package ru.yandex.intranet.d.services.provisions;

import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple4;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.quotas.ExpandedQuotas;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Provisions read service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ProvisionsReadService {

    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final YdbTableClient tableClient;
    private final ProvidersLoader providersLoader;
    private final ResourcesLoader resourcesLoader;
    private final FolderDao folderDao;
    private final AccountsDao accountsDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final ObjectReader byAccountContinuationTokenReader;
    private final ObjectWriter byAccountContinuationTokenWriter;

    @SuppressWarnings("ParameterNumber")
    public ProvisionsReadService(@Qualifier("messageSource") MessageSource messages,
                                 @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
                                 SecurityManagerService securityManagerService,
                                 YdbTableClient tableClient,
                                 ProvidersLoader providersLoader,
                                 ResourcesLoader resourcesLoader,
                                 FolderDao folderDao,
                                 AccountsDao accountsDao,
                                 AccountsQuotasDao accountsQuotasDao,
                                 UnitsEnsemblesLoader unitsEnsemblesLoader) {
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.tableClient = tableClient;
        this.providersLoader = providersLoader;
        this.resourcesLoader = resourcesLoader;
        this.folderDao = folderDao;
        this.accountsDao = accountsDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.byAccountContinuationTokenReader = objectMapper.getObjectMapper()
                .readerFor(ProvisionsByAccountContinuationToken.class);
        this.byAccountContinuationTokenWriter = objectMapper.getObjectMapper()
                .writerFor(ProvisionsByAccountContinuationToken.class);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Result<ExpandedQuotas<AccountsQuotasModel>>> getOneProvision(String folderId, String accountId,
                                                                             String providerId, String resourceId,
                                                                             YaUserDetails currentUser,
                                                                             Locale locale) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res ->
                res.andThen(v -> validateIds(folderId, accountId, providerId, resourceId, locale)).andThenMono(u ->
                        tableClient.usingSessionMonoRetryable(session ->
                                getEntities(session, folderId, accountId, providerId, resourceId, currentUser,
                                        locale, tenantId).flatMap(tr -> tr
                                            .andThenMono(t -> getProvision(session, t.getT1(), t.getT2(), t.getT3(),
                                                    t.getT4(), tenantId))))));
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Result<ExpandedQuotas<Page<AccountsQuotasModel>>>> getAccountProvisions(String folderId,
                                                                                        String accountId,
                                                                                        PageRequest pageRequest,
                                                                                        YaUserDetails currentUser,
                                                                                        Locale locale) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res ->
                res.andThen(v -> validateFolderIdAccountId(folderId, accountId, locale)).andThenMono(u -> {
                    Result<PageRequest.Validated<ProvisionsByAccountContinuationToken>> pageValidation
                            = pageRequest.validate(byAccountContinuationTokenReader, messages, locale);
                    return pageValidation
                            .andThenDo(p -> validateByAccountContinuationToken(p, locale)).andThenMono(p -> {
                                int limit = p.getLimit();
                                String fromResourceId = p.getContinuationToken()
                                        .map(ProvisionsByAccountContinuationToken::getResourceId).orElse(null);
                                return tableClient.usingSessionMonoRetryable(session -> getFolderAccount(session,
                                        folderId, accountId, currentUser, locale, tenantId).flatMap(tr ->
                                        tr.andThenMono(t -> accountsQuotasDao.getPageByAccount(immediateTx(session),
                                                        t.getT2().getId(), fromResourceId, tenantId, limit + 1)
                                                .map(values -> values.size() > limit
                                                        ? Page.page(values.subList(0, limit),
                                                        prepareByAccountToken(values.get(limit - 1)))
                                                        : Page.lastPage(values))
                                                .flatMap(page -> expand(session, page, tenantId))
                                                .map(Result::success))));
                            });
                }));
    }

    private Mono<ExpandedQuotas<Page<AccountsQuotasModel>>> expand(YdbSession session,
                                                                   Page<AccountsQuotasModel> page,
                                                                   TenantId tenantId) {
        Set<String> resourceIds = new HashSet<>();
        page.getItems().forEach(provision -> resourceIds.add(provision.getResourceId()));
        // At most 100 items per page so no need to paginate resource loading
        List<Tuple2<String, TenantId>> resourceIdPairs = resourceIds.stream()
                .map(id -> Tuples.of(id, tenantId)).collect(Collectors.toList());
        return resourcesLoader.getResourcesByIds(immediateTx(session), resourceIdPairs).flatMap(loadedResources -> {
            Map<String, ResourceModel> resources = loadedResources.stream()
                    .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
            Set<String> unitsEnsembleIds = new HashSet<>();
            loadedResources.forEach(r -> unitsEnsembleIds.add(r.getUnitsEnsembleId()));
            List<Tuple2<String, TenantId>> unitsEnsembleIdPairs = unitsEnsembleIds.stream()
                    .map(id -> Tuples.of(id, tenantId)).collect(Collectors.toList());
            return unitsEnsemblesLoader.getUnitsEnsemblesByIds(immediateTx(session), unitsEnsembleIdPairs)
                    .map(loadedUnitsEnsembles -> {
                        Map<String, UnitsEnsembleModel> unitsEnsembles = loadedUnitsEnsembles.stream()
                                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
                        return new ExpandedQuotas<>(page, resources, unitsEnsembles);
                    });
        });
    }

    private Result<Void> validateIds(String folderId, String accountId, String providerId, String resourceId,
                                     Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!Uuids.isValidUuid(folderId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        }
        if (!Uuids.isValidUuid(accountId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.account.not.found", null, locale)));
        }
        if (!Uuids.isValidUuid(providerId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)));
        }
        if (!Uuids.isValidUuid(resourceId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.resource.not.found", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Result<Tuple4<FolderModel, AccountModel, ProviderModel, ResourceModel>>> getEntities(
            YdbSession session, String folderId, String accountId, String providerId, String resourceId,
            YaUserDetails currentUser, Locale locale, TenantId tenantId) {
        return Mono.zip(providersLoader.getProviderByIdImmediate(providerId, tenantId),
                resourcesLoader.getResourceByIdImmediate(resourceId, tenantId)).flatMap(providerOAndResourceO ->
                folderDao.getById(immediateTx(session), folderId, tenantId).flatMap(folderO ->
                accountsDao.getById(immediateTx(session), accountId, tenantId).flatMap(accountO -> {
                    Optional<ProviderModel> providerO = providerOAndResourceO.getT1();
                    Optional<ResourceModel> resourceO = providerOAndResourceO.getT2();
                    ErrorCollection.Builder errors = ErrorCollection.builder();
                    if (providerO.isEmpty() || providerO.get().isDeleted()) {
                        errors.addError(TypedError
                                .notFound(messages.getMessage("errors.provider.not.found", null, locale)));
                    }
                    if (resourceO.isEmpty() || resourceO.get().isDeleted()) {
                        errors.addError(TypedError
                                .notFound(messages.getMessage("errors.resource.not.found", null, locale)));
                    }
                    if (resourceO.isPresent() && providerO.isPresent()
                            && !resourceO.get().getProviderId().equals(providerO.get().getId())) {
                        errors.addError(TypedError.notFound(messages
                                .getMessage("errors.resource.not.found.in.this.provider", null, locale)));
                    }
                    if (folderO.isEmpty() || folderO.get().isDeleted()) {
                        errors.addError(TypedError
                                .notFound(messages.getMessage("errors.folder.not.found", null, locale)));
                    }
                    if (accountO.isEmpty() || accountO.get().isDeleted()) {
                        errors.addError(TypedError
                                .notFound(messages.getMessage("errors.account.not.found", null, locale)));
                    }
                    if (accountO.isPresent() && folderO.isPresent()
                            && !accountO.get().getFolderId().equals(folderO.get().getId())) {
                        errors.addError(TypedError.notFound(messages
                                .getMessage("errors.account.not.found.in.this.folder", null, locale)));
                    }
                    if (errors.hasAnyErrors()) {
                        return Mono.just(Result.failure(errors.build()));
                    }
                    return securityManagerService.checkReadPermissions(folderO.orElseThrow(), currentUser, locale,
                            Tuples.of(folderO.orElseThrow(), accountO.orElseThrow(), providerO.orElseThrow(),
                                    resourceO.orElseThrow()));
                })));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Result<ExpandedQuotas<AccountsQuotasModel>>> getProvision(
            YdbSession session, FolderModel folder, AccountModel account, ProviderModel provider,
            ResourceModel resource, TenantId tenantId) {
        AccountsQuotasModel.Identity id = new AccountsQuotasModel.Identity(account.getId(), resource.getId());
        return accountsQuotasDao.getById(immediateTx(session), id, tenantId).flatMap(provisionO ->
                unitsEnsemblesLoader.getUnitsEnsembleById(immediateTx(session), resource.getUnitsEnsembleId(),
                        tenantId).flatMap(unitsEnsembleO -> {
                    if (unitsEnsembleO.isEmpty()) {
                        return Mono.error(new IllegalStateException("Missing units ensemble "
                                + resource.getUnitsEnsembleId()));
                    }
                    AccountsQuotasModel provision = provisionO.orElseGet(() -> new AccountsQuotasModel.Builder()
                            .setTenantId(tenantId)
                            .setFolderId(folder.getId())
                            .setAccountId(account.getId())
                            .setProviderId(provider.getId())
                            .setResourceId(resource.getId())
                            .setProvidedQuota(0L)
                            .setAllocatedQuota(0L)
                            .setLastProvisionUpdate(Instant.now())
                            .build());
                    return Mono.just(Result.success(new ExpandedQuotas<>(provision, Map.of(resource.getId(), resource),
                            Map.of(unitsEnsembleO.get().getId(), unitsEnsembleO.get()))));
                }));
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private Result<Void> validateFolderIdAccountId(String folderId, String accountId, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!Uuids.isValidUuid(folderId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        }
        if (!Uuids.isValidUuid(accountId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.account.not.found", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private Mono<Result<Tuple2<FolderModel, AccountModel>>> getFolderAccount(YdbSession session,
                                                                             String folderId,
                                                                             String accountId,
                                                                             YaUserDetails currentUser,
                                                                             Locale locale,
                                                                             TenantId tenantId) {
        return folderDao.getById(immediateTx(session), folderId, tenantId).flatMap(folderO ->
                accountsDao.getById(immediateTx(session), accountId, tenantId).flatMap(accountO -> {
            ErrorCollection.Builder errors = ErrorCollection.builder();
            if (folderO.isEmpty() || folderO.get().isDeleted()) {
                errors.addError(TypedError
                        .notFound(messages.getMessage("errors.folder.not.found", null, locale)));
            }
            if (accountO.isEmpty() || accountO.get().isDeleted()) {
                errors.addError(TypedError
                        .notFound(messages.getMessage("errors.account.not.found", null, locale)));
            }
            if (accountO.isPresent() && folderO.isPresent()
                    && !accountO.get().getFolderId().equals(folderO.get().getId())) {
                errors.addError(TypedError
                        .notFound(messages.getMessage("errors.account.not.found", null, locale)));
            }
            if (errors.hasAnyErrors()) {
                return Mono.just(Result.failure(errors.build()));
            }
            return securityManagerService.checkReadPermissions(folderO.orElseThrow(), currentUser, locale,
                    Tuples.of(folderO.orElseThrow(), accountO.orElseThrow()));
        }));
    }

    private Result<Void> validateByAccountContinuationToken(
            PageRequest.Validated<ProvisionsByAccountContinuationToken> pageRequest, Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!Uuids.isValidUuid(pageRequest.getContinuationToken().get().getResourceId())) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.invalid.continuation.token", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private String prepareByAccountToken(AccountsQuotasModel lastItem) {
        return ContinuationTokens.encode(new ProvisionsByAccountContinuationToken(lastItem.getResourceId()),
                byAccountContinuationTokenWriter);
    }

}
