package ru.yandex.intranet.d.services.provisions

import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao
import ru.yandex.intranet.d.dao.accounts.FolderProviderAccountsSpace
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao
import ru.yandex.intranet.d.dao.quotas.QuotasDao
import ru.yandex.intranet.d.dao.resources.ResourcesDao
import ru.yandex.intranet.d.dao.users.UsersDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.kotlin.AccountId
import ru.yandex.intranet.d.kotlin.AccountsSpacesId
import ru.yandex.intranet.d.kotlin.ExternalAccountId
import ru.yandex.intranet.d.kotlin.FolderId
import ru.yandex.intranet.d.kotlin.OperationId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ResourceTypeId
import ru.yandex.intranet.d.kotlin.SegmentId
import ru.yandex.intranet.d.kotlin.SegmentationId
import ru.yandex.intranet.d.kotlin.UnitsEnsembleId
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.elapsed
import ru.yandex.intranet.d.kotlin.getOrNull
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader
import ru.yandex.intranet.d.model.WithTenant
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel.RequestStatus
import ru.yandex.intranet.d.model.accounts.OperationChangesModel
import ru.yandex.intranet.d.model.accounts.OperationErrorCollections
import ru.yandex.intranet.d.model.accounts.OperationErrorKind
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel
import ru.yandex.intranet.d.model.accounts.OperationSource
import ru.yandex.intranet.d.model.folders.AccountHistoryModel
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel
import ru.yandex.intranet.d.model.folders.FolderOperationType
import ru.yandex.intranet.d.model.folders.OperationPhase
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel
import ru.yandex.intranet.d.model.folders.ProvisionsByResource
import ru.yandex.intranet.d.model.folders.QuotasByAccount
import ru.yandex.intranet.d.model.folders.QuotasByResource
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.quotas.QuotaModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.model.users.UserModel
import ru.yandex.intranet.d.services.accounts.ReserveAccountsService
import ru.yandex.intranet.d.services.integration.providers.ProviderError
import ru.yandex.intranet.d.services.integration.providers.ProvidersIntegrationService
import ru.yandex.intranet.d.services.integration.providers.Response
import ru.yandex.intranet.d.services.integration.providers.getExternalAccountIdFromReceivedAccount
import ru.yandex.intranet.d.services.integration.providers.getFolderFromReceivedAccount
import ru.yandex.intranet.d.services.integration.providers.getOperationsFromReceivedAccount
import ru.yandex.intranet.d.services.integration.providers.getOperationsFromReceivedUpdatedProvision
import ru.yandex.intranet.d.services.integration.providers.getUsersFromReceivedAccount
import ru.yandex.intranet.d.services.integration.providers.getUsersFromReceivedUpdatedProvision
import ru.yandex.intranet.d.services.integration.providers.isProvisionChangesApplied
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.GetAccountRequestDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownAccountProvisionsDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownProvisionDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionRequestDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceKeyRequestDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyRequestDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionRequestDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionResponseDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.UserIdDto
import ru.yandex.intranet.d.services.integration.providers.validateProviderAccountDto
import ru.yandex.intranet.d.services.integration.providers.validateReceivedAccount
import ru.yandex.intranet.d.services.integration.providers.validateReceivedUpdatedProvision
import ru.yandex.intranet.d.services.integration.providers.validateUpdateProvisionResponseDto
import ru.yandex.intranet.d.services.operations.OperationsObservabilityService
import ru.yandex.intranet.d.services.operations.model.ReceivedAccount
import ru.yandex.intranet.d.services.operations.model.ReceivedUserId
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedAccount
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedUpdatedProvision
import ru.yandex.intranet.d.services.quotas.ProvideReserveOperationContext
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.services.uniques.ProvideReserveOperation
import ru.yandex.intranet.d.services.uniques.RequestUniqueService
import ru.yandex.intranet.d.util.Uuids
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.util.units.Units
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.provisions.ProvideReserveOperationFailureMeta
import ru.yandex.intranet.d.web.model.provisions.ProviderReserveProvisionResponseValueDto
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsRequestDto
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsResponseDto
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsResultDto
import ru.yandex.intranet.d.web.model.provisions.UpdateProviderReserveProvisionsStatusDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.math.BigInteger
import java.time.Instant
import java.util.*

private val logger = KotlinLogging.logger {}

/**
 * Reserve provisions service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ReserveProvisionsService(
    private val requestUniqueService: RequestUniqueService,
    private val securityManagerService: SecurityManagerService,
    private val reserveAccountsService: ReserveAccountsService,
    private val operationsObservabilityService: OperationsObservabilityService,
    private val providersIntegrationService: ProvidersIntegrationService,
    private val providersLoader: ProvidersLoader,
    private val resourcesLoader: ResourcesLoader,
    private val unitsEnsemblesLoader: UnitsEnsemblesLoader,
    private val resourceTypesLoader: ResourceTypesLoader,
    private val resourceSegmentationsLoader: ResourceSegmentationsLoader,
    private val resourceSegmentsLoader: ResourceSegmentsLoader,
    private val accountsSpacesDao: AccountsSpacesDao,
    private val accountsDao: AccountsDao,
    private val folderDao: FolderDao,
    private val quotasDao: QuotasDao,
    private val accountsQuotasDao: AccountsQuotasDao,
    private val resourcesDao: ResourcesDao,
    private val accountsQuotasOperationsDao: AccountsQuotasOperationsDao,
    private val folderOperationLogDao: FolderOperationLogDao,
    private val operationsInProgressDao: OperationsInProgressDao,
    private val usersDao: UsersDao,
    private val tableClient: YdbTableClient,
    @Qualifier("messageSource") private val messages: MessageSource
) {

    fun provideReserveMono(providerId: String,
                           request: UpdateProviderReserveProvisionsRequestDto,
                           idempotencyKey: String?,
                           currentUser: YaUserDetails,
                           locale: Locale): Mono<Result<UpdateProviderReserveProvisionsResponseDto>> {
        return mono {
            provideReserve(providerId, request, idempotencyKey, currentUser, locale)
        }
    }

    suspend fun provideReserve(providerId: String,
                               request: UpdateProviderReserveProvisionsRequestDto,
                               idempotencyKey: String?,
                               currentUser: YaUserDetails,
                               locale: Locale): Result<UpdateProviderReserveProvisionsResponseDto> = binding {
        // Check if user can read anything
        meter({ securityManagerService.checkReadPermissions(currentUser, locale).awaitSingle() },
            "Provide reserve, check read permissions").bind()
        // Sanity check for incoming DTO
        val preValidatedRequest = preValidateRequest(providerId, request, currentUser, locale).bind()!!
        // Sanity check for idempotency key, substitute random default value if missing
        val checkedIdempotencyKey = requestUniqueService.validateIdempotencyKey(idempotencyKey, locale).bind()!!
        // Validate provider
        val provider = meter({ validateProvider(providerId, locale) },
            "Provide reserve, validate provider").bind()!!
        // Validate accounts space
        val accountsSpace = meter({ validateAccountsSpace(provider, preValidatedRequest, locale) },
            "Provide reserve, validate accounts space").bind()
        val (toApply, alreadyApplied) = dbSessionRetryable(tableClient) { rwTxRetryable {
            provideReservePreRequest(txSession, checkedIdempotencyKey, provider, accountsSpace, preValidatedRequest,
                currentUser, locale)
        } }!!.bind()!!
        if (toApply != null) {
            val resultR: Result<UpdateProviderReserveProvisionsResponseDto>
            try {
                operationsObservabilityService.observeOperationSubmitted(toApply.op)
                val updateProvisionResult = meter({
                    providersIntegrationService.updateProvision(toApply.account.outerAccountIdInProvider,
                        toApply.provider, toApply.request, locale).awaitSingle()
                }, "Provide reserve, update provision in provider")
                resultR = provideReservePostRequest(updateProvisionResult, toApply.request, toApply.op.operationId,
                    toApply.folder.id, toApply.folder.serviceId, toApply.account.outerAccountIdInProvider,
                    toApply.account.id, provider, accountsSpace, toApply.resourcesDictionary, currentUser, locale)
            } catch (e: Exception) {
                logger.error(e) { "Unexpected error while providing reserve" }
                if (e is CancellationException) {
                    throw e
                }
                // Unexpected error but operation was already started, it will be finished in background,
                // return 'in progress' in the meantime
                return@binding Result.success(UpdateProviderReserveProvisionsResponseDto(
                    UpdateProviderReserveProvisionsStatusDto.IN_PROGRESS, null, toApply.op.operationId))
            }
            val result = resultR.bind()!!
            Result.success(result)
        } else {
            Result.success(alreadyApplied!!)
        }
    }

    private suspend fun provideReservePreRequest(
        txSession: YdbTxSession, checkedIdempotencyKey: String, provider: ProviderModel,
        accountsSpace: AccountSpaceModel?, preValidatedRequest: UpdateProviderReserveProvisionsRequest,
        currentUser: YaUserDetails, locale: Locale
    ): Result<Pair<UpdatedQuotas?, UpdateProviderReserveProvisionsResponseDto?>> = binding {
        val existingOperation = meter({ requestUniqueService.checkProvideReserve(txSession, checkedIdempotencyKey,
            currentUser, locale) }, "Provide reserve, check for existing idempotency key").bind()
        if (existingOperation != null) {
            checkExistingOperationMatch(existingOperation, provider, accountsSpace, locale).bind()
            checkExistingOperationResult(existingOperation, locale).bind()
            Result.success(Pair(null, prepareExistingOperationResponse(existingOperation)))
        } else {
            meter({ securityManagerService.checkWritePermissionsForProvider(provider.id, currentUser, locale)
                .awaitSingle() }, "Provide reserve, check write permissions").bind()
            val validatedRequest = meter({ validateRequestValues(txSession, provider, accountsSpace,
                preValidatedRequest, locale) }, "Provide reserve, validate supplied provisions").bind()!!
            validateMutability(validatedRequest, locale).bind()
            val reserveAccount = meter({ findReserveAccount(txSession, validatedRequest, locale) },
                "Provide reserve, find reserve account").bind()!!
            val resourcesDictionary = meter({ loadResourceDictionary(txSession, validatedRequest, accountsSpace,
                provider) }, "Provide reserve, load resource metadata")
            val currentQuotas = meter({ loadCurrentQuotas(txSession, reserveAccount,
                resourcesDictionary.resources.keys) }, "Provide reserve, load current quotas")
            val now = Instant.now()
            val updatedQuotas = prepareUpdatedQuotas(currentQuotas, reserveAccount, validatedRequest,
                resourcesDictionary, now, currentUser, locale).bind()!!
            meter({ requestUniqueService.addProviderReserve(txSession, checkedIdempotencyKey, updatedQuotas.op,
                now, currentUser) }, "Provide reserve, save idempotency key")
            meter({ folderDao.upsertOneRetryable(txSession, updatedQuotas.folder).awaitSingle() },
                "Provide reserve, upsert updated folder")
            if (updatedQuotas.quotas.isNotEmpty()) {
                meter({ quotasDao.upsertAllRetryable(txSession, updatedQuotas.quotas).awaitSingleOrNull() },
                    "Provide reserve, upsert updated quotas")
            }
            meter({ accountsQuotasOperationsDao.upsertOneRetryable(txSession, updatedQuotas.op).awaitSingle() },
                "Provide reserve, upsert operation")
            meter({ folderOperationLogDao.upsertOneRetryable(txSession, updatedQuotas.opLog).awaitSingle() },
                "Provide reserve, upsert folder history")
            meter({ operationsInProgressDao.upsertOneRetryable(txSession,
                updatedQuotas.operationInProgress).awaitSingle() }, "Provide reserve, upsert operation in progress")
            Result.success(Pair(updatedQuotas, null))
        }
    }

    fun prepareReserveProvisionForTransferRequest(
        transferRequest: TransferRequestModel,
        provisionTransfer: ProvisionTransfer,
        payOff: Boolean,
        account: AccountModel,
        folder: FolderModel,
        provider: ProviderModel,
        accountsSpaceId: AccountsSpacesId?,
        resourceById: Map<ResourceId, ResourceModel>,
        accountsQuotas: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>,
        folderQuotas: Map<ResourceId, QuotaModel>,
        currentUser: YaUserDetails,
        locale: Locale
    ): ProvideReserveOperationContext {
        if (true != transferRequest.loanMeta.getOrNull()?.provideOverCommitReserve && !payOff) {
            throw IllegalArgumentException("'Provide over commit reserve' flag is not set for provision transfer!")
        }
        val transfers = if (payOff) {
            provisionTransfer.sourceAccountTransfers
        } else {
            provisionTransfer.destinationAccountTransfers
        }
        val currentQuotas = CurrentQuotas(folderQuotas, accountsQuotas, mapOf(account.id to account), folder)
        val updatedQuotasResult = prepareUpdatedQuotas(transferRequest, currentQuotas, account, transfers, resourceById,
            provider, accountsSpaceId, Instant.now(), currentUser, locale)
        return updatedQuotasResult.match({ it }, { errors ->
            logger.error {
                "Failed to prepare reserve provision for transfer request ${transferRequest.id}: $errors"
            }
            throw IllegalArgumentException("Failed to prepare reserve provision for transfer request")
        })
    }

    private fun prepareUpdatedQuotas(
        transferRequest: TransferRequestModel,
        currentQuotas: CurrentQuotas,
        account: AccountModel,
        transfers: Set<ResourceQuotaTransfer>,
        resourceById: Map<ResourceId, ResourceModel>,
        provider: ProviderModel,
        accountsSpaceId: String?,
        now: Instant,
        currentUser: YaUserDetails,
        locale: Locale
    ): Result<ProvideReserveOperationContext> {
        val errors = ErrorCollection.builder()
        val updatedQuotas = mutableListOf<QuotaModel>()
        val updatedProvisions = mutableMapOf<ResourceId, Long>()
        val frozenAmounts = mutableMapOf<ResourceId, Long>()
        val oldQuotas = mutableMapOf<ResourceId, Long>()
        val newQuotas = mutableMapOf<ResourceId, Long>()

        prepareUpdatedQuotasDelta(transfers, account, currentQuotas, updatedQuotas, updatedProvisions,
            frozenAmounts, oldQuotas, newQuotas, errors, locale)

        (currentQuotas.folderProvisions[account.id] ?: emptyMap()).forEach { (resourceId, provision) ->
            val resource = resourceById[resourceId]!!
            val currentProvision = provision.providedQuota ?: 0L
            if (!resource.isReadOnly && resource.isManaged && !resource.isDeleted && currentProvision > 0L
                && !updatedProvisions.containsKey(resourceId)
            ) {
                // Current values for unchanged non-zero provisions of the updated account
                updatedProvisions[resourceId] = currentProvision
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        val knownProvisions = mutableMapOf<AccountId, MutableMap<ResourceId, Long>>()
        currentQuotas.accounts.keys.forEach { accountId ->
            val accountKnownProvisions = knownProvisions.computeIfAbsent(accountId) { mutableMapOf() }
            val accountProvisions = currentQuotas.folderProvisions[accountId] ?: emptyMap()
            accountProvisions.forEach { (resourceId, provision) ->
                val currentProvision = provision.providedQuota ?: 0L
                if (currentProvision != 0L || updatedProvisions.containsKey(resourceId)) {
                    // Current values for non-zero provisions and provisions of resources in updatedProvisions
                    accountKnownProvisions[resourceId] = currentProvision
                }
            }
            updatedProvisions.keys.forEach { resourceId ->
                if (!accountKnownProvisions.containsKey(resourceId)) {
                    // Zero for all remaining provisions of resources in updatedProvisions
                    accountKnownProvisions[resourceId] = 0L
                }
            }
        }
        val operation = prepareOperation(now, currentUser, provider, accountsSpaceId,
            currentQuotas.folder, account, updatedProvisions, frozenAmounts, transferRequest)
        val operationLog = prepareSubmitOperationLog(operation, currentQuotas.folder, currentUser,
            oldQuotas, newQuotas)
        val updatedFolder = currentQuotas.folder.toBuilder()
            .setNextOpLogOrder(currentQuotas.folder.nextOpLogOrder + 1L).build()
        val operationInProgress = OperationInProgressModel(Tenants.DEFAULT_TENANT_ID, operation.operationId,
            currentQuotas.folder.id, account.id, 0L)
        return Result.success(ProvideReserveOperationContext(updatedQuotas, updatedFolder, operationLog, operation,
            operationInProgress))
    }

    private suspend fun provideReservePostRequest(providerResponse: Result<Response<UpdateProvisionResponseDto>>,
                                                  request: UpdateProvisionRequestDto,
                                                  operationId: OperationId,
                                                  folderId: FolderId,
                                                  serviceId: Long,
                                                  externalAccountId: ExternalAccountId,
                                                  accountId: AccountId,
                                                  provider: ProviderModel,
                                                  accountsSpace: AccountSpaceModel?,
                                                  resourceDictionary: ResourceDictionary,
                                                  currentUser: YaUserDetails,
                                                  locale: Locale
    ): Result<UpdateProviderReserveProvisionsResponseDto> {
        val now = Instant.now()
        return providerResponse.matchSuspend(
            { response ->
                response.matchSuspend(
                    { providerResponseDto, reqId ->
                        val preValidatedResponse = validateUpdateProvisionResponseDto(providerResponseDto,
                            messages, locale)
                        preValidatedResponse.matchSuspend(
                            { preValidatedProviderResponseDto ->
                                val receivedUsers = getUsersFromReceivedUpdatedProvision(preValidatedProviderResponseDto)
                                val receivedOperations = getOperationsFromReceivedUpdatedProvision(preValidatedProviderResponseDto)
                                val knownUsers = loadUsers(receivedUsers)
                                val knownOperations = loadOperations(receivedOperations)
                                val validatedResponse = validateReceivedUpdatedProvision(
                                    preValidatedProviderResponseDto, knownUsers, knownOperations, provider,
                                    accountsSpace, resourceDictionary.resources, resourceDictionary.segmentations,
                                    resourceDictionary.segments, resourceDictionary.resourceTypes,
                                    resourceDictionary.unitsEnsembles, messages, locale)
                                validatedResponse.matchSuspend(
                                    { validatedResponseDto ->
                                        val updateResult = dbSessionRetryable(tableClient) { rwTxRetryable {
                                            val postRequestContext = preparePostRequestContext(txSession, operationId,
                                                folderId, accountId, resourceDictionary)
                                            completeOperation(txSession, postRequestContext, validatedResponseDto,
                                                provider, resourceDictionary, now, currentUser)
                                        }}!!
                                        if (updateResult.updatedOperation != null) {
                                            operationsObservabilityService.observeOperationFinished(updateResult.updatedOperation)
                                        }
                                        Result.success(updateResult.response)
                                    },
                                    { errors -> onPostRequestValidationError(errors, reqId, request,
                                        providerResponseDto, operationId, folderId, accountId, provider,
                                        resourceDictionary, now, currentUser, locale) })
                            },
                            { errors -> onPostRequestValidationError(errors, reqId, request, providerResponseDto,
                                operationId, folderId, accountId, provider, resourceDictionary, now,
                                currentUser, locale) })
                    },
                    { ex -> onPostRequestUnexpectedError(ex, request, operationId, folderId, accountId, provider,
                        resourceDictionary, now, currentUser, locale) },
                    { providerError, reqId ->
                        logger.error { "Unexpected error during reserve provision for provider ${provider.key} " +
                            "(requestId = ${reqId}): ${providerError}. Request = ${request}." }
                        if (providerError.isConflict) {
                            val getAccountRequest = prepareGetAccountRequest(folderId, serviceId,
                                accountsSpace, resourceDictionary)
                            val getAccountResult = meter({
                                providersIntegrationService.getAccount(externalAccountId, provider, getAccountRequest,
                                    locale).awaitSingle() }, "Provide reserve, get account from provider")
                            handleReloadedAccount(getAccountResult, getAccountRequest, providerError, operationId,
                                folderId, accountId, provider, accountsSpace, resourceDictionary, now, currentUser,
                                locale)
                        } else {
                            val errorMessage = Errors.flattenProviderErrorResponse(providerError, null)
                            val providerErrorCollection = Errors.providerErrorResponseToErrorCollection(providerError)
                            val operationErrors = OperationErrorCollections.builder()
                                .addErrorCollection(Locales.ENGLISH, providerErrorCollection)
                                .addErrorCollection(Locales.RUSSIAN, providerErrorCollection)
                                .build()
                            val operationErrorKind = Errors.providerErrorResponseToOperationErrorKind(providerError)
                            val rollbackResult = dbSessionRetryable(tableClient) {
                                rwTxRetryable {
                                    val postRequestContext = preparePostRequestContext(txSession, operationId,
                                        folderId, accountId, resourceDictionary)
                                    rollbackOperation(txSession, postRequestContext.currentOperation,
                                        postRequestContext.currentOperationInProgress,
                                        postRequestContext.currentQuotas.folderQuotas,
                                        postRequestContext.currentQuotas.folder, now, errorMessage,
                                        operationErrors, operationErrorKind, currentUser)
                                }
                            }
                            if (rollbackResult != null) {
                                operationsObservabilityService.observeOperationFinished(rollbackResult.updatedOperation)
                            }
                            Result.failure(providerErrorCollection)
                        }
                    })
            },
            { errors -> onPreRequestValidationError(errors, request, operationId, folderId, accountId, provider,
                resourceDictionary, now, currentUser) })
    }

    private fun prepareGetAccountRequest(folderId: FolderId,
                                         serviceId: Long,
                                         accountsSpace: AccountSpaceModel?,
                                         resourceDictionary: ResourceDictionary): GetAccountRequestDto {
        val accountsSpaceKey = prepareAccountsSpaceKey(accountsSpace, resourceDictionary)
        return GetAccountRequestDto(true, false, folderId, serviceId, accountsSpaceKey)
    }

    private suspend fun handleReloadedAccount(
        getAccountResult: Result<Response<AccountDto>>,
        getAccountRequest: GetAccountRequestDto,
        originalProviderError: ProviderError,
        operationId: OperationId,
        folderId: FolderId,
        accountId: AccountId,
        provider: ProviderModel,
        accountsSpace: AccountSpaceModel?,
        resourceDictionary: ResourceDictionary,
        now: Instant,
        currentUser: YaUserDetails,
        locale: Locale
    ): Result<UpdateProviderReserveProvisionsResponseDto> {
        return getAccountResult.matchSuspend(
            { response ->
                response.matchSuspend(
                    { providerAccountDto, reqId ->
                        val preValidatedAccountDto = validateProviderAccountDto(providerAccountDto, messages, locale)
                        preValidatedAccountDto.matchSuspend(
                            { receivedAccount ->
                                val validatedReceivedAccount = validateReceivedExternalAccount(receivedAccount,
                                    provider, accountsSpace, resourceDictionary, locale)
                                validatedReceivedAccount.matchSuspend(
                                    { receivedProviderAccount ->
                                        handleRefreshedAccountOnConflict(receivedProviderAccount,
                                            originalProviderError, operationId, folderId, accountId, provider,
                                            resourceDictionary, now, currentUser, locale)
                                    },
                                    { errors ->
                                        logger.error {
                                            "Unexpected error while getting account $accountId for " +
                                                "provider ${provider.key} (requestId = ${reqId}): ${errors}. " +
                                                "Request = ${getAccountRequest}. Response = ${providerAccountDto}."
                                        }
                                        handleRollbackOnConflict(originalProviderError, operationId, folderId, accountId,
                                            provider, resourceDictionary, now, currentUser, locale)
                                    })
                            },
                            { errors ->
                                logger.error {
                                    "Unexpected error while getting account $accountId for " +
                                        "provider ${provider.key} (requestId = ${reqId}): ${errors}. " +
                                        "Request = ${getAccountRequest}. Response = ${providerAccountDto}."
                                }
                                handleRollbackOnConflict(originalProviderError, operationId, folderId, accountId,
                                    provider, resourceDictionary, now, currentUser, locale)
                            })
                    },
                    { ex ->
                        logger.error(ex) {
                            "Unexpected error while getting account $accountId for provider ${provider.key}. " +
                                "Request = ${getAccountRequest}."
                        }
                        handleRollbackOnConflict(originalProviderError, operationId, folderId, accountId, provider,
                            resourceDictionary, now, currentUser, locale)
                    },
                    { providerError, reqId ->
                        logger.error { "Unexpected error while getting account $accountId for provider ${provider.key} " +
                            "(requestId = ${reqId}): ${providerError}. Request = ${getAccountRequest}." }
                        handleRollbackOnConflict(originalProviderError, operationId, folderId, accountId, provider,
                            resourceDictionary, now, currentUser, locale)
                    })
            },
            { errors ->
                logger.error {
                    "Unexpected error while getting account $accountId for provider ${provider.key}: ${errors}. " +
                        "Request = ${getAccountRequest}."
                }
                handleRollbackOnConflict(originalProviderError, operationId, folderId, accountId, provider,
                    resourceDictionary, now, currentUser, locale)
            })
    }

    private suspend fun handleRefreshedAccountOnConflict(
        refreshedAccount: ValidatedReceivedAccount,
        originalProviderError: ProviderError,
        operationId: OperationId,
        folderId: FolderId,
        accountId: AccountId,
        provider: ProviderModel,
        resourceDictionary: ResourceDictionary,
        now: Instant,
        currentUser: YaUserDetails,
        locale: Locale
    ): Result<UpdateProviderReserveProvisionsResponseDto> {
        val (updatedOperation, updateResult) = dbSessionRetryable(tableClient) { rwTxRetryable {
            val postRequestContext = preparePostRequestContext(txSession, operationId, folderId,
                accountId, resourceDictionary)
            if (checkOperationApplied(refreshedAccount, postRequestContext, provider)) {
                val completedOperation = completeOperationAfterRefresh(txSession,
                    postRequestContext.currentOperation, postRequestContext.currentOperationInProgress,
                    postRequestContext.currentAccount, postRequestContext.currentQuotas.folder,
                    postRequestContext.currentQuotas.folderProvisions, postRequestContext.currentQuotas.folderQuotas,
                    refreshedAccount, provider, resourceDictionary.resources, resourceDictionary.unitsEnsembles,
                    now, currentUser)
                Pair(completedOperation.updatedOperation, Result.success(completedOperation.response))
            } else {
                val prefix = messages.getMessage("errors.accounts.quotas.out.of.sync.with.provider", null, locale)
                val errorMessage = Errors.flattenProviderErrorResponse(originalProviderError, prefix)
                val providerErrorCollection = Errors.providerErrorResponseToErrorCollection(originalProviderError)
                val operationErrors = OperationErrorCollections.builder()
                    .addErrorCollection(Locales.ENGLISH, providerErrorCollection)
                    .addErrorCollection(Locales.RUSSIAN, providerErrorCollection)
                    .build()
                val operationErrorKind = Errors.providerErrorResponseToOperationErrorKind(originalProviderError)
                val rollbackResult = rollbackOperation(txSession, postRequestContext.currentOperation,
                    postRequestContext.currentOperationInProgress, postRequestContext.currentQuotas.folderQuotas,
                    postRequestContext.currentQuotas.folder, now, errorMessage, operationErrors,
                    operationErrorKind, currentUser)
                Pair(rollbackResult?.updatedOperation, Result.failure(providerErrorCollection))
            }
        }}!!
        if (updatedOperation != null) {
            operationsObservabilityService.observeOperationFinished(updatedOperation)
        }
        return updateResult
    }

    suspend fun completeOperationAfterRefresh(txSession: YdbTxSession,
                                              currentOperation: AccountsQuotasOperationsModel,
                                              currentOperationInProgress: OperationInProgressModel?,
                                              currentAccount: AccountModel,
                                              currentFolder: FolderModel,
                                              currentFolderProvisions: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>,
                                              folderQuotas: Map<ResourceId, QuotaModel>,
                                              refreshedAccount: ValidatedReceivedAccount,
                                              provider: ProviderModel,
                                              resources: Map<ResourceId, ResourceModel>,
                                              unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>,
                                              now: Instant,
                                              currentUser: YaUserDetails?
    ): ReserveProvisionCompletionResult {
        if (currentOperation.requestStatus.isPresent
            && currentOperation.requestStatus.get() != RequestStatus.WAITING
            && currentOperationInProgress == null
        ) {
            // Operation was already finished, do nothing
            val resultProvisions = (currentFolderProvisions[currentAccount.id] ?: emptyMap())
                .mapValues { e -> e.value.providedQuota ?: 0L}
            val result = prepareCompletionResponse(currentOperation, currentAccount, resultProvisions,
                resources, unitsEnsembles)
            return ReserveProvisionCompletionResult(null, result, null)
        }
        val updatedQuotas = mutableListOf<QuotaModel>()
        val updatedProvisions = mutableListOf<AccountsQuotasModel>()
        val oldQuotas = mutableMapOf<ResourceId, Long>()
        val newQuotas = mutableMapOf<ResourceId, Long>()
        val oldProvisions = mutableMapOf<ResourceId, Long>()
        val newProvisions = mutableMapOf<ResourceId, Long>()
        val actuallyAppliedProvisions = mutableMapOf<ResourceId, Long>()
        val oldProvisionsVersions = mutableMapOf<ResourceId, Long>()
        val resultProvisions = mutableMapOf<ResourceId, Long>()
        val oldBalances = mutableMapOf<ResourceId, Long>()
        val newBalances = mutableMapOf<ResourceId, Long>()
        val actuallyAppliedProvisionsVersions = mutableMapOf<ResourceId, Long>()
        val oldAccounts = mutableMapOf<AccountId, AccountHistoryModel>()
        val newAccounts = mutableMapOf<AccountId, AccountHistoryModel>()
        val updatedAccount = prepareUpdatedAccountAfterRefresh(refreshedAccount, currentAccount, provider,
            oldAccounts, newAccounts)
        prepareQuotaApplication(refreshedAccount.provisions, refreshedAccount.accountVersion.orElse(null),
            currentOperation, folderQuotas, currentFolderProvisions, currentAccount,
            currentFolder, provider, updatedQuotas, updatedProvisions, oldQuotas, newQuotas,
            oldProvisions, newProvisions, actuallyAppliedProvisions, oldProvisionsVersions,
            actuallyAppliedProvisionsVersions, resultProvisions, oldBalances, newBalances, now)
        val opLog = prepareApplyCloseOperationLog(currentOperation, currentFolder,
            currentAccount, currentUser, oldQuotas, newQuotas, oldProvisions, newProvisions,
            actuallyAppliedProvisions, oldProvisionsVersions, actuallyAppliedProvisionsVersions,
            oldBalances, newBalances, oldAccounts, newAccounts, now)
        val updatedOperation = prepareOperationApplication(currentOperation, now, currentFolder)
        val updatedFolder = currentFolder.toBuilder()
            .setNextOpLogOrder(currentFolder.nextOpLogOrder + 1L).build()
        val result = prepareCompletionResponse(updatedOperation, currentAccount, resultProvisions,
            resources, unitsEnsembles)
        if (currentOperationInProgress != null) {
            meter({ operationsInProgressDao.deleteOneRetryable(txSession, WithTenant(Tenants.DEFAULT_TENANT_ID,
                currentOperationInProgress.key)).awaitSingleOrNull() }, "Provide reserve, remove operation in progress")
        }
        meter({ accountsQuotasOperationsDao.upsertOneRetryable(txSession, updatedOperation).awaitSingle() },
            "Provide reserve, upsert operation success after refresh")
        meter({ folderDao.upsertOneRetryable(txSession, updatedFolder).awaitSingle() },
            "Provide reserve, upsert updated folder on operation success after refresh")
        if (updatedQuotas.isNotEmpty()) {
            meter({ quotasDao.upsertAllRetryable(txSession, updatedQuotas).awaitSingleOrNull() },
                "Provide reserve, upsert updated quotas on operation success after refresh")
        }
        if (updatedProvisions.isNotEmpty()) {
            meter({ accountsQuotasDao.upsertAllRetryable(txSession, updatedProvisions).awaitSingleOrNull() },
                "Provide reserve, upsert updated provisions on operation success after refresh")
        }
        meter({ folderOperationLogDao.upsertOneRetryable(txSession, opLog).awaitSingle() },
            "Provide reserve, upsert folder history on operation success after refresh")
        if (updatedAccount != null) {
            meter({ accountsDao.upsertOneRetryable(txSession, updatedAccount).awaitSingle() },
                "Provide reserve, upsert account on operation success after refresh")
        }
        return ReserveProvisionCompletionResult(updatedOperation, result, opLog.id)
    }

    private fun prepareUpdatedAccountAfterRefresh(refreshedAccount: ValidatedReceivedAccount,
                                                  currentAccount: AccountModel,
                                                  provider: ProviderModel,
                                                  oldAccounts: MutableMap<AccountId, AccountHistoryModel>,
                                                  newAccounts: MutableMap<AccountId, AccountHistoryModel>): AccountModel? {
        // If account version is also a provision version
        val accountAndProvisionsVersionedTogether = provider.accountsSettings.isPerAccountVersionSupported
            && !provider.accountsSettings.isPerProvisionVersionSupported
        // Current account version in DB
        val currentLastReceivedAccountVersion = currentAccount.lastReceivedVersion.orElse(null)
        // Received account version
        val actualLastReceivedAccountVersion = refreshedAccount.accountVersion.orElse(null)
        if (!accountAndProvisionsVersionedTogether
            || Objects.equals(currentLastReceivedAccountVersion, actualLastReceivedAccountVersion)
        ) {
            // Account version does not include provisions or version is unchanged
            return null
        }
        if (currentLastReceivedAccountVersion != null && actualLastReceivedAccountVersion != null
            && currentLastReceivedAccountVersion > actualLastReceivedAccountVersion) {
            // Version should not be decreased
            logger.warn { "Version is decreased for account ${currentAccount.id} from $currentLastReceivedAccountVersion " +
                "to $actualLastReceivedAccountVersion" }
            return null
        }
        // Set new received account fields, prepare history
        val updatedAccountBuilder = AccountModel.Builder(currentAccount)
            .setLastReceivedVersion(actualLastReceivedAccountVersion)
            .setVersion(currentAccount.version + 1L)
        val oldAccountBuilder = AccountHistoryModel.builder()
            .lastReceivedVersion(currentLastReceivedAccountVersion)
            .version(currentAccount.version)
        val newAccountBuilder = AccountHistoryModel.builder()
            .lastReceivedVersion(actualLastReceivedAccountVersion)
            .version(currentAccount.version + 1L)
        if (provider.accountsSettings.isDisplayNameSupported
            && currentAccount.displayName != refreshedAccount.displayName
        ) {
            updatedAccountBuilder.setDisplayName(refreshedAccount.displayName.orElse(null))
            oldAccountBuilder.displayName(currentAccount.displayName.orElse(null))
            newAccountBuilder.displayName(refreshedAccount.displayName.orElse(null))
        }
        if (provider.accountsSettings.isKeySupported
            && currentAccount.outerAccountKeyInProvider != refreshedAccount.key
        ) {
            updatedAccountBuilder.setOuterAccountKeyInProvider(refreshedAccount.key.orElse(null))
            oldAccountBuilder.outerAccountKeyInProvider(currentAccount
                .outerAccountKeyInProvider.orElse(null))
            newAccountBuilder.outerAccountKeyInProvider(refreshedAccount.key.orElse(null))
        }
        val updatedAccount = updatedAccountBuilder.build()
        oldAccounts[currentAccount.id] = oldAccountBuilder.build()
        newAccounts[currentAccount.id] = newAccountBuilder.build()
        return updatedAccount
    }

    private fun checkOperationApplied(refreshedAccount: ValidatedReceivedAccount,
                                      postRequestContext: PostRequestContext,
                                      provider: ProviderModel): Boolean {
        return isProvisionChangesApplied(refreshedAccount, postRequestContext.currentOperation, provider,
            postRequestContext.currentAccount, postRequestContext.currentQuotas.folder,
            postRequestContext.currentQuotas.folderProvisions)
    }

    private suspend fun validateReceivedExternalAccount(receivedAccount: ReceivedAccount,
                                                        provider: ProviderModel,
                                                        accountsSpace: AccountSpaceModel?,
                                                        resourceDictionary: ResourceDictionary,
                                                        locale: Locale): Result<ValidatedReceivedAccount> {
        val receivedUserIds = getUsersFromReceivedAccount(receivedAccount)
        val receivedOperations = getOperationsFromReceivedAccount(receivedAccount)
        val receivedFolderId = getFolderFromReceivedAccount(receivedAccount)
        val receivedExternalAccountId = getExternalAccountIdFromReceivedAccount(receivedAccount)
        val knownUsers = loadUsers(receivedUserIds)
        val knownOperations = loadOperations(receivedOperations)
        val accountFolder = dbSessionRetryable(tableClient) {
            folderDao.getById(roStaleSingleRetryableCommit(), receivedFolderId, Tenants.DEFAULT_TENANT_ID)
                .awaitSingle().orElse(null)
        }
        val existingAccount = dbSessionRetryable(tableClient) {
            accountsDao.getAllByExternalId(rwSingleRetryableCommit(), WithTenant(Tenants.DEFAULT_TENANT_ID,
                AccountModel.ExternalId(provider.id, receivedExternalAccountId, accountsSpace?.id)))
                .awaitSingle().orElse(null)
        }
        return validateReceivedAccount(receivedAccount, knownUsers, knownOperations, accountFolder, existingAccount,
            provider, accountsSpace, resourceDictionary.resources, resourceDictionary.segmentations,
            resourceDictionary.segments, resourceDictionary.resourceTypes, resourceDictionary.unitsEnsembles,
            messages, locale)
    }

    private suspend fun handleRollbackOnConflict(originalProviderError: ProviderError,
                                                 operationId: OperationId,
                                                 folderId: FolderId,
                                                 accountId: AccountId,
                                                 provider: ProviderModel,
                                                 resourceDictionary: ResourceDictionary,
                                                 now: Instant,
                                                 currentUser: YaUserDetails,
                                                 locale: Locale): Result<UpdateProviderReserveProvisionsResponseDto> {
        val prefix = messages.getMessage("errors.accounts.quotas.out.of.sync.with.provider", null, locale)
        val errorMessage = Errors.flattenProviderErrorResponse(originalProviderError, prefix)
        val providerErrorCollection = Errors.providerErrorResponseToErrorCollection(originalProviderError)
        val operationErrors = OperationErrorCollections.builder()
            .addErrorCollection(Locales.ENGLISH, providerErrorCollection)
            .addErrorCollection(Locales.RUSSIAN, providerErrorCollection)
            .build()
        val operationErrorKind = Errors.providerErrorResponseToOperationErrorKind(originalProviderError)
        val rollbackResult = dbSessionRetryable(tableClient) {
            rwTxRetryable {
                val postRequestContext = preparePostRequestContext(txSession, operationId,
                    folderId, accountId, resourceDictionary)
                rollbackOperation(txSession, postRequestContext.currentOperation,
                    postRequestContext.currentOperationInProgress, postRequestContext.currentQuotas.folderQuotas,
                    postRequestContext.currentQuotas.folder, now, errorMessage, operationErrors,
                    operationErrorKind, currentUser)
            }
        }
        if (rollbackResult != null) {
            operationsObservabilityService.observeOperationFinished(rollbackResult.updatedOperation)
        }
        return Result.failure(providerErrorCollection)
    }

    private suspend fun completeOperation(
        txSession: YdbTxSession,
        context: PostRequestContext,
        response: ValidatedReceivedUpdatedProvision,
        provider: ProviderModel,
        resourceDictionary: ResourceDictionary,
        now: Instant,
        currentUser: YaUserDetails
    ): ReserveProvisionCompletionResult {
        return completeOperation(
            txSession, context.currentOperation, context.currentOperationInProgress,
            context.currentQuotas.folder, context.currentAccount, context.currentQuotas.folderQuotas,
            context.currentQuotas.folderProvisions, response, provider, resourceDictionary.resources,
            resourceDictionary.unitsEnsembles, now, currentUser
        )
    }

    suspend fun completeOperation(
        txSession: YdbTxSession,
        currentOperation: AccountsQuotasOperationsModel,
        currentOperationInProgress: OperationInProgressModel?,
        currentFolder: FolderModel,
        currentAccount: AccountModel,
        folderQuotas: Map<ResourceId, QuotaModel>,
        folderProvisions: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>,
        response: ValidatedReceivedUpdatedProvision,
        provider: ProviderModel,
        resources: Map<ResourceId, ResourceModel>,
        unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>,
        now: Instant,
        currentUser: YaUserDetails?
    ): ReserveProvisionCompletionResult {
        if (currentOperation.requestStatus.isPresent
            && currentOperation.requestStatus.get() != RequestStatus.WAITING
            && currentOperationInProgress == null
        ) {
            // Operation was already finished, do nothing
            val resultProvisions = (folderProvisions[currentAccount.id] ?: emptyMap())
                .mapValues { e -> e.value.providedQuota ?: 0L}
            val result = prepareCompletionResponse(currentOperation, currentAccount, resultProvisions,
                resources, unitsEnsembles)
            return ReserveProvisionCompletionResult(null, result, null)
        }
        val updatedQuotas = mutableListOf<QuotaModel>()
        val updatedProvisions = mutableListOf<AccountsQuotasModel>()
        val oldQuotas = mutableMapOf<ResourceId, Long>()
        val newQuotas = mutableMapOf<ResourceId, Long>()
        val oldProvisions = mutableMapOf<ResourceId, Long>()
        val newProvisions = mutableMapOf<ResourceId, Long>()
        val actuallyAppliedProvisions = mutableMapOf<ResourceId, Long>()
        val oldProvisionsVersions = mutableMapOf<ResourceId, Long>()
        val resultProvisions = mutableMapOf<ResourceId, Long>()
        val oldBalances = mutableMapOf<ResourceId, Long>()
        val newBalances = mutableMapOf<ResourceId, Long>()
        val actuallyAppliedProvisionsVersions = mutableMapOf<ResourceId, Long>()
        val oldAccounts = mutableMapOf<AccountId, AccountHistoryModel>()
        val newAccounts = mutableMapOf<AccountId, AccountHistoryModel>()
        val updatedAccount = prepareUpdatedAccount(response, currentAccount, provider, oldAccounts, newAccounts)
        prepareQuotaApplication(response.provisions, response.accountVersion.orElse(null),
            currentOperation, folderQuotas, folderProvisions,
            currentAccount, currentFolder, provider, updatedQuotas, updatedProvisions,
            oldQuotas, newQuotas, oldProvisions, newProvisions, actuallyAppliedProvisions, oldProvisionsVersions,
            actuallyAppliedProvisionsVersions, resultProvisions, oldBalances, newBalances, now)
        val opLog = prepareApplyCloseOperationLog(currentOperation, currentFolder, currentAccount, currentUser,
            oldQuotas, newQuotas, oldProvisions, newProvisions,
            actuallyAppliedProvisions, oldProvisionsVersions, actuallyAppliedProvisionsVersions,
            oldBalances, newBalances, oldAccounts, newAccounts, now)
        val updatedOperation = prepareOperationApplication(currentOperation, now, currentFolder)
        val updatedFolder = currentFolder.toBuilder()
            .setNextOpLogOrder(currentFolder.nextOpLogOrder + 1L).build()
        val result = prepareCompletionResponse(updatedOperation, currentAccount, resultProvisions,
            resources, unitsEnsembles)
        if (currentOperationInProgress != null) {
            meter({ operationsInProgressDao.deleteOneRetryable(txSession, WithTenant(Tenants.DEFAULT_TENANT_ID,
                currentOperationInProgress.key)).awaitSingleOrNull() }, "Provide reserve, remove operation in progress")
        }
        meter({ accountsQuotasOperationsDao.upsertOneRetryable(txSession, updatedOperation).awaitSingle() },
            "Provide reserve, upsert operation success")
        meter({ folderDao.upsertOneRetryable(txSession, updatedFolder).awaitSingle() },
            "Provide reserve, upsert updated folder on operation success")
        if (updatedQuotas.isNotEmpty()) {
            meter({ quotasDao.upsertAllRetryable(txSession, updatedQuotas).awaitSingleOrNull() },
                "Provide reserve, upsert updated quotas on operation success")
        }
        if (updatedProvisions.isNotEmpty()) {
            meter({ accountsQuotasDao.upsertAllRetryable(txSession, updatedProvisions).awaitSingleOrNull() },
                "Provide reserve, upsert updated provisions on operation success")
        }
        meter({ folderOperationLogDao.upsertOneRetryable(txSession, opLog).awaitSingle() },
            "Provide reserve, upsert folder history on operation success")
        if (updatedAccount != null) {
            meter({ accountsDao.upsertOneRetryable(txSession, updatedAccount).awaitSingle() },
                "Provide reserve, upsert account on operation success")
        }
        return ReserveProvisionCompletionResult(updatedOperation, result, opLog.id)
    }

    private fun prepareCompletionResponse(operation: AccountsQuotasOperationsModel,
                                          account: AccountModel,
                                          newProvisions: Map<ResourceId, Long>,
                                          resources: Map<ResourceId, ResourceModel>,
                                          unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>
    ): UpdateProviderReserveProvisionsResponseDto {
        val provisions: List<ProviderReserveProvisionResponseValueDto> = newProvisions.map { entry ->
            val resource = resources[entry.key]!!
            val unitsEnsemble = unitsEnsembles[resource.unitsEnsembleId]!!
            val converted = Units.convertToApi(entry.value, resource, unitsEnsemble)
            ProviderReserveProvisionResponseValueDto(entry.key, converted.t1.toLong(), converted.t2.key)
        }
        val result = UpdateProviderReserveProvisionsResultDto(account.id, provisions)
        return UpdateProviderReserveProvisionsResponseDto(UpdateProviderReserveProvisionsStatusDto.SUCCESS, result,
            operation.operationId)
    }

    private fun prepareUpdatedAccount(response: ValidatedReceivedUpdatedProvision,
                                      currentAccount: AccountModel,
                                      provider: ProviderModel,
                                      oldAccounts: MutableMap<AccountId, AccountHistoryModel>,
                                      newAccounts: MutableMap<AccountId, AccountHistoryModel>): AccountModel? {
        // If account version is also a provision version
        val accountAndProvisionsVersionedTogether = provider.accountsSettings.isPerAccountVersionSupported
            && !provider.accountsSettings.isPerProvisionVersionSupported
        // Current account version in DB
        val currentLastReceivedAccountVersion = currentAccount.lastReceivedVersion.orElse(null)
        // Received account version
        val actualLastReceivedAccountVersion = response.accountVersion.orElse(null)
        if (!accountAndProvisionsVersionedTogether
            || Objects.equals(currentLastReceivedAccountVersion, actualLastReceivedAccountVersion)
        ) {
            // Account version does not include provisions or version is unchanged
            return null
        }
        if (currentLastReceivedAccountVersion != null && actualLastReceivedAccountVersion != null
            && currentLastReceivedAccountVersion > actualLastReceivedAccountVersion) {
            // Version should not be decreased
            logger.warn { "Version is decreased for account ${currentAccount.id} from $currentLastReceivedAccountVersion " +
                "to $actualLastReceivedAccountVersion" }
            return null
        }
        // Set new received account version, increment internal version, prepare history
        val updatedAccount = AccountModel.Builder(currentAccount)
            .setLastReceivedVersion(actualLastReceivedAccountVersion)
            .setVersion(currentAccount.version + 1L)
            .build()
        oldAccounts[currentAccount.id] = AccountHistoryModel.builder()
            .lastReceivedVersion(currentLastReceivedAccountVersion)
            .version(currentAccount.version)
            .build()
        newAccounts[currentAccount.id] = AccountHistoryModel.builder()
            .lastReceivedVersion(actualLastReceivedAccountVersion)
            .version(updatedAccount.version)
            .build()
        return updatedAccount
    }

    private fun prepareQuotaApplication(responseProvisions: List<ValidatedReceivedProvision>,
                                        responseAccountVersion: Long?,
                                        operation: AccountsQuotasOperationsModel,
                                        folderQuotas: Map<ResourceId, QuotaModel>,
                                        currentFolderProvisions: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>,
                                        account: AccountModel,
                                        folder: FolderModel,
                                        provider: ProviderModel,
                                        updatedQuotas: MutableList<QuotaModel>,
                                        updatedProvisions: MutableList<AccountsQuotasModel>,
                                        oldQuotas: MutableMap<ResourceId, Long>,
                                        newQuotas: MutableMap<ResourceId, Long>,
                                        oldProvisions: MutableMap<ResourceId, Long>,
                                        newProvisions: MutableMap<ResourceId, Long>,
                                        actuallyAppliedProvisions: MutableMap<ResourceId, Long>,
                                        oldProvisionsVersions: MutableMap<ResourceId, Long>,
                                        actuallyAppliedProvisionsVersions: MutableMap<ResourceId, Long>,
                                        resultProvisions: MutableMap<ResourceId, Long>,
                                        oldBalances: MutableMap<ResourceId, Long>,
                                        newBalances: MutableMap<ResourceId, Long>,
                                        now: Instant) {
        // If provisions are versioned
        val provisionsVersionedSeparately: Boolean = provider.accountsSettings.isPerProvisionVersionSupported
        // If account version is also a provision version
        val accountAndProvisionsVersionedTogether = provider.accountsSettings.isPerAccountVersionSupported
            && !provider.accountsSettings.isPerProvisionVersionSupported
        // Current account version in DB
        val currentLastReceivedAccountVersion = account.lastReceivedVersion.orElse(null)
        // Received account version
        // Account version is also a provision version and received version is stale
        val staleAccountProvisions = accountAndProvisionsVersionedTogether && currentLastReceivedAccountVersion != null
            && responseAccountVersion != null && currentLastReceivedAccountVersion > responseAccountVersion
        // Updated frozen quotas to save
        val updatedFrozenQuotas = mutableMapOf<ResourceId, Long>()
        // Amounts added to quotas before request, only if frozen quotas are as expected
        // If frozen quotas are unexpected then we won't unfreeze them and ignore that increase
        val existingQuotaIncrease = mutableMapOf<ResourceId, Long>()
        operation.requestedChanges.frozenProvisions.ifPresent { it.forEach { frozenProvision ->
            val currentFrozenQuota = folderQuotas[frozenProvision.resourceId]?.frozenQuota ?: 0L
            if (currentFrozenQuota >= frozenProvision.amount) {
                val updatedFrozenQuotaO = Units.subtract(currentFrozenQuota, frozenProvision.amount)
                if (updatedFrozenQuotaO.isPresent) {
                    updatedFrozenQuotas[frozenProvision.resourceId] = updatedFrozenQuotaO.get()
                    existingQuotaIncrease[frozenProvision.resourceId] = frozenProvision.amount
                } else {
                    // Underflow while unfreezing, something went wrong
                    logger.error { "Frozen quota underflow, current frozen quota: ${currentFrozenQuota}, quota to " +
                        "unfreeze: ${frozenProvision.amount} in ${folder.id} for ${frozenProvision.resourceId}" }
                }
            } else {
                // Not enough quota to unfreeze, something went wrong
                logger.error { "Frozen quota $currentFrozenQuota is less then quota to " +
                    "unfreeze ${frozenProvision.amount} in ${folder.id} for ${frozenProvision.resourceId}" }
            }
        }}
        // Updated provisions
        val updatedProvidedAmounts = mutableMapOf<ResourceId, Long>()
        // Updated allocations
        val updatedAllocatedAmounts = mutableMapOf<ResourceId, Long>()
        // Updated provision versions
        val updatedProvisionVersions = mutableMapOf<ResourceId, Long>()
        // Deltas to apply to quotas
        val targetQuotaDeltas = mutableMapOf<ResourceId, BigInteger>()
        // Unless previous increase is accounted for - negate it
        existingQuotaIncrease.forEach { (resourceId, value) -> targetQuotaDeltas[resourceId] = BigInteger.valueOf(-value) }
        // Expected new provision values
        val expectedProvisions = operation.requestedChanges.updatedProvisions.orElse(emptyList())
            .associateBy { it.resourceId }
        // Process provisions in provider response
        responseProvisions.forEach { responseProvision ->
            val resourceId = responseProvision.resource.id
            // Expected new provision value
            val expectedProvision = expectedProvisions[resourceId]
            // Amounts added to quota before request, or zero
            val currentQuotaIncrease = existingQuotaIncrease[resourceId] ?: 0L
            // Actual provision version in provider
            val actualVersion = responseProvision.quotaVersion.orElse(null)
            // Current provision version, if any
            val currentVersion = (currentFolderProvisions[account.id] ?: emptyMap())[resourceId]
                ?.lastReceivedProvisionVersion?.orElse(null)
            // Provisions are versioned and received version is stale
            val staleProvision = provisionsVersionedSeparately && currentVersion != null && actualVersion != null
                && currentVersion > actualVersion
            if (staleProvision) {
                logger.warn { "Version is decreased for provision of $resourceId in account ${account.id} from " +
                    "$currentVersion to $actualVersion" }
            }
            val receivedTimestamp = responseProvision.lastUpdate.orElse(null)?.timestamp?.orElse(null)
            val currentTimestamp = (currentFolderProvisions[account.id] ?: emptyMap())[resourceId]?.lastProvisionUpdate
            if (receivedTimestamp != null && currentTimestamp != null && receivedTimestamp.isBefore(currentTimestamp)) {
                logger.warn { "Timestamp for received provision is earlier then currently saved timestamp, " +
                    "resource is ${resourceId}, account id ${account.id}, received timestamp is ${receivedTimestamp}, " +
                    "current timestamp is $currentTimestamp" }
            }
            val receivedOperationId = responseProvision.lastUpdate.orElse(null)?.operationId?.orElse(null)
            if (receivedOperationId != null && provider.accountsSettings.isPerProvisionLastUpdateSupported
                && receivedOperationId != operation.operationId) {
                logger.warn { "Received operation id does not match current operation id, " +
                    "resource is ${resourceId}, account id ${account.id}, received id is ${receivedOperationId}, " +
                    "current id is ${operation.operationId}" }
            }
            // If expectedProvision is null then it's new value was not sent in update and was expected to be zero.
            // Skip such resources, we won't update their provisions, they will be updated during background sync.
            // Also skip if received provision is stale
            if (expectedProvision != null && !staleAccountProvisions && !staleProvision) {
                // Current provision amount in DB
                val currentProvision = (currentFolderProvisions[account.id] ?: emptyMap())[resourceId]?.providedQuota ?: 0L
                // Actual provision in provider
                val actualProvision = responseProvision.providedAmount
                // Actual allocation in provider
                val actualAllocation = responseProvision.allocatedAmount
                // Remember to save new provision value
                updatedProvidedAmounts[resourceId] = actualProvision
                // Remember to save new allocation value
                updatedAllocatedAmounts[resourceId] = actualAllocation
                if (actualVersion != null) {
                    // Remember to save new provision version, if any
                    updatedProvisionVersions[resourceId] = actualVersion
                }
                if (currentProvision > actualProvision) {
                    // Provision is decreased
                    // Decrease quota, take into account unfrozen quota
                    // How much provision was decreased?
                    val provisionDecrease = BigInteger.valueOf(currentProvision)
                        .subtract(BigInteger.valueOf(actualProvision))
                    val currentQuotaIncreaseBigInteger = BigInteger.valueOf(currentQuotaIncrease)
                    val remainingQuotaDecrease = provisionDecrease.add(currentQuotaIncreaseBigInteger)
                    targetQuotaDeltas[resourceId] = remainingQuotaDecrease.negate()
                } else if (currentProvision < actualProvision) {
                    // Provision is increased
                    // Increase quota if unfrozen quota is not enough
                    // How much provision was increased?
                    val provisionIncrease = BigInteger.valueOf(actualProvision)
                        .subtract(BigInteger.valueOf(currentProvision))
                    val currentQuotaIncreaseBigInteger = BigInteger.valueOf(currentQuotaIncrease)
                    if (provisionIncrease > currentQuotaIncreaseBigInteger) {
                        // Provision was increased more than expected, need to increment quota
                        val remainingQuotaIncrease = provisionIncrease.subtract(currentQuotaIncreaseBigInteger)
                        targetQuotaDeltas[resourceId] = remainingQuotaIncrease
                    } else if (provisionIncrease < currentQuotaIncreaseBigInteger) {
                        // Provision was increased less than expected, need to decrement quota
                        val remainingQuotaDecrease = currentQuotaIncreaseBigInteger.subtract(provisionIncrease)
                        targetQuotaDeltas[resourceId] = remainingQuotaDecrease.negate()
                    } else {
                        // Provision was increased exactly as expected, quota stays unchanged
                        targetQuotaDeltas[resourceId] = BigInteger.ZERO
                    }
                } else {
                    // Provision is unchanged
                    if (currentQuotaIncrease > 0) {
                        // Decrease quota if it was increased earlier and is unfrozen now
                        targetQuotaDeltas[resourceId] = BigInteger.valueOf(-currentQuotaIncrease)
                    }
                }
            }
        }
        // Prepare updated quotas
        val updatedQuotaAmounts = mutableMapOf<ResourceId, Long>()
        targetQuotaDeltas.forEach { (resourceId, targetDelta) ->
            // Try to apply prepared deltas
            val currentQuota = folderQuotas[resourceId]?.quota ?: 0L
            val currentQuotaBigDecimal = BigInteger.valueOf(currentQuota)
            val updatedQuota = currentQuotaBigDecimal.add(targetDelta)
            val actualUpdatedQuota = if (updatedQuota < BigInteger.ZERO) {
                // Not enough quota to take away, settle at zero quota
                logger.warn { "Quota of $resourceId in ${folder.id} is not enough to account for provision " +
                    "difference: $currentQuotaBigDecimal vs $targetDelta" }
                0L
            } else if (Units.longValue(updatedQuota).isEmpty) {
                // Overflow, can not change quota
                logger.error { "Overflow for quota of $resourceId in ${folder.id}, current " +
                    "value: ${currentQuotaBigDecimal}, delta: $targetDelta" }
                null
            } else {
                updatedQuota.toLong()
            }
            if (actualUpdatedQuota != null) {
                if (currentQuota != actualUpdatedQuota) {
                    updatedQuotaAmounts[resourceId] = actualUpdatedQuota
                }
            }
        }
        // Quotas with updates to find updated balance
        val quotasForBalance = mutableMapOf<ResourceId, Long>()
        // Frozen quotas with updates to find balance
        val frozenQuotasForBalance = mutableMapOf<ResourceId, Long>()
        quotasForBalance.putAll(updatedQuotaAmounts)
        frozenQuotasForBalance.putAll(updatedFrozenQuotas)
        folderQuotas.forEach { (resourceId, quota) ->
            if (!quotasForBalance.containsKey(resourceId)) {
                quotasForBalance[resourceId] = quota.quota ?: 0L
            }
            if (!frozenQuotasForBalance.containsKey(resourceId)) {
                frozenQuotasForBalance[resourceId] = quota.frozenQuota
            }
        }
        // Provisions with updates to find balance
        val provisionsForBalance = mutableMapOf<AccountId, MutableMap<ResourceId, Long>>()
        provisionsForBalance.computeIfAbsent(account.id) { mutableMapOf() }.putAll(updatedProvidedAmounts)
        currentFolderProvisions.forEach { (accountId, accountProvisions) ->
            val accountProvisionsForBalance = provisionsForBalance.computeIfAbsent(accountId) { mutableMapOf() }
            accountProvisions.forEach { (resourceId, provision) ->
                if (!accountProvisionsForBalance.containsKey(resourceId)) {
                    accountProvisionsForBalance[resourceId] = provision.providedQuota ?: 0L
                }
            }
        }
        // Find new balances
        val updatedBalances = mutableMapOf<ResourceId, BigInteger>()
        quotasForBalance.forEach { (resourceId, quota) ->
            updatedBalances[resourceId] = BigInteger.valueOf(quota)
        }
        frozenQuotasForBalance.forEach { (resourceId, frozenQuota) ->
            updatedBalances[resourceId] = (updatedBalances[resourceId] ?: BigInteger.ZERO)
                .subtract(BigInteger.valueOf(frozenQuota))
        }
        provisionsForBalance.forEach { (_, accountProvisions) ->
            accountProvisions.forEach { (resourceId, provision) ->
                updatedBalances[resourceId] = (updatedBalances[resourceId] ?: BigInteger.ZERO)
                    .subtract(BigInteger.valueOf(provision))
            }
        }
        // Check balances for underflow
        val actualUpdatedBalances = mutableMapOf<ResourceId, Long>()
        updatedBalances.forEach { (resourceId, balance) ->
            val balanceO = Units.longValue(balance)
            if (balanceO.isPresent) {
                actualUpdatedBalances[resourceId] = balanceO.asLong
            } else {
                logger.error { "Underflow for balance of $resourceId in ${folder.id}: $balance" }
            }
        }
        val resourcesToUpdateQuotas = mutableSetOf<ResourceId>()
        // Remember quota update history and data
        updatedQuotaAmounts.forEach { (resourceId, updatedQuota) ->
            val currentQuota = folderQuotas[resourceId]?.quota ?: 0L
            // Quotas won't be updated if balance underflows
            val hasBalance = actualUpdatedBalances[resourceId] != null
            if (hasBalance) {
                oldQuotas[resourceId] = currentQuota
                newQuotas[resourceId] = updatedQuota
                resourcesToUpdateQuotas.add(resourceId)
            }
        }
        // Remember balance history
        actualUpdatedBalances.forEach { (resourceId, updatedBalance) ->
            val currentBalance = folderQuotas[resourceId]?.balance ?: 0L
            if (currentBalance != updatedBalance) {
                oldBalances[resourceId] = currentBalance
                newBalances[resourceId] = updatedBalance
                resourcesToUpdateQuotas.add(resourceId)
            }
        }
        // Which frozen quotas require update?
        updatedFrozenQuotas.forEach { (resourceId, _) ->
            // Quotas won't be updated if balance underflows
            val hasBalance = actualUpdatedBalances[resourceId] != null
            if (hasBalance) {
                resourcesToUpdateQuotas.add(resourceId)
            }
        }
        // Prepare updated quotas
        resourcesToUpdateQuotas.forEach { resourceId ->
            val currentQuota = folderQuotas[resourceId]
            val updatedQuota = updatedQuotaAmounts[resourceId]
            val updatedBalance = actualUpdatedBalances[resourceId]
            val updatedFrozenQuota = updatedFrozenQuotas[resourceId]
            if (currentQuota != null) {
                val quotaBuilder = QuotaModel.builder(currentQuota)
                if (updatedQuota != null) {
                    quotaBuilder.quota(updatedQuota)
                }
                if (updatedBalance != null) {
                    quotaBuilder.balance(updatedBalance)
                }
                if (updatedFrozenQuota != null) {
                    quotaBuilder.frozenQuota(updatedFrozenQuota)
                }
                updatedQuotas.add(quotaBuilder.build())
            } else {
                updatedQuotas.add(QuotaModel.builder()
                    .tenantId(folder.tenantId)
                    .folderId(folder.id)
                    .providerId(provider.id)
                    .resourceId(resourceId)
                    .quota(updatedQuota ?: 0L)
                    .balance(updatedBalance ?: 0L)
                    .frozenQuota(updatedFrozenQuota ?: 0L)
                    .build())
            }
        }
        // Remember actual new provisions for history and data
        updatedProvidedAmounts.forEach { (resourceId, updatedProvision) ->
            val currentProvisionModel = (currentFolderProvisions[account.id] ?: emptyMap())[resourceId]
            // Current provision amount in DB
            val currentProvision = currentProvisionModel?.providedQuota ?: 0L
            // Updated provision version, if any
            val updatedVersion = updatedProvisionVersions[resourceId]
            // Current provision version, if any
            val currentVersion = currentProvisionModel?.lastReceivedProvisionVersion?.orElse(null)
            // Provisions won't be updated if balance underflows
            val hasBalance = actualUpdatedBalances[resourceId] != null
            if (((updatedProvision != currentProvision) || (provisionsVersionedSeparately
                    && !Objects.equals(currentVersion, updatedVersion))) && hasBalance) {
                // Only if provision or version was actually changed
                actuallyAppliedProvisions[resourceId] = updatedProvision
                oldProvisions[resourceId] = currentProvision
                if (provisionsVersionedSeparately && !Objects.equals(currentVersion, updatedVersion)) {
                    // Only if version was actually changed
                    if (currentVersion != null) {
                        oldProvisionsVersions[resourceId] = currentVersion
                    }
                    if (updatedVersion != null) {
                        actuallyAppliedProvisionsVersions[resourceId] = updatedVersion
                    }
                }
                // Prepare provision to upsert
                if (currentProvisionModel != null) {
                    // Update existing provision
                    val updatedProvisionModelBuilder = AccountsQuotasModel.Builder(currentProvisionModel)
                    updatedProvisionModelBuilder.setProvidedQuota(updatedProvision)
                    updatedProvisionModelBuilder.setAllocatedQuota(updatedAllocatedAmounts[resourceId] ?: 0L)
                    if (provisionsVersionedSeparately) {
                        updatedProvisionModelBuilder.setLastReceivedProvisionVersion(updatedVersion)
                    }
                    updatedProvisionModelBuilder.setLatestSuccessfulProvisionOperationId(operation.operationId)
                    updatedProvisionModelBuilder.setLastProvisionUpdate(now)
                    updatedProvisions.add(updatedProvisionModelBuilder.build())
                } else {
                    // Create new provision
                    val newProvisionModelBuilder = AccountsQuotasModel.Builder()
                    newProvisionModelBuilder.setTenantId(Tenants.DEFAULT_TENANT_ID)
                    newProvisionModelBuilder.setAccountId(account.id)
                    newProvisionModelBuilder.setResourceId(resourceId)
                    newProvisionModelBuilder.setFolderId(folder.id)
                    newProvisionModelBuilder.setProviderId(provider.id)
                    newProvisionModelBuilder.setFrozenProvidedQuota(0L)
                    newProvisionModelBuilder.setProvidedQuota(updatedProvision)
                    newProvisionModelBuilder.setAllocatedQuota(updatedAllocatedAmounts[resourceId] ?: 0L)
                    if (provisionsVersionedSeparately) {
                        newProvisionModelBuilder.setLastReceivedProvisionVersion(updatedVersion)
                    }
                    newProvisionModelBuilder.setLatestSuccessfulProvisionOperationId(operation.operationId)
                    newProvisionModelBuilder.setLastProvisionUpdate(now)
                    updatedProvisions.add(newProvisionModelBuilder.build())
                }
            }
        }
        // Remember expected provisions for history
        expectedProvisions.forEach { (resourceId, provision) ->
            // Current provision amount in DB
            val currentProvision = (currentFolderProvisions[account.id] ?: emptyMap())[resourceId]?.providedQuota ?: 0L
            val currentVersion = (currentFolderProvisions[account.id] ?: emptyMap())[resourceId]
                ?.lastReceivedProvisionVersion?.orElse(null)
            if (provision.amount != currentProvision) {
                // Only if change was expected
                newProvisions[resourceId] = provision.amount
                if (!oldProvisions.containsKey(resourceId)) {
                    // If not yet added for actual provision change
                    oldProvisions[resourceId] = currentProvision
                }
                if (provisionsVersionedSeparately && currentVersion != null
                    && !oldProvisionsVersions.containsKey(resourceId)
                ) {
                    // If not yet added for actual provision change
                    oldProvisionsVersions[resourceId] = currentVersion
                }
            }
        }
        // Prepare updated provisions for response
        updatedProvidedAmounts.forEach { (resourceId, updatedProvision) ->
            // Provisions won't be updated if balance underflows
            val hasBalance = actualUpdatedBalances[resourceId] != null
            if (hasBalance) {
                resultProvisions[resourceId] = updatedProvision
            }
        }
        // Prepare unchanged provisions for response
        (currentFolderProvisions[account.id] ?: emptyMap()).forEach { (resourceId, provision) ->
            if (!resultProvisions.containsKey(resourceId)) {
                resultProvisions[resourceId] = provision.providedQuota ?: 0L
            }
        }
    }

    private fun prepareApplyCloseOperationLog(operation: AccountsQuotasOperationsModel,
                                              folder: FolderModel,
                                              account: AccountModel,
                                              currentUser: YaUserDetails?,
                                              oldQuotas: Map<ResourceId, Long>,
                                              newQuotas: Map<ResourceId, Long>,
                                              oldProvisions: Map<ResourceId, Long>,
                                              newProvisions: Map<ResourceId, Long>,
                                              actuallyAppliedProvisions: Map<ResourceId, Long>,
                                              oldProvisionsVersions: Map<ResourceId, Long>,
                                              actuallyAppliedProvisionsVersions: Map<ResourceId, Long>,
                                              oldBalances: Map<ResourceId, Long>,
                                              newBalances: Map<ResourceId, Long>,
                                              oldAccounts: MutableMap<AccountId, AccountHistoryModel>,
                                              newAccounts: MutableMap<AccountId, AccountHistoryModel>,
                                              now: Instant): FolderOperationLogModel {
        val user = currentUser?.user?.orElse(null)
        return FolderOperationLogModel.builder()
            .setTenantId(Tenants.DEFAULT_TENANT_ID)
            .setFolderId(folder.id)
            .setOperationDateTime(now)
            .setId(UUID.randomUUID().toString())
            .setProviderRequestId(operation.lastRequestId.orElseThrow())
            .setOperationType(FolderOperationType.PROVIDE_RESERVE)
            .setAuthorUserId(user?.id ?: operation.authorUserId)
            .setAuthorUserUid(user?.passportUid?.orElse(null) ?: operation.authorUserUid.orElseThrow())
            .setAuthorProviderId(null)
            .setSourceFolderOperationsLogId(null)
            .setDestinationFolderOperationsLogId(null)
            .setOldFolderFields(null)
            .setOldQuotas(QuotasByResource(oldQuotas))
            .setOldBalance(QuotasByResource(oldBalances))
            .setOldProvisions(QuotasByAccount(mapOf(account.id to ProvisionsByResource(oldProvisions
                .mapValues { ProvisionHistoryModel(it.value, oldProvisionsVersions[it.key]) }))))
            .setOldAccounts(if (oldAccounts.isNotEmpty()) {
                AccountsHistoryModel(oldAccounts)
            } else {
                null
            })
            .setNewFolderFields(null)
            .setNewQuotas(QuotasByResource(newQuotas))
            .setNewBalance(QuotasByResource(newBalances))
            .setNewProvisions(QuotasByAccount(mapOf(account.id to ProvisionsByResource(newProvisions
                .mapValues { ProvisionHistoryModel(it.value, null) }))))
            .setActuallyAppliedProvisions(QuotasByAccount(mapOf(account.id to ProvisionsByResource(actuallyAppliedProvisions
                .mapValues { ProvisionHistoryModel(it.value, actuallyAppliedProvisionsVersions[it.key]) }))))
            .setNewAccounts(if (newAccounts.isNotEmpty()) {
                AccountsHistoryModel(newAccounts)
            } else {
                null
            })
            .setAccountsQuotasOperationsId(operation.operationId)
            .setQuotasDemandsId(null)
            .setOperationPhase(OperationPhase.CLOSE)
            .setCommentId(null)
            .setOrder(folder.nextOpLogOrder)
            .setDeliveryMeta(null)
            .setTransferMeta(null)
            .build()
    }

    private fun prepareOperationApplication(operation: AccountsQuotasOperationsModel,
                                            now: Instant,
                                            folder: FolderModel): AccountsQuotasOperationsModel {
        return AccountsQuotasOperationsModel.Builder(operation)
            .setUpdateDateTime(now)
            .setRequestStatus(RequestStatus.OK)
            .setOrders(OperationOrdersModel.builder(operation.orders)
                .closeOrder(folder.nextOpLogOrder)
                .build())
            .build()
    }

    private suspend fun onPostRequestValidationError(errors: ErrorCollection,
                                                     reqId: String?,
                                                     request: UpdateProvisionRequestDto,
                                                     providerResponseDto: UpdateProvisionResponseDto,
                                                     operationId: OperationId,
                                                     folderId: FolderId,
                                                     accountId: AccountId,
                                                     provider: ProviderModel,
                                                     resourceDictionary: ResourceDictionary,
                                                     now: Instant,
                                                     currentUser: YaUserDetails,
                                                     locale: Locale
    ): Result<UpdateProviderReserveProvisionsResponseDto> {
        logger.error {
            "Unexpected error during reserve provision for " +
                "provider ${provider.key} (requestId = ${reqId}): ${errors}. " +
                "Request = ${request}. Response = ${providerResponseDto}."
        }
        val errorMessage = messages.getMessage("errors.unexpected.provider.response",
            null, locale)
        val operationErrors = OperationErrorCollections.builder()
            .addErrorCollection(Locales.ENGLISH, ErrorCollection.builder().addError(TypedError
                .unavailable(messages.getMessage("errors.unexpected.provider.response",
                    null, Locales.ENGLISH))).build())
            .addErrorCollection(Locales.RUSSIAN, ErrorCollection.builder().addError(TypedError
                .unavailable(messages.getMessage("errors.unexpected.provider.response",
                    null, Locales.RUSSIAN))).build())
            .build()
        val rollbackResult = dbSessionRetryable(tableClient) {
            rwTxRetryable {
                val postRequestContext = preparePostRequestContext(txSession, operationId,
                    folderId, accountId, resourceDictionary)
                rollbackOperation(txSession, postRequestContext.currentOperation,
                    postRequestContext.currentOperationInProgress, postRequestContext.currentQuotas.folderQuotas,
                    postRequestContext.currentQuotas.folder, now, errorMessage, operationErrors,
                    OperationErrorKind.UNAVAILABLE, currentUser)
            }
        }
        if (rollbackResult != null) {
            operationsObservabilityService.observeOperationFinished(rollbackResult.updatedOperation)
        }
        val resultErrors = ErrorCollection.builder()
            .addError(TypedError.unavailable(errorMessage)).build()
        return Result.failure(resultErrors)
    }

    private suspend fun onPostRequestUnexpectedError(ex: Throwable,
                                                     request: UpdateProvisionRequestDto,
                                                     operationId: OperationId,
                                                     folderId: FolderId,
                                                     accountId: AccountId,
                                                     provider: ProviderModel,
                                                     resourceDictionary: ResourceDictionary,
                                                     now: Instant,
                                                     currentUser: YaUserDetails,
                                                     locale: Locale
    ): Result<UpdateProviderReserveProvisionsResponseDto> {
        logger.error(ex) {
            "Unexpected error during reserve provision for provider ${provider.key}. " +
                "Request = ${request}."
        }
        val errorMessage = messages.getMessage("errors.unexpected.provider.communication.failure",
            null, locale)
        val operationErrors = OperationErrorCollections.builder()
            .addErrorCollection(Locales.ENGLISH, ErrorCollection.builder().addError(TypedError
                .unavailable(messages.getMessage("errors.unexpected.provider.communication.failure",
                    null, Locales.ENGLISH))).build())
            .addErrorCollection(Locales.RUSSIAN, ErrorCollection.builder().addError(TypedError
                .unavailable(messages.getMessage("errors.unexpected.provider.communication.failure",
                    null, Locales.RUSSIAN))).build())
            .build()
        val rollbackResult = dbSessionRetryable(tableClient) {
            rwTxRetryable {
                val postRequestContext = preparePostRequestContext(txSession, operationId,
                    folderId, accountId, resourceDictionary)
                rollbackOperation(txSession, postRequestContext.currentOperation,
                    postRequestContext.currentOperationInProgress, postRequestContext.currentQuotas.folderQuotas,
                    postRequestContext.currentQuotas.folder, now, errorMessage, operationErrors,
                    OperationErrorKind.UNAVAILABLE, currentUser)
            }
        }
        if (rollbackResult != null) {
            operationsObservabilityService.observeOperationFinished(rollbackResult.updatedOperation)
        }
        val errors = ErrorCollection.builder().addError(TypedError.unavailable(errorMessage)).build()
        return Result.failure(errors)
    }

    private suspend fun onPreRequestValidationError(errors: ErrorCollection,
                                                    request: UpdateProvisionRequestDto,
                                                    operationId: OperationId,
                                                    folderId: FolderId,
                                                    accountId: AccountId,
                                                    provider: ProviderModel,
                                                    resourceDictionary: ResourceDictionary,
                                                    now: Instant,
                                                    currentUser: YaUserDetails
    ): Result<UpdateProviderReserveProvisionsResponseDto> {
        logger.error {
            "Unexpected error during reserve provision for provider ${provider.key}: ${errors}. " +
                "Request = ${request}."
        }
        val errorMessage = Errors.flattenErrors(errors)
        val operationErrors = OperationErrorCollections.builder()
            // Provider integration does not support multi-locale errors yet
            .addErrorCollection(Locales.ENGLISH, errors)
            .addErrorCollection(Locales.RUSSIAN, errors)
            .build()
        val rollbackResult = dbSessionRetryable(tableClient) {
            rwTxRetryable {
                val postRequestContext = preparePostRequestContext(txSession, operationId,
                    folderId, accountId, resourceDictionary)
                rollbackOperation(txSession, postRequestContext.currentOperation,
                    postRequestContext.currentOperationInProgress, postRequestContext.currentQuotas.folderQuotas,
                    postRequestContext.currentQuotas.folder, now, errorMessage, operationErrors,
                    OperationErrorKind.INVALID_ARGUMENT, currentUser)
            }
        }
        if (rollbackResult != null) {
            operationsObservabilityService.observeOperationFinished(rollbackResult.updatedOperation)
        }
        return Result.failure(errors)
    }

    private suspend fun loadUsers(userIds: Set<ReceivedUserId>): List<UserModel> {
        if (userIds.isEmpty()) {
            return emptyList()
        }
        val uidSet = userIds.mapNotNull { it.passportUid.orElse(null) }.toSet()
        val loginSet = userIds.mapNotNull { it.staffLogin.orElse(null) }.toSet()
        if (uidSet.isEmpty() && loginSet.isEmpty()) {
            return emptyList()
        }
        return dbSessionRetryable(tableClient) {
            usersDao.getByExternalIds(roStaleSingleRetryableCommit(),
                uidSet.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) },
                loginSet.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) },
                emptyList()).awaitSingle()
        }!!
    }

    private suspend fun loadOperations(operationIds: Set<String>): List<AccountsQuotasOperationsModel> {
        if (operationIds.isEmpty()) {
            return emptyList()
        }
        return dbSessionRetryable(tableClient) {
            accountsQuotasOperationsDao.getAllByIds(rwSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID,
                operationIds.toList()).awaitSingle()
        }!!
    }

    private suspend fun preparePostRequestContext(txSession: YdbTxSession,
                                                  operationId: OperationId,
                                                  folderId: FolderId,
                                                  accountId: AccountId,
                                                  resourceDictionary: ResourceDictionary): PostRequestContext {
        val currentOperation: AccountsQuotasOperationsModel = meter({ accountsQuotasOperationsDao.getById(txSession,
            operationId, Tenants.DEFAULT_TENANT_ID).awaitSingle().orElseThrow() }, "Provide reserve, reload operation")
        val currentOperationInProgress: OperationInProgressModel? = meter({ operationsInProgressDao.getById(txSession,
            OperationInProgressModel.Key(operationId, folderId), Tenants.DEFAULT_TENANT_ID)
            .awaitSingle().orElse(null) }, "Provide reserve, reload operation in progress")
        val currentAccount = meter({ accountsDao.getById(txSession, accountId, Tenants.DEFAULT_TENANT_ID)
            .awaitSingle().orElse(null) }, "Provide reserve, reload account")
        val currentQuotas = meter({ loadCurrentQuotas(txSession, currentAccount, resourceDictionary.resources.keys) },
            "Provide reserve, reload current quotas")
        return PostRequestContext(currentOperation, currentOperationInProgress, currentAccount, currentQuotas)
    }

    suspend fun rollbackOperation(txSession: YdbTxSession,
                                  currentOperation: AccountsQuotasOperationsModel,
                                  currentOperationInProgress: OperationInProgressModel?,
                                  currentFolderQuotas: Map<ResourceId, QuotaModel>,
                                  currentFolder: FolderModel,
                                  now: Instant,
                                  errorMessage: String?,
                                  operationErrors: OperationErrorCollections,
                                  errorKind: OperationErrorKind,
                                  currentUser: YaUserDetails?): ReserveProvisionRollbackResult? {
        if (currentOperation.requestStatus.isPresent
            && currentOperation.requestStatus.get() != RequestStatus.WAITING
            && currentOperationInProgress == null
        ) {
            // Operation was already finished, do nothing
            return null
        }
        val updatedQuotas = mutableListOf<QuotaModel>()
        val oldQuotas = mutableMapOf<ResourceId, Long>()
        val newQuotas = mutableMapOf<ResourceId, Long>()
        prepareQuotaRollback(currentOperation, currentFolderQuotas, currentFolder, updatedQuotas, oldQuotas, newQuotas)
        val opLog = prepareRollbackCloseOperationLog(currentOperation, currentFolder,
            currentUser, oldQuotas, newQuotas, now)
        val updatedOperation = prepareOperationRollback(currentOperation, now, errorMessage, operationErrors,
            errorKind, currentFolder)
        val updatedFolder = currentFolder.toBuilder()
            .setNextOpLogOrder(currentFolder.nextOpLogOrder + 1L).build()
        if (currentOperationInProgress != null) {
            meter({ operationsInProgressDao.deleteOneRetryable(txSession, WithTenant(Tenants.DEFAULT_TENANT_ID,
                currentOperationInProgress.key)).awaitSingleOrNull() }, "Provide reserve, remove operation in progress")
        }
        meter({ accountsQuotasOperationsDao.upsertOneRetryable(txSession, updatedOperation).awaitSingle() },
            "Provide reserve, upsert operation failure")
        meter({ folderDao.upsertOneRetryable(txSession, updatedFolder).awaitSingle() },
            "Provide reserve, upsert updated folder on operation failure")
        if (updatedQuotas.isNotEmpty()) {
            meter({ quotasDao.upsertAllRetryable(txSession, updatedQuotas).awaitSingleOrNull() },
                "Provide reserve, upsert updated quotas on operation failure")
        }
        meter({ folderOperationLogDao.upsertOneRetryable(txSession, opLog).awaitSingle() },
            "Provide reserve, upsert folder history on operation failure")
        return ReserveProvisionRollbackResult(updatedOperation, updatedFolder, opLog, updatedQuotas)
    }

    private fun prepareQuotaRollback(operation: AccountsQuotasOperationsModel,
                                     folderQuotas: Map<ResourceId, QuotaModel>,
                                     folder: FolderModel,
                                     updatedQuotas: MutableList<QuotaModel>,
                                     oldQuotas: MutableMap<ResourceId, Long>,
                                     newQuotas: MutableMap<ResourceId, Long>) {
        val frozenProvisions = operation.requestedChanges.frozenProvisions.orElse(emptyList())
        frozenProvisions.forEach { frozenProvision ->
            val currentQuota = folderQuotas[frozenProvision.resourceId]?.quota ?: 0L
            val currentFrozen = folderQuotas[frozenProvision.resourceId]?.frozenQuota ?: 0L
            val currentBalance = folderQuotas[frozenProvision.resourceId]?.balance ?: 0L
            if (currentQuota >= frozenProvision.amount && currentFrozen >= frozenProvision.amount) {
                val updatedQuotaO = Units.subtract(currentQuota, frozenProvision.amount)
                val updatedFrozenO = Units.subtract(currentFrozen, frozenProvision.amount)
                if (updatedQuotaO.isPresent && updatedFrozenO.isPresent) {
                    updatedQuotas.add(QuotaModel.builder()
                        .tenantId(folder.tenantId)
                        .folderId(folder.id)
                        .providerId(operation.providerId)
                        .resourceId(frozenProvision.resourceId)
                        .quota(updatedQuotaO.get())
                        .balance(currentBalance)
                        .frozenQuota(updatedFrozenO.get())
                        .build())
                    oldQuotas[frozenProvision.resourceId] = currentQuota
                    newQuotas[frozenProvision.resourceId] = updatedQuotaO.get()
                } else {
                    logger.error { "Underflow while unfreezing ${frozenProvision.amount} " +
                        "for ${frozenProvision.resourceId} in folder ${folder.id}, current quota $currentQuota and " +
                        "frozen quota $currentFrozen" }
                }
            } else {
                logger.error { "Invalid current quota $currentQuota and frozen quota $currentFrozen in " +
                    "folder ${folder.id}, can not unfreeze ${frozenProvision.amount} for ${frozenProvision.resourceId}" }
            }
        }
    }

    private fun prepareOperationRollback(operation: AccountsQuotasOperationsModel,
                                         now: Instant,
                                         errorMessage: String?,
                                         operationErrors: OperationErrorCollections,
                                         errorKind: OperationErrorKind,
                                         folder: FolderModel): AccountsQuotasOperationsModel {
        return AccountsQuotasOperationsModel.Builder(operation)
            .setUpdateDateTime(now)
            .setRequestStatus(RequestStatus.ERROR)
            .setErrorMessage(errorMessage)
            .setFullErrorMessage(operationErrors)
            .setOrders(OperationOrdersModel.builder(operation.orders)
                .closeOrder(folder.nextOpLogOrder)
                .build())
            .setErrorKind(errorKind)
            .build()
    }

    private fun prepareRollbackCloseOperationLog(operation: AccountsQuotasOperationsModel,
                                                 folder: FolderModel,
                                                 currentUser: YaUserDetails?,
                                                 oldQuotas: Map<ResourceId, Long>,
                                                 newQuotas: Map<ResourceId, Long>,
                                                 now: Instant): FolderOperationLogModel {
        val user = currentUser?.user?.orElse(null)
        return FolderOperationLogModel.builder()
            .setTenantId(Tenants.DEFAULT_TENANT_ID)
            .setFolderId(folder.id)
            .setOperationDateTime(now)
            .setId(UUID.randomUUID().toString())
            .setProviderRequestId(operation.lastRequestId.orElseThrow())
            .setOperationType(FolderOperationType.PROVIDE_RESERVE)
            .setAuthorUserId(user?.id ?: operation.authorUserId)
            .setAuthorUserUid(user?.passportUid?.orElse(null) ?: operation.authorUserUid.orElseThrow())
            .setAuthorProviderId(null)
            .setSourceFolderOperationsLogId(null)
            .setDestinationFolderOperationsLogId(null)
            .setOldFolderFields(null)
            .setOldQuotas(QuotasByResource(oldQuotas))
            .setOldBalance(QuotasByResource(emptyMap()))
            .setOldProvisions(QuotasByAccount(emptyMap()))
            .setOldAccounts(null)
            .setNewFolderFields(null)
            .setNewQuotas(QuotasByResource(newQuotas))
            .setNewBalance(QuotasByResource(emptyMap()))
            .setNewProvisions(QuotasByAccount(emptyMap()))
            .setActuallyAppliedProvisions(null)
            .setNewAccounts(null)
            .setAccountsQuotasOperationsId(operation.operationId)
            .setQuotasDemandsId(null)
            .setOperationPhase(OperationPhase.CLOSE)
            .setCommentId(null)
            .setOrder(folder.nextOpLogOrder)
            .setDeliveryMeta(null)
            .setTransferMeta(null)
            .build()
    }

    private fun checkExistingOperationMatch(existingOperation: ProvideReserveOperation, provider: ProviderModel,
                                            accountsSpace: AccountSpaceModel?, locale: Locale): Result<Unit> {
        if (existingOperation.operation.providerId != provider.id
            || !Objects.equals(accountsSpace?.id, existingOperation.operation.accountsSpaceId.orElse(null))) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.idempotency.key.mismatch", null, locale))).build())
        }
        return Result.success(Unit)
    }

    private fun prepareExistingOperationResponse(
        existingOperation: ProvideReserveOperation
    ): UpdateProviderReserveProvisionsResponseDto {
        val status = getOperationStatus(existingOperation)
        val result = if (existingOperation.result != null) {
            val values = mutableListOf<ProviderReserveProvisionResponseValueDto>()
            val provisionsByResource = existingOperation.result.provisions.associateBy { it.resourceId }
            val unitsEnsemblesById = existingOperation.result.unitsEnsembles.associateBy { it.id }
            existingOperation.result.resources.forEach { resource ->
                val provision = provisionsByResource[resource.id]
                val providedValue = if (provision == null) {
                    0L
                } else {
                    provision.providedQuota ?: 0L
                }
                val converted = Units.convertToApi(providedValue, resource, unitsEnsemblesById[resource.unitsEnsembleId])
                values.add(ProviderReserveProvisionResponseValueDto(resource.id, converted.t1.toLong(),
                    converted.t2.key))
            }
            UpdateProviderReserveProvisionsResultDto(existingOperation.result.account.id, values)
        } else {
            null
        }
        return UpdateProviderReserveProvisionsResponseDto(status, result, existingOperation.operation.operationId)
    }

    private fun checkExistingOperationResult(operation: ProvideReserveOperation, locale: Locale): Result<Unit> {
        if (!operationHasError(operation)) {
            return Result.success(Unit)
        }
        val isFailedPrecondition = operation.operation.errorKind.map { k -> k == OperationErrorKind.FAILED_PRECONDITION }
            .orElse(false)
        val isConflict = operation.operation.errorKind.map { k -> k == OperationErrorKind.ALREADY_EXISTS }
            .orElse(false)
        val isUnavailable = operation.operation.errorKind.map { k -> k == OperationErrorKind.UNAVAILABLE }
            .orElse(false)
        val errors = ErrorCollection.builder()
        if (operation.operation.fullErrorMessage.isPresent) {
            errors.add(operation.operation.fullErrorMessage.get().getErrorCollection(locale).toErrorCollection())
        } else {
            val errorMessage = operation.operation.errorMessage.orElseGet {
                if (isFailedPrecondition) {
                    messages.getMessage("errors.grpc.code.failed.precondition", null, locale)
                } else if (isConflict) {
                    messages.getMessage("errors.grpc.code.already.exists", null, locale)
                } else if (isUnavailable) {
                    messages.getMessage("errors.grpc.code.unavailable", null, locale)
                } else {
                    messages.getMessage("errors.bad.request", null, locale)
                }
            }
            if (isFailedPrecondition) {
                errors.addError(TypedError.versionMismatch(errorMessage))
            } else if (isConflict) {
                errors.addError(TypedError.conflict(errorMessage))
            } else if (isUnavailable) {
                errors.addError(TypedError.unavailable(errorMessage))
            } else {
                errors.addError(TypedError.invalid(errorMessage))
            }
        }
        return Result.failure(errors
            .addDetail("operationMeta", ProvideReserveOperationFailureMeta(operation.operation.operationId))
            .build())
    }

    private fun getOperationStatus(operation: ProvideReserveOperation): UpdateProviderReserveProvisionsStatusDto {
        if (operation.operation.requestStatus.isEmpty
            || operation.operation.requestStatus.get() == RequestStatus.WAITING) {
            return UpdateProviderReserveProvisionsStatusDto.IN_PROGRESS
        }
        if (operation.operation.requestStatus.get() == RequestStatus.OK) {
            return UpdateProviderReserveProvisionsStatusDto.SUCCESS
        }
        throw IllegalStateException("Unexpected operation status")
    }

    private fun operationHasError(operation: ProvideReserveOperation): Boolean {
        return operation.operation.requestStatus.isPresent
            && operation.operation.requestStatus.get() == RequestStatus.ERROR
    }

    private fun preValidateRequest(providerId: String,
                                   request: UpdateProviderReserveProvisionsRequestDto,
                                   currentUser: YaUserDetails,
                                   locale: Locale): Result<UpdateProviderReserveProvisionsRequest> {
        val errors = ErrorCollection.builder()
        if (!Uuids.isValidUuid(providerId)) {
            errors.addError("providerId", TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale)))
        }
        if (request.accountsSpaceId != null && !Uuids.isValidUuid(request.accountsSpaceId)) {
            errors.addError("accountsSpaceId", TypedError.invalid(messages
                .getMessage("errors.accounts.space.not.found", null, locale)))
        }
        if (request.deltaValues == null) {
            errors.addError("deltaValues", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (request.values == null || request.values.isEmpty()) {
            errors.addError("values", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            request.values.forEachIndexed { index, value ->
                if (value == null) {
                    errors.addError("values.${index}", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)))
                } else {
                    if (value.resourceId == null) {
                        errors.addError("values.${index}.resourceId", TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)))
                    } else if (!Uuids.isValidUuid(value.resourceId)) {
                        errors.addError("values.${index}.resourceId", TypedError.invalid(messages
                            .getMessage("errors.resource.not.found", null, locale)))
                    }
                    if (value.provided == null) {
                        errors.addError("values.${index}.provided", TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)))
                    }
                    if (value.providedUnitKey == null || value.providedUnitKey.isBlank()) {
                        errors.addError("values.${index}.providedUnitKey", TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)))
                    }
                }
            }
        }
        if (currentUser.user.isEmpty || currentUser.user.get().passportUid.isEmpty
            || currentUser.user.get().passportLogin.isEmpty) {
            errors.addError(TypedError.forbidden(messages.getMessage("errors.access.denied", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(UpdateProviderReserveProvisionsRequest(request.accountsSpaceId,
            request.values!!.map { v -> ProviderReserveProvisionRequestValue(v!!.resourceId!!, v.provided!!,
                v.providedUnitKey!!) }, request.deltaValues!!))
    }

    private suspend fun validateProvider(providerId: String, locale: Locale): Result<ProviderModel> {
        val providerO = providersLoader.getProviderByIdImmediate(providerId, Tenants.DEFAULT_TENANT_ID).awaitSingle()
        if (providerO.isEmpty || providerO.get().isDeleted) {
            return Result.failure(ErrorCollection.builder().addError("providerId", TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale))).build())
        }
        return Result.success(providerO.get())
    }

    private suspend fun validateAccountsSpace(provider: ProviderModel, request: UpdateProviderReserveProvisionsRequest,
                                              locale: Locale): Result<AccountSpaceModel?> {
        return if (request.accountsSpaceId != null) {
            val accountsSpaceO = dbSessionRetryable(tableClient) {
                accountsSpacesDao.getById(roStaleSingleRetryableCommit(), request.accountsSpaceId,
                    Tenants.DEFAULT_TENANT_ID).awaitSingle()
            }!!
            if (accountsSpaceO.isEmpty || accountsSpaceO.get().isDeleted
                    || !Objects.equals(accountsSpaceO.get().providerId, provider.id)) {
                Result.failure(ErrorCollection.builder().addError("accountsSpaceId", TypedError.invalid(messages
                        .getMessage("errors.accounts.space.not.found", null, locale))).build())
            } else {
                Result.success(accountsSpaceO.get())
            }
        } else {
            Result.success(null)
        }
    }

    private suspend fun validateRequestValues(txSession: YdbTxSession, provider: ProviderModel,
                                              accountsSpace: AccountSpaceModel?,
                                              request: UpdateProviderReserveProvisionsRequest,
                                              locale: Locale): Result<UpdateProviderReserveProvisionsRequestModel> {
        val errors = ErrorCollection.builder()
        val values = mutableListOf<ProviderReserveProvisionRequestValueModel>()
        val resourceIds = request.values.map { it.resourceId }.distinct()
        val resourcesById = resourceIds.chunked(1000).flatMap { p -> resourcesLoader
            .getResourcesByIds(txSession, p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
            .associateBy { it.id }
        val unitsEnsemblesIds = resourcesById.values.map { it.unitsEnsembleId }.distinct()
        val unitsEnsemblesById = unitsEnsemblesIds.chunked(1000).flatMap { p -> unitsEnsemblesLoader
            .getUnitsEnsemblesByIds(txSession, p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
            .associateBy { it.id }
        request.values.forEachIndexed { index, value ->
            var canAdd = true
            val resource = resourcesById[value.resourceId]
            if (resource == null || resource.isDeleted
                || !Objects.equals(resource.providerId, provider.id)
                || !Objects.equals(resource.accountsSpacesId, accountsSpace?.id)
            ) {
                errors.addError("values.${index}.resourceId", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)))
                canAdd = false
            }
            val unitsEnsemble = if (resource != null) {
                unitsEnsemblesById[resource.unitsEnsembleId]!!
            } else {
                null
            }
            val provided = if (unitsEnsemble != null) {
                val unitO = unitsEnsemble.unitByKey(value.providedUnitKey)
                if (unitO.isEmpty || unitO.get().isDeleted) {
                    errors.addError("values.${index}.providedUnitKey", TypedError.invalid(messages
                        .getMessage("errors.unit.not.found", null, locale)))
                    canAdd = false
                    null
                } else if (resource != null && !resource.resourceUnits.allowedUnitIds.contains(unitO.get().id)) {
                    errors.addError("values.${index}.providedUnitKey", TypedError.invalid(messages
                        .getMessage("errors.unit.not.allowed", null, locale)))
                    canAdd = false
                    null
                } else {
                    if (!request.deltaValues && value.provided < 0) {
                        errors.addError("values.${index}.provided", TypedError.invalid(messages
                            .getMessage("errors.number.must.be.non.negative", null, locale)))
                        canAdd = false
                        null
                    } else {
                        val convertedValueO = Units.convertFromApi(value.provided, resource, unitsEnsemble, unitO.get())
                        if (convertedValueO.isEmpty) {
                            errors.addError("values.${index}.provided", TypedError.invalid(messages
                                .getMessage("errors.value.can.not.be.converted.to.unit", null, locale)))
                            canAdd = false
                            null
                        } else {
                            convertedValueO.get()
                        }
                    }
                }
            } else {
                null
            }
            if (canAdd) {
                values.add(ProviderReserveProvisionRequestValueModel(resource!!, provided!!, unitsEnsemble!!))
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(UpdateProviderReserveProvisionsRequestModel(provider, accountsSpace,
            values, request.deltaValues))
    }

    private fun validateMutability(request: UpdateProviderReserveProvisionsRequestModel, locale: Locale): Result<Unit> {
        val errors = ErrorCollection.builder()
        if (!request.provider.isManaged) {
            errors.addError("providerId", TypedError.invalid(messages
                .getMessage("errors.provider.is.not.managed", null, locale)))
        }
        if (request.provider.isReadOnly) {
            errors.addError("providerId", TypedError.invalid(messages
                .getMessage("errors.provider.is.read.only", null, locale)))
        }
        if (request.accountsSpace != null && request.accountsSpace.isReadOnly) {
            errors.addError("providerId", TypedError.invalid(messages
                .getMessage("errors.provider.account.space.is.read.only", null, locale)))
        }
        request.values.forEachIndexed { index, value ->
            if (!value.resource.isManaged) {
                errors.addError("values.${index}.resourceId", TypedError.invalid(messages
                    .getMessage("errors.resource.not.managed", null, locale)))
            }
            if (value.resource.isReadOnly) {
                errors.addError("values.${index}.resourceId", TypedError.invalid(messages
                    .getMessage("errors.resource.is.read.only", null, locale)))
            }
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(Unit)
        }
    }

    private suspend fun findReserveAccount(txSession: YdbTxSession, request: UpdateProviderReserveProvisionsRequestModel,
                                           locale: Locale): Result<AccountModel> {
        val reserveAccount = reserveAccountsService.findProviderReserveAccount(txSession, request.provider,
            request.accountsSpace) ?: return Result.failure(ErrorCollection.builder().addError(TypedError
            .invalid(messages.getMessage("errors.provider.reserve.account.not.found", null, locale))).build())
        val accountO = accountsDao.getById(txSession, reserveAccount.key.accountId,
                reserveAccount.key.tenantId).awaitSingle()
        if (accountO.isEmpty || accountO.get().isDeleted) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.provider.reserve.account.not.found", null, locale))).build())
        }
        return Result.success(accountO.get())
    }

    private suspend fun loadResourceDictionary(txSession: YdbTxSession,
                                               request: UpdateProviderReserveProvisionsRequestModel,
                                               accountsSpace: AccountSpaceModel?,
                                               provider: ProviderModel): ResourceDictionary {
        val accountsSpaceId = accountsSpace?.id
        val resources = if (accountsSpaceId == null) {
            resourcesDao.getAllByProvider(txSession, provider.id, provider.tenantId,
                true).awaitSingle().filter { it.accountsSpacesId == null }
        } else {
            resourcesDao.getAllByProviderAccountsSpace(txSession, provider.id, accountsSpaceId,
                provider.tenantId, true).awaitSingle()
        }
        val allResources = (request.values.map { it.resource } + resources).distinctBy { it.id }
        val resourceTypeIds = allResources.map { it.resourceTypeId }.toSet()
        val segmentationIds = (allResources.flatMap { it.segments.map { s -> s.segmentationId } } + (request
            .accountsSpace?.segments ?: emptySet()).map { it.segmentationId }).distinct()
        val segmentIds = (allResources.flatMap { it.segments.map { s -> s.segmentId } } + (request
            .accountsSpace?.segments ?: emptySet()).map { it.segmentId }).distinct()
        val resourceTypesById = resourceTypeIds.chunked(1000).flatMap { p -> resourceTypesLoader
            .getResourceTypesByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
            .associateBy { it.id }
        val segmentationsById = segmentationIds.chunked(1000).flatMap { p -> resourceSegmentationsLoader
            .getResourceSegmentationsByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle() }.associateBy { it.id }
        val segmentsById = segmentIds.chunked(1000).flatMap { p -> resourceSegmentsLoader
            .getResourceSegmentsByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle() }.associateBy { it.id }
        val unitsEnsembleIds = allResources.map { it.unitsEnsembleId }.distinct()
        val unitsEnsembles = unitsEnsembleIds.chunked(1000).flatMap { p -> unitsEnsemblesLoader
            .getUnitsEnsemblesByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
            .associateBy { it.id }
        return ResourceDictionary(allResources.associateBy { it.id }, resourceTypesById, segmentationsById,
            segmentsById, unitsEnsembles)
    }

    private suspend fun loadCurrentQuotas(txSession: YdbTxSession, account: AccountModel,
                                          resourceIds: Set<ResourceId>): CurrentQuotas {
        val accountsSpaceId = account.accountsSpacesId.orElse(null)
        val folderQuotas = if (accountsSpaceId == null) {
            quotasDao.getByFoldersAndProvider(txSession, listOf(account.folderId), account.tenantId,
                account.providerId).awaitSingle().filter { resourceIds.contains(it.resourceId) }
        } else {
            quotasDao.getByProviderFoldersResources(txSession, account.tenantId, setOf(account.folderId),
                account.providerId, resourceIds).awaitSingle()
        }
        val accounts = if (accountsSpaceId == null) {
            accountsDao.getByFoldersForProvider(txSession, account.tenantId, account.providerId, account.folderId,
                false).awaitSingle().filter { it.accountsSpacesId.isEmpty }
        } else {
            accountsDao.getAllByFoldersProvidersAccountsSpaces(txSession,
                setOf(FolderProviderAccountsSpace(account.tenantId, account.folderId, account.providerId,
                    accountsSpaceId)), false).awaitSingle()
        }
        val accountIds = accounts.map { it.id }.toSet()
        val provisions = accountsQuotasDao.getAllByAccountIds(txSession, account.tenantId, accountIds).awaitSingle()
        val folder = folderDao.getById(txSession, account.folderId, account.tenantId).awaitSingle().orElseThrow()
        return CurrentQuotas(folderQuotas.associateBy { it.resourceId },
            provisions.groupBy { it.accountId }.mapValues { e -> e.value.associateBy { it.resourceId } },
            accounts.associateBy { it.id }, folder)
    }

    private fun prepareUpdatedQuotas(currentQuotas: CurrentQuotas,
                                     account: AccountModel,
                                     request: UpdateProviderReserveProvisionsRequestModel,
                                     resourceDictionary: ResourceDictionary,
                                     now: Instant,
                                     currentUser: YaUserDetails,
                                     locale: Locale): Result<UpdatedQuotas> {
        val errors = ErrorCollection.builder()
        val updatedQuotas = mutableListOf<QuotaModel>()
        val folderId = currentQuotas.folder.id
        val serviceId = currentQuotas.folder.serviceId
        val author = UserIdDto(currentUser.user.orElseThrow().passportUid.orElseThrow(),
            currentUser.user.orElseThrow().passportLogin.orElseThrow())
        val accountsSpaceKey = prepareAccountsSpaceKey(request.accountsSpace, resourceDictionary)
        val updatedProvisions = mutableMapOf<ResourceId, Long>()
        val frozenAmounts = mutableMapOf<ResourceId, Long>()
        val oldQuotas = mutableMapOf<ResourceId, Long>()
        val newQuotas = mutableMapOf<ResourceId, Long>()
        if (request.deltaValues) {
            // New values are delta values
            prepareUpdatedQuotasDelta(request, account, currentQuotas, updatedQuotas, updatedProvisions,
                frozenAmounts, oldQuotas, newQuotas, errors, locale)
        } else {
            // New values are absolute values
            prepareUpdatedQuotasAbsolute(request, account, currentQuotas, updatedQuotas, updatedProvisions,
                frozenAmounts, oldQuotas, newQuotas, errors, locale)
        }
        (currentQuotas.folderProvisions[account.id] ?: emptyMap()).forEach { (resourceId, provision) ->
            val resource = resourceDictionary.resources[resourceId]!!
            val currentProvision = provision.providedQuota ?: 0L
            if (!resource.isReadOnly && resource.isManaged && !resource.isDeleted && currentProvision > 0L
                && !updatedProvisions.containsKey(resourceId)
            ) {
                // Current values for unchanged non-zero provisions of the updated account
                updatedProvisions[resourceId] = currentProvision
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        val resourceKeyCache = mutableMapOf<ResourceId, ResourceKeyRequestDto>()
        val requestUpdatedProvisions = updatedProvisions.map { (resourceId, provision) ->
            val resource = resourceDictionary.resources[resourceId]!!
            val unitsEnsemble = resourceDictionary.unitsEnsembles[resource.unitsEnsembleId]!!
            val resourceKey = resourceKeyCache.computeIfAbsent(resourceId) { prepareResourceKey(resource,
                resourceDictionary, request.accountsSpace) }
            val convertedProvision = Units.convertToApi(provision, resource, unitsEnsemble)
            ProvisionRequestDto(resourceKey, convertedProvision.t1.toLong(), convertedProvision.t2.key)
        }
        val knownProvisions = mutableMapOf<AccountId, MutableMap<ResourceId, Long>>()
        currentQuotas.accounts.keys.forEach { accountId ->
            val accountKnownProvisions = knownProvisions.computeIfAbsent(accountId) { mutableMapOf() }
            val accountProvisions = currentQuotas.folderProvisions[accountId] ?: emptyMap()
            accountProvisions.forEach { (resourceId, provision) ->
                val currentProvision = provision.providedQuota ?: 0L
                if (currentProvision != 0L || updatedProvisions.containsKey(resourceId)) {
                    // Current values for non-zero provisions and provisions of resources in updatedProvisions
                    accountKnownProvisions[resourceId] = currentProvision
                }
            }
            updatedProvisions.keys.forEach { resourceId ->
                if (!accountKnownProvisions.containsKey(resourceId)) {
                    // Zero for all remaining provisions of resources in updatedProvisions
                    accountKnownProvisions[resourceId] = 0L
                }
            }
        }
        val requestKnownProvisions = mutableListOf<KnownAccountProvisionsDto>()
        knownProvisions.forEach { (accountId, accountProvisions) ->
            val requestAccountKnownProvisions = mutableListOf<KnownProvisionDto>()
            accountProvisions.forEach { (resourceId, provision) ->
                val resource = resourceDictionary.resources[resourceId]!!
                val unitsEnsemble = resourceDictionary.unitsEnsembles[resource.unitsEnsembleId]!!
                val resourceKey = resourceKeyCache.computeIfAbsent(resourceId) { prepareResourceKey(resource,
                    resourceDictionary, request.accountsSpace) }
                val convertedProvision = Units.convertToApi(provision, resource, unitsEnsemble)
                requestAccountKnownProvisions.add(KnownProvisionDto(resourceKey, convertedProvision.t1.toLong(),
                    convertedProvision.t2.key))
            }
            val folderAccount = currentQuotas.accounts[accountId]!!
            requestKnownProvisions.add(KnownAccountProvisionsDto(folderAccount.outerAccountIdInProvider,
                requestAccountKnownProvisions))
        }
        val operation = prepareOperation(now, currentUser, request.provider, request.accountsSpace?.id,
            currentQuotas.folder, account, updatedProvisions, frozenAmounts, null)
        val operationLog = prepareSubmitOperationLog(operation, currentQuotas.folder, currentUser, oldQuotas, newQuotas)
        val updatedFolder = currentQuotas.folder.toBuilder()
            .setNextOpLogOrder(currentQuotas.folder.nextOpLogOrder + 1L).build()
        val providerRequest = UpdateProvisionRequestDto(folderId, serviceId,
            requestUpdatedProvisions, requestKnownProvisions, author, operation.operationId, accountsSpaceKey)
        val operationInProgress = OperationInProgressModel(Tenants.DEFAULT_TENANT_ID, operation.operationId,
            currentQuotas.folder.id, account.id, 0L)
        return Result.success(UpdatedQuotas(updatedQuotas, providerRequest, updatedFolder, operationLog, operation,
            operationInProgress, account, request.provider, resourceDictionary))
    }

    private fun prepareUpdatedQuotasDelta(request: UpdateProviderReserveProvisionsRequestModel,
                                          account: AccountModel,
                                          currentQuotas: CurrentQuotas,
                                          updatedQuotas: MutableList<QuotaModel>,
                                          updatedProvisions: MutableMap<ResourceId, Long>,
                                          frozenAmounts: MutableMap<ResourceId, Long>,
                                          oldQuotas: MutableMap<ResourceId, Long>,
                                          newQuotas: MutableMap<ResourceId, Long>,
                                          errors: ErrorCollection.Builder,
                                          locale: Locale) {
        request.values.forEachIndexed { index, value ->
            processDelta(index, account, value.resource.id, value.provided, currentQuotas, updatedQuotas,
                updatedProvisions, frozenAmounts, oldQuotas, newQuotas, errors, locale)
        }
    }

    private fun prepareUpdatedQuotasDelta(transfers: Set<ResourceQuotaTransfer>,
                                          account: AccountModel,
                                          currentQuotas: CurrentQuotas,
                                          updatedQuotas: MutableList<QuotaModel>,
                                          updatedProvisions: MutableMap<ResourceId, Long>,
                                          frozenAmounts: MutableMap<ResourceId, Long>,
                                          oldQuotas: MutableMap<ResourceId, Long>,
                                          newQuotas: MutableMap<ResourceId, Long>,
                                          errors: ErrorCollection.Builder,
                                          locale: Locale) {
        transfers.forEachIndexed { index, value ->
            processDelta(index, account, value.resourceId, value.delta, currentQuotas, updatedQuotas,
                updatedProvisions, frozenAmounts, oldQuotas, newQuotas, errors, locale)
        }
    }

    private fun processDelta(
        index: Int,
        account: AccountModel,
        resourceId: String,
        provisionDelta: Long,
        currentQuotas: CurrentQuotas,
        updatedQuotas: MutableList<QuotaModel>,
        updatedProvisions: MutableMap<ResourceId, Long>,
        frozenAmounts: MutableMap<ResourceId, Long>,
        oldQuotas: MutableMap<ResourceId, Long>,
        newQuotas: MutableMap<ResourceId, Long>,
        errors: ErrorCollection.Builder,
        locale: Locale
    ) {
        val currentBalance = currentQuotas.folderQuotas[resourceId]?.balance ?: 0L
        if (currentBalance < 0) {
            errors.addError(
                "values.${index}.provided", TypedError.invalid(
                    messages
                        .getMessage("errors.negative.balance.is.not.allowed", null, locale)
                )
            )
        } else {
            val currentQuota = currentQuotas.folderQuotas[resourceId]?.quota ?: 0L
            val currentFrozenQuota = currentQuotas.folderQuotas[resourceId]?.frozenQuota ?: 0L
            val currentProvision = (currentQuotas
                .folderProvisions[account.id] ?: emptyMap())[resourceId]?.providedQuota ?: 0L
            if (provisionDelta > 0) {
                // Provision is increased
                val updatedProvisionO = Units.add(currentProvision, provisionDelta)
                // Quota must be increased by provisionDelta to leave balance unchanged
                val updatedQuotaO = Units.add(currentQuota, provisionDelta)
                // Frozen quota is also increased by provisionDelta until provision is actually updated
                // because provision will be updated only after successful provider response
                val updatedFrozenQuotaO = Units.add(currentFrozenQuota, provisionDelta)
                if (updatedProvisionO.isEmpty || updatedQuotaO.isEmpty || updatedFrozenQuotaO.isEmpty) {
                    if (updatedProvisionO.isEmpty) {
                        errors.addError(
                            "values.${index}.provided", TypedError.invalid(
                                messages
                                    .getMessage("errors.provision.amount.overflow", null, locale)
                            )
                        )
                    }
                    if (updatedQuotaO.isEmpty || updatedFrozenQuotaO.isEmpty) {
                        errors.addError(
                            "values.${index}.provided", TypedError.invalid(
                                messages
                                    .getMessage("errors.quota.amount.overflow", null, locale)
                            )
                        )
                    }
                } else {
                    updatedQuotas.add(
                        QuotaModel.builder()
                            .tenantId(account.tenantId)
                            .folderId(account.folderId)
                            .providerId(account.providerId)
                            .resourceId(resourceId)
                            .quota(updatedQuotaO.get())
                            .balance(currentBalance)
                            .frozenQuota(updatedFrozenQuotaO.get())
                            .build()
                    )
                    updatedProvisions[resourceId] = updatedProvisionO.get()
                    frozenAmounts[resourceId] = provisionDelta
                    oldQuotas[resourceId] = currentQuota
                    newQuotas[resourceId] = updatedQuotaO.get()
                }
            } else if (provisionDelta < 0) {
                // Provision is decreased
                val updatedProvisionO = Units.add(currentProvision, provisionDelta)
                // Quota must be decreased by provisionDelta to leave balance unchanged
                val updatedQuotaO = Units.add(currentQuota, provisionDelta)
                if (updatedProvisionO.isEmpty || updatedQuotaO.isEmpty) {
                    if (updatedProvisionO.isEmpty) {
                        errors.addError(
                            "values.${index}.provided", TypedError.invalid(
                                messages
                                    .getMessage("errors.provision.amount.overflow", null, locale)
                            )
                        )
                    }
                    if (updatedQuotaO.isEmpty) {
                        errors.addError(
                            "values.${index}.provided", TypedError.invalid(
                                messages
                                    .getMessage("errors.quota.amount.overflow", null, locale)
                            )
                        )
                    }
                } else if (updatedQuotaO.get() < 0 || updatedProvisionO.get() < 0) {
                    if (updatedQuotaO.get() < 0) {
                        errors.addError(
                            "values.${index}.provided", TypedError.invalid(
                                messages
                                    .getMessage("errors.negative.quota.amount", null, locale)
                            )
                        )
                    }
                    if (updatedProvisionO.get() < 0) {
                        errors.addError(
                            "values.${index}.provided", TypedError.invalid(
                                messages
                                    .getMessage("errors.negative.provision.amount", null, locale)
                            )
                        )
                    }
                } else {
                    // Quota will be decreased only after successful provider response
                    updatedProvisions[resourceId] = updatedProvisionO.get()
                }
            } else {
                // Provision is unchanged
                updatedProvisions[resourceId] = currentProvision
            }
        }
    }

    private fun prepareUpdatedQuotasAbsolute(request: UpdateProviderReserveProvisionsRequestModel,
                                             account: AccountModel,
                                             currentQuotas: CurrentQuotas,
                                             updatedQuotas: MutableList<QuotaModel>,
                                             updatedProvisions: MutableMap<ResourceId, Long>,
                                             frozenAmounts: MutableMap<ResourceId, Long>,
                                             oldQuotas: MutableMap<ResourceId, Long>,
                                             newQuotas: MutableMap<ResourceId, Long>,
                                             errors: ErrorCollection.Builder,
                                             locale: Locale) {
        request.values.forEachIndexed { index, value ->
            val currentBalance = currentQuotas.folderQuotas[value.resource.id]?.balance ?: 0L
            if (currentBalance < 0) {
                errors.addError("values.${index}.provided", TypedError.invalid(messages
                    .getMessage("errors.negative.balance.is.not.allowed", null, locale)))
            } else {
                val currentQuota = currentQuotas.folderQuotas[value.resource.id]?.quota ?: 0L
                val currentFrozenQuota = currentQuotas.folderQuotas[value.resource.id]?.frozenQuota ?: 0L
                val currentProvision = (currentQuotas
                    .folderProvisions[account.id] ?: emptyMap())[value.resource.id]?.providedQuota ?: 0L
                val updatedProvision = value.provided
                if (updatedProvision > currentProvision) {
                    // Provision is increased
                    val provisionDeltaO = Units.subtract(updatedProvision, currentProvision)
                    if (provisionDeltaO.isEmpty) {
                        errors.addError("values.${index}.provided", TypedError.invalid(messages
                            .getMessage("errors.quota.amount.overflow", null, locale)))
                    } else {
                        // Quota must be increased by provisionDelta to leave balance unchanged
                        val updatedQuotaO = Units.add(currentQuota, provisionDeltaO.get())
                        // Frozen quota is also increased by provisionDelta until provision is actually updated
                        // because provision will be updated only after successful provider response
                        val updatedFrozenQuotaO = Units.add(currentFrozenQuota, provisionDeltaO.get())
                        if (updatedQuotaO.isEmpty || updatedFrozenQuotaO.isEmpty) {
                            errors.addError("values.${index}.provided", TypedError.invalid(messages
                                .getMessage("errors.quota.amount.overflow", null, locale)))
                        } else {
                            updatedQuotas.add(QuotaModel.builder()
                                .tenantId(account.tenantId)
                                .folderId(account.folderId)
                                .providerId(account.providerId)
                                .resourceId(value.resource.id)
                                .quota(updatedQuotaO.get())
                                .balance(currentBalance)
                                .frozenQuota(updatedFrozenQuotaO.get())
                                .build())
                            updatedProvisions[value.resource.id] = updatedProvision
                            frozenAmounts[value.resource.id] = provisionDeltaO.get()
                            oldQuotas[value.resource.id] = currentQuota
                            newQuotas[value.resource.id] = updatedQuotaO.get()
                        }
                    }
                } else if (updatedProvision < currentProvision) {
                    // Provision is decreased
                    val provisionDeltaO = Units.subtract(currentProvision, updatedProvision)
                    if (provisionDeltaO.isEmpty) {
                        errors.addError("values.${index}.provided", TypedError.invalid(messages
                            .getMessage("errors.quota.amount.overflow", null, locale)))
                    } else {
                        // Quota must be decreased by provisionDelta to leave balance unchanged
                        val updatedQuotaO = Units.subtract(currentQuota, provisionDeltaO.get())
                        if (updatedQuotaO.isEmpty) {
                            errors.addError("values.${index}.provided", TypedError.invalid(messages
                                .getMessage("errors.quota.amount.overflow", null, locale)))
                        } else if (updatedQuotaO.get() < 0) {
                            errors.addError("values.${index}.provided", TypedError.invalid(messages
                                .getMessage("errors.negative.quota.amount", null, locale)))
                        } else {
                            // Quota will be decreased only after successful provider response
                            updatedProvisions[value.resource.id] = updatedProvision
                        }
                    }
                } else {
                    // Provision is unchanged
                    updatedProvisions[value.resource.id] = currentProvision
                }
            }
        }
    }

    private fun prepareResourceKey(resource: ResourceModel, resourceDictionary: ResourceDictionary,
                                   accountsSpace: AccountSpaceModel?): ResourceKeyRequestDto {
        val resourceType = resourceDictionary.resourceTypes[resource.resourceTypeId]!!
        val accountsSpaceSegmentationIds = (accountsSpace?.segments ?: setOf())
            .map { it.segmentationId }.toSet()
        val segmentation = resource.segments.filter { !accountsSpaceSegmentationIds.contains(it.segmentationId) }
            .map { SegmentKeyRequestDto(resourceDictionary.segmentations[it.segmentationId]!!.key,
                resourceDictionary.segments[it.segmentId]!!.key) }
        return ResourceKeyRequestDto(resourceType.key, segmentation)
    }

    private fun prepareAccountsSpaceKey(accountsSpace: AccountSpaceModel?,
                                        resourceDictionary: ResourceDictionary): AccountsSpaceKeyRequestDto? {
        if (accountsSpace == null) {
            return null
        }
        val segments = accountsSpace.segments.map { s ->
            val segmentation = resourceDictionary.segmentations[s.segmentationId]!!
            val segment = resourceDictionary.segments[s.segmentId]!!
            SegmentKeyRequestDto(segmentation.key, segment.key)
        }
        return AccountsSpaceKeyRequestDto(segments)
    }

    private fun prepareOperation(now: Instant,
                                 currentUser: YaUserDetails,
                                 provider: ProviderModel,
                                 accountsSpaceId: AccountsSpacesId?,
                                 folder: FolderModel,
                                 account: AccountModel,
                                 updatedProvisions: Map<ResourceId, Long>,
                                 frozenAmounts: Map<ResourceId, Long>,
                                 transferRequestModel: TransferRequestModel?
    ): AccountsQuotasOperationsModel {
        val operationChanges = OperationChangesModel.builder()
            .accountId(account.id)
            .updatedProvisions(updatedProvisions.map { e -> OperationChangesModel.Provision(e.key, e.value) })
            .frozenProvisions(frozenAmounts.map { e -> OperationChangesModel.Provision(e.key, e.value) })
            .transferRequestId(transferRequestModel?.id)
            .build()
        return AccountsQuotasOperationsModel.builder()
            .setTenantId(Tenants.DEFAULT_TENANT_ID)
            .setOperationId(UUID.randomUUID().toString())
            .setLastRequestId(UUID.randomUUID().toString())
            .setCreateDateTime(now)
            .setOperationSource(OperationSource.USER)
            .setOperationType(AccountsQuotasOperationsModel.OperationType.PROVIDE_RESERVE)
            .setAuthorUserId(currentUser.user.orElseThrow().id)
            .setAuthorUserUid(currentUser.user.orElseThrow().passportUid.orElseThrow())
            .setProviderId(provider.id)
            .setAccountsSpaceId(accountsSpaceId)
            .setUpdateDateTime(null)
            .setRequestStatus(RequestStatus.WAITING)
            .setErrorMessage(null)
            .setFullErrorMessage(null)
            .setRequestedChanges(operationChanges)
            .setOrders(OperationOrdersModel.builder()
                .submitOrder(folder.nextOpLogOrder)
                .build())
            .setErrorKind(null)
            .setLogs(emptyList())
            .build()
    }

    private fun prepareSubmitOperationLog(operation: AccountsQuotasOperationsModel,
                                          folder: FolderModel,
                                          currentUser: YaUserDetails,
                                          oldQuotas: Map<ResourceId, Long>,
                                          newQuotas: Map<ResourceId, Long>): FolderOperationLogModel {
        return FolderOperationLogModel.builder()
            .setTenantId(Tenants.DEFAULT_TENANT_ID)
            .setFolderId(folder.id)
            .setOperationDateTime(operation.createDateTime)
            .setId(UUID.randomUUID().toString())
            .setProviderRequestId(operation.lastRequestId.orElseThrow())
            .setOperationType(FolderOperationType.PROVIDE_RESERVE)
            .setAuthorUserId(currentUser.user.orElseThrow().id)
            .setAuthorUserUid(currentUser.user.orElseThrow().passportUid.orElseThrow())
            .setAuthorProviderId(null)
            .setSourceFolderOperationsLogId(null)
            .setDestinationFolderOperationsLogId(null)
            .setOldFolderFields(null)
            .setOldQuotas(QuotasByResource(oldQuotas))
            .setOldBalance(QuotasByResource(emptyMap()))
            .setOldProvisions(QuotasByAccount(emptyMap()))
            .setOldAccounts(null)
            .setNewFolderFields(null)
            .setNewQuotas(QuotasByResource(newQuotas))
            .setNewBalance(QuotasByResource(emptyMap()))
            .setNewProvisions(QuotasByAccount(emptyMap()))
            .setActuallyAppliedProvisions(null)
            .setNewAccounts(null)
            .setAccountsQuotasOperationsId(operation.operationId)
            .setQuotasDemandsId(null)
            .setOperationPhase(OperationPhase.SUBMIT)
            .setCommentId(null)
            .setOrder(folder.nextOpLogOrder)
            .setDeliveryMeta(null)
            .setTransferMeta(null)
            .build()
    }

    private inline fun <T> meter(statement: () -> T,
                                 label: String): T {
        return elapsed(statement)
        { millis, success -> logger.info { "$label: duration = $millis ms, success = $success" } }
    }

    private data class UpdateProviderReserveProvisionsRequest(
        val accountsSpaceId: String?,
        val values: List<ProviderReserveProvisionRequestValue>,
        val deltaValues: Boolean
    )

    private data class ProviderReserveProvisionRequestValue(
        val resourceId: String,
        val provided: Long,
        val providedUnitKey: String
    )

    private data class UpdateProviderReserveProvisionsRequestModel(
        val provider: ProviderModel,
        val accountsSpace: AccountSpaceModel?,
        val values: List<ProviderReserveProvisionRequestValueModel>,
        val deltaValues: Boolean
    )

    private data class ProviderReserveProvisionRequestValueModel(
        val resource: ResourceModel,
        // Converted to base unit
        val provided: Long,
        val unitsEnsemble: UnitsEnsembleModel
    )

    private data class ResourceDictionary(
        val resources: Map<ResourceId, ResourceModel>,
        val resourceTypes: Map<ResourceTypeId, ResourceTypeModel>,
        val segmentations: Map<SegmentationId, ResourceSegmentationModel>,
        val segments: Map<SegmentId, ResourceSegmentModel>,
        val unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>
    )

    private data class CurrentQuotas(
        val folderQuotas: Map<ResourceId, QuotaModel>,
        val folderProvisions: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>,
        val accounts: Map<AccountId, AccountModel>,
        val folder: FolderModel
    )

    private data class UpdatedQuotas(
        val quotas: List<QuotaModel>,
        val request: UpdateProvisionRequestDto,
        val folder: FolderModel,
        val opLog: FolderOperationLogModel,
        val op: AccountsQuotasOperationsModel,
        val operationInProgress: OperationInProgressModel,
        val account: AccountModel,
        val provider: ProviderModel,
        val resourcesDictionary: ResourceDictionary
    )

    private data class PostRequestContext(
        val currentOperation: AccountsQuotasOperationsModel,
        val currentOperationInProgress: OperationInProgressModel?,
        val currentAccount: AccountModel,
        val currentQuotas: CurrentQuotas
    )

}
