package ru.yandex.intranet.d.services.quotas;

import java.math.BigDecimal;
import java.math.MathContext;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import kotlin.Pair;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.providers.ProviderId;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeId;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.util.FrontStringUtil;
import ru.yandex.intranet.d.web.model.AccountDto;
import ru.yandex.intranet.d.web.model.AmountDto;
import ru.yandex.intranet.d.web.model.folders.FrontAmountsDto;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedAccount;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedAccountResource;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedProvider;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedResource;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedResourceBuilder;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedResourceType;
import ru.yandex.intranet.d.web.model.folders.front.ProviderPermission;

import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.groupingBy;
import static ru.yandex.intranet.d.services.quotas.QuotasHelper.convertToReadable;
import static ru.yandex.intranet.d.services.quotas.QuotasHelper.getMinAllowedUnit;
import static ru.yandex.intranet.d.services.quotas.QuotasHelper.toProvidedAndNotAllocated;
import static ru.yandex.intranet.d.services.quotas.QuotasHelper.zeroAmount;

/**
 * ExpandedProviderBuilder.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 24-02-2021
 */
public class ExpandedProviderBuilder {
    private final Locale locale;
    private final Map<String, ResourceModel> resourceMap;
    private final Map<String, UnitsEnsembleModel> unitsEnsembleMap;
    private final ExternalAccountUrlFactory externalAccountUrlFactory;

    public ExpandedProviderBuilder(
            Locale locale,
            Map<String, ResourceModel> resourceMap,
            Map<String, UnitsEnsembleModel> unitsEnsembleMap,
            ExternalAccountUrlFactory externalAccountUrlFactory
    ) {
        this.locale = locale;
        this.resourceMap = resourceMap;
        this.unitsEnsembleMap = unitsEnsembleMap;
        this.externalAccountUrlFactory = externalAccountUrlFactory;
    }

    public ExpandedProvider toExpandedProvider(
            ProviderId providerId,
            Map<ResourceTypeId, List<QuotaSums>> quotasByResourceTypeId,
            List<AccountModel> accounts,
            Map<String, List<AccountsQuotasModel>> accountsQuotasByAccountId,
            Map<String, QuotaModel> quotasByResourceId,
            Set<ProviderPermission> providerPermissions,
            boolean emptyIfAllResourcesVirtual
    ) {
        boolean hasNonVirtualResources = hasNonVirtualResources(quotasByResourceId, providerId);
        Map<String, List<AccountsQuotasModel>> accountsQuotasByResourceId = accounts.stream()
                .flatMap(account -> accountsQuotasByAccountId.getOrDefault(account.getId(),
                        List.of()).stream())
                .collect(groupingBy(AccountsQuotasModel::getResourceId));
        List<ExpandedResourceType> types = new ArrayList<>();
        if (!emptyIfAllResourcesVirtual || hasNonVirtualResources) {
            quotasByResourceTypeId.forEach((resourceTypeId, quotas) -> {
                List<ExpandedResource> resources = new ArrayList<>();
                for (QuotaSums quota : quotas) {
                    for (AccountsQuotasModel accountQuota :
                            accountsQuotasByResourceId.getOrDefault(quota.getResourceId(), List.of())
                    ) {
                        quota.add(accountQuota);
                    }
                    ResourceModel resourceModel = resourceMap.get(quota.getResourceId());
                    UnitsEnsembleModel unitsEnsemble = unitsEnsembleMap.get(resourceModel.getUnitsEnsembleId());
                    resources.add(
                            toExpandedResource(resourceModel, quota, unitsEnsemble, locale).buildExpandedResource()
                    );
                }
                FrontAmountsDto sums = sumQuotas(
                        quotas,
                        resourceMap,
                        unitsEnsembleMap
                );

                types.add(new ExpandedResourceType(resourceTypeId.getValue(), resources, sums));
            });
        }

        // get accounts
        List<ExpandedAccount> expandedAccounts = accounts.stream()
                .map(account -> {
                            List<AccountsQuotasModel> accountQuotas = accountsQuotasByAccountId
                                    .getOrDefault(account.getId(), List.of());
                            List<ResourceModel> accountsResources = accountQuotas.stream()
                                    .map(quota -> resourceMap.get(quota.getResourceId()))
                                    .filter(Objects::nonNull)
                                    .collect(Collectors.toList());
                            Pair<Boolean, Map<String, String>> externalAccountUrlsPair =
                                    externalAccountUrlFactory != null ?
                                            externalAccountUrlFactory.generateUrl(account, accountsResources) : null;
                            Boolean urlForSegments = externalAccountUrlsPair != null ?
                                    externalAccountUrlsPair.getFirst() : null;
                            Map<String, String> generatedUrls = externalAccountUrlsPair != null ?
                                    externalAccountUrlsPair.getSecond() : null;
                            return new ExpandedAccount(
                                    AccountDto.fromModel(account),
                                    emptyIfAllResourcesVirtual && !hasNonVirtualResources ? List.of() :
                                            collectQuotasForAccount(
                                                    accountQuotas,
                                                    locale,
                                                    quotasByResourceId,
                                                    unitsEnsembleMap,
                                                    resourceMap),
                                    generatedUrls,
                                    urlForSegments
                            );
                        }
                )
                .collect(Collectors.toList());

        return new ExpandedProvider(
                providerId.getValue(), types, expandedAccounts, providerPermissions
        );
    }

    private List<ExpandedAccountResource> collectQuotasForAccount(
            List<AccountsQuotasModel> accountQuotas,
            Locale locale,
            Map<String, QuotaModel> quotasByResourceId,
            Map<String, UnitsEnsembleModel> unitsEnsembleMap,
            Map<String, ResourceModel> resourceMap
    ) {
        List<ExpandedAccountResource> result = new ArrayList<>();
        for (AccountsQuotasModel quota : accountQuotas) {
            ResourceModel resource = resourceMap.get(quota.getResourceId());
            QuotaModel folderQuota = quotasByResourceId.get(quota.getResourceId());
            if (folderQuota == null) {
                continue;
            }
            UnitsEnsembleModel unitsEnsembleModel = unitsEnsembleMap.get(resource.getUnitsEnsembleId());
            result.add(toExpandedResource(
                    resource,
                    QuotaSums.from(quota, folderQuota),
                    unitsEnsembleModel,
                    locale
            ).buildExpandedAccountResource());
        }
        return result;
    }

    private boolean hasNonZeroQuota(QuotaModel quotaModel) {
        return quotaModel.getQuota() != 0 || quotaModel.getBalance() != 0 || quotaModel.getFrozenQuota() != 0;
    }

    private boolean hasNonVirtualResources(Map<String, QuotaModel> quotasByResourceId, ProviderId providerId) {
        return quotasByResourceId.keySet().stream()
                .anyMatch(id ->
                        !resourceMap.get(id).isVirtual() &&
                                quotasByResourceId.get(id).getProviderId().equals(providerId.getValue()) &&
                                hasNonZeroQuota(quotasByResourceId.get(id))
                );
    }

    private FrontAmountsDto sumQuotas(
            List<QuotaSums> quotas,
            Map<String, ResourceModel> resourceMap,
            Map<String, UnitsEnsembleModel> unitsEnsembleMap
    ) {
        Set<String> resourceIds = quotas.stream()
                .map(QuotaSums::getResourceId)
                .collect(Collectors.toSet());
        String unitsEnsembleId = null;
        UnitsEnsembleModel ensemble = null;
        UnitModel baseUnit = null;
        UnitModel defaultUnit = null;
        Set<String> allowedUnits = null;
        for (String resourceId : resourceIds) {
            ResourceModel resourceModel = resourceMap.get(resourceId);
            if (unitsEnsembleId != null && !unitsEnsembleId.equals(resourceModel.getUnitsEnsembleId())) {
                throw new IllegalArgumentException("Can't sum units from different ensembles");
            }
            unitsEnsembleId = resourceModel.getUnitsEnsembleId();
            ensemble = unitsEnsembleMap.get(unitsEnsembleId);
            UnitModel newUnit = ensemble.unitById(resourceModel.getBaseUnitId()).get();
            if (baseUnit == null || UnitsComparator.INSTANCE.compare(baseUnit, newUnit) < 0) {
                baseUnit = newUnit;
            }
            UnitModel newDefaultUnit = ensemble.unitById(resourceModel.getResourceUnits().getDefaultUnitId()).get();
            if (defaultUnit == null || UnitsComparator.INSTANCE.compare(defaultUnit, newDefaultUnit) < 0) {
                defaultUnit = newDefaultUnit;
            }
            if (allowedUnits == null) {
                allowedUnits = Set.copyOf(resourceModel.getResourceUnits().getAllowedUnitIds());
            } else {
                allowedUnits = Sets.intersection(allowedUnits, resourceModel.getResourceUnits().getAllowedUnitIds());
            }
        }

        BigDecimal negativeBalance = null;
        BigDecimal positiveBalance = null;
        BigDecimal quotaSum = BigDecimal.ZERO;
        BigDecimal allocatedSum = BigDecimal.ZERO;
        BigDecimal frozenSum = BigDecimal.ZERO;
        for (QuotaSums quota : quotas) {
            final BigDecimal decimalBalance = BigDecimal.valueOf(quota.getBalance());
            if (quota.getBalance() >= 0) {
                positiveBalance = positiveBalance == null ? decimalBalance : positiveBalance.add(decimalBalance);
            } else {
                negativeBalance = negativeBalance == null ? decimalBalance : negativeBalance.add(decimalBalance);
            }
            quotaSum = quotaSum.add(BigDecimal.valueOf(quota.getQuota()));
            if (quota.getAllocatedQuota() != null) {
                allocatedSum = allocatedSum.add(BigDecimal.valueOf(quota.getAllocatedQuota()));
            }
            if (quota.getFrozenQuota() != null) {
                frozenSum = frozenSum.add(BigDecimal.valueOf(quota.getFrozenQuota()));
            }
        }
        BigDecimal balanceSum = BigDecimal.ZERO;
        BigDecimal providedSum = quotaSum;
        if (positiveBalance != null) {
            providedSum = providedSum.subtract(positiveBalance);
            balanceSum = balanceSum.add(positiveBalance);
        }
        if (negativeBalance != null) {
            providedSum = providedSum.subtract(negativeBalance);
            balanceSum = balanceSum.add(negativeBalance);
        }
        providedSum = providedSum.subtract(frozenSum);

        Set<String> finalAllowedUnits = allowedUnits != null ? Set.copyOf(allowedUnits) : emptySet();
        List<UnitModel> allowedSortedUnits = ensemble.getUnits().stream()
                .filter(unit -> finalAllowedUnits.contains(unit.getId()))
                .sorted(UnitsComparator.INSTANCE).collect(Collectors.toList());
        UnitModel minAllowedUnit = allowedSortedUnits.stream().findFirst().orElse(baseUnit);

        AmountDto positiveAmount = positiveBalance != null ?
                getAmountDto(positiveBalance, allowedSortedUnits, baseUnit, defaultUnit,
                        minAllowedUnit, locale) : null;

        AmountDto negativeAmount = negativeBalance != null ?
                getAmountDto(negativeBalance, allowedSortedUnits, baseUnit, defaultUnit,
                        minAllowedUnit, locale) : null;

        AmountDto balanceAmount = getAmountDto(balanceSum, allowedSortedUnits, baseUnit, defaultUnit,
                minAllowedUnit, locale);
        AmountDto quotaAmount = getAmountDto(quotaSum, allowedSortedUnits, baseUnit, defaultUnit,
                minAllowedUnit, locale);
        AmountDto frozenQuotaAmount = getAmountDto(frozenSum, allowedSortedUnits, baseUnit, defaultUnit,
                minAllowedUnit, locale);

        AmountDto providedAmount = getAmountDto(providedSum, allowedSortedUnits, baseUnit, defaultUnit,
                minAllowedUnit, locale);
        BigDecimal providedRatio = quotaSum.signum() == 0 ? BigDecimal.ZERO :
                providedSum.divide(quotaSum, MathContext.DECIMAL64);
        AmountDto allocated = getAmountDto(allocatedSum, allowedSortedUnits, baseUnit, defaultUnit,
                minAllowedUnit, locale);
        BigDecimal allocatedRatio = quotaSum.signum() == 0 ? BigDecimal.ZERO :
                FrontStringUtil.toBigDecimal(allocated.getRawAmount()).divide(quotaSum, MathContext.DECIMAL64);

        return new FrontAmountsDto(
                quotaAmount,
                balanceAmount,
                positiveAmount,
                negativeAmount,
                frozenQuotaAmount,
                providedAmount,
                providedRatio,
                allocated,
                allocatedRatio
        );
    }

    private ExpandedResourceBuilder toExpandedResource(
            ResourceModel resourceModel,
            QuotaSums quotaSums,
            UnitsEnsembleModel unitsEnsembleModel,
            Locale locale
    ) {
        List<UnitModel> sortedUnits = unitsEnsembleModel.getUnits().stream()
                .sorted(UnitsComparator.INSTANCE).collect(Collectors.toList());
        List<UnitModel> allowedSortedUnits = QuotasHelper.getAllowedUnits(resourceModel, sortedUnits);
        UnitModel baseUnit = unitsEnsembleModel.unitById(resourceModel.getBaseUnitId()).get();
        UnitModel defaultUnit = unitsEnsembleModel.unitById(resourceModel.getResourceUnits().getDefaultUnitId()).get();
        UnitModel minAllowedUnit = getMinAllowedUnit(resourceModel.getResourceUnits().getAllowedUnitIds(), sortedUnits)
                .orElse(baseUnit);

        AmountDto zeroAmount = zeroAmount(defaultUnit, defaultUnit, minAllowedUnit, locale);
        AmountDto quota = zeroAmount;
        AmountDto balance = zeroAmount;
        AmountDto negativeBalance = null;
        AmountDto positiveBalance = null;
        AmountDto frozenQuota = zeroAmount;
        AmountDto provided = zeroAmount;
        BigDecimal providedRatio = BigDecimal.ZERO;
        AmountDto allocated = zeroAmount;
        BigDecimal allocatedRatio = BigDecimal.ZERO;
        AmountDto providedAndNotAllocated = zeroAmount;

        if (quotaSums.getQuota() != null && quotaSums.getBalance() != null) {
            quota = getAmountDto(quotaSums.getQuota(), allowedSortedUnits, baseUnit, defaultUnit,
                    minAllowedUnit, locale);
            balance = getAmountDto(quotaSums.getBalance(), allowedSortedUnits, baseUnit, defaultUnit,
                    minAllowedUnit, locale);
            if (quotaSums.getBalance() >= 0) {
                positiveBalance = balance;
            } else {
                negativeBalance = balance;
            }
            long providedQuota = quotaSums.getQuota() - quotaSums.getBalance();
            if (quotaSums.getFrozenQuota() != null) {
                providedQuota = providedQuota - quotaSums.getFrozenQuota();
            }
            provided = getAmountDto(providedQuota, allowedSortedUnits, baseUnit, defaultUnit,
                    minAllowedUnit, locale);
            if (quotaSums.getQuota() != 0) {
                providedRatio = BigDecimal.valueOf(providedQuota)
                        .divide(BigDecimal.valueOf(quotaSums.getQuota()), MathContext.DECIMAL64);
            }
        } else if (quotaSums.getProvidedQuota() != null) {
            provided = getAmountDto(
                    quotaSums.getProvidedQuota(), allowedSortedUnits, baseUnit, defaultUnit,
                    minAllowedUnit, locale);
            if (quotaSums.getBalance() != null) {
                balance = getAmountDto(quotaSums.getBalance(), allowedSortedUnits, baseUnit, defaultUnit,
                        minAllowedUnit, locale);
            }
            if (quotaSums.getQuota() != null) {
                quota = getAmountDto(quotaSums.getQuota(), allowedSortedUnits, baseUnit, defaultUnit,
                        minAllowedUnit, locale);
                if (quotaSums.getQuota() != 0) {
                    providedRatio = BigDecimal.valueOf(quotaSums.getProvidedQuota())
                            .divide(BigDecimal.valueOf(quotaSums.getQuota()), MathContext.DECIMAL64);
                }
            }
        }
        if (quotaSums.getAllocatedQuota() != null) {
            allocated = getAmountDto(quotaSums.getAllocatedQuota(), allowedSortedUnits, baseUnit, defaultUnit,
                    minAllowedUnit, locale);
            if (quotaSums.getQuota() != null && quotaSums.getQuota() != 0) {
                allocatedRatio = BigDecimal.valueOf(quotaSums.getAllocatedQuota())
                        .divide(BigDecimal.valueOf(quotaSums.getQuota()), MathContext.DECIMAL64);
            }
        }
        if (quotaSums.getFrozenQuota() != null) {
            frozenQuota = getAmountDto(quotaSums.getFrozenQuota(), allowedSortedUnits, baseUnit, defaultUnit,
                    minAllowedUnit, locale);
        }
        if (!provided.equals(zeroAmount) || !allocated.equals(zeroAmount)) {
            providedAndNotAllocated = getAmountDto(toProvidedAndNotAllocated(provided.getRawAmount(),
                    allocated.getRawAmount()), allowedSortedUnits, baseUnit, defaultUnit, minAllowedUnit, locale);
        }
        return new ExpandedResourceBuilder()
                .setResourceId(quotaSums.getResourceId())
                .setQuota(quota)
                .setBalance(balance)
                .setPositiveBalance(positiveBalance)
                .setNegativeBalance(negativeBalance)
                .setFrozenQuota(frozenQuota)
                .setProvided(provided)
                .setProvidedRatio(providedRatio)
                .setAllocated(allocated)
                .setAllocatedRatio(allocatedRatio)
                .setProvidedAndNotAllocated(providedAndNotAllocated);
    }

    private AmountDto getAmountDto(long amount, List<UnitModel> allowedSortedUnits, UnitModel baseUnit,
                                   UnitModel defaultUnit, UnitModel minAllowedUnit, Locale locale) {
        return getAmountDto(BigDecimal.valueOf(amount), allowedSortedUnits, baseUnit, defaultUnit, minAllowedUnit,
                locale);
    }

    private AmountDto getAmountDto(BigDecimal amount, List<UnitModel> allowedSortedUnits, UnitModel baseUnit,
                                   UnitModel defaultUnit, UnitModel minAllowedUnit, Locale locale) {
        UnitModel forEditUnit = convertToReadable(amount, allowedSortedUnits, baseUnit).getUnit();
        return QuotasHelper.getAmountDto(amount, allowedSortedUnits, baseUnit, forEditUnit, defaultUnit, minAllowedUnit,
                locale);
    }
}
