package ru.yandex.intranet.d.services.quotas

import org.springframework.web.util.UriComponentsBuilder
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.providers.ExternalAccountUrlTemplate
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.web.model.resources.AccountsSpaceDto

data class ExternalAccountUrlFactory(
    private val allTemplates: Set<ExternalAccountUrlTemplate>? = setOf(),
    private val serviceId: Long? = null,
    private val segmentationsById: Map<String, ResourceSegmentationModel> = mapOf(),
    private val segmentsById: Map<String, ResourceSegmentModel> = mapOf(),
    private val accountsSpaces: List<AccountsSpaceDto> = listOf(),
) {
    fun generateUrl(
        account: AccountModel,
        accountsResources: List<ResourceModel>
    ): Pair<Boolean, Map<String, String>>? {
        if (allTemplates == null || allTemplates.isEmpty()) return null
        val accountSegmentKeysBySegmentationKeys =
            if (account.accountsSpacesId.isPresent && accountsSpaces.isNotEmpty()) {
                accountsSpaces
                    .filter { s -> s.id == account.accountsSpacesId.get() }
                    .flatMap { s -> s.segments }
                    .associateBy(
                        { s -> s.segmentation.key },
                        { s -> s.segment.key })
            } else {
                mapOf()
            }
        val pickedTemplate = pickUrlTemplates(accountSegmentKeysBySegmentationKeys)
        val urlTemplates = pickedTemplate.urlTemplates
        val urlsForSegments = pickedTemplate.urlsForSegments

        if (urlTemplates.isEmpty()) return null

        val urlVariables = mutableMapOf<String, String?>()
        urlVariables.putAllIfNotNull(
            listOf(
                "service_id" to serviceId?.toString(),
                "outer_account_id_in_provider" to account.outerAccountIdInProvider,
                "outer_account_key_in_provider" to account.outerAccountKeyInProvider.orElse(null),
                "display_name" to account.displayName.orElse(null)
            )
        )
        accountSegmentKeysBySegmentationKeys.forEach { (segmentationKey, segmentKey) ->
            urlVariables.putIfNotNull("account_segment_$segmentationKey" to segmentKey)
        }
        return if (urlsForSegments && accountsResources.isNotEmpty()) {
            generateUrlsWithSegmentations(urlTemplates, urlVariables, accountsResources)
        } else {
            generateUrlsWithoutSegmentations(urlTemplates, urlVariables)
        }
    }

    private fun generateUrlsWithSegmentations(
        urlTemplates: Map<String, String>,
        urlVariables: MutableMap<String, String?>,
        accountsResources: List<ResourceModel>
    ): Pair<Boolean, Map<String, String>>? {
        val result = mutableMapOf<String, String>()
        val segmentEnsembles: List<List<ResourceSegmentModel>> = accountsResources
            .map { resource ->
                resource.segments
                    .map { s -> segmentationsById[s.segmentationId]?.groupingOrder to segmentsById[s.segmentId] }
                    .distinct()
                    .sortedBy { (order, _) -> order }
                    .mapNotNull { (_, segment) -> segment }
            }

        for ((name, template) in urlTemplates) {
            for (segmentEnsemble in segmentEnsembles) {
                urlVariables.putAllIfNotNull(segmentEnsemble.map { segment ->
                    "resource_segment_${segmentationsById[segment.segmentationId]?.key}" to segment.key
                })
                try {
                    val url = UriComponentsBuilder.fromHttpUrl(template)
                        .buildAndExpand(urlVariables)
                    result[segmentEnsemble.joinToString(".") { s -> s.id } + ":" + name] = url.toUriString()
                } catch (exception: Exception) {
                    continue
                }
            }
        }
        return if (result.isNotEmpty()) {
            true to result
        } else {
            null
        }
    }

    private fun generateUrlsWithoutSegmentations(
        urlTemplates: Map<String, String>,
        urlVariables: Map<String, String?>
    ): Pair<Boolean, Map<String, String>>? {
        val result = mutableMapOf<String, String>()
        for ((name, template) in urlTemplates) {
            try {
                val url = UriComponentsBuilder.fromHttpUrl(template)
                    .buildAndExpand(urlVariables)
                result[name] = url.toUriString()
            } catch (exception: Exception) {
                continue
            }
        }
        return if (result.isNotEmpty()) {
            false to result
        } else {
            null
        }
    }

    private fun pickUrlTemplates(accountSegmentKeysBySegmentationKeys: Map<String, String>)
        : ExternalAccountUrlTemplate {
        if (accountSegmentKeysBySegmentationKeys.isNotEmpty()) {
            val pickedTemplate = allTemplates!!
                .firstOrNull { template ->
                    template.segments.isNotEmpty() && template.segments.all { (segmentationKey, segmentKeys) ->
                        segmentKeys.contains(accountSegmentKeysBySegmentationKeys[segmentationKey])
                    }
                }
            if (pickedTemplate != null) {
                return pickedTemplate
            }
        }
        return allTemplates!!.firstOrNull { template -> template.defaultTemplate } ?: ExternalAccountUrlTemplate()

    }
}

private fun MutableMap<String, String?>.putIfNotNull(pair: Pair<String?, String?>) {
    val key = pair.first
    val value = pair.second
    if (key != null && value != null) {
        this[key] = value
    }
}

private fun MutableMap<String, String?>.putAllIfNotNull(pairs: List<Pair<String?, String?>>) {
    pairs.forEach { pair -> putIfNotNull(pair) }
}
