package ru.yandex.intranet.d.services.quotas;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderId;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderId;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeId;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.resources.AccountsSpacesUtils;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;
import ru.yandex.intranet.d.web.model.ProviderDto;
import ru.yandex.intranet.d.web.model.ResourceDto;
import ru.yandex.intranet.d.web.model.folders.FrontFolderDto;
import ru.yandex.intranet.d.web.model.folders.FrontFolderWithQuotesDto;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedFolder;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedProvider;
import ru.yandex.intranet.d.web.model.folders.front.ProviderPermission;
import ru.yandex.intranet.d.web.model.folders.front.ResourceTypeDto;
import ru.yandex.intranet.d.web.model.resources.AccountsSpaceDto;
import ru.yandex.intranet.d.web.util.ModelDtoConverter;

import static java.util.stream.Collectors.groupingBy;
import static ru.yandex.intranet.d.web.util.ModelDtoConverter.providerDtoFromModel;

/**
 * FrontFolderWithQuotesDtoBuilder.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 20-02-2021
 */
public class FrontFolderWithQuotesDtoBuilder {
    private final Locale locale;
    private final AccountsSpacesUtils accountsSpacesUtils;

    public FrontFolderWithQuotesDtoBuilder(Locale locale, AccountsSpacesUtils accountsSpacesUtils) {
        this.locale = locale;
        this.accountsSpacesUtils = accountsSpacesUtils;
    }

    public <CONTEXT extends
            WithContinuationToken &
            WithFolders &
            WithQuotas &
            WithResources &
            WithResourceTypes &
            WithAccounts &
            WithAccountsQuotas &
            WithUnitsEnsembles &
            WithProviders &
            WithFolderPermissions &
            WithSegmentations &
            WithSegments>
    FrontFolderWithQuotesDto build(CONTEXT c) {
        // Preparation
        Map<String, ResourceModel> resourceMap = c.getResources().stream().collect(
                Collectors.toMap(ResourceModel::getId, Function.identity()));
        Function<QuotaModel, ResourceTypeId> getResourceTypeId = q -> new ResourceTypeId(
                resourceMap.get(q.getResourceId()).getResourceTypeId()
        );
        Map<String, List<AccountsQuotasModel>> accountsQuotasByAccountId =
                c.getAccountsQuotas().stream().collect(groupingBy(AccountsQuotasModel::getAccountId));
        Map<String, UnitsEnsembleModel> unitsEnsembleMap =
                c.getUnitsEnsembles().stream().collect(Collectors.toMap(UnitsEnsembleModel::getId,
                        Function.identity()));
        Map<String, ProviderModel> providersByIds = c.getProviders().stream()
                .collect(Collectors.toMap(
                        ProviderModel::getId,
                        Function.identity()));
        List<AccountsSpaceDto> accountsSpaces = accountsSpacesUtils.toDtos(c.getAccountsSpaces(), locale);

        // Grouping
        Map<FolderId,
                Map<ProviderId,
                        Map<ResourceTypeId,
                                List<QuotaSums>>>> collectedQuotas = c.getQuotas().stream().collect(
                groupingBy(FolderId::from,
                        groupingBy(ProviderId::from,
                                groupingBy(getResourceTypeId,
                                        Collectors.mapping(QuotaSums::from, Collectors.toList())))));

        Map<FolderId,
                Map<ProviderId,
                        List<AccountModel>>> collectedAccounts = c.getAccounts().stream().collect(
                groupingBy(FolderId::from,
                        groupingBy(ProviderId::from)));

        Map<FolderId, Map<String, QuotaModel>> quotasByFolderIdByResourceId = c.getQuotas().stream().collect(
                groupingBy(FolderId::from,
                        Collectors.toMap(QuotaModel::getResourceId, Function.identity())));

        List<ResourceDto> resources = c.getResources().stream()
                .map(res -> new ResourceDto(
                        res, locale, unitsEnsembleMap, c.getSegmentationsById(), c.getSegmentsById()))
                .collect(Collectors.toList());

        // Assembling result
        List<ExpandedFolder> folders = new ArrayList<>();
        for (FolderModel folderModel : c.getFolders()) {
            Map<ProviderId, Map<ResourceTypeId, List<QuotaSums>>> quotasByProviderId =
                    collectedQuotas.getOrDefault(FolderId.from(folderModel), Map.of());

            Map<ProviderId, List<AccountModel>> accountsByProvider = collectedAccounts
                    .getOrDefault(FolderId.from(folderModel), Map.of());

            Map<String, QuotaModel> quotasByResourceId =
                    quotasByFolderIdByResourceId.getOrDefault(FolderId.from(folderModel), Map.of());

            Set<ProviderId> providerIds = Sets.union(quotasByProviderId.keySet(), accountsByProvider.keySet());

            List<ExpandedProvider> providers = new ArrayList<>();

            for (ProviderId providerId : providerIds) {
                ProviderModel provider = providersByIds.get(providerId.getValue());
                var urlFactoryBuilder = new ExternalAccountUrlFactory(
                        provider == null ? null : provider.getAccountsSettings().getExternalAccountUrlTemplates(),
                        folderModel.getServiceId(),
                        c.getSegmentationsById(),
                        c.getSegmentsById(),
                        accountsSpaces
                );

                List<AccountModel> accounts = accountsByProvider
                        .getOrDefault(providerId, List.of());
                Map<ResourceTypeId, List<QuotaSums>> quotasByResourceTypeId = quotasByProviderId
                        .getOrDefault(providerId, Map.of());

                Set<ProviderPermission> providerPermissions = c.getPermissionsByFolder()
                        .getFolderProviderPermissions()
                        .getOrDefault(folderModel.getId(), Map.of())
                        .getOrDefault(providerId.getValue(), Set.of());
                ExpandedProvider expandedProvider = new ExpandedProviderBuilder(
                        locale,
                        resourceMap,
                        unitsEnsembleMap,
                        urlFactoryBuilder
                ).toExpandedProvider(
                        providerId,
                        quotasByResourceTypeId,
                        accounts,
                        accountsQuotasByAccountId,
                        quotasByResourceId,
                        providerPermissions,
                        true
                );
                providers.add(expandedProvider);
            }
            //Папка
            folders.add(new ExpandedFolder(FrontFolderDto.from(folderModel), providers, c.getPermissionsByFolder()
                    .getFolderPermissions().get(folderModel.getId())));
        }

        List<ProviderDto> providers = c.getProviders().stream().map(
                provider -> providerDtoFromModel(provider, locale))
                .collect(Collectors.toList());

        List<ResourceTypeDto> resourceTypes = c.getResourceTypes().stream()
                .map(resourceType -> ModelDtoConverter.resourceTypeDtoFromModel(resourceType, locale, unitsEnsembleMap))
                .collect(Collectors.toList());

        return new FrontFolderWithQuotesDto(
                folders,
                resources,
                providers,
                resourceTypes,
                accountsSpaces,
                c.getContinuationToken()
        );
    }

    public interface WithContinuationToken {
        String getContinuationToken();
    }

    public interface WithFolders {
        List<FolderModel> getFolders();

        default List<String> getFoldersIds() {
            return getFolders().stream().map(FolderModel::getId).collect(Collectors.toList());
        }

        default List<WithTenant<String>> getFolderIdsWithTenants() {
            return getFolders().stream()
                    .map(folder -> new WithTenant<>(folder.getTenantId(), folder.getId()))
                    .collect(Collectors.toList());
        }
    }

    public interface WithQuotas {
        List<QuotaModel> getQuotas();

        default List<Tuple2<String, TenantId>> getResourcesIds() {
            return getQuotas().stream()
                    .map(q -> Tuples.of(q.getResourceId(), q.getTenantId()))
                    .distinct()
                    .collect(Collectors.toList());
        }
    }

    public interface WithResources {
        List<ResourceModel> getResources();

        default List<Tuple2<String, TenantId>> getUnitsEnsemblesIds() {
            return getResources().stream()
                    .map(resource -> Tuples.of(resource.getUnitsEnsembleId(), resource.getTenantId()))
                    .distinct()
                    .collect(Collectors.toList());
        }

        default List<Tuple2<String, TenantId>> getResourceTypesIds() {
            return getResources().stream()
                    .filter(resource -> resource.getResourceTypeId() != null)
                    .map(resource -> Tuples.of(resource.getResourceTypeId(), resource.getTenantId()))
                    .distinct()
                    .collect(Collectors.toList());
        }

        default List<Tuple2<String, TenantId>> getProvidersIds() {
            return getResources().stream()
                    .map(resource -> Tuples.of(resource.getProviderId(), resource.getTenantId()))
                    .distinct()
                    .collect(Collectors.toList());
        }

        default List<Tuple2<String, TenantId>> getSegmentationsIds() {
            return getResources().stream()
                    .flatMap(resource -> resource.getSegments().stream()
                        .map(segment -> Tuples.of(segment.getSegmentationId(), resource.getTenantId()))
                    )
                    .collect(Collectors.toList());
        }

        default List<Tuple2<String, TenantId>> getSegmentsIds() {
            return getResources().stream()
                    .flatMap(resource -> resource.getSegments().stream()
                            .map(segment -> Tuples.of(segment.getSegmentId(), resource.getTenantId()))
                    )
                    .collect(Collectors.toList());
        }
    }

    public interface AccountAcceptor<T> {
        T withAccounts(List<AccountModel> accounts, ExpandedAccountsSpaces<List<AccountSpaceModel>> spaces);
    }

    public interface WithAccounts {
        List<AccountModel> getAccounts();

        ExpandedAccountsSpaces<List<AccountSpaceModel>> getAccountsSpaces();

        default Set<String> getAccountIds() {
            return getAccounts().stream().map(AccountModel::getId).collect(Collectors.toSet());
        }
    }

    public interface WithAccountsQuotas {
        List<AccountsQuotasModel> getAccountsQuotas();
    }

    public interface WithUnitsEnsembles {
        List<UnitsEnsembleModel> getUnitsEnsembles();
    }

    public interface WithProviders {
        List<ProviderModel> getProviders();
    }

    public interface WithResourceTypes {
        List<ResourceTypeModel> getResourceTypes();
    }

    public interface WithFolderPermissions {
        FoldersPermissionsCollection getPermissionsByFolder();
    }

    public interface WithSegmentations {
        Map<String, ResourceSegmentationModel> getSegmentationsById();
    }

    public interface WithSegments {
        Map<String, ResourceSegmentModel> getSegmentsById();
    }
}
