package ru.yandex.intranet.d.services.quotas;

import java.util.List;
import java.util.Map;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;

/**
 * Validation and loading context for methods, which returns FrontFolderWithQuotesDto.
 * Immutable boilerplate for collecting data in Mono chain.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @see QuotasService#getFrontPageForService
 * @since 25-02-2021
 */
public class FrontFolderWithQuotesDtoContext {
    public static class Impl implements
            FrontFolderWithQuotesDtoBuilder.WithContinuationToken,
            FrontFolderWithQuotesDtoBuilder.WithFolders,
            FrontFolderWithQuotesDtoBuilder.WithQuotas,
            FrontFolderWithQuotesDtoBuilder.WithResources,
            FrontFolderWithQuotesDtoBuilder.WithResourceTypes,
            FrontFolderWithQuotesDtoBuilder.WithAccounts,
            FrontFolderWithQuotesDtoBuilder.WithAccountsQuotas,
            FrontFolderWithQuotesDtoBuilder.WithUnitsEnsembles,
            FrontFolderWithQuotesDtoBuilder.WithProviders,
            FrontFolderWithQuotesDtoBuilder.WithFolderPermissions,
            FrontFolderWithQuotesDtoBuilder.WithSegmentations,
            FrontFolderWithQuotesDtoBuilder.WithSegments {
        private final List<FolderModel> folders;
        private final List<QuotaModel> quotas;
        private final List<ResourceModel> resources;
        private final List<ResourceTypeModel> resourceTypes;
        private final List<AccountModel> accounts;
        private final ExpandedAccountsSpaces<List<AccountSpaceModel>> accountsSpaces;
        private final List<AccountsQuotasModel> accountsQuotas;
        private final List<UnitsEnsembleModel> unitsEnsembles;
        private final List<ProviderModel> providers;
        private final FoldersPermissionsCollection permissionsByFolder;
        private final String continuationToken;
        private final Map<String, ResourceSegmentationModel> segmentationsById;
        private final Map<String, ResourceSegmentModel> segmentsById;

        public Impl() {
            this(new Builder());
        }

        Impl(Builder builder) {
            folders = builder.folders;
            quotas = builder.quotas;
            resources = builder.resources;
            resourceTypes = builder.resourceTypes;
            accounts = builder.accounts;
            accountsQuotas = builder.accountsQuotas;
            unitsEnsembles = builder.unitsEnsembles;
            providers = builder.providers;
            continuationToken = builder.continuationToken;
            permissionsByFolder = builder.permissionsByFolder;
            accountsSpaces = builder.accountsSpaces;
            segmentationsById = builder.segmentationsById;
            segmentsById = builder.segmentsById;
        }

        public Impl withFolders(List<FolderModel> folders) {
            return new Builder(this).setFolders(folders).build();
        }

        public Impl withQuotas(List<QuotaModel> quotas) {
            return new Builder(this).setQuotas(quotas).build();
        }

        public Impl withResources(List<ResourceModel> resources) {
            return new Builder(this).setResources(resources).build();
        }

        public Impl withResourceTypes(List<ResourceTypeModel> resourceTypes) {
            return new Builder(this).setResourceTypes(resourceTypes).build();
        }

        public Impl withAccounts(List<AccountModel> accounts) {
            return new Builder(this).setAccounts(accounts).build();
        }

        public Impl withAccountsSpaces(ExpandedAccountsSpaces<List<AccountSpaceModel>> spaces) {
            return new Builder(this).setAccountsSpaces(spaces).build();
        }

        public Impl withAccountsQuotas(List<AccountsQuotasModel> accountsQuotas) {
            return new Builder(this).setAccountsQuotas(accountsQuotas).build();
        }

        public Impl withUnitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            return new Builder(this).setUnitsEnsembles(unitsEnsembles).build();
        }

        public Impl withProviders(List<ProviderModel> providers) {
            return new Builder(this).setProviders(providers).build();
        }

        public Impl withFolderPermissions(FoldersPermissionsCollection folderPermissions) {
            return new Builder(this).setPermissionsByFolder(folderPermissions).build();
        }

        public Impl withContinuationToken(String continuationToken) {
            return new Builder(this).setContinuationToken(continuationToken).build();
        }

        public Impl withSegmentationsById(Map<String, ResourceSegmentationModel> segmentationsById) {
            return new Builder(this).setSegmentationsById(segmentationsById).build();
        }

        public Impl withSegmentsById(Map<String, ResourceSegmentModel> segmentsById) {
            return new Builder(this).setSegmentsById(segmentsById).build();
        }

        @Override
        public List<FolderModel> getFolders() {
            return folders;
        }

        @Override
        public List<QuotaModel> getQuotas() {
            return quotas;
        }

        @Override
        public List<ResourceModel> getResources() {
            return resources;
        }

        @Override
        public List<AccountModel> getAccounts() {
            return accounts;
        }

        @Override
        public ExpandedAccountsSpaces<List<AccountSpaceModel>> getAccountsSpaces() {
            return accountsSpaces;
        }

        @Override
        public List<AccountsQuotasModel> getAccountsQuotas() {
            return accountsQuotas;
        }

        @Override
        public List<UnitsEnsembleModel> getUnitsEnsembles() {
            return unitsEnsembles;
        }

        @Override
        public List<ResourceTypeModel> getResourceTypes() {
            return resourceTypes;
        }

        @Override
        public List<ProviderModel> getProviders() {
            return providers;
        }

        @Override
        public FoldersPermissionsCollection getPermissionsByFolder() {
            return permissionsByFolder;
        }

        @Override
        public String getContinuationToken() {
            return continuationToken;
        }

        public Impl get() {
            return this;
        }

        @Override
        public Map<String, ResourceSegmentationModel> getSegmentationsById() {
            return segmentationsById;
        }

        @Override
        public Map<String, ResourceSegmentModel> getSegmentsById() {
            return segmentsById;
        }
    }

    private static class Builder {
        private List<FolderModel> folders;
        private List<QuotaModel> quotas;
        private List<ResourceModel> resources;
        private List<ResourceTypeModel> resourceTypes;
        private List<AccountModel> accounts;
        private ExpandedAccountsSpaces<List<AccountSpaceModel>> accountsSpaces;
        private List<AccountsQuotasModel> accountsQuotas;
        private List<UnitsEnsembleModel> unitsEnsembles;
        private List<ProviderModel> providers;
        private FoldersPermissionsCollection permissionsByFolder;
        private String continuationToken;
        private Map<String, ResourceSegmentationModel> segmentationsById;
        private Map<String, ResourceSegmentModel> segmentsById;

        Builder() {
        }

        Builder(Impl impl) {
            folders = impl.folders;
            quotas = impl.quotas;
            resources = impl.resources;
            resourceTypes = impl.resourceTypes;
            accounts = impl.accounts;
            accountsQuotas = impl.accountsQuotas;
            unitsEnsembles = impl.unitsEnsembles;
            providers = impl.providers;
            continuationToken = impl.continuationToken;
            permissionsByFolder = impl.permissionsByFolder;
            accountsSpaces = impl.accountsSpaces;
            segmentationsById = impl.segmentationsById;
            segmentsById = impl.segmentsById;
        }

        Builder setFolders(List<FolderModel> folders) {
            this.folders = folders;
            return this;
        }

        Builder setQuotas(List<QuotaModel> quotas) {
            this.quotas = quotas;
            return this;
        }

        Builder setResources(List<ResourceModel> resources) {
            this.resources = resources;
            return this;
        }

        Builder setResourceTypes(List<ResourceTypeModel> resourceTypes) {
            this.resourceTypes = resourceTypes;
            return this;
        }

        Builder setAccounts(List<AccountModel> accounts) {
            this.accounts = accounts;
            return this;
        }

        Builder setAccountsSpaces(ExpandedAccountsSpaces<List<AccountSpaceModel>> spaces) {
            this.accountsSpaces = spaces;
            return this;
        }

        Builder setAccountsQuotas(List<AccountsQuotasModel> accountsQuotas) {
            this.accountsQuotas = accountsQuotas;
            return this;
        }

        Builder setUnitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            this.unitsEnsembles = unitsEnsembles;
            return this;
        }

        Builder setProviders(List<ProviderModel> providers) {
            this.providers = providers;
            return this;
        }

        Builder setPermissionsByFolder(FoldersPermissionsCollection permissionsByFolder) {
            this.permissionsByFolder = permissionsByFolder;
            return this;
        }

        Builder setContinuationToken(String continuationToken) {
            this.continuationToken = continuationToken;
            return this;
        }

        Builder setSegmentationsById(Map<String, ResourceSegmentationModel> segmentationsById) {
            this.segmentationsById = segmentationsById;
            return this;
        }

        Builder setSegmentsById(Map<String, ResourceSegmentModel> segmentsById) {
            this.segmentsById = segmentsById;
            return this;
        }

        Impl build() {
            return new Impl(this);
        }
    }
}
