package ru.yandex.intranet.d.services.quotas;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;

/**
 * GetFrontForServiceContext.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 01-03-2021
 */
public class GetFrontForServiceContext {
    public static class Impl implements
            FrontFolderWithQuotesDtoBuilder.WithContinuationToken,
            FrontFolderWithQuotesDtoBuilder.WithFolders,
            FrontFolderWithQuotesDtoBuilder.WithQuotas,
            FrontFolderWithQuotesDtoBuilder.WithResources,
            FrontFolderWithQuotesDtoBuilder.WithResourceTypes,
            FrontFolderWithQuotesDtoBuilder.WithAccounts,
            FrontFolderWithQuotesDtoBuilder.AccountAcceptor<Impl>,
            FrontFolderWithQuotesDtoBuilder.WithAccountsQuotas,
            FrontFolderWithQuotesDtoBuilder.WithUnitsEnsembles,
            FrontFolderWithQuotesDtoBuilder.WithProviders,
            FrontFolderWithQuotesDtoBuilder.WithFolderPermissions,
            FrontFolderWithQuotesDtoBuilder.WithSegmentations,
            FrontFolderWithQuotesDtoBuilder.WithSegments {
        private final FrontFolderWithQuotesDtoContext.Impl impl;
        private final TenantId tenantId;

        public Impl(TenantId tenantId) {
            this.tenantId = tenantId;
            impl = new FrontFolderWithQuotesDtoContext.Impl();
        }

        public Impl(Builder builder) {
            impl = builder.impl;
            tenantId = builder.tenantId;
        }

        public TenantId getTenantId() {
            return tenantId;
        }

        public Impl withResources(List<ResourceModel> resources) {
            return new Builder(this).withResources(resources).build();
        }

        @Override
        public String getContinuationToken() {
            return impl.getContinuationToken();
        }

        public Impl withContinuationToken(String continuationToken) {
            return new Builder(this).withContinuationToken(continuationToken).build();
        }

        @Override
        public List<FolderModel> getFolders() {
            return impl.getFolders();
        }

        public Impl withFolders(List<FolderModel> folders) {
            return new Builder(this).withFolders(folders).build();
        }

        @Override
        public List<String> getFoldersIds() {
            return impl.getFoldersIds();
        }

        @Override
        public List<WithTenant<String>> getFolderIdsWithTenants() {
            return impl.getFolderIdsWithTenants();
        }

        @Override
        public List<Tuple2<String, TenantId>> getResourcesIds() {
            return impl.getResourcesIds();
        }

        @Override
        public List<Tuple2<String, TenantId>> getUnitsEnsemblesIds() {
            return impl.getUnitsEnsemblesIds();
        }

        @Override
        public List<Tuple2<String, TenantId>> getResourceTypesIds() {
            return impl.getResourceTypesIds();
        }

        @Override
        public List<Tuple2<String, TenantId>> getProvidersIds() {
            return impl.getProvidersIds();
        }

        @Override
        public Set<String> getAccountIds() {
            return impl.getAccountIds();
        }

        public Impl withQuotas(List<QuotaModel> quotas) {
            return new Builder(this).withQuotas(quotas).build();
        }

        public Impl withResourceTypes(List<ResourceTypeModel> resourceTypes) {
            return new Builder(this).withResourceTypes(resourceTypes).build();
        }

        @Override
        public Impl withAccounts(List<AccountModel> accounts, ExpandedAccountsSpaces<List<AccountSpaceModel>> spaces) {
            return new Builder(this).withAccounts(accounts).withAccountsSpaces(spaces).build();
        }

        public Impl withAccountsQuotas(List<AccountsQuotasModel> accountsQuotas) {
            return new Builder(this).withAccountsQuotas(accountsQuotas).build();
        }

        public Impl withUnitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            return new Builder(this).withUnitsEnsembles(unitsEnsembles).build();
        }

        public Impl withProviders(List<ProviderModel> providers) {
            return new Builder(this).withProviders(providers).build();
        }

        public Impl withFolderPermissions(FoldersPermissionsCollection folderPermissions) {
            return new Builder(this).withFolderPermissions(folderPermissions).build();
        }

        public Impl withSegmentations(List<ResourceSegmentationModel> segmentationsById) {
            return new Builder(this).withSegmentationsById(segmentationsById.stream()
                    .collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity()))
            ).build();
        }

        public Impl withSegments(List<ResourceSegmentModel> segments) {
            return new Builder(this).withSegmentsById(segments.stream()
                    .collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity()))
            ).build();
        }

        @Override
        public List<QuotaModel> getQuotas() {
            return impl.getQuotas();
        }

        @Override
        public List<ResourceModel> getResources() {
            return impl.getResources();
        }

        @Override
        public List<AccountModel> getAccounts() {
            return impl.getAccounts();
        }

        @Override
        public ExpandedAccountsSpaces<List<AccountSpaceModel>> getAccountsSpaces() {
            return impl.getAccountsSpaces();
        }

        public Map<String, AccountModel> getAccountsById() {
            return impl.getAccounts().stream().collect(Collectors.toMap(AccountModel::getId, Function.identity()));
        }

        @Override
        public List<AccountsQuotasModel> getAccountsQuotas() {
            return impl.getAccountsQuotas();
        }

        @Override
        public List<UnitsEnsembleModel> getUnitsEnsembles() {
            return impl.getUnitsEnsembles();
        }

        @Override
        public List<ResourceTypeModel> getResourceTypes() {
            return impl.getResourceTypes();
        }

        @Override
        public List<ProviderModel> getProviders() {
            return impl.getProviders();
        }

        @Override
        public FoldersPermissionsCollection getPermissionsByFolder() {
            return impl.getPermissionsByFolder();
        }

        @Override
        public Map<String, ResourceSegmentationModel> getSegmentationsById() {
            return impl.getSegmentationsById();
        }

        @Override
        public Map<String, ResourceSegmentModel> getSegmentsById() {
            return impl.getSegmentsById();
        }
    }

    private static class Builder {
        private FrontFolderWithQuotesDtoContext.Impl impl;
        private TenantId tenantId;

        private Builder(Impl impl) {
            this.impl = impl.impl;
            this.tenantId = impl.tenantId;
        }

        Impl build() {
            return new Impl(this);
        }

        public Builder withResources(List<ResourceModel> resources) {
            impl = impl.withResources(resources);
            return this;
        }

        public Builder withContinuationToken(String continuationToken) {
            impl = impl.withContinuationToken(continuationToken);
            return this;
        }

        public Builder withFolders(List<FolderModel> folders) {
            impl = impl.withFolders(folders);
            return this;
        }

        public Builder withQuotas(List<QuotaModel> quotas) {
            impl = impl.withQuotas(quotas);
            return this;
        }

        public Builder withResourceTypes(List<ResourceTypeModel> resourceTypes) {
            impl = impl.withResourceTypes(resourceTypes);
            return this;
        }

        public Builder withAccounts(List<AccountModel> accounts) {
            impl = impl.withAccounts(accounts);
            return this;
        }

        public Builder withAccountsSpaces(ExpandedAccountsSpaces<List<AccountSpaceModel>> spaces) {
            impl = impl.withAccountsSpaces(spaces);
            return this;
        }

        public Builder withAccountsQuotas(List<AccountsQuotasModel> accountsQuotas) {
            impl = impl.withAccountsQuotas(accountsQuotas);
            return this;
        }

        public Builder withUnitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            impl = impl.withUnitsEnsembles(unitsEnsembles);
            return this;
        }

        public Builder withProviders(List<ProviderModel> providers) {
            impl = impl.withProviders(providers);
            return this;
        }

        public Builder withFolderPermissions(FoldersPermissionsCollection folderPermissions) {
            impl = impl.withFolderPermissions(folderPermissions);
            return this;
        }

        public Builder withSegmentationsById(Map<String, ResourceSegmentationModel> segmentationsById) {
            impl = impl.withSegmentationsById(segmentationsById);
            return this;
        }

        public Builder withSegmentsById(Map<String, ResourceSegmentModel> segmentsById) {
            impl = impl.withSegmentsById(segmentsById);
            return this;
        }
    }
}
