package ru.yandex.intranet.d.services.quotas;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;
import ru.yandex.intranet.d.util.result.Result;

/**
 * Validation and loading context for getQuotasInFolder method.
 * Immutable boilerplate for collecting data in Mono chain.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @see QuotasService#getQuotasInFolders
 * @since 19.02.2021
 */
public class GetQuotasInFoldersContext {
    private GetQuotasInFoldersContext() {
    }

    public static Mono<Result<Start>> start() {
        return Mono.just(Result.success(new Builder().build()));
    }

    public static class Impl implements Start, WithUnitsEnsembles {
        private final List<FolderModel> folders;
        private final List<QuotaModel> quotas;
        private final List<ResourceModel> resources;
        private final List<ResourceTypeModel> resourceTypes;
        private final List<AccountModel> accounts;
        private final List<AccountsQuotasModel> accountsQuotas;
        private final List<UnitsEnsembleModel> unitsEnsembles;
        private final List<ProviderModel> providers;
        private final FoldersPermissionsCollection permissionsByFolder;
        private final ExpandedAccountsSpaces<List<AccountSpaceModel>> accountsSpaces;
        private final Map<String, ResourceSegmentationModel> segmentationsById;
        private final Map<String, ResourceSegmentModel> segmentsById;

        Impl(Builder builder) {
            folders = builder.folders;
            quotas = builder.quotas;
            resources = builder.resources;
            resourceTypes = builder.resourceTypes;
            accounts = builder.accounts;
            accountsQuotas = builder.accountsQuotas;
            unitsEnsembles = builder.unitsEnsembles;
            providers = builder.providers;
            permissionsByFolder = builder.permissionsByFolder;
            accountsSpaces = builder.accountsSpaces;
            segmentationsById = builder.segmentationsById;
            segmentsById = builder.segmentsById;
        }

        @Override
        public WithFolders withFolders(List<FolderModel> folders) {
            return new Builder(this).setFolders(folders).build();
        }

        @Override
        public List<FolderModel> getFolders() {
            return folders;
        }

        @Override
        public WithQuotas withQuotas(List<QuotaModel> quotas) {
            return new Builder(this).setQuotas(quotas).build();
        }

        @Override
        public List<QuotaModel> getQuotas() {
            return quotas;
        }

        @Override
        public List<ResourceModel> getResources() {
            return resources;
        }

        @Override
        public List<AccountModel> getAccounts() {
            return accounts;
        }

        @Override
        public ExpandedAccountsSpaces<List<AccountSpaceModel>> getAccountsSpaces() {
            return accountsSpaces;
        }

        @Override
        public List<AccountsQuotasModel> getAccountsQuotas() {
            return accountsQuotas;
        }

        @Override
        public List<UnitsEnsembleModel> getUnitsEnsembles() {
            return unitsEnsembles;
        }

        @Override
        public FoldersPermissionsCollection getPermissionsByFolder() {
            return permissionsByFolder;
        }

        @Override
        public WithResources withResources(List<ResourceModel> resources) {
            return new Builder(this).setResources(resources).build();
        }

        @Override
        public List<ResourceTypeModel> getResourceTypes() {
            return resourceTypes;
        }

        @Override
        public WithResourceTypes withResourceTypes(List<ResourceTypeModel> resourceTypes) {
            return new Builder(this).setResourceTypes(resourceTypes).build();
        }

        @Override
        public WithResources filterUnmanagedResources() {
            List<ResourceModel> mutableResources =
                    resources.stream()
                            .filter(resourceModel ->
                                    resourceModel.isManaged() && !resourceModel.isReadOnly())
                            .collect(Collectors.toList());
            Set<String> mutableResourceIds = mutableResources.stream()
                    .map(ResourceModel::getId)
                    .collect(Collectors.toSet());
            List<QuotaModel> mutableQuotas = quotas.stream()
                    .filter(quotaModel -> mutableResourceIds.contains(quotaModel.getResourceId()))
                    .collect(Collectors.toList());
            return new Builder(this).setQuotas(mutableQuotas).setResources(mutableResources).build();
        }

        @Override
        public WithAccounts withAccounts(List<AccountModel> accounts,
                                         ExpandedAccountsSpaces<List<AccountSpaceModel>> spaces) {
            return new Builder(this).setAccounts(accounts)
                    .setAccountsSpaces(spaces)
                    .build();
        }

        @Override
        public WithAccountsQuotas withAccountsQuotas(List<AccountsQuotasModel> accountsQuotas) {
            return new Builder(this).setAccountsQuotas(accountsQuotas).build();
        }

        @Override
        public WithUnitsEnsembles withUnitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            return new Builder(this).setUnitsEnsembles(unitsEnsembles).build();
        }

        @Override
        public List<ProviderModel> getProviders() {
            return providers;
        }

        @Override
        public WithProviders withProviders(List<ProviderModel> providers) {
            return new Builder(this).setProviders(providers).build();
        }

        @Override
        public WithFolderPermissions withFolderPermissions(FoldersPermissionsCollection folderPermissions) {
            return new Builder(this).setFolderPermissions(folderPermissions).build();
        }

        @Override
        public String getContinuationToken() {
            return null;
        }

        @Override
        public Impl get() {
            return this;
        }

        @Override
        public Map<String, ResourceSegmentationModel> getSegmentationsById() {
            return segmentationsById;
        }

        @Override
        public WithSegmentations withSegmentations(List<ResourceSegmentationModel> segmentations) {
            return new Builder(this).setSegmentationsById(
                    segmentations.stream().collect(
                            Collectors.toMap(ResourceSegmentationModel::getId, Function.identity()))
            ).build();
        }

        @Override
        public Map<String, ResourceSegmentModel> getSegmentsById() {
            return segmentsById;
        }

        @Override
        public WithSegments withSegments(List<ResourceSegmentModel> segments) {
            return new Builder(this).setSegmentsById(
                    segments.stream().collect(
                        Collectors.toMap(ResourceSegmentModel::getId, Function.identity()))
            ).build();
        }
    }

    private static class Builder {
        private List<FolderModel> folders;
        private List<QuotaModel> quotas;
        private List<ResourceModel> resources;
        private List<ResourceTypeModel> resourceTypes;
        private List<AccountModel> accounts;
        private List<AccountsQuotasModel> accountsQuotas;
        private List<UnitsEnsembleModel> unitsEnsembles;
        private List<ProviderModel> providers;
        private FoldersPermissionsCollection permissionsByFolder;
        private ExpandedAccountsSpaces<List<AccountSpaceModel>> accountsSpaces;
        private Map<String, ResourceSegmentationModel> segmentationsById;
        private Map<String, ResourceSegmentModel> segmentsById;

        Builder() {
        }

        Builder(Impl impl) {
            folders = impl.folders;
            quotas = impl.quotas;
            resources = impl.resources;
            resourceTypes = impl.resourceTypes;
            accounts = impl.accounts;
            accountsQuotas = impl.accountsQuotas;
            unitsEnsembles = impl.unitsEnsembles;
            providers = impl.providers;
            permissionsByFolder = impl.permissionsByFolder;
            accountsSpaces = impl.accountsSpaces;
            segmentationsById = impl.segmentationsById;
            segmentsById = impl.segmentsById;
        }

        Builder setFolders(List<FolderModel> folders) {
            this.folders = folders;
            return this;
        }

        Builder setQuotas(List<QuotaModel> quotas) {
            this.quotas = quotas;
            return this;
        }

        Builder setResources(List<ResourceModel> resources) {
            this.resources = resources;
            return this;
        }

        Builder setResourceTypes(List<ResourceTypeModel> resourceTypes) {
            this.resourceTypes = resourceTypes;
            return this;
        }

        Builder setAccounts(List<AccountModel> accounts) {
            this.accounts = accounts;
            return this;
        }

        Builder setAccountsSpaces(ExpandedAccountsSpaces<List<AccountSpaceModel>> spaces) {
            this.accountsSpaces = spaces;
            return this;
        }

        Builder setAccountsQuotas(List<AccountsQuotasModel> accountsQuotas) {
            this.accountsQuotas = accountsQuotas;
            return this;
        }

        Builder setUnitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            this.unitsEnsembles = unitsEnsembles;
            return this;
        }

        Builder setProviders(List<ProviderModel> providers) {
            this.providers = providers;
            return this;
        }

        Builder setFolderPermissions(FoldersPermissionsCollection permissionsByFolder) {
            this.permissionsByFolder = permissionsByFolder;
            return this;
        }

        Builder setSegmentationsById(Map<String, ResourceSegmentationModel> segmentationsById) {
            this.segmentationsById = segmentationsById;
            return this;
        }

        Builder setSegmentsById(Map<String, ResourceSegmentModel> segmentsById) {
            this.segmentsById = segmentsById;
            return this;
        }

        Impl build() {
            return new Impl(this);
        }
    }

    public interface Start extends FrontFolderWithQuotesDtoBuilder.WithContinuationToken {
        WithFolders withFolders(List<FolderModel> folders);
    }

    public interface WithFolders extends FrontFolderWithQuotesDtoBuilder.WithFolders {
        WithQuotas withQuotas(List<QuotaModel> quotas);
    }

    public interface WithQuotas extends FrontFolderWithQuotesDtoBuilder.WithQuotas, WithFolders {
        WithResources withResources(List<ResourceModel> resources);
    }

    public interface WithResources extends FrontFolderWithQuotesDtoBuilder.WithResources, WithQuotas {
        WithResourceTypes withResourceTypes(List<ResourceTypeModel> resourceTypes);
        WithResources filterUnmanagedResources();
    }

    public interface WithResourceTypes extends FrontFolderWithQuotesDtoBuilder.WithResourceTypes, WithResources {
        WithProviders withProviders(List<ProviderModel> providers);
    }

    public interface WithProviders extends FrontFolderWithQuotesDtoBuilder.WithProviders, WithResourceTypes,
            FrontFolderWithQuotesDtoBuilder.AccountAcceptor<WithAccounts> {
    }

    public interface WithAccounts extends FrontFolderWithQuotesDtoBuilder.WithAccounts, WithProviders {
        WithAccountsQuotas withAccountsQuotas(List<AccountsQuotasModel> accountsQuotas);
    }

    public interface WithAccountsQuotas extends FrontFolderWithQuotesDtoBuilder.WithAccountsQuotas, WithAccounts {
        WithFolderPermissions withFolderPermissions(FoldersPermissionsCollection folderPermissions);
    }

    public interface WithFolderPermissions extends FrontFolderWithQuotesDtoBuilder.WithFolderPermissions,
            WithAccountsQuotas {
        WithSegmentations withSegmentations(List<ResourceSegmentationModel> segmentations);
    }

    public interface WithSegmentations extends FrontFolderWithQuotesDtoBuilder.WithSegmentations,
            WithFolderPermissions {
        WithSegments withSegments(List<ResourceSegmentModel> unitsEnsembles);
    }

    public interface WithSegments extends FrontFolderWithQuotesDtoBuilder.WithSegments,
            WithSegmentations {
        WithUnitsEnsembles withUnitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles);
    }

    public interface WithUnitsEnsembles extends FrontFolderWithQuotesDtoBuilder.WithUnitsEnsembles,
            WithSegments {
        Impl get();
    }

}
