package ru.yandex.intranet.d.services.quotas

import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.quotas.QuotaModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.services.operations.model.RetryResult
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.web.security.model.YaUserDetails

/**
 * Нужен для работы с операциями в провайдере при перемещении спущенной квоты.
 * Для применения заявки надо:
 * 1. В той же транзакции, в которой выполняется последнее голосование (возможно, прямо при создании заявки),
 *    создать записи об операции в провайдере (createOperationRecord).
 * 2. Закоммитить транзакцию.
 * 3. Вне транзакции вызвать runOperation (это последнее действие в коде на стороне сервиса заявок).
 * 4. В сервисе заявок надо реализовать TransferRequestCloser.
 *    По завершении операции в нем будет вызван closeTransferRequest, которы должен в той же транзакции,
 *    что и запись результатов операции, поменять статус заявки.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 13-12-2021
 */
interface MoveProvisionLogicService {
    /**
     * Только создает записи об операции (включая первые записи истории в фолдерах).
     * Поскольку прочитать из базы ничего не получится, на вход передается модель заявки целиком.
     * На вход передается открытая транзакция, в которой и происходит запись.
     * Ответственность за коммит транзакции лежит на вызывающем коде.
     */
    suspend fun createMoveProvisionOperationRecord(
        tx: YdbTxSession,
        transferRequestModel: TransferRequestModel,
        context: MoveProvisionApplicationContext,
        currentUser: YaUserDetails
    ): MoveProvisionOperationRecord

    /**
     * На вход получает только id операции.
     * Внутри открывает транзакцию, вычитывает все данные об операции и выполняет её в провайдере.
     * Ничего не возвращает, по завершении операции вызывает closeTransferRequest
     */
    suspend fun runOperation(operationId: String)
}

interface TransferRequestCloser {
    /**
     * Чтобы не мешать вычитать необходимую информацию, вызывается до начала записи в таблицы операций и истории.
     * В конце метода будет запись в базу, так что поле него ничего прочесть не получится, только писать.
     *
     * @return Возвращает лямбду, которую надо вызвать после коммита транзакции.
     */
    suspend fun moveProvisionOperationFinished(
        tx: YdbTxSession,
        transferRequest: TransferRequestModel,
        operationId: String,
        operationResult: MoveProvisionOperationResult,
        borrowedLoanId: LoanId?
    ): suspend () -> Unit
}

data class MoveProvisionOperationResult(
    val requestStatus: RetryResult,
    val opLogIdsByFolderId: Map<String, Set<String>>,
    val errorsRu: ErrorCollection,
    val errorsEn: ErrorCollection,
)

data class MoveProvisionOperationRecord(
    val operationIds: Set<String>,
    val opLogIdsByFolderId: Map<String, Set<String>>,
    val operationIdByProvisionTransfer: Map<ProvisionTransfer, String>
)

data class MoveProvisionApplicationContext(
    val providers: List<ProviderModel>,
    val resources: List<ResourceModel>,
    val accounts: List<AccountModel>,
    val folders: List<FolderModel>,
    val foldersQuotas: List<QuotaModel>,
    val accountsQuotas: List<AccountsQuotasModel>,
)

data class ProvideReserveOperationContext(
    val updatedQuotas: List<QuotaModel>,
    val updatedFolder: FolderModel,
    val opLog: FolderOperationLogModel,
    val operation: AccountsQuotasOperationsModel,
    val operationInProgress: OperationInProgressModel,
)
