package ru.yandex.intranet.d.services.quotas

import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import mu.KotlinLogging
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.Tenants.DEFAULT_TENANT_ID
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao
import ru.yandex.intranet.d.dao.loans.LoansDao
import ru.yandex.intranet.d.dao.quotas.QuotasDao
import ru.yandex.intranet.d.dao.resources.ResourcesDao
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.kotlin.AccountId
import ru.yandex.intranet.d.kotlin.AccountsQuotasOperationsDaoAdapter
import ru.yandex.intranet.d.kotlin.FolderId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.dao.OperationsInProgressDaoAdapter
import ru.yandex.intranet.d.kotlin.dao.TransferRequestsDaoAdapter
import ru.yandex.intranet.d.kotlin.elapsed
import ru.yandex.intranet.d.kotlin.getOrNull
import ru.yandex.intranet.d.kotlin.orFalse
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel.OperationType.MOVE_PROVISION
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel.OperationType.PROVIDE_RESERVE
import ru.yandex.intranet.d.model.accounts.OperationChangesModel
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel
import ru.yandex.intranet.d.model.accounts.OperationSource
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel
import ru.yandex.intranet.d.model.folders.FolderOperationType
import ru.yandex.intranet.d.model.folders.OperationPhase
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel
import ru.yandex.intranet.d.model.folders.ProvisionsByResource
import ru.yandex.intranet.d.model.folders.QuotasByAccount
import ru.yandex.intranet.d.model.folders.QuotasByResource
import ru.yandex.intranet.d.model.loans.LoanType
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.quotas.QuotaModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.transfers.LoanOperationType
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.services.operations.OperationsObservabilityService
import ru.yandex.intranet.d.services.operations.OperationsRetryService
import ru.yandex.intranet.d.services.operations.model.RetryableOperation
import ru.yandex.intranet.d.services.provisions.ReserveProvisionsService
import ru.yandex.intranet.d.util.units.Units
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.time.Instant
import java.util.*
import kotlin.math.max
import kotlin.math.min

private val logger = KotlinLogging.logger {}
private val ALLOWED_OPERATION_TYPES = setOf(MOVE_PROVISION, PROVIDE_RESERVE)

/**
 * MoveProvisionLogicServiceImpl.
 *
 * @author Vladimir Zaytsev <vzay></vzay>@yandex-team.ru>
 * @since 13-12-2021
 */
@Component
class MoveProvisionLogicServiceImpl(
    private val operationsInProgressDao: OperationsInProgressDaoAdapter,
    private val accountsQuotasOperationsDao: AccountsQuotasOperationsDaoAdapter,
    private val folderOperationLogDao: FolderOperationLogDao,
    private val accountsQuotasDao: AccountsQuotasDao,
    private val folderDao: FolderDao,
    private val operationsRetryService: OperationsRetryService,
    private val transferRequestsDao: TransferRequestsDaoAdapter,
    private val providersLoader: ProvidersLoader,
    private val operationsObservabilityService: OperationsObservabilityService,
    private val reserveProvisionsService: ReserveProvisionsService,
    private val quotasDao: QuotasDao,
    private val loansDao: LoansDao,
    private val resourcesDao: ResourcesDao,
) : MoveProvisionLogicService {

    companion object {
        @JvmStatic
        fun generateFolderOperationLogs(
            sourceAccountId: AccountId,
            destinationAccountId: AccountId,
            sourceFolderId: FolderId,
            destinationFolderId: FolderId,
            operation: AccountsQuotasOperationsModel,
            folderQuotaByFolderIdResourceId: Map<FolderId, Map<ResourceId, QuotaModel>>,
            accountQuotaByAccountIdResourceId: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>,
            sourceDeltaByResourceId: Map<ResourceId, Long>,
            destinationDeltaByResourceId: Map<ResourceId, Long>,
            operationPhase: OperationPhase,
            sourceFolderOpLogOrder: Long,
            destinationFolderOpLogOrder: Long,
            updatedAccountQuotas: MutableList<AccountsQuotasModel>
        ): List<FolderOperationLogModel> {
            val folderOperationLog = mutableListOf<FolderOperationLogModel>()

            val newSourceProvisionsByResourceId = operation.requestedChanges.updatedProvisions.get()
                .associateBy({ it.resourceId }, { it.amount })
            val newDestinationProvisionsByResourceId = operation.requestedChanges.updatedDestinationProvisions.get()
                .associateBy({ it.resourceId }, { it.amount })
            val sourceFolderQuotas: Map<ResourceId, QuotaModel>
            val destinationFolderQuotas: Map<ResourceId, QuotaModel>
            if (sourceFolderId == destinationFolderId) {
                sourceFolderQuotas = mapOf()
                destinationFolderQuotas = mapOf()
            } else {
                sourceFolderQuotas = folderQuotaByFolderIdResourceId[sourceFolderId] ?: mapOf()
                destinationFolderQuotas = folderQuotaByFolderIdResourceId[destinationFolderId] ?: mapOf()
            }
            val newSourceFolderQuotas = sourceFolderQuotas
                .filter { (resourceId, _) -> newSourceProvisionsByResourceId.containsKey(resourceId) }
                .map { (resourceId, quotaModel) ->
                    calculateNewQuotaModel(
                        delta = sourceDeltaByResourceId[resourceId] ?: 0L,
                        quotaModel,
                        destinationFolderQuotas[resourceId]
                    )
                }
            val newDestFolderQuotas = destinationFolderQuotas
                .filter { (resourceId, _) -> newSourceProvisionsByResourceId.containsKey(resourceId) }
                .map { (resourceId, quotaModel) ->
                    calculateNewQuotaModel(
                        delta = destinationDeltaByResourceId[resourceId] ?: 0L,
                        quotaModel,
                        sourceFolderQuotas[resourceId]
                    )
                }

            val sourceAccountQuotas = accountQuotaByAccountIdResourceId[sourceAccountId] ?: mapOf()
            val destinationAccountQuotas = accountQuotaByAccountIdResourceId[destinationAccountId] ?: mapOf()
            operation.requestedChanges.frozenProvisions.orElse(listOf()).forEach { p ->
                val quota = sourceAccountQuotas[p.resourceId] ?: return@forEach
                val updatedQuota = AccountsQuotasModel.Builder(quota)
                    .setFrozenProvidedQuota(checkedSum(quota.frozenProvidedQuota, p.amount))
                    .build()
                updatedAccountQuotas.add(updatedQuota)
            }
            operation.requestedChanges.frozenDestinationProvisions.orElse(listOf()).forEach { p ->
                val quota = destinationAccountQuotas[p.resourceId] ?: return@forEach
                val updatedQuota = AccountsQuotasModel.Builder(quota)
                    .setFrozenProvidedQuota(checkedSum(quota.frozenProvidedQuota, p.amount))
                    .build()
                updatedAccountQuotas.add(updatedQuota)
            }

            val sourceFolderOpLogId = UUID.randomUUID().toString()
            val destinationFolderOpLogId = UUID.randomUUID().toString()
            folderOperationLog.add(
                prepareFolderOperationLog(
                    opLogId = sourceFolderOpLogId,
                    operation = operation,
                    accountId = sourceAccountId,
                    folderId = sourceFolderId,
                    oldSourceQuotas = sourceFolderQuotas.values.toList(),
                    oldAccountsQuotas = sourceAccountQuotas.values.toList(),
                    newSourceQuotas = newSourceFolderQuotas,
                    newProvisionsByResourceId = newSourceProvisionsByResourceId,
                    destinationFolderOpLogId = destinationFolderOpLogId,
                    operationPhase = operationPhase,
                    order = sourceFolderOpLogOrder
                )
            )
            folderOperationLog.add(
                prepareFolderOperationLog(
                    opLogId = destinationFolderOpLogId,
                    operation = operation,
                    accountId = destinationAccountId,
                    folderId = destinationFolderId,
                    oldSourceQuotas = destinationFolderQuotas.values.toList(),
                    oldAccountsQuotas = destinationAccountQuotas.values.toList(),
                    newSourceQuotas = newDestFolderQuotas,
                    newProvisionsByResourceId = newDestinationProvisionsByResourceId,
                    sourceFolderOpLogId = sourceFolderOpLogId,
                    operationPhase = operationPhase,
                    order = destinationFolderOpLogOrder
                )
            )
            return folderOperationLog
        }

        private fun calculateNewQuotaModel(
            delta: Long, quotaModel: QuotaModel, anotherFolderQuota: QuotaModel?
        ): QuotaModel {
            val updatedQuota = if (delta < 0) {
                max(checkedSum(quotaModel.quota, delta), 0)
            } else {
                checkedSum(quotaModel.quota, min(delta, anotherFolderQuota?.quota ?: delta))
            }
            val quotaDelta = checkedSum(updatedQuota, -quotaModel.quota)
            val balanceDelta = checkedSum(quotaDelta, -delta)
            val balance = checkedSum(quotaModel.balance, balanceDelta)
            return QuotaModel.builder(quotaModel)
                .balance(balance)
                .quota(updatedQuota)
                .build()
        }

        private fun checkedSum(x: Long, y: Long) = Units.add(x, y)
            .orElseThrow { IllegalArgumentException("Overflow with sum: $x + $y") }

        private fun prepareFolderOperationLog(
            opLogId: String,
            operation: AccountsQuotasOperationsModel,
            accountId: String,
            folderId: String,
            oldSourceQuotas: List<QuotaModel>,
            oldAccountsQuotas: List<AccountsQuotasModel>,
            newSourceQuotas: List<QuotaModel>,
            newProvisionsByResourceId: Map<String, Long>,
            operationPhase: OperationPhase,
            order: Long,
            sourceFolderOpLogId: String? = null,
            destinationFolderOpLogId: String? = null,

        ): FolderOperationLogModel {
            val affectedOldQuotas = oldSourceQuotas.filter { newProvisionsByResourceId.containsKey(it.resourceId) }
            val oldQuotasByResource = affectedOldQuotas.associateBy({ it.resourceId }, { it.quota })
            val oldBalanceByResource = affectedOldQuotas.associateBy({ it.resourceId }, { it.balance })
            val oldProvisionHistoryModelByResource = oldAccountsQuotas
                .filter { accountId == it.accountId && newProvisionsByResourceId.containsKey(it.resourceId) }
                .associateBy(
                    { it.resourceId },
                    { ProvisionHistoryModel(it.providedQuota, it.lastReceivedProvisionVersion.orElse(null)) }
                )
            val oldQuotasByAccountMap = mapOf(
                accountId to ProvisionsByResource(oldProvisionHistoryModelByResource)
            )
            val newQuotasByResource = newSourceQuotas.associateBy({ it.resourceId }, { it.quota })
            val newBalanceByResource = newSourceQuotas.associateBy({ it.resourceId }, { it.balance })
            val newQuotasByAccountMap = mapOf(
                accountId to ProvisionsByResource(newProvisionsByResourceId.mapValues { (key, value) ->
                    ProvisionHistoryModel(
                        value,
                        oldProvisionHistoryModelByResource[key]?.version?.map { it + 1L }?.orElse(null) ?: 0L
                    )
                })
            )
            return FolderOperationLogModel.builder()
                .setTenantId(operation.tenantId)
                .setFolderId(folderId)
                .setOperationDateTime(operation.createDateTime)
                .setId(opLogId)
                .setProviderRequestId(operation.lastRequestId.orElseThrow())
                .setOperationType(FolderOperationType.PROVISION_TRANSFER)
                .setAuthorUserId(operation.authorUserId)
                .setAuthorUserUid(operation.authorUserUid.orElse(null))
                .setAuthorProviderId(null)
                .setSourceFolderOperationsLogId(sourceFolderOpLogId)
                .setDestinationFolderOperationsLogId(destinationFolderOpLogId)
                .setOldFolderFields(null)
                .setOldQuotas(QuotasByResource(oldQuotasByResource))
                .setOldBalance(QuotasByResource(oldBalanceByResource))
                .setOldProvisions(QuotasByAccount(oldQuotasByAccountMap))
                .setOldAccounts(null)
                .setOldFolderFields(null)
                .setNewQuotas(QuotasByResource(newQuotasByResource))
                .setNewBalance(QuotasByResource(newBalanceByResource))
                .setNewProvisions(QuotasByAccount(newQuotasByAccountMap))
                .setActuallyAppliedProvisions(null)
                .setNewAccounts(null)
                .setAccountsQuotasOperationsId(operation.operationId)
                .setQuotasDemandsId(null)
                .setOperationPhase(operationPhase)
                .setOrder(order)
                .build()
        }
    }

    override suspend fun createMoveProvisionOperationRecord(
        tx: YdbTxSession,
        transferRequestModel: TransferRequestModel,
        context: MoveProvisionApplicationContext,
        currentUser: YaUserDetails
    ): MoveProvisionOperationRecord {
        val sourceFolderIds = transferRequestModel.parameters.provisionTransfers.map {
            it.sourceFolderId
        }.toSet()
        if (sourceFolderIds.size != 1) {
            throw IllegalArgumentException("Multiple source folders unsupported.")
        }
        val destinationFolderIds = transferRequestModel.parameters.provisionTransfers.map {
            it.destinationFolderId
        }.toSet()
        if (destinationFolderIds.size != 1) {
            throw IllegalArgumentException("Multiple destination folders unsupported.")
        }
        val sourceFolder: FolderModel = context.folders.find { it.id == sourceFolderIds.first() }!!
        val destinationFolder: FolderModel = context.folders.find { it.id == destinationFolderIds.first() }!!
        val providersById = context.providers.associateBy { it.id }
        val accountsById = context.accounts.associateBy { it.id }

        val tenantId = transferRequestModel.tenantId
        val operations = mutableListOf<AccountsQuotasOperationsModel>()
        val operationsInProgress = mutableListOf<OperationInProgressModel>()
        val folderOperationLog = mutableListOf<FolderOperationLogModel>()
        val updatedQuotas = mutableListOf<QuotaModel>()

        val accountQuotasByAccountResource = context.accountsQuotas.groupBy { it.accountId }
            .mapValues { (_, v) ->
                v.associateBy { it.resourceId }
            }
        val folderQuotasByFolderResource = context.foldersQuotas.groupBy { it.folderId }
            .mapValues { (_, v) ->
                v.associateBy { it.resourceId }
            }

        val operationIdByProvisionTransfer = mutableMapOf<ProvisionTransfer, String>()
        val updatedAccountQuotas = mutableListOf<AccountsQuotasModel>()
        val loanId = transferRequestModel.loanMeta.map { it.payOffLoanId }.orElse(null)
        val loan = if (loanId != null) {
            loansDao.getById(tx, loanId, DEFAULT_TENANT_ID)
        } else {
            null
        }
        val providersResources = resourcesDao.getAllByProviders(tx, providersById.keys, tenantId, true).awaitSingle()
        val resourcesById = providersResources.associateBy { it.id }
        transferRequestModel.parameters.provisionTransfers
            .forEach { transfer ->
                if (transferRequestModel.loanMeta.getOrNull()?.provideOverCommitReserve.orFalse()
                    || LoanType.PROVIDER_RESERVE_OVER_COMMIT == loan?.type) {
                    val sourceAccount = accountsById[transfer.sourceAccountId]!!
                    val destAccount = accountsById[transfer.destinationAccountId]!!
                    val provider = providersById[transfer.providerId]!!
                    processReserveProvisionTransfer(transferRequestModel, transfer, provider, sourceAccount, destAccount,
                        sourceFolder, destinationFolder, resourcesById, folderQuotasByFolderResource,
                        accountQuotasByAccountResource, operations, operationIdByProvisionTransfer, operationsInProgress,
                        folderOperationLog, updatedQuotas, currentUser, Locales.ENGLISH)
                } else {
                    processMoveProvisionTransfer(transferRequestModel, tenantId, currentUser, transfer, sourceFolder,
                        destinationFolder, accountQuotasByAccountResource, operations, operationIdByProvisionTransfer,
                        operationsInProgress, folderOperationLog, folderQuotasByFolderResource, updatedAccountQuotas
                    )
                }
            }

        if (updatedAccountQuotas.isNotEmpty()) {
            meter({
                accountsQuotasDao.upsertAllRetryable(tx, updatedAccountQuotas).awaitSingleOrNull()
            }, "Move provision, write account quotas")
        }
        meter({
            accountsQuotasOperationsDao.upsertAllRetryable(tx, operations)
        }, "Move provision, write operation")
        operations.forEach { operationsObservabilityService.observeOperationSubmitted(it) }
        meter({
            operationsInProgressDao.upsertAllRetryable(tx, operationsInProgress)
        }, "Move provision, write operation in progress")
        meter({
            folderOperationLogDao.upsertAllRetryable(tx, folderOperationLog).awaitSingleOrNull()
        }, "Move provision, write history log")
        meter({
            folderDao.upsertAllRetryable(
                tx, listOf(
                    sourceFolder.incrementOpLogOrder(),
                    destinationFolder.incrementOpLogOrder()
                )
            ).awaitSingleOrNull()
        }, "Move provision, update folder")
        if (updatedQuotas.isNotEmpty()) {
            meter({
                quotasDao.upsertAllRetryable(tx, updatedQuotas).awaitSingleOrNull()
            }, "Move provision, write folder quotas")
        }

        val opLogIdsByFolderId = folderOperationLog
            .groupBy { it.folderId }
            .mapValues { (_, value) -> value.map { it.id }.toSet() }
        return MoveProvisionOperationRecord(
            operations.map { it.operationId }.toSet(),
            opLogIdsByFolderId,
            operationIdByProvisionTransfer
        )
    }

    private fun processReserveProvisionTransfer(
        transferRequestModel: TransferRequestModel,
        provisionTransfer: ProvisionTransfer,
        providerModel: ProviderModel,
        sourceAccount: AccountModel,
        destinationAccount: AccountModel,
        sourceFolder: FolderModel,
        destinationFolder: FolderModel,
        resourcesById: Map<ResourceId, ResourceModel>,
        folderQuotas: Map<FolderId, Map<ResourceId, QuotaModel>>,
        accountQuotasByAccountResource: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>,
        operations: MutableList<AccountsQuotasOperationsModel>,
        operationIdByProvisionTransfer: MutableMap<ProvisionTransfer, String>,
        operationsInProgress: MutableList<OperationInProgressModel>,
        folderOperationLog: MutableList<FolderOperationLogModel>,
        quotas: MutableList<QuotaModel>,
        currentUser: YaUserDetails,
        locale: Locale
    ) {
        val payOff = if (transferRequestModel.loanMeta.isPresent) {
            transferRequestModel.loanMeta.get().operationType == LoanOperationType.PAY_OFF
        } else {
            throw IllegalArgumentException("Reserve operation without loan!")
        }
        val (currentAccount, currentFolder) = if (payOff) {
            Pair(sourceAccount, sourceFolder)
        } else {
            Pair(destinationAccount, destinationFolder)
        }
        val (updatedQuotas, updatedFolder, opLog, operation, operationInProgress) = reserveProvisionsService
            .prepareReserveProvisionForTransferRequest(transferRequestModel, provisionTransfer, payOff,
                currentAccount, currentFolder, providerModel, provisionTransfer.accountsSpacesId, resourcesById,
                accountQuotasByAccountResource, folderQuotas[currentFolder.id] ?: mapOf(), currentUser,
                locale
        )
        operations += operation
        operationIdByProvisionTransfer[provisionTransfer] = operation.operationId
        operationsInProgress += operationInProgress
        folderOperationLog += opLog
        quotas += updatedQuotas
    }

    private fun processMoveProvisionTransfer(
        transferRequestModel: TransferRequestModel,
        tenantId: TenantId,
        currentUser: YaUserDetails,
        transfer: ProvisionTransfer,
        sourceFolder: FolderModel,
        destinationFolder: FolderModel,
        accountQuotasByAccountResource: Map<String, Map<String, AccountsQuotasModel>>,
        operations: MutableList<AccountsQuotasOperationsModel>,
        operationIdByProvisionTransfer: MutableMap<ProvisionTransfer, String>,
        operationsInProgress: MutableList<OperationInProgressModel>,
        folderOperationLog: MutableList<FolderOperationLogModel>,
        folderQuotasByFolderResource: Map<String, Map<String, QuotaModel>>,
        updatedAccountQuotas: MutableList<AccountsQuotasModel>
    ) {
        val operation = prepareOperation(
            transferRequestModel.id, tenantId, currentUser, transfer, sourceFolder.nextOpLogOrder,
            destinationFolder.nextOpLogOrder, accountQuotasByAccountResource
        )
        operations.add(operation)
        operationIdByProvisionTransfer[transfer] = operation.operationId

        operationsInProgress.add(
            OperationInProgressModel(
                tenantId,
                operation.operationId,
                transfer.sourceFolderId,
                transfer.sourceAccountId,
                0L
            )
        )
        operationsInProgress.add(
            OperationInProgressModel(
                tenantId,
                operation.operationId,
                transfer.destinationFolderId,
                transfer.destinationAccountId,
                0L
            )
        )

        val sourceDeltas = transfer.sourceAccountTransfers.associateBy({ it.resourceId }, { it.delta })
        val destinationDeltas = transfer.destinationAccountTransfers.associateBy({ it.resourceId }, { it.delta })
        folderOperationLog.addAll(
            generateFolderOperationLogs(
                sourceAccountId = transfer.sourceAccountId,
                destinationAccountId = transfer.destinationAccountId,
                sourceFolderId = sourceFolder.id,
                destinationFolderId = destinationFolder.id,
                operation = operation,
                folderQuotaByFolderIdResourceId = folderQuotasByFolderResource,
                accountQuotaByAccountIdResourceId = accountQuotasByAccountResource,
                sourceDeltaByResourceId = sourceDeltas,
                destinationDeltaByResourceId = destinationDeltas,
                updatedAccountQuotas = updatedAccountQuotas,
                operationPhase = OperationPhase.SUBMIT,
                sourceFolderOpLogOrder = sourceFolder.nextOpLogOrder,
                destinationFolderOpLogOrder = destinationFolder.nextOpLogOrder
            )
        )
    }

    override suspend fun runOperation(operationId: String) {
        val operation = accountsQuotasOperationsDao.getById(operationId)
        if (operation == null || operation.operationType !in ALLOWED_OPERATION_TYPES) {
            return
        }
        val transferRequest = transferRequestsDao.getById(operation.requestedChanges.transferRequestId.get())!!
        val operationInProgressKeys = transferRequest.parameters.provisionTransfers
            .flatMap { listOf(it.sourceFolderId, it.destinationFolderId) }
            .distinct()
            .map { folderId ->
                OperationInProgressModel.Key(operationId, folderId)
            }
        val operationInProgress = operationsInProgressDao.getByIds(operationInProgressKeys)
        val provider = providersLoader.getProviderByIdImmediate(operation.providerId, DEFAULT_TENANT_ID)
                .awaitSingleOrNull()!!.get()
        operationsRetryService.retryOneOperation(
            RetryableOperation(operation, operationInProgress),
            mapOf(provider.id to provider)
        ).awaitSingleOrNull()
    }

    private fun prepareOperation(
        transferRequestId: String,
        tenantId: TenantId,
        currentUser: YaUserDetails,
        transfer: ProvisionTransfer,
        sourceOrder: Long,
        destinationOrder: Long,
        accountQuotasByAccountResource: Map<AccountId, Map<ResourceId, AccountsQuotasModel>>
    ): AccountsQuotasOperationsModel {
        val operation: AccountsQuotasOperationsModel = AccountsQuotasOperationsModel.builder()
            .setTenantId(tenantId)
            .setOperationId(UUID.randomUUID().toString())
            .setLastRequestId(UUID.randomUUID().toString())
            .setCreateDateTime(Instant.now())
            .setOperationSource(OperationSource.USER)
            .setOperationType(MOVE_PROVISION)
            .setAuthorUserId(currentUser.user.orElseThrow().id)
            .setAuthorUserUid(currentUser.user.orElseThrow().passportUid.orElseThrow())
            .setProviderId(transfer.providerId)
            .setAccountsSpaceId(transfer.accountsSpacesId)
            .setUpdateDateTime(null)
            .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.WAITING)
            .setErrorMessage(null)
            .setRequestedChanges(
                OperationChangesModel.builder()
                    .accountId(transfer.sourceAccountId)
                    .transferRequestId(transferRequestId)
                    .updatedProvisions(transfer.sourceAccountTransfers.map {
                        val quota = accountQuotasByAccountResource[transfer.sourceAccountId]?.get(it.resourceId)
                        OperationChangesModel.Provision(
                            it.resourceId, checkedSum(quota?.providedQuota ?: 0L, it.delta)
                        )
                    })
                    .frozenProvisions(transfer.sourceAccountTransfers.filter { it.delta < 0 }.map {
                        OperationChangesModel.Provision(it.resourceId, -it.delta)
                    })
                    .destinationAccountId(transfer.destinationAccountId)
                    .updatedDestinationProvisions(transfer.destinationAccountTransfers.map {
                        val quota = accountQuotasByAccountResource[transfer.destinationAccountId]?.get(it.resourceId)
                        OperationChangesModel.Provision(it.resourceId, checkedSum(quota?.providedQuota ?: 0L, it.delta))
                    })
                    .frozenDestinationProvisions(transfer.destinationAccountTransfers
                        .filter { it.delta < 0 }.map {
                        OperationChangesModel.Provision(it.resourceId, -it.delta)
                    })
                    .build()
            )
            .setOrders(OperationOrdersModel.builder()
                .submitOrder(sourceOrder)
                .destinationSubmitOrder(destinationOrder)
                .build())
            .setErrorKind(null)
            .build()
        return operation
    }

    private fun FolderModel.incrementOpLogOrder(): FolderModel {
        return this.toBuilder()
            .setNextOpLogOrder(this.nextOpLogOrder + 1L).build()
    }

    private inline fun <T> meter(
        statement: () -> T,
        label: String
    ): T {
        return elapsed(statement)
        { millis, success -> logger.info { "$label: duration = $millis ms, success = $success" } }
    }
}
