package ru.yandex.intranet.d.services.quotas;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.providers.RelatedCoefficient;
import ru.yandex.intranet.d.model.providers.RelatedResourceMapping;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.util.FrontStringUtil;
import ru.yandex.intranet.d.util.ProviderUtil;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.errors.ValidationMessages;
import ru.yandex.intranet.d.web.model.AmountDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionDryRunAmounts;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionDryRunAnswerDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionDryRunFolderQuotaDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionDryRunRequestDto;
import ru.yandex.intranet.d.web.model.quotas.ValidatedUpdateProvisionDryRunAmounts;
import ru.yandex.intranet.d.web.model.quotas.ValidatedUpdateProvisionDryRunFolderQuota;
import ru.yandex.intranet.d.web.model.quotas.ValidatedUpdateProvisionDryRunRequest;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.services.quotas.QuotasHelper.isInteger;
import static ru.yandex.intranet.d.services.quotas.QuotasHelper.roundForDisplay;
import static ru.yandex.intranet.d.util.Util.isEmpty;
import static ru.yandex.intranet.d.util.units.Units.convert;

/**
 * Update provision dry run service.
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class ProvisionDryRunService {

    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final ResourcesLoader resourcesLoader;
    private final ProvidersLoader providersLoader;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;

    public ProvisionDryRunService(@Qualifier("messageSource") MessageSource messages,
                                  SecurityManagerService securityManagerService,
                                  ResourcesLoader resourcesLoader,
                                  ProvidersLoader providersLoader,
                                  UnitsEnsemblesLoader unitsEnsemblesLoader) {
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.resourcesLoader = resourcesLoader;
        this.providersLoader = providersLoader;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
    }

    public Mono<Result<UpdateProvisionDryRunAnswerDto>> updateProvisionDryRun(
            UpdateProvisionDryRunRequestDto request,
            YaUserDetails currentUser,
            Locale locale
    ) {
        TenantId tenant = Tenants.getTenantId(currentUser);
        return validateRequest(request, locale)
                .andThenMono(req -> securityManagerService.checkReadPermissions(currentUser, locale)
                .flatMap(r -> r.andThenMono(unused ->
                        resourcesLoader.getResourceByIdImmediate(req.getResourceId(), tenant)
                .flatMap(mainResourceOptional -> validateMainResource(mainResourceOptional, locale)
                .andThenMono(mainResource -> unitsEnsemblesLoader.getUnitsEnsembleByIdImmediate(
                        mainResource.getUnitsEnsembleId(), tenant)
                .flatMap(mainUnitsEnsemble -> mainUnitsEnsemble.isEmpty() ?
                        Mono.error(new IllegalStateException("Units ensemble not found")) :
                        providersLoader.getProviderByIdImmediate(mainResource.getProviderId(), tenant)
                                .flatMap(providerOptional -> validateProvider(providerOptional, locale))
                .flatMap(resultProvider -> resultProvider.andThenMono(provider ->
                        getRelatedResourceIds(provider, req.getResourceId(), tenant)
                .flatMap(relatedResourceIds -> resourcesLoader.getResourcesByIdsImmediate(relatedResourceIds)
                        .map(relatedResources ->
                                validateRelatedResources(relatedResources, relatedResourceIds, req.getResourceId()))
                .flatMap(relatedResources -> getUnitsEnsembleIds(relatedResources, tenant)
                .flatMap(relatedUnitsEnsembleIds -> unitsEnsemblesLoader
                        .getUnitsEnsemblesByIdsImmediate(relatedUnitsEnsembleIds)
                        .map(relatedUnitsEnsembles ->
                                validateRelatedUnitsEnsembles(relatedUnitsEnsembles, relatedUnitsEnsembleIds))
                .map(relatedUnitsEnsembles -> calculateNewUpdateProvisionValues(req, mainResource, provider,
                        mainUnitsEnsemble.get(), locale).andThen(answer ->
                                calculateNewUpdateProvisionValuesWithRelatedResources(req, answer, mainResource,
                                        mainUnitsEnsemble.get(), relatedResources, relatedUnitsEnsembles,
                                        provider, locale))))))))))))));
    }

    private Result<ResourceModel> validateMainResource(Optional<ResourceModel> mainResourceOptional, Locale locale) {
        if (mainResourceOptional.isEmpty()) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.notFound(
                    messages.getMessage("errors.resource.not.found", null, locale))).build());
        }
        ResourceModel mainResource = mainResourceOptional.get();
        if (mainResource.isDeleted()) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.badRequest(
                    messages.getMessage("errors.resource.deleted", null, locale))).build());
        }
        if (mainResource.isReadOnly()) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.badRequest(
                    messages.getMessage("errors.resource.is.read.only", null, locale))).build());
        }
        if (!mainResource.isManaged()) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.badRequest(
                    messages.getMessage("errors.resource.not.managed", null, locale))).build());
        }
        return Result.success(mainResource);
    }

    private Mono<Result<ProviderModel>> validateProvider(Optional<ProviderModel> providerOptional, Locale locale) {
        if (providerOptional.isEmpty()) {
            return Mono.error(new IllegalStateException("Provider not found"));
        }
        ProviderModel provider = providerOptional.get();
        if (provider.isDeleted()) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.badRequest(
                    messages.getMessage("errors.provider.deleted", null, locale))).build()));
        }
        if (provider.isReadOnly()) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.badRequest(
                    messages.getMessage("errors.provider.is.read.only", null, locale))).build()));
        }
        if (!provider.isManaged()) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.badRequest(
                    messages.getMessage("errors.provider.is.not.managed", null, locale))).build()));
        }
        return Mono.just(Result.success(provider));
    }

    private Mono<List<Tuple2<String, TenantId>>> getRelatedResourceIds(ProviderModel provider, String resourceId,
                                                                       TenantId tenantId) {
        if (providerHasNotRelatedResources(provider, resourceId)) {
            return Mono.just(Collections.emptyList());
        }
        //noinspection OptionalGetWithoutIsPresent -- providerHasNotRelatedResources checks presents
        return Mono.just(provider.getRelatedResourcesByResourceId().get().get(resourceId)
                .getRelatedCoefficientMap().keySet().stream()
                .distinct()
                .map(relatedResourceId -> Tuples.of(relatedResourceId, tenantId))
                .collect(Collectors.toList()));
    }

    private List<ResourceModel> validateRelatedResources(List<ResourceModel> relatedResources,
                                                         List<Tuple2<String, TenantId>> relatedResourceIds,
                                                         String mainResourceId) {
        Map<String, ResourceModel> relatedResourceById = relatedResources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        if (relatedResources.size() != relatedResourceIds.size()) {
            Set<String> notFoundResourcesIds = relatedResourceIds.stream().map(Tuple2::getT1)
                    .filter(id -> !relatedResourceById.containsKey(id))
                    .collect(Collectors.toSet());
            throw new IllegalStateException("Related resources not found: " + notFoundResourcesIds);
        }
        return relatedResourceIds.stream()
                .map(Tuple2::getT1)
                .filter(id -> !relatedResourceById.get(id).isReadOnly() && !relatedResourceById.get(id).isDeleted() &&
                        relatedResourceById.get(id).isManaged() && !id.equals(mainResourceId))
                .map(relatedResourceById::get)
                .collect(Collectors.toList());
    }

    private Mono<List<Tuple2<String, TenantId>>> getUnitsEnsembleIds(List<ResourceModel> resources, TenantId tenantId) {
        return Mono.just(resources.stream()
                .map(ResourceModel::getUnitsEnsembleId)
                .distinct()
                .map(unitsEnsembleId -> Tuples.of(unitsEnsembleId, tenantId))
                .collect(Collectors.toList()));
    }

    private List<UnitsEnsembleModel> validateRelatedUnitsEnsembles(
            List<UnitsEnsembleModel> relatedUnitsEnsembles, List<Tuple2<String, TenantId>> relatedUnitsEnsembleIds) {
        Map<String, UnitsEnsembleModel> relatedUnitsEnsembleById = relatedUnitsEnsembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        if (relatedUnitsEnsembles.size() != relatedUnitsEnsembleIds.size()) {
            Set<String> notFoundUnitsEnsembleIds = relatedUnitsEnsembleIds.stream().map(Tuple2::getT1)
                    .filter(id -> !relatedUnitsEnsembleById.containsKey(id))
                    .collect(Collectors.toSet());
            throw new IllegalStateException("Units ensembles of related resource not found: " +
                    notFoundUnitsEnsembleIds);
        }
        return relatedUnitsEnsembles;
    }

    private boolean providerHasNotRelatedResources(ProviderModel provider, String mainResourceId) {
        Optional<Map<String, RelatedResourceMapping>> relatedResourcesMapping =
                provider.getRelatedResourcesByResourceId();
        return relatedResourcesMapping.isEmpty() ||
                !relatedResourcesMapping.get().containsKey(mainResourceId) ||
                relatedResourcesMapping.get().get(mainResourceId).getRelatedCoefficientMap() == null;
    }

    @SuppressWarnings("ParameterNumber")
    private Result<UpdateProvisionDryRunAnswerDto> calculateNewUpdateProvisionValuesWithRelatedResources(
            ValidatedUpdateProvisionDryRunRequest req, UpdateProvisionDryRunAnswerDto answer,
            ResourceModel mainResource, UnitsEnsembleModel mainUnitsEnsemble, List<ResourceModel> relatedResources,
            List<UnitsEnsembleModel> relatedUnitsEnsembles, ProviderModel provider, Locale locale) {
        if (providerHasNotRelatedResources(provider, mainResource.getId())) {
            return Result.success(new UpdateProvisionDryRunAnswerDto.Builder(answer).build());
        }

        UnitModel forEditUnit = mainUnitsEnsemble.unitById(answer.getForEditUnitId())
                .orElseThrow(() -> new IllegalStateException("New edit unit not found"));
        UnitModel baseUnit = mainUnitsEnsemble.unitById(mainResource.getBaseUnitId())
                .orElseThrow(() -> new IllegalStateException("Base unit not found"));
        Map<String, ResourceModel> relatedResourcesById = relatedResources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, UnitsEnsembleModel> relatedUnitsEnsembleById = relatedUnitsEnsembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));

        BigDecimal newDelta = convert(new BigDecimal(answer.getProvidedDelta()), forEditUnit, baseUnit);
        //noinspection OptionalGetWithoutIsPresent -- providerHasNotRelatedResources checks presents
        Map<String, RelatedResourceMapping> relatedResourceMappingByResourceId =
                provider.getRelatedResourcesByResourceId().get();
        Result<Map<String, UpdateProvisionDryRunAnswerDto>> relatedResourcesValuesResult =
                calculateRelatedResourcesValues(relatedResourcesById, relatedUnitsEnsembleById,
                        relatedResourceMappingByResourceId.get(req.getResourceId()).getRelatedCoefficientMap(),
                        req.getFolderQuotasByResourceId().values(), newDelta, locale);
        return relatedResourcesValuesResult.apply(relatedResourcesValues ->
                new UpdateProvisionDryRunAnswerDto.Builder(answer).setRelatedResources(relatedResourcesValues).build());
    }

    private Result<UpdateProvisionDryRunAnswerDto> calculateNewUpdateProvisionValues(
            ValidatedUpdateProvisionDryRunRequest req, ResourceModel resource, ProviderModel provider,
            UnitsEnsembleModel unitsEnsemble, Locale locale
    ) {
        Optional<UnitModel> oldAmountsUnitOptional = unitsEnsemble.unitById(req.getOldAmounts().getForEditUnitId());
        if (oldAmountsUnitOptional.isEmpty()) {
            return notFound(locale, "errors.unit.not.found");
        }
        UnitModel oldAmountsUnit = oldAmountsUnitOptional.get();
        Optional<UnitModel> oldFormFieldsUnitOptional = unitsEnsemble.unitById(req.getOldFormFields().getUnitId());
        if (oldFormFieldsUnitOptional.isEmpty()) {
            return notFound(locale, "errors.unit.not.found");
        }
        UnitModel formFieldsUnit = oldFormFieldsUnitOptional.get();

        UnitModel minAllowedUnit = Units.getMinAllowedUnit(resource, unitsEnsemble)
                .orElseThrow(() -> new IllegalStateException("Allowed unit not found"));
        UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId())
                .orElseThrow(() -> new IllegalStateException("Base unit not found"));
        UnitModel defaultUnit = unitsEnsemble.unitById(resource.getResourceUnits().getDefaultUnitId())
                .orElseThrow(() -> new IllegalStateException("Default unit not found"));

        BigDecimal oldProvided = convert(req.getOldAmounts().getProvided(), oldAmountsUnit, minAllowedUnit);
        BigDecimal quota = convert(req.getOldAmounts().getQuota(), oldAmountsUnit, minAllowedUnit);
        BigDecimal oldBalance = convert(req.getOldAmounts().getBalance(), oldAmountsUnit, minAllowedUnit);
        BigDecimal allocated = convert(req.getOldAmounts().getAllocated(), oldAmountsUnit, minAllowedUnit);

        ValidationMessages validationMessages = new ValidationMessages();
        BigDecimal newBalance;
        BigDecimal newProvided;
        BigDecimal newDelta;
        UnitModel newFormFieldsUnit;
        int scale;
        BigDecimal newProvidedForField = null;
        BigDecimal newDeltaForField = null;
        switch (req.getEditedField()) {
            case ABSOLUTE:
                newFormFieldsUnit = formFieldsUnit;
                newProvidedForField = req.getNewFormFields().getProvidedAbsolute();
                scale = getScale(newProvidedForField);
                newProvided = providedToMinAllowedUnitRounded(
                        newProvidedForField, formFieldsUnit, minAllowedUnit, validationMessages, locale
                );
                newDelta = newProvided.subtract(oldProvided);
                break;
            case DELTA:
                newFormFieldsUnit = formFieldsUnit;
                newDeltaForField = req.getNewFormFields().getProvidedDelta();
                scale = getScale(newDeltaForField);
                newDelta = convert(newDeltaForField, formFieldsUnit, minAllowedUnit);
                newProvided = providedToMinAllowedUnitRounded(
                        oldProvided.add(newDelta), minAllowedUnit, minAllowedUnit, validationMessages, locale
                );
                newDelta = newProvided.subtract(oldProvided); // чтобы согласовать с округленным newProvided
                break;
            case UNIT:
                scale = getScale(req.getOldFormFields().getProvidedAbsolute());
                Optional<UnitModel> newFormFieldsUnitOptional = unitsEnsemble.unitById(
                        req.getNewFormFields().getUnitId()
                );
                if (newFormFieldsUnitOptional.isEmpty()) {
                    return notFound(locale, "errors.unit.not.found");
                }
                newFormFieldsUnit = newFormFieldsUnitOptional.get();
                Optional<UnitModel> oldFormFieldMinAllowedUnitOptional =
                        unitsEnsemble.unitById(req.getOldFormFields().getMinAllowedUnitId());
                if (oldFormFieldMinAllowedUnitOptional.isEmpty()) {
                    return notFound(locale, "errors.unit.not.found");
                }
                UnitModel oldFormFieldMinAllowedUnit = oldFormFieldMinAllowedUnitOptional.get();

                newProvided = convert(req.getOldFormFields().getProvidedAbsoluteInMinAllowedUnit(),
                        oldFormFieldMinAllowedUnit, minAllowedUnit
                ).setScale(0, RoundingMode.FLOOR);
                newDelta = newProvided.subtract(oldProvided);
                break;
            default:
                throw new IllegalStateException("Unexpected EditedField: " + req.getEditedField());
        }
        Optional<Long> newProvidedInBaseUnit = Units.convertToBaseUnit(
                newProvided, resource, unitsEnsemble, minAllowedUnit
        );
        if (newProvidedInBaseUnit.isEmpty()) {
            validationMessages.addError("newProvided", messages.getMessage(
                    "errors.value.can.not.be.converted.to.base.unit", null, locale));
        } else if (!Units.canConvertToApi(newProvidedInBaseUnit.get(), resource, unitsEnsemble)) {
            validationMessages.addError("newProvided", messages.getMessage(
                    "errors.value.can.not.be.converted.to.providers.api.unit", null, locale));
        }
        newBalance = oldBalance.subtract(newDelta);
        if (newBalance.signum() < 0) {
            validationMessages.addError("newProvided", messages.getMessage(
                    "errors.negative.balance.is.not.allowed", null, locale));
        }
        boolean allocatedSupported = ProviderUtil.isAllocatedSupported(provider, resource);
        if (allocatedSupported && newProvided.compareTo(allocated) < 0) {
            validationMessages.addWarning(
                    "newProvided",
                    messages.getMessage("errors.provided.less.than.allocated.warning", null, locale));
        }

        BigDecimal providedRatio = quota.signum() != 0 ?
                newProvided.divide(quota, MathContext.DECIMAL64) : BigDecimal.ZERO;
        BigDecimal allocatedRatio = quota.signum() != 0 ?
                allocated.divide(quota, MathContext.DECIMAL64) : BigDecimal.ZERO;

        List<UnitModel> sortedUnits = unitsEnsemble.getUnits().stream()
                .sorted(UnitsComparator.INSTANCE).collect(Collectors.toList());
        List<UnitModel> allowedSortedUnits = QuotasHelper.getAllowedUnits(resource, sortedUnits);

        if (newProvidedForField == null) {
            newProvidedForField = convert(newProvided, minAllowedUnit, newFormFieldsUnit)
                    .setScale(scale, RoundingMode.HALF_UP);
        }
        if (newDeltaForField == null) {
            newDeltaForField = convert(newDelta, minAllowedUnit, newFormFieldsUnit)
                    .setScale(scale, RoundingMode.HALF_UP);
        }

        AmountDto balanceAmount = getAmountDto(newBalance, allowedSortedUnits, baseUnit,
                newFormFieldsUnit, defaultUnit, minAllowedUnit, locale, null);
        AmountDto providedAmount = getAmountDto(newProvided, allowedSortedUnits, baseUnit,
                newFormFieldsUnit, defaultUnit, minAllowedUnit, locale, newProvidedForField);
        AmountDto deltaAmount = getAmountDto(newDelta, allowedSortedUnits, baseUnit,
                newFormFieldsUnit, defaultUnit, minAllowedUnit, locale, newDeltaForField);

        var builder = new UpdateProvisionDryRunAnswerDto.Builder()
                .setBalance(FrontStringUtil.toString(roundForDisplay(
                        convert(newBalance, minAllowedUnit, newFormFieldsUnit))))
                .setProvidedAbsolute(FrontStringUtil.toString(newProvidedForField))
                .setProvidedDelta(FrontStringUtil.toString(newDeltaForField))
                .setProvidedRatio(FrontStringUtil.toString(providedRatio))
                .setAllocated(FrontStringUtil.toString(convert(allocated, minAllowedUnit, newFormFieldsUnit)))
                .setAllocatedRatio(FrontStringUtil.toString(allocatedRatio))
                .setForEditUnitId(newFormFieldsUnit.getId())
                .setProvidedAbsoluteInMinAllowedUnit(FrontStringUtil.toString(newProvided))
                .setMinAllowedUnitId(minAllowedUnit.getId())
                .setBalanceAmount(balanceAmount)
                .setProvidedAmount(providedAmount)
                .setDeltaAmount(deltaAmount);
        if (validationMessages.isNotEmpty()) {
            builder.setValidationMessages(validationMessages.toDto());
        }
        return Result.success(builder.build());
    }

    private int getScale(BigDecimal value) {
        int scale = value.stripTrailingZeros().scale();
        return scale > 0 ? scale : QuotasHelper.SCALE_FOR_EDIT;
    }

    private BigDecimal providedToMinAllowedUnitRounded(
            BigDecimal newProvided, UnitModel formFieldsUnit, UnitModel minAllowedUnit,
            ValidationMessages validationMessages, Locale locale
    ) {
        BigDecimal newProvidedConverted = convert(newProvided, formFieldsUnit, minAllowedUnit).stripTrailingZeros();
        BigDecimal newProvidedConvertedRounded = newProvidedConverted.setScale(0, RoundingMode.FLOOR);
        if (!isInteger(newProvidedConverted)) {
            validationMessages.addInfo(
                    "newProvided",
                    messages.getMessage("warning.provided.quota.value.rounded", null, locale));
        }
        return newProvidedConvertedRounded;
    }

    private Result<Map<String, UpdateProvisionDryRunAnswerDto>> calculateRelatedResourcesValues(
            Map<String, ResourceModel> resourcesById, Map<String, UnitsEnsembleModel> unitsEnsembleById,
            Map<String, RelatedCoefficient> coefficientByRelatedResourceId,
            Collection<ValidatedUpdateProvisionDryRunFolderQuota> folderQuotaRequests,
            BigDecimal newDeltaMainResource, Locale locale) {
        Map<String, UpdateProvisionDryRunAnswerDto> relatedResourcesValuesByResourceId = new HashMap<>();
        for (ValidatedUpdateProvisionDryRunFolderQuota folderQuotaRequest : folderQuotaRequests) {
            String resourceId = folderQuotaRequest.getResourceId();
            if (coefficientByRelatedResourceId.containsKey(resourceId) && resourcesById.containsKey(resourceId)) {
                ResourceModel resource = resourcesById.get(resourceId);
                RelatedCoefficient relatedCoefficientFraction = coefficientByRelatedResourceId.get(resourceId);
                BigDecimal numerator = BigDecimal.valueOf(relatedCoefficientFraction.getNumerator());
                BigDecimal denominator = BigDecimal.valueOf(relatedCoefficientFraction.getDenominator());
                BigDecimal newDeltaRelatedResource = newDeltaMainResource.multiply(numerator)
                        .divide(denominator, Units.MATH_CONTEXT);
                Result<UpdateProvisionDryRunAnswerDto> relatedResourceValues = calculateRelatedResourceValue(
                        folderQuotaRequest, resource, unitsEnsembleById.get(resource.getUnitsEnsembleId()),
                        newDeltaRelatedResource, locale);

                if (relatedResourceValues.isSuccess()) {
                    relatedResourcesValuesByResourceId.put(resourceId,
                            relatedResourceValues.match(Function.identity(), u -> null));
                } else {
                    return relatedResourceValues.apply(u -> null);
                }
            }
        }
        return Result.success(relatedResourcesValuesByResourceId);
    }

    private Result<UpdateProvisionDryRunAnswerDto> calculateRelatedResourceValue(
            ValidatedUpdateProvisionDryRunFolderQuota request, ResourceModel resource,
            UnitsEnsembleModel unitsEnsemble, BigDecimal newDeltaInBaseUnit, Locale locale) {
        Optional<UnitModel> oldAmountsUnitOptional = unitsEnsemble.unitById(request.getForEditUnitId());
        Optional<UnitModel> oldFormFieldsUnitOptional = unitsEnsemble.unitById(request.getFormFieldUnitId());
        if (oldAmountsUnitOptional.isEmpty() ||
                oldFormFieldsUnitOptional.isEmpty()) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.notFound(
                    messages.getMessage("errors.unit.not.found", null, locale))).build());
        }
        UnitModel oldAmountsUnit = oldAmountsUnitOptional.get();
        UnitModel formFieldsUnit = oldFormFieldsUnitOptional.get();
        UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId())
                .orElseThrow(() -> new IllegalStateException("Base unit not found"));
        UnitModel minAllowedUnit = Units.getMinAllowedUnit(resource, unitsEnsemble)
                .orElseThrow(() -> new IllegalStateException("Allowed unit not found"));
        UnitModel defaultUnit = unitsEnsemble.unitById(resource.getResourceUnits().getDefaultUnitId())
                .orElseThrow(() -> new IllegalStateException("Default unit not found"));

        BigDecimal oldProvided = convert(request.getProvided(), oldAmountsUnit, minAllowedUnit);
        BigDecimal quota = convert(request.getQuota(), oldAmountsUnit, minAllowedUnit);
        BigDecimal oldBalance = convert(request.getBalance(), oldAmountsUnit, minAllowedUnit);
        BigDecimal allocated = convert(request.getAllocated(), oldAmountsUnit, minAllowedUnit);
        BigDecimal newDelta = convert(newDeltaInBaseUnit, baseUnit, minAllowedUnit)
                .setScale(0, RoundingMode.CEILING);

        ValidationMessages validationMessages = new ValidationMessages();
        if (oldProvided.add(newDelta).signum() < 0) {
            newDelta = oldProvided.multiply(BigDecimal.valueOf(-1));
        }
        BigDecimal newProvided = oldProvided.add(newDelta);
        BigDecimal newBalance = oldBalance.subtract(newDelta);
        if (newBalance.signum() < 0) {
            String newProvidedText = FrontStringUtil.toString(convert(newProvided, minAllowedUnit, formFieldsUnit));
            String balanceText = FrontStringUtil.toString(convert(oldBalance, minAllowedUnit, formFieldsUnit));
            validationMessages.addWarning("newProvided", messages.getMessage(
                    "warning.provided.recommended", new String[] {balanceText, newProvidedText}, locale));
            newProvided = newProvided.add(newBalance);
            newDelta = newDelta.add(newBalance);
            newBalance = BigDecimal.ZERO;
        }
        // TODO DISPENSER-4687 Allocation may be important here
        BigDecimal providedRatio = quota.signum() != 0 ?
                newProvided.divide(quota, MathContext.DECIMAL64) : BigDecimal.ZERO;
        BigDecimal allocatedRatio = quota.signum() != 0 ?
                allocated.divide(quota, MathContext.DECIMAL64) : BigDecimal.ZERO;

        List<UnitModel> sortedUnits = unitsEnsemble.getUnits().stream()
                .sorted(UnitsComparator.INSTANCE).collect(Collectors.toList());
        List<UnitModel> allowedSortedUnits = QuotasHelper.getAllowedUnits(resource, sortedUnits);
        AmountDto balanceAmount = getAmountDto(newBalance, allowedSortedUnits, baseUnit,
                formFieldsUnit, defaultUnit, minAllowedUnit, locale, null);
        AmountDto providedAmount = getAmountDto(newProvided, allowedSortedUnits, baseUnit,
                formFieldsUnit, defaultUnit, minAllowedUnit, locale, null);
        AmountDto deltaAmount = getAmountDto(newDelta, allowedSortedUnits, baseUnit,
                formFieldsUnit, defaultUnit, minAllowedUnit, locale, null);

        var builder = new UpdateProvisionDryRunAnswerDto.Builder()
                .setBalance(FrontStringUtil.toString(convert(newBalance, minAllowedUnit, formFieldsUnit)))
                .setProvidedAbsolute(FrontStringUtil.toString(convert(newProvided, minAllowedUnit, formFieldsUnit)))
                .setProvidedDelta(FrontStringUtil.toString(convert(newDelta, minAllowedUnit, formFieldsUnit)))
                .setProvidedRatio(FrontStringUtil.toString(providedRatio))
                .setAllocated(FrontStringUtil.toString(convert(allocated, minAllowedUnit, formFieldsUnit)))
                .setAllocatedRatio(FrontStringUtil.toString(allocatedRatio))
                .setForEditUnitId(formFieldsUnit.getId())
                .setProvidedAbsoluteInMinAllowedUnit(FrontStringUtil.toString(newProvided))
                .setMinAllowedUnitId(minAllowedUnit.getId())
                .setBalanceAmount(balanceAmount)
                .setProvidedAmount(providedAmount)
                .setDeltaAmount(deltaAmount);
        if (validationMessages.isNotEmpty()) {
            builder.setValidationMessages(validationMessages.toDto());
        }
        return Result.success(builder.build());
    }

    @SuppressWarnings("checkstyle:ParameterNumber")
    private AmountDto getAmountDto(BigDecimal amountInMinAllowedUnit, List<UnitModel> sortedUnits, UnitModel baseUnit,
                                   UnitModel formFieldsUnit, UnitModel defaultUnit, UnitModel minAllowedUnit,
                                   Locale locale, BigDecimal forField) {
        if (amountInMinAllowedUnit.compareTo(BigDecimal.ZERO) == 0) {
            return QuotasHelper.zeroAmount(defaultUnit, formFieldsUnit, minAllowedUnit, locale);
        }
        return QuotasHelper.getAmountDto(convert(amountInMinAllowedUnit, minAllowedUnit, baseUnit),
                sortedUnits, baseUnit, formFieldsUnit, defaultUnit, minAllowedUnit, locale, forField);
    }

    private Result<ValidatedUpdateProvisionDryRunRequest> validateRequest(
            UpdateProvisionDryRunRequestDto request, Locale locale) {
        Result<ValidatedUpdateProvisionDryRunRequest> validatedUpdateProvisionDryRunRequest = validate(request, locale);
        Map<String, ValidatedUpdateProvisionDryRunFolderQuota> validatedRelatedResourcesById = new HashMap<>();
        if (request.getFolderQuotasByResourceId() != null) {
            for (UpdateProvisionDryRunFolderQuotaDto relatedResource : request.getFolderQuotasByResourceId().values()) {
                Result<ValidatedUpdateProvisionDryRunFolderQuota> validatedRelatedResource =
                        validateRelatedResource(relatedResource, locale);
                if (validatedRelatedResource.isFailure()) {
                    return Result.failure(validatedRelatedResource.match(u -> null, Function.identity()));
                }
                validatedRelatedResourcesById.put(relatedResource.getResourceId(),
                        validatedRelatedResource.match(Function.identity(), u -> null));
            }
        }
        return validatedUpdateProvisionDryRunRequest.apply(validatedRequest ->
                new ValidatedUpdateProvisionDryRunRequest(request.getResourceId(),
                        validatedRequest.getOldAmounts(), validatedRequest.getOldFormFields(),
                        validatedRequest.getEditedField(),
                        validatedRequest.getNewFormFields(), validatedRelatedResourcesById));
    }

    @SuppressWarnings("MethodLength")
    private Result<ValidatedUpdateProvisionDryRunRequest> validate(UpdateProvisionDryRunRequestDto request,
                                                                   Locale locale) {
        if (isEmpty(request.getResourceId())) {
            return failure("resourceId", locale, "errors.resource.id.is.required");
        }

        UpdateProvisionDryRunAmounts oldAmounts = request.getOldAmounts();
        if (oldAmounts == null) {
            return failure("oldAmounts", locale, "errors.field.is.required");
        }
        String oldAmountsQuota = oldAmounts.getQuota();
        if (oldAmountsQuota == null) {
            return failure("oldAmounts.quota", locale, "errors.field.is.required");
        }
        BigDecimal quota;
        try {
            quota = FrontStringUtil.toBigDecimal(oldAmountsQuota);
        } catch (NumberFormatException e) {
            return failure("oldAmounts.quota", locale, "errors.number.invalid.format");
        }
        String oldAmountsBalance = oldAmounts.getBalance();
        if (oldAmountsBalance == null) {
            return failure("oldAmounts.balance", locale, "errors.field.is.required");
        }
        BigDecimal balance;
        try {
            balance = FrontStringUtil.toBigDecimal(oldAmountsBalance);
        } catch (NumberFormatException e) {
            return failure("oldAmounts.balance", locale, "errors.number.invalid.format");
        }
        String oldAmountsProvided = oldAmounts.getProvided();
        if (oldAmountsProvided == null) {
            return failure("oldAmounts.provided", locale, "errors.field.is.required");
        }
        BigDecimal provided;
        try {
            provided = FrontStringUtil.toBigDecimal(oldAmountsProvided);
        } catch (NumberFormatException e) {
            return failure("oldAmounts.provided", locale, "errors.number.invalid.format");
        }
        String oldAmountsAllocated = oldAmounts.getAllocated();
        if (oldAmountsAllocated == null) {
            return failure("oldAmounts.allocated", locale, "errors.field.is.required");
        }
        BigDecimal allocated;
        try {
            allocated = FrontStringUtil.toBigDecimal(oldAmountsAllocated);
        } catch (NumberFormatException e) {
            return failure("oldAmounts.allocated", locale, "errors.number.invalid.format");
        }
        String oldAmountsForEditUnitId = oldAmounts.getForEditUnitId();
        if (isEmpty(oldAmountsForEditUnitId)) {
            return failure("oldAmounts.forEditUnitId", locale, "errors.field.is.required");
        }

        ValidatedUpdateProvisionDryRunAmounts validatedUpdateProvisionDryRunAmounts =
                new ValidatedUpdateProvisionDryRunAmounts(quota, balance, provided, allocated, oldAmountsForEditUnitId);

        UpdateProvisionDryRunRequestDto.OldEditFormFields oldFormFields = request.getOldFormFields();
        if (oldFormFields == null) {
            return failure("oldFormFields", locale, "errors.field.is.required");
        }
        String oldFormFieldsProvidedAbsolute = oldFormFields.getProvidedAbsolute();
        if (oldFormFieldsProvidedAbsolute == null) {
            return failure("oldFormFields.providedAbsolute", locale, "errors.field.is.required");
        }
        BigDecimal providedAbsolute;
        try {
            providedAbsolute = FrontStringUtil.toBigDecimal(oldFormFieldsProvidedAbsolute);
        } catch (NumberFormatException e) {
            return failure("oldFormFields.providedAbsolute", locale, "errors.number.invalid.format");
        }
        String oldFormFieldsProvidedDelta = oldFormFields.getProvidedDelta();
        if (oldFormFieldsProvidedDelta == null) {
            return failure("oldFormFields.providedDelta", locale, "errors.field.is.required");
        }
        BigDecimal providedDelta;
        try {
            providedDelta = FrontStringUtil.toBigDecimal(oldFormFieldsProvidedDelta);
        } catch (NumberFormatException e) {
            return failure("oldFormFields.providedDelta", locale, "errors.number.invalid.format");
        }
        String unitId = oldFormFields.getUnitId();
        if (isEmpty(unitId)) {
            return failure("oldFormFields.unitId", locale, "errors.field.is.required");
        }
        String absoluteInMinAllowedUnit = oldFormFields.getProvidedAbsoluteInMinAllowedUnit();
        if (absoluteInMinAllowedUnit == null) {
            return failure("oldFormFields.providedAbsoluteInMinAllowedUnit", locale, "errors.field.is.required");
        }
        BigDecimal providedAbsoluteInMinAllowedUnit;
        try {
            providedAbsoluteInMinAllowedUnit = FrontStringUtil.toBigDecimal(absoluteInMinAllowedUnit);
        } catch (NumberFormatException e) {
            return failure("oldFormFields.providedAbsoluteInMinAllowedUnit", locale, "errors.number.invalid.format");
        }
        if (isEmpty(oldFormFields.getMinAllowedUnitId())) {
            return failure("oldFormFields.minAllowedUnitId", locale, "errors.field.is.required");
        }

        ValidatedUpdateProvisionDryRunRequest.ValidatedOldEditFormFields validatedOldEditFormFields =
                new ValidatedUpdateProvisionDryRunRequest.ValidatedOldEditFormFields(providedAbsolute, providedDelta,
                        unitId, providedAbsoluteInMinAllowedUnit, oldFormFields.getMinAllowedUnitId());

        UpdateProvisionDryRunRequestDto.EditedField editedField = request.getEditedField();
        if (editedField == null) {
            return failure("editedField", locale, "errors.field.is.required");
        }
        if (request.getNewFormFields() == null) {
            return failure("newFormFields", locale, "errors.field.is.required");
        }

        BigDecimal changedEditFormFieldProvidedAbsolute;
        BigDecimal changedEditFormFieldProvidedProvidedDelta;
        String changedEditFormFieldUnitId = request.getNewFormFields().getUnitId();
        switch (editedField) {
            case ABSOLUTE:
                String absolute = request.getNewFormFields().getProvidedAbsolute();
                if (absolute == null) {
                    return failure("newFormFields.providedAbsolute", locale, "errors.field.is.required");
                }
                try {
                    changedEditFormFieldProvidedAbsolute = FrontStringUtil.toBigDecimal(absolute);
                } catch (NumberFormatException e) {
                    return failure("newFormFields.providedAbsolute", locale, "errors.number.invalid.format");
                }
                changedEditFormFieldProvidedProvidedDelta = null;
                break;
            case DELTA:
                String delta = request.getNewFormFields().getProvidedDelta();
                if (delta == null) {
                    return failure("newFormFields.providedDelta", locale, "errors.field.is.required");
                }
                try {
                    changedEditFormFieldProvidedProvidedDelta = FrontStringUtil.toBigDecimal(delta);
                } catch (NumberFormatException e) {
                    return failure("newFormFields.providedDelta", locale, "errors.number.invalid.format");
                }
                changedEditFormFieldProvidedAbsolute = null;
                break;
            case UNIT:
                if (isEmpty(changedEditFormFieldUnitId)) {
                    return failure("newFormFields.unitId", locale, "errors.field.is.required");
                }
                changedEditFormFieldProvidedAbsolute = null;
                changedEditFormFieldProvidedProvidedDelta = null;
                break;
            default:
                throw new IllegalStateException("Unexpected EditedField: " + editedField);
        }

        ValidatedUpdateProvisionDryRunRequest.ValidatedChangedEditFormField validatedChangedEditFormField =
                new ValidatedUpdateProvisionDryRunRequest.ValidatedChangedEditFormField(
                        changedEditFormFieldProvidedAbsolute, changedEditFormFieldProvidedProvidedDelta,
                        changedEditFormFieldUnitId);

        return Result.success(new ValidatedUpdateProvisionDryRunRequest(request.getResourceId(),
                validatedUpdateProvisionDryRunAmounts, validatedOldEditFormFields, editedField,
                validatedChangedEditFormField, null));
    }

    private Result<ValidatedUpdateProvisionDryRunFolderQuota> validateRelatedResource(
            UpdateProvisionDryRunFolderQuotaDto request, Locale locale) {
        if (isEmpty(request.getResourceId())) {
            return failure("resourceId", locale, "errors.resource.id.is.required");
        }

        String oldAmountsQuota = request.getQuota();
        if (oldAmountsQuota == null) {
            return failure("quota", locale, "errors.field.is.required");
        }
        BigDecimal quota;
        try {
            quota = FrontStringUtil.toBigDecimal(oldAmountsQuota);
        } catch (NumberFormatException e) {
            return failure("quota", locale, "errors.number.invalid.format");
        }
        String oldAmountsBalance = request.getBalance();
        if (oldAmountsBalance == null) {
            return failure("balance", locale, "errors.field.is.required");
        }
        BigDecimal balance;
        try {
            balance = FrontStringUtil.toBigDecimal(oldAmountsBalance);
        } catch (NumberFormatException e) {
            return failure("balance", locale, "errors.number.invalid.format");
        }
        String oldAmountsProvided = request.getProvided();
        if (oldAmountsProvided == null) {
            return failure("provided", locale, "errors.field.is.required");
        }
        BigDecimal provided;
        try {
            provided = FrontStringUtil.toBigDecimal(oldAmountsProvided);
        } catch (NumberFormatException e) {
            return failure("provided", locale, "errors.number.invalid.format");
        }
        String oldAmountsAllocated = request.getAllocated();
        if (oldAmountsAllocated == null) {
            return failure("allocated", locale, "errors.field.is.required");
        }
        BigDecimal allocated;
        try {
            allocated = FrontStringUtil.toBigDecimal(oldAmountsAllocated);
        } catch (NumberFormatException e) {
            return failure("allocated", locale, "errors.number.invalid.format");
        }
        String oldAmountsForEditUnitId = request.getForEditUnitId();
        if (isEmpty(oldAmountsForEditUnitId)) {
            return failure("forEditUnitId", locale, "errors.field.is.required");
        }
        String formFieldUnitId = request.getFormFieldUnitId();
        if (isEmpty(formFieldUnitId)) {
            return failure("formFieldUnitId", locale, "errors.field.is.required");
        }

        return Result.success(new ValidatedUpdateProvisionDryRunFolderQuota(request.getResourceId(),
                quota, balance, provided, allocated, oldAmountsForEditUnitId, formFieldUnitId));
    }

    private <T> Result<T> failure(String field, Locale locale, String code) {
        return Result.failure(ErrorCollection.builder().addError(
                field, TypedError.badRequest(
                        messages.getMessage(code, null, locale)
                )
        ).build());
    }

    @SuppressWarnings("SameParameterValue") // API
    private <T> Result<T> notFound(Locale locale, String code) {
        return Result.failure(ErrorCollection.builder().addError(TypedError.notFound(
                messages.getMessage(code, null, locale)
        )).build());
    }

}
