package ru.yandex.intranet.d.services.quotas

import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao
import ru.yandex.intranet.d.dao.quotas.QuotasDao
import ru.yandex.intranet.d.dao.resources.ResourcesDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.elapsed
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.accounts.*
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel
import ru.yandex.intranet.d.model.providers.ProviderId
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.quotas.QuotaModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeId
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.services.accounts.AccountService
import ru.yandex.intranet.d.services.operations.OperationsObservabilityService
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.services.uniques.RequestUniqueService
import ru.yandex.intranet.d.services.uniques.UpdateProvisionOperation
import ru.yandex.intranet.d.services.validators.AbcServiceValidator
import ru.yandex.intranet.d.util.FrontStringUtil
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.util.units.Units
import ru.yandex.intranet.d.web.model.folders.front.ExpandedProvider
import ru.yandex.intranet.d.web.model.folders.front.ProviderPermission
import ru.yandex.intranet.d.web.model.quotas.AccountsQuotasOperationsDto
import ru.yandex.intranet.d.web.model.quotas.ProvisionLiteWithBigIntegers
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsAnswerDto
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsRequestDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import ru.yandex.intranet.d.web.util.ModelDtoConverter
import java.math.BigDecimal
import java.math.BigInteger
import java.time.Instant
import java.util.*

private val logger = KotlinLogging.logger {}

/**
 * Provision logic service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ProvisionLogicService(private val securityManagerService: SecurityManagerService,
                            private val requestUniqueService: RequestUniqueService,
                            private val provisionService: ProvisionService,
                            private val abcServiceValidator: AbcServiceValidator,
                            private val resourcesLoader: ResourcesLoader,
                            private val resourceTypesLoader: ResourceTypesLoader,
                            private val unitsEnsemblesLoader: UnitsEnsemblesLoader,
                            private val accountsSpacesLoader: AccountSpacesLoader,
                            private val resourceSegmentationsLoader: ResourceSegmentationsLoader,
                            private val resourceSegmentsLoader: ResourceSegmentsLoader,
                            private val providersLoader: ProvidersLoader,
                            private val accountsService: AccountService,
                            private val accountsDao: AccountsDao,
                            private val folderDao: FolderDao,
                            private val quotasDao: QuotasDao,
                            private val resourcesDao: ResourcesDao,
                            private val accountsQuotasDao: AccountsQuotasDao,
                            private val accountsQuotasOperationsDao: AccountsQuotasOperationsDao,
                            private val operationsInProgressDao: OperationsInProgressDao,
                            private val folderOperationLogDao: FolderOperationLogDao,
                            private val tableClient: YdbTableClient,
                            private val operationsObservabilityService: OperationsObservabilityService,
                            @Qualifier("messageSource") private val messages: MessageSource) {

    fun updateProvisionMono(updateProvisionsRequestDto: UpdateProvisionsRequestDto,
                            currentUser: YaUserDetails,
                            locale: Locale,
                            publicApi: Boolean,
                            idempotencyKey: String?): Mono<Result<ProvisionOperationResult>> {
        return mono { updateProvision(updateProvisionsRequestDto, currentUser, locale, publicApi, idempotencyKey) }
    }

    suspend fun updateProvision(updateProvisionsRequestDto: UpdateProvisionsRequestDto,
                                currentUser: YaUserDetails,
                                locale: Locale,
                                publicApi: Boolean,
                                idempotencyKey: String?,
                                validateAllowedUnits: Boolean = true
    ): Result<ProvisionOperationResult> = binding {
        validateUser(currentUser, locale).bind()
        val tenantId = Tenants.getTenantId(currentUser)
        provisionService.preValidateInputDto(updateProvisionsRequestDto, locale, publicApi).bind()
        val checkedIdempotencyKey = requestUniqueService.validateIdempotencyKey(idempotencyKey, locale).bind()
        meter({ provisionService.checkReadPermissions(updateProvisionsRequestDto, currentUser, locale, publicApi)
            .awaitSingle() }, "Provide, check read permissions").bind()
        val (created, existing) = dbSessionRetryable(tableClient) {
            val (createdData, existingData) = rwTxRetryable {
                val existingAccountOperation = meter({ requestUniqueService.checkUpdateProvisions(txSession,
                    checkedIdempotencyKey, currentUser) }, "Provide, check idempotency key")
                if (existingAccountOperation != null) {
                    val folder = existingAccountOperation.folder
                    meter({ securityManagerService.checkReadPermissions(folder, currentUser, locale, folder)
                        .awaitSingle() }, "Provide, check folder read permissions").bind()
                    val urlFactory = getExternalAccountUrlFactory(txSession, existingAccountOperation, locale)
                    Pair (null, ExistingOperationContext(existingAccountOperation, urlFactory))
                } else {
                    val validatedResources = meter({ validateResources(txSession, updateProvisionsRequestDto, locale,
                        tenantId, publicApi) }, "Provide, validate resources").bind()!!
                    val resourceTypes = meter({ getResourceTypes(txSession, validatedResources.resources, locale) },
                        "Provide, get resource types").bind()!!
                    val units = meter({ getAndValidateUnits(txSession, validatedResources.parsedUpdatedProvisions,
                        validatedResources.resources, locale, publicApi, validateAllowedUnits)},
                        "Provide, validate units").bind()!!
                    val accountsSpace = meter({ getAccountsSpace(txSession,validatedResources.accountsSpaceId,
                        tenantId, validatedResources.providerId, locale) }, "Provide, get accounts space").bind()
                    val segmentations = meter({ getSegmentations(txSession, validatedResources.resources,
                        accountsSpace, tenantId, locale) }, "Provide, get segmentations").bind()!!
                    val segments = meter({ getSegments(txSession, validatedResources.resources,
                        accountsSpace, tenantId, locale) }, "Provide, get segments").bind()!!
                    val account = meter({ validateAccount(txSession, updateProvisionsRequestDto,
                        validatedResources.providerId, validatedResources.accountsSpaceId, locale, tenantId) },
                        "Provide, validate account").bind()!!
                    val provider = meter({ getProvider(txSession, validatedResources.providerId, locale, tenantId) },
                        "Provide, get provider").bind()!!
                    val folder = meter({ validateFolder(txSession, updateProvisionsRequestDto, account, locale,
                        tenantId, publicApi) }, "Provider, validate folder").bind()!!
                    meter({ checkWritePermissions(updateProvisionsRequestDto, folder, provider, currentUser, locale,
                        publicApi) }, "Provide, check write permissions").bind()
                    val quotas = meter({ validateQuotas(txSession, validatedResources.parsedUpdatedProvisions, folder,
                        provider, tenantId, locale) }, "Provide, validate quotas").bind()!!
                    val absentResources = meter({ loadAbsentResources(txSession, provider,
                        validatedResources.resources, resourceTypes, segmentations, segments, tenantId) },
                        "Provide, load provider resources")
                    val folderAccountsById = meter({ getFolderAccounts(txSession, provider, folder, accountsSpace,
                        tenantId) }, "Provide, load folder accounts")
                    val oldProvisions = meter({ getProvisions(txSession, folderAccountsById, tenantId) },
                        "Provide, load provisions")
                    val quotasAndResources = prepareQuotasAndResources(validatedResources.resources,
                        absentResources.resources, units, absentResources.unitsEnsembles, quotas,
                        accountsSpace, oldProvisions)
                    val validatedProvisions = validateProvisions(quotasAndResources, account,
                        validatedResources.parsedUpdatedProvisions, locale, publicApi).bind()!!
                    meter({ validateService(txSession, validatedProvisions.changingUpdatedProvisions, oldProvisions,
                        quotasAndResources.resourceByIdMap, quotasAndResources.ensembleModelByIdMap, account, folder,
                        locale, publicApi) }, "Provide, validate service").bind()
                    val preparedBuilder = prepareBuilder(updateProvisionsRequestDto, validatedResources, resourceTypes,
                        units, accountsSpace, segmentations, segments, account, provider, folder, quotas,
                        absentResources, folderAccountsById, oldProvisions, validatedProvisions, tenantId, currentUser,
                        locale, publicApi)
                    meter({ quotasDao.upsertAllRetryable(txSession, preparedBuilder.updatedQuota).awaitSingleOrNull() },
                        "Provide, write frozen quota")
                    meter({ accountsQuotasOperationsDao.upsertOneRetryable(txSession, preparedBuilder.operation).awaitSingle() },
                        "Provide, write operation")
                    operationsObservabilityService.observeOperationSubmitted(preparedBuilder.operation)
                    meter({ operationsInProgressDao.upsertOneRetryable(txSession, preparedBuilder.operationInProgress)
                        .awaitSingle() }, "Provide, write operation in progress")
                    meter({ folderOperationLogDao.upsertOneRetryable(txSession, preparedBuilder.folderOperationLog)
                        .awaitSingle() }, "Provide, write history log")
                    meter({ folderDao.upsertOneRetryable(txSession, preparedBuilder.updatedFolder).awaitSingle() },
                        "Provide, update folder")
                    if (checkedIdempotencyKey != null) {
                        meter({ requestUniqueService.addUpdateProvisions(txSession, checkedIdempotencyKey,
                            preparedBuilder.operation, preparedBuilder.operation.createDateTime, currentUser) },
                            "Provide, add idempotency key")
                    }
                    Pair (preparedBuilder, null)
                }
            }!!
            Pair (createdData, existingData)
        }!!
        return if (created != null) {
            provisionService.applyOperation(created.builder, locale, publicApi).awaitSingle()
        } else {
            processExistingOperation(existing!!, locale, publicApi)
        }
    }

    private fun processExistingOperation(context: ExistingOperationContext,
                                         locale: Locale,
                                         publicApi: Boolean): Result<ProvisionOperationResult> {
        val existingOp = context.existingOperation
        if (existingOp.provisions == null) {
            val isError = existingOp.operation.requestStatus
                .map { s -> s == AccountsQuotasOperationsModel.RequestStatus.ERROR }.orElse(false)
            val isConflict = existingOp.operation.errorKind.map { k -> k == OperationErrorKind.FAILED_PRECONDITION }
                .orElse(false)
            return if (isError) {
                accountsService.generateResultForFailedOperation(existingOp.operation, isConflict, locale) {
                    errorMessage -> if (isConflict && publicApi) {
                        TypedError.versionMismatch(errorMessage)
                    } else {
                        TypedError.badRequest(errorMessage)
                    }
                }
            } else {
                if (publicApi) {
                    Result.success(ProvisionOperationResult.inProgress(existingOp.operation.operationId))
                } else {
                    Result.failure(ErrorCollection.builder().addError(TypedError.badRequest(messages
                        .getMessage("errors.provision.update.scheduled", null, locale))).build())
                }
            }
        }
        val operationId = existingOp.operation.operationId
        val opResources = existingOp.provisions.resources
        val folderResources = existingOp.provisions.folderResources
        val opUnits = existingOp.provisions.unitsEnsembles
        val folderUnits = existingOp.provisions.folderUnitsEnsembles
        val opResourceById = opResources.associateBy { k -> k.id }
        val opUnitsById = opUnits.associateBy { k -> k.id }
        val account = existingOp.provisions.account
        val provisions = existingOp.provisions.provisions
        val quotas = existingOp.provisions.folderQuotas
        val folderAccounts = existingOp.provisions.folderAccounts
        val folderProvisions = existingOp.provisions.folderProvisions
        val provisionsByResource = provisions.associateBy { k -> k.resourceId }
        val provisionsWithZeroes = opResources.map { r -> provisionsByResource[r.id] ?: emptyProvision(account, r) }
        val expandedResult = ExpandedProvisionResult(opResources,
            provisionsWithZeroes, provisionsWithZeroes, opResourceById, opUnitsById, account)
        val accountsQuotasOperationsDto = AccountsQuotasOperationsDto(operationId,
            AccountsQuotasOperationsModel.RequestStatus.OK)
        val expandedProvider = prepareExpandedAnswer(account.providerId, opResources, folderResources,
            folderUnits, quotas, folderAccounts, folderProvisions, account, context.externalAccountUrlFactory, locale)
        val answerDto = UpdateProvisionsAnswerDto(expandedProvider, accountsQuotasOperationsDto)
        return Result.success(ProvisionOperationResult.success(answerDto, operationId, expandedResult))
    }

    private fun prepareExpandedAnswer(providerId: String,
                                      operationResources: List<ResourceModel>,
                                      folderResources: List<ResourceModel>,
                                      folderUnits: List<UnitsEnsembleModel>,
                                      folderQuotas: List<QuotaModel>,
                                      folderAccounts: List<AccountModel>,
                                      folderProvisions: List<AccountsQuotasModel>,
                                      account: AccountModel,
                                      externalAccountUrlFactory: ExternalAccountUrlFactory,
                                      locale: Locale): ExpandedProvider {
        val targetResourceIds = folderQuotas.map { q -> q.resourceId }.toSet()
            .union(operationResources.map { r -> r.id }.toSet())
        val folderQuotasByResourceId = folderQuotas.associateBy { q -> q.resourceId }
        val folderResourcesById = folderResources.associateBy { r -> r.id }
        val folderUnitsById = folderUnits.associateBy { u -> u.id }
        val quotaModelsMap = targetResourceIds
            .map { r -> folderQuotasByResourceId[r] ?: emptyQuota(account, folderResourcesById[r]!!) }
            .associateBy { q -> q.resourceId }
        val collectedQuotas = quotaModelsMap.values
            .groupBy { q -> ResourceTypeId(folderResourcesById[q.resourceId]!!.resourceTypeId) }
            .mapValues { t -> t.value.map { q -> QuotaSums.from(q) } }
        val folderAccountsById = folderAccounts.associateBy { a -> a.id }
        val accountsQuotasByAccountId = folderProvisions.groupBy { p -> p.accountId }.mapValues { e ->
            val accountResourceIds = e.value.map { v -> v.resourceId }.toSet().union(targetResourceIds)
            val provisionsByResourceId = e.value.associateBy { v -> v.resourceId }
            accountResourceIds.map { r -> provisionsByResourceId[r] ?: emptyProvision(folderAccountsById[e.key]!!,
                folderResourcesById[r]!!) }
        }
        val quotasByResourceId = folderQuotas.associateBy { q -> q.resourceId }
        return ExpandedProviderBuilder(locale, folderResourcesById, folderUnitsById, externalAccountUrlFactory)
            .toExpandedProvider(ProviderId(providerId), collectedQuotas, folderAccounts,
            accountsQuotasByAccountId,
            quotasByResourceId,
            setOf(ProviderPermission.CAN_UPDATE_PROVISION, ProviderPermission.CAN_MANAGE_ACCOUNT),
            false
        )
    }

    private fun emptyProvision(account: AccountModel, resource: ResourceModel): AccountsQuotasModel {
        return AccountsQuotasModel.Builder()
            .setTenantId(account.tenantId)
            .setAccountId(account.id)
            .setResourceId(resource.id)
            .setProvidedQuota(0L)
            .setAllocatedQuota(0L)
            .setFolderId(account.folderId)
            .setProviderId(resource.providerId)
            .setLastProvisionUpdate(Instant.now())
            .build()
    }

    private fun emptyQuota(account: AccountModel, resource: ResourceModel): QuotaModel {
        return QuotaModel.builder()
            .tenantId(account.tenantId)
            .folderId(account.folderId)
            .resourceId(resource.id)
            .providerId(resource.providerId)
            .quota(0L)
            .balance(0L)
            .frozenQuota(0L)
            .build()
    }

    private fun validateUser(currentUser: YaUserDetails,
                             locale: Locale): Result<Unit> {
        return if (currentUser.uid.isEmpty
            || currentUser.user.isEmpty
            || currentUser.user.get().id.isEmpty()
            || currentUser.user.get().passportLogin.isEmpty) {
            Result.failure(ErrorCollection.builder().addError(
                TypedError.forbidden(messages.getMessage("errors.access.denied", null, locale))).build())
        } else {
            Result.success(Unit)
        }

    }

    private suspend fun validateResources(txSession: YdbTxSession,
                                          updateProvisionsRequestDto: UpdateProvisionsRequestDto,
                                          locale: Locale,
                                          tenantId: TenantId,
                                          publicApi: Boolean): Result<ValidatedResources> {
        val errors = ErrorCollection.builder()
        val updatedProvisions = updateProvisionsRequestDto.updatedProvisions.orElseThrow()
        val parsedProvisions = mutableListOf<ProvisionLiteWithBigIntegers>()
        for (i in updatedProvisions.indices) {
            val provisionErrors = ErrorCollection.builder()
            val updatedProvision = updatedProvisions[i]
            val parsedOldProvidedAmount = if (!publicApi) {
                validateAmount(updatedProvision.oldProvidedAmount.orElseThrow(),
                    "updatedProvisions.$i.oldProvidedAmount", provisionErrors, locale)
            } else {
                null
            }
            val parsedProvidedAmount = validateAmount(updatedProvision.providedAmount.orElseThrow(),
                "updatedProvisions.$i.providedAmount", provisionErrors, locale)
            if (provisionErrors.hasAnyErrors()) {
                errors.add(provisionErrors)
            } else {
                parsedProvisions.add(ProvisionLiteWithBigIntegers(updatedProvision, parsedOldProvidedAmount,
                    parsedProvidedAmount))
            }
        }
        val resourceIds = updatedProvisions.map { v -> v.resourceId.orElseThrow() }
            .distinct().map { v -> Tuples.of(v, tenantId) }.toList()
        if (resourceIds.size != updatedProvisions.size) {
            errors.addError("updatedProvisions", TypedError
                .invalid(messages.getMessage("errors.resource.id.duplicate", null, locale)))
        }
        val resources = resourceIds.chunked(500)
            .map { p -> resourcesLoader.getResourcesByIds(txSession, p).awaitSingle() }.flatten()
        val resourceById = resources.associateBy { v -> v.id }
        for (i in 0 until updatedProvisions.size) {
            val updatedProvision = updatedProvisions[i]
            val resource = resourceById[updatedProvision.resourceId.orElseThrow()]
            if (resource == null || resource.isDeleted) {
                errors.addError("updatedProvisions.$i.resourceId", TypedError
                    .invalid(messages.getMessage("errors.resource.not.found", null, locale)))
            } else {
                if (publicApi && resource.isReadOnly) {
                    errors.addError("updatedProvisions.$i.resourceId", TypedError
                        .invalid(messages.getMessage("errors.resource.is.read.only", null, locale)))
                }
                if (!resource.isManaged) {
                    errors.addError("updatedProvisions.$i.resourceId", TypedError
                        .invalid(messages.getMessage("errors.resource.not.managed", null, locale)))
                }
            }
        }
        val providerIds = resources.map { v -> v.providerId }.distinct().toList()
        val accountsSpaceIds = resources.mapNotNull { v -> v.accountsSpacesId }.distinct().toList()
        if (providerIds.size != 1) {
            errors.addError("updatedProvisions", TypedError
                .invalid(messages.getMessage("errors.wrong.provider.for.resource", null, locale)))
        }
        if (accountsSpaceIds.size > 1) {
            errors.addError("updatedProvisions", TypedError
                .invalid(messages.getMessage("errors.wrong.resource.account.space.for.account", null, locale)))
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(ValidatedResources(parsedProvisions.toList(), resources, providerIds[0],
                if (accountsSpaceIds.isEmpty()) { null } else { accountsSpaceIds[0] }))
        }
    }

    private suspend fun getResourceTypes(txSession: YdbTxSession,
                                         resources: List<ResourceModel>,
                                         locale: Locale): Result<List<ResourceTypeModel>> {
        val resourceTypeIds = resources.map { v -> Tuples.of(v.resourceTypeId, v.tenantId) }.distinct().toList()
        val resourceTypes = resourceTypeIds.chunked(500)
            .map { p -> resourceTypesLoader.getResourceTypesByIds(txSession, p).awaitSingle() }.flatten()
        if (resourceTypes.size != resourceTypeIds.size || resourceTypes.any { v -> v.isDeleted }) {
            return Result.failure(ErrorCollection.builder().addError(
                TypedError.invalid(messages.getMessage("errors.resource.type.not.found", null, locale))).build())
        }
        return Result.success(resourceTypes)
    }

    private suspend fun getAndValidateUnits(
        txSession: YdbTxSession,
        parsedUpdatedProvisions: List<ProvisionLiteWithBigIntegers>,
        resources: List<ResourceModel>,
        locale: Locale,
        publicApi: Boolean,
        validateAllowedUnits: Boolean
    ): Result<List<UnitsEnsembleModel>> {
        val errors = ErrorCollection.builder()
        val unitsEnsembleIds = resources.map { v -> Tuples.of(v.unitsEnsembleId, v.tenantId) }.distinct().toList()
        val unitsEnsembles = unitsEnsembleIds.chunked(500)
            .map { p -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(txSession, p).awaitSingle() }.flatten()
        if (unitsEnsembles.size != unitsEnsembleIds.size || unitsEnsembles.any { v -> v.isDeleted }) {
            return Result.failure(ErrorCollection.builder().addError(
                TypedError.invalid(messages.getMessage("errors.units.ensemble.not.found", null, locale))).build())
        }
        val unitsEnsemblesById = unitsEnsembles.associateBy { v -> v.id }
        val resourceById = resources.associateBy { v -> v.id }
        for (i in parsedUpdatedProvisions.indices) {
            val parsedUpdatedProvision = parsedUpdatedProvisions[i]
            val resource = resourceById[parsedUpdatedProvision.resourceId.orElseThrow()]!!
            val unitsEnsemble = unitsEnsemblesById[resource.unitsEnsembleId]!!
            val unit = if (publicApi) {
                unitsEnsemble.unitByKey(parsedUpdatedProvision.providedAmountUnitId.orElseThrow())
            } else {
                unitsEnsemble.unitById(parsedUpdatedProvision.providedAmountUnitId.orElseThrow())
            }
            if (unit.isEmpty) {
                errors.addError("updatedProvisions.$i.providedAmountUnitId", TypedError
                    .invalid(messages.getMessage("errors.unit.not.found", null, locale)))
            } else if (
                validateAllowedUnits && !resource.resourceUnits.allowedUnitIds.contains(unit.orElseThrow().id)
            ) {
                errors.addError("updatedProvisions.$i.providedAmountUnitId", TypedError
                    .invalid(messages.getMessage("errors.unit.not.allowed", null, locale)))
            }
            if (!publicApi) {
                val oldUnit = unitsEnsemble.unitById(parsedUpdatedProvision.oldProvidedAmountUnitId.orElseThrow())
                if (oldUnit.isEmpty) {
                    errors.addError("updatedProvisions.$i.oldProvidedAmountUnitId", TypedError
                        .invalid(messages.getMessage("errors.unit.not.found", null, locale)))
                } else if (
                    validateAllowedUnits && !resource.resourceUnits.allowedUnitIds.contains(oldUnit.orElseThrow().id)
                ) {
                    errors.addError("updatedProvisions.$i.oldProvidedAmountUnitId", TypedError
                        .invalid(messages.getMessage("errors.unit.not.allowed", null, locale)))
                }
            }
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(unitsEnsembles)
        }
    }

    private suspend fun getAccountsSpace(txSession: YdbTxSession,
                                         accountsSpaceId: String?,
                                         tenantId: TenantId,
                                         providerId: String,
                                         locale: Locale): Result<AccountSpaceModel?> {
        if (accountsSpaceId == null) {
            return Result.success(null)
        }
        val accountsSpaceO = accountsSpacesLoader.getAccountSpaces(txSession, tenantId, providerId, accountsSpaceId)
            .awaitSingle()
        if (accountsSpaceO.isEmpty || accountsSpaceO.get().isEmpty() || accountsSpaceO.get()[0].isDeleted) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.accounts.space.not.found", null, locale))).build())
        } else if (accountsSpaceO.get()[0].isReadOnly) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.provider.account.space.is.read.only", null, locale))).build())
        }
        return Result.success(accountsSpaceO.get()[0])
    }

    private suspend fun getSegmentations(txSession: YdbTxSession,
                                         resources: List<ResourceModel>,
                                         accountsSpace: AccountSpaceModel?,
                                         tenantId: TenantId,
                                         locale: Locale): Result<List<ResourceSegmentationModel>> {
        val resourceSegmentsSettings = resources.flatMap { v -> v.segments ?: emptySet() }
        val accountsSpaceSegmentsSettings = accountsSpace?.segments ?: emptySet()
        val segmentsSettings = listOf(resourceSegmentsSettings, accountsSpaceSegmentsSettings).flatten()
        val segmentationIds = segmentsSettings.map { v -> Tuples.of(v.segmentationId, tenantId) }.distinct().toList()
        val segmentations = segmentationIds.chunked(500)
            .map { p -> resourceSegmentationsLoader.getResourceSegmentationsByIds(txSession, p).awaitSingle() }
            .flatten()
        if (segmentations.size != segmentationIds.size || segmentations.any { v -> v.isDeleted }) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.resource.segmentation.not.found", null, locale))).build())
        }
        return Result.success(segmentations)
    }

    private suspend fun getSegments(txSession: YdbTxSession,
                                    resources: List<ResourceModel>,
                                    accountsSpace: AccountSpaceModel?,
                                    tenantId: TenantId,
                                    locale: Locale): Result<List<ResourceSegmentModel>> {
        val resourceSegmentsSettings = resources.flatMap { v -> v.segments ?: emptySet() }
        val accountsSpaceSegmentsSettings = accountsSpace?.segments ?: emptySet()
        val segmentsSettings = listOf(resourceSegmentsSettings, accountsSpaceSegmentsSettings).flatten()
        val segmentIds = segmentsSettings.map { v -> Tuples.of(v.segmentId, tenantId) }.distinct().toList()
        val segments = segmentIds.chunked(500)
            .map { p -> resourceSegmentsLoader.getResourceSegmentsByIds(txSession, p).awaitSingle() }
            .flatten()
        if (segments.size != segmentIds.size || segments.any { v -> v.isDeleted }) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.resource.segment.not.found", null, locale))).build())
        }
        return Result.success(segments)
    }

    private suspend fun validateAccount(txSession: YdbTxSession,
                                        updateProvisionsRequestDto: UpdateProvisionsRequestDto,
                                        providerId: String,
                                        accountsSpaceId: String?,
                                        locale: Locale,
                                        tenantId: TenantId): Result<AccountModel> {
        val accountO = accountsDao.getById(txSession, updateProvisionsRequestDto.accountId.orElseThrow(), tenantId)
            .awaitSingle()
        if (accountO.isEmpty) {
            return Result.failure(ErrorCollection.builder().addError("accountId",
                TypedError.invalid(messages.getMessage("errors.account.not.found", null, locale))).build())
        }
        val account = accountO.get()
        if (account.isDeleted) {
            return Result.failure(ErrorCollection.builder().addError("accountId",
                TypedError.invalid(messages.getMessage("errors.account.deleted", null, locale))).build())
        }
        if (account.providerId != providerId) {
            return Result.failure(ErrorCollection.builder().addError("accountId", TypedError.invalid(messages
                .getMessage("errors.wrong.provider.for.resource", null, locale))).build())
        }
        if (accountsSpaceId != account.accountsSpacesId.orElse(null)) {
            return Result.failure(ErrorCollection.builder().addError("accountId", TypedError.invalid(messages
                .getMessage("errors.wrong.resource.account.space.for.account", null, locale))).build())
        }
        return Result.success(account)
    }

    private suspend fun getProvider(txSession: YdbTxSession,
                                    providerId: String,
                                    locale: Locale,
                                    tenantId: TenantId): Result<ProviderModel> {
        val providerO = providersLoader.getProviderById(txSession, providerId, tenantId).awaitSingle()
        if (providerO.isEmpty) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale))).build())
        }
        val provider = providerO.get()
        if (provider.isDeleted) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.provider.deleted", null, locale))).build())
        }
        if (!provider.isManaged) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.provider.is.not.managed", null, locale))).build())
        }
        if (provider.isReadOnly) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.provider.is.read.only", null, locale))).build())
        }
        return Result.success(provider)
    }

    private suspend fun validateFolder(txSession: YdbTxSession,
                                       updateProvisionsRequestDto: UpdateProvisionsRequestDto,
                                       account: AccountModel,
                                       locale: Locale,
                                       tenantId: TenantId,
                                       publicApi: Boolean): Result<FolderModel> {
        val folderId = updateProvisionsRequestDto.folderId.orElseThrow()
        if (folderId != account.folderId) {
            return Result.failure(ErrorCollection.builder().addError("folderId", TypedError.invalid(messages
                .getMessage("errors.folder.not.match.account.one", arrayOf(account.folderId), locale))).build())
        }
        val folderO = folderDao.getById(txSession, folderId, tenantId).awaitSingle()
        if (folderO.isEmpty) {
            return Result.failure(ErrorCollection.builder().addError("folderId", TypedError.invalid(messages
                .getMessage("errors.folder.not.found", null, locale))).build())
        }
        val folder = folderO.get()
        if (folder.isDeleted) {
            return Result.failure(ErrorCollection.builder().addError("folderId", TypedError.invalid(messages
                .getMessage("errors.folder.deleted", null, locale))).build())
        }
        if (!publicApi) {
            val serviceId = updateProvisionsRequestDto.serviceId.orElseThrow()
            if (serviceId != folder.serviceId) {
                return Result.failure(ErrorCollection.builder().addError("serviceId", TypedError.invalid(messages
                    .getMessage("errors.service.not.match.folder.one", arrayOf(folder.serviceId.toString()),
                        locale))).build())
            }
        }
        return Result.success(folder)
    }

    private suspend fun checkWritePermissions(updateProvisionsRequestDto: UpdateProvisionsRequestDto,
                                              folder: FolderModel,
                                              provider: ProviderModel,
                                              currentUser: YaUserDetails,
                                              locale: Locale,
                                              publicApi: Boolean): Result<Void?> {
        val serviceId = if (publicApi) { folder.serviceId } else { updateProvisionsRequestDto.serviceId.orElseThrow() }
        return securityManagerService.checkProvisionAndAccountPermissions(serviceId, provider, currentUser, locale)
            .awaitSingle()
    }

    private suspend fun validateQuotas(txSession: YdbTxSession,
                                       parsedUpdatedProvisions: List<ProvisionLiteWithBigIntegers>,
                                       folder: FolderModel,
                                       provider: ProviderModel,
                                       tenantId: TenantId,
                                       locale: Locale): Result<List<QuotaModel>> {
        val quotas = quotasDao.getByFoldersAndProvider(txSession, listOf(folder.id), tenantId, provider.id, false)
            .awaitSingle()
        val quotasByResourceId = quotas.associateBy { v -> v.resourceId }
        val errors = ErrorCollection.builder()
        for (i in parsedUpdatedProvisions.indices) {
            val parsedUpdatedProvision = parsedUpdatedProvisions[i]
            if (!quotasByResourceId.containsKey(parsedUpdatedProvision.resourceId.orElseThrow())) {
                errors.addError("updatedProvisions.$i", TypedError.invalid(messages
                    .getMessage("errors.quota.not.found", null, locale)))
            }
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(quotas)
        }
    }

    private suspend fun loadAbsentResources(txSession: YdbTxSession,
                                            provider: ProviderModel,
                                            requestResources: List<ResourceModel>,
                                            requestResourceTypes: List<ResourceTypeModel>,
                                            requestSegmentations: List<ResourceSegmentationModel>,
                                            requestSegments: List<ResourceSegmentModel>,
                                            tenantId: TenantId): AbsentFolderResources {
        val requestResourceIds = requestResources.map { r -> r.id }.toSet()
        val providerResources = resourcesDao.getAllByProvider(txSession, provider.id, tenantId, false)
            .awaitSingle()
        val absentResources = providerResources.filterNot { r -> requestResourceIds.contains(r.id) }.toList()
        val requestResourceTypeIds = requestResourceTypes.map { t -> t.id }.toSet()
        val absentResourceTypeIds = absentResources.asSequence().map { r -> r.resourceTypeId }.distinct()
            .filterNot { v -> requestResourceTypeIds.contains(v) }.map { v -> Tuples.of(v, tenantId) }.toList()
        val absentResourceTypes = absentResourceTypeIds.chunked(500)
            .map { p -> resourceTypesLoader.getResourceTypesByIds(txSession, p).awaitSingle() }.flatten()
        val requestUnitsEnsembleIds = requestResources.map { r -> r.unitsEnsembleId }.toSet()
        val absentUnitsEnsembleIds = absentResources.asSequence().map { r -> r.unitsEnsembleId }.distinct()
            .filterNot { v -> requestUnitsEnsembleIds.contains(v) }.map { v -> Tuples.of(v, tenantId) }.toList()
        val absentUnitsEnsembles = absentUnitsEnsembleIds.chunked(500)
            .map { p -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(txSession, p).awaitSingle() }.flatten()
        val requestSegmentationIds = requestSegmentations.map { s -> s.id }.toSet()
        val absentSegmentationIds = absentResources.asSequence().flatMap { v -> v.segments ?: emptySet() }
            .map { v -> v.segmentationId }.distinct().filterNot { v -> requestSegmentationIds.contains(v) }
            .map { v -> Tuples.of(v, tenantId) }.toList()
        val absentSegmentations = absentSegmentationIds.chunked(500)
            .map { p -> resourceSegmentationsLoader.getResourceSegmentationsByIds(txSession, p).awaitSingle() }
            .flatten()
        val requestSegmentIds = requestSegments.map { s -> s.id }.toSet()
        val absentSegmentIds = absentResources.asSequence().flatMap { v -> v.segments ?: emptySet() }
            .map { v -> v.segmentId }.distinct().filterNot { v -> requestSegmentIds.contains(v) }
            .map { v -> Tuples.of(v, tenantId) }.toList()
        val absentSegments = absentSegmentIds.chunked(500)
            .map { p -> resourceSegmentsLoader.getResourceSegmentsByIds(txSession, p).awaitSingle() }
            .flatten()
        return AbsentFolderResources(absentResources, absentResourceTypes, absentUnitsEnsembles, absentSegmentations,
            absentSegments)
    }

    private suspend fun getFolderAccounts(txSession: YdbTxSession,
                                          provider: ProviderModel,
                                          folder: FolderModel,
                                          accountsSpace: AccountSpaceModel?,
                                          tenantId: TenantId): Map<String, AccountModel> {
        val accounts = accountsDao.getByFoldersForProvider(txSession, tenantId, provider.id, setOf(folder.id),
            accountsSpace?.id, false).awaitSingle()
        return accounts.associateBy { a -> a.id }
    }

    private suspend fun getProvisions(txSession: YdbTxSession,
                                      accountById: Map<String, AccountModel>,
                                      tenantId: TenantId): List<AccountsQuotasModel> {
        return accountsQuotasDao.getAllByAccountIds(txSession, tenantId, accountById.keys).awaitSingle()
    }

    private suspend fun validateService(txSession: YdbTxSession,
                                        changingUpdatedProvisions: List<ProvisionLiteWithBigIntegers>,
                                        provisions: List<AccountsQuotasModel>,
                                        resourceByIdMap: Map<String, ResourceModel>,
                                        ensembleModelByIdMap: Map<String, UnitsEnsembleModel>,
                                        account: AccountModel,
                                        folder: FolderModel,
                                        locale: Locale,
                                        publicApi: Boolean): Result<Unit> {
        val currentProvisionsByResourceId = if (publicApi) {
            provisions.filter { p -> p.accountId == account.id }.associateBy { p -> p.resourceId }
        } else {
            emptyMap<String, AccountsQuotasModel>()
        }
        val isOnlyRemovingProvisions = changingUpdatedProvisions
            .all { p -> isRemoveProvisions(p, currentProvisionsByResourceId, resourceByIdMap, ensembleModelByIdMap,
                publicApi) }
        val allowedServiceReadOnlyState = if (isOnlyRemovingProvisions) {
            AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES_FOR_QUOTA_EXPORT
        } else {
            AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES
        }
        val allowedServiceState = if (isOnlyRemovingProvisions) {
            AbcServiceValidator.ALLOWED_SERVICE_STATES_FOR_QUOTA_EXPORT
        } else {
            AbcServiceValidator.ALLOWED_SERVICE_STATES
        }
        return abcServiceValidator.validateAbcService(Optional.of(folder.serviceId), locale, txSession, "serviceId",
            allowedServiceState, allowedServiceReadOnlyState, isOnlyRemovingProvisions).awaitSingle().toResult()
            .toUnit()
    }

    private fun isRemoveProvisions(provision: ProvisionLiteWithBigIntegers,
                                   currentProvisionsByResourceId: Map<String, AccountsQuotasModel>,
                                   resourceByIdMap: Map<String, ResourceModel>,
                                   ensembleModelByIdMap: Map<String, UnitsEnsembleModel>,
                                   publicApi: Boolean): Boolean {
        val resource = resourceByIdMap[provision.resourceId.orElseThrow()]!!
        val unitsEnsemble = ensembleModelByIdMap[resource.unitsEnsembleId]!!
        val baseUnit = unitsEnsemble.unitById(resource.baseUnitId).orElseThrow()
        val newUnit = if (publicApi) {
            unitsEnsemble.unitByKey(provision.providedAmountUnitId.orElseThrow())
        } else {
            unitsEnsemble.unitById(provision.providedAmountUnitId.orElseThrow())
        }
        if (newUnit.isEmpty) {
            throw IllegalStateException("UnitsEnsemble from resource '" + provision.resourceId +
                "' not contain UnitModel for new unit '" + provision.providedAmountUnitId.orElseThrow() + "'")
        }
        val newProvidedAmount = Units.convert(BigDecimal(provision.providedAmountBI), newUnit.get(), baseUnit)
        val oldProvidedAmount = if (!publicApi) {
            val oldUnit = unitsEnsemble.unitById(provision.oldProvidedAmountUnitId.orElseThrow())
            if (oldUnit.isEmpty) {
                throw IllegalStateException("UnitsEnsemble from resource '" + provision.resourceId +
                    "' not contain UnitModel for old unit '" + provision.oldProvidedAmountUnitId.orElseThrow() + "'")
            }
            Units.convert(BigDecimal(provision.oldProvidedAmountBI.orElseThrow()), oldUnit.get(), baseUnit)
        } else {
            val currentProvisionModel = currentProvisionsByResourceId[provision.resourceId.orElseThrow()]
            if (currentProvisionModel != null) {
                BigDecimal(currentProvisionModel.providedQuota)
            } else {
                BigDecimal.ZERO
            }
        }
        return newProvidedAmount < oldProvidedAmount
    }

    private fun validateProvisions(quotasAndResources: QuotasAndResources,
                                   account: AccountModel,
                                   parsedUpdatedProvisions: List<ProvisionLiteWithBigIntegers>,
                                   locale: Locale,
                                   publicApi: Boolean): Result<ValidatedProvisions> {
        val resourceModelMap = quotasAndResources.resourceByIdMap
        val ensembleModelMap = quotasAndResources.ensembleModelByIdMap
        val accountsQuotasModelByResourceByAccountMap = quotasAndResources
            .oldAccountsQuotasModelByResourceIdByAccountMap
        val accountsQuotasModelByResourcesMap = accountsQuotasModelByResourceByAccountMap[account.id] ?: emptyMap()
        val quotaModelByResourceMap = quotasAndResources.oldQuotaModelByResourceIdMap
        val changingUpdatedProvisions = mutableListOf<ProvisionLiteWithBigIntegers>()
        val oldProvisionsMap = mutableMapOf<String, Long>()
        val newProvisionsMap = mutableMapOf<String, Long>()
        val errors = ErrorCollection.builder()
        for (i in parsedUpdatedProvisions.indices) {
            val provisionErrors = ErrorCollection.builder()
            val parsedUpdatedProvision = parsedUpdatedProvisions[i]
            val resourceId = parsedUpdatedProvision.resourceId.orElseThrow()
            val resource = resourceModelMap[resourceId]!!
            val unitsEnsemble = ensembleModelMap[resource.unitsEnsembleId]!!
            val provision = accountsQuotasModelByResourcesMap[resourceId]
            val cachedProvidedQuota = provision?.providedQuota ?: 0L
            val oldProvidedQuota = if (!publicApi) {
                val oldProvidedAmount = parsedUpdatedProvision.oldProvidedAmountBI.orElseThrow()
                val oldUnit = unitsEnsemble.unitById(parsedUpdatedProvision
                    .oldProvidedAmountUnitId.orElseThrow()).orElseThrow()
                val convertedOldValueO = Units.convertFromApi(oldProvidedAmount.toLong(), resource,
                    unitsEnsemble, oldUnit)
                if (convertedOldValueO.isEmpty) {
                    provisionErrors.addError("updatedProvisions.$i.oldProvidedAmount", TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.unit", null, locale)))
                    null
                } else {
                    val convertedOldValue = convertedOldValueO.get()
                    if (convertedOldValue != cachedProvidedQuota) {
                        provisionErrors.addError("updatedProvisions.$i.oldProvidedAmount", TypedError.invalid(messages
                            .getMessage("errors.provision.mismatch", null, locale)))
                        null
                    } else {
                        convertedOldValue
                    }
                }
            } else {
                cachedProvidedQuota
            }
            val providedAmount = parsedUpdatedProvision.providedAmountBI
            val unit = if (publicApi) {
                unitsEnsemble.unitByKey(parsedUpdatedProvision.providedAmountUnitId.orElseThrow()).orElseThrow()
            } else {
                unitsEnsemble.unitById(parsedUpdatedProvision.providedAmountUnitId.orElseThrow()).orElseThrow()
            }
            val convertedValueO = Units.convertFromApi(providedAmount.toLong(), resource, unitsEnsemble, unit)
            val providedQuota = if (convertedValueO.isEmpty) {
                provisionErrors.addError("updatedProvisions.$i.providedAmount", TypedError.invalid(messages
                    .getMessage("errors.value.can.not.be.converted.to.unit", null, locale)))
                null
            } else {
                convertedValueO.get()
            }
            if (oldProvidedQuota != null && providedQuota != null) {
                if(!Units.canConvertToApi(providedQuota, resource, unitsEnsemble)) {
                    provisionErrors.addError("updatedProvisions.$i.providedAmount", TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.providers.api.unit", null, locale)))
                }
                val compared = cachedProvidedQuota.compareTo(providedQuota)
                if (compared > 0) {
                    if (resource.defaultQuota.isPresent && providedQuota < resource.defaultQuota.get()) {
                        provisionErrors.addError("updatedProvisions.$i.providedAmount", TypedError.invalid(messages
                            .getMessage("errors.provision.less.than.default", null, locale)))
                    } else {
                        changingUpdatedProvisions.add(parsedUpdatedProvision)
                        oldProvisionsMap[resourceId] = oldProvidedQuota
                        newProvisionsMap[resourceId] = providedQuota
                    }
                } else if (compared < 0) {
                    val quota = quotaModelByResourceMap[resourceId]!!
                    val balance = quota.balance
                    if (balance < (providedQuota - cachedProvidedQuota)) {
                        provisionErrors.addError("updatedProvisions.$i.providedAmount", TypedError.invalid(messages
                            .getMessage("errors.balance.is.to.low", null, locale)))
                    } else {
                        changingUpdatedProvisions.add(parsedUpdatedProvision)
                        oldProvisionsMap[resourceId] = oldProvidedQuota
                        newProvisionsMap[resourceId] = providedQuota
                    }
                }
            }
            if (provisionErrors.hasAnyErrors()) {
                errors.add(provisionErrors)
            }
        }
        if (!errors.hasAnyErrors() && changingUpdatedProvisions.isEmpty()) {
            errors.addError("updatedProvisions", TypedError.invalid(messages
                .getMessage("errors.provision.no.change", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(ValidatedProvisions(changingUpdatedProvisions.toList(), oldProvisionsMap.toMap(),
            newProvisionsMap.toMap()))
    }

    private fun prepareQuotasAndResources(requestResources: List<ResourceModel>,
                                          absentResources: List<ResourceModel>,
                                          requestUnitsEnsembles: List<UnitsEnsembleModel>,
                                          absentUnitsEnsembles: List<UnitsEnsembleModel>,
                                          quotas: List<QuotaModel>,
                                          accountsSpace: AccountSpaceModel?,
                                          oldProvisions: List<AccountsQuotasModel>): QuotasAndResources {
        val resourceModels = listOf(requestResources, absentResources).flatten()
        val resourceByIdMap = resourceModels.filter { r -> r.accountsSpacesId == accountsSpace?.id }
            .associateBy { r -> r.id }
        val unitsEnsembles = listOf(requestUnitsEnsembles, absentUnitsEnsembles).flatten()
        val ensembleModelByIdMap = unitsEnsembles.associateBy { v -> v.id }
        val oldQuotaModelByResourceIdMap = quotas.filter { q -> resourceByIdMap.containsKey(q.resourceId) }
            .associateBy { q -> q.resourceId }
        val oldAccountsQuotasModelByResourceIdByAccountMap = if (oldProvisions.isNotEmpty()) {
            oldProvisions.groupBy{ q -> q.accountId }
                .mapValues { v -> v.value.associateBy { k -> k.identity.resourceId } }
        } else {
            emptyMap()
        }
        return QuotasAndResources(resourceByIdMap, ensembleModelByIdMap, oldQuotaModelByResourceIdMap,
            oldAccountsQuotasModelByResourceIdByAccountMap)
    }

    private fun prepareBuilder(updateProvisionsRequestDto: UpdateProvisionsRequestDto,
                               validatedResources: ValidatedResources,
                               resourceTypes: List<ResourceTypeModel>,
                               unitsEnsembles: List<UnitsEnsembleModel>,
                               accountsSpace: AccountSpaceModel?,
                               segmentations: List<ResourceSegmentationModel>,
                               segments: List<ResourceSegmentModel>,
                               account: AccountModel,
                               provider: ProviderModel,
                               folder: FolderModel,
                               quotas: List<QuotaModel>,
                               absentResources: AbsentFolderResources,
                               accountsById: Map<String, AccountModel>,
                               provisions: List<AccountsQuotasModel>,
                               validatedProvisions: ValidatedProvisions,
                               tenantId: TenantId,
                               currentUser: YaUserDetails,
                               locale: Locale,
                               publicApi: Boolean): PreparedBuilder {
        val builder = QuotasProvisionAnswerBuilder(updateProvisionsRequestDto, currentUser, locale, tenantId, publicApi)
        builder.setRequestProvisionsBI(validatedResources.parsedUpdatedProvisions)
        builder.requestResourceModels = validatedResources.resources
        builder.providerId = validatedResources.providerId
        builder.accountsSpaceId = validatedResources.accountsSpaceId
        builder.setRequestResourceTypeModels(resourceTypes)
        builder.setRequestUnitsEnsembleModels(unitsEnsembles)
        builder.setAccountSpace(accountsSpace)
        builder.setRequestResourceSegmentationModels(segmentations)
        builder.setRequestResourceSegmentModels(segments)
        builder.accountModel = account
        builder.providerModel = provider
        builder.folderModel = folder
        builder.setAllQuotaModels(quotas)
        builder.setAbsentResourceModels(absentResources.resources)
        builder.setAbsentResourceType(absentResources.resourceTypes)
        builder.setAbsentUnitsEnsembleModels(absentResources.unitsEnsembles)
        builder.setAbsentResourceSegmentationModels(absentResources.segmentations)
        builder.setAbsentResourceSegmentModels(absentResources.segments)
        builder.prepareQuotasAndResources()
        builder.accountModelByIdMap = accountsById
        builder.setOldAccountsQuotasModel(provisions)
        builder.setChangingUpdatedProvisions(validatedProvisions.changingUpdatedProvisions)
        builder.setOldProvisionsByResourceIdMap(validatedProvisions.oldProvisionsMap)
        builder.setNewProvisionsByResourceIdMap(validatedProvisions.newProvisionsMap)
        val updatedQuotas = builder.calculateNewQuotaModels()
        val operation = builder.prepareAccountsQuotasOperation()
        val operationInProgress = builder.prepareOperationInProgress()
        val folderHistory = builder.prepareFolderOperationLogModel()
        val updatedFolder = builder.incrementedFolderModel
        return PreparedBuilder(builder, updatedQuotas.toList(), operation, operationInProgress,
            folderHistory, updatedFolder)
    }

    private fun validateAmount(amount: String,
                               key: String,
                               errors: ErrorCollection.Builder,
                               locale: Locale): BigInteger? {
        try {
            val parsedAmount = FrontStringUtil.toBigInteger(amount)
            if (parsedAmount < BigInteger.ZERO) {
                errors.addError(key, TypedError
                    .invalid(messages.getMessage("errors.number.must.be.non.negative", null, locale)))
                return null
            }
            return parsedAmount
        } catch (e: NumberFormatException) {
            errors.addError(key, TypedError
                .invalid(messages.getMessage("errors.number.invalid.format", null, locale)))
            return null
        }
    }

    private suspend fun getExternalAccountUrlFactory(
        ydbTxSession: YdbTxSession,
        existingOperation: UpdateProvisionOperation,
        locale: Locale
    ): ExternalAccountUrlFactory {
        if (existingOperation.provisions == null) return ExternalAccountUrlFactory()
        val tenantId = existingOperation.provisions.account.tenantId
        val providerO = meter({
            providersLoader.getProviderById(
                ydbTxSession,
                existingOperation.provisions.account.providerId,
                tenantId
            )
                .awaitSingle()
        }, "Provide, get provider")
        if (providerO.isEmpty) return ExternalAccountUrlFactory()
        val provider = providerO.get()
        val templates = provider.accountsSettings.externalAccountUrlTemplates ?: return ExternalAccountUrlFactory()
        val accountSpaceIdO = existingOperation.provisions.account.accountsSpacesId
        val accountSpace = meter({
            if (accountSpaceIdO.isEmpty) {
                null
            } else {
                accountsSpacesLoader.getAccountSpaces(ydbTxSession, tenantId, provider.id, accountSpaceIdO.get())
                    .awaitSingle()
                    .map { list -> list.getOrNull(0) }
                    .orElse(null)
            }
        }, "Provide, get accounts space")
        val resourceSegmentsSettings = existingOperation.provisions.resources.flatMap { v -> v.segments ?: emptySet() }
        val accountsSpaceSegmentsSettings = accountSpace?.segments ?: emptySet()
        val segmentsSettings = listOf(resourceSegmentsSettings, accountsSpaceSegmentsSettings).flatten()
        val segmentationIds = segmentsSettings.map { v -> Tuples.of(v.segmentationId, tenantId) }.distinct().toList()
        val segmentationsByIds = meter({
            segmentationIds.chunked(500)
                .map { p -> resourceSegmentationsLoader.getResourceSegmentationsByIds(ydbTxSession, p).awaitSingle() }
                .flatten()
                .associateBy { s -> s.id }
        }, "Provide, get segmentations")
        val segmentIds = segmentsSettings.map { v -> Tuples.of(v.segmentId, tenantId) }.distinct().toList()
        val segmentsByIds = meter({
            segmentIds.chunked(500)
                .map { p -> resourceSegmentsLoader.getResourceSegmentsByIds(ydbTxSession, p).awaitSingle() }
                .flatten()
                .associateBy { s -> s.id }
        }, "Provide, get segments")
        val accountSpaces = if (accountSpace == null) {
            emptyList()
        } else {
            listOf(ModelDtoConverter.toDto(accountSpace, segmentationsByIds, segmentsByIds, locale))
        }

        return ExternalAccountUrlFactory(
            templates,
            existingOperation.folder.serviceId,
            segmentationsByIds,
            segmentsByIds,
            accountSpaces
        )
    }

    private inline fun <T> meter(statement: () -> T,
                                  label: String): T {
        return elapsed(statement)
                { millis, success -> logger.info { "$label: duration = $millis ms, success = $success" } }
    }

    private data class ValidatedResources(val parsedUpdatedProvisions: List<ProvisionLiteWithBigIntegers>,
                                          val resources: List<ResourceModel>,
                                          val providerId: String,
                                          val accountsSpaceId: String?)

    private data class AbsentFolderResources(val resources: List<ResourceModel>,
                                             val resourceTypes: List<ResourceTypeModel>,
                                             val unitsEnsembles: List<UnitsEnsembleModel>,
                                             val segmentations: List<ResourceSegmentationModel>,
                                             val segments: List<ResourceSegmentModel>)

    private data class QuotasAndResources(val resourceByIdMap: Map<String, ResourceModel>,
                                          val ensembleModelByIdMap: Map<String, UnitsEnsembleModel>,
                                          val oldQuotaModelByResourceIdMap: Map<String, QuotaModel>,
                                          val oldAccountsQuotasModelByResourceIdByAccountMap:
                                                Map<String, Map<String, AccountsQuotasModel>>)

    private data class ValidatedProvisions(val changingUpdatedProvisions: List<ProvisionLiteWithBigIntegers>,
                                           val oldProvisionsMap: Map<String, Long>,
                                           val newProvisionsMap: Map<String, Long>)

    private data class PreparedBuilder(val builder: QuotasProvisionAnswerBuilder,
                                       val updatedQuota: List<QuotaModel>,
                                       val operation: AccountsQuotasOperationsModel,
                                       val operationInProgress: OperationInProgressModel,
                                       val folderOperationLog: FolderOperationLogModel,
                                       val updatedFolder: FolderModel)

    private data class ExistingOperationContext(
        val existingOperation: UpdateProvisionOperation,
        val externalAccountUrlFactory: ExternalAccountUrlFactory
    )
}
