package ru.yandex.intranet.d.services.quotas;

import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao;
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationErrorCollections;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.services.integration.providers.ProviderError;
import ru.yandex.intranet.d.services.integration.providers.ProvidersIntegrationService;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ErrorMessagesDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionResponseDto;
import ru.yandex.intranet.d.services.operations.OperationsObservabilityService;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.Details;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.quotas.ProvisionLiteDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsRequestDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static io.grpc.Status.Code.INTERNAL;

/**
 * Provision service.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 * @since 7-12-2020
 */
@Component
public class ProvisionService {

    private static final Logger LOG = LoggerFactory.getLogger(ProvisionService.class);

    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final ProvidersIntegrationService providersIntegrationService;
    private final QuotasDao quotasDao;
    private final AccountsDao accountsDao;
    private final FolderDao folderDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final AccountsQuotasOperationsDao accountsQuotasOperationsDao;
    private final OperationsInProgressDao operationsInProgressDao;
    private final YdbTableClient tableClient;
    private final OperationsObservabilityService operationsObservabilityService;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public ProvisionService(YdbTableClient tableClient,
                            @Qualifier("messageSource") MessageSource messages,
                            SecurityManagerService securityManagerService,
                            ProvidersIntegrationService providersIntegrationService,
                            QuotasDao quotasDao,
                            AccountsDao accountsDao,
                            FolderDao folderDao,
                            AccountsQuotasDao accountsQuotasDao,
                            FolderOperationLogDao folderOperationLogDao,
                            AccountsQuotasOperationsDao accountsQuotasOperationsDao,
                            OperationsInProgressDao operationsInProgressDao,
                            OperationsObservabilityService operationsObservabilityService) {
        this.tableClient = tableClient;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.providersIntegrationService = providersIntegrationService;
        this.quotasDao = quotasDao;
        this.accountsDao = accountsDao;
        this.folderDao = folderDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.accountsQuotasOperationsDao = accountsQuotasOperationsDao;
        this.operationsInProgressDao = operationsInProgressDao;
        this.operationsObservabilityService = operationsObservabilityService;
    }

    public Mono<Result<ProvisionOperationResult>> applyOperation(QuotasProvisionAnswerBuilder builder,
                                                                 Locale locale, boolean publicApi) {
        return updateProvisionRequest(builder, locale, publicApi).map(r -> r.apply(b -> b.isShortCircuit()
                ? ProvisionOperationResult.inProgress(b.getBuilder().getAccountsQuotasOperationsModel()
                        .getOperationId())
                : b.getBuilder().buildAnswer())
                .mapFailure(e -> {
                    AccountsQuotasOperationsModel op = builder.getAccountsQuotasOperationsModel();
                    if (publicApi && op != null) {
                        return ErrorCollection.builder().add(e)
                                .addDetail("operationMeta", new ProvisionOperationFailureMeta(op
                                        .getOperationId())).build();
                    }
                    return e;
                }));
    }

    public Mono<Result<Void>> checkReadPermissions(UpdateProvisionsRequestDto updateProvisionsRequestDto,
                                                   YaUserDetails currentUser, Locale locale, boolean publicApi) {
        if (publicApi) {
            return securityManagerService.checkReadPermissions(currentUser, locale);
        } else {
            long serviceId = updateProvisionsRequestDto.getServiceId().orElseThrow();
            return securityManagerService.checkReadPermissions(serviceId, currentUser, locale);
        }
    }

    public Result<Void> preValidateInputDto(UpdateProvisionsRequestDto updateProvisionsRequestDto,
                                            Locale locale, boolean publicApi) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!publicApi && updateProvisionsRequestDto.getServiceId().isEmpty()) {
            errors.addError("serviceId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        if (updateProvisionsRequestDto.getAccountId().isEmpty()) {
            errors.addError("accountId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (!Uuids.isValidUuid(updateProvisionsRequestDto.getAccountId().get())) {
            errors.addError("accountId", TypedError.invalid(messages
                    .getMessage("errors.account.not.found", null, locale)));
        }
        if (updateProvisionsRequestDto.getFolderId().isEmpty()) {
            errors.addError("folderId", TypedError.invalid(messages
                    .getMessage("errors.folder.is.required", null, locale)));
        } else if (!Uuids.isValidUuid(updateProvisionsRequestDto.getFolderId().get())) {
            errors.addError("folderId", TypedError.invalid(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        }
        if (updateProvisionsRequestDto.getUpdatedProvisions().isEmpty()
                || updateProvisionsRequestDto.getUpdatedProvisions().get().isEmpty()) {
            errors.addError("updatedProvisions", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            for (int i = 0; i < updateProvisionsRequestDto.getUpdatedProvisions().get().size(); i++) {
                ProvisionLiteDto provision = updateProvisionsRequestDto.getUpdatedProvisions().get().get(i);
                if (provision == null) {
                    errors.addError("updatedProvisions." + i, TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)));
                } else {
                    if (provision.getResourceId().isEmpty()) {
                        errors.addError("updatedProvisions." + i + ".resourceId", TypedError.invalid(messages
                                .getMessage("errors.resource.id.is.required", null, locale)));
                    } else if (!Uuids.isValidUuid(provision.getResourceId().get())) {
                        errors.addError("updatedProvisions." + i + ".resourceId", TypedError.invalid(messages
                                .getMessage("errors.resource.not.found", null, locale)));
                    }
                    if (provision.getProvidedAmount().isEmpty()) {
                        errors.addError("updatedProvisions." + i +
                                        (publicApi ? "provided" : ".providedAmount"), TypedError.invalid(messages
                                .getMessage("errors.account.provided.amount.is.required", null, locale)));
                    }
                    if (provision.getProvidedAmountUnitId().isEmpty()) {
                        errors.addError("updatedProvisions." + i +
                                (publicApi ? "providedUnitKey" : ".providedAmountUnitId"), TypedError.invalid(messages
                                .getMessage("errors.unit.id.is.required", null, locale)));
                    } else if (!publicApi && !Uuids.isValidUuid(provision.getProvidedAmountUnitId().get())) {
                        errors.addError("updatedProvisions." + i + ".providedAmountUnitId", TypedError.invalid(messages
                                .getMessage("errors.unit.not.found", null, locale)));
                    }
                    if (!publicApi && provision.getOldProvidedAmount().isEmpty()) {
                        errors.addError("updatedProvisions." + i + ".oldProvidedAmount", TypedError.invalid(messages
                                .getMessage("errors.account.provided.amount.is.required", null, locale)));
                    }
                    if (!publicApi && provision.getOldProvidedAmountUnitId().isEmpty()) {
                        errors.addError("updatedProvisions." + i + ".oldProvidedAmountUnitId",
                                TypedError.invalid(messages.getMessage("errors.unit.id.is.required", null, locale)));
                    } else if (!publicApi && !Uuids.isValidUuid(provision.getOldProvidedAmountUnitId().get())) {
                        errors.addError("updatedProvisions." + i + ".oldProvidedAmountUnitId",
                                TypedError.invalid(messages.getMessage("errors.unit.not.found", null, locale)));
                    }
                }
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private Mono<QuotasProvisionAnswerBuilder> getFolder(YdbTxSession ts,
                                                         QuotasProvisionAnswerBuilder builder) {
        return folderDao.getById(ts, builder.getAccountModel().getFolderId(),
                builder.getTenantId())
                .map(folderModelO -> {
                            FolderModel folderModel = folderModelO.orElse(builder.getFolderModel());
                            return builder.setActualFolderModel(folderModel);
                        }
                );
    }

    private Mono<QuotasProvisionAnswerBuilder> getFolderAccountsProvisions(
            YdbTxSession session, QuotasProvisionAnswerBuilder builder,
            Function<List<AccountsQuotasModel>, QuotasProvisionAnswerBuilder> setter) {
        Set<String> accountIds = builder.getAccountModelByIdMap().keySet();
        return accountsQuotasDao.getAllByAccountIds(session, builder.getTenantId(), accountIds).map(setter);
    }

    private Mono<QuotasProvisionAnswerBuilder> getFolderAccountsAllSpaces(YdbTxSession ts,
                                                                          QuotasProvisionAnswerBuilder b) {
        return accountsDao.getByFoldersForProvider(ts, b.getTenantId(),
                b.getProviderId(), b.getFolderModel().getId(), false).map(accounts -> {
            Map<String, AccountModel> accountsById = accounts.stream()
                    .collect(Collectors.toMap(AccountModel::getId, Function.identity()));
            return b.setAccountModelByIdMap(accountsById);
        });
    }

    private Mono<Result<ProvisionOperationState>> updateProvisionRequest(QuotasProvisionAnswerBuilder builder,
                                                                         Locale locale, boolean publicApi) {
        AccountModel accountModel = builder.getAccountModel();
        return meter(providersIntegrationService.updateProvision(
                        accountModel.getOuterAccountIdInProvider(),
                        builder.getProviderModel(),
                        builder.getUpdateProvisionRequestDto(),
                        locale),
                        "Provide, provider request"
                ).flatMap(r -> r.match(updateProvisionResponseDtoResponse -> {
                    if (updateProvisionResponseDtoResponse.isSuccess()) {
                        return validateResponseAndUpdate(
                            builder, unpack(updateProvisionResponseDtoResponse), locale, publicApi);
                    }
                    if (!isVersionConflict(updateProvisionResponseDtoResponse)
                            && !isRetryable(updateProvisionResponseDtoResponse)) {
                        // Return error to user unless operation commit failed in public API,
                        // in this case return 'in progress'
                        return doUpdateProvisionRequestIncorrect(builder, locale, updateProvisionResponseDtoResponse,
                                publicApi);
                    }
                    if (is5xx(updateProvisionResponseDtoResponse)) {
                        if (isRetryable(updateProvisionResponseDtoResponse) && builder.canRetryUpdateProvision()) {
                            // Recursive retry
                            return updateProvisionRequest(builder, locale, publicApi);
                        } else {
                            if (builder.getAccountsQuotasOperationsModel() != null) {
                                operationsObservabilityService.observeOperationTransientFailure(builder
                                        .getAccountsQuotasOperationsModel());
                            }
                            // Return error unless it is public API, in this case return 'in progress'
                            return publicApi
                                    ? Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)))
                                    : this.<ProvisionOperationState>failureMono(locale,
                                            "errors.provision.update.scheduled");
                        }
                    } else {
                        if (isVersionConflict(updateProvisionResponseDtoResponse)) {
                            // Try to resolve version conflict
                            // If unexpected or fatal error during resolution then return 'in progress' for public API
                            // and error otherwise
                            // If resolution is 'operation completed' then return success to user unless operation
                            // commit failed, in this case return 'in progress' for public API and error otherwise
                            // If resolution is 'conflict' then return error to user unless
                            // operation commit failed in public API, in this case return 'in progress'
                            return doOnVersionConflict(builder, locale, updateProvisionResponseDtoResponse, publicApi);
                        } else {
                            if (isRetryable(updateProvisionResponseDtoResponse) && builder.canRetryUpdateProvision()) {
                                // Recursive retry
                                return updateProvisionRequest(builder, locale, publicApi);
                            } else {
                                if (builder.getAccountsQuotasOperationsModel() != null) {
                                    operationsObservabilityService.observeOperationTransientFailure(builder
                                            .getAccountsQuotasOperationsModel());
                                }
                                // Return error unless it is public API, in this case return 'in progress'
                                return publicApi
                                        ? Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)))
                                        : this.<ProvisionOperationState>failureMono(locale,
                                                "errors.provision.update.scheduled");
                            }
                        }
                    }
                }, e -> {
                    if (builder.getAccountsQuotasOperationsModel() != null) {
                        operationsObservabilityService
                                .observeOperationTransientFailure(builder.getAccountsQuotasOperationsModel());
                    }
                    if (publicApi) {
                        LOG.warn("Failed to send provision update request: {}", e);
                        return Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)));
                    } else {
                        return Mono.just(Result.<ProvisionOperationState>failure(e));
                    }
                })).onErrorResume(e -> {
                    if (builder.getAccountsQuotasOperationsModel() != null) {
                        operationsObservabilityService
                                .observeOperationTransientFailure(builder.getAccountsQuotasOperationsModel());
                    }
                    if (publicApi) {
                        LOG.warn("Failed to send and process provision update request", e);
                        return Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)));
                    } else {
                        return Mono.error(e);
                    }
                });
    }

    private Mono<Result<ProvisionOperationState>> doUpdateProvisionRequestIncorrect(
            QuotasProvisionAnswerBuilder builder, Locale locale, Response<UpdateProvisionResponseDto> response,
            boolean publicApi
    ) {
        ErrorCollection errorsForActualLocale = doUpdateProvisionRequestIncorrectGenerateError(
                builder, locale, response, publicApi);
        ErrorCollection errorsForEnLocale = doUpdateProvisionRequestIncorrectGenerateError(
                builder, Locales.ENGLISH, response, publicApi);
        ErrorCollection errorsForRuLocale = doUpdateProvisionRequestIncorrectGenerateError(
                builder, Locales.RUSSIAN, response, publicApi);
        OperationErrorCollections operationErrorCollections = OperationErrorCollections.builder()
                .addErrorCollection(Locales.ENGLISH, errorsForEnLocale)
                .addErrorCollection(Locales.RUSSIAN, errorsForRuLocale)
                .build();

        return updateProvisionRequestIncorrect(operationErrorCollections, builder, false)
                .map(b -> Result.<ProvisionOperationState>failure(errorsForActualLocale))
                .onErrorResume(e -> {
                    if (publicApi) {
                        LOG.warn("Failed to save failed provision update", e);
                        return Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)));
                    } else {
                        return Mono.error(e);
                    }
                });
    }

    private ErrorCollection doUpdateProvisionRequestIncorrectGenerateError(
            QuotasProvisionAnswerBuilder builder, Locale locale, Response<UpdateProvisionResponseDto> response,
            boolean publicApi
    ) {
        String errorMessage = getErrorMessage(builder, locale);
        Optional<String> errorMessageFromProvider = toErrorMessage(response);

        ErrorCollection.Builder errorBuilder = ErrorCollection.builder();
        if (publicApi) {
            String flattenError = errorMessageFromProvider.map(s -> errorMessage + " " + s).orElse(errorMessage);
            errorBuilder.addError(TypedError.badRequest(flattenError));
        } else {
            errorBuilder.addError(TypedError.badRequest(errorMessage));
            errorMessageFromProvider.ifPresent(message ->
                    errorBuilder.addDetail(Details.ERROR_FROM_PROVIDER,
                            addRequestIdToError(message, response.requestId())));
        }
        return errorBuilder.build();
    }

    private <R> boolean is5xx(Response<R> updateProvisionResponseDtoResponse) {
        return updateProvisionResponseDtoResponse.match(new Response.Cases<>() {
            @Override
            public Boolean success(R result, String requestId) {
                return false;
            }

            @Override
            public Boolean failure(Throwable error) {
                return false;
            }

            @Override
            public Boolean error(ProviderError error, String requestId) {
                return error.match(new ProviderError.Cases<>() {
                    @Override
                    public Boolean httpError(int statusCode) {
                        return is5xx(statusCode);
                    }

                    @Override
                    public Boolean httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                        return is5xx(statusCode);
                    }

                    @Override
                    public Boolean grpcError(Status.Code statusCode, String message) {
                        return is5xx(statusCode);
                    }

                    @Override
                    public Boolean grpcExtendedError(Status.Code statusCode, String message,
                                                     Map<String, String> badRequestDetails) {
                        return is5xx(statusCode);
                    }
                });
            }
        });
    }

    private boolean isVersionConflict(Response<UpdateProvisionResponseDto> updateProvisionResponseDtoResponse) {
        return updateProvisionResponseDtoResponse.match(new Response.Cases<>() {
            @Override
            public Boolean success(UpdateProvisionResponseDto result, String requestId) {
                return false;
            }

            @Override
            public Boolean failure(Throwable error) {
                return false;
            }

            @Override
            public Boolean error(ProviderError error, String requestId) {
                return error.isConflict();
            }
        });
    }

    private <R> boolean isRetryable(Response<R> updateProvisionResponseDtoResponse) {
        return updateProvisionResponseDtoResponse.match(new Response.Cases<>() {
            @Override
            public Boolean success(R result, String requestId) {
                return false;
            }

            @Override
            public Boolean failure(Throwable error) {
                return false;
            }

            @Override
            public Boolean error(ProviderError error, String requestId) {
                return error.isRetryable();
            }
        });
    }

    private boolean is5xx(int code) {
        return code >= 500;
    }

    private boolean is5xx(Status.Code code) {
        return INTERNAL.equals(code);
    }

    private Mono<QuotasProvisionAnswerBuilder> updateProvisionRequestIncorrect(
            OperationErrorCollections errorMessage, QuotasProvisionAnswerBuilder builder, boolean conflict
    ) {
        return tableClient.usingSessionMonoRetryable(session ->
                session.usingCompTxRetryable(
                        ts -> meter(getQuotas(builder, ts), "Provide, second get quotas"),
                        (ts, quotas) -> meter(unfreezeQuotaFailure(quotas, builder, ts), "Provide, unfreeze quota")
                                .flatMap(b ->
                                        meter(updateAccountsQuotasOperationOnFailure(errorMessage, b, ts, conflict),
                                                "Provide, update operation"))
                                .flatMap(b -> meter(removeOperationInProgress(ts, b),
                                        "Provide, remove operation in progress")),
                        (ts, b) -> meter(ts.commitTransaction().thenReturn(b), "Provide, second commit")
                ));
    }

    private Mono<QuotasProvisionAnswerBuilder>
    updateAccountsQuotasOperationOnFailure(OperationErrorCollections errorMessage,
                                           QuotasProvisionAnswerBuilder b, YdbTxSession ts, boolean conflict) {
        return accountsQuotasOperationsDao.upsertOneRetryable(ts, prepareFailedOperation(errorMessage, b, conflict))
                .thenReturn(b);
    }

    private AccountsQuotasOperationsModel prepareFailedOperation(OperationErrorCollections errorMessage,
                                                                 QuotasProvisionAnswerBuilder b,
                                                                 boolean conflict) {
        AccountsQuotasOperationsModel result = b.toAccountsQuotasOperationsFailure(errorMessage, conflict);
        operationsObservabilityService.observeOperationFinished(result);
        return result;
    }

    private Optional<String> toErrorMessage(Response<UpdateProvisionResponseDto> updateProvisionResponseDto) {
        return updateProvisionResponseDto.match(new Response.Cases<>() {
            @Override
            public Optional<String> success(UpdateProvisionResponseDto result, String requestId) {
                return Optional.empty();
            }

            @Override
            public Optional<String> failure(Throwable error) {
                return Optional.ofNullable(error.getMessage());
            }

            @Override
            public Optional<String> error(ProviderError error, String requestId) {
                return Optional.ofNullable(error.match(new ProviderError.Cases<>() {
                    @Override
                    public String httpError(int statusCode) {
                        return "HttpError with code " + statusCode;
                    }

                    @Override
                    public String httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                        return errors.getMessage().orElse(null);
                    }

                    @Override
                    public String grpcError(Status.Code statusCode, String message) {
                        return message;
                    }

                    @Override
                    public String grpcExtendedError(Status.Code statusCode, String message,
                                                    Map<String, String> badRequestDetails) {
                        return message;
                    }
                }));
            }
        });
    }

    private Mono<Result<ProvisionOperationState>> doOnVersionConflict(
            QuotasProvisionAnswerBuilder builder, Locale locale,
            Response<UpdateProvisionResponseDto> updateProvisionResponse, boolean publicApi) {
        return meter(providersIntegrationService.getAccount(builder.getAccountModel().getOuterAccountIdInProvider(),
                builder.getProviderModel(), builder.prepareGetAccountRequestDto(), locale),
                        "Provide, another get account")
                .flatMap(r -> r.match(accountDtoResponse -> {
                    if (accountDtoResponse.isSuccess()) {
                        // If resolution is 'operation completed' then return success to user unless operation
                        // commit failed, in this case return 'in progress' for public API and error otherwise
                        // If resolution is 'conflict' then return error to user unless
                        // operation commit failed in public API, in this case return 'in progress'
                        return onSuccessProvidersIntegrationServiceGetAccount(builder,
                                accountDtoResponse,
                                updateProvisionResponse,
                                locale,
                                publicApi);
                    } else {
                        // If unexpected or fatal error during resolution then return 'in progress' for public API
                        // and error otherwise
                        // Retry other errors
                        return onErrorProvidersIntegrationServiceGetAccount(builder,
                                accountDtoResponse,
                                updateProvisionResponse,
                                locale,
                                publicApi);
                    }
                },
                e -> {
                    if (builder.getAccountsQuotasOperationsModel() != null) {
                        operationsObservabilityService
                                .observeOperationTransientFailure(builder.getAccountsQuotasOperationsModel());
                    }
                    if (publicApi) {
                        LOG.warn("Failed to resolve version conflict for provision update: {}", e);
                        return Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)));
                    } else {
                        return Mono.just(Result.<ProvisionOperationState>failure(e));
                    }
                })).onErrorResume(e -> {
                    if (builder.getAccountsQuotasOperationsModel() != null) {
                        operationsObservabilityService
                                .observeOperationTransientFailure(builder.getAccountsQuotasOperationsModel());
                    }
                    if (publicApi) {
                        LOG.warn("Failed to resolve version conflict for provision update", e);
                        return Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)));
                    } else {
                        return Mono.error(e);
                    }
                });
    }

    private Mono<Result<ProvisionOperationState>> onErrorProvidersIntegrationServiceGetAccount(
            QuotasProvisionAnswerBuilder builder,
            Response<AccountDto> accountDtoResponse,
            Response<UpdateProvisionResponseDto> updateProvisionResponseDtoResponse,
            Locale locale,
            boolean publicApi) {
        if ((!is5xx(accountDtoResponse)
                && !isRetryable(accountDtoResponse))
                || !builder.canRetryGetAccount()) {
            if (builder.getAccountsQuotasOperationsModel() != null) {
                operationsObservabilityService.observeOperationTransientFailure(builder
                        .getAccountsQuotasOperationsModel());
            }
            // Return error on fatal conflict resolution error unless it is public API, then return 'in progress'
            return publicApi
                    ? Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)))
                    : failureMono(locale, "errors.provision.update.scheduled");
        } else {
            // Recursive retry
            return doOnVersionConflict(builder, locale, updateProvisionResponseDtoResponse, publicApi);
        }
    }

    private Mono<Result<ProvisionOperationState>> onSuccessProvidersIntegrationServiceGetAccount(
            QuotasProvisionAnswerBuilder builder,
            Response<AccountDto> response,
            Response<UpdateProvisionResponseDto> updateProvisionResponseDtoResponse,
            Locale locale,
            boolean publicApi) {
        if (builder.isOperationComplete(response)) {
            return validateResponseAndUpdate(
                builder, builder.toUpdateProvisionResponseDto(response), locale, publicApi);
        } else {
            // If resolution is 'conflict' then return error to user unless
            // operation commit failed in public API, in this case return 'in progress'
            Optional<ProviderError> providerError =
                    updateProvisionResponseDtoResponse.match(new Response.Cases<>() {
                        @Override
                        public Optional<ProviderError> success(UpdateProvisionResponseDto result, String requestId) {
                            return Optional.empty();
                        }

                        @Override
                        public Optional<ProviderError> failure(Throwable error) {
                            return Optional.empty();
                        }

                        @Override
                        public Optional<ProviderError> error(ProviderError error, String requestId) {
                            return Optional.of(error);
                        }
                    });
            String requestId = updateProvisionResponseDtoResponse.requestId();
            ErrorCollection errorsForActualLocale = onSuccessProvidersIntegrationServiceGetAccountGenerateError(
                    builder, providerError, locale, publicApi, requestId);
            ErrorCollection errorsForEnLocale = onSuccessProvidersIntegrationServiceGetAccountGenerateError(
                    builder, providerError, Locales.ENGLISH, publicApi, requestId);
            ErrorCollection errorsForRuLocale = onSuccessProvidersIntegrationServiceGetAccountGenerateError(
                    builder, providerError, Locales.RUSSIAN, publicApi, requestId);
            OperationErrorCollections operationErrorCollections = OperationErrorCollections.builder()
                    .addErrorCollection(Locales.ENGLISH, errorsForEnLocale)
                    .addErrorCollection(Locales.RUSSIAN, errorsForRuLocale)
                    .build();

            return updateProvisionRequestIncorrect(operationErrorCollections, builder, true)
                    .map(b -> Result.<ProvisionOperationState>failure(errorsForActualLocale))
                    .onErrorResume(e -> {
                        if (publicApi) {
                            LOG.warn("Failed to save failed provision update", e);
                            return Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)));
                        } else {
                            return Mono.error(e);
                        }
                    });
        }
    }

    private ErrorCollection onSuccessProvidersIntegrationServiceGetAccountGenerateError(
            QuotasProvisionAnswerBuilder builder, Optional<ProviderError> providerError, Locale locale,
            boolean publicApi, String requestId) {
        String errorMessage = getErrorMessage(builder, locale);
        ErrorCollection.Builder errorBuilder = ErrorCollection.builder();

        if (publicApi) {
            String flattenError = providerError
                    .map(error -> Errors.flattenProviderErrorResponse(error, errorMessage)).orElse(errorMessage);
            errorBuilder.addError(TypedError.versionMismatch(flattenError));
        } else {
            errorBuilder.addError(TypedError.badRequest(errorMessage));
            providerError.map(error -> Errors.flattenProviderErrorResponse(error, null))
                    .ifPresent(flattenErrorWithoutPrefix -> errorBuilder.addDetail(Details.ERROR_FROM_PROVIDER,
                            addRequestIdToError(flattenErrorWithoutPrefix, requestId)));
        }
        return errorBuilder.build();
    }

    private Mono<QuotasProvisionAnswerBuilder>
    unfreezeQuotaFailure(List<QuotaModel> quotas, QuotasProvisionAnswerBuilder b, YdbTxSession ts) {
        return quotasDao.upsertAllRetryable(ts, b.calculateUnfreezeQuotasFailure(quotas))
                .thenReturn(b);
    }

    private Mono<Result<ProvisionOperationState>> validateResponseAndUpdate(
        QuotasProvisionAnswerBuilder builder,
        UpdateProvisionResponseDto updateProvisionResponse,
        Locale locale,
        boolean publicApi
    ) {
        Result<Void> validationResult = validateResponse(updateProvisionResponse, locale);
        // if validation fails, return 'in progress' for public API and an error otherwise
        if (validationResult.isFailure()) {
            if (builder.getAccountsQuotasOperationsModel() != null) {
                operationsObservabilityService
                    .observeOperationTransientFailure(builder.getAccountsQuotasOperationsModel());
            }
            if (publicApi) {
                return Mono.just(Result.success(ProvisionOperationState.shortCircuited(builder)));
            }
        }
        return validationResult
            .andThenMono(validatedR ->
                // Return success to user unless operation commit failed,
                // in this case return 'in progress' for public API and error otherwise
                update(builder, updateProvisionResponse, publicApi).map(Result::success)
            );
    }

    private Result<Void> validateResponse(UpdateProvisionResponseDto response, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (response.getProvisions().isPresent()) {
            for (int i = 0; i < response.getProvisions().get().size(); i++) {
                String fieldKeyPrefix = "provisions." + i;
                ProvisionDto provision = response.getProvisions().get().get(i);
                if (provision == null) {
                    errors.addError(fieldKeyPrefix,
                        TypedError.invalid(
                            messages.getMessage("errors.provision.is.required", null, locale)
                        ));
                    continue;
                }

                if (provision.getResourceKey().isEmpty()) {
                    errors.addError(fieldKeyPrefix + ".resourceKey",
                        TypedError.invalid(
                            messages.getMessage("errors.resource.not.found", null, locale)
                        ));
                }

                if (provision.getProvidedAmount().isEmpty()) {
                   errors.addError(fieldKeyPrefix + ".providedAmount",
                       TypedError.invalid(
                           messages.getMessage(
                               "errors.value.can.not.be.converted.to.base.unit", null, locale)
                       ));
                } else if (provision.getProvidedAmount().get() < 0L) {
                    errors.addError(fieldKeyPrefix + ".providedAmount",
                        TypedError.invalid(
                            messages.getMessage("errors.number.must.be.non.negative", null, locale)
                        ));
                }

                if (provision.getProvidedAmountUnitKey().isEmpty()) {
                    errors.addError(fieldKeyPrefix + ".providedAmountUnitKey",
                        TypedError.invalid(
                            messages.getMessage("errors.unit.not.found", null, locale)
                        ));
                } else if (provision.getProvidedAmountUnitKey().get().isEmpty()) {
                   errors.addError(fieldKeyPrefix + ".providedAmountUnitKey",
                       TypedError.invalid(
                            messages.getMessage(
                            "errors.value.can.not.be.converted.to.base.unit", null, locale)
                       ));
                }

                if (provision.getAllocatedAmount().isPresent()) {
                    if (provision.getAllocatedAmount().get() < 0L) {
                        errors.addError(fieldKeyPrefix + ".allocatedAmount",
                            TypedError.invalid(
                                messages.getMessage("errors.number.must.be.non.negative", null, locale)
                            ));
                    }
                    if (provision.getAllocatedAmountUnitKey().isEmpty()) {
                        errors.addError(fieldKeyPrefix + ".allocatedAmountUnitKey",
                            TypedError.invalid(
                                messages.getMessage("errors.unit.not.found", null, locale)
                            ));
                    } else if (provision.getAllocatedAmountUnitKey().get().isEmpty()) {
                        errors.addError(fieldKeyPrefix + ".allocatedAmountUnitKey",
                            TypedError.invalid(
                                messages.getMessage(
                                "errors.value.can.not.be.converted.to.base.unit", null, locale)
                            ));
                    }
                }
            }
        }
        if (errors.hasAnyErrors()) {
            LOG.warn("Invalid provider provision response: {} in {}", errors, response);
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private Mono<ProvisionOperationState> update(
        QuotasProvisionAnswerBuilder builder, UpdateProvisionResponseDto responseDto, boolean publicApi
    ) {
        return tableClient.usingSessionMonoRetryable(session ->
                session.usingCompTxRetryable(
                        ts -> meter(getQuotas(builder, ts), "Provide, second get quotas"),
                        (ts, quotas) ->
                                meter(getFolderAccountsAllSpaces(ts, builder),
                                        "Provide, second get more accounts")
                                .flatMap(b -> meter(getFolderAccountsProvisions(ts, b, b::setCurrentActualQuotas),
                                        "Provide, second get provisions"))
                                .flatMap(b -> meter(getAccount(b, ts), "Provide, second get account"))
                                .flatMap(b -> meter(getFolder(ts, b), "Provide, second get folder"))
                                .map(b -> b.validateAndPrepareStateAfterProvide(quotas, responseDto))
                                .flatMap(b -> meter(unfreezeQuota(builder, ts),
                                        "Provide, unfreeze quota"))
                                .flatMap(b -> meter(updateAccountsQuotasOperationOnSuccess(b, ts),
                                        "Provide, update operation"))
                                .flatMap(b -> meter(updateAccountQuota(b, ts), "Provide, update provisions"))
                                .flatMap(b -> meter(incrementFolderOpLogOnSuccess(ts, b),
                                        "Provide, second folder update"))
                                .flatMap(b -> meter(removeOperationInProgress(ts, b),
                                        "Provide, remove operation in progress"))
                                .flatMap(b -> meter(writeFolderOperationLogOnSuccess(b, ts),
                                        "Provide, write second history log")),
                        (ts, b) -> meter(ts.commitTransaction().thenReturn(b), "Provide, second commit")
                ))
                .map(ProvisionOperationState::continued)
                .onErrorResume(e -> {
                    if (publicApi) {
                        LOG.warn("Failed to save successful provision update", e);
                        return Mono.just(ProvisionOperationState.shortCircuited(builder));
                    } else {
                        return Mono.error(e);
                    }
                });
    }

    private Mono<QuotasProvisionAnswerBuilder> updateAccountsQuotasOperationOnSuccess(
            QuotasProvisionAnswerBuilder b,
            YdbTxSession ts) {
        return accountsQuotasOperationsDao.upsertOneRetryable(ts, prepareSuccessfulOperation(b))
                .thenReturn(b);
    }

    private AccountsQuotasOperationsModel prepareSuccessfulOperation(QuotasProvisionAnswerBuilder b) {
        AccountsQuotasOperationsModel result = b.toAccountsQuotasOperationsSuccess();
        operationsObservabilityService.observeOperationFinished(result);
        return result;
    }

    private Mono<QuotasProvisionAnswerBuilder> incrementFolderOpLogOnSuccess(YdbTxSession ts,
                                                                             QuotasProvisionAnswerBuilder b) {
        return folderDao.upsertOneRetryable(ts, b.getIncrementedFolderModelSuccess())
                .thenReturn(b);
    }

    private Mono<QuotasProvisionAnswerBuilder> removeOperationInProgress(YdbTxSession ts,
                                                                         QuotasProvisionAnswerBuilder b) {
        return operationsInProgressDao.deleteOneRetryable(ts, b.getOperationInProgressKey())
                .thenReturn(b);
    }

    private Mono<QuotasProvisionAnswerBuilder> writeFolderOperationLogOnSuccess(
            QuotasProvisionAnswerBuilder b,
            YdbTxSession ts) {
        return folderOperationLogDao.upsertOneRetryable(ts, b.toFolderOperationLogSuccess())
                .thenReturn(b);
    }

    private UpdateProvisionResponseDto unpack(
            Response<UpdateProvisionResponseDto> updateProvisionResponseDtoResponse
    ) {
        return updateProvisionResponseDtoResponse.<Optional<UpdateProvisionResponseDto>>match(
                (result, requestId) -> Optional.ofNullable(result),
                throwable -> Optional.empty(),
                (providerError, requestId) -> Optional.empty()
        ).orElseThrow();
    }

    private Mono<QuotasProvisionAnswerBuilder> updateAccountQuota(QuotasProvisionAnswerBuilder b, YdbTxSession ts) {
        List<AccountsQuotasModel> models = b.calculateUpdatedAccountQuota();
        if (models.isEmpty()) {
            return Mono.just(b);
        }

        return accountsQuotasDao.upsertAllRetryable(ts, models)
                .thenReturn(b);
    }

    private Mono<WithTxId<List<QuotaModel>>> getQuotas(QuotasProvisionAnswerBuilder builder, YdbTxSession ts) {
        return quotasDao.getByFoldersAndProviderStartTx(ts,
                Collections.singletonList(builder.getFolderModel().getId()),
                builder.getTenantId(), builder.getProviderModel().getId(),
                true)
                .map(tuple2 -> new WithTxId<>(tuple2.getT1(), tuple2.getT2()));
    }

    private Mono<QuotasProvisionAnswerBuilder> unfreezeQuota(QuotasProvisionAnswerBuilder b, YdbTxSession ts) {
        return quotasDao.upsertAllRetryable(ts, b.getUnfreezeQuotaModels())
                .thenReturn(b);
    }

    private Mono<QuotasProvisionAnswerBuilder> getAccount(QuotasProvisionAnswerBuilder builder,
                                                          YdbTxSession ts) {
        return accountsDao.getById(ts, builder.getUpdateProvisionsRequestDto().getAccountId().orElseThrow(),
                        builder.getTenantId())
                .map(accountO -> {
                    AccountModel accountModel;
                    if (accountO.isEmpty()) {
                        accountModel = builder.getAccountModel();
                    } else {
                        accountModel = accountO.get();
                    }

                    return builder.setCurrentAccount(accountModel);
                });
    }

    @SuppressWarnings("SameParameterValue")
    private <T> Mono<Result<T>> failureMono(Locale locale, String code) {
        return Mono.just(failure(locale, code));
    }

    private <T> Result<T> failure(Locale locale, String code) {
        return Result.failure(ErrorCollection.builder().addError(
                TypedError.badRequest(
                        messages.getMessage(code, null, locale)
                )
        ).build());
    }

    private String getErrorMessage(QuotasProvisionAnswerBuilder builder, Locale locale) {
        return builder.isProvidedLessThanAllocated() ?
                messages.getMessage("errors.provided.less.than.allocated", null, locale) :
                messages.getMessage("errors.provider.has.error", null, locale);
    }

    private String addRequestIdToError(String errorMessage, String requestId) {
        return errorMessage + "\nRequest id: " + requestId;
    }

    private static <T> Mono<T> meter(Mono<T> mono, String label) {
        return AsyncMetrics.metric(mono,
                (millis, success) -> LOG.info("{}: duration = {} ms, success = {}", label, millis, success));
    }

}
