package ru.yandex.intranet.d.services.quotas;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.DecimalWithUnit;
import ru.yandex.intranet.d.model.units.GrammaticalCase;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.util.FrontStringUtil;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.AmountDto;

/**
 * QuotasHelper.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 25-02-2021
 */
public class QuotasHelper {
    /**
     * Default scale for displaying quotas
     */
    public static final int SCALE_FOR_DISPLAY = 2;

    /**
     * Default scale for editing quotas
     */
    public static final int SCALE_FOR_EDIT = 2;

    /**
     * Default scale for editing quotas
     */
    public static final int SCALE_FOR_AGGREGATES_DISPLAY = 0;

    private QuotasHelper() {
    }

    public static boolean isInteger(BigDecimal v) {
        return v.signum() == 0 || v.scale() <= 0 || v.stripTrailingZeros().scale() <= 0;
    }

    public static BigDecimal toBigDecimal(Long value) {
        return value != null ? BigDecimal.valueOf(value) : BigDecimal.ZERO;
    }

    public static BigDecimal toBigDecimal(String value) {
        return value != null ? new BigDecimal(value) : BigDecimal.ZERO;
    }

    public static BigDecimal toBigDecimal(Long value, Supplier<Long> orElse) {
        return value != null ? BigDecimal.valueOf(value) : toBigDecimal(orElse.get());
    }

    public static String getUnitName(UnitModel unitModel, Locale locale) {
        return Locales.select(unitModel.getShortNamePluralEn(),
                unitModel.getShortNamePluralRu().get(GrammaticalCase.NOMINATIVE), locale);
    }

    public static DecimalWithUnit convertToMaxUnitWithCondition(
            List<UnitModel> sortedUnits, BigDecimal amount, UnitModel unit,
            Predicate<BigDecimal> condition
    ) {
        BigDecimal result = amount;
        UnitModel resultUnit = unit;
        for (int i = sortedUnits.size() - 1; i >= 0; i--) {
            UnitModel newUnit = sortedUnits.get(i);
            BigDecimal newAmount = Units.convert(amount, unit, newUnit);
            if (condition.test(newAmount)) {
                result = newAmount;
                resultUnit = newUnit;
                break;
            }
        }
        return new DecimalWithUnit(result, resultUnit);
    }

    public static List<DecimalWithUnit> convertToMaxUnitWithCondition(
            List<UnitModel> sortedUnits, List<BigDecimal> amounts, UnitModel unit,
            Predicate<BigDecimal> condition
    ) {
        List<BigDecimal> result = new ArrayList<>(amounts);
        List<BigDecimal> newAmounts = new ArrayList<>(amounts.size());
        UnitModel resultUnit = unit;
        for (int i = sortedUnits.size() - 1; i >= 0; i--) {
            UnitModel newUnit = sortedUnits.get(i);
            boolean isOk = true;
            for (int j = 0; j < amounts.size(); j++) {
                BigDecimal amount = amounts.get(j);
                BigDecimal newAmount = Units.convert(amount, unit, newUnit);
                newAmounts.set(j, newAmount);
                isOk &= condition.test(newAmount);
            }
            if (isOk) {
                result = newAmounts;
                resultUnit = newUnit;
                break;
            }
        }
        UnitModel finalResultUnit = resultUnit;
        return result.stream().map(amount -> new DecimalWithUnit(amount, finalResultUnit)).collect(Collectors.toList());
    }

    public static AmountDto zeroAmount(UnitModel defaultUnit, UnitModel forEditUnit, UnitModel minAllowedUnit,
                                       Locale locale) {
        String defaultUnitName = getUnitName(defaultUnit, locale);
        return new AmountDto(
                FrontStringUtil.toString(BigDecimal.ZERO), defaultUnitName,
                FrontStringUtil.toString(BigDecimal.ZERO), defaultUnitName,
                FrontStringUtil.toString(BigDecimal.ZERO), forEditUnit.getId(),
                FrontStringUtil.toString(BigDecimal.ZERO), minAllowedUnit.getId());
    }

    public static List<UnitModel> getAllowedUnits(ResourceModel resourceModel, List<UnitModel> sortedUnits) {
        Set<String> allowedUnitIds = Set.copyOf(resourceModel.getResourceUnits().getAllowedUnitIds());
        return sortedUnits.stream()
                .filter(unit -> allowedUnitIds.contains(unit.getId()))
                .collect(Collectors.toList());
    }

    public static List<UnitModel> getAllowedSortedUnits(ResourceModel resourceModel, Collection<UnitModel> units) {
        Set<String> allowedUnitIds = Set.copyOf(resourceModel.getResourceUnits().getAllowedUnitIds());
        return units.stream()
                .filter(unit -> allowedUnitIds.contains(unit.getId()))
                .sorted(UnitsComparator.INSTANCE)
                .collect(Collectors.toList());
    }

    public static Optional<UnitModel> getMinAllowedUnit(Set<String> allowedUnitIds, List<UnitModel> sortedUnits) {
        return sortedUnits.stream()
                .filter(unit -> allowedUnitIds.contains(unit.getId()))
                .findFirst();
    }

    public static AmountDto getAmountDto(
            BigDecimal amount, List<UnitModel> allowedSortedUnits, UnitModel baseUnit, UnitModel forEditUnit,
            UnitModel defaultUnit,
            UnitModel minAllowedUnit,
            Locale locale
    ) {
        return getAmountDto(
                amount, allowedSortedUnits, baseUnit, forEditUnit, defaultUnit, minAllowedUnit, locale, null);
    }

    @SuppressWarnings("checkstyle:ParameterNumber")
    public static AmountDto getAmountDto(
            BigDecimal amount, List<UnitModel> allowedSortedUnits, UnitModel baseUnit, UnitModel forEditUnit,
            UnitModel defaultUnit,
            UnitModel minAllowedUnit,
            Locale locale,
            @Nullable
            BigDecimal forEditValue
    ) {
        if (amount.compareTo(BigDecimal.ZERO) == 0) {
            return zeroAmount(defaultUnit, defaultUnit, minAllowedUnit, locale);
        }
        DecimalWithUnit readable = convertToReadable(amount, allowedSortedUnits, baseUnit);
        BigDecimal forEdit = forEditValue != null ? forEditValue :
                roundForEdit(Units.convert(amount, baseUnit, forEditUnit));
        BigDecimal amountInMinAllowedUnit = Units.convert(amount, baseUnit, minAllowedUnit)
                .setScale(0, RoundingMode.HALF_UP);
        return new AmountDto(
                FrontStringUtil.toString(roundForDisplay(readable.getAmount())),
                getUnitName(readable.getUnit(), locale),
                FrontStringUtil.toString(amount), getUnitName(baseUnit, locale),
                FrontStringUtil.toString(forEdit), forEditUnit.getId(),
                FrontStringUtil.toString(amountInMinAllowedUnit), minAllowedUnit.getId());
    }

    public static AmountDto getAmountDto(
            BigDecimal amount, ResourceModel resource, UnitsEnsembleModel unitsEnsemble, Locale locale
    ) {
        List<UnitModel> allowedSortedUnits = getAllowedSortedUnits(resource, unitsEnsemble.getUnits());
        if (allowedSortedUnits.isEmpty()) {
            throw new IllegalStateException(
                    "No allowed units found " + resource.getResourceUnits().getAllowedUnitIds());
        }
        UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId())
                .orElseThrow(() -> new IllegalStateException("Base unit not found " + resource.getBaseUnitId()));
        UnitModel minAllowedUnit = allowedSortedUnits.get(0);
        UnitModel defaultUnit = unitsEnsemble.unitById(resource.getResourceUnits().getDefaultUnitId())
                .orElseThrow(() -> new IllegalStateException(
                        "Default unit not found " + resource.getResourceUnits().getDefaultUnitId()));
        UnitModel forEditUnit = convertToReadable(amount, allowedSortedUnits, baseUnit).getUnit();
        return getAmountDto(
                amount,
                allowedSortedUnits,
                baseUnit,
                forEditUnit,
                defaultUnit,
                minAllowedUnit,
                locale
        );
    }

    public static AmountDto getAmountDtoForAggregates(
            BigDecimal amount,
            ResourceModel resource,
            UnitsEnsembleModel unitsEnsemble,
            @Nullable UnitModel humanReadableUnit,
            Locale locale
    ) {
        UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId())
                .orElseThrow(() -> new IllegalStateException("Base unit not found: " + resource.getBaseUnitId()));
        List<UnitModel> allowedSortedUnits = getAllowedSortedUnits(resource, unitsEnsemble.getUnits());
        if (allowedSortedUnits.isEmpty()) {
            throw new IllegalStateException("No allowed units: " + resource.getResourceUnits().getAllowedUnitIds());
        }
        UnitModel minAllowedUnit = allowedSortedUnits.get(0);
        return getAmountDtoForAggregates(
                amount, allowedSortedUnits, baseUnit, minAllowedUnit, humanReadableUnit, locale
        );
    }

    public static AmountDto getAmountDtoForAggregates(
            BigDecimal amount,
            List<UnitModel> allowedSortedUnits,
            UnitModel baseUnit,
            UnitModel minAllowedUnit,
            @Nullable UnitModel humanReadableUnit,
            Locale locale
    ) {
        if (allowedSortedUnits.isEmpty()) {
            throw new IllegalStateException("No allowed units: " + allowedSortedUnits);
        }
        boolean isZero = amount.compareTo(BigDecimal.ZERO) == 0;
        DecimalWithUnit readable = readableForAggregates(
                amount, isZero, baseUnit, humanReadableUnit, allowedSortedUnits
        );
        BigDecimal amountInMinAllowedUnit = isZero ? BigDecimal.ZERO : Units.convert(amount, baseUnit, minAllowedUnit)
                .setScale(0, RoundingMode.HALF_UP);
        return new AmountDto(
                FrontStringUtil.toString(roundForAggregatesDisplay(readable.getAmount())),
                getUnitName(readable.getUnit(), locale),
                FrontStringUtil.toString(amount), getUnitName(baseUnit, locale),
                FrontStringUtil.toString(roundForEdit(readable.getAmount())), readable.getUnit().getId(),
                FrontStringUtil.toString(amountInMinAllowedUnit), minAllowedUnit.getId());
    }

    private static DecimalWithUnit readableForAggregates(
            BigDecimal amount,
            boolean isZero,
            UnitModel baseUnit,
            @Nullable UnitModel selectedUnit,
            List<UnitModel> allowedSortedUnits
    ) {
        if (isZero) {
            return new DecimalWithUnit(
                    BigDecimal.ZERO,
                    selectedUnit == null ? allowedSortedUnits.get(0) : selectedUnit
            );
        }
        if (selectedUnit == null) {
            return convertToReadable(amount, allowedSortedUnits, baseUnit);
        }
        return new DecimalWithUnit(Units.convert(amount, baseUnit, selectedUnit), selectedUnit);
    }

    public static BigDecimal roundForDisplay(BigDecimal value) {
        return value.setScale(SCALE_FOR_DISPLAY, RoundingMode.HALF_UP);
    }

    public static BigDecimal roundForEdit(BigDecimal value) {
        return value.setScale(SCALE_FOR_EDIT, RoundingMode.FLOOR);
    }

    public static BigDecimal roundForAggregatesDisplay(BigDecimal value) {
        return value.setScale(SCALE_FOR_AGGREGATES_DISPLAY, RoundingMode.HALF_UP);
    }

    public static DecimalWithUnit convertToReadable(
            BigDecimal amount, List<UnitModel> sortedUnits, UnitModel baseUnit
    ) {
        if (!sortedUnits.contains(baseUnit) && !sortedUnits.isEmpty()) {
            amount = Units.convert(amount, baseUnit, sortedUnits.get(0));
            baseUnit = sortedUnits.get(0);
        }
        return convertToMaxUnitWithCondition(sortedUnits, amount, baseUnit,
                newAmount -> newAmount.abs().compareTo(BigDecimal.ONE) >= 0
        );
    }

    public static List<DecimalWithUnit> convertToReadable(
            List<BigDecimal> amounts, List<UnitModel> sortedUnits, UnitModel baseUnit
    ) {
        return convertToMaxUnitWithCondition(sortedUnits, amounts, baseUnit,
                newAmount -> newAmount.abs().compareTo(BigDecimal.ONE) >= 0
        );
    }

    /**
     * Convert to provided and not allocated amount
     * @param providedQuota expected in raw amount
     * @param allocatedQuota  expected in raw amount
     * @return provided and not allocated amount
     */
    public static BigDecimal toProvidedAndNotAllocated(String providedQuota, String allocatedQuota) {
        return toProvidedAndNotAllocated(toBigDecimal(providedQuota), toBigDecimal(allocatedQuota));
    }

    /**
     * Convert to provided and not allocated amount
     * @param providedQuota providedQuota expected in raw amount
     * @param allocatedQuota allocatedQuota  expected in raw amount
     * @return provided and not allocated amount
     */
    public static BigDecimal toProvidedAndNotAllocated(Long providedQuota, Long allocatedQuota) {
        return toProvidedAndNotAllocated(toBigDecimal(providedQuota), toBigDecimal(allocatedQuota));
    }

    private static BigDecimal toProvidedAndNotAllocated(BigDecimal providedQuota, BigDecimal allocatedQuota) {
        return providedQuota.subtract(allocatedQuota);
    }
}
