package ru.yandex.intranet.d.services.quotas;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.transfers.TransferRequestsDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceIdAndParentId;
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer;
import ru.yandex.intranet.d.model.transfers.QuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.transfer.TextFormatterService;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.model.quotas.QuotasMoveStatisticDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.util.result.TypedError.forbidden;

/**
 * Get statistic about quotas move
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class QuotasMoveStatisticService {

    private final YdbTableClient tableClient;
    private final ServicesDao servicesDao;
    private final TransferRequestsDao transferRequestsDao;
    private final TextFormatterService textFormatterService;
    private final MessageSource messages;
    private final ResourcesLoader resourcesLoader;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final Set<String> usersHavePermissions;

    @SuppressWarnings("ParameterNumber")
    public QuotasMoveStatisticService(YdbTableClient tableClient,
                                      ServicesDao servicesDao,
                                      TransferRequestsDao transferRequestsDao,
                                      TextFormatterService textFormatterService,
                                      @Qualifier("messageSource") MessageSource messages,
                                      ResourcesLoader resourcesLoader,
                                      UnitsEnsemblesLoader unitsEnsemblesLoader,
                                      @Value("${quotaMove.usersHavePermissions}") Set<String> usersHavePermissions) {
        this.tableClient = tableClient;
        this.servicesDao = servicesDao;
        this.transferRequestsDao = transferRequestsDao;
        this.textFormatterService = textFormatterService;
        this.messages = messages;
        this.resourcesLoader = resourcesLoader;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.usersHavePermissions = usersHavePermissions;
    }

    public Mono<Result<QuotasMoveStatisticDto>> getQuotasMoveStatisticDto(Instant dateFrom, Instant dateTo,
                                                                          List<Long> parentServiceIds,
                                                                          YaUserDetails user, Locale locale) {
        return validatePermission(user, locale)
                .flatMap(res -> res.andThenMono(u ->
                        getQuotasMoveStatistic(dateFrom, dateTo, parentServiceIds,
                                Tenants.getTenantId(user), locale)
                                .map(res2 -> res2.andThen(r -> Result.success(toDto(r))))
                ));
    }

    public Mono<Result<QuotasMoveStatisticResult>> getQuotasMoveStatistic(Instant dateFrom, Instant dateTo,
                                                                          List<Long> parentServiceIds,
                                                                          TenantId tenantId, Locale locale) {
        return tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                servicesDao.getAllServiceIdsWithParents(session).collect(Collectors.toList())
                        .map(serviceIdAndParentIds ->
                                getServicesSubtree(serviceIdAndParentIds, parentServiceIds))
                        .flatMap(ids -> transferRequestsDao.getByServicesAndDate(txSession,
                                                new ArrayList<>(ids), tenantId, dateFrom, dateTo)
                        .flatMap(transfers -> resourcesLoader
                                .getResourcesByIds(txSession, getResourceIds(transfers, tenantId))
                        .flatMap(resources -> unitsEnsemblesLoader
                                .getUnitsEnsemblesByIds(txSession, getUnitsEnsemblesIds(resources, tenantId))
                        .map(unitsEnsembles -> {
                            Predicate<TransferRequestModel> filterResourcesInput = transfer -> {
                                if (transfer.getType().equals(TransferRequestType.QUOTA_TRANSFER)) {
                                    return transfer.getParameters().getQuotaTransfers().stream().anyMatch(
                                        quotaTransfer ->
                                            !ids.contains(quotaTransfer.getDestinationServiceId()) &&
                                            transferHasNegativeQuota(quotaTransfer.getTransfers())
                                    );
                                }
                                if (transfer.getType().equals(TransferRequestType.PROVISION_TRANSFER)) {
                                    return transfer.getParameters().getProvisionTransfers().stream().anyMatch(
                                        provisionTransfer ->
                                            !ids.contains(provisionTransfer.getSourceServiceId())
                                    );
                                }
                                return false;
                            };

                            List<TransferRequestModel> created = transfers.stream()
                                    .filter(filterResourcesInput)
                                    .filter(transfer -> transfer.getAppliedAt().isEmpty())
                                    .toList();
                            List<TransferRequestModel> applied = transfers.stream()
                                    .filter(filterResourcesInput)
                                    .filter(transfer -> transfer.getAppliedAt().isPresent())
                                    .toList();
                            Map<String, ResourceModel> resourceById = resources.stream()
                                    .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
                            Map<String, UnitsEnsembleModel> unitsEnsembleById = unitsEnsembles.stream()
                                    .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
                            return Result.success(new QuotasMoveStatisticResult(
                                    created, applied, resourceById, unitsEnsembleById));
                        })
        )))));
    }

    private Set<Long> getServicesSubtree(List<ServiceIdAndParentId> serviceIdAndParentIds, List<Long> parentIds) {
        Map<Long, Set<Long>> serviceIdsByParent = new HashMap<>();
        for (ServiceIdAndParentId serviceIdAndParentId : serviceIdAndParentIds) {
            serviceIdsByParent.computeIfAbsent(serviceIdAndParentId.getParentId(), u -> new HashSet<>())
                    .add(serviceIdAndParentId.getServiceId());
        }

        Set<Long> serviceIds = new HashSet<>();
        for (long parentId : parentIds) {
            addParentAndAllChild(serviceIdsByParent, parentId, serviceIds);
        }
        return serviceIds;
    }

    private void addParentAndAllChild(Map<Long, Set<Long>> serviceIdsByParent, long parentId, Set<Long> result) {
        result.add(parentId);
        for (Long child : serviceIdsByParent.getOrDefault(parentId, Collections.emptySet())) {
            addParentAndAllChild(serviceIdsByParent, child, result);
        }
    }

    private Mono<Result<Void>> validatePermission(YaUserDetails user, Locale locale) {
        if (user.getUser().map(UserModel::getDAdmin).orElse(false)) {
            return Mono.just(Result.success(null));
        }
        if (user.getUser().isPresent() &&
                usersHavePermissions.contains(user.getUser().get().getPassportLogin().orElse(""))) {
            return Mono.just(Result.success(null));
        }
        return Mono.just(Result.failure(ErrorCollection.builder()
                .addError(forbidden(messages.getMessage("errors.access.denied", null, locale)))
                .build()));
    }

    private boolean transferHasNegativeQuota(Collection<ResourceQuotaTransfer> transfers) {
        return transfers.stream()
                .anyMatch(resourceQuotaTransfer -> resourceQuotaTransfer.getDelta() < 0);
    }

    private List<Tuple2<String, TenantId>> getResourceIds(List<TransferRequestModel> transfers,
                                                          TenantId tenantId) {
        Set<Tuple2<String, TenantId>> resourceIds = new HashSet<>();
        for (TransferRequestModel transfer : transfers) {
            if (transfer.getType() == TransferRequestType.QUOTA_TRANSFER) {
                for (QuotaTransfer quotaTransfer : transfer.getParameters().getQuotaTransfers()) {
                    for (ResourceQuotaTransfer resourceTransfer : quotaTransfer.getTransfers()) {
                        resourceIds.add(Tuples.of(resourceTransfer.getResourceId(), tenantId));
                    }
                }
            }
            if (transfer.getType() == TransferRequestType.PROVISION_TRANSFER) {
                for (ProvisionTransfer provisionTransfer : transfer.getParameters().getProvisionTransfers()) {
                    for (ResourceQuotaTransfer resourceTransfer : provisionTransfer.getSourceAccountTransfers()) {
                        resourceIds.add(Tuples.of(resourceTransfer.getResourceId(), tenantId));
                    }
                    for (ResourceQuotaTransfer resourceTransfer : provisionTransfer.getDestinationAccountTransfers()) {
                        resourceIds.add(Tuples.of(resourceTransfer.getResourceId(), tenantId));
                    }
                }
            }
        }
        return new ArrayList<>(resourceIds);
    }

    private List<Tuple2<String, TenantId>> getUnitsEnsemblesIds(List<ResourceModel> resources,
                                                                TenantId tenantId) {
        Set<Tuple2<String, TenantId>> unitsEnsemblesIds = new HashSet<>();
        for (ResourceModel resource : resources) {
            unitsEnsemblesIds.add(Tuples.of(resource.getUnitsEnsembleId(), tenantId));
        }
        return new ArrayList<>(unitsEnsemblesIds);
    }

    private QuotasMoveStatisticDto toDto(QuotasMoveStatisticResult result) {
        return new QuotasMoveStatisticDto(
                result.getTransfersCreated().stream()
                        .map(TransferRequestModel::getId)
                        .toList(),
                result.getTransfersApplied().stream()
                        .map(TransferRequestModel::getId)
                        .toList(),
                result.getTransfersCreated().stream()
                        .map(transfer -> textFormatterService.buildTransferUrl(transfer.getId()))
                        .toList(),
                result.getTransfersApplied().stream()
                        .map(transfer -> textFormatterService.buildTransferUrl(transfer.getId()))
                        .toList());
    }
}
