package ru.yandex.intranet.d.services.quotas;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import com.google.common.collect.Streams;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.accounts.OperationErrorCollections;
import ru.yandex.intranet.d.model.accounts.OperationErrorKind;
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel;
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel;
import ru.yandex.intranet.d.model.accounts.OperationSource;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.OperationPhase;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionsByResource;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.providers.AccountsSettingsModel;
import ru.yandex.intranet.d.model.providers.ProviderId;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeId;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.integration.providers.ProviderError;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.GetAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownAccountProvisionsDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.KnownProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.LastUpdateDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceComplexKey;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UpdateProvisionResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UserIdDto;
import ru.yandex.intranet.d.util.BiLongPredicate;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedProvider;
import ru.yandex.intranet.d.web.model.folders.front.ProviderPermission;
import ru.yandex.intranet.d.web.model.quotas.AccountsQuotasOperationsDto;
import ru.yandex.intranet.d.web.model.quotas.ProvisionLiteDto;
import ru.yandex.intranet.d.web.model.quotas.ProvisionLiteWithBigIntegers;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsAnswerDto;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionsRequestDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.util.ModelDtoConverter;

import static com.google.common.collect.ImmutableMap.toImmutableMap;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.intranet.d.model.folders.OperationPhase.SUBMIT;
import static ru.yandex.intranet.d.services.quotas.QuotasProvisionAnswerBuilder.UnitEnsembleUnitKey.fromUnitEnsembleIdUnitKey;

/**
 * Quotas provision answer builder.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 * @since 15-12-2020
 */
public class QuotasProvisionAnswerBuilder {
    private static final Logger LOG = LoggerFactory.getLogger(QuotasProvisionAnswerBuilder.class);

    public static final int MAX_RETRY_COUNT = 3;
    private final AtomicInteger updateProvisionRetryCount = new AtomicInteger(MAX_RETRY_COUNT);
    private final AtomicInteger getAccountRetryCount = new AtomicInteger(MAX_RETRY_COUNT);
    /**Actual provision update request from a user**/
    private final UpdateProvisionsRequestDto updateProvisionsRequestDto;
    private final YaUserDetails currentUser;
    private final Locale locale;
    private final TenantId tenantId;
    private final boolean publicApi;
    private UpdateProvisionResponseDto updateProvisionResponseDto;
    private Map<String, ProvisionDto> responseProvisionsByResourceId;
    private UpdateProvisionRequestDto updateProvisionRequestDto;
    private AccountModel accountModel;
    private AccountModel currentAccount;
    /**On first transaction - all accounts for the current account space grouped by account id
     * or just all provider accounts grouped by account id if no accounts spaces are defined.
     * On second transaction - all provider accounts grouped by account id.**/
    private Map<String, AccountModel> accountModelByIdMap;
    private ProviderModel providerModel;
    private FolderModel folderModel;
    private FolderModel actualFolderModel;
    private String providerId;
    private String accountsSpaceId;
    private AccountSpaceModel accountSpaceModel;
    private List<ProvisionLiteWithBigIntegers> requestProvisionsBI;
    private List<ProvisionLiteWithBigIntegers> changingRequestProvisionsBI;
    /**List of resources in the provision update request**/
    private List<ResourceModel> requestResourceModels;
    /**All provider resources except those in the provision update request**/
    private List<ResourceModel> absentResourceModels;
    /**All provider resources for the current account space grouped by resource id, or just all provider resources
     * grouped by resource id if no accounts spaces are defined**/
    private Map<String, ResourceModel> resourceByIdMap;
    /**All provider resources grouped by resource id**/
    private Map<String, ResourceModel> allResourceByIdMap;
    private Map<ResourceComplexKey, ResourceModel> resourceByResourceKeyResponse;
    private List<ResourceTypeModel> requestResourceTypeModels;
    private List<ResourceTypeModel> absentResourceTypeModels;
    private Map<String, String> resourceTypeByIdMap;
    private List<UnitsEnsembleModel> requestUnitsEnsembleModels;
    private List<UnitsEnsembleModel> absentUnitsEnsembleModels;
    private Map<String, UnitsEnsembleModel> ensembleModelByIdMap;
    private Map<UnitEnsembleUnitKey, UnitModel> unitModelByUnitEnsembleUnitKeyMap;
    private List<ResourceSegmentationModel> requestResourceSegmentationModels;
    private List<ResourceSegmentationModel> absentResourceSegmentationModels;
    private List<ResourceSegmentModel> requestResourceSegmentModels;
    private List<ResourceSegmentModel> absentResourceSegmentModels;
    private Map<String, ResourceSegmentationModel> resourceSegmentationsById;
    private Map<String, ResourceSegmentModel> resourceSegmentsById;
    /**Quotas for folder and provider, only those that exist in the DB**/
    private List<QuotaModel> allQuotaModels;
    private List<QuotaModel> newQuotaModels;
    private Map<String, QuotaModel> oldQuotaModelByResourceIdMap;
    private List<QuotaModel> unfreezeQuotaModels;
    private List<QuotaModel> unfreezeQuotaFailureModels;
    /**On first transaction - current provisions in the folder for the current account space
     * or for the whole provider if no accounts spaces are defined.
     * On second transaction - current provisions for the whole provider.**/
    private List<AccountsQuotasModel> oldAccountsQuotasModel;
    /**{@link #oldAccountsQuotasModel} grouped by account and then by resource id**/
    private Map<String, Map<String, AccountsQuotasModel>> oldAccountsQuotasModelByResourceIdByAccountMap;
    private List<AccountsQuotasModel> currentActualQuotas;
    private List<AccountsQuotasModel> actualUpdatedQuotas;
    private Map<String, Long> oldProvisionsByResourceIdMap;
    private Map<String, Long> newProvisionsByResourceIdMap;
    private AccountsQuotasOperationsModel accountsQuotasOperationsModel;
    private AccountsQuotasOperationsModel accountsQuotasOperationsModelSuccess;
    private FolderOperationLogModel folderOperationLogModel;
    private FolderOperationLogModel folderOperationLogModelSuccess;
    private OperationInProgressModel operationInProgressModel;

    public QuotasProvisionAnswerBuilder(UpdateProvisionsRequestDto updateProvisionsRequestDto,
                                        YaUserDetails currentUser, Locale locale, TenantId tenantId,
                                        boolean publicApi) {
        this.updateProvisionsRequestDto = updateProvisionsRequestDto;
        this.currentUser = currentUser;
        this.locale = locale;
        this.tenantId = tenantId;
        this.publicApi = publicApi;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public List<ResourceModel> getRequestResourceModels() {
        return requestResourceModels;
    }

    public AccountModel getAccountModel() {
        return accountModel;
    }

    public ProviderModel getProviderModel() {
        return providerModel;
    }

    public FolderModel getFolderModel() {
        return folderModel;
    }

    public UpdateProvisionsRequestDto getUpdateProvisionsRequestDto() {
        return updateProvisionsRequestDto;
    }

    public String getProviderId() {
        return providerId;
    }

    public YaUserDetails getCurrentUser() {
        return currentUser;
    }

    public AccountsQuotasOperationsModel getAccountsQuotasOperationsModel() {
        return accountsQuotasOperationsModel;
    }

    public FolderOperationLogModel getFolderOperationLogModel() {
        return folderOperationLogModel;
    }

    public List<QuotaModel> getUnfreezeQuotaModels() {
        return unfreezeQuotaModels;
    }

    @Nullable
    public String getAccountsSpaceId() {
        return accountsSpaceId;
    }

    @Nullable
    public AccountSpaceModel getAccountSpaceModel() {
        return accountSpaceModel;
    }

    public Map<String, AccountModel> getAccountModelByIdMap() {
        return accountModelByIdMap;
    }

    public QuotasProvisionAnswerBuilder setAccountSpace(AccountSpaceModel accountSpaceModel) {
        this.accountSpaceModel = accountSpaceModel;
        return this;
    }

    public QuotasProvisionAnswerBuilder setRequestUnitsEnsembleModels(
            List<UnitsEnsembleModel> requestUnitsEnsembleModels) {
        this.requestUnitsEnsembleModels = requestUnitsEnsembleModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setAbsentUnitsEnsembleModels(
            List<UnitsEnsembleModel> absentUnitsEnsembleModels) {
        this.absentUnitsEnsembleModels = absentUnitsEnsembleModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setOldAccountsQuotasModel(List<AccountsQuotasModel> oldAccountsQuotasModel) {
        this.oldAccountsQuotasModel = oldAccountsQuotasModel;
        if (!oldAccountsQuotasModel.isEmpty()) {
            this.oldAccountsQuotasModelByResourceIdByAccountMap = oldAccountsQuotasModel.stream()
                    .collect(Collectors.groupingBy(AccountsQuotasModel::getAccountId,
                            toMap(key -> key.getIdentity().getResourceId(), Function.identity())));
        } else {
            this.oldAccountsQuotasModelByResourceIdByAccountMap = Collections.emptyMap();
        }
        return this;
    }

    public QuotasProvisionAnswerBuilder setAllQuotaModels(List<QuotaModel> allQuotaModels) {
        this.allQuotaModels = allQuotaModels;
        return this;
    }

    @SuppressWarnings("UnusedReturnValue")
    public QuotasProvisionAnswerBuilder setRequestProvisionsBI(List<ProvisionLiteWithBigIntegers> requestProvisionsBI) {
        this.requestProvisionsBI = requestProvisionsBI;
        return this;
    }

    public QuotasProvisionAnswerBuilder setAccountModel(AccountModel accountModel) {
        this.accountModel = accountModel;
        return this;
    }

    public QuotasProvisionAnswerBuilder setProviderModel(ProviderModel providerModel) {
        this.providerModel = providerModel;
        return this;
    }

    public QuotasProvisionAnswerBuilder setFolderModel(FolderModel folderModel) {
        this.folderModel = folderModel;
        return this;
    }

    public QuotasProvisionAnswerBuilder setActualFolderModel(FolderModel actualFolderModel) {
        this.actualFolderModel = actualFolderModel;
        return this;
    }

    public QuotasProvisionAnswerBuilder setRequestResourceModels(List<ResourceModel> requestResourceModels) {
        this.requestResourceModels = requestResourceModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setAbsentResourceModels(List<ResourceModel> absentResourceModels) {
        this.absentResourceModels = absentResourceModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setRequestResourceTypeModels(
            List<ResourceTypeModel> requestResourceTypeModels) {
        this.requestResourceTypeModels = requestResourceTypeModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setAbsentResourceType(List<ResourceTypeModel> absentResourceTypeModels) {
        this.absentResourceTypeModels = absentResourceTypeModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setRequestResourceSegmentationModels(
            List<ResourceSegmentationModel> requestResourceSegmentationModels) {
        this.requestResourceSegmentationModels = requestResourceSegmentationModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setAbsentResourceSegmentationModels(
            List<ResourceSegmentationModel> absentResourceSegmentationModels) {
        this.absentResourceSegmentationModels = absentResourceSegmentationModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setRequestResourceSegmentModels(
            List<ResourceSegmentModel> requestResourceSegmentModels) {
        this.requestResourceSegmentModels = requestResourceSegmentModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setAbsentResourceSegmentModels(
            List<ResourceSegmentModel> absentResourceSegmentModels) {
        this.absentResourceSegmentModels = absentResourceSegmentModels;
        return this;
    }

    public QuotasProvisionAnswerBuilder setChangingUpdatedProvisions(
            List<ProvisionLiteWithBigIntegers> changingUpdatedProvisionsBI) {
        this.changingRequestProvisionsBI = changingUpdatedProvisionsBI;
        return this;
    }

    public QuotasProvisionAnswerBuilder setAccountModelByIdMap(Map<String, AccountModel> accountModelByIdMap) {
        this.accountModelByIdMap = accountModelByIdMap;
        return this;
    }

    public Result<QuotasProvisionAnswerBuilder> prepareQuotasAndResources() {
        // Union to get all provider resources
        List<ResourceModel> resourceModels = Streams.concat(absentResourceModels.stream(),
                requestResourceModels.stream())
                .toList();
        this.allResourceByIdMap = resourceModels.stream()
                .collect(toImmutableMap(ResourceModel::getId, Function.identity()));
        this.resourceByIdMap = resourceModels.stream()
                .filter(resourceModel -> Objects.equals(resourceModel.getAccountsSpacesId(), accountsSpaceId))
                .collect(toMap(ResourceModel::getId, Function.identity()));

        this.resourceTypeByIdMap = Streams.concat(requestResourceTypeModels.stream(), absentResourceTypeModels.stream())
                .collect(toMap(ResourceTypeModel::getId, ResourceTypeModel::getKey));

        List<UnitsEnsembleModel> unitsEnsembleModels = Streams.concat(requestUnitsEnsembleModels.stream(),
                absentUnitsEnsembleModels.stream())
                .toList();
        this.ensembleModelByIdMap = unitsEnsembleModels.stream()
                .collect(toMap(UnitsEnsembleModel::getId, Function.identity()));
        this.unitModelByUnitEnsembleUnitKeyMap = unitsEnsembleModels.stream()
                .flatMap(unitsEnsembleModel -> unitsEnsembleModel.getUnits().stream()
                        .map(unitModel -> Tuples.of(unitsEnsembleModel.getId(), unitModel)))
                .collect(toMap(tuple2 -> fromUnitEnsembleIdUnitKey(tuple2.getT1(), tuple2.getT2().getKey()),
                        Tuple2::getT2));

        this.resourceSegmentationsById = Streams.concat(requestResourceSegmentationModels.stream(),
                        absentResourceSegmentationModels.stream())
                .collect(toMap(ResourceSegmentationModel::getId, Function.identity()));

        this.resourceSegmentsById = Streams.concat(requestResourceSegmentModels.stream(),
                        absentResourceSegmentModels.stream())
                .collect(toMap(ResourceSegmentModel::getId, Function.identity()));

        this.oldQuotaModelByResourceIdMap = allQuotaModels.stream()
                .filter(quotaModel -> resourceByIdMap.containsKey(quotaModel.getResourceId()))
                .collect(toMap(QuotaModel::getResourceId, Function.identity()));

        this.resourceByResourceKeyResponse = resourceByIdMap.keySet().stream()
                .collect(toMap(this::toResourceComplexKey, resourceByIdMap::get));

        return Result.success(this);
    }

    private ResourceTypeId toResourceTypeId(QuotaModel q) {
        String resourceTypeId =
                allResourceByIdMap.get(q.getResourceId()).getResourceTypeId();
        return new ResourceTypeId(resourceTypeId);
    }

    public ProvisionOperationResult buildAnswer() {
        Map<String, QuotaModel> quotaModelsMap = unfreezeQuotaModels.stream()
                .collect(toMap(QuotaModel::getResourceId, Function.identity()));
        allQuotaModels.forEach(quotaModel -> quotaModelsMap.putIfAbsent(quotaModel.getResourceId(), quotaModel));
        Map<ResourceTypeId, List<QuotaSums>> collectedQuotas = quotaModelsMap.values().stream()
                .collect(groupingBy(this::toResourceTypeId, Collectors.mapping(QuotaSums::from,
                        Collectors.toList())));

        List<AccountModel> accountsFromProvider = accountModelByIdMap.values().stream()
                .filter(account -> account.getProviderId().equals(providerId)).collect(Collectors.toList());

        Map<String, List<AccountsQuotasModel>> accountsQuotasByAccountId = new HashMap<>();
        for (AccountsQuotasModel accountsQuotas : actualUpdatedQuotas) {
            accountsQuotasByAccountId.putIfAbsent(accountsQuotas.getAccountId(), new ArrayList<>());
            accountsQuotasByAccountId.get(accountsQuotas.getAccountId()).add(accountsQuotas);
        }
        Set<AccountsQuotasModel.Identity> actualUpdatedQuotaIdentities = actualUpdatedQuotas.stream()
                .map(AccountsQuotasModel::getIdentity).collect(Collectors.toSet());
        for (AccountsQuotasModel accountsQuotas : currentActualQuotas) {
            if (!actualUpdatedQuotaIdentities.contains(accountsQuotas.getIdentity())) {
                accountsQuotasByAccountId.putIfAbsent(accountsQuotas.getAccountId(), new ArrayList<>());
                accountsQuotasByAccountId.get(accountsQuotas.getAccountId()).add(accountsQuotas);
            }
        }

        Map<String, QuotaModel> quotasByResourceId = newQuotaModels.stream()
                .collect(toMap(QuotaModel::getResourceId, Function.identity()));

        ExternalAccountUrlFactory externalAccountUrlFactory =
                providerModel.getAccountsSettings().getExternalAccountUrlTemplates() == null ? null :
                        new ExternalAccountUrlFactory(
                                providerModel.getAccountsSettings().getExternalAccountUrlTemplates(),
                                providerModel.getServiceId(),
                                resourceSegmentationsById,
                                resourceSegmentsById,
                                accountSpaceModel == null ? List.of() : List.of(ModelDtoConverter.toDto(
                                        accountSpaceModel, resourceSegmentationsById, resourceSegmentsById, locale))
                        );

        ExpandedProvider expandedProvider = new ExpandedProviderBuilder(
                locale,
                allResourceByIdMap,
                ensembleModelByIdMap,
                externalAccountUrlFactory
        ).toExpandedProvider(
                new ProviderId(providerId),
                collectedQuotas,
                accountsFromProvider,
                accountsQuotasByAccountId,
                quotasByResourceId,
                // If provision was updated then user has both permissions for this provider
                Set.of(ProviderPermission.CAN_UPDATE_PROVISION, ProviderPermission.CAN_MANAGE_ACCOUNT),
                false
        );
        AccountsQuotasOperationsDto accountsQuotasOperationsDto =
                new AccountsQuotasOperationsDto(accountsQuotasOperationsModel.getOperationId(),
                        AccountsQuotasOperationsModel.RequestStatus.OK);
        ExpandedProvisionResult expandedResult = new ExpandedProvisionResult(requestResourceModels,
                actualUpdatedQuotas, currentActualQuotas, allResourceByIdMap, ensembleModelByIdMap, currentAccount);
        return ProvisionOperationResult.success(new UpdateProvisionsAnswerDto(expandedProvider,
                accountsQuotasOperationsDto), accountsQuotasOperationsModel.getOperationId(), expandedResult);
    }

    public QuotasProvisionAnswerBuilder setProviderId(String providerId) {
        this.providerId = providerId;
        return this;
    }

    public QuotasProvisionAnswerBuilder setAccountsSpaceId(String accountsSpaceId) {
        this.accountsSpaceId = accountsSpaceId;
        return this;
    }

    public QuotasProvisionAnswerBuilder setOldProvisionsByResourceIdMap(
            Map<String, Long> oldProvisionsByResourceIdMap) {
        this.oldProvisionsByResourceIdMap = oldProvisionsByResourceIdMap;
        return this;
    }

    public QuotasProvisionAnswerBuilder setNewProvisionsByResourceIdMap(
            Map<String, Long> newProvisionsByResourceIdMap) {
        this.newProvisionsByResourceIdMap = newProvisionsByResourceIdMap;
        return this;
    }

    public QuotasProvisionAnswerBuilder setCurrentActualQuotas(List<AccountsQuotasModel> currentActualQuotas) {
        this.currentActualQuotas = currentActualQuotas;
        return this;
    }

    public List<QuotaModel> calculateNewQuotaModels() {
        newQuotaModels = new ArrayList<>();

        for (ProvisionLiteWithBigIntegers provision : changingRequestProvisionsBI) {
            String resourceId = provision.getResourceId().orElseThrow();
            QuotaModel quotaModel = oldQuotaModelByResourceIdMap.get(resourceId);

            long oldProvision = oldProvisionsByResourceIdMap.get(resourceId);
            long newProvision = newProvisionsByResourceIdMap.get(resourceId);

            if (newProvision > oldProvision) {
                long diff = Math.abs(newProvision - oldProvision);
                newQuotaModels.add(QuotaModel.builder(quotaModel)
                        .frozenQuota(quotaModel.getFrozenQuota() + diff)
                        .balance(quotaModel.getBalance() - diff)
                        .build());
            } else {
                newQuotaModels.add(quotaModel);
            }
        }

        return newQuotaModels;
    }

    public AccountsQuotasOperationsModel prepareAccountsQuotasOperation() {
        List<OperationChangesModel.Provision> provisionList = newProvisionsByResourceIdMap.entrySet().stream()
                .map(e -> new OperationChangesModel.Provision(e.getKey(), e.getValue()))
                .collect(Collectors.toList());

        List<OperationChangesModel.Provision> frozenProvisionList = new ArrayList<>();
        for (ProvisionLiteWithBigIntegers provision : changingRequestProvisionsBI) {
            String resourceId = provision.getResourceId().orElseThrow();
            long oldProvision = oldProvisionsByResourceIdMap.get(resourceId);
            long newProvision = newProvisionsByResourceIdMap.get(resourceId);
            if (newProvision > oldProvision) {
                long diff = Math.abs(newProvision - oldProvision);
                frozenProvisionList.add(new OperationChangesModel.Provision(resourceId, diff));
            } else {
                frozenProvisionList.add(new OperationChangesModel.Provision(resourceId, 0L));
            }
        }

        accountsQuotasOperationsModel = AccountsQuotasOperationsModel.builder()
                .setTenantId(tenantId)
                .setOperationId(UUID.randomUUID().toString())
                .setLastRequestId(UUID.randomUUID().toString())
                .setCreateDateTime(Instant.now())
                .setOperationSource(OperationSource.USER)
                .setOperationType(AccountsQuotasOperationsModel.OperationType.UPDATE_PROVISION)
                .setAuthorUserId(currentUser.getUser().orElseThrow().getId())
                .setAuthorUserUid(currentUser.getUid().orElseThrow())
                .setProviderId(providerModel.getId())
                .setAccountsSpaceId(accountModel.getAccountsSpacesId().orElse(null))
                .setUpdateDateTime(null)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.WAITING)
                .setErrorMessage(null)
                .setRequestedChanges(OperationChangesModel.builder()
                        .accountId(accountModel.getId())
                        .updatedProvisions(provisionList)
                        .frozenProvisions(frozenProvisionList)
                        .build())
                .setOrders(OperationOrdersModel.builder().submitOrder(folderModel.getNextOpLogOrder()).build())
                .setErrorKind(null)
                .build();

        return accountsQuotasOperationsModel;
    }

    public OperationInProgressModel prepareOperationInProgress() {
        operationInProgressModel = new OperationInProgressModel(tenantId,
                accountsQuotasOperationsModel.getOperationId(),
                folderModel.getId(), accountModel.getId(), 0L);
        return operationInProgressModel;
    }

    public WithTenant<OperationInProgressModel.Key> getOperationInProgressKey() {
        return WithTenant.from(Tuples.of(operationInProgressModel.getKey(), tenantId));
    }

    public FolderOperationLogModel prepareFolderOperationLogModel() {
        Map<String, Long> oldQuotasByResource = oldQuotaModelByResourceIdMap.values().stream()
                .filter(quotaModel -> newProvisionsByResourceIdMap.containsKey(quotaModel.getResourceId()))
                .collect(toMap(QuotaModel::getResourceId, QuotaModel::getQuota));

        Map<String, Long> oldBalanceByResource = oldQuotaModelByResourceIdMap.values().stream()
                .filter(quotaModel -> newProvisionsByResourceIdMap.containsKey(quotaModel.getResourceId()))
                .collect(toMap(QuotaModel::getResourceId, QuotaModel::getBalance));

        var oldProvisionHistoryModelByResource = oldAccountsQuotasModel.stream()
                .filter(accountModel -> this.accountModel.getId().equals(accountModel.getAccountId())
                        && newProvisionsByResourceIdMap.containsKey(accountModel.getResourceId()))
                .collect(toMap(AccountsQuotasModel::getResourceId, accountsQuotasModel ->
                        new ProvisionHistoryModel(accountsQuotasModel.getProvidedQuota(),
                                accountsQuotasModel.getLastReceivedProvisionVersion().orElse(null))));

        var oldQuotasByAccountMap = Map.of(accountModel.getId(),
                new ProvisionsByResource(oldProvisionHistoryModelByResource));

        Map<String, Long> newQuotasByResource = newQuotaModels.stream()
                .collect(toMap(QuotaModel::getResourceId, QuotaModel::getQuota));

        Map<String, Long> newBalanceByResource = newQuotaModels.stream()
                .collect(toMap(QuotaModel::getResourceId, QuotaModel::getBalance));

        var newQuotasByAccountMap = Map.of(accountModel.getId(),
                new ProvisionsByResource(newProvisionsByResourceIdMap.entrySet().stream()
                        .collect(toMap(Map.Entry::getKey, v -> new ProvisionHistoryModel(v.getValue(),
                                oldProvisionHistoryModelByResource.getOrDefault(v.getKey(),
                                        new ProvisionHistoryModel(0, null)).getVersion().orElse(-1L) + 1L))))
        );

        folderOperationLogModel = FolderOperationLogModel.builder()
                .setTenantId(tenantId)
                .setFolderId(folderModel.getId())
                .setOperationDateTime(accountsQuotasOperationsModel.getCreateDateTime())
                .setId(UUID.randomUUID().toString())
                .setProviderRequestId(accountsQuotasOperationsModel.getLastRequestId().orElseThrow())
                .setOperationType(FolderOperationType.PROVIDE_REVOKE_QUOTAS_TO_ACCOUNT)
                .setAuthorUserId(currentUser.getUser().orElseThrow().getId())
                .setAuthorUserUid(currentUser.getUid().orElseThrow())
                .setAuthorProviderId(null)
                .setSourceFolderOperationsLogId(null)
                .setDestinationFolderOperationsLogId(null)
                .setOldFolderFields(null)
                .setOldQuotas(new QuotasByResource(oldQuotasByResource))
                .setOldBalance(new QuotasByResource(oldBalanceByResource))
                .setOldProvisions(new QuotasByAccount(oldQuotasByAccountMap))
                .setOldAccounts(null)
                .setNewQuotas(new QuotasByResource(newQuotasByResource))
                .setNewBalance(new QuotasByResource(newBalanceByResource))
                .setNewProvisions(new QuotasByAccount(newQuotasByAccountMap))
                .setActuallyAppliedProvisions(null)
                .setNewAccounts(null)
                .setAccountsQuotasOperationsId(accountsQuotasOperationsModel.getOperationId())
                .setQuotasDemandsId(null)
                .setOperationPhase(SUBMIT)
                .setOrder(accountsQuotasOperationsModel.getOrders().getSubmitOrder())
                .build();

        return folderOperationLogModel;
    }

    public FolderModel getIncrementedFolderModel() {
        return folderModel.toBuilder()
                .setNextOpLogOrder(folderModel.getNextOpLogOrder() + 1L).build();
    }

    public FolderModel getIncrementedFolderModelSuccess() {
        return actualFolderModel.toBuilder()
                .setNextOpLogOrder(actualFolderModel.getNextOpLogOrder() + 1L).build();
    }

    public UpdateProvisionRequestDto getUpdateProvisionRequestDto() {
        if (updateProvisionRequestDto == null) {

            updateProvisionRequestDto = new UpdateProvisionRequestDto(
                    folderModel.getId(),
                    publicApi
                            ? folderModel.getServiceId()
                            : getUpdateProvisionsRequestDto().getServiceId().orElseThrow(),
                    toProvisionRequestDto(),
                    toKnownProvisions(),
                    toAuthor(),
                    accountsQuotasOperationsModel.getOperationId(),
                    toAccountsSpaceKey()
            );
        }

        return updateProvisionRequestDto;
    }

    private List<ProvisionRequestDto> toProvisionRequestDto() {
        List<ProvisionRequestDto> result = new ArrayList<>();
        Set<String> updatedResourceIds = changingRequestProvisionsBI.stream()
                .map(r -> r.getResourceId().orElseThrow())
                .collect(Collectors.toSet());
        Set<String> nonZeroCurrentResourceIds = oldAccountsQuotasModelByResourceIdByAccountMap
                .getOrDefault(accountModel.getId(), Map.of()).entrySet()
                .stream().filter(e -> e.getValue().getProvidedQuota() != null && e.getValue().getProvidedQuota() != 0L)
                .map(Map.Entry::getKey).collect(Collectors.toSet());
        Set<String> resourceIdsToSend = Sets.union(updatedResourceIds, nonZeroCurrentResourceIds).stream()
                .filter(id -> {
                    ResourceModel resource = allResourceByIdMap.get(id);
                    return !resource.isDeleted() && resource.isManaged() && !resource.isReadOnly();
                }).collect(Collectors.toSet());
        resourceIdsToSend.stream().sorted().forEach(resourceId -> {
            if (updatedResourceIds.contains(resourceId)) {
                long provision = newProvisionsByResourceIdMap.get(resourceId);
                Tuple2<BigDecimal, UnitModel> valueUnit = toProviderApiUnit(provision, resourceId);
                result.add(new ProvisionRequestDto(toResourceKeyRequestDto(resourceId),
                        valueUnit.getT1().longValue(), valueUnit.getT2().getKey()));
            } else {
                long provision = oldAccountsQuotasModelByResourceIdByAccountMap
                        .getOrDefault(accountModel.getId(), Map.of())
                        .get(resourceId)
                        .getProvidedQuota();
                Tuple2<BigDecimal, UnitModel> valueUnit = toProviderApiUnit(provision, resourceId);
                result.add(new ProvisionRequestDto(toResourceKeyRequestDto(resourceId),
                        valueUnit.getT1().longValue(), valueUnit.getT2().getKey()));
            }
        });
        return result;
    }

    private Tuple2<BigDecimal, UnitModel> toProviderApiUnit(long value, String resourceId) {
        ResourceModel resourceModel = resourceByIdMap.get(resourceId);
        UnitsEnsembleModel unitsEnsembleModel = ensembleModelByIdMap.get(resourceModel.getUnitsEnsembleId());
        return Units.convertToApiStrictly(value, resourceModel, unitsEnsembleModel);
    }

    private ResourceKeyRequestDto toResourceKeyRequestDto(String resourceId) {
        ResourceModel resourceModel = resourceByIdMap.get(resourceId);
        return new ResourceKeyRequestDto(resourceTypeByIdMap.get(resourceModel.getResourceTypeId()),
                toSegmentKeyRequestDto(resourceModel.getSegments() != null
                        ? resourceModel.getSegments().stream()
                        .filter(resourceSegmentSettingsModel -> accountSpaceModel == null
                                || !accountSpaceModel.getSegments()
                                .contains(resourceSegmentSettingsModel))
                        .collect(Collectors.toSet())
                        : Set.of()
                ));
    }

    private List<SegmentKeyRequestDto> toSegmentKeyRequestDto(Set<ResourceSegmentSettingsModel> segments) {
        return segments.stream()
                .map(segment -> new SegmentKeyRequestDto(
                        resourceSegmentationsById.get(segment.getSegmentationId()).getKey(),
                        resourceSegmentsById.get(segment.getSegmentId()).getKey()))
                .collect(Collectors.toList());
    }

    private List<KnownAccountProvisionsDto> toKnownProvisions() {
        List<KnownAccountProvisionsDto> result = new ArrayList<>();
        Set<String> updatedResourceIds = changingRequestProvisionsBI.stream()
                .map(r -> r.getResourceId().orElseThrow())
                .collect(Collectors.toSet());
        Set<String> nonZeroCurrentResourceIds = oldAccountsQuotasModelByResourceIdByAccountMap
                .getOrDefault(accountModel.getId(), Map.of()).entrySet()
                .stream().filter(e -> e.getValue().getProvidedQuota() != null && e.getValue().getProvidedQuota() != 0L)
                .map(Map.Entry::getKey).collect(Collectors.toSet());
        Set<String> resourceIdsToSend = Sets.union(updatedResourceIds, nonZeroCurrentResourceIds).stream()
                .filter(id -> {
                    ResourceModel resource = allResourceByIdMap.get(id);
                    return !resource.isDeleted() && resource.isManaged() && !resource.isReadOnly();
                }).collect(Collectors.toSet());
        accountModelByIdMap.values().stream().sorted(Comparator.comparing(AccountModel::getId)).forEach(account -> {
            List<KnownProvisionDto> accountResult = new ArrayList<>();
            resourceIdsToSend.stream().sorted().forEach(resourceId -> {
                accountResult.add(Optional.ofNullable(oldAccountsQuotasModelByResourceIdByAccountMap
                                .getOrDefault(account.getId(), Map.of())
                                .getOrDefault(resourceId, null))
                        .map(provision -> {
                            Tuple2<BigDecimal, UnitModel> valueUnit = toProviderApiUnit(provision
                                    .getProvidedQuota(), provision.getResourceId());
                            return new KnownProvisionDto(toResourceKeyRequestDto(resourceId),
                                    valueUnit.getT1().longValue(), valueUnit.getT2().getKey());
                        }).orElseGet(() -> {
                            Tuple2<BigDecimal, UnitModel> valueUnit = toProviderApiUnit(0L, resourceId);
                            return new KnownProvisionDto(toResourceKeyRequestDto(resourceId),
                                    valueUnit.getT1().longValue(), valueUnit.getT2().getKey());
                        }));
            });
            result.add(new KnownAccountProvisionsDto(account.getOuterAccountIdInProvider(), accountResult));
        });
        return result;
    }

    private UserIdDto toAuthor() {
        return new UserIdDto(currentUser.getUid().orElseThrow(),
                currentUser.getUser().orElseThrow().getPassportLogin().orElseThrow());
    }

    private AccountsSpaceKeyRequestDto toAccountsSpaceKey() {
        Set<ResourceSegmentSettingsModel> segments = accountSpaceModel != null ? accountSpaceModel.getSegments() :
                Set.of();
        return new AccountsSpaceKeyRequestDto(toSegmentKeyRequestDto(segments));
    }

    private List<QuotaModel> calculateFrozenQuotas(
            List<QuotaModel> quotas,
            Map<String, Long> actuallyAppliedAmountByResourceIdMap,
            BiLongPredicate newOldProvisionsSetBalancePredicate
    ) {
        List<QuotaModel> result = new ArrayList<>();

        Map<String, QuotaModel> quotasModelByResourceIdMap = quotas.stream()
                .collect(toMap(QuotaModel::getResourceId, Function.identity()));

        for (ProvisionLiteWithBigIntegers provision : changingRequestProvisionsBI) {
            String resourceId = provision.getResourceId().orElseThrow();
            QuotaModel quotaModel = quotasModelByResourceIdMap.get(resourceId);

            long oldProvision = oldProvisionsByResourceIdMap.get(resourceId);
            long newProvision = newProvisionsByResourceIdMap.get(resourceId);

            long diff = Math.abs(newProvision - oldProvision);

            QuotaModel.Builder builder = QuotaModel.builder(quotaModel);

            if (newProvision > oldProvision) {
                builder.frozenQuota(quotaModel.getFrozenQuota() - diff);
            }

            Long newAmount = actuallyAppliedAmountByResourceIdMap.get(resourceId);
            if (newOldProvisionsSetBalancePredicate.test(newProvision, oldProvision)) {
                if (newAmount != null) {
                    diff = oldProvision - newAmount;
                }
                builder.balance(quotaModel.getBalance() + diff);
            } else if (newAmount != null && newAmount != newProvision) {
                builder.balance(quotaModel.getBalance() + newProvision - newAmount);
            }

            result.add(builder.build());
        }

        return result;
    }

    /**
     * Готовим данные для обновления квот.
     *
     * @param quotas              Текущее состояние обещанных квот в фолдере (прочитано в открытой транзакции)
     * @param response            Ответ провайдера о фактически спущенных квотах
     * кроме параметров метода, на вход используются поля:
     * accountModel               Аккаунт, в который производится спуск
     * currentActualQuotas        Текущее состояние спущенных квот в аккаунте (прочитано в открытой транзакции)
     * updateProvisionsRequestDto Запрос, направленный в провайдера
     * resourceByIdMap            Используемые ресурсы
     * ensembleModelByIdMap       Используемые ансамбли единиц измерения
     *
     * Результаты записываются в поля:
     * updateProvisionResponseDto Ответ провайдера о фактически спущенных квотах (зачем?)
     * unfreezeQuotaModels        Обновленные данные о квоте на фолдере (разморозка + уточненный баланс)
     *
     * todo: перестать использовать мутабельные объекты вместо входных и выходных параметров
     *
     * Должно заменить calculateUnfreezeQuotas
     * @see QuotasProvisionAnswerBuilder#calculateUnfreezeQuotas
     */
    QuotasProvisionAnswerBuilder validateAndPrepareStateAfterProvide(
            List<QuotaModel> quotas, UpdateProvisionResponseDto response
    ) {
        updateProvisionResponseDto = response; // todo: зачем?
        List<QuotaModel> updatedQuotas = new ArrayList<>(); // результат

        // первичная валидация и подготовка входных данных
        Map<String, ProvisionDto> responseProvisionsByResourceIdTmp = new HashMap<>();
        for (ProvisionDto provision : response.getProvisions().orElseThrow()) {
            ResourceModel provisionResource = resourceByResourceKeyResponse.get(
                    new ResourceComplexKey(provision.getResourceKey().orElseThrow())
            );
            if (provisionResource == null) {
                LOG.info("Resource for responded provision not found in request: " + provision.getResourceKey());
                continue;
            }
            if (responseProvisionsByResourceIdTmp.containsKey(provisionResource.getId())) {
                LOG.error("Duplicate resource in provision: " + provision);
                continue;
            }
            responseProvisionsByResourceIdTmp.put(provisionResource.getId(), provision);
        }
        responseProvisionsByResourceId = responseProvisionsByResourceIdTmp;
        Map<String, QuotaModel> currentQuotasByResourceId = quotas.stream().collect(
                toMap(QuotaModel::getResourceId, Function.identity()));
        Map<String, AccountsQuotasModel> currentProvisionsByResourceId = currentActualQuotas.stream()
                .filter(accountsQuotasModel -> accountsQuotasModel.getAccountId().equals(accountModel.getId()))
                .collect(toMap(AccountsQuotasModel::getResourceId, Function.identity()));

        // По каждому ресурсу у нас есть:
        // (ресурсы, которых не было в запросе, отбрасываем)
        for (ProvisionLiteDto requested: updateProvisionsRequestDto.getUpdatedProvisions().orElseThrow()) {
            String resourceId = requested.getResourceId().orElseThrow();
            ResourceModel resource = resourceByIdMap.get(resourceId);
            UnitsEnsembleModel unitsEnsemble = ensembleModelByIdMap.get(resource.getUnitsEnsembleId());
            UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId()).orElseThrow();

            //   1. текущее состояние обещанных квот (включая замороженные)
            QuotaModel currentQuotaModel = currentQuotasByResourceId.get(resourceId);
            BigDecimal currentBalance = BigDecimal.valueOf(currentQuotaModel.getBalance());
            BigDecimal currentFrozenQuota = BigDecimal.valueOf(currentQuotaModel.getFrozenQuota());

            //   2. текущее состояние спущенных квот (в этом аккаунте)
            AccountsQuotasModel currentProvisionModel = currentProvisionsByResourceId.get(resourceId);
            BigDecimal currentProvision = currentProvisionModel != null ?
                    new BigDecimal(currentProvisionModel.getProvidedQuota()) : BigDecimal.ZERO;

            //   3. состояние квот, в которое хотел привести наш запрос
            UnitModel fromUnit;
            if (publicApi) {
                fromUnit = unitsEnsemble.unitByKey(requested.getProvidedAmountUnitId().orElseThrow()).orElseThrow();
            } else {
                fromUnit = unitsEnsemble.unitById(requested.getProvidedAmountUnitId().orElseThrow()).orElseThrow();
            }
            BigDecimal requestedProvision = Units.convert(
                    new BigDecimal(requested.getProvidedAmount().orElseThrow()),
                    fromUnit,
                    baseUnit
            );

            //   4. замороженная запросом квота
            BigDecimal oldProvision = publicApi
                    ? currentProvision
                    : Units.convert(new BigDecimal(requested.getOldProvidedAmount().orElseThrow()),
                    unitsEnsemble.unitById(requested.getOldProvidedAmountUnitId().orElseThrow()).orElseThrow(),
                    baseUnit);
            BigDecimal frozenQuotaForRequest = requestedProvision.compareTo(oldProvision) > 0 ?
                    requestedProvision.subtract(oldProvision) :
                    BigDecimal.ZERO;

            //   5. ответ провайдера о новом состоянии спущенной квоты
            BigDecimal responseProvided;
            ProvisionDto responseProvidedDto = responseProvisionsByResourceId.get(resourceId);
            if (responseProvidedDto == null) {
                responseProvided = BigDecimal.ZERO;
                if (requestedProvision.compareTo(BigDecimal.ZERO) != 0) {
                    LOG.error("Provision for requested non-zero resource not found in provider response." +
                            " Resource: " + resource + ". Response: " + response);
                }
            } else {
                responseProvided = Units.convert(
                        new BigDecimal(responseProvidedDto.getProvidedAmount().orElseThrow()),
                        unitModelByUnitEnsembleUnitKeyMap.get(new UnitEnsembleUnitKey(
                                resource.getUnitsEnsembleId(),
                                responseProvidedDto.getProvidedAmountUnitKey().orElseThrow()
                        )),
                        baseUnit
                );
            }
            // todo: добавить контроль версий данных, если они провайдером поддерживаются

            // =====================================================================================================
            // В результате нам нужно:

            //  1. разморозить квоту в том объёме, в каком заморозили на старте операции
            //     (потому что кто ещё её разморозит?), разморозить -- значит вернуть на баланс
            BigDecimal newFrozenQuota = currentFrozenQuota.subtract(frozenQuotaForRequest);
            BigDecimal newBalance = currentBalance.add(frozenQuotaForRequest);

            //  2. вычислить дельту спущенной квоты по итогу операции --
            //     -- это разность между новым (по ответу провайдера) и текущим состоянием спущенной квоты
            BigDecimal provisionDelta = responseProvided.subtract(currentProvision);

            //  3. изменить баланс на дельту спущенной квоты по итогу операции (вычесть её из баланса)
            newBalance = newBalance.subtract(provisionDelta);

            //  4. если результат отличается от запроса -- добавить в ответ предупреждение
            if (requestedProvision.compareTo(responseProvided) != 0) {
                LOG.error("Responded provision not match requested: requested " + requestedProvision
                        + ", responded " + responseProvided + ". Account: " + accountModel + ". Resource: " + resource);
            }

            //  5. изменить спущенную и аллоцированную квоту в аккаунте по ответу провайдера
            // todo

            //  6. запомнить разницу нового и старого значения обещанных квот для истории
            // todo

            //  7. запомнить разницу нового и старого значения спущенной квоты для истории
            // todo

            updatedQuotas.add(QuotaModel.builder(currentQuotaModel)
                    .frozenQuota(newFrozenQuota.longValue())
                    .balance(newBalance.longValue())
                    .build()
            );
        }

        unfreezeQuotaModels = updatedQuotas;
        return this;
    }

    public List<QuotaModel> calculateUnfreezeQuotas(List<QuotaModel> quotas, UpdateProvisionResponseDto responseDto) {
        updateProvisionResponseDto = responseDto;
        Set<String> requestResourceIds = requestResourceModels.stream()
                .map(ResourceModel::getId).collect(toSet());
        Map<String, Long> actuallyAppliedAmountByResourceIdMap =
                responseDto.getProvisions().map(provisionDtos -> provisionDtos.stream()
                                .filter(provisionDto -> provisionDto.getResourceKey().isPresent()
                                        && provisionDto.getProvidedAmount().isPresent()
                                        && provisionDto.getProvidedAmountUnitKey().isPresent())
                                .map((Function<ProvisionDto, Optional<Tuple2<String, Long>>>) provisionDto -> {
                                    ResourceModel resourceModel = resourceByResourceKeyResponse
                                            .get(new ResourceComplexKey(provisionDto.getResourceKey().get()));
                                    String resourceId = resourceModel == null ? null : resourceModel.getId();
                                    if (resourceId == null) {
                                        return Optional.empty();
                                    }
                                    if (!requestResourceIds.contains(resourceId)) {
                                        // Skip received resources missing from the original request
                                        return Optional.empty();
                                    }
                                    ResourceModel resource = resourceByIdMap.get(resourceId);
                                    String unitsEnsembleId = resource.getUnitsEnsembleId();
                                    return Units.convertFromApi(provisionDto.getProvidedAmount().get(),
                                                    resource, ensembleModelByIdMap.get(unitsEnsembleId),
                                                    unitModelByUnitEnsembleUnitKeyMap.get(fromUnitEnsembleIdUnitKey(
                                                            unitsEnsembleId,
                                                            provisionDto.getProvidedAmountUnitKey().get())))
                                            .map(amount -> Tuples.of(resourceId, amount));
                                })
                                .filter(Optional::isPresent)
                                .collect(toMap(k -> k.get().getT1(),
                                        v -> v.get().getT2())))
                        .orElse(Collections.emptyMap());

        unfreezeQuotaModels = calculateFrozenQuotas(quotas, actuallyAppliedAmountByResourceIdMap,
                (newProvision, oldProvision) -> newProvision < oldProvision);

        return unfreezeQuotaModels;
    }

    public List<QuotaModel> calculateUnfreezeQuotasFailure(List<QuotaModel> quotas) {
        unfreezeQuotaFailureModels =  calculateFrozenQuotas(quotas, Collections.emptyMap(),
                (newProvision, oldProvision) -> newProvision > oldProvision
        );

        return unfreezeQuotaFailureModels;
    }

    public List<AccountsQuotasModel> calculateUpdatedAccountQuota() {
        actualUpdatedQuotas = new ArrayList<>();

        Optional<Long> accountVersion = updateProvisionResponseDto.getAccountVersion();

        Map<String, AccountsQuotasModel> currentAccountQuotasByResourceId = currentActualQuotas.stream()
                .filter(accountsQuotasModel -> accountsQuotasModel.getAccountId().equals(accountModel.getId()))
                .collect(toMap(AccountsQuotasModel::getResourceId, Function.identity()));

        AccountsSettingsModel accountsSettings = providerModel.getAccountsSettings();
        boolean perAccountVersionSupported = accountsSettings.isPerAccountVersionSupported();
        boolean perProvisionVersionSupported = accountsSettings.isPerProvisionVersionSupported();
        boolean perProvisionLastUpdateSupported = accountsSettings.isPerProvisionLastUpdateSupported();

        if (perAccountVersionSupported && accountVersion.isPresent()) {
            if (accountVersion.get() <= currentAccount.getVersion()) {
                return actualUpdatedQuotas;
            }
        }

        // перебираем все ресурсы из запроса
        for (ResourceModel resource: requestResourceModels) {
            // определяем единицы измерения для ресурса
            String unitsEnsembleId = resource.getUnitsEnsembleId();
            UnitsEnsembleModel unitsEnsemble = ensembleModelByIdMap.get(unitsEnsembleId);
            UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId()).orElseThrow();

            // определяем текущее состояние квоты по ресурсу
            AccountsQuotasModel currentAccountsQuotasModel = currentAccountQuotasByResourceId.computeIfAbsent(
                    resource.getId(),
                    key -> new AccountsQuotasModel.Builder()
                            .setTenantId(tenantId)
                            .setAccountId(accountModel.getId())
                            .setResourceId(resource.getId())
                            .setProvidedQuota(0L)
                            .setAllocatedQuota(0L)
                            .setFolderId(folderModel.getId())
                            .setProviderId(providerModel.getId())
                            .setLastProvisionUpdate(accountsQuotasOperationsModelSuccess.getUpdateDateTime()
                                    .orElseThrow())
                            .setLastReceivedProvisionVersion(null)
                            .setLatestSuccessfulProvisionOperationId(null)
                            .build());

            // получаем ответ провайдера по этому ресурсу
            Optional<ProvisionDto> responseProvision = Optional.ofNullable(
                    responseProvisionsByResourceId.get(resource.getId())
            );
            Optional<Long> responseVersion = responseProvision.flatMap(ProvisionDto::getQuotaVersion);
            Optional<Instant> responseLastUpdate = responseProvision.flatMap(ProvisionDto::getLastUpdate)
                    .flatMap(LastUpdateDto::getTimestamp)
                    .map(Instant::ofEpochMilli);

            // пропускаем устаревшие ответы
            if (responseProvision.isPresent()) {
                Optional<Long> currentVersion = currentAccountsQuotasModel.getLastReceivedProvisionVersion();
                Optional<Instant> currentLastUpdate = Optional.ofNullable(
                        currentAccountsQuotasModel.getLastProvisionUpdate());
                if (perProvisionVersionSupported && responseVersion.isPresent() && currentVersion.isPresent()) {
                    if (responseVersion.get() <= currentVersion.get()) {
                        LOG.warn("Current version is greater than the response one. "
                                + resource + responseProvision + currentAccountsQuotasModel);
                        continue;
                    }
                }
                if (perProvisionLastUpdateSupported
                        && responseLastUpdate.isPresent() && currentLastUpdate.isPresent()
                ) {
                    if (currentLastUpdate.get().isAfter(responseLastUpdate.get())) {
                        LOG.warn("Current last update timestamp is later than the response one. "
                                + resource + responseProvision + currentAccountsQuotasModel);
                        // не пропускаем квоту в этом случае; считаем, что часы разъехались; надеемся на синк
                    }
                }
            }

            // получаем запрошенную квоту
            Long requestedProvision = newProvisionsByResourceIdMap.get(resource.getId());
            if (requestedProvision == null) {
                LOG.warn("Not found requested provision of " + resource);
                continue;
            }
            // если запрошена нулевая квота, разрешаем отсутствующий ответ, иначе пропускаем
            // https://st.yandex-team.ru/DISPENSER-4097
            if (responseProvision.isEmpty() && !requestedProvision.equals(0L)) {
                LOG.warn("Not found response for non-zero provision of " + resource);
                continue;
            }
            // вычисляем новое значение спущенной квоты
            long respondedProvidedAmount = responseProvision.flatMap(ProvisionDto::getProvidedAmount).orElse(0L);
            // вычисляем единицу измерения для нового значения спущенной квоты
            UnitModel respondedProvidedAmountUnit = responseProvision.flatMap(ProvisionDto::getProvidedAmountUnitKey)
                    .map(unitKey -> fromUnitEnsembleIdUnitKey(unitsEnsembleId, unitKey))
                    .map(unitModelByUnitEnsembleUnitKeyMap::get)
                    .orElse(baseUnit);
            // переводим в базовую единицу
            Optional<Long> newProvidedAmount = Units.convertFromApi(
                    respondedProvidedAmount, resource, unitsEnsemble, respondedProvidedAmountUnit
            );
            if (newProvidedAmount.isEmpty()) {
                LOG.error("Can't convert to base unit provision of " + resource + responseProvision);
                continue;
            }

            // вычисляем новое значение алоцированной квоты
            long respondedAllocatedAmount = responseProvision.flatMap(ProvisionDto::getAllocatedAmount).orElse(0L);
            UnitModel respondedAllocatedAmountUnit = responseProvision.flatMap(ProvisionDto::getAllocatedAmountUnitKey)
                    .map(unitKey -> fromUnitEnsembleIdUnitKey(unitsEnsembleId, unitKey))
                    .map(unitModelByUnitEnsembleUnitKeyMap::get)
                    .orElse(baseUnit);
            Optional<Long> newAllocatedAmount = Units.convertFromApi(
                    respondedAllocatedAmount, resource, unitsEnsemble, respondedAllocatedAmountUnit
            );
            if (newAllocatedAmount.isEmpty()) {
                LOG.error("Can't convert to base unit allocation of " + resource + responseProvision);
            }

            // заполняем результат
            AccountsQuotasModel.Builder builder = currentAccountsQuotasModel.copyBuilder()
                    .setProvidedQuota(newProvidedAmount.get())
                    .setLatestSuccessfulProvisionOperationId(accountsQuotasOperationsModel.getOperationId())
                    .setLastProvisionUpdate(accountsQuotasOperationsModel.getCreateDateTime());
            newAllocatedAmount.ifPresent(builder::setAllocatedQuota);
            if (perProvisionVersionSupported) {
                responseVersion.ifPresent(builder::setLastReceivedProvisionVersion);
            }
            actualUpdatedQuotas.add(builder.build());
        }

        return actualUpdatedQuotas;
    }

    public boolean isOperationComplete(Response<AccountDto> response) {
        AccountDto accountDto = getAccountDtoFromResponse(response);

        AccountsSettingsModel accountsSettings = providerModel.getAccountsSettings();
        boolean perAccountVersionSupported = accountsSettings.isPerAccountVersionSupported();
        boolean perProvisionVersionSupported = accountsSettings.isPerProvisionVersionSupported();
        boolean perProvisionLastUpdateSupported = accountsSettings.isPerProvisionLastUpdateSupported();

        if (perAccountVersionSupported && accountDto.getAccountVersion().isPresent()) {
            if (accountDto.getAccountVersion().get().equals(accountModel.getVersion())) {
                return false;
            }
        }

        Map<String, ProvisionDto> provisionDtoByResourceKeyMap =
                accountDto.getProvisions().orElse(emptyList()).stream()
                .filter(provisionDto -> provisionDto.getResourceKey().isPresent() &&
                        provisionDto.getResourceKey().get().getResourceTypeKey().isPresent()
                        && resourceByResourceKeyResponse.containsKey(
                        new ResourceComplexKey(provisionDto.getResourceKey().get()))
                )
                .collect(
                        toMap(k -> resourceByResourceKeyResponse.get(new ResourceComplexKey(k.getResourceKey().get()))
                                        .getKey(),
                                Function.identity()));

        long minTimestamp = Long.MAX_VALUE;

        for (ProvisionLiteWithBigIntegers provisionLiteWithBigIntegers : changingRequestProvisionsBI) {
            ResourceModel resourceModel = resourceByIdMap
                    .get(provisionLiteWithBigIntegers.getResourceId().orElseThrow());
            ProvisionDto provisionDto = provisionDtoByResourceKeyMap.get(resourceModel.getKey());

            if (provisionDto == null) {
                if (!provisionLiteWithBigIntegers.getProvidedAmountBI().equals(BigInteger.ZERO)) {
                    return false;
                }
                continue;
            }

            if (perProvisionVersionSupported && accountDto.getProvisions().isPresent()) {
                AccountsQuotasModel accountsQuotasModel = oldAccountsQuotasModelByResourceIdByAccountMap
                        .getOrDefault(accountModel.getId(), Collections.emptyMap())
                        .getOrDefault(resourceModel.getId(), null);
                if (provisionDto.getQuotaVersion().isPresent() && provisionDto.getQuotaVersion().get().
                        equals(accountsQuotasModel != null ? accountsQuotasModel.getLastReceivedProvisionVersion()
                                .orElse(null) : null)) {
                    return false;
                }
            }

            if (perProvisionLastUpdateSupported && provisionDto.getLastUpdate().isPresent()) {
                LastUpdateDto lastUpdateDto = provisionDto.getLastUpdate().get();

                if (lastUpdateDto.getOperationId().isPresent()) {
                    String lastOperationId = lastUpdateDto.getOperationId().get();
                    if (accountsQuotasOperationsModel.getOperationId().equals(lastOperationId)) {
                        return true;
                    }
                }

                if (lastUpdateDto.getTimestamp().isPresent()) {
                    minTimestamp = Math.min(lastUpdateDto.getTimestamp().get(), minTimestamp);
                }
            }
        }

        if (minTimestamp != Long.MAX_VALUE && accountsQuotasOperationsModel.getCreateDateTime()
                .isAfter(Instant.ofEpochSecond(minTimestamp))) {
            return false;
        }

        for (ProvisionLiteWithBigIntegers provisionLiteWithBigIntegers : changingRequestProvisionsBI) {
            ResourceModel resourceModel = resourceByIdMap
                    .get(provisionLiteWithBigIntegers.getResourceId().orElseThrow());
            ProvisionDto provisionDto = provisionDtoByResourceKeyMap.get(resourceModel.getKey());

            if (provisionDto == null) {
                continue;
            }

            if (provisionDto.getProvidedAmount().isPresent() && provisionDto.getProvidedAmountUnitKey().isPresent()) {
                String unitsEnsembleId = resourceModel.getUnitsEnsembleId();
                Optional<Long> amountO = Units.convertFromApi(provisionDto.getProvidedAmount().get(), resourceModel,
                        ensembleModelByIdMap.get(unitsEnsembleId),
                        unitModelByUnitEnsembleUnitKeyMap.get(fromUnitEnsembleIdUnitKey(unitsEnsembleId,
                                provisionDto.getProvidedAmountUnitKey().get())));
                if (amountO.isPresent() && newProvisionsByResourceIdMap.containsKey(resourceModel.getId())
                        && amountO.get().equals(
                        newProvisionsByResourceIdMap.get(resourceModel.getId()))) {
                    return true;
                }

                if (amountO.isPresent() && amountO.get().equals(
                        oldProvisionsByResourceIdMap.getOrDefault(resourceModel.getId(), 0L))) {
                    return false;
                }
            }
        }

        return true;
    }

    private AccountDto getAccountDtoFromResponse(Response<AccountDto> response) {
        return response.match(new Response.Cases<>() {
            @Override
            public AccountDto success(AccountDto result, String requestId) {
                return result;
            }

            @Override
            public AccountDto failure(Throwable error) {
                return null;
            }

            @Override
            public AccountDto error(ProviderError error, String requestId) {
                return null;
            }
        });
    }

    public UpdateProvisionResponseDto toUpdateProvisionResponseDto(Response<AccountDto> response) {
        return response.match(new Response.Cases<>() {
            @Override
            public UpdateProvisionResponseDto success(AccountDto result, String requestId) {
                return new UpdateProvisionResponseDto(result.getProvisions().map(l -> l.stream()
                                .filter(provisionDto -> provisionDto.getResourceKey().isPresent() &&
                                        provisionDto.getResourceKey().get().getResourceTypeKey().isPresent()
                                        && resourceByResourceKeyResponse.containsKey(
                                        new ResourceComplexKey(provisionDto.getResourceKey().get()))
                                )
                                .collect(Collectors.toList()))
                        .orElse(null),
                        result.getAccountVersion().orElse(null), result.getAccountsSpaceKey().orElse(null));
            }

            @Override
            public UpdateProvisionResponseDto failure(Throwable error) {
                return null;
            }

            @Override
            public UpdateProvisionResponseDto error(ProviderError error, String requestId) {
                return null;
            }
        });
    }

    private ResourceComplexKey toResourceComplexKey(String resourceId) {
        ResourceModel resourceModel = resourceByIdMap.get(resourceId);
        return new ResourceComplexKey(
                resourceTypeByIdMap.get(resourceModel.getResourceTypeId()),
                toSegmentKeyBySegmentationKeyMap(resourceModel.getSegments())
        );
    }

    private Map<String, String> toSegmentKeyBySegmentationKeyMap(Set<ResourceSegmentSettingsModel> segments) {
        if (segments == null) {
            return Map.of();
        }
        var stream = segments.stream();
        if (accountSpaceModel != null) {
            var accountSpaceSegments = accountSpaceModel.getSegments();
            stream = stream.filter(segment -> !accountSpaceSegments.contains(segment));
        }
        return stream.collect(toMap(
                segment -> resourceSegmentationsById.get(segment.getSegmentationId()).getKey(),
                segment -> resourceSegmentsById.get(segment.getSegmentId()).getKey())
        );
    }

    private List<SegmentKeyResponseDto> toSegmentKeyResponseDto(List<SegmentKeyRequestDto> segmentKeyRequestDtoList) {
        return segmentKeyRequestDtoList.stream()
                .map(segmentKeyRequestDto -> new SegmentKeyResponseDto(segmentKeyRequestDto.getSegmentationKey(),
                        segmentKeyRequestDto.getSegmentKey()))
                .collect(Collectors.toList());

    }

    public QuotasProvisionAnswerBuilder setCurrentAccount(AccountModel account) {
        this.currentAccount = account;
        return this;
    }

    public AccountsQuotasOperationsModel toAccountsQuotasOperationsSuccess() {
        this.accountsQuotasOperationsModelSuccess =
                new AccountsQuotasOperationsModel.Builder(accountsQuotasOperationsModel)
                        .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                        .setUpdateDateTime(Instant.now())
                        .setOrders(accountsQuotasOperationsModel.getOrders().copyBuilder()
                                .closeOrder(actualFolderModel.getNextOpLogOrder()).build())
                        .setErrorKind(null)
                        .build();
        return accountsQuotasOperationsModelSuccess;
    }

    public AccountsQuotasOperationsModel toAccountsQuotasOperationsFailure(OperationErrorCollections errorMessage,
                                                                           boolean conflict) {
        return new AccountsQuotasOperationsModel.Builder(accountsQuotasOperationsModel)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                .setFullErrorMessage(errorMessage)
                .setErrorKind(conflict ? OperationErrorKind.FAILED_PRECONDITION : OperationErrorKind.INVALID_ARGUMENT)
                .build();
    }

    public FolderOperationLogModel toFolderOperationLogSuccess() {
        Map<String, Long> newBalanceByResource = new HashMap<>(folderOperationLogModel.getNewBalance().asMap());

        unfreezeQuotaModels.forEach(q -> newBalanceByResource.put(q.getResourceId(), q.getBalance()));

        this.folderOperationLogModelSuccess = this.folderOperationLogModel.copyBuilder()
                .setId(UUID.randomUUID().toString())
                .setNewBalance(new QuotasByResource(newBalanceByResource))
                .setOperationDateTime(accountsQuotasOperationsModelSuccess.getUpdateDateTime().orElseThrow())
                .setOrder(accountsQuotasOperationsModelSuccess.getOrders().getCloseOrder().orElseThrow())
                .setActuallyAppliedProvisions(toActuallyAppliedProvisions())
                .setOperationPhase(OperationPhase.CLOSE)
                .build();
        return folderOperationLogModelSuccess;
    }

    private QuotasByAccount toActuallyAppliedProvisions() {
        if (updateProvisionResponseDto.getProvisions().isEmpty()) {
            return null;
        }

        List<ProvisionDto> provisionDtoList = updateProvisionResponseDto.getProvisions().get();

        Map<String, ProvisionHistoryModel> provisionHistoryModelByResourceIdMap = provisionDtoList.stream()
                .filter(provisionDto -> provisionDto.getResourceKey().isPresent()
                        && provisionDto.getResourceKey().get().getResourceTypeKey().isPresent()
                        && provisionDto.getProvidedAmount().isPresent()
                        && provisionDto.getProvidedAmountUnitKey().isPresent()
                        && resourceByResourceKeyResponse.containsKey(
                        new ResourceComplexKey(provisionDto.getResourceKey().get()))
                        && newProvisionsByResourceIdMap.containsKey(
                        resourceByResourceKeyResponse.get(
                                new ResourceComplexKey(provisionDto.getResourceKey().get())
                        ).getId()))
                .collect(toMap(provisionDto -> resourceByResourceKeyResponse
                                .get(new ResourceComplexKey(provisionDto.getResourceKey().get())).getId(),
                        provisionDto -> {
                            ResourceModel resource = resourceByResourceKeyResponse
                                    .get(new ResourceComplexKey(provisionDto.getResourceKey().get()));
                            String unitsEnsembleId = resource.getUnitsEnsembleId();
                            Optional<Long> providedQuotaO = Units.convertFromApi(provisionDto.getProvidedAmount().get(),
                                    resource, ensembleModelByIdMap.get(unitsEnsembleId),
                                    unitModelByUnitEnsembleUnitKeyMap.get(fromUnitEnsembleIdUnitKey(unitsEnsembleId,
                                            provisionDto.getProvidedAmountUnitKey().get())));

                            return Tuples.of(providedQuotaO, provisionDto.getQuotaVersion());
                        })).entrySet().stream()
                .filter(stringTuple2Entry -> stringTuple2Entry.getValue().getT1().isPresent())
                .collect(toMap(Map.Entry::getKey, e -> new ProvisionHistoryModel(e.getValue().getT1().get(),
                        e.getValue().getT2().orElse(null))));

        return new QuotasByAccount(Collections.singletonMap(accountModel.getId(),
                new ProvisionsByResource(provisionHistoryModelByResourceIdMap)));
    }

    public GetAccountRequestDto prepareGetAccountRequestDto() {
        return new GetAccountRequestDto(true, false, folderModel.getId(),
                folderModel.getServiceId(), toAccountsSpaceKey());
    }

    public boolean canRetryUpdateProvision() {
        return updateProvisionRetryCount.decrementAndGet() >= 0;
    }

    public boolean canRetryGetAccount() {
        return getAccountRetryCount.decrementAndGet() >= 0;
    }

    public boolean isProvidedLessThanAllocated(Response<AccountDto> response) {
        AccountDto accountDto = getAccountDtoFromResponse(response);

        String accountId = updateProvisionsRequestDto.getAccountId().orElseThrow();
        String responseAccountId = accountDto.getAccountId().orElse(null);

        Map<ResourceComplexKey, ProvisionDto> responseProvisionsMap = Objects.equals(accountId, responseAccountId)
                ? accountDto.getProvisions().orElse(List.of())
                .stream()
                .filter(r -> r.getResourceKey().isPresent())
                .collect(toMap(k -> new ResourceComplexKey(k.getResourceKey().orElseThrow()), Function.identity()))
                : Map.of();

        return isProvidedLessThanAllocated(responseProvisionsMap);
    }

    /**
     * Calculate is 'provided > allocated', where 'provided' uses from user update provision request
     * and 'allocated' from our db, or from provider get-account-response from @param responseProvisionsMap.
     * @param responseProvisionsMap with allocated values from provider get-account-response
     * @return true if 'provided > allocated' in any resource from request, false otherwise.
     */
    private boolean isProvidedLessThanAllocated(Map<ResourceComplexKey, ProvisionDto> responseProvisionsMap) {
        boolean providerSupportsAllocation = providerModel.isAllocatedSupported().orElse(true);
        Set<String> updatedResourceIds = changingRequestProvisionsBI.stream()
                .map(r -> r.getResourceId().orElseThrow())
                .collect(Collectors.toSet());
        return updateProvisionRequestDto.getUpdatedProvisions().stream().anyMatch(provisionRequest -> {
            if (provisionRequest.getResourceKey().getResourceTypeKey().isEmpty() ||
                    provisionRequest.getResourceKey().getSegmentation().isEmpty()) {
                return false;
            }
            Map<String, String> segmentKeyBySegmentationKey = provisionRequest.getResourceKey().getSegmentation().get()
                    .stream()
                    .collect(toMap(SegmentKeyRequestDto::getSegmentationKey, SegmentKeyRequestDto::getSegmentKey));
            ResourceComplexKey resourceComplexKey = new ResourceComplexKey(
                    provisionRequest.getResourceKey().getResourceTypeKey().get(), segmentKeyBySegmentationKey);
            String accountId = updateProvisionsRequestDto.getAccountId().orElseThrow();
            String resourceId = resourceByResourceKeyResponse.get(resourceComplexKey).getId();
            if (!oldAccountsQuotasModelByResourceIdByAccountMap.containsKey(accountId) ||
                    !oldAccountsQuotasModelByResourceIdByAccountMap.get(accountId).containsKey(resourceId)) {
                return false;
            }
            Optional<ResourceModel> resourceModelO = Optional.ofNullable(allResourceByIdMap.get(resourceId));
            boolean resourceAllocationSupported = resourceModelO
                    .flatMap(ResourceModel::isAllocatedSupported).orElse(providerSupportsAllocation);
            if (!resourceAllocationSupported) {
                return false;
            }
            Long allocated;
            ProvisionDto provisionDto;
            if ((provisionDto = responseProvisionsMap.get(resourceComplexKey)) != null
                    && provisionDto.getAllocatedAmount().isPresent()
                    && provisionDto.getAllocatedAmountUnitKey().isPresent()
                    && resourceModelO.isPresent()) {
                ResourceModel resourceModel = resourceModelO.get();
                String unitsEnsembleId = resourceModel.getUnitsEnsembleId();
                Optional<Long> amountO = Units.convertFromApi(provisionDto.getAllocatedAmount().get(),
                        resourceModel, ensembleModelByIdMap.get(unitsEnsembleId),
                        unitModelByUnitEnsembleUnitKeyMap.get(fromUnitEnsembleIdUnitKey(unitsEnsembleId,
                                provisionDto.getAllocatedAmountUnitKey().get())));
                allocated = amountO.orElse(oldAccountsQuotasModelByResourceIdByAccountMap.get(accountId).get(resourceId)
                        .getAllocatedQuota());
            } else {
                allocated = oldAccountsQuotasModelByResourceIdByAccountMap.get(accountId).get(resourceId)
                        .getAllocatedQuota();
            }
            long newProvided;
            if (updatedResourceIds.contains(resourceId)) {
                newProvided = newProvisionsByResourceIdMap.get(resourceId);
            } else {
                newProvided = oldAccountsQuotasModelByResourceIdByAccountMap
                        .getOrDefault(accountModel.getId(), Map.of())
                        .get(resourceId)
                        .getProvidedQuota();
            }
            return newProvided < allocated;
        });
    }

    public boolean isProvidedLessThanAllocated() {
        return isProvidedLessThanAllocated(Map.of());
    }

    static final class UnitEnsembleUnitKey {
        private final String unitsEnsembleId;
        private final String unitKey;

        UnitEnsembleUnitKey(String unitsEnsembleId, String unitKey) {
            this.unitsEnsembleId = unitsEnsembleId;
            this.unitKey = unitKey;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            UnitEnsembleUnitKey that = (UnitEnsembleUnitKey) o;
            return Objects.equals(unitsEnsembleId, that.unitsEnsembleId) &&
                    Objects.equals(unitKey, that.unitKey);
        }

        @Override
        public int hashCode() {
            return Objects.hash(unitsEnsembleId, unitKey);
        }

        @Override
        public String toString() {
            return "UnitEnsembleUnitKey{" +
                    "unitsEnsembleId='" + unitsEnsembleId + '\'' +
                    ", unitKey='" + unitKey + '\'' +
                    '}';
        }

        static UnitEnsembleUnitKey fromUnitEnsembleIdUnitKey(String unitEnsembleId, String unitKey) {
            return new UnitEnsembleUnitKey(unitEnsembleId, unitKey);
        }
    }
}
