package ru.yandex.intranet.d.services.quotas;

import java.math.BigDecimal;
import java.math.MathContext;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.collect.Maps;
import com.google.common.collect.Streams;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.DecimalWithUnit;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.elements.FolderServiceElement;
import ru.yandex.intranet.d.services.elements.StringContinuationTokenPager;
import ru.yandex.intranet.d.services.resources.AccountsSpacesUtils;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.util.FrontStringUtil;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.paging.StringContinuationToken;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.ResultTx;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.AmountDto;
import ru.yandex.intranet.d.web.model.FrontProvisionsDto;
import ru.yandex.intranet.d.web.model.folders.FrontFolderWithQuotesDto;
import ru.yandex.intranet.d.web.model.folders.front.FolderPermission;
import ru.yandex.intranet.d.web.model.folders.front.ProviderPermission;
import ru.yandex.intranet.d.web.model.quotas.UpdateProvisionDryRunAnswerDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static com.yandex.ydb.table.transaction.TransactionMode.SERIALIZABLE_READ_WRITE;
import static ru.yandex.intranet.d.services.quotas.QuotasHelper.getMinAllowedUnit;
import static ru.yandex.intranet.d.util.units.Units.convert;

/**
 * Quotas service.
 *
 * @author Nikita Minin <spasitel@yandex-team.ru>
 */
@Component
public class QuotasService {

    private final MessageSource messages;
    private final QuotasDao quotasDao;
    private final FolderServiceElement folderServiceElement;
    private final ResourcesLoader resourcesLoader;
    private final ProvidersLoader providersLoader;
    private final ResourceTypesLoader resourceTypesLoader;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final FolderDao folderDao;
    private final ObjectReader continuationTokenReader;
    private final ObjectWriter continuationTokenWriter;
    private final SecurityManagerService securityManagerService;
    private final AccountsDao accountsDao;
    private final AccountsSpacesUtils accountsSpacesUtils;
    private final AccountsQuotasDao accountsQuotasDao;
    private final YdbTableClient tableClient;
    private final StringContinuationTokenPager pager;
    private final ResourceSegmentationsLoader segmentationsLoader;
    private final ResourceSegmentsLoader segmentsLoader;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public QuotasService(
            @Qualifier("messageSource") MessageSource messages,
            QuotasDao quotasDao,
            FolderServiceElement folderServiceElement,
            YdbTableClient tableClient,
            ResourcesLoader resourcesLoader,
            ProvidersLoader providersLoader,
            ResourceTypesLoader resourceTypesLoader,
            UnitsEnsemblesLoader unitsEnsemblesLoader,
            FolderDao folderDao,
            SecurityManagerService securityManagerService,
            @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
            AccountsDao accountsDao,
            AccountsSpacesUtils accountsSpacesUtils,
            AccountsQuotasDao accountsQuotasDao,
            StringContinuationTokenPager pager,
            ResourceSegmentationsLoader segmentationsLoader,
            ResourceSegmentsLoader segmentsLoader
    ) {
        this.messages = messages;
        this.quotasDao = quotasDao;
        this.folderServiceElement = folderServiceElement;
        this.tableClient = tableClient;
        this.resourcesLoader = resourcesLoader;
        this.providersLoader = providersLoader;
        this.resourceTypesLoader = resourceTypesLoader;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.folderDao = folderDao;
        this.securityManagerService = securityManagerService;
        this.continuationTokenReader = objectMapper.getObjectMapper().readerFor(QuotaContinuationToken.class);
        this.continuationTokenWriter = objectMapper.getObjectMapper().writerFor(QuotaContinuationToken.class);
        this.accountsDao = accountsDao;
        this.accountsSpacesUtils = accountsSpacesUtils;
        this.accountsQuotasDao = accountsQuotasDao;
        this.pager = pager;
        this.segmentationsLoader = segmentationsLoader;
        this.segmentsLoader = segmentsLoader;
    }

    /**
     * Get quotas in folders.
     *
     * @param includeCompletelyZero - включать ли в результат полностью нулевые квоты, т.е. такие,
     *                              что их quota, balance и frozen_quota ноль.
     */
    public Mono<Result<FrontFolderWithQuotesDto>> getQuotasInFolders(
            Set<String> folderIds,
            boolean includeCompletelyZero,
            YaUserDetails currentUser,
            Locale locale
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return GetQuotasInFoldersContext.start().flatMap(r -> r.andThenMono(c ->
                securityManagerService.checkReadPermissions(currentUser, locale)
                        .map(result -> result.apply(u -> c)))).flatMap(r -> r.andThenMono(c ->
                tableClient.usingSessionMonoRetryable(s -> s.usingCompTxRetryable(
                        (ts) -> folderDao.getByIdsStartTx(ts, List.copyOf(folderIds), tenantId)
                                .map(withTxId -> withTxId.map(c::withFolders))
                                .map(withTxId -> withTxId.map(con -> validateAllFoldersFound(folderIds, con, locale))),
                        (ts, rco) -> rco.applyMono(co -> Mono.just(co)
                                .flatMap(con -> quotasDao.getByFolders(ts,
                                        con.getFoldersIds(),
                                        tenantId,
                                        includeCompletelyZero)
                                        .map(con::withQuotas))
                                .flatMap(con -> resourcesLoader.getResourcesByIds(ts, con.getResourcesIds())
                                        .map(con::withResources))
                                .flatMap(con -> resourceTypesLoader.getResourceTypesByIds(ts, con.getResourceTypesIds())
                                        .map(con::withResourceTypes))
                                .flatMap(con -> providersLoader.getProvidersByIds(ts, con.getProvidersIds())
                                        .map(con::withProviders))
                                .flatMap(con -> addAccounts(ts, null,  con.getFolderIdsWithTenants(), con))
                                .flatMap(con -> accountsQuotasDao.getAllByAccountIds(ts, tenantId, con.getAccountIds())
                                        .map(con::withAccountsQuotas))
                                .flatMap(con -> getFolderPermissions(ts, con.getFolders(), con.getProviders(),
                                        currentUser).map(con::withFolderPermissions))
                                .flatMap(con -> segmentationsLoader
                                        .getResourceSegmentationsByIds(ts, con.getSegmentationsIds())
                                        .map(con::withSegmentations))
                                .flatMap(con -> segmentsLoader
                                        .getResourceSegmentsByIds(ts, con.getSegmentsIds())
                                        .map(con::withSegments))
                        ),
                        (ts, rco) -> rco.applyMono(co -> Mono.just(co)
                                .flatMap(con -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(
                                        ts, con.getUnitsEnsemblesIds()
                                ).map(con::withUnitsEnsembles))
                        )
                ))))
                .map(r -> r.apply(c -> new FrontFolderWithQuotesDtoBuilder(locale, accountsSpacesUtils)
                        .build(c.get())));
    }

    public Mono<Result<FrontFolderWithQuotesDto>> getProviderReservedQuotas(
            String providerId,
            String reservedFolderId,
            YaUserDetails currentUser,
            Locale locale
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return GetQuotasInFoldersContext.start().flatMap(r -> r.andThenMono(c ->
                securityManagerService.checkReadPermissions(currentUser, locale)
                        .map(result -> result.apply(u -> c)))).flatMap(r -> r.andThenMono(c ->
                tableClient.usingSessionMonoRetryable(s -> s.usingCompTxRetryable(
                        (ts) -> folderDao.getByIdsStartTx(ts, List.of(reservedFolderId), tenantId)
                                .map(withTxId -> withTxId.map(c::withFolders))
                                .map(withTxId -> withTxId.map(con ->
                                        validateAllFoldersFound(Set.of(reservedFolderId), con, locale))),
                        (ts, rco) -> rco.applyMono(co -> Mono.just(co)
                                .flatMap(con ->
                                        quotasDao.getByFoldersAndProvider(ts, con.getFoldersIds(), tenantId, providerId)
                                        .map(con::withQuotas))
                                .flatMap(con -> resourcesLoader.getResourcesByIds(ts, con.getResourcesIds())
                                        .map(con::withResources))
                                .map(GetQuotasInFoldersContext.WithResources::filterUnmanagedResources)
                                .flatMap(con -> resourceTypesLoader.getResourceTypesByIds(ts, con.getResourceTypesIds())
                                        .map(con::withResourceTypes))
                                .flatMap(con -> providersLoader.getProvidersByIds(ts, con.getProvidersIds())
                                        .map(con::withProviders))
                                .flatMap(con -> addAccounts(ts, null,  con.getFolderIdsWithTenants(), con))
                                .flatMap(con -> accountsQuotasDao.getAllByAccountIds(ts, tenantId, con.getAccountIds())
                                        .map(con::withAccountsQuotas))
                                .flatMap(con -> getFolderPermissions(ts, con.getFolders(), con.getProviders(),
                                        currentUser).map(con::withFolderPermissions))
                                .flatMap(con -> segmentationsLoader
                                        .getResourceSegmentationsByIds(ts, con.getSegmentationsIds())
                                        .map(con::withSegmentations))
                                .flatMap(con -> segmentsLoader
                                        .getResourceSegmentsByIds(ts, con.getSegmentsIds())
                                        .map(con::withSegments))
                        ),
                        (ts, rco) -> rco.applyMono(co -> Mono.just(co)
                                .flatMap(con -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(
                                        ts, con.getUnitsEnsemblesIds()
                                ).map(con::withUnitsEnsembles))
                        )
                ))))
                .map(r -> r.apply(c -> new FrontFolderWithQuotesDtoBuilder(locale, accountsSpacesUtils)
                        .build(c.get())));
    }

    public Mono<Result<FrontProvisionsDto>> getAllBalances(
            String folderId,
            String accountId,
            YaUserDetails currentUser,
            Locale locale
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale)
                .flatMap(r -> r.andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
                session.usingTxMonoRetryable(SERIALIZABLE_READ_WRITE, ts ->
        getAccount(ts, accountId, tenantId, locale)
                .flatMap(accountModelResult -> accountModelResult.applyMono(accountModel ->
        quotasDao.getByFoldersAndProvider(ts, List.of(folderId), tenantId, accountModel.getProviderId(), false)
                .flatMap(quotaModels ->
        accountsQuotasDao.getAllByAccountIds(ts, tenantId, Set.of(accountId))
                .flatMap(accountsQuotasModels ->
        getResourcesById(ts, quotaModels, accountsQuotasModels)
                .flatMap(resourceModels ->
        getUnitsEnsembles(ts, resourceModels)
                .map(unitsEnsembleModels ->
        toGetAllBalancesResponseDto(
                accountModel, quotaModels, accountsQuotasModels, resourceModels, unitsEnsembleModels, locale
        )
        ))))))))));
    }

    public Mono<Result<FrontProvisionsDto>> getAllNonAllocated(
            String folderId,
            String accountId,
            YaUserDetails currentUser,
            Locale locale
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale)
        .flatMap(r -> r
        .andThen(v -> validateFolderId(folderId, locale))
        .andThen(a -> validateAccountId(accountId, locale))
        .andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
            session.usingTxMonoRetryable(SERIALIZABLE_READ_WRITE, ts ->
        getAccount(ts, accountId, tenantId, locale)
            .flatMap(accountModelResult -> accountModelResult.applyMono(accountModel ->
        quotasDao.getByFoldersAndProvider(ts, List.of(folderId), tenantId, accountModel.getProviderId(), false)
            .flatMap(quotaModels ->
        accountsQuotasDao.getAllByAccountIds(ts, tenantId, Set.of(accountId))
                .flatMap(accountsQuotasModels ->
        getResourcesById(ts, quotaModels, accountsQuotasModels)
                .flatMap(resourceModels ->
        getUnitsEnsembles(ts, resourceModels)
                .map(unitsEnsembleModels ->
        toGetAllNonAllocatedResponseDto(
                accountModel, quotaModels, accountsQuotasModels, resourceModels, unitsEnsembleModels, locale
        )
        ))))))))));
    }

    private FrontProvisionsDto toGetAllNonAllocatedResponseDto(
            AccountModel accountModel,
            List<QuotaModel> quotaModels,
            List<AccountsQuotasModel> accountsQuotasModels,
            List<ResourceModel> resourceModels,
            List<UnitsEnsembleModel> unitsEnsembleModels,
            Locale locale
    ) {
        Map<String, AccountsQuotasModel> accountsQuotasByResourceId = accountsQuotasModels.stream().collect(
                Collectors.toMap(AccountsQuotasModel::getResourceId, Function.identity())
        );
        Map<String, ResourceModel> resourcesById = resourceModels.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, UnitsEnsembleModel> ensemblesById = unitsEnsembleModels.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        Map<String, UpdateProvisionDryRunAnswerDto> provisionsByResourceId = new HashMap<>();
        for (QuotaModel quotaModel : quotaModels) {
            ResourceModel resource = resourcesById.get(quotaModel.getResourceId());
            if (resourceQuotaCanNotBeChanged(resource, accountModel)) {
                continue;
            }
            AccountsQuotasModel accountsQuotas = accountsQuotasByResourceId.get(quotaModel.getResourceId());
            if (accountsQuotas != null && accountsQuotas.getProvidedQuota() > accountsQuotas.getAllocatedQuota()) {
                long deltaPrimitive = accountsQuotas.getAllocatedQuota() - accountsQuotas.getProvidedQuota();
                BigDecimal balance = BigDecimal.valueOf(quotaModel.getBalance() - deltaPrimitive);
                BigDecimal provided = BigDecimal.valueOf(accountsQuotas.getProvidedQuota() + deltaPrimitive);
                BigDecimal delta = BigDecimal.valueOf(deltaPrimitive);
                BigDecimal allocated = BigDecimal.valueOf(accountsQuotas.getAllocatedQuota());

                var dryRunAnswerDto = getDryRunAnswerDto(
                        resource, ensemblesById, quotaModel, balance, provided, delta, allocated, locale
                );
                provisionsByResourceId.put(quotaModel.getResourceId(), dryRunAnswerDto);
            }
        }
        return new FrontProvisionsDto(provisionsByResourceId);
    }

    private Mono<Result<AccountModel>> getAccount(YdbTxSession ts, String accountId, TenantId tenantId, Locale locale) {
        return accountsDao.getById(ts, accountId, tenantId)
                .map(accountModelOptional -> accountModelOptional.map(Result::success)
                        .orElseGet(() -> Result.failure(ErrorCollection.builder()
                        .addError("folderIds", TypedError.notFound(messages
                                .getMessage("errors.account.not.found", null, locale)))
                        .addDetail("unknownAccountId", accountId)
                        .build())
                ));
    }

    private Mono<List<ResourceModel>> getResourcesById(
            YdbTxSession ts,
            List<QuotaModel> quotaModels,
            List<AccountsQuotasModel> accountsQuotasModels
    ) {
        List<Tuple2<String, TenantId>> ids = Stream.concat(
                quotaModels.stream().map(quota ->
                        toIdTuple(quota, QuotaModel::getResourceId, QuotaModel::getTenantId)),
                accountsQuotasModels.stream().map(quota ->
                        toIdTuple(quota, AccountsQuotasModel::getResourceId, AccountsQuotasModel::getTenantId))
        ).distinct().collect(Collectors.toList());
        return resourcesLoader.getResourcesByIds(ts, ids);
    }

    private Mono<List<UnitsEnsembleModel>> getUnitsEnsembles(
            YdbTxSession ts,
            List<ResourceModel> resources
    ) {
        List<Tuple2<String, TenantId>> ids = resources.stream()
                .map(resource -> toIdTuple(resource, ResourceModel::getUnitsEnsembleId, ResourceModel::getTenantId))
                .distinct().collect(Collectors.toList());
        return unitsEnsemblesLoader.getUnitsEnsemblesByIds(ts, ids);
    }
    private <T> Tuple2<String, TenantId> toIdTuple(
            T data,
            Function<T, String> idGetter,
            Function<T, TenantId> tenantIdGetter
    ) {
        return Tuples.of(idGetter.apply(data), tenantIdGetter.apply(data));
    }

    private FrontProvisionsDto toGetAllBalancesResponseDto(
            AccountModel accountModel,
            List<QuotaModel> quotaModels,
            List<AccountsQuotasModel> accountsQuotasModels,
            List<ResourceModel> resourceModels,
            List<UnitsEnsembleModel> unitsEnsembleModels,
            Locale locale
    ) {
        Map<String, AccountsQuotasModel> accountsQuotasByResourceId = accountsQuotasModels.stream().collect(
                Collectors.toMap(AccountsQuotasModel::getResourceId, Function.identity())
        );
        Map<String, ResourceModel> resourcesById = resourceModels.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, UnitsEnsembleModel> ensemblesById = unitsEnsembleModels.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        Map<String, UpdateProvisionDryRunAnswerDto> provisionsByResourceId = new HashMap<>();
        for (QuotaModel quotaModel : quotaModels) {
            ResourceModel resource = resourcesById.get(quotaModel.getResourceId());
            if (resourceQuotaCanNotBeChanged(resource, accountModel)) {
                continue;
            }
            AccountsQuotasModel accountsQuotas = accountsQuotasByResourceId.get(quotaModel.getResourceId());
            long deltaPrimitive = quotaModel.getBalance() > 0 ? quotaModel.getBalance() : 0;
            long oldProvided = accountsQuotas != null ? accountsQuotas.getProvidedQuota() : 0;
            if (quotaModel.getBalance() > 0 || accountsQuotas != null) {
                BigDecimal balance = BigDecimal.ZERO;
                BigDecimal provided = BigDecimal.valueOf(oldProvided + deltaPrimitive);
                BigDecimal delta = BigDecimal.valueOf(deltaPrimitive);
                BigDecimal allocated = accountsQuotas != null ?
                        BigDecimal.valueOf(accountsQuotas.getAllocatedQuota()) : BigDecimal.ZERO;

                var dryRunAnswer = getDryRunAnswerDto(
                        resource, ensemblesById, quotaModel, balance, provided, delta, allocated, locale
                );
                provisionsByResourceId.put(quotaModel.getResourceId(), dryRunAnswer);
            }
        }
        return new FrontProvisionsDto(provisionsByResourceId);
    }

    private boolean resourceQuotaCanNotBeChanged(ResourceModel resource, AccountModel accountModel) {
        return accountModel.getAccountsSpacesId().map(accountsSpacesId ->
                        !accountsSpacesId.equals(resource.getAccountsSpacesId()))
                .orElse(false) ||
                resource.isReadOnly() || !resource.isManaged();
    }

    @SuppressWarnings("ParameterNumber")
    private UpdateProvisionDryRunAnswerDto getDryRunAnswerDto(
            ResourceModel resource,
            Map<String, UnitsEnsembleModel> ensemblesById,
            QuotaModel quotaModel,
            BigDecimal balance,
            BigDecimal provided,
            BigDecimal delta,
            BigDecimal allocated,
            Locale locale
    ) {
        Set<String> allowedUnitIds = resource.getResourceUnits().getAllowedUnitIds();
        UnitsEnsembleModel unitsEnsemble = ensemblesById.get(resource.getUnitsEnsembleId());
        List<UnitModel> sortedUnits = unitsEnsemble.getUnits().stream()
                .filter(unit -> allowedUnitIds.contains(unit.getId()))
                .sorted(UnitsComparator.INSTANCE).collect(Collectors.toList());
        UnitModel baseUnit = unitsEnsemble.unitById(resource.getBaseUnitId()).orElseThrow();
        UnitModel defaultUnit = unitsEnsemble.unitById(resource.getResourceUnits().getDefaultUnitId())
                .orElseThrow(() -> new IllegalStateException("Default unit not found"));
        UnitModel minAllowedUnit = getMinAllowedUnit(allowedUnitIds, sortedUnits).orElse(baseUnit);

        DecimalWithUnit newProvidedWithUnit = QuotasHelper.convertToReadable(
                provided, sortedUnits, baseUnit
        );
        UnitModel newFormFieldsUnit = newProvidedWithUnit.getUnit();

        BigDecimal quota = BigDecimal.valueOf(quotaModel.getQuota());
        BigDecimal providedRatio = quota.signum() != 0 ?
                provided.divide(quota, MathContext.DECIMAL64) : BigDecimal.ZERO;
        BigDecimal allocatedRatio = quota.signum() != 0 ?
                allocated.divide(quota, MathContext.DECIMAL64) : BigDecimal.ZERO;

        AmountDto balanceAmount = getAmountDto(balance, sortedUnits, baseUnit,
                newFormFieldsUnit, defaultUnit, minAllowedUnit, locale);
        AmountDto providedAmount = getAmountDto(provided, sortedUnits, baseUnit,
                newFormFieldsUnit, defaultUnit, minAllowedUnit, locale);
        AmountDto deltaAmount = getAmountDto(delta, sortedUnits, baseUnit,
                newFormFieldsUnit, defaultUnit, minAllowedUnit, locale);

        return new UpdateProvisionDryRunAnswerDto.Builder()
                .setBalance(FrontStringUtil.toString(convert(balance, baseUnit, newFormFieldsUnit)))
                .setProvidedAbsolute(
                        FrontStringUtil.toString(convert(provided, baseUnit, newFormFieldsUnit)))
                .setProvidedDelta(FrontStringUtil.toString(convert(delta, baseUnit, newFormFieldsUnit)))
                .setProvidedRatio(FrontStringUtil.toString(providedRatio))
                .setAllocated(FrontStringUtil.toString(convert(allocated, baseUnit, newFormFieldsUnit)))
                .setAllocatedRatio(FrontStringUtil.toString(allocatedRatio))
                .setForEditUnitId(newFormFieldsUnit.getId())
                .setProvidedAbsoluteInMinAllowedUnit(
                        FrontStringUtil.toString(convert(provided, baseUnit, minAllowedUnit)))
                .setMinAllowedUnitId(minAllowedUnit.getId())
                .setBalanceAmount(balanceAmount)
                .setProvidedAmount(providedAmount)
                .setDeltaAmount(deltaAmount)
                .build();
    }

    private AmountDto getAmountDto(BigDecimal amountInBaseUnit, List<UnitModel> sortedUnits, UnitModel baseUnit,
                                   UnitModel formFieldsUnit, UnitModel defaultUnit, UnitModel minAllowedUnit,
                                   Locale locale) {
        if (amountInBaseUnit.compareTo(BigDecimal.ZERO) == 0) {
            return QuotasHelper.zeroAmount(defaultUnit, formFieldsUnit, minAllowedUnit, locale);
        }
        return QuotasHelper.getAmountDto(amountInBaseUnit, sortedUnits, baseUnit, formFieldsUnit,
                defaultUnit, minAllowedUnit, locale);
    }

    private Mono<FoldersPermissionsCollection> getFolderPermissions(YdbTxSession ts, List<FolderModel> folders,
                                                                    List<ProviderModel> providers,
                                                                    YaUserDetails user) {
        Set<Long> serviceIds = folders.stream()
                .map(FolderModel::getServiceId)
                .collect(Collectors.toSet());

        return securityManagerService.hasUserWritePermissionsByServiceId(ts, serviceIds, user)
                .flatMap(perms -> securityManagerService.filterProvidersWithProviderAdminRole(providers, user)
                .map(allowedProviders -> {
                    Map<Long, Set<FolderPermission>> permissionsByService = Maps.transformValues(perms, b -> b ?
                            Set.of(FolderPermission.CAN_MANAGE_ACCOUNT, FolderPermission.CAN_UPDATE_PROVISION) :
                            Set.of());
                    Map<String, Set<FolderPermission>> permissionsByFolder = folders.stream()
                            .collect(Collectors.toMap(FolderModel::getId,
                                    f -> permissionsByService.get(f.getServiceId())));
                    Set<String> allowedProviderIds = allowedProviders.stream().map(ProviderModel::getId)
                            .collect(Collectors.toSet());
                    Map<String, Map<String, Set<ProviderPermission>>> permissionsByFolderProvider = new HashMap<>();
                    folders.forEach(folder -> {
                        Map<String, Set<ProviderPermission>> permissionsByProvider = permissionsByFolderProvider
                                .computeIfAbsent(folder.getId(), k -> new HashMap<>());
                        boolean folderAllowed = perms.getOrDefault(folder.getServiceId(), false);
                        if (folderAllowed) {
                            providers.forEach(provider -> permissionsByProvider.put(provider.getId(),
                                    Set.of(ProviderPermission.CAN_MANAGE_ACCOUNT,
                                            ProviderPermission.CAN_UPDATE_PROVISION)));
                        } else {
                            providers.forEach(provider -> {
                                if (allowedProviderIds.contains(provider.getId())) {
                                    permissionsByProvider.put(provider.getId(),
                                            Set.of(ProviderPermission.CAN_MANAGE_ACCOUNT,
                                                    ProviderPermission.CAN_UPDATE_PROVISION));
                                } else {
                                    permissionsByProvider.put(provider.getId(), Set.of());
                                }
                            });
                        }
                    });
                    return new FoldersPermissionsCollection(permissionsByFolder, permissionsByFolderProvider);
                }));
    }

    private Result<GetQuotasInFoldersContext.WithFolders> validateAllFoldersFound(
            Set<String> folderIds,
            GetQuotasInFoldersContext.WithFolders con,
            Locale locale
    ) {
        Set<String> foundedFolderIds = con.getFolders().stream().map(FolderModel::getId).collect(Collectors.toSet());
        List<String> notFoundedFolderIds = folderIds.stream()
                .filter(folderId -> !foundedFolderIds.contains(folderId))
                .collect(Collectors.toList());
        if (notFoundedFolderIds.isEmpty()) {
            return Result.success(con);
        }
        return Result.failure(ErrorCollection.builder()
                .addError("folderIds", TypedError.notFound(messages
                        .getMessage("errors.folder.not.found", null, locale)))
                .addDetail("unknownFolderIds", notFoundedFolderIds)
                .build());
    }

    public Mono<Result<FrontFolderWithQuotesDto>> getFrontPageForService(
            PageRequest pageRequest,
            long serviceId,
            @Nullable String providerId,
            @Nullable String resourceId,
            boolean includeCompletelyZero,
            YaUserDetails currentUser,
            Locale locale
    ) {
        GetFrontForServiceContext.Impl con = new GetFrontForServiceContext.Impl(Tenants.getTenantId(currentUser));
        return securityManagerService.checkReadPermissions(currentUser, locale)
                //Если заданы, проверяем провайдера и ресурс по кешу. Фолдер и сервис не проверяем
                .then(checkProviderAndResource(
                        locale, providerId, resourceId, con))
                .flatMap(u -> u.andThenMono(context -> tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompResultTxRetryable(SERIALIZABLE_READ_WRITE,
                                //квоты и фолдеры в первом запросе, остальное во втором - без result
                                ts -> getQuotasPageAndFoldersForService(context,
                                        pageRequest,
                                        serviceId,
                                        providerId,
                                        resourceId,
                                        includeCompletelyZero,
                                        locale,
                                        ts
                                ),
                                (ts, r) ->
                                        addAccounts(
                                                ts,
                                                getProviderIdIfResourceIdIsPresent(resourceId, providerId, r),
                                                r.getFolderIdsWithTenants(),
                                                r
                                        )
                                        .flatMap(q -> addResourcesAndProvidersAndUnits(ts, q))
                                        .flatMap(q -> getFolderPermissions(ts, q.getFolders(), q.getProviders(),
                                                currentUser).map(q::withFolderPermissions))
                                        .flatMap(q -> fillAllocated(ts, q, resourceId))
                                        .map(q -> new WithTxId<>(q, ts.getId())),
                                (ts, r) -> ts.commitTransaction().thenReturn(new WithTxId<>(r, null))
                        )))).map(r -> r.apply(b -> new FrontFolderWithQuotesDtoBuilder(locale, accountsSpacesUtils)
                        .build(b)));
    }

    private String getProviderIdIfResourceIdIsPresent(
            String resourceId,
            String providerId,
            GetFrontForServiceContext.Impl context
    ) {
        if ((providerId == null || providerId.isEmpty()) && resourceId != null && !resourceId.isEmpty()
                && !context.getQuotas().isEmpty()
        ) {
            return context.getQuotas().get(0).getProviderId();
        }
        return providerId;
    }

    private Mono<GetFrontForServiceContext.Impl> fillAllocated(
            YdbTxSession ts, GetFrontForServiceContext.Impl q, String resourceId
    ) {
        Map<String, AccountModel> accountById = q.getAccountsById();
        return accountsQuotasDao.getAllByAccountIds(ts, q.getTenantId(), accountById.keySet())
                .map(quotas -> (resourceId == null || resourceId.isEmpty()) ? quotas :
                        quotas.stream().filter(quota -> quota.getResourceId().equals(resourceId))
                                .collect(Collectors.toList())
                )
                .map(q::withAccountsQuotas);
    }

    private <R, T extends FrontFolderWithQuotesDtoBuilder.AccountAcceptor<R>> Mono<R> addAccounts(
            YdbTxSession ts,
            String providerId,
            List<WithTenant<String>> folderIds,
            T con
    ) {
        Mono<List<AccountModel>> accountsMono;
        if (providerId == null || providerId.isEmpty()) {
            accountsMono = accountsDao.getAllByFolderIds(ts, folderIds, false);
        } else {
            if (!folderIds.isEmpty()) {
                TenantId tenantId = null;
                Set<String> rawFolderIds = new HashSet<>();
                for (WithTenant<String> folderId : folderIds) {
                    tenantId = folderId.getTenantId();
                    rawFolderIds.add(folderId.getIdentity());
                }
                accountsMono = accountsDao.getByFoldersForProvider(ts, tenantId, providerId, rawFolderIds, null, false);
            } else {
                accountsMono = Mono.just(List.of());
            }
        }

        return accountsMono.flatMap(accounts -> accountsSpacesUtils
                .getExpandedByProviderIds(ts, accounts.stream().map(a ->
                        new WithTenant<>(a.getTenantId(), a.getProviderId()))
                        .collect(Collectors.toSet()))
                .map(spaces -> con.withAccounts(accounts, spaces)));
    }

    public Mono<Result<ExpandedQuotas<Page<QuotaModel>>>> getPageForFolder(String folderId,
                                                                           PageRequest pageRequest,
                                                                           YaUserDetails currentUser,
                                                                           Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(r -> r
                .andThen(v -> validateFolderId(folderId, locale))
                .andThen(v -> pageRequest.<QuotaContinuationToken>validate(continuationTokenReader, messages, locale))
                .andThenDo(v -> validateContinuationToken(v, locale)).andThenMono(pageToken ->
                        tableClient.usingSessionMonoRetryable(session ->
                                validateFolder(immediateTx(session), folderId, locale)
                                        .flatMap(f -> loadQuotasPage(immediateTx(session), pageToken, f,
                                                locale, currentUser))
                        ).flatMap(this::expandQuotasPage)));
    }

    public Mono<Result<ExpandedQuotas<QuotaModel>>> getQuotaForFolderAndResource(String folderId,
                                                                                 String resourceId,
                                                                                 String providerId,
                                                                                 YaUserDetails currentUser,
                                                                                 Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(r -> r
                .andThen(v -> validateFolderId(folderId, locale))
                .andThen(v -> validateResourceId(resourceId, locale))
                .andThen(v -> validateProviderId(providerId, locale)).andThenMono(v ->
                        loadQuota(folderId, resourceId, providerId, currentUser, locale))
                .flatMap(this::expandQuota));
    }

    private Mono<Result<GetFrontForServiceContext.Impl>> checkProviderAndResource(
            Locale locale,
            @Nullable String providerId,
            @Nullable String resourceId,
            GetFrontForServiceContext.Impl context
    ) {
        if (providerId == null && resourceId == null) {
            return Mono.just(Result.success(context));
        }
        return checkResource(locale, providerId, resourceId, context)
                .zipWith(checkProvider(locale, providerId, context))
                .map(tuple2 -> tuple2.getT1().andThen(b -> tuple2.getT2()));
    }

    private Mono<Result<GetFrontForServiceContext.Impl>> checkResource(
            Locale locale,
            String providerId,
            String resourceId,
            GetFrontForServiceContext.Impl context
    ) {
        if (resourceId == null || resourceId.isEmpty()) {
            return Mono.just(Result.success(context));
        }
        return resourcesLoader.getResourceByIdImmediate(resourceId, context.getTenantId()).map(resourceModelO ->
                resourceModelO.map(r -> {
                    if (r.isDeleted()) {
                        return Result.<GetFrontForServiceContext.Impl>failure(ErrorCollection.builder().addError(
                                "resourceId", TypedError.badRequest(
                                        messages.getMessage("errors.resource.deleted", null, locale)
                                )
                        ).build());
                    }
                    if ((providerId == null || providerId.isEmpty()) || r.getProviderId().equals(providerId)) {
                        return Result.success(context.withResources(List.of(r)));
                    } else {
                        return Result.<GetFrontForServiceContext.Impl>failure(ErrorCollection.builder().addError(
                                "resourceId", TypedError.badRequest(
                                        messages.getMessage("errors.wrong.provider.for.resource", null, locale)
                                )
                        ).build());
                    }
                }).orElse(Result.failure(ErrorCollection.builder().addError(
                        "resourceId", TypedError.badRequest(
                                messages.getMessage("errors.resource.not.found", null, locale)
                        )
                        ).build())
                ));
    }

    private Mono<Result<GetFrontForServiceContext.Impl>> checkProvider(
            Locale locale,
            String providerId,
            GetFrontForServiceContext.Impl context
    ) {
        if (providerId == null || providerId.isEmpty()) {
            return Mono.just(Result.success(context));
        }
        return providersLoader.getProviderByIdImmediate(providerId, context.getTenantId()).map(providerModelO -> {
            if (providerModelO.isEmpty()) {
                return Result.failure(ErrorCollection.builder().addError(
                        "providerId", TypedError.badRequest(
                                messages.getMessage("errors.provider.not.found", null, locale)
                        )
                ).build());
            }
            final ProviderModel providerModel = providerModelO.get();
            if (providerModel.isDeleted()) {
                return Result.failure(ErrorCollection.builder().addError(
                        "providerId", TypedError.badRequest(
                                messages.getMessage("errors.provider.deleted", null, locale)
                        )
                ).build());
            }
            return Result.success(context.withProviders(List.of(providerModel)));
        });
    }

    private Mono<GetFrontForServiceContext.Impl> addResourcesAndProvidersAndUnits(
            YdbTxSession ts, GetFrontForServiceContext.Impl con
    ) {
        return addResources(ts, con)
                .flatMap(context -> addProviders(ts, context))
                .flatMap(context -> addUnits(ts, context))
                .flatMap(context -> addResourceTypes(ts, context))
                .flatMap(context -> addSegmentations(ts, context))
                .flatMap(context -> addSegments(ts, context));
    }

    private Mono<GetFrontForServiceContext.Impl> addUnits(
            YdbTxSession ts, GetFrontForServiceContext.Impl builder) {
        List<Tuple2<String, TenantId>> unitsEnsembleId = builder.getResources().stream()
                .map(resourceModel -> Tuples.of(resourceModel.getUnitsEnsembleId(), resourceModel.getTenantId()))
                .distinct().collect(Collectors.toList());

        return unitsEnsemblesLoader.getUnitsEnsemblesByIds(ts, unitsEnsembleId)
                .map(builder::withUnitsEnsembles);
    }

    private Mono<GetFrontForServiceContext.Impl> addResourceTypes(
            YdbTxSession ts, GetFrontForServiceContext.Impl builder) {
        final List<Tuple2<String, TenantId>> resourceTypeIds = builder.getResources().stream()
                .filter(res -> res.getResourceTypeId() != null)
                .map(res -> Tuples.of(res.getResourceTypeId(), res.getTenantId()))
                .collect(Collectors.toList());

        return resourceTypesLoader.getResourceTypesByIds(ts, resourceTypeIds)
                .map(builder::withResourceTypes);
    }

    private Mono<GetFrontForServiceContext.Impl> addResources(
            YdbTxSession ts, GetFrontForServiceContext.Impl builder) {
        //инфо о ресурсах
        if (builder.getResources() != null) {
            return Mono.just(builder);
        }
        List<QuotaModel> quotaModels = builder.getQuotas();
        List<Tuple2<String, TenantId>> resourceIds = quotaModels.stream()
                .map(quotaModel -> Tuples.of(quotaModel.getResourceId(), quotaModel.getTenantId()))
                .distinct().collect(Collectors.toList());
        return resourcesLoader.getResourcesByIds(ts, resourceIds).map(builder::withResources);
    }

    private Mono<GetFrontForServiceContext.Impl> addProviders(
            YdbTxSession ts, GetFrontForServiceContext.Impl builder) {
        if (builder.getProviders() != null) {
            return Mono.just(builder);
        }
        List<Tuple2<String, TenantId>> providersIds = Streams.concat(
                builder.getResources().stream().map(res -> Tuples.of(res.getProviderId(), res.getTenantId())),
                builder.getAccounts().stream().map(acc -> Tuples.of(acc.getProviderId(), acc.getTenantId()))
        ).distinct().collect(Collectors.toList());

        return providersLoader.getProvidersByIds(ts, providersIds)
                .map(builder::withProviders);
    }

    private Mono<GetFrontForServiceContext.Impl> addSegmentations(
            YdbTxSession ts, GetFrontForServiceContext.Impl builder
    ) {
        if (builder.getSegmentationsById() != null) {
            return Mono.just(builder);
        }
        List<Tuple2<String, TenantId>> segmentationIds = builder.getResources().stream()
                .flatMap(resource -> resource.getSegments().stream()
                        .map(segment -> Tuples.of(segment.getSegmentationId(), resource.getTenantId()))
                )
                .collect(Collectors.toList());
        return segmentationsLoader.getResourceSegmentationsByIds(ts, segmentationIds)
                .map(builder::withSegmentations);
    }

    private Mono<GetFrontForServiceContext.Impl> addSegments(YdbTxSession ts, GetFrontForServiceContext.Impl builder) {
        if (builder.getSegmentsById() != null) {
            return Mono.just(builder);
        }
        List<Tuple2<String, TenantId>> segmentIds = builder.getResources().stream()
                .flatMap(resource -> resource.getSegments().stream()
                        .map(segment -> Tuples.of(segment.getSegmentId(), resource.getTenantId()))
                )
                .collect(Collectors.toList());
        return segmentsLoader.getResourceSegmentsByIds(ts, segmentIds).map(builder::withSegments);
    }

    @SuppressWarnings("checkstyle:ParameterNumber")
    private Mono<ResultTx<GetFrontForServiceContext.Impl>> getQuotasPageAndFoldersForService(
            GetFrontForServiceContext.Impl con,
            PageRequest pageRequest,
            long serviceId,
            String providerId,
            String resourceId,
            boolean includeCompletelyZero,
            Locale locale,
            YdbTxSession session
    ) {
        Result<PageRequest.Validated<StringContinuationToken>> pageResult = pager.validatePageRequest(
                pageRequest, locale);
        if (pageResult.isFailure()) {
            return pageResult.applyMonoTx(u -> Mono.empty());
        }
        PageRequest.Validated<StringContinuationToken> pageRequestValidated = pageResult
                .match(Function.identity(), e -> null);
        int limit = pageRequestValidated.getLimit();
        String fromId = pageRequestValidated.getContinuationToken()
                .map(StringContinuationToken::getId).orElse(null);
        if ((providerId == null || providerId.isEmpty()) && (resourceId == null || resourceId.isEmpty())) {
            //TODO поход в базу в одной транзакции
            return folderServiceElement
                    .listFoldersByService(con.getTenantId(), serviceId, false, limit, fromId, locale)
                    .flatMap(r -> r.applyMonoTx(folderModelPage ->
                            quotasDao.getByFoldersStartTx(session,
                                    folderModelPage.getItems()
                                            .stream().map(FolderModel::getId).collect(Collectors.toList()),
                                    con.getTenantId(),
                                    includeCompletelyZero
                            ).map(tuple2 -> {
                                List<QuotaModel> quotas = tuple2.getT1();
                                String transactionId = tuple2.getT2();
                                return new WithTxId<>(con
                                        .withContinuationToken(folderModelPage.getContinuationToken().orElse(null))
                                        .withFolders(folderModelPage.getItems())
                                        .withQuotas(quotas),
                                        transactionId
                                );
                            })));
        }

        return ((resourceId == null || resourceId.isEmpty()) ?
                getByServiceAndProviderStartTx(con, serviceId, providerId, session, limit, fromId) :
                quotasDao.getByServiceAndProviderAndResourceStartTx(
                        session, serviceId, providerId, resourceId, con.getTenantId(), limit + 1, fromId
                )
        ).map(tuple3 -> {
            List<FolderModel> folders = tuple3.getT1();
            List<QuotaModel> quotas = tuple3.getT2();
            String transactionId = tuple3.getT3();
            if (folders.size() > limit + 1) {
                throw new IllegalStateException("Number of folders in result is more than limit");
            }

            String lastFolderId = folders.size() > 0 ? folders.get(folders.size() - 1).getId() : null;
            Page<FolderModel> page = pager.toPage(folders, limit, FolderModel::getId);
            return ResultTx.success(
                    page.getContinuationToken().map(continuationToken ->
                            con
                                    .withContinuationToken(continuationToken)
                                    .withFolders(page.getItems())
                                    .withQuotas(quotas.stream()
                                            .filter(quota -> !quota.getFolderId().equals(lastFolderId))
                                            .collect(Collectors.toList()))
                    ).orElseGet(() ->
                            con
                                    .withFolders(folders)
                                    .withQuotas(quotas)
                    ),
                    transactionId
            );
        });
    }

    private Mono<Tuple3<List<FolderModel>, List<QuotaModel>, String>> getByServiceAndProviderStartTx(
            GetFrontForServiceContext.Impl con,
            long serviceId, String providerId,
            YdbTxSession session,
            int limit,
            String fromId
    ) {
        return quotasDao.getFoldersWithQuotas(session, serviceId, providerId, con.getTenantId(), limit + 1, fromId)
                .flatMap(qfs -> folderDao
                        .getFolderWithAccounts(session, serviceId, providerId, con.getTenantId(), limit + 1, fromId)
                        .map(afs -> qfs
                                .map(quotaFolders -> Streams.concat(quotaFolders.stream(), afs.stream())
                                        .distinct()
                                        .sorted(Comparator.comparing(FolderModel::getId))
                                        .limit(limit + 1)
                                        .collect(Collectors.toList()))))
                .flatMap(folders ->
                        folders.get().isEmpty() ?
                                Mono.just(Tuples.of(folders.get(), List.of(), folders.getTransactionId())) :
                                quotasDao.getByFoldersAndProvider(
                                                session,
                                                folders.get().stream()
                                                        .map(FolderModel::getId)
                                                        .collect(Collectors.toList()),
                                                con.getTenantId(), providerId)
                                        .map(quotas -> Tuples.of(folders.get(), quotas, folders.getTransactionId())
                                        ));
    }

    private Result<Void> validateFolderId(String folderId, Locale locale) {
        if (!Uuids.isValidUuid(folderId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.folder.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateAccountId(String accountId, Locale locale) {
        if (!Uuids.isValidUuid(accountId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                            .getMessage("errors.account.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateResourceId(String resourceId, Locale locale) {
        if (!Uuids.isValidUuid(resourceId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.resource.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateProviderId(String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateContinuationToken(PageRequest.Validated<QuotaContinuationToken> pageRequest,
                                                   Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!Uuids.isValidUuid(pageRequest.getContinuationToken().get().getProviderId())) {
            errors.addError(TypedError.badRequest(messages
                    .getMessage("errors.provider.not.found", null, locale)));
        }
        if (!Uuids.isValidUuid(pageRequest.getContinuationToken().get().getResourceId())) {
            errors.addError(TypedError.badRequest(messages
                    .getMessage("errors.resource.not.found", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private Mono<Result<FolderModel>> validateFolder(YdbTxSession session, String folderId, Locale locale) {
        return folderDao.getById(session, folderId, Tenants.DEFAULT_TENANT_ID)
                .map(r -> {
                    if (r.isPresent() && !r.get().isDeleted()) {
                        return Result.success(r.get());
                    }
                    ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                            .getMessage("errors.folder.not.found", null, locale)))
                            .build();
                    return Result.failure(error);
                });
    }

    private Mono<Result<Page<QuotaModel>>> loadQuotasPage(YdbTxSession session,
                                                          PageRequest.Validated<QuotaContinuationToken> page,
                                                          Result<FolderModel> folderR,
                                                          Locale locale,
                                                          YaUserDetails currentUser) {
        return folderR.andThenMono(
                f -> securityManagerService.checkReadPermissions(currentUser, locale, f)
        ).flatMap(r -> r.andThenMono(folder -> {
            int limit = page.getLimit();
            String providerId = page.getContinuationToken().map(QuotaContinuationToken::getProviderId).orElse(null);
            String resourceId = page.getContinuationToken().map(QuotaContinuationToken::getResourceId).orElse(null);
            return quotasDao.listQuotasByFolder(session, Tenants.DEFAULT_TENANT_ID, folder.getId(), limit + 1,
                            providerId, resourceId)
                .flatMap(quotaModels -> {
                    Set<String> resourceIds = new HashSet<>();
                    Set<String> providerIds = new HashSet<>();
                    quotaModels.forEach(quota -> {
                        resourceIds.add(quota.getResourceId());
                        providerIds.add(quota.getProviderId());
                    });
                    List<Tuple2<String, TenantId>> resourceIdPairs = resourceIds.stream()
                            .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                            .collect(Collectors.toList());
                    List<Tuple2<String, TenantId>> providerIdPairs = providerIds.stream()
                            .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                            .collect(Collectors.toList());
                    return resourcesLoader.getResourcesByIds(session, resourceIdPairs)
                            .map(resources -> resources.stream()
                                    .collect(Collectors.toMap(ResourceModel::getId, Function.identity())))
                            .flatMap(resources -> providersLoader.getProvidersByIds(session, providerIdPairs)
                                .map(providers -> providers.stream()
                                    .collect(Collectors.toMap(ProviderModel::getId, Function.identity())))
                                .map(providers -> quotaModels.stream()
                                    .map(quotaModel ->
                                        providers.get(quotaModel.getProviderId()).isManaged() &&
                                            resources.get(quotaModel.getResourceId()).isManaged() ?
                                                quotaModel : zeroQuotaModel(folder.getId(), quotaModel.getProviderId(),
                                                        quotaModel.getResourceId()))
                                    .collect(Collectors.toList())));
                })
                .map(values -> Result.success(values.size() > limit
                        ? Page.page(values.subList(0, limit), prepareToken(values.get(limit - 1)))
                        : Page.lastPage(values)));
        }));
    }

    private Mono<Result<QuotaModel>> loadQuota(String folderId, String resourceId, String providerId,
                                               YaUserDetails currentUser, Locale locale) {
        return Mono.zip(providersLoader.getProviderByIdImmediate(providerId, Tenants.DEFAULT_TENANT_ID),
                resourcesLoader.getResourceByIdImmediate(resourceId, Tenants.DEFAULT_TENANT_ID))
                .flatMap(tuple -> {
                    ErrorCollection.Builder errors = ErrorCollection.builder();
                    if (tuple.getT1().isEmpty() || tuple.getT1().get().isDeleted()) {
                        errors.addError(TypedError.notFound(messages
                                .getMessage("errors.provider.not.found", null, locale)));
                    }
                    if (tuple.getT2().isEmpty() || tuple.getT2().get().isDeleted()
                            || !tuple.getT2().get().getProviderId().equals(providerId)) {
                        errors.addError(TypedError.notFound(messages
                                .getMessage("errors.resource.not.found", null, locale)));
                    }
                    if (errors.hasAnyErrors()) {
                        return Mono.just(Result.failure(errors.build()));
                    }
                    if (!tuple.getT1().get().isManaged() || !tuple.getT2().get().isManaged()) {
                        return Mono.just(Result.success(zeroQuotaModel(folderId, providerId, resourceId)));
                    }
                    return loadQuota(folderId, tuple.getT2().get(), tuple.getT1().get(), currentUser, locale);
                });
    }

    private Mono<Result<QuotaModel>> loadQuota(String folderId, ResourceModel resource, ProviderModel provider,
                                               YaUserDetails currentUser, Locale locale) {
        return tableClient.usingSessionMonoRetryable(session ->
                validateFolder(immediateTx(session), folderId, locale)
                        .flatMap(f -> loadQuota(immediateTx(session), f, resource, provider, currentUser, locale)));
    }

    private Mono<Result<QuotaModel>> loadQuota(YdbTxSession session, Result<FolderModel> folderR,
                                               ResourceModel resource, ProviderModel provider,
                                               YaUserDetails currentUser, Locale locale) {
        return folderR.andThenMono(
                f -> securityManagerService.checkReadPermissions(currentUser, locale, f)
        ).flatMap(r -> r.andThenMono(folder ->
                quotasDao.getOneQuota(session, folder.getId(), provider.getId(), resource.getId(),
                        Tenants.DEFAULT_TENANT_ID).map(quotaO -> {
                    if (quotaO.isEmpty()) {
                        return Result.success(zeroQuotaModel(folder.getId(), provider.getId(), resource.getId()));
                    }
                    return Result.success(quotaO.get());
                })));
    }

    private String prepareToken(QuotaModel lastItem) {
        return ContinuationTokens.encode(new QuotaContinuationToken(lastItem.getProviderId(),
                lastItem.getResourceId()), continuationTokenWriter);
    }

    private Mono<Result<ExpandedQuotas<Page<QuotaModel>>>> expandQuotasPage(Result<Page<QuotaModel>> pageResult) {
        return pageResult.andThenMono(page -> {
            Set<String> resourceIds = new HashSet<>();
            page.getItems().forEach(quota -> resourceIds.add(quota.getResourceId()));
            List<Tuple2<String, TenantId>> resourceIdPairs = resourceIds.stream()
                    .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
            return resourcesLoader.getResourcesByIdsImmediate(resourceIdPairs).flatMap(loadedResources -> {
                Map<String, ResourceModel> resources = loadedResources.stream()
                        .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
                Set<String> unitsEnsembleIds = new HashSet<>();
                loadedResources.forEach(r -> unitsEnsembleIds.add(r.getUnitsEnsembleId()));
                List<Tuple2<String, TenantId>> unitsEnsembleIdPairs = unitsEnsembleIds.stream()
                        .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
                return unitsEnsemblesLoader.getUnitsEnsemblesByIdsImmediate(unitsEnsembleIdPairs)
                        .map(loadedUnitsEnsembles -> {
                            Map<String, UnitsEnsembleModel> unitsEnsembles = loadedUnitsEnsembles.stream()
                                    .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
                            return Result.success(new ExpandedQuotas<>(page, resources, unitsEnsembles));
                        });
            });

        });
    }

    private Mono<Result<ExpandedQuotas<QuotaModel>>> expandQuota(Result<QuotaModel> quotaR) {
        return quotaR.andThenMono(quota ->
                resourcesLoader.getResourceByIdImmediate(quota.getResourceId(), Tenants.DEFAULT_TENANT_ID)
                        .flatMap(resource -> {
                            if (resource.isEmpty()) {
                                return Mono.error(new IllegalStateException("Missing resource "
                                        + quota.getResourceId()));
                            }
                            return unitsEnsemblesLoader.getUnitsEnsembleByIdImmediate(resource.get()
                                    .getUnitsEnsembleId(), Tenants.DEFAULT_TENANT_ID)
                                    .flatMap(unitsEnsemble -> {
                                        if (unitsEnsemble.isEmpty()) {
                                            return Mono.error(new IllegalStateException("Missing units ensemble "
                                                    + resource.get().getUnitsEnsembleId()));
                                        }
                                        return Mono.just(Result.success(new ExpandedQuotas<>(quota,
                                                Map.of(resource.get().getId(), resource.get()),
                                                Map.of(unitsEnsemble.get().getId(), unitsEnsemble.get()))));
                                    });
                        }));
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private QuotaModel zeroQuotaModel(String folderId, String providerId, String resourceId) {
        return QuotaModel.builder()
                .folderId(folderId)
                .providerId(providerId)
                .resourceId(resourceId)
                .quota(0L)
                .balance(0L)
                .frozenQuota(0L)
                .tenantId(Tenants.DEFAULT_TENANT_ID)
                .build();
    }
}
