package ru.yandex.intranet.d.services.recipe;

import java.math.BigDecimal;
import java.time.DateTimeException;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.yandex.ydb.table.transaction.TransactionMode;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import it.unimi.dsi.fastutil.longs.LongSet;
import it.unimi.dsi.fastutil.longs.LongSets;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao;
import ru.yandex.intranet.d.dao.users.AbcIntranetStaffDao;
import ru.yandex.intranet.d.dao.users.AbcServiceMemberDao;
import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.services.ServiceRecipeModel;
import ru.yandex.intranet.d.model.services.ServiceState;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.AbcServiceMemberModel;
import ru.yandex.intranet.d.model.users.AbcServiceMemberState;
import ru.yandex.intranet.d.model.users.AbcUserModel;
import ru.yandex.intranet.d.model.users.StaffAffiliation;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.model.users.UserServiceRoles;
import ru.yandex.intranet.d.util.Long2LongMultimap;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeDto;
import ru.yandex.intranet.d.web.model.recipe.AccountDto;
import ru.yandex.intranet.d.web.model.recipe.AccountQuotaDto;
import ru.yandex.intranet.d.web.model.recipe.AccountsDto;
import ru.yandex.intranet.d.web.model.recipe.AccountsQuotasDto;
import ru.yandex.intranet.d.web.model.recipe.CreateServiceDto;
import ru.yandex.intranet.d.web.model.recipe.CreateUserDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Recipe service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
@Profile({"test-recipe"})
public class RecipeService {

    private static final int MAX_NAME_LENGTH = 256;
    private static final int MAX_SLUG_LENGTH = 256;

    private final MessageSource messages;
    private final ServicesDao servicesDao;
    private final UsersDao usersDao;
    private final AbcIntranetStaffDao abcIntranetStaffDao;
    private final AbcServiceMemberDao abcServiceMemberDao;
    private final AccountsDao accountsDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final ResourcesDao resourcesDao;
    private final UnitsEnsemblesDao unitsEnsemblesDao;
    private final YdbTableClient tableClient;
    private final Set<Long> validRoles;
    private final Map<Long, UserServiceRoles> roleByIdMap;

    @SuppressWarnings("ParameterNumber")
    public RecipeService(@Qualifier("messageSource") MessageSource messages,
                         ServicesDao servicesDao,
                         UsersDao usersDao,
                         AbcIntranetStaffDao abcIntranetStaffDao,
                         AbcServiceMemberDao abcServiceMemberDao,
                         AccountsDao accountsDao,
                         AccountsQuotasDao accountsQuotasDao,
                         ResourcesDao resourcesDao,
                         UnitsEnsemblesDao unitsEnsemblesDao,
                         YdbTableClient tableClient,
                         @Value("${abc.roles.quotaManager}") long quotaManagerRoleId,
                         @Value("${abc.roles.responsibleOfProvider}") long responsibleOfProviderRoleId,
                         @Value("${abc.roles.serviceProductHead}") long serviceProductHeadRoleId,
                         @Value("${abc.roles.serviceProductDeputyHead}") long serviceProductDeputyHeadRoleId,
                         @Value("${abc.roles.serviceResponsible}") long serviceResponsibleRoleId) {
        this.messages = messages;
        this.servicesDao = servicesDao;
        this.usersDao = usersDao;
        this.abcIntranetStaffDao = abcIntranetStaffDao;
        this.abcServiceMemberDao = abcServiceMemberDao;
        this.accountsDao = accountsDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.resourcesDao = resourcesDao;
        this.unitsEnsemblesDao = unitsEnsemblesDao;
        this.tableClient = tableClient;
        this.roleByIdMap = Map.of(
                quotaManagerRoleId, UserServiceRoles.QUOTA_MANAGER,
                responsibleOfProviderRoleId, UserServiceRoles.RESPONSIBLE_OF_PROVIDER,
                serviceProductHeadRoleId, UserServiceRoles.SERVICE_PRODUCT_HEAD,
                serviceProductDeputyHeadRoleId, UserServiceRoles.SERVICE_PRODUCT_DEPUTY_HEAD,
                serviceResponsibleRoleId, UserServiceRoles.SERVICE_RESPONSIBLE
        );
        this.validRoles = roleByIdMap.keySet();
    }

    public Mono<Result<Void>> createService(CreateServiceDto service, YaUserDetails currentUser, Locale locale) {
        return checkAdminPermissions(currentUser, locale).andThenMono(v -> preValidateService(service, locale)
                .andThenMono(s -> tableClient.usingSessionMonoRetryable(session ->
                        session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                                validateService(ts, s, locale).flatMap(r -> r.andThenMono(validatedService -> {
                                    Mono<LongSet> parentParents;
                                    if (validatedService.getParentId().isPresent()) {
                                        parentParents = servicesDao.getAllParents(ts,
                                                validatedService.getParentId().get(), Tenants.DEFAULT_TENANT_ID);
                                    } else {
                                        parentParents = Mono.just(LongSets.EMPTY_SET);
                                    }
                                    return parentParents.flatMap(p -> {
                                        Long2LongMultimap parentsByServiceId = new Long2LongMultimap();
                                        if (validatedService.getParentId().isEmpty()) {
                                            parentsByServiceId.resetAll(validatedService.getId(),
                                                    new LongOpenHashSet());
                                        } else {
                                            LongOpenHashSet parents = new LongOpenHashSet(p);
                                            parents.add(validatedService.getParentId().get().longValue());
                                            parentsByServiceId.resetAll(validatedService.getId(), parents);
                                        }
                                        return servicesDao.upsertRecipeRetryable(ts, validatedService)
                                                .then(Mono.defer(() -> servicesDao
                                                        .upsertAllParentsRetryable(ts, parentsByServiceId,
                                                                Tenants.DEFAULT_TENANT_ID)))
                                                .thenReturn(Result.success(null));
                                    });

                                }))))
                ));
    }

    public Mono<Result<Void>> createUser(CreateUserDto user, YaUserDetails currentUser, Locale locale) {
        return checkAdminPermissions(currentUser, locale).andThenMono(v -> preValidateUser(user, locale)
                .andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
                        session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> handleUser(ts, user, u, locale)))));
    }

    public Mono<Result<AccountsDto>> getAccounts(YaUserDetails currentUser, Locale locale) {
        return checkAdminPermissions(currentUser, locale).andThenMono(v ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                                accountsDao.getAllByTenant(ts, Tenants.DEFAULT_TENANT_ID)
                                .map(accounts -> Result.success(new AccountsDto(accounts.stream()
                                        .map(account -> new AccountDto(account.getId(),
                                                account.getProviderId(),
                                                account.getOuterAccountIdInProvider(),
                                                account.getOuterAccountKeyInProvider().orElse(null),
                                                account.getFolderId(),
                                                account.getDisplayName().orElse(null),
                                                account.isDeleted(),
                                                account.getLastAccountUpdate(),
                                                account.getLastReceivedVersion().orElse(null),
                                                account.getLatestSuccessfulAccountOperationId().orElse(null),
                                                account.getAccountsSpacesId().orElse(null),
                                                account.isFreeTier(),
                                                AccountReserveTypeDto.fromModel(account
                                                        .getReserveType().orElse(null))))
                                        .collect(Collectors.toList())))))));
    }

    public Mono<Result<AccountsQuotasDto>> getAccountsQuotas(YaUserDetails currentUser, Locale locale) {
        return checkAdminPermissions(currentUser, locale).andThenMono(v ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts -> accountsQuotasDao
                                .getAllByTenant(ts, Tenants.DEFAULT_TENANT_ID).flatMap(quotas -> {
                                    Set<String> accountIds = quotas.stream().map(AccountsQuotasModel::getAccountId)
                                            .collect(Collectors.toSet());
                                    return loadAccounts(ts, accountIds).flatMap(accountsById -> {
                                        Set<String> resourceIds = quotas.stream()
                                                .map(AccountsQuotasModel::getResourceId)
                                                .collect(Collectors.toSet());
                                        return loadResources(ts, resourceIds).flatMap(resourcesById -> {
                                            Set<String> unitsEnsemblesIds = resourcesById.values().stream()
                                                    .map(ResourceModel::getUnitsEnsembleId).collect(Collectors.toSet());
                                            return loadUnitsEnsembles(ts, unitsEnsemblesIds).map(unitsEnsemblesById ->
                                                    Result.success(new AccountsQuotasDto(quotas.stream()
                                                            .map(quota -> toAccountQuotaDto(accountsById, resourcesById,
                                                                    unitsEnsemblesById, quota))
                                                            .collect(Collectors.toList()))));
                                        });
                                    });
                                }))));
    }

    @NotNull
    public static AccountQuotaDto toAccountQuotaDto(Map<String, AccountModel> accountsById,
                                                    Map<String, ResourceModel> resourcesById,
                                                    Map<String, UnitsEnsembleModel> unitsEnsemblesById,
                                                    AccountsQuotasModel quota) {
        AccountModel account = accountsById.get(quota.getAccountId());
        ResourceModel resource = resourcesById.get(quota.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsemblesById
                .get(resource.getUnitsEnsembleId());
        long providedAmount = Objects.requireNonNullElse(
                quota.getProvidedQuota(), 0L);
        long allocatedAmount = Objects.requireNonNullElse(
                quota.getAllocatedQuota(), 0L);
        Tuple2<BigDecimal, UnitModel> provided = Units
                .convertToApi(providedAmount, resource, unitsEnsemble);
        Tuple2<BigDecimal, UnitModel> allocated = Units
                .convertToApi(allocatedAmount, resource, unitsEnsemble);
        Tuple2<BigDecimal, UnitModel> frozen = Units
                .convertToApi(quota.getFrozenProvidedQuota(), resource, unitsEnsemble);
        return new AccountQuotaDto(quota.getAccountId(),
                quota.getResourceId(),
                provided.getT1().longValue(),
                allocated.getT1().longValue(),
                frozen.getT1().longValue(),
                quota.getFolderId(),
                quota.getProviderId(),
                quota.getLastProvisionUpdate(),
                quota.getLastReceivedProvisionVersion().orElse(null),
                quota.getLatestSuccessfulProvisionOperationId().orElse(null),
                provided.getT2().getKey(),
                allocated.getT2().getKey(),
                frozen.getT2().getKey(),
                account.getOuterAccountIdInProvider(),
                account.getAccountsSpacesId().orElse(null));
    }

    private Mono<Map<String, ResourceModel>> loadResources(YdbTxSession session, Set<String> resourceIds) {
        if (resourceIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<Tuple2<String, TenantId>> ids = resourceIds.stream().map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                .collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500)).concatMap(v -> resourcesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(ResourceModel::getId, Function.identity())));
    }

    private Mono<Map<String, UnitsEnsembleModel>> loadUnitsEnsembles(YdbTxSession session,
                                                                    Set<String> unitsEnsemblesIds) {
        if (unitsEnsemblesIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<Tuple2<String, TenantId>> ids = unitsEnsemblesIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500)).concatMap(v -> unitsEnsemblesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity())));
    }

    private Mono<Map<String, AccountModel>> loadAccounts(YdbTxSession session, Set<String> accountIds) {
        if (accountIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        List<WithTenant<String>> ids = accountIds.stream().map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id))
                .collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500)).concatMap(v -> accountsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .collect(Collectors.toMap(AccountModel::getId, Function.identity())));
    }

    private Result<Void> checkAdminPermissions(YaUserDetails currentUser, Locale locale) {
        if (currentUser.getUser().isEmpty() || !currentUser.getUser().get().getDAdmin()) {
            return Result.failure(ErrorCollection.builder().addError(TypedError
                    .forbidden(messages.getMessage("errors.access.denied", null, locale))).build());
        }
        return Result.success(null);
    }

    private Result<ServiceRecipeModel> preValidateService(CreateServiceDto service, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        ServiceRecipeModel.Builder builder = ServiceRecipeModel.builder();
        builder.state(ServiceState.DEVELOP);
        builder.exportable(true);
        validateText(service::getNameEn, builder::nameEn, errors, "nameEn", MAX_NAME_LENGTH, locale);
        validateText(service::getNameRu, builder::name, errors, "nameRu", MAX_NAME_LENGTH, locale);
        validateText(service::getSlug, builder::slug, errors, "slug", MAX_SLUG_LENGTH, locale);
        validateRequired(service::getId, builder::id, errors, "id", locale);
        service.getParentId().ifPresent(builder::parentId);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(builder.build());
    }

    private Mono<Result<ServiceRecipeModel>> validateService(YdbTxSession session, ServiceRecipeModel service,
                                                             Locale locale) {
        return servicesDao.getByIdMinimal(session, service.getId()).map(WithTxId::get).flatMap(serviceO -> {
            if (serviceO.isPresent()) {
                return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.conflict(
                        messages.getMessage("errors.service.already.exists", null, locale))).build()));
            }
            if (service.getParentId().isEmpty()) {
                return Mono.just(Result.success(service));
            }
            return servicesDao.getByIdMinimal(session, service.getParentId().get()).map(WithTxId::get)
                    .flatMap(parentServiceO -> {
                        if (parentServiceO.isEmpty()) {
                            return Mono.just(Result.failure(ErrorCollection.builder().addError("parentId",
                                    TypedError.invalid(messages.getMessage("errors.service.not.found",
                                            null, locale))).build()));
                        }
                        return Mono.just(Result.success(service));
                    });
        });
    }

    private Result<AbcUserModel.Builder> preValidateUser(CreateUserDto user, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        AbcUserModel.Builder builder = AbcUserModel.newBuilder();
        validateText(user::getFirstNameEn, builder::setFirstNameEn, errors, "firstNameEn", MAX_NAME_LENGTH, locale);
        validateText(user::getFirstNameRu, builder::setFirstName, errors, "firstNameRu", MAX_NAME_LENGTH, locale);
        validateText(user::getLastNameEn, builder::setLastNameEn, errors, "lastNameEn", MAX_NAME_LENGTH, locale);
        validateText(user::getLastNameRu, builder::setLastName, errors, "lastNameRu", MAX_NAME_LENGTH, locale);
        validateText(user::getLogin, builder::setLogin, errors, "login", MAX_NAME_LENGTH, locale);
        validateRequired(user::getUid, v -> builder.setUid(v.toString()), errors, "uid", locale);
        validateText(user::getWorkEmail, builder::setWorkEmail, errors, "workEmail", MAX_NAME_LENGTH, locale);
        validateGender(user::getGender, v -> builder.setGender(v.charAt(0)), errors, "gender", locale);
        validateLangUi(user::getLangUi, builder::setLangUi, errors, "langUi", locale);
        validateTimeZone(user::getTimeZone, builder::setTz, errors, "timeZone", locale);
        validateRequired(user::getAdmin, v -> { }, errors, "admin", locale);
        if (user.getRolesByService().isPresent()) {
            user.getRolesByService().get().forEach((serviceId, roleIds) -> {
                if (serviceId == null || roleIds == null || roleIds.isEmpty()) {
                    errors.addError("rolesByService", TypedError.invalid(messages
                            .getMessage("errors.invalid.value", null, locale)));
                }
                if (roleIds != null) {
                    boolean invalidRole = roleIds.stream()
                            .anyMatch(roleId -> roleId == null || !validRoles.contains(roleId));
                    if (invalidRole) {
                        errors.addError("rolesByService", TypedError.invalid(messages
                                .getMessage("errors.invalid.value", null, locale)));
                    }
                }
            });
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        builder.setAffiliation("yandex");
        builder.setIsDismissed(false);
        builder.setIsRobot(false);
        return Result.success(builder);
    }

    private Mono<Result<Void>> handleUser(YdbTxSession session, CreateUserDto user,
                                                  AbcUserModel.Builder preValidatedUser, Locale locale) {
        List<Tuple2<String, TenantId>> uid = List.of(Tuples.of(user.getLogin().get(), Tenants.DEFAULT_TENANT_ID));
        List<Tuple2<String, TenantId>> login
                = List.of(Tuples.of(user.getLogin().get(), Tenants.DEFAULT_TENANT_ID));
        List<Tuple2<Long, TenantId>> staffId = List.of();
        Set<Long> serviceIds = user.getRolesByService().orElse(Map.of()).keySet();
        return usersDao.getByExternalIds(session, uid, login, staffId).flatMap(users -> {
            if (!users.isEmpty()) {
                return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.conflict(
                        messages.getMessage("errors.user.already.exists", null, locale))).build()));
            }
            return getServices(session, new ArrayList<>(serviceIds)).flatMap(existingServices -> {
                if (existingServices.size() < serviceIds.size()) {
                    return Mono.just(Result.failure(ErrorCollection.builder().addError("rolesByService",
                            TypedError.invalid(messages.getMessage("errors.invalid.value", null, locale)))
                            .build()));
                }
                return abcIntranetStaffDao.getMaxId(session).flatMap(maxUserIdO -> {
                    long nextUserId = maxUserIdO.orElse(-1L) + 1L;
                    AbcUserModel abcUser = preValidatedUser
                            .setId(nextUserId)
                            .build();
                    UserModel userModel = UserModel.builder()
                            .firstNameEn(abcUser.getFirstNameEn())
                            .firstNameRu(abcUser.getFirstName())
                            .lastNameEn(abcUser.getLastNameEn())
                            .lastNameRu(abcUser.getLastName())
                            .passportLogin(abcUser.getLogin())
                            .passportUid(abcUser.getUid())
                            .staffId(abcUser.getId())
                            .workEmail(abcUser.getWorkEmail())
                            .gender(String.valueOf(abcUser.getGender()))
                            .langUi(abcUser.getLangUi())
                            .timeZone(abcUser.getTz())
                            .roles(prepareRoles(user.getRolesByService().orElse(null)))
                            .dAdmin(user.getAdmin().get())
                            .id(UUID.randomUUID().toString())
                            .tenantId(Tenants.DEFAULT_TENANT_ID)
                            .deleted(false)
                            .staffAffiliation(StaffAffiliation.YANDEX)
                            .staffRobot(false)
                            .staffDismissed(false)
                            .build();
                    return abcServiceMemberDao.getMaxId(session).flatMap(maxMembershipIdO -> {
                        long nextMembershipId = maxMembershipIdO.orElse(-1L) + 1L;
                        List<AbcServiceMemberModel> memberships = new ArrayList<>();
                        for (Map.Entry<Long, Set<Long>> entry : user.getRolesByService().orElse(Map.of()).entrySet()) {
                            for (Long roleId : entry.getValue()) {
                                memberships.add(AbcServiceMemberModel.newBuilder()
                                        .id(nextMembershipId)
                                        .serviceId(entry.getKey())
                                        .staffId(userModel.getStaffId().get())
                                        .roleId(roleId)
                                        // TODO Support membership state
                                        .state(AbcServiceMemberState.ACTIVE)
                                        .build());
                                nextMembershipId++;
                            }
                        }
                        return usersDao.upsertUserRetryable(session, userModel)
                                .then(abcIntranetStaffDao.upsertOneRetryable(session, abcUser))
                                .then(abcServiceMemberDao.upsertManyRetryable(session, memberships))
                                .thenReturn(Result.success(null));
                    });
                });
            });
        });
    }

    private Map<UserServiceRoles, Set<Long>> prepareRoles(Map<Long, Set<Long>> rolesByService) {
        if (rolesByService == null) {
            return Map.of();
        }
        Map<UserServiceRoles, Set<Long>> result = new HashMap<>();
        rolesByService.forEach((serviceId, roleIds) -> roleIds.forEach(roleId -> result
                .computeIfAbsent(roleByIdMap.get(roleId), r -> new HashSet<>()).add(serviceId)));
        return result;
    }

    private Mono<List<ServiceMinimalModel>> getServices(YdbTxSession session, List<Long> serviceIds) {
        if (serviceIds.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(serviceIds, 500))
                .concatMap(v -> servicesDao.getByIdsMinimal(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    private void validateText(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private <T> void validateRequired(Supplier<Optional<T>> getter, Consumer<T> setter,
                                      ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<T> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    private void validateGender(Supplier<Optional<String>> getter, Consumer<String> setter,
                                ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (!text.get().equals("F") && !text.get().equals("M")) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.invalid.value", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private void validateLangUi(Supplier<Optional<String>> getter, Consumer<String> setter,
                                ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (!text.get().equals("en") && !text.get().equals("ru")) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.invalid.value", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private void validateTimeZone(Supplier<Optional<String>> getter, Consumer<String> setter,
                                  ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (!checkZoneId(text.get())) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.invalid.value", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private boolean checkZoneId(String timezone) {
        try {
            ZoneId.of(timezone);
            return true;
        } catch (DateTimeException e) {
            return false;
        }
    }

}
