package ru.yandex.intranet.d.services.resources;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.providers.ProviderId;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.services.elements.StringContinuationTokenPager;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.validators.ProviderValidator;
import ru.yandex.intranet.d.services.validators.SegmentationsValidator;
import ru.yandex.intranet.d.services.validators.TextValidator;
import ru.yandex.intranet.d.services.validators.VersionValidator;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.paging.StringContinuationToken;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.resources.directory.CreateResourceSegmentationSegmentDto;
import ru.yandex.intranet.d.web.model.resources.directory.spaces.AccountsSpaceCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.spaces.AccountsSpacePutDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * AccountsSpacesService.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 19.01.2021
 */
@Component
public class AccountsSpacesService {
    private static final int MAX_KEY_LENGTH = 256;
    private static final int MAX_NAME_LENGTH = 256;
    private static final int MAX_DESCRIPTION_LENGTH = 1024;

    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final StringContinuationTokenPager pager;
    private final YdbTableClient tableClient;

    private final AccountsSpacesDao accountsSpacesDao;
    private final ProviderValidator providerValidator;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final ResourceSegmentsLoader resourceSegmentsLoader;
    private final AccountSpacesLoader accountSpacesLoader;
    private final TextValidator textValidator;
    private final SegmentationsValidator segmentationsValidator;
    private final VersionValidator versionValidator;
    private final AccountsSpacesUtils accountsSpacesUtils;

    @SuppressWarnings("ParameterNumber")
    public AccountsSpacesService(
            @Qualifier("messageSource") MessageSource messages,
            SecurityManagerService securityManagerService,
            StringContinuationTokenPager pager,
            YdbTableClient tableClient,
            AccountsSpacesDao accountsSpacesDao,
            ProviderValidator providerValidator,
            ResourceSegmentationsLoader resourceSegmentationsLoader,
            ResourceSegmentsLoader resourceSegmentsLoader,
            AccountSpacesLoader accountSpacesLoader,
            TextValidator textValidator,
            SegmentationsValidator segmentationsValidator,
            VersionValidator versionValidator,
            AccountsSpacesUtils accountsSpacesUtils
    ) {
        this.tableClient = tableClient;
        this.accountsSpacesDao = accountsSpacesDao;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.pager = pager;
        this.providerValidator = providerValidator;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.resourceSegmentsLoader = resourceSegmentsLoader;
        this.accountSpacesLoader = accountSpacesLoader;
        this.textValidator = textValidator;
        this.segmentationsValidator = segmentationsValidator;
        this.versionValidator = versionValidator;
        this.accountsSpacesUtils = accountsSpacesUtils;
    }

    public Mono<Result<ExpandedAccountsSpaces<AccountSpaceModel>>> getByIdExpanded(
            String accountSpaceId,
            String providerId,
            YaUserDetails currentUser,
            Locale locale
    ) {
        return getById(accountSpaceId, providerId, currentUser, locale).flatMap(r -> r.applyMono(this::expand));
    }

    public Mono<Result<AccountSpaceModel>> getById(
            String accountSpaceId,
            String providerId,
            YaUserDetails currentUser,
            Locale locale
    ) {
        return securityManagerService
                .checkReadPermissions(currentUser, locale).flatMap(result -> result.andThenMono(u -> providerValidator
                        .validateProvider(Tenants.getTenantId(currentUser), providerId, locale))
                ).flatMap(r -> r.andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
                        accountsSpacesDao.getById(
                                session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                                accountSpaceId,
                                Tenants.getTenantId(currentUser)
                        )).map(o -> o.map(Result::success)
                        .orElse(Result.failure(ErrorCollection.builder().addError(
                                TypedError.notFound(messages.getMessage(
                                        "errors.accounts.space.not.found", null, locale
                                ))).build()))
                )))
                .map(r -> r.andThen(accountSpace -> accountSpace.getProviderId().equals(providerId) ?
                        Result.success(accountSpace) :
                        Result.failure(ErrorCollection.builder().addError(
                                TypedError.notFound(messages.getMessage(
                                        "errors.wrong.provider.for.accounts.space", null, locale
                                ))).build())
                ));
    }


    public Mono<Result<ExpandedAccountsSpaces<Page<AccountSpaceModel>>>> getPageExpanded(
            String providerId,
            PageRequest pageRequest,
            YaUserDetails currentUser,
            Locale locale,
            boolean withDeleted
    ) {
        return getPage(providerId, pageRequest, currentUser, locale, withDeleted)
                .flatMap(r -> r.andThenMono(page -> accountsSpacesUtils.expandPage(page).map(Result::success)));
    }

    public Mono<Result<Page<AccountSpaceModel>>> getPage(
            String providerId,
            PageRequest pageRequest,
            YaUserDetails currentUser,
            Locale locale,
            boolean withDeleted
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService
                .checkReadPermissions(currentUser, locale).flatMap(result -> result.andThenMono(u -> providerValidator
                        .validateProvider(tenantId, providerId, locale))).map(r -> r.andThen(u -> pager
                        .validatePageRequest(pageRequest, locale)))
                .flatMap(r -> r.andThenMono(token -> tableClient.usingSessionMonoRetryable(session ->
                        accountsSpacesDao.getByProvider(
                                session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                                providerId,
                                tenantId,
                                token.getContinuationToken().map(StringContinuationToken::getId).orElse(null),
                                token.getLimit() + 1,
                                withDeleted
                        ).map(models -> pager.toPage(
                                models, token.getLimit(), AccountSpaceModel::getId
                        ))
                                .map(Result::success)
                )));
    }

    public Mono<ExpandedAccountsSpaces<AccountSpaceModel>> expand(AccountSpaceModel accountSpace) {
        List<Tuple2<String, TenantId>> segmentsIds = accountSpace.getSegments().stream().map(segment ->
                Tuples.of(segment.getSegmentId(), accountSpace.getTenantId())
        ).distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> segmentationsIds = accountSpace.getSegments().stream().map(segment ->
                Tuples.of(segment.getSegmentationId(), accountSpace.getTenantId())
        ).distinct().collect(Collectors.toList());
        return Mono.zip(
                resourceSegmentationsLoader.getResourceSegmentationsByIdsImmediate(segmentationsIds),
                resourceSegmentsLoader.getResourceSegmentsByIdsImmediate(segmentsIds)
        ).map(tuple -> new ExpandedAccountsSpaces<>(accountSpace,
                tuple.getT1().stream().collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity())),
                tuple.getT2().stream().collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity())))
        );
    }

    public Mono<Result<AccountSpaceModel>> create(
            AccountsSpaceCreateDto resource, String providerId, YaUserDetails currentUser, Locale locale
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        String newId = UUID.randomUUID().toString();
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                                validateCreation(
                                        resource, providerId, newId, currentUser, locale, ts, tenantId
                                ).flatMap(r -> r.andThenMono(validated ->
                                        accountsSpacesDao.upsertOneRetryable(ts, validated)
                                                .doOnSuccess(v -> accountSpacesLoader.refresh(
                                                        new ProviderId(validated.getProviderId()),
                                                        validated.getTenantId())
                                                )
                                                .thenReturn(Result.success(validated))
                                ))
                        )
                ));
    }

    public Mono<Result<AccountSpaceModel>> put(
            String id,
            String providerId,
            Long version,
            AccountsSpacePutDto accountsSpaceDto,
            YaUserDetails currentUser,
            Locale locale
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(
                v -> textValidator.validateId(id, locale, "errors.accounts.space.not.found")).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxRetryable(
                                ts -> accountsSpacesDao.getByIdStartTx(ts, id, tenantId),
                                (ts, r) -> validateExists(r.orElse(null), providerId, locale)
                                        .andThenMono(p -> validatePut(accountsSpaceDto, providerId, version,
                                                p, currentUser, locale, tenantId)),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return accountsSpacesDao.upsertOneRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> accountSpacesLoader
                                                                .refresh(new ProviderId(providerId), tenantId))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    public Mono<Result<ExpandedAccountsSpaces<SelectionTreeNode>>> getSelectionTree(
            String providerId,
            YaUserDetails currentUser,
            Locale locale
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return
            securityManagerService.checkReadPermissions(currentUser, locale).flatMap(result -> result.andThenMono(u ->
            providerValidator.validateProvider(tenantId, providerId, locale))).flatMap(r -> r.applyMono(provider ->
            accountSpacesLoader.getAllByProviderId(tenantId, providerId).flatMap(accountSpaces ->
            resourceSegmentationsLoader.getResourceSegmentationsByIdsImmediate(accountSpaces.stream()
                    .map(AccountSpaceModel::getSegments).flatMap(Collection::stream)
                    .map(ResourceSegmentSettingsModel::getSegmentationId)
                    .distinct()
                    .map(segmentationId -> Tuples.of(segmentationId, tenantId))
                    .collect(Collectors.toList())
            ).flatMap(segmentationModels -> {
                List<String> segmentationIdsInOrder = new ArrayList<>();
                List<String> segmentationNamesInOrder = new ArrayList<>();
                segmentationModels.stream()
                        .sorted(Comparator.comparingInt(ResourceSegmentationModel::getGroupingOrder))
                        .forEachOrdered(model -> {
                            segmentationIdsInOrder.add(model.getId());
                            segmentationNamesInOrder.add(Locales.select(model.getNameEn(), model.getNameRu(), locale));
                        });
                SelectionTreeNode selectionTree = groupToTree(0, accountSpaces, segmentationIdsInOrder,
                        segmentationNamesInOrder);
                return accountsSpacesUtils.expand(selectionTree, accountSpaces);
            }))));
    }

    private SelectionTreeNode groupToTree(
            int segmentationIdPosition,
            List<AccountSpaceModel> accountSpaces,
            List<String> segmentationIdsInOrder,
            List<String> segmentationNamesInOrder
    ) {
        if (accountSpaces == null || accountSpaces.isEmpty()) {
            return SelectionTreeNode.empty();
        }
        int nextSegmentationIdPosition = findNextSegmentationIdPosition(segmentationIdPosition, accountSpaces,
                segmentationIdsInOrder);
        if (nextSegmentationIdPosition < 0) {
            // todo: fall back to flatten list
            if (accountSpaces.size() == 1) {
                return SelectionTreeNode.builder().value(accountSpaces.get(0)).build();
            }
            return SelectionTreeNode.empty();
        }
        String segmentationId = segmentationIdsInOrder.get(nextSegmentationIdPosition);
        Function<AccountSpaceModel, String> segmentIdFunction = getSegmentIdFunction(segmentationId);
        Map<String, List<AccountSpaceModel>> collectedBySegmentId = accountSpaces.stream()
                .collect(Collectors.groupingBy(segmentIdFunction));
        Map<String, SelectionTreeNode> childrenBySegmentId = new HashMap<>(collectedBySegmentId.size());
        collectedBySegmentId.forEach((segmentId, spaces) ->
                childrenBySegmentId.put(segmentId, groupToTree(nextSegmentationIdPosition + 1, spaces,
                        segmentationIdsInOrder, segmentationNamesInOrder))
        );
        return SelectionTreeNode.builder()
                .segmentationName(segmentationNamesInOrder.get(nextSegmentationIdPosition))
                .childrenBySegmentId(childrenBySegmentId)
                .build();
    }

    private int findNextSegmentationIdPosition(
            int currentIndex, List<AccountSpaceModel> accountSpaces, List<String> segmentationIdsInOrder
    ) {
        for (int i = currentIndex; i < segmentationIdsInOrder.size(); i++) {
            String segmentationId = segmentationIdsInOrder.get(i);
            Function<AccountSpaceModel, String> segmentIdFunction = getSegmentIdFunction(segmentationId);
            if (accountSpaces.stream().allMatch(accountSpace -> segmentIdFunction.apply(accountSpace) != null)) {
                return i;
            }
        }
        return -1;
    }

    private Function<AccountSpaceModel, String> getSegmentIdFunction(String segmentationId) {
        return accountSpace -> accountSpace.getSegments().stream()
                .filter(segment -> segment.getSegmentationId().equals(segmentationId))
                .findFirst()
                .map(ResourceSegmentSettingsModel::getSegmentId)
                .orElse(null);
    }

    private Result<AccountSpaceModel> validateExists(
            AccountSpaceModel accountSpaceModel, String providerId, Locale locale
    ) {
        if (accountSpaceModel == null || accountSpaceModel.isDeleted()
                || !accountSpaceModel.getProviderId().equals(providerId)
        ) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.accounts.space.not.found", null, locale)))
                    .build());
        }
        return Result.success(accountSpaceModel);
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Result<Tuple2<AccountSpaceModel, Boolean>>> validatePut(
            AccountsSpacePutDto accountsSpace,
            String providerId,
            Long version,
            AccountSpaceModel existingAccountSpaceModel,
            YaUserDetails currentUser,
            Locale locale,
            TenantId tenantId
    ) {
        return providerValidator.validateProvider(tenantId, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        p.getId(), currentUser, locale, p
                ))
                .map(result -> result.andThen(provider -> {
                    AccountSpaceModel.Builder builder = AccountSpaceModel.newBuilder(existingAccountSpaceModel);
                    ErrorCollection.Builder errors = ErrorCollection.builder();
                    versionValidator.validate(() -> Optional.ofNullable(version), existingAccountSpaceModel::getVersion,
                            builder::setVersion, errors, "version", locale);
                    textValidator.with(errors, locale)
                            .validateAndSet(accountsSpace::getNameEn, builder::setNameEn, "nameEn", MAX_NAME_LENGTH)
                            .validateAndSet(accountsSpace::getNameRu, builder::setNameRu, "nameRu", MAX_NAME_LENGTH)
                            .validateAndSet(accountsSpace::getDescriptionEn, builder::setDescriptionEn, "descriptionEn",
                                    MAX_DESCRIPTION_LENGTH)
                            .validateAndSet(accountsSpace::getDescriptionRu, builder::setDescriptionRu, "descriptionRu",
                                    MAX_DESCRIPTION_LENGTH);
                    accountsSpace.getUiSettings().ifPresent(it -> builder.setUiSettings(it.toModel()));
                    builder.setSyncEnabled(accountsSpace.isSyncEnabled());
                    if (errors.hasAnyErrors()) {
                        return Result.failure(errors.build());
                    }
                    AccountSpaceModel accountSpaceModel = builder.build();
                    return Result.success(Tuples.of(
                            accountSpaceModel,
                            !accountSpaceModel.equals(existingAccountSpaceModel)
                    ));
                })));
    }

    public Mono<Result<AccountSpaceModel>> setReadOnly(
            String accountsSpaceId,
            String providerId,
            boolean readOnly,
            YaUserDetails currentUser,
            Locale locale
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(
                v -> textValidator.validateId(accountsSpaceId, locale, "errors.accounts.space.not.found"
                )).andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxRetryable(
                                ts -> accountsSpacesDao.getByIdStartTx(ts, accountsSpaceId, tenantId),
                                (ts, r) -> validateExists(r.orElse(null), providerId, locale).andThenMono(as ->
                                        providerValidator.validateProvider(
                                                tenantId, providerId, locale
                                        ).flatMap(pr -> pr.andThenMono(p ->
                                                securityManagerService.checkWritePermissionsForProvider(
                                                        p.getId(), currentUser, locale, as
                                                )
                                        ))
                                ),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.isReadOnly() != readOnly) {
                                                return accountsSpacesDao.setReadOnly(
                                                        ts, accountsSpaceId, tenantId, readOnly
                                                ).doOnSuccess(v -> accountSpacesLoader
                                                        .refresh(new ProviderId(providerId), tenantId))
                                                        .thenReturn(Result.success(m));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    private Mono<Result<AccountSpaceModel>> validateCreation(
            AccountsSpaceCreateDto accountsSpace,
            String providerId,
            String newId,
            YaUserDetails currentUser,
            Locale locale,
            YdbTxSession session,
            TenantId tenantId
    ) {
        return providerValidator.validateProvider(tenantId, providerId, locale)
                .map(result -> result.andThen(provider -> provider.isAccountsSpacesSupported() ?
                        Result.success(provider) :
                        Result.failure(ErrorCollection.builder().addError("key", TypedError.invalid(messages
                                .getMessage("errors.accounts.spaces.is.not.supported", null, locale)))
                                .build()))
                )
                .flatMap(result -> result.andThenMono(provider ->
                        securityManagerService.checkWritePermissionsForProvider(
                                providerId, currentUser, locale, provider
                        )))
                .flatMap(result -> result
                        .andThenMono(provider -> validateSegments(session, accountsSpace.getSegments(), tenantId,
                                provider.getId(), locale)
                                .flatMap(segmentsResult ->
                                        validateKey(tenantId,
                                                session, accountsSpace.getKey(), provider, locale
                                        ).map(keyResult -> validateAndBuild(
                                                accountsSpace, newId, locale, tenantId, provider.getId(),
                                                keyResult,
                                                segmentsResult
                                        ))
                                )
                        )
                );
    }

    @Nonnull
    private Result<AccountSpaceModel> validateAndBuild(
            AccountsSpaceCreateDto accountsSpace,
            String newId,
            Locale locale,
            TenantId tenantId,
            String providerId,
            Result<String> key,
            Result<Set<ResourceSegmentSettingsModel>> segments
    ) {
        AccountSpaceModel.Builder builder = AccountSpaceModel.newBuilder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        builder.setId(newId);
        builder.setTenantId(tenantId);
        builder.setDeleted(false);
        builder.setVersion(0L);
        builder.setReadOnly(accountsSpace.isReadOnly());
        builder.setSyncEnabled(accountsSpace.isSyncEnabled());
        builder.setProviderId(providerId);
        key.doOnSuccess(builder::setOuterKeyInProvider);
        key.doOnFailure(errors::add);
        textValidator.with(errors, locale)
                .validateAndSet(accountsSpace::getNameEn, builder::setNameEn, "nameEn", MAX_NAME_LENGTH)
                .validateAndSet(accountsSpace::getNameRu, builder::setNameRu, "nameRu", MAX_NAME_LENGTH)
                .validateAndSet(accountsSpace::getDescriptionEn, builder::setDescriptionEn, "descriptionEn",
                        MAX_DESCRIPTION_LENGTH)
                .validateAndSet(accountsSpace::getDescriptionRu, builder::setDescriptionRu, "descriptionRu",
                        MAX_DESCRIPTION_LENGTH);
        segments.doOnSuccess(builder::setSegments);
        segments.doOnFailure(errors::add);
        accountsSpace.getUiSettings().ifPresent(it -> builder.setUiSettings(it.toModel()));
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(builder.build());
    }

    private Mono<Result<String>> validateKey(
            TenantId tenantId, YdbTxSession session, Optional<String> key, ProviderModel provider, Locale locale
    ) {
        if (provider.getAccountsSettings().isKeySupported()) {
            if (key.isEmpty()) {
                ErrorCollection error = ErrorCollection.builder().addError("key", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)))
                        .build();
                return Mono.just(Result.failure(error));
            }
            if (key.get().isBlank()) {
                ErrorCollection error = ErrorCollection.builder().addError("key", TypedError.invalid(messages
                        .getMessage("errors.non.blank.text.is.required", null, locale)))
                        .build();
                return Mono.just(Result.failure(error));
            }
            if (key.get().length() > MAX_KEY_LENGTH) {
                ErrorCollection error = ErrorCollection.builder().addError("key", TypedError.invalid(messages
                        .getMessage("errors.text.is.too.long", null, locale)))
                        .build();
                return Mono.just(Result.failure(error));
            }
            return accountsSpacesDao.getByKey(
                    session, tenantId, provider.getId(), key.get()
            ).map(exists -> {
                if (exists.get().isPresent()) {
                    ErrorCollection error = ErrorCollection.builder()
                            .addError("key", TypedError.conflict(messages
                                    .getMessage("errors.accounts.space.already.exists", null, locale))
                            )
                            .addDetail("segments", Map.of(
                                    "exists_accounts_space_id", exists.get().get().getId()
                            ))
                            .build();
                    return Result.failure(error);
                }
                return Result.success(key.get());
            });
        } else {
            if (key.isEmpty()) {
                ErrorCollection error = ErrorCollection.builder().addError("key", TypedError.invalid(messages
                        .getMessage("errors.accounts.space.key.is.not.supported", null, locale)))
                        .build();
                return Mono.just(Result.failure(error));
            }
            return Mono.just(Result.success(""));
        }
    }

    private Mono<Result<Set<ResourceSegmentSettingsModel>>> validateSegments(
            YdbTxSession session,
            List<CreateResourceSegmentationSegmentDto> segmentsDto,
            TenantId tenantId,
            String providerId,
            Locale locale
    ) {
        return segmentationsValidator.validateSegmentations(session,
                () -> Optional.ofNullable(segmentsDto),
                CreateResourceSegmentationSegmentDto::getSegmentationId,
                CreateResourceSegmentationSegmentDto::getSegmentId,
                providerId,
                locale,
                "segments",
                true
        ).map(result -> result.apply(segments -> segments.stream().map(tuple ->
                new ResourceSegmentSettingsModel(tuple.getT1().getId(), tuple.getT2().getId()))
                .collect(Collectors.toSet()))
        ).flatMap(result -> result.andThenMono(segments -> accountSpacesLoader
                .getAccountSpaces(session, tenantId, providerId, segments).map(accountSpaceModels ->
                        accountSpaceModels.isEmpty() || accountSpaceModels.get().isEmpty() ?
                                Result.success(segments) :
                                Result.failure(ErrorCollection.builder()
                                        .addError("segments", TypedError.conflict(
                                                messages.getMessage("errors.accounts.space.already.exists", null,
                                                        locale))
                                        )
                                        .addDetail("segments", Map.of(
                                                "exists_accounts_space_id", accountSpaceModels.get().get(0).getId()
                                        ))
                                        .build())))
        );
    }
}
