package ru.yandex.intranet.d.services.resources;

import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.resources.AccountsSpaceDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceSegmentDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceSegmentationDto;
import ru.yandex.intranet.d.web.model.resources.InnerResourceSegmentationSegmentDto;
import ru.yandex.intranet.d.web.util.ModelDtoConverter;

/**
 * AccountsSpacesUtils
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@Component
public class AccountsSpacesUtils {
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final ResourceSegmentsLoader resourceSegmentsLoader;
    private final AccountSpacesLoader accountSpacesLoader;

    public AccountsSpacesUtils(ResourceSegmentationsLoader resourceSegmentationsLoader,
                               ResourceSegmentsLoader resourceSegmentsLoader,
                               AccountSpacesLoader accountSpacesLoader
    ) {
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.resourceSegmentsLoader = resourceSegmentsLoader;
        this.accountSpacesLoader = accountSpacesLoader;
    }

    public Mono<ExpandedAccountsSpaces<List<AccountSpaceModel>>> getExpandedByProviderIds(
            YdbTxSession session,
            Set<WithTenant<String>> providerIds) {

        return accountSpacesLoader.getAccountSpaces(session, providerIds)
                .flatMap(this::expandCollection);
    }

    public <T extends Page<AccountSpaceModel>> Mono<ExpandedAccountsSpaces<T>> expandPage(T page) {
        return expand(page, page.getItems());
    }

    public <T extends Collection<AccountSpaceModel>> Mono<ExpandedAccountsSpaces<T>> expandCollection(T collection) {
        return expand(collection, collection);
    }

    public <T extends Collection<AccountSpaceModel>> List<AccountsSpaceDto> toDtos(
            ExpandedAccountsSpaces<T> em, Locale locale) {
        return toDtos(em, em.getAccountsSpaces(), locale);
    }

    public <T extends Page<AccountSpaceModel>> List<AccountsSpaceDto> toDtosFromPage(
            ExpandedAccountsSpaces<T> em, Locale locale) {
        return toDtos(em, em.getAccountsSpaces().getItems(), locale);
    }

    public <T> Mono<ExpandedAccountsSpaces<T>> expand(T collection, Collection<AccountSpaceModel> models) {
        List<Tuple2<String, TenantId>> segmentsIds = models.stream().flatMap(accountSpace ->
                accountSpace.getSegments().stream().map(segment ->
                        Tuples.of(segment.getSegmentId(), accountSpace.getTenantId())
                )).distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> segmentationsIds = models.stream().flatMap(accountSpace ->
                accountSpace.getSegments().stream().map(segment ->
                        Tuples.of(segment.getSegmentationId(), accountSpace.getTenantId())
                )).distinct().collect(Collectors.toList());
        return Mono.zip(
                resourceSegmentationsLoader.getResourceSegmentationsByIdsImmediate(segmentationsIds),
                resourceSegmentsLoader.getResourceSegmentsByIdsImmediate(segmentsIds)
        ).map(tuple -> new ExpandedAccountsSpaces<>(collection,
                tuple.getT1().stream().collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity())),
                tuple.getT2().stream().collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity())))
        );
    }

    public AccountsSpaceDto toDto(
            AccountSpaceModel model,
            Map<String, ResourceSegmentationModel> resourceSegmentations,
            Map<String, ResourceSegmentModel> resourceSegments,
            Locale locale
    ) {
        return new AccountsSpaceDto(
                model.getId(),
                model.getOuterKeyInProvider(),
                model.getProviderId(),
                model.getVersion(),
                Locales.select(model.getNameEn(), model.getNameRu(), locale),
                Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale),
                model.isReadOnly(), // readOnly
                toSegmentations(model, resourceSegmentations, resourceSegments, locale),
                model.getUiSettings().map(ProviderUISettingsDto::new).orElse(null),
                model.isSyncEnabled()
        );
    }

    private Set<InnerResourceSegmentationSegmentDto> toSegmentations(
            AccountSpaceModel model,
            Map<String, ResourceSegmentationModel> resourceSegmentations,
            Map<String, ResourceSegmentModel> resourceSegments,
            Locale locale
    ) {
        return model.getSegments().stream().map(s -> {
            InnerResourceSegmentationDto segmentation = ModelDtoConverter.toResourceSegmentation(
                    resourceSegmentations.get(s.getSegmentationId()), locale);
            InnerResourceSegmentDto segment = ModelDtoConverter.toResourceSegment(
                    resourceSegments.get(s.getSegmentId()), locale);
            return new InnerResourceSegmentationSegmentDto(segmentation, segment);
        }).collect(Collectors.toSet());
    }

    private <T> List<AccountsSpaceDto> toDtos(ExpandedAccountsSpaces<T> em,
                                             Collection<AccountSpaceModel> models, Locale locale) {
        return models.stream()
                .map(m -> toDto(m, em.getSegmentations(), em.getSegments(), locale))
                .collect(Collectors.toList());
    }

}
