package ru.yandex.intranet.d.services.resources;

import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuples;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.accounts.ProviderReserveAccountsDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.ProviderReserveAccountKey;
import ru.yandex.intranet.d.model.accounts.ProviderReserveAccountModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.validators.ProviderValidator;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static com.yandex.ydb.table.transaction.TransactionMode.STALE_READ_ONLY;
import static ru.yandex.intranet.d.util.Util.isEmpty;

/**
 * Resource tree service.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class ResourceTreeService {
    private final SecurityManagerService securityManagerService;
    private final ProviderValidator providerValidator;
    private final ResourcesService resourcesService;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final ResourceUtils resourceUtils;
    private final YdbTableClient tableClient;
    private final AccountsQuotasDao accountsQuotasDao;
    private final AccountsDao accountsDao;
    private final FolderDao folderDao;
    private final ResourceTreeUtils resourceTreeUtils;
    private final ProviderReserveAccountsDao providerReserveAccountsDao;
    private final QuotasDao quotasDao;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public ResourceTreeService(
            SecurityManagerService securityManagerService,
            ProviderValidator providerValidator,
            ResourcesService resourcesService,
            ResourceSegmentationsLoader resourceSegmentationsLoader,
            ResourceUtils resourceUtils,
            YdbTableClient tableClient,
            AccountsQuotasDao accountsQuotasDao,
            AccountsDao accountsDao,
            FolderDao folderDao,
            ResourceTreeUtils resourceTreeUtils,
            ProviderReserveAccountsDao providerReserveAccountsDao,
            QuotasDao quotasDao) {
        this.securityManagerService = securityManagerService;
        this.providerValidator = providerValidator;
        this.resourcesService = resourcesService;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.resourceUtils = resourceUtils;
        this.tableClient = tableClient;
        this.accountsQuotasDao = accountsQuotasDao;
        this.accountsDao = accountsDao;
        this.folderDao = folderDao;
        this.resourceTreeUtils = resourceTreeUtils;
        this.providerReserveAccountsDao = providerReserveAccountsDao;
        this.quotasDao = quotasDao;
    }

    public Mono<Result<ExpandedResources<SelectionResourceTreeNode>>> getResourceSelectionTree(
            String providerId,
            ResourceFilterParameters resourceFilterParameters,
            ByAccountsFilterParameters byAccountsFilterParameters,
            YaUserDetails currentUser,
            Locale locale
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return toPredicate(byAccountsFilterParameters, providerId, tenantId).flatMap(byAccountsFilterPredicate ->
            securityManagerService.checkReadPermissions(currentUser, locale).flatMap(r -> r.andThenMono(u ->
                providerValidator.validateProvider(tenantId, providerId,
                        locale))).flatMap(r -> r.andThenMono(provider ->
                resourcesService.getAllProviderResources(currentUser, providerId,
                        locale).flatMap(r1 -> r1.apply(resourceModels -> resourceModels.stream()
                        .filter(resourceFilterParameters.getPredicate())
                        .filter(byAccountsFilterPredicate)
                        .collect(Collectors.toList())).applyMono(resources ->
                        resourceSegmentationsLoader.getResourceSegmentationsByIdsImmediate(resources.stream()
                                .flatMap(resourceModel -> resourceModel.getSegments().stream())
                                .map(ResourceSegmentSettingsModel::getSegmentationId)
                                .distinct()
                                .map(segmentationId -> Tuples.of(segmentationId, tenantId))
                                .collect(Collectors.toList())
                        ).flatMap(segmentationModels -> {
                            String[] segmentationIdsInOrder = segmentationModels.stream()
                                    .sorted(Comparator.comparingInt(ResourceSegmentationModel::getGroupingOrder)
                                            .thenComparing(ResourceSegmentationModel::getId))
                                    .map(ResourceSegmentationModel::getId).toArray(String[]::new);
                            SelectionResourceTreeNode selectionResourceTreeNode =
                                    resourceTreeUtils.groupToResourceTree(resources, segmentationIdsInOrder, 0);
                            return resourceUtils.expand(
                                    selectionResourceTreeNode, resources, true, false);
                        })
                )))));
    }

    private Mono<Predicate<ResourceModel>> toPredicate(ByAccountsFilterParameters p, String providerId,
                                                       TenantId tenantId) {
        if (!isEmpty(p.getAccountIdForFilter())) {
            return toPredicate(Set.of(p.getAccountIdForFilter()), tenantId);
        } else if (!isEmpty(p.getFolderIdForFilter())) {
            if (p.getBalanceOnly()) {
                return toFolderBalanceOnlyPredicate(tenantId, p.getFolderIdForFilter(), providerId);
            } else {
                return tableClient.usingSessionMonoRetryable(session ->
                        accountsDao.getAllByFolderIds(session.asTxCommitRetryable(STALE_READ_ONLY),
                                List.of(new WithTenant<>(tenantId, p.getFolderIdForFilter())), false
                        )
                ).flatMap(accountModels -> p.getProviderReserveOnly() ?
                        toPredicateProviderReserve(providerId, tenantId) :
                        toPredicate(accountModels.stream().map(AccountModel::getId).collect(Collectors.toSet()),
                                tenantId)
                );
            }
        } else if (!isEmpty(p.getServiceIdForFilter())) {
            if (p.getBalanceOnly()) {
                return toServiceBalanceOnlyPredicate(tenantId, p.getServiceIdForFilter(), providerId,
                        p.getProviderReserveOnly());
            } else {
                return tableClient.usingSessionMonoRetryable(session ->
                    folderDao.getAllFoldersByServiceIds(session.asTxCommitRetryable(STALE_READ_ONLY),
                            Set.of(new WithTenant<>(tenantId, p.getServiceIdForFilter()))
                    )
                    .map(listWithTxId -> listWithTxId.get().stream()
                            .map(folder -> new WithTenant<>(folder.getTenantId(), folder.getId())).toList()
                    ).flatMap(folderIds -> accountsDao.getAllByFolderIds(
                            session.asTxCommitRetryable(STALE_READ_ONLY), folderIds, false)
                    ).flatMap(accountModels -> p.getProviderReserveOnly() ?
                        toPredicateProviderReserve(providerId, tenantId) :
                        toPredicate(accountModels.stream().map(AccountModel::getId).collect(Collectors.toSet()),
                                tenantId)
                    )
                );
            }
        }
        return Mono.just(ignored -> true);
    }

    private Mono<Predicate<ResourceModel>> toFolderBalanceOnlyPredicate(TenantId tenantId,
                                                                        String folderId,
                                                                        String providerId) {
        return tableClient.usingSessionMonoRetryable(session -> quotasDao.getByFoldersAndProvider(
                session.asTxCommitRetryable(STALE_READ_ONLY), List.of(folderId), tenantId, providerId, false)
                .map(quotas -> {
                    Set<String> resourceIds = quotas.stream()
                            .filter(q -> q.getBalance() > 0)
                            .map(QuotaModel::getResourceId)
                            .collect(Collectors.toUnmodifiableSet());
                    return resource -> resourceIds.contains(resource.getId());
                })
        );
    }

    private Mono<Predicate<ResourceModel>> toServiceBalanceOnlyPredicate(TenantId tenantId,
                                                                         long serviceId,
                                                                         String providerId,
                                                                         boolean providerReserveOnly) {
        return tableClient.usingSessionMonoRetryable(session ->
                folderDao.getAllFoldersByServiceIds(session.asTxCommitRetryable(STALE_READ_ONLY),
                        Set.of(new WithTenant<>(tenantId, serviceId)))
                        .map(l -> l.get().stream()
                                .filter(f -> !providerReserveOnly || f.getFolderType() == FolderType.PROVIDER_RESERVE)
                                .map(FolderModel::getId).toList())
                        .flatMap(folders -> {
                            if (folders.isEmpty()) {
                                return Mono.just(List.<QuotaModel>of());
                            }
                            return quotasDao.getByFoldersAndProvider(session.asTxCommitRetryable(STALE_READ_ONLY),
                                    folders, tenantId, providerId, false);
                        })
                        .map(quotas -> {
                            Set<String> resourceIds = quotas.stream()
                                    .filter(q -> q.getBalance() > 0)
                                    .map(QuotaModel::getResourceId)
                                    .collect(Collectors.toUnmodifiableSet());
                            return resource -> resourceIds.contains(resource.getId());
                        })
        );
    }

    private Mono<Predicate<ResourceModel>> toPredicate(Set<String> accountIds, TenantId tenantId) {
        return tableClient.usingSessionMonoRetryable((YdbSession session) ->
                        accountsQuotasDao.getAllByAccountIds(
                                session.asTxCommitRetryable(STALE_READ_ONLY), tenantId, accountIds
                        )
                ).map(quotas -> {
                    Set<String> resourceIds = quotas.stream()
                            .filter(q -> q.getProvidedQuota() > 0)
                            .map(AccountsQuotasModel::getResourceId)
                            .collect(Collectors.toUnmodifiableSet());
                    return resource -> resourceIds.contains(resource.getId());
                });
    }

    private Mono<Predicate<ResourceModel>> toPredicateProviderReserve(String providerId, TenantId tenantId) {
        return tableClient.usingSessionMonoRetryable((YdbSession session) ->
                providerReserveAccountsDao.getAllByProviderMono(
                        session.asTxCommitRetryable(STALE_READ_ONLY), tenantId, providerId)
                .flatMap(reserveAccounts -> {
                    Set<String> providerReserveAccountIds = reserveAccounts.stream()
                            .map(ProviderReserveAccountModel::getKey)
                            .map(ProviderReserveAccountKey::getAccountId)
                            .collect(Collectors.toSet());
                    return accountsQuotasDao.getAllByAccountIds(
                            session.asTxCommitRetryable(STALE_READ_ONLY), tenantId, providerReserveAccountIds
                    ).map(providerReserveQuotas -> {
                        Set<String> providerResourceIds = providerReserveQuotas.stream()
                                .filter(q -> q.getProvidedQuota() > 0)
                                .map(AccountsQuotasModel::getResourceId)
                                .collect(Collectors.toUnmodifiableSet());
                        return resource -> providerResourceIds.contains(resource.getId());
                    });
                })
        );
    }

}
