package ru.yandex.intranet.d.services.resources;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;

/**
 * Resource utils.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class ResourceUtils {
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final ResourceTypesLoader resourceTypesLoader;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final ResourceSegmentsLoader resourceSegmentsLoader;

    public ResourceUtils(UnitsEnsemblesLoader unitsEnsemblesLoader, ResourceTypesLoader resourceTypesLoader,
                         ResourceSegmentationsLoader resourceSegmentationsLoader,
                         ResourceSegmentsLoader resourceSegmentsLoader) {
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.resourceTypesLoader = resourceTypesLoader;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.resourceSegmentsLoader = resourceSegmentsLoader;
    }


    public <T> Mono<ExpandedResources<T>> expand(T collection, ResourceModel resource, boolean withSegmentations,
                                                 boolean directoryEndpoint) {
        if (directoryEndpoint) {
            return Mono.just(new ExpandedResources<>(collection, Map.of(), Map.of(), Map.of(), Map.of()));
        }
        if (!withSegmentations) {
            List<Tuple2<String, TenantId>> unitsEnsembleKeys = List.of(Tuples.of(resource.getUnitsEnsembleId(),
                    resource.getTenantId()));
            return loadUnitsEnsembles(unitsEnsembleKeys)
                    .map(t -> new ExpandedResources<>(collection, Map.of(), Map.of(), Map.of(), t));
        }
        List<Tuple2<String, TenantId>> resourceTypeKeys = Optional.ofNullable(resource.getResourceTypeId())
                .stream().map(k -> Tuples.of(k, resource.getTenantId())).collect(Collectors.toList());
        List<Tuple2<String, TenantId>> segmentationKeys = Optional.ofNullable(resource.getSegments())
                .stream().flatMap(Collection::stream).map(ResourceSegmentSettingsModel::getSegmentationId)
                .distinct().map(k -> Tuples.of(k, resource.getTenantId())).collect(Collectors.toList());
        List<Tuple2<String, TenantId>> segmentKeys = Optional.ofNullable(resource.getSegments())
                .stream().flatMap(Collection::stream).map(ResourceSegmentSettingsModel::getSegmentId)
                .distinct().map(k -> Tuples.of(k, resource.getTenantId())).collect(Collectors.toList());
        List<Tuple2<String, TenantId>> unitsEnsembleKeys = List.of(Tuples.of(resource.getUnitsEnsembleId(),
                resource.getTenantId()));
        return Mono.zip(loadResourceTypes(resourceTypeKeys), loadResourceSegmentations(segmentationKeys),
                        loadResourceSegments(segmentKeys), loadUnitsEnsembles(unitsEnsembleKeys))
                .map(t -> new ExpandedResources<>(collection, t.getT1(), t.getT2(), t.getT3(), t.getT4()));
    }

    public <T> Mono<ExpandedResources<T>> expand(T collection, List<ResourceModel> resources,
                                                 boolean withSegmentations,
                                                 boolean directoryEndpoint) {
        if (directoryEndpoint) {
            return Mono.just(new ExpandedResources<>(collection, Map.of(), Map.of(), Map.of(), Map.of()));
        }
        if (!withSegmentations) {
            List<Tuple2<String, TenantId>> unitsEnsembleKeys = resources.stream()
                    .map(i -> Tuples.of(i.getUnitsEnsembleId(), i.getTenantId()))
                    .distinct().collect(Collectors.toList());
            return loadUnitsEnsembles(unitsEnsembleKeys)
                    .map(t -> new ExpandedResources<>(collection, Map.of(), Map.of(), Map.of(), t));
        }
        List<Tuple2<String, TenantId>> resourceTypeKeys = resources.stream()
                .flatMap(i -> Optional.ofNullable(i.getResourceTypeId())
                        .stream().map(k -> Tuples.of(k, i.getTenantId())))
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> segmentationKeys = resources.stream()
                .flatMap(i -> Optional.ofNullable(i.getSegments())
                        .stream().flatMap(m -> m.stream()
                                .map(ResourceSegmentSettingsModel::getSegmentationId))
                        .map(k -> Tuples.of(k, i.getTenantId())))
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> segmentKeys = resources.stream()
                .flatMap(i -> Optional.ofNullable(i.getSegments())
                        .stream().flatMap(m -> m.stream()
                                .map(ResourceSegmentSettingsModel::getSegmentId))
                        .map(k -> Tuples.of(k, i.getTenantId())))
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> unitsEnsembleKeys = resources.stream()
                .map(i -> Tuples.of(i.getUnitsEnsembleId(), i.getTenantId()))
                .distinct().collect(Collectors.toList());
        return Mono.zip(loadResourceTypes(resourceTypeKeys), loadResourceSegmentations(segmentationKeys),
                        loadResourceSegments(segmentKeys), loadUnitsEnsembles(unitsEnsembleKeys))
                .map(t -> new ExpandedResources<>(collection, t.getT1(), t.getT2(), t.getT3(), t.getT4()));
    }

    private Mono<Map<String, ResourceTypeModel>> loadResourceTypes(List<Tuple2<String, TenantId>> resourceTypeKeys) {
        return resourceTypesLoader.getResourceTypesByIdsImmediate(resourceTypeKeys)
                .map(types -> types.stream().collect(Collectors.toMap(ResourceTypeModel::getId, t -> t)));
    }

    private Mono<Map<String, ResourceSegmentationModel>> loadResourceSegmentations(
            List<Tuple2<String, TenantId>> resourceSegmentationKeys) {
        return resourceSegmentationsLoader.getResourceSegmentationsByIdsImmediate(resourceSegmentationKeys)
                .map(types -> types.stream().collect(Collectors.toMap(ResourceSegmentationModel::getId, t -> t)));
    }

    private Mono<Map<String, ResourceSegmentModel>> loadResourceSegments(
            List<Tuple2<String, TenantId>> resourceSegmentKeys) {
        return resourceSegmentsLoader.getResourceSegmentsByIdsImmediate(resourceSegmentKeys)
                .map(types -> types.stream().collect(Collectors.toMap(ResourceSegmentModel::getId, t -> t)));
    }

    private Mono<Map<String, UnitsEnsembleModel>> loadUnitsEnsembles(
            List<Tuple2<String, TenantId>> unitsEnsembleKeys) {
        return unitsEnsemblesLoader.getUnitsEnsemblesByIdsImmediate(unitsEnsembleKeys)
                .map(types -> types.stream().collect(Collectors.toMap(UnitsEnsembleModel::getId, t -> t)));
    }
}
