package ru.yandex.intranet.d.services.resources

import kotlinx.coroutines.reactor.awaitSingle
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.resources.types.ExchangeableResourceTypesDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.model.resources.types.ExchangeableResourceTypeModel
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.util.Uuids
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.providers.FrontGetProvidersForExchangeResponseDto
import ru.yandex.intranet.d.web.model.resources.exchangeable.FrontExchangeableResourceTypesDto
import ru.yandex.intranet.d.web.model.resources.exchangeable.FrontGetExchangeableResourcesResponseDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import ru.yandex.intranet.d.web.util.ModelDtoConverter
import java.util.*

@Component
class ResourcesExchangeService(
    private val tableClient: YdbTableClient,
    private val securityManagerService: SecurityManagerService,
    private val exchangeableResourceTypesDao: ExchangeableResourceTypesDao,
    private val providersLoader: ProvidersLoader,
    @Qualifier("messageSource") private val messages: MessageSource
) {
    fun getExchangeableResourceTypesByProviderMono(
        providerId: ProviderId?, userDetails: YaUserDetails, locale: Locale
    ): Mono<Result<FrontGetExchangeableResourcesResponseDto>> = mono {
        getExchangeableResourceTypesByProvider(providerId, userDetails, locale)
    }

    fun getProvidersForResourcesExchangeMono(
        userDetails: YaUserDetails, locale: Locale
    ): Mono<Result<FrontGetProvidersForExchangeResponseDto>> = mono {
        getProvidersForResourcesExchange(userDetails, locale)
    }

    private suspend fun getExchangeableResourceTypesByProvider(
        providerId: ProviderId?, userDetails: YaUserDetails, locale: Locale
    ): Result<FrontGetExchangeableResourcesResponseDto> = binding {
        securityManagerService.checkReadPermissions(userDetails, locale).awaitSingle().bind()
        validateProviderId(providerId, locale).bind()
        val models = loadExchangeableResourceTypesByProvider(providerId!!)
            .filter { it.key.fromResourceTypeId == it.key.toResourceTypeId } // MVP
        val dtos = models.map { FrontExchangeableResourceTypesDto(it) }.toSet()
        return Result.success(FrontGetExchangeableResourcesResponseDto(dtos))
    }

    private suspend fun getProvidersForResourcesExchange(
        userDetails: YaUserDetails, locale: Locale
    ): Result<FrontGetProvidersForExchangeResponseDto> = binding {
        securityManagerService.checkReadPermissions(userDetails, locale).awaitSingle().bind()
        val ids = loadProviderIdsResourcesExchange()
        val models = loadProviders(ids)
        val dtos = models.map { ModelDtoConverter.toProvider(it, locale) }.associateBy { it.id }
        return Result.success(FrontGetProvidersForExchangeResponseDto(dtos))
    }

    private fun validateProviderId(providerId: ProviderId?, locale: Locale): Result<ProviderId> {
        val errors = ErrorCollection.builder()
        if (providerId == null) {
            errors.addError(
                "providerId", TypedError.invalid(
                    messages.getMessage("errors.field.is.required", null, locale)
                )
            )
        }
        if (!Uuids.isValidUuid(providerId)) {
            errors.addError(
                "providerId", TypedError.invalid(
                    messages.getMessage("errors.provider.not.found", null, locale)
                )
            )
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(providerId!!)
    }

    private suspend fun loadExchangeableResourceTypesByProvider(
        providerId: ProviderId
    ): List<ExchangeableResourceTypeModel> = dbSessionRetryable(tableClient) {
        exchangeableResourceTypesDao.getByProvider(
            roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, providerId
        )
    }!!

    private suspend fun loadProviderIdsResourcesExchange(): List<ProviderId> = dbSessionRetryable(tableClient) {
        exchangeableResourceTypesDao.getProviders(roStaleSingleRetryableCommit())
    }!!.distinct()

    private suspend fun loadProviders(ids: List<ProviderId>) =
        providersLoader.getProvidersByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }).awaitSingle()
}
