package ru.yandex.intranet.d.services.resources;

import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.resources.types.ResourceTypesDao;
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.ResourceUnitsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.validators.SegmentationsValidator;
import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.util.FeatureStateHelper;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.resources.directory.CreateResourceSegmentationSegmentDto;
import ru.yandex.intranet.d.web.model.resources.directory.CreateResourceTypeSegmentsDto;
import ru.yandex.intranet.d.web.model.resources.directory.ResourceCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.ResourcePutDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.services.units.UnitsComparator.getBaseUnit;
import static ru.yandex.intranet.d.util.Util.isEmpty;

/**
 * Resources service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourcesService {

    private static final int MAX_KEY_LENGTH = 256;
    private static final int MAX_NAME_LENGTH = 256;
    private static final int MAX_DESCRIPTION_LENGTH = 1024;
    public static final String ERROR_MESSAGE = "errors.resource.with.non.unique.combination.of.resource.type.and" +
            ".segments";

    private final ResourcesDao resourcesDao;
    private final ProvidersDao providersDao;
    private final UnitsEnsemblesDao unitsEnsemblesDao;
    private final ResourceTypesDao resourceTypesDao;
    private final YdbTableClient tableClient;
    private final ResourcesLoader resourcesLoader;
    private final ProvidersLoader providersLoader;
    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final ObjectReader continuationTokenReader;
    private final ObjectWriter continuationTokenWriter;
    private final AccountSpacesLoader accountSpacesLoader;
    private final SegmentationsValidator segmentationsValidator;
    private final ResourceUtils resourceUtils;

    @SuppressWarnings("ParameterNumber")
    public ResourcesService(ResourcesDao resourcesDao,
                            ProvidersDao providersDao,
                            UnitsEnsemblesDao unitsEnsemblesDao,
                            ResourceTypesDao resourceTypesDao,
                            YdbTableClient tableClient,
                            ResourcesLoader resourcesLoader,
                            ProvidersLoader providersLoader,
                            @Qualifier("messageSource") MessageSource messages,
                            SecurityManagerService securityManagerService,
                            @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
                            AccountSpacesLoader accountSpacesLoader,
                            SegmentationsValidator segmentationsValidator,
                            ResourceUtils resourceUtils) {
        this.resourcesDao = resourcesDao;
        this.providersDao = providersDao;
        this.unitsEnsemblesDao = unitsEnsemblesDao;
        this.resourceTypesDao = resourceTypesDao;
        this.tableClient = tableClient;
        this.resourcesLoader = resourcesLoader;
        this.providersLoader = providersLoader;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.continuationTokenReader = objectMapper.getObjectMapper().readerFor(ResourceContinuationToken.class);
        this.continuationTokenWriter = objectMapper.getObjectMapper().writerFor(ResourceContinuationToken.class);
        this.accountSpacesLoader = accountSpacesLoader;
        this.segmentationsValidator = segmentationsValidator;
        this.resourceUtils = resourceUtils;
    }

    public Mono<Result<ExpandedResources<ResourceModel>>> getById(String id,
                                                                  String providerId,
                                                                  YaUserDetails currentUser,
                                                                  Locale locale,
                                                                  boolean directoryEndpoint,
                                                                  boolean withSegmentations) {
        return checkDirectoryEndpointPermissions(currentUser, locale, directoryEndpoint).andThenMono(
                v -> securityManagerService.checkReadPermissions(currentUser, locale)
        ).flatMap(result -> result
                .andThen(v -> validateId(id, locale))
                .andThenMono(u -> validateProvider(providerId, locale)
                        .map(r -> r.andThen(p -> checkDirectoryEndpointPermissions(
                                currentUser, locale, directoryEndpoint, p
                        )))
                        .flatMap(r -> r.andThenMono(pr ->
                                tableClient.usingSessionMonoRetryable(session -> resourcesDao
                                        .getById(immediateTx(session), id, Tenants.DEFAULT_TENANT_ID)
                                        .flatMap(t -> validateExists(t.orElse(null), pr.getId(), locale)
                                                .applyMono(x -> resourceUtils.expand(
                                                        x, x, withSegmentations, directoryEndpoint)))
                                )
                        ))
                ));
    }

    private Result<Void> checkDirectoryEndpointPermissions(
            YaUserDetails currentUser, Locale locale, boolean directoryEndpoint
    ) {
        return checkDirectoryEndpointPermissions(currentUser, locale, directoryEndpoint, null);
    }

    private <T> Result<T> checkDirectoryEndpointPermissions(
            YaUserDetails currentUser, Locale locale, boolean directoryEndpoint, T payload
    ) {
        return directoryEndpoint ?
                securityManagerService.checkDirectoryEndpointPermissions(currentUser, locale, payload) :
                Result.success(payload);
    }

    public Mono<Result<ExpandedResources<ResourceModel>>> getByIdFront(String id,
                                                                       YaUserDetails currentUser,
                                                                       Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(r -> r
                .andThen(v -> validateId(id, locale))
                .andThenMono(u -> tableClient.usingSessionMonoRetryable(session -> resourcesDao
                                .getById(immediateTx(session), id, Tenants.DEFAULT_TENANT_ID)
                                .flatMap(t -> validateExists(t.orElse(null),
                                        t.map(ResourceModel::getProviderId).orElse(null), locale)
                                        .applyMono(x -> resourceUtils.expand(x, x, true, false)))
                        )
                ));
    }

    public Mono<Result<ExpandedResources<Page<ResourceModel>>>> getPage(String providerId,
                                                                        PageRequest pageRequest,
                                                                        YaUserDetails currentUser,
                                                                        Locale locale,
                                                                        boolean directoryEndpoint,
                                                                        boolean withSegmentations) {
        return checkDirectoryEndpointPermissions(currentUser, locale, directoryEndpoint).andThenMono(
                v -> securityManagerService.checkReadPermissions(currentUser, locale)
        ).flatMap(result -> result
                .andThenMono(u -> {
                    Result<PageRequest.Validated<ResourceContinuationToken>> pageValidation
                            = pageRequest.validate(continuationTokenReader, messages, locale);
                    return pageValidation.andThenDo(p -> validateContinuationToken(p, locale)).andThenMono(p -> {
                        int limit = p.getLimit();
                        String fromId = p.getContinuationToken().map(ResourceContinuationToken::getId).orElse(null);
                        return validateProvider(providerId, locale)
                                .map(r -> r.andThen(pr -> checkDirectoryEndpointPermissions(
                                        currentUser, locale, directoryEndpoint, pr
                                )))
                                .flatMap(r -> r.andThenMono(pr -> tableClient.usingSessionMonoRetryable(session ->
                                        resourcesDao.getByProvider(immediateTx(session), pr.getId(),
                                                        Tenants.DEFAULT_TENANT_ID, fromId, limit + 1, false)
                                                .map(values -> values.size() > limit
                                                        ? Page.page(values.subList(0, limit),
                                                        prepareToken(values.get(limit - 1)))
                                                        : Page.lastPage(values))
                                                .flatMap(x -> resourceUtils.expand(x, x.getItems(), withSegmentations,
                                                        directoryEndpoint))
                                                .map(Result::success))));
                    });
                }));
    }

    @SuppressWarnings("checkstyle:ParameterNumber")
    public Mono<Result<ExpandedResources<Page<ResourceModel>>>> getPageByProviderTypeAndSegments(
            String providerId,
            String resourceTypeId,
            Set<String> segmentsIds,
            PageRequest pageRequest,
            YaUserDetails currentUser,
            Locale locale,
            boolean withSegmentations,
            boolean directoryEndpoint
    ) {
        return checkDirectoryEndpointPermissions(currentUser, locale, directoryEndpoint)
            .andThen(v -> validateResourceTypeId(resourceTypeId, locale))
            .andThen(v -> validateSegmentIds(segmentsIds, locale))
            .andThenMono(v -> securityManagerService.checkReadPermissions(currentUser, locale))
            .flatMap(result -> result
            .andThenMono(u -> {
                Result<PageRequest.Validated<ResourceContinuationToken>> pageValidation
                        = pageRequest.validate(continuationTokenReader, messages, locale);
                return pageValidation.andThenDo(p -> validateContinuationToken(p, locale)).andThenMono(p -> {
                    int limit = p.getLimit();
                    String fromId = p.getContinuationToken().map(ResourceContinuationToken::getId).orElse(null);
                    return validateProvider(providerId, locale)
                        .map(r -> r.andThen(pr -> checkDirectoryEndpointPermissions(
                            currentUser, locale, directoryEndpoint, pr)))
                        .flatMap(r -> r.andThenMono(pr -> tableClient.usingSessionMonoRetryable(session ->
                            resourcesDao.getByProviderResourceTypeAndSegments(
                                immediateTx(session), pr.getTenantId(), pr.getId(), resourceTypeId,
                                segmentsIds, fromId, limit + 1, false)
                        .map(values -> values.size() > limit ?
                            Page.page(values.subList(0, limit), prepareToken(values.get(limit - 1))) :
                            Page.lastPage(values))
                        .flatMap(x -> resourceUtils.expand(x, x.getItems(), withSegmentations,
                            directoryEndpoint))
                        .map(Result::success))));
                });
            }));
    }

    public Mono<Result<ResourceModel>> create(ResourceCreateDto resource, String providerId,
                                              YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThenMono(u -> {
            String newId = UUID.randomUUID().toString();
            return tableClient.usingSessionMonoRetryable(session ->
                    session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                            validateCreation(resource, providerId, newId, currentUser, locale, ts)
                                    .flatMap(r -> r.applyMono(validated ->
                                            setProviderHasDefaultQuotas(
                                                    ts,
                                                    validated.getT1().getTenantId(),
                                                    providerId,
                                                    validated.getT1().getId(),
                                                    hasDefaultQuota(validated.getT1())
                                            )).map(ignored -> r))
                                    .flatMap(r -> r.andThenMono(validated ->
                                                    resourcesDao.upsertResourceRetryable(ts, validated.getT1())
                                                            .doOnSuccess(v -> resourcesLoader.update(validated.getT1()))
                                                            .thenReturn(Result.success(validated))
                                            )
                                    )
                                    .flatMap(r -> r
                                            .apply(Tuple2::getT2)
                                            .applyMono(loadedR -> updateResourceType(loadedR, ts))
                                            .thenReturn(r.apply(Tuple2::getT1)))
                    )
            );
        });
    }

    private Mono<Void> updateResourceType(ValidatedCreateFields loadedR, YdbTxSession ts) {
        if (loadedR.getSegments().isEmpty()) {
            return Mono.empty();
        }
        ResourceTypeModel resourceType = loadedR.getSegments().get().getResourceType()
                .match(Function.identity(), error -> null);
        UnitsEnsembleModel unitsEnsemble = loadedR.getUnitsEnsemble()
                .match(Function.identity(), error -> null);
        if (resourceType == null || unitsEnsemble == null ||
                (resourceType.getUnitsEnsembleId() != null && resourceType.getBaseUnitId() != null)
        ) {
            return Mono.empty();
        }
        ResourceTypeModel.Builder builder = ResourceTypeModel.builder(resourceType);
        if (resourceType.getUnitsEnsembleId() == null) {
            builder.unitsEnsembleId(unitsEnsemble.getId());
        }
        if (resourceType.getBaseUnitId() == null) {
            builder.baseUnitId(getBaseUnit(unitsEnsemble).getId());
        }
        return resourceTypesDao.updateResourceTypeRetryable(ts, builder.build());
    }

    public Mono<Result<ResourceModel>> put(String id, String providerId, Long version,
                                           ResourcePutDto resourcePutDto,
                                           YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> resourcesDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), providerId, locale)
                                        .andThenMono(p -> validatePut(resourcePutDto, providerId, version, p,
                                                currentUser, locale, ts)
                                        )
                                        .flatMap(res -> res.applyMono(m -> {
                                            if (m.hasDefaultQuotaChanged) {
                                                return setProviderHasDefaultQuotas(
                                                        ts, m.resource.getTenantId(), providerId,
                                                        m.resource.getId(), m.resourceHasDefaultQuota
                                                ).map(ignored -> m);
                                            }
                                            return Mono.just(m);
                                        })),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.hasChanges) {
                                                return resourcesDao.updateResourceRetryable(ts, m.resource)
                                                        .doOnSuccess(v -> resourcesLoader.update(m.resource))
                                                        .thenReturn(Result.success(m.resource));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.resource));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    private Mono<Boolean> setProviderHasDefaultQuotas(
            YdbTxSession tx, TenantId tenantId, String providerId, String resourceId, boolean resourceHasDefaultQuotas
    ) {
        return resourcesDao.getAllByProvider(tx, providerId, tenantId, false, true)
                .flatMap(resourcesWithDefaultQuota -> {
                    Set<String> resourceIds =
                            resourcesWithDefaultQuota.stream().map(ResourceModel::getId).collect(Collectors.toSet());
                    boolean oldProviderHasDefaultQuotas = !resourceIds.isEmpty();
                    if (resourceHasDefaultQuotas) {
                        resourceIds.add(resourceId);
                    } else {
                        resourceIds.remove(resourceId);
                    }
                    boolean newProviderHasDefaultQuotas = !resourceIds.isEmpty();
                    if (newProviderHasDefaultQuotas != oldProviderHasDefaultQuotas) {
                        return providersDao.setHasDefaultQuotasRetryable(tx, tenantId, providerId,
                                        newProviderHasDefaultQuotas).map(ignored -> newProviderHasDefaultQuotas);
                    }
                    return Mono.just(newProviderHasDefaultQuotas);
                });
    }

    public Mono<Result<ResourceModel>> setReadOnly(String id, String providerId, boolean readOnly,
                                                   YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> resourcesDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), providerId, locale)
                                        .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                                                providerId, currentUser, locale, p
                                        ))
                                        .flatMap(result -> result
                                                .andThenMono(p -> validateSetReadOnly(readOnly, providerId, p,
                                                        currentUser, locale, ts))),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return resourcesDao.updateResourceRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> resourcesLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    public Mono<Result<List<ResourceModel>>> getAllProviderResources(YaUserDetails currentUser, String providerId,
                                                                     Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(r -> r
                .andThen(v -> validateId(providerId, locale))
                .andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
                                resourcesDao.getAllByProvider(immediateTx(session),
                                                providerId, Tenants.DEFAULT_TENANT_ID, false)
                                        .map(Result::success)
                        )
                ));
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private String prepareToken(ResourceModel lastItem) {
        return ContinuationTokens.encode(new ResourceContinuationToken(lastItem.getId()),
                continuationTokenWriter);
    }

    private Result<ResourceModel> validateExists(ResourceModel resource, String providerId,
                                                 Locale locale) {
        if (resource == null || resource.isDeleted() || !resource.getProviderId().equals(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                            .getMessage("errors.resource.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(resource);
    }

    private Result<Void> validateId(String resourceId, Locale locale) {
        if (!Uuids.isValidUuid(resourceId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                            .getMessage("errors.resource.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateResourceTypeId(String id, Locale locale) {
        if (!Uuids.isValidUuid(id)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(
                            messages.getMessage("errors.resource.type.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateSegmentIds(Set<String> ids, Locale locale) {
        if (ids != null && !ids.isEmpty()) {
            var errors = ErrorCollection.builder();
            for (String id : ids) {
                if (!Uuids.isValidUuid(id)) {
                    errors.addError(TypedError.notFound(
                            messages.getMessage("errors.resource.segment.not.found", null, locale))
                    );
                }
            }
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
        }
        return Result.success(null);
    }

    private Result<Void> validateContinuationToken(PageRequest.Validated<ResourceContinuationToken> pageRequest,
                                                   Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        return validateId(pageRequest.getContinuationToken().get().getId(), locale);
    }

    private Mono<Result<ProviderModel>> validateProvider(String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                            .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersLoader.getProviderByIdImmediate(providerId, Tenants.DEFAULT_TENANT_ID).map(provider -> {
            if (provider.isEmpty() || provider.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                                .getMessage("errors.provider.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider.get());
        });
    }

    private Mono<Result<ProviderModel>> validateProvider(YdbTxSession session, String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                            .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersDao.getById(session, providerId, Tenants.DEFAULT_TENANT_ID).map(provider -> {
            if (provider.isEmpty() || provider.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                                .getMessage("errors.provider.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider.get());
        });
    }

    private Mono<Result<Tuple2<ResourceModel, ValidatedCreateFields>>> validateCreation(
            ResourceCreateDto resource,
            String providerId,
            String newId,
            YaUserDetails currentUser,
            Locale locale,
            YdbTxSession session
    ) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return validateProvider(session, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        providerId, currentUser, locale, p
                ))
                .flatMap(result -> result.andThenMono(provider ->
                        validateCreateLoadable(resource, provider, locale, session).map(loadedR -> {
                            ResourceModel.Builder builder = ResourceModel.builder();
                            ErrorCollection.Builder errors = ErrorCollection.builder();
                            builder.id(newId);
                            builder.tenantId(tenantId);
                            builder.deleted(false);
                            builder.version(0L);
                            builder.providerId(provider.getId());
                            loadedR.getKey().doOnSuccess(builder::key);
                            loadedR.getKey().doOnFailure(errors::add);
                            validateText(resource::getNameEn, builder::nameEn, errors, "nameEn",
                                    MAX_NAME_LENGTH, locale);
                            validateText(resource::getNameRu, builder::nameRu, errors, "nameRu",
                                    MAX_NAME_LENGTH, locale);
                            validateText(resource::getDescriptionEn, builder::descriptionEn, errors,
                                    "descriptionEn", MAX_DESCRIPTION_LENGTH, locale);
                            validateText(resource::getDescriptionRu, builder::descriptionRu, errors,
                                    "descriptionRu", MAX_DESCRIPTION_LENGTH, locale);
                            loadedR.getUnitsEnsemble().doOnSuccess(unitsEnsemble -> {
                                builder.unitsEnsembleId(unitsEnsemble.getId());
                                validateUnits(resource::getAllowedUnitIds, resource::getDefaultUnitId,
                                        resource::getProviderApiUnitId,
                                        unitsEnsemble, builder, errors, locale);
                                loadedR.getResourceType().doOnSuccess(resourceType -> {
                                    if (!isEmpty(resourceType.getBaseUnitId())) {
                                        builder.baseUnitId(resourceType.getBaseUnitId());
                                    } else {
                                        builder.baseUnitId(getBaseUnit(unitsEnsemble).getId());
                                    }
                                });
                            });
                            loadedR.getUnitsEnsemble().doOnFailure(errors::add);
                            loadedR.getResourceType().doOnFailure(errors::add);
                            validateRequired(resource::getManaged, builder::managed, errors, "managed", locale);
                            validateRequired(resource::getOrderable, builder::orderable, errors, "orderable", locale);
                            validateRequired(resource::getReadOnly, builder::readOnly, errors, "readOnly", locale);
                            builder.virtual(resource.getVirtual().orElse(false));
                            buildSegmentation(loadedR::getSegments, builder, errors);
                            validateUnitsEnsembleMatch(loadedR, errors, locale);
                            loadedR.getAccountsSpace().doOnSuccess(a -> a.map(AccountSpaceModel::getId)
                                    .ifPresent(builder::accountsSpacesId));
                            loadedR.getAccountsSpace().doOnFailure(errors::add);
                            loadedR.resourceType.doOnSuccess(resourceType ->
                                    builder.resourceTypeId(resourceType.getId()));
                            loadedR.resourceType.doOnFailure(errors::add);
                            validateQuota(resource.getDefaultQuota(), builder::defaultQuota,
                                    errors, "defaultQuota", locale);
                            FeatureStateHelper.validateFeatureState(resource::getAllocatedSupported,
                                    builder::allocatedSupported, errors, "allocatedSupported", locale, messages);
                            AggregationSettingsHelper.validateAggregationSettings(resource::getAggregationSettings,
                                    builder::aggregationSettings, errors, "aggregationSettings", locale,
                                    messages);
                            if (errors.hasAnyErrors()) {
                                return Result.failure(errors.build());
                            }
                            return Result.success(Tuples.of(builder.build(), loadedR));
                        })
                )
        ));
    }

    private void validateUnitsEnsembleMatch(
            ValidatedCreateFields loadedR,
            ErrorCollection.Builder errors,
            Locale locale
    ) {
        Optional<ValidatedSegmentationFields> segments = loadedR.getSegments();
        Result<UnitsEnsembleModel> unitsEnsembleModelResult = loadedR.getUnitsEnsemble();
        segments.ifPresent(f -> Result.zip(Tuples.of(f.getResourceType(), unitsEnsembleModelResult))
                .doOnSuccess(resourceTypeAndUnitsEnsemble -> {
                    ResourceTypeModel resourceType = resourceTypeAndUnitsEnsemble.getT1();
                    UnitsEnsembleModel unitsEnsemble = resourceTypeAndUnitsEnsemble.getT2();
                    if (resourceType.getUnitsEnsembleId() != null
                            && !resourceType.getUnitsEnsembleId().equals(unitsEnsemble.getId())
                    ) {
                        errors.addError(TypedError.invalid(messages.getMessage(
                                "errors.resource.type.units.ensemble.dont.match", null, locale
                        )));
                    }
                })
        );
    }

    private void buildSegmentation(Supplier<Optional<ValidatedSegmentationFields>> segmentsSupplier,
                                   ResourceModel.Builder builder,
                                   ErrorCollection.Builder errors) {
        Optional<ValidatedSegmentationFields> segmentationO = segmentsSupplier.get();
        if (segmentationO.isPresent()) {
            ValidatedSegmentationFields segmentation = segmentationO.get();
            segmentation.getResourceType().doOnSuccess(type -> builder.resourceTypeId(type.getId()));
            segmentation.getResourceType().doOnFailure(errors::add);
            segmentation.getSegments().doOnSuccess(segments -> builder.segments(segments.stream().map(t ->
                            ResourceSegmentSettingsModel.builder()
                                    .segmentationId(t.getT1().getId())
                                    .segmentId(t.getT2().getId()).build()
                    ).collect(Collectors.toSet()))
            );
            segmentation.getSegments().doOnFailure(errors::add);
        } else {
            builder.resourceTypeId(null);
            builder.segments(null);
        }
    }

    private void validateUnits(Supplier<Optional<List<String>>> allowedUnitIdsGetter,
                               Supplier<Optional<String>> defaultUnitIdGetter,
                               Supplier<Optional<String>> providerApiUnitIdGetter,
                               UnitsEnsembleModel unitsEnsemble,
                               ResourceModel.Builder builder, ErrorCollection.Builder errors,
                               Locale locale) {
        Optional<List<String>> allowedUnitIds = allowedUnitIdsGetter.get();
        Optional<String> defaultUnitId = defaultUnitIdGetter.get();
        Optional<String> providerApiUnitId = providerApiUnitIdGetter.get();
        Set<String> validUnitIds = unitsEnsemble.getUnits().stream()
                .map(UnitModel::getId).collect(Collectors.toSet());
        ErrorCollection.Builder unitErrors = ErrorCollection.builder();
        ResourceUnitsModel.Builder unitsBuilder = ResourceUnitsModel.builder();
        if (defaultUnitId.isEmpty()) {
            unitErrors.addError("defaultUnitId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (defaultUnitId.get().isBlank()) {
            unitErrors.addError("defaultUnitId", TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (!Uuids.isValidUuid(defaultUnitId.get())) {
            unitErrors.addError("defaultUnitId", TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)));
        } else if (!validUnitIds.contains(defaultUnitId.get())) {
            unitErrors.addError("defaultUnitId", TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)));
        } else {
            unitsBuilder.defaultUnitId(defaultUnitId.get());
        }
        if (allowedUnitIds.isEmpty() || allowedUnitIds.get().isEmpty()) {
            unitErrors.addError("allowedUnitIds", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            if (Set.copyOf(allowedUnitIds.get()).size() != allowedUnitIds.get().size()) {
                unitErrors.addError("allowedUnitIds", TypedError.invalid(messages
                        .getMessage("errors.duplicate.resource.allowed.unit.ids", null, locale)));
            }
            for (int i = 0; i < allowedUnitIds.get().size(); i++) {
                String allowedUnitId = allowedUnitIds.get().get(i);
                if (allowedUnitId == null) {
                    unitErrors.addError("allowedUnitIds." + i, TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)));
                } else if (allowedUnitId.isBlank()) {
                    unitErrors.addError("allowedUnitIds." + i, TypedError.invalid(messages
                            .getMessage("errors.non.blank.text.is.required", null, locale)));
                } else if (!Uuids.isValidUuid(allowedUnitId)) {
                    unitErrors.addError("allowedUnitIds." + i, TypedError.invalid(messages
                            .getMessage("errors.unit.not.found", null, locale)));
                } else if (!validUnitIds.contains(allowedUnitId)) {
                    unitErrors.addError("allowedUnitIds." + i, TypedError.invalid(messages
                            .getMessage("errors.unit.not.found", null, locale)));
                } else {
                    unitsBuilder.addAllowedUnitId(allowedUnitId);
                }
            }
        }
        if (!unitsBuilder.getAllowedUnitIds().isEmpty() && unitsBuilder.getDefaultUnitId().isPresent()) {
            if (!unitsBuilder.getAllowedUnitIds().contains(unitsBuilder.getDefaultUnitId().get())) {
                unitErrors.addError("allowedUnitIds", TypedError.invalid(messages
                        .getMessage("errors.default.resource.unit.id.must.be.allowed", null, locale)));
            }
        }
        if (providerApiUnitId.isPresent() && !providerApiUnitId.get().isBlank()) {
            if (!Uuids.isValidUuid(providerApiUnitId.get())) {
                unitErrors.addError("providerApiUnitId", TypedError.invalid(messages
                        .getMessage("errors.unit.not.found", null, locale)));
            } else if (!validUnitIds.contains(providerApiUnitId.get())) {
                unitErrors.addError("providerApiUnitId", TypedError.invalid(messages
                        .getMessage("errors.unit.not.found", null, locale)));
            } else {
                unitsBuilder.providerApiUnitId(providerApiUnitId.get());
            }
        } else {
            if (allowedUnitIds.isPresent()) {
                Optional<UnitModel> minAllowedUnit =
                        Units.getMinAllowedUnit(Set.copyOf(allowedUnitIds.get()), unitsEnsemble);
                if (minAllowedUnit.isPresent()) {
                    unitsBuilder.providerApiUnitId(minAllowedUnit.get().getId());
                } else {
                    unitErrors.addError("providerApiUnitId", TypedError.invalid(messages
                            .getMessage("errors.unit.not.found", null, locale)));
                }
            } else {
                unitErrors.addError("providerApiUnitId", TypedError.invalid(messages
                        .getMessage("errors.unit.not.found", null, locale)));
            }
        }
        if (unitErrors.hasAnyErrors()) {
            errors.add(unitErrors);
        } else {
            builder.resourceUnits(unitsBuilder.build());
        }
    }

    private Mono<ValidatedCreateFields> validateCreateLoadable(ResourceCreateDto resource, ProviderModel provider,
                                                               Locale locale, YdbTxSession session) {
        return validateKey(session, resource::getKey, provider.getId(), "key", locale)
                .flatMap(k -> validateAccountsSpace(provider, resource.getSegmentations(), locale)
                        .flatMap(accountsSpacesId -> validateResourceType(session,
                                () -> resource.getSegmentations()
                                        .flatMap(CreateResourceTypeSegmentsDto::getResourceTypeId),
                                provider.getId(), locale, "segmentations.resourceTypeId"
                        )
                                .flatMap(resourceType -> validateUnitsEnsemble(session, resource::getUnitsEnsembleId,
                                        "unitsEnsembleId", locale, resourceType)
                                        .flatMap(e -> validateTypeSegmentations(provider.getId(),
                                                resource::getSegmentations,
                                                CreateResourceTypeSegmentsDto::getResourceTypeId,
                                                CreateResourceTypeSegmentsDto::getSegmentations,
                                                CreateResourceSegmentationSegmentDto::getSegmentationId,
                                                CreateResourceSegmentationSegmentDto::getSegmentId, locale,
                                                "segmentations", session, resourceType)
                                                .map(s -> new ValidatedCreateFields(
                                                        k, e, s.orElse(null), accountsSpacesId, resourceType
                                                ))))));
    }

    private Mono<Result<Optional<AccountSpaceModel>>> validateAccountsSpace(
            ProviderModel provider,
            Optional<CreateResourceTypeSegmentsDto> segmentations,
            Locale locale
    ) {
        if (!provider.isAccountsSpacesSupported()) {
            return Mono.just(Result.success(Optional.empty()));
        }
        if (
                segmentations.isEmpty() ||
                        segmentations.get().getSegmentations().isEmpty()
        ) {
            return Mono.just(Result.failure(ErrorCollection.builder()
                    .addError("segmentations", TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)))
                    .build()));
        }
        Set<ResourceSegmentSettingsModel> segments = new HashSet<>();
        List<CreateResourceSegmentationSegmentDto> segmentDtos = segmentations.get().getSegmentations().get();
        for (int i = 0; i < segmentDtos.size(); i++) {
            CreateResourceSegmentationSegmentDto segment = segmentDtos.get(i);
            if (segment.getSegmentationId().isEmpty()) {
                return Mono.just(Result.failure(ErrorCollection.builder()
                        .addError("segmentationId", TypedError.invalid(messages
                                .getMessage("errors.field.is.required", null, locale)))
                        .addDetail("segmentationsOrder", i)
                        .build()));
            }
            if (segment.getSegmentId().isEmpty()) {
                return Mono.just(Result.failure(ErrorCollection.builder()
                        .addError("segmentId", TypedError.invalid(messages
                                .getMessage("errors.field.is.required", null, locale)))
                        .addDetail("segmentationsOrder", i)
                        .build()));
            }
            segments.add(new ResourceSegmentSettingsModel(
                    segment.getSegmentationId().get(), segment.getSegmentId().get()));
        }
        return accountSpacesLoader.getAccountSpaces(provider.getTenantId(), provider.getId(), segments)
                .map(accountSpaceModels -> {
                    if (accountSpaceModels.isEmpty() || accountSpaceModels.get().isEmpty()) {
                        return Result.failure(ErrorCollection.builder()
                                .addError(TypedError.notFound(messages
                                        .getMessage("errors.accounts.space.not.found", null, locale)))
                                .addDetail("segments", ResourceSegmentSettingsModel.toString(segments))
                                .build());
                    }
                    if (accountSpaceModels.get().size() > 1) {
                        return Result.failure(ErrorCollection.builder()
                                .addError(TypedError.notFound(messages
                                        .getMessage("errors.provider.accounts.space.ambiguous", null, locale)))
                                .addDetail("segments", ResourceSegmentSettingsModel.toString(segments))
                                .addDetail("accountsSpaces", accountSpaceModels.get().toString())
                                .build());
                    }
                    return Result.success(Optional.of(accountSpaceModels.get().get(0)));
                });
    }

    @SuppressWarnings("ParameterNumber")
    private <U, V> Mono<Optional<ValidatedSegmentationFields>> validateTypeSegmentations(
            String providerId,
            Supplier<Optional<U>> typeSegmentationsSupplier,
            Function<U, Optional<String>> resourceTypeGetter,
            Function<U, Optional<List<V>>> segmentationsGetter,
            Function<V, Optional<String>> segmentationIdGetter,
            Function<V, Optional<String>> segmentIdGetter,
            Locale locale,
            String fieldKey,
            YdbTxSession session,
            Result<ResourceTypeModel> resourceType
    ) {
        Optional<U> typeSegmentations = typeSegmentationsSupplier.get();
        if (typeSegmentations.isEmpty()) {
            return Mono.just(Optional.empty());
        }
        Supplier<Optional<List<V>>> segmentationGetter = () -> segmentationsGetter.apply(typeSegmentations.get());
        String segmentationFieldKey = fieldKey + ".segmentations";
        return segmentationsValidator.validateSegmentations(session, segmentationGetter, segmentationIdGetter,
                        segmentIdGetter, providerId, locale, segmentationFieldKey, false)
                .flatMap(s -> validateTypeAndSegmentIds(resourceType, s, locale, session)
                        .map(Optional::of));
    }

    private Mono<ValidatedSegmentationFields> validateTypeAndSegmentIds(Result<ResourceTypeModel> resourceTypeResult,
                                                                        Result<List<Tuple2<ResourceSegmentationModel,
                                                                                ResourceSegmentModel>>> segmentsResult,
                                                                        Locale locale,
                                                                        YdbTxSession session) {
        return resourceTypeResult.match(resourceType -> segmentsResult.andThenMono(segments -> {
                            Set<String> segmentsIds = segments.stream()
                                    .map(segment -> segment.getT2().getId())
                                    .collect(Collectors.toUnmodifiableSet());
                            return resourcesDao.getAllByProvider(session,
                                            resourceType.getProviderId(),
                                            resourceType.getTenantId(),
                                            true)
                                    .map(existingResources -> existingResources.stream()
                                            .filter(existingResource -> {
                                                if (!existingResource.getResourceTypeId()
                                                        .equals(resourceType.getId())) {
                                                    return false;
                                                }
                                                Set<ResourceSegmentSettingsModel> existingSegmentation =
                                                        existingResource.getSegments();
                                                Set<String> existingSegmentsIds = existingSegmentation.stream()
                                                        .map(ResourceSegmentSettingsModel::getSegmentId)
                                                        .collect(Collectors.toUnmodifiableSet());
                                                return segmentsIds.containsAll(existingSegmentsIds) ||
                                                        existingSegmentsIds.containsAll(segmentsIds);
                                            })
                                            .findAny()
                                            .map(conflictResource ->
                                                    Result.<List<Tuple2<ResourceSegmentationModel,
                                                            ResourceSegmentModel>>>failure(ErrorCollection.builder()
                                                            .addError(TypedError.invalid(messages.getMessage(
                                                                    ERROR_MESSAGE,
                                                                    new Object[]{conflictResource.getId()},
                                                                    locale)))
                                                            .build()))
                                            .orElse(Result.success(segments))
                                    );
                        }),
                        error -> Mono.just(segmentsResult))
                .map(s -> new ValidatedSegmentationFields(resourceTypeResult, s));
    }

    private Mono<Result<ResourceTypeModel>> validateResourceType(YdbTxSession session,
                                                                 Supplier<Optional<String>> getter,
                                                                 String providerId, Locale locale,
                                                                 String fieldKey) {
        Optional<String> resourceTypeId = getter.get();
        if (resourceTypeId.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (resourceTypeId.get().isBlank()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.non.blank.text.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (!Uuids.isValidUuid(resourceTypeId.get())) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.resource.type.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return resourceTypesDao.getById(session, resourceTypeId.get(), Tenants.DEFAULT_TENANT_ID).map(type -> {
            if (type.isEmpty() || type.get().isDeleted() || !type.get().getProviderId().equals(providerId)) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                                .getMessage("errors.resource.type.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(type.get());
        });
    }

    private Mono<Result<String>> validateKey(YdbTxSession session, Supplier<Optional<String>> getter,
                                             String providerId, String fieldKey, Locale locale) {
        Optional<String> value = getter.get();
        if (value.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().isBlank()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.non.blank.text.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().length() > MAX_KEY_LENGTH) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.text.is.too.long", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return resourcesDao.existsByProviderAndKey(session, providerId, Tenants.DEFAULT_TENANT_ID,
                value.get(), false).map(exists -> {
            if (exists) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey,
                                TypedError.conflict(messages
                                        .getMessage("errors.resource.key.already.exists", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(value.get());
        });
    }

    private void validateText(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private void validateQuota(
            Optional<Long> value, Consumer<Long> setter,
            ErrorCollection.Builder errors, String fieldKey, Locale locale
    ) {
        if (value.isEmpty()) {
            return;
        }
        if (value.get() < 0) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.non.negative", null, locale)));
            return;
        }
        setter.accept(value.get());
    }

    private Mono<Result<UnitsEnsembleModel>> validateUnitsEnsemble(
            YdbTxSession session,
            Supplier<Optional<String>> getter,
            String fieldKey,
            Locale locale,
            Result<ResourceTypeModel> resourceType
    ) {
        Optional<String> unitsEnsembleIdByResourceType = Optional.ofNullable(
                resourceType.match(ResourceTypeModel::getUnitsEnsembleId, u -> null)
        );
        Optional<String> unitsEnsembleIdByResource = getter.get();
        Optional<String> unitsEnsembleId = coalesce(unitsEnsembleIdByResource, unitsEnsembleIdByResourceType);
        if (unitsEnsembleId.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (unitsEnsembleId.get().isBlank()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.non.blank.text.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (notMatch(unitsEnsembleIdByResource, unitsEnsembleIdByResourceType)) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.resource.units.ensemble.dont.match.resource.type", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (!Uuids.isValidUuid(unitsEnsembleId.get())) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                            .getMessage("errors.units.ensemble.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return unitsEnsemblesDao.getById(session, unitsEnsembleId.get(), Tenants.DEFAULT_TENANT_ID).map(ensemble -> {
            if (ensemble.isEmpty() || ensemble.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                                .getMessage("errors.units.ensemble.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(ensemble.get());
        });
    }

    private Optional<String> coalesce(Optional<String> o1, Optional<String> o2) {
        if (o1.isEmpty() || o1.get().isEmpty()) {
            return o2;
        }
        return o1;
    }

    private boolean notMatch(Optional<String> o1, Optional<String> o2) {
        if (o1.isEmpty() || o1.get().isEmpty()) {
            return false;
        }
        if (o2.isEmpty() || o2.get().isEmpty()) {
            return false;
        }
        return !o1.get().equals(o2.get());
    }

    private <T> void validateRequired(Supplier<Optional<T>> getter, Consumer<T> setter,
                                      ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<T> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Result<ValidatedResource>> validatePut(ResourcePutDto resource,
                                                        String providerId,
                                                        Long version,
                                                        ResourceModel existingResource,
                                                        YaUserDetails currentUser,
                                                        Locale locale,
                                                        YdbTxSession session) {
        return validateProvider(session, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        providerId, currentUser, locale, p
                ))
                .flatMap(result -> result.andThenMono(provider ->
                        validatePutLoadable(resource, existingResource, providerId, locale, session).map(loadedR -> {
                            ResourceModel.Builder builder = ResourceModel.builder(existingResource);
                            ErrorCollection.Builder errors = ErrorCollection.builder();
                            validateText(resource::getNameEn, builder::nameEn, errors, "nameEn",
                                    MAX_NAME_LENGTH, locale);
                            validateText(resource::getNameRu, builder::nameRu, errors, "nameRu",
                                    MAX_NAME_LENGTH, locale);
                            validateText(resource::getDescriptionEn, builder::descriptionEn, errors,
                                    "descriptionEn", MAX_DESCRIPTION_LENGTH, locale);
                            validateText(resource::getDescriptionRu, builder::descriptionRu, errors,
                                    "descriptionRu", MAX_DESCRIPTION_LENGTH, locale);
                            validateVersion(() -> Optional.ofNullable(version), existingResource::getVersion,
                                    builder::version, errors, "version", locale);
                            validateUnits(resource::getAllowedUnitIds, resource::getDefaultUnitId,
                                    resource::getProviderApiUnitId,
                                    loadedR.getUnitsEnsemble(), builder, errors, locale);
                            validateRequired(resource::getManaged, builder::managed, errors, "managed", locale);
                            validateRequired(resource::getOrderable, builder::orderable, errors, "orderable", locale);
                            builder.virtual(resource.getVirtual().orElse(false));
                            validateQuota(resource.getDefaultQuota(), builder::defaultQuota,
                                    errors, "defaultQuota", locale);
                            FeatureStateHelper.validateFeatureState(resource::getAllocatedSupported,
                                    builder::allocatedSupported, errors, "allocatedSupported", locale, messages);
                            AggregationSettingsHelper.validateAggregationSettings(resource::getAggregationSettings,
                                    builder::aggregationSettings, errors, "aggregationSettings", locale,
                                    messages);
                            if (errors.hasAnyErrors()) {
                                return Result.failure(errors.build());
                            }
                            boolean hasChanges = builder.hasChanges(existingResource);
                            boolean resourceHasDefaultQuota = hasDefaultQuota(resource);
                            boolean existingResourceHasDefaultQuota = hasDefaultQuota(existingResource);
                            return Result.success(new ValidatedResource(
                                    hasChanges ? builder.build() : existingResource,
                                    hasChanges,
                                    hasChanges ? resourceHasDefaultQuota : existingResourceHasDefaultQuota,
                                    resourceHasDefaultQuota != existingResourceHasDefaultQuota
                            ));
                        })
                )
        ));
    }

    private boolean hasDefaultQuota(ResourcePutDto resource) {
        return resource.getDefaultQuota().map(quota -> quota > 0).orElse(false);
    }

    private boolean hasDefaultQuota(ResourceModel resource) {
        return resource.getDefaultQuota().map(quota -> quota > 0).orElse(false);
    }

    private Mono<ValidatedPutFields> validatePutLoadable(ResourcePutDto resource, ResourceModel existingResource,
                                                         String providerId, Locale locale, YdbTxSession session) {
        return unitsEnsemblesDao.getById(session, existingResource.getUnitsEnsembleId(), Tenants.DEFAULT_TENANT_ID)
                .map(e -> new ValidatedPutFields(e.get()));
    }

    private void validateVersion(Supplier<Optional<Long>> getter, Supplier<Long> existingGetter,
                                 Consumer<Long> setter, ErrorCollection.Builder errors, String fieldKey,
                                 Locale locale) {
        Optional<Long> value = getter.get();
        Long existingValue = existingGetter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (!Objects.equals(value.get(), existingValue)) {
            errors.addError(fieldKey, TypedError.versionMismatch(messages
                    .getMessage("errors.version.mismatch", null, locale)));
        } else {
            setter.accept(existingValue + 1L);
        }
    }

    private Mono<Result<Tuple2<ResourceModel, Boolean>>> validateSetReadOnly(boolean readOnly,
                                                                             String providerId,
                                                                             ResourceModel existingResource,
                                                                             YaUserDetails currentUser,
                                                                             Locale locale,
                                                                             YdbTxSession session) {
        return validateProvider(session, providerId, locale).map(providerR -> providerR
                .andThen(provider -> {
                    ResourceModel.Builder builder = ResourceModel.builder(existingResource);
                    builder.readOnly(readOnly)
                            .version(existingResource.getVersion() + 1L);
                    if (builder.hasChanges(existingResource)) {
                        return Result.success(Tuples.of(builder.build(), true));
                    }
                    return Result.success(Tuples.of(existingResource, false));
                })
        );
    }

    private static final class ValidatedCreateFields {

        private final Result<String> key;
        private final Result<UnitsEnsembleModel> unitsEnsemble;
        private final ValidatedSegmentationFields segments;
        private final Result<Optional<AccountSpaceModel>> accountsSpace;
        private final Result<ResourceTypeModel> resourceType;


        private ValidatedCreateFields(Result<String> key,
                                      Result<UnitsEnsembleModel> unitsEnsemble,
                                      ValidatedSegmentationFields segments,
                                      Result<Optional<AccountSpaceModel>> accountsSpace,
                                      Result<ResourceTypeModel> resourceType
        ) {
            this.key = key;
            this.unitsEnsemble = unitsEnsemble;
            this.segments = segments;
            this.accountsSpace = accountsSpace;
            this.resourceType = resourceType;
        }

        public Result<String> getKey() {
            return key;
        }

        public Result<UnitsEnsembleModel> getUnitsEnsemble() {
            return unitsEnsemble;
        }

        public Optional<ValidatedSegmentationFields> getSegments() {
            return Optional.ofNullable(segments);
        }

        public Result<Optional<AccountSpaceModel>> getAccountsSpace() {
            return accountsSpace;
        }

        public Result<ResourceTypeModel> getResourceType() {
            return resourceType;
        }
    }

    private static final class ValidatedPutFields {

        private final UnitsEnsembleModel unitsEnsemble;

        private ValidatedPutFields(UnitsEnsembleModel unitsEnsemble) {
            this.unitsEnsemble = unitsEnsemble;
        }

        public UnitsEnsembleModel getUnitsEnsemble() {
            return unitsEnsemble;
        }
    }

    private static final class ValidatedSegmentationFields {

        private final Result<ResourceTypeModel> resourceType;
        private final Result<List<Tuple2<ResourceSegmentationModel, ResourceSegmentModel>>> segments;

        private ValidatedSegmentationFields(
                Result<ResourceTypeModel> resourceType,
                Result<List<Tuple2<ResourceSegmentationModel, ResourceSegmentModel>>> segments) {
            this.resourceType = resourceType;
            this.segments = segments;
        }

        public Result<ResourceTypeModel> getResourceType() {
            return resourceType;
        }

        public Result<List<Tuple2<ResourceSegmentationModel, ResourceSegmentModel>>> getSegments() {
            return segments;
        }

    }

    private static class ValidatedResource {
        private final ResourceModel resource;
        private final boolean hasChanges;
        private final boolean resourceHasDefaultQuota;
        private final boolean hasDefaultQuotaChanged;

        private ValidatedResource(
                ResourceModel resource,
                boolean hasChanges,
                boolean resourceHasDefaultQuota,
                boolean hasDefaultQuotaChanged
        ) {
            this.resource = resource;
            this.hasChanges = hasChanges;
            this.resourceHasDefaultQuota = resourceHasDefaultQuota;
            this.hasDefaultQuotaChanged = hasDefaultQuotaChanged;
        }
    }
}
