package ru.yandex.intranet.d.services.resources.segmentations;

import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Supplier;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.ResourceSegmentationCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.ResourceSegmentationPutDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Resource segmentations service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourceSegmentationsService {

    private static final int MAX_KEY_LENGTH = 256;
    private static final int MAX_NAME_LENGTH = 256;
    private static final int MAX_DESCRIPTION_LENGTH = 1024;

    private final ResourceSegmentationsDao resourceSegmentationsDao;
    private final ProvidersDao providersDao;
    private final YdbTableClient tableClient;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final ProvidersLoader providersLoader;
    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final ObjectReader continuationTokenReader;
    private final ObjectWriter continuationTokenWriter;

    @SuppressWarnings("ParameterNumber")
    public ResourceSegmentationsService(
            ResourceSegmentationsDao resourceSegmentationsDao,
            ProvidersDao providersDao,
            YdbTableClient tableClient,
            ResourceSegmentationsLoader resourceSegmentationsLoader,
            ProvidersLoader providersLoader,
            @Qualifier("messageSource") MessageSource messages,
            SecurityManagerService securityManagerService,
            @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper) {
        this.resourceSegmentationsDao = resourceSegmentationsDao;
        this.providersDao = providersDao;
        this.tableClient = tableClient;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.providersLoader = providersLoader;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.continuationTokenReader = objectMapper.getObjectMapper()
                .readerFor(ResourceSegmentationContinuationToken.class);
        this.continuationTokenWriter = objectMapper.getObjectMapper()
                .writerFor(ResourceSegmentationContinuationToken.class);
    }

    public Mono<Result<ResourceSegmentationModel>> getById(String id, String providerId,
                                                           YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                validateProvider(providerId, locale)
                        .flatMap(r -> r.andThenMono(pr ->
                                tableClient.usingSessionMonoRetryable(session -> resourceSegmentationsDao
                                        .getById(immediateTx(session), id, Tenants.DEFAULT_TENANT_ID)
                                        .map(t -> validateExists(t.orElse(null), pr.getId(), locale))
                                )
                        ))
        ));
    }

    public Mono<Result<Page<ResourceSegmentationModel>>> getPage(String providerId, PageRequest pageRequest,
                                                                 YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u -> {
            Result<PageRequest.Validated<ResourceSegmentationContinuationToken>> pageValidation
                    = pageRequest.validate(continuationTokenReader, messages, locale);
            return pageValidation.andThenDo(p -> validateContinuationToken(p, locale)).andThenMono(p -> {
                int limit = p.getLimit();
                String fromId = p.getContinuationToken().map(ResourceSegmentationContinuationToken::getId)
                        .orElse(null);
                return validateProvider(providerId, locale)
                        .flatMap(r -> r.andThenMono(pr -> tableClient.usingSessionMonoRetryable(session ->
                                resourceSegmentationsDao.getByProvider(immediateTx(session), pr.getId(),
                                        Tenants.DEFAULT_TENANT_ID, fromId, limit + 1, false)
                                        .map(values -> values.size() > limit
                                                ? Page.page(values.subList(0, limit),
                                                prepareToken(values.get(limit - 1)))
                                                : Page.lastPage(values))
                                        .map(Result::success))));
            });
        }));
    }

    public Mono<Result<ResourceSegmentationModel>> create(ResourceSegmentationCreateDto resourceSegmentation,
                                                          String providerId, YaUserDetails currentUser,
                                                          Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThenMono(u -> {
            String newId = UUID.randomUUID().toString();
            return tableClient.usingSessionMonoRetryable(session ->
                    session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                            validateCreation(resourceSegmentation, providerId, newId, currentUser, locale, ts)
                                    .flatMap(r -> r.andThenMono(validated ->
                                                    resourceSegmentationsDao.upsertResourceSegmentationRetryable(ts,
                                                                    validated)
                                                            .doOnSuccess(v -> resourceSegmentationsLoader
                                                                    .update(validated))
                                                            .thenReturn(Result.success(validated))
                                            )
                                    )
                    )
            );
        });
    }

    public Mono<Result<ResourceSegmentationModel>> put(String id, String providerId, Long version,
                                                       ResourceSegmentationPutDto resourceSegmentationPutDto,
                                                       YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(
                v -> validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> resourceSegmentationsDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), providerId, locale)
                                        .andThenMono(p -> validatePut(resourceSegmentationPutDto, providerId, version,
                                                p, currentUser, locale, ts)),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return resourceSegmentationsDao
                                                        .updateResourceSegmentationRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> resourceSegmentationsLoader
                                                                .update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private String prepareToken(ResourceSegmentationModel lastItem) {
        return ContinuationTokens.encode(new ResourceSegmentationContinuationToken(lastItem.getId()),
                continuationTokenWriter);
    }

    private Result<Void> validateId(String resourceId, Locale locale) {
        if (!Uuids.isValidUuid(resourceId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.resource.segmentation.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateContinuationToken(
            PageRequest.Validated<ResourceSegmentationContinuationToken> pageRequest,
            Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        return validateId(pageRequest.getContinuationToken().get().getId(), locale);
    }

    private Result<ResourceSegmentationModel> validateExists(ResourceSegmentationModel resourceSegmentation,
                                                             String providerId, Locale locale) {
        if (resourceSegmentation == null || resourceSegmentation.isDeleted()
                || !resourceSegmentation.getProviderId().equals(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.resource.segmentation.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(resourceSegmentation);
    }

    private Mono<Result<ProviderModel>> validateProvider(String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersLoader.getProviderByIdImmediate(providerId, Tenants.DEFAULT_TENANT_ID).map(provider -> {
            if (provider.isEmpty() || provider.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.provider.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider.get());
        });
    }

    private Mono<Result<ProviderModel>> validateProvider(YdbTxSession session, String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersDao.getById(session, providerId, Tenants.DEFAULT_TENANT_ID).map(provider -> {
            if (provider.isEmpty() || provider.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.provider.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider.get());
        });
    }

    private Mono<Result<ResourceSegmentationModel>> validateCreation(
            ResourceSegmentationCreateDto resourceSegmentation,
            String providerId,
            String newId,
            YaUserDetails currentUser,
            Locale locale,
            YdbTxSession session) {
        return validateProvider(session, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        p.getId(), currentUser, locale, p
                ))
                .flatMap(result -> result.andThenMono(provider ->
                        validateKey(session, resourceSegmentation::getKey, provider.getId(), "key", locale)
                                .map(key -> {
                                    ResourceSegmentationModel.Builder builder = ResourceSegmentationModel.builder();
                                    ErrorCollection.Builder errors = ErrorCollection.builder();
                                    builder.id(newId);
                                    builder.tenantId(Tenants.DEFAULT_TENANT_ID);
                                    builder.deleted(false);
                                    builder.version(0L);
                                    builder.providerId(provider.getId());
                                    key.doOnSuccess(builder::key);
                                    key.doOnFailure(errors::add);
                                    validateText(resourceSegmentation::getNameEn, builder::nameEn, errors,
                                            "nameEn", MAX_NAME_LENGTH, locale);
                                    validateText(resourceSegmentation::getNameRu, builder::nameRu, errors,
                                            "nameRu", MAX_NAME_LENGTH, locale);
                                    validateText(resourceSegmentation::getDescriptionEn, builder::descriptionEn,
                                            errors, "descriptionEn", MAX_DESCRIPTION_LENGTH, locale);
                                    validateText(resourceSegmentation::getDescriptionRu, builder::descriptionRu,
                                            errors, "descriptionRu", MAX_DESCRIPTION_LENGTH, locale);
                                    builder.groupingOrder(resourceSegmentation.getGroupingOrder().orElse(0));
                                    resourceSegmentation.getUiSettings().ifPresent(it ->
                                            builder.uiSettings(it.toModel()));
                                    if (errors.hasAnyErrors()) {
                                        return Result.failure(errors.build());
                                    }
                                    return Result.success(builder.build());
                                })
                ))
        );
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Result<Tuple2<ResourceSegmentationModel, Boolean>>> validatePut(
            ResourceSegmentationPutDto resourceSegmentation,
            String providerId,
            Long version,
            ResourceSegmentationModel existingResourceSegmentation,
            YaUserDetails currentUser,
            Locale locale,
            YdbTxSession session) {
        return validateProvider(session, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        p.getId(), currentUser, locale, p
                ))
                .map(result -> result.andThen(provider -> {
                    ResourceSegmentationModel.Builder builder = ResourceSegmentationModel.builder();
                    ErrorCollection.Builder errors = ErrorCollection.builder();
                    builder.id(existingResourceSegmentation.getId());
                    builder.tenantId(existingResourceSegmentation.getTenantId());
                    builder.deleted(existingResourceSegmentation.isDeleted());
                    builder.providerId(existingResourceSegmentation.getProviderId());
                    builder.key(existingResourceSegmentation.getKey());
                    validateVersion(() -> Optional.ofNullable(version), existingResourceSegmentation::getVersion,
                            builder::version, errors, "version", locale);
                    validateText(resourceSegmentation::getNameEn, builder::nameEn, errors, "nameEn",
                            MAX_NAME_LENGTH, locale);
                    validateText(resourceSegmentation::getNameRu, builder::nameRu, errors, "nameRu",
                            MAX_NAME_LENGTH, locale);
                    validateText(resourceSegmentation::getDescriptionEn, builder::descriptionEn, errors,
                            "descriptionEn", MAX_DESCRIPTION_LENGTH, locale);
                    validateText(resourceSegmentation::getDescriptionRu, builder::descriptionRu, errors,
                            "descriptionRu", MAX_DESCRIPTION_LENGTH, locale);
                    builder.groupingOrder(resourceSegmentation.getGroupingOrder().orElse(0));
                    resourceSegmentation.getUiSettings().ifPresent(it -> builder.uiSettings(it.toModel()));
                    if (errors.hasAnyErrors()) {
                        return Result.failure(errors.build());
                    }
                    if (builder.hasChanges(existingResourceSegmentation)) {
                        return Result.success(Tuples.of(builder.build(), true));
                    }
                    return Result.success(Tuples.of(existingResourceSegmentation, false));
                })));
    }

    private void validateText(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private Mono<Result<String>> validateKey(YdbTxSession session, Supplier<Optional<String>> getter,
                                             String providerId, String fieldKey, Locale locale) {
        Optional<String> value = getter.get();
        if (value.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().isBlank()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().length() > MAX_KEY_LENGTH) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return resourceSegmentationsDao.existsByProviderAndKey(session, providerId, Tenants.DEFAULT_TENANT_ID,
                value.get(), false).map(exists -> {
            if (exists) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey,
                        TypedError.conflict(messages
                                .getMessage("errors.resource.segmentation.key.already.exists", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(value.get());
        });
    }

    private void validateVersion(Supplier<Optional<Long>> getter, Supplier<Long> existingGetter,
                                 Consumer<Long> setter, ErrorCollection.Builder errors, String fieldKey,
                                 Locale locale) {
        Optional<Long> value = getter.get();
        Long existingValue = existingGetter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (!Objects.equals(value.get(), existingValue)) {
            errors.addError(fieldKey, TypedError.versionMismatch(messages
                    .getMessage("errors.version.mismatch", null, locale)));
        } else {
            setter.accept(existingValue + 1L);
        }
    }

}
