package ru.yandex.intranet.d.services.resources.segments;

import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Supplier;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao;
import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.resources.directory.segments.ResourceSegmentCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.segments.ResourceSegmentPutDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Resource segments service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourceSegmentsService {

    private static final int MAX_KEY_LENGTH = 256;
    private static final int MAX_NAME_LENGTH = 256;
    private static final int MAX_DESCRIPTION_LENGTH = 1024;

    private final ResourceSegmentsDao resourceSegmentsDao;
    private final ProvidersDao providersDao;
    private final ResourceSegmentationsDao resourceSegmentationsDao;
    private final YdbTableClient tableClient;
    private final ResourceSegmentsLoader resourceSegmentsLoader;
    private final ProvidersLoader providersLoader;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final ObjectReader continuationTokenReader;
    private final ObjectWriter continuationTokenWriter;

    @SuppressWarnings("ParameterNumber")
    public ResourceSegmentsService(
            ResourceSegmentsDao resourceSegmentsDao,
            ProvidersDao providersDao,
            ResourceSegmentationsDao resourceSegmentationsDao,
            YdbTableClient tableClient,
            ResourceSegmentsLoader resourceSegmentsLoader,
            ProvidersLoader providersLoader,
            ResourceSegmentationsLoader resourceSegmentationsLoader,
            @Qualifier("messageSource") MessageSource messages,
            SecurityManagerService securityManagerService,
            @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper) {
        this.resourceSegmentsDao = resourceSegmentsDao;
        this.providersDao = providersDao;
        this.resourceSegmentationsDao = resourceSegmentationsDao;
        this.tableClient = tableClient;
        this.resourceSegmentsLoader = resourceSegmentsLoader;
        this.providersLoader = providersLoader;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.continuationTokenReader = objectMapper.getObjectMapper()
                .readerFor(ResourceSegmentContinuationToken.class);
        this.continuationTokenWriter = objectMapper.getObjectMapper()
                .writerFor(ResourceSegmentContinuationToken.class);
    }

    public Mono<Result<ResourceSegmentModel>> getById(String id, String providerId, String segmentationId,
                                                      YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                Mono.zip(validateProvider(providerId, locale),
                        validateSegmentation(segmentationId, providerId, locale))
                        .map(Tuple2::getT2)
                        .flatMap(r -> r.andThenMono(pr ->
                                tableClient.usingSessionMonoRetryable(session -> resourceSegmentsDao
                                        .getById(immediateTx(session), id, Tenants.DEFAULT_TENANT_ID)
                                        .map(t -> validateExists(t.orElse(null), segmentationId, locale))
                                )
                        ))
        ));
    }

    public Mono<Result<Page<ResourceSegmentModel>>> getPage(String providerId, String segmentationId,
                                                            PageRequest pageRequest,
                                                            YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u -> {
            Result<PageRequest.Validated<ResourceSegmentContinuationToken>> pageValidation
                    = pageRequest.validate(continuationTokenReader, messages, locale);
            return pageValidation.andThenDo(p -> validateContinuationToken(p, locale)).andThenMono(p -> {
                int limit = p.getLimit();
                String fromId = p.getContinuationToken().map(ResourceSegmentContinuationToken::getId)
                        .orElse(null);
                return Mono.zip(validateProvider(providerId, locale),
                        validateSegmentation(segmentationId, providerId, locale))
                        .map(Tuple2::getT2)
                        .flatMap(r -> r.andThenMono(seg -> tableClient.usingSessionMonoRetryable(session ->
                                resourceSegmentsDao.getBySegmentation(immediateTx(session), seg.getId(),
                                        Tenants.DEFAULT_TENANT_ID, fromId, limit + 1, false)
                                        .map(values -> values.size() > limit
                                                ? Page.page(values.subList(0, limit),
                                                prepareToken(values.get(limit - 1)))
                                                : Page.lastPage(values))
                                        .map(Result::success))));
            });
        }));
    }

    public Mono<Result<ResourceSegmentModel>> create(ResourceSegmentCreateDto resourceSegment, String providerId,
                                                     String segmentationId, YaUserDetails currentUser,
                                                     Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThenMono(u -> {
            String newId = UUID.randomUUID().toString();
            return tableClient.usingSessionMonoRetryable(session ->
                    session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                            validateCreation(resourceSegment, providerId, segmentationId, newId, currentUser,
                                    locale, ts).flatMap(rs -> rs.andThenMono(validated ->
                                            resourceSegmentsDao.upsertResourceSegmentRetryable(ts, validated)
                                                    .doOnSuccess(v -> resourceSegmentsLoader.update(validated))
                                                    .thenReturn(Result.success(validated))
                                    )
                            )
                    )
            );
        });
    }

    public Mono<Result<ResourceSegmentModel>> put(String id, String providerId, String segmentationId, Long version,
                                                  ResourceSegmentPutDto resourceSegmentPutDto,
                                                  YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> resourceSegmentsDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), segmentationId, locale)
                                        .andThenMono(rs -> validatePut(resourceSegmentPutDto, providerId,
                                                segmentationId, version, rs, currentUser, locale, ts)),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return resourceSegmentsDao.updateResourceSegmentRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> resourceSegmentsLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private String prepareToken(ResourceSegmentModel lastItem) {
        return ContinuationTokens.encode(new ResourceSegmentContinuationToken(lastItem.getId()),
                continuationTokenWriter);
    }

    private Result<Void> validateId(String resourceId, Locale locale) {
        if (!Uuids.isValidUuid(resourceId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.resource.segment.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateContinuationToken(
            PageRequest.Validated<ResourceSegmentContinuationToken> pageRequest,
            Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        return validateId(pageRequest.getContinuationToken().get().getId(), locale);
    }

    private Result<ResourceSegmentModel> validateExists(ResourceSegmentModel resourceSegment, String segmentationId,
                                                        Locale locale) {
        if (resourceSegment == null || resourceSegment.isDeleted()
                || !resourceSegment.getSegmentationId().equals(segmentationId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.resource.segment.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(resourceSegment);
    }

    private Mono<Result<ProviderModel>> validateProvider(String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersLoader.getProviderByIdImmediate(providerId, Tenants.DEFAULT_TENANT_ID).map(provider -> {
            if (provider.isEmpty() || provider.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.provider.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider.get());
        });
    }

    private Mono<Result<ResourceSegmentationModel>> validateSegmentation(String segmentationId, String providerId,
                                                                         Locale locale) {
        if (!Uuids.isValidUuid(segmentationId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.resource.segmentation.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return resourceSegmentationsLoader.getResourceSegmentationByIdImmediate(segmentationId,
                Tenants.DEFAULT_TENANT_ID).map(segmentation -> {
            if (segmentation.isEmpty() || segmentation.get().isDeleted()
                    || !segmentation.get().getProviderId().equals(providerId)) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.resource.segmentation.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(segmentation.get());
        });
    }

    private Mono<Result<ProviderModel>> validateProvider(YdbTxSession session, String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersDao.getById(session, providerId, Tenants.DEFAULT_TENANT_ID).map(provider -> {
            if (provider.isEmpty() || provider.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.provider.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider.get());
        });
    }

    private Mono<Result<ResourceSegmentationModel>> validateSegmentation(YdbTxSession session, String segmentationId,
                                                                         String providerId, Locale locale) {
        if (!Uuids.isValidUuid(segmentationId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.resource.segmentation.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return resourceSegmentationsDao.getById(session, segmentationId, Tenants.DEFAULT_TENANT_ID).map(s -> {
            if (s.isEmpty() || s.get().isDeleted() || !s.get().getProviderId().equals(providerId)) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.resource.segmentation.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(s.get());
        });
    }

    private Mono<Result<ResourceSegmentModel>> validateCreation(ResourceSegmentCreateDto resourceSegment,
                                                                String providerId, String segmentationId,
                                                                String newId, YaUserDetails currentUser,
                                                                Locale locale, YdbTxSession session) {
        return validateProvider(session, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        p.getId(), currentUser, locale, p
                ))
                .flatMap(result -> result.andThenMono(pr -> validateSegmentation(session, segmentationId, providerId,
                        locale).flatMap(rs ->
                        rs.andThenMono(segmentation ->
                                validateKey(session, resourceSegment::getKey, segmentation.getId(), "key", locale)
                                        .map(key -> {
                                            ResourceSegmentModel.Builder builder = ResourceSegmentModel.builder();
                                            ErrorCollection.Builder errors = ErrorCollection.builder();
                                            builder.id(newId);
                                            builder.tenantId(Tenants.DEFAULT_TENANT_ID);
                                            builder.deleted(false);
                                            builder.version(0L);
                                            builder.segmentationId(segmentation.getId());
                                            key.doOnSuccess(builder::key);
                                            key.doOnFailure(errors::add);
                                            validateText(resourceSegment::getNameEn, builder::nameEn, errors,
                                                    "nameEn", MAX_NAME_LENGTH, locale);
                                            validateText(resourceSegment::getNameRu, builder::nameRu, errors,
                                                    "nameRu", MAX_NAME_LENGTH, locale);
                                            validateText(resourceSegment::getDescriptionEn, builder::descriptionEn,
                                                    errors, "descriptionEn", MAX_DESCRIPTION_LENGTH, locale);
                                            validateText(resourceSegment::getDescriptionRu, builder::descriptionRu,
                                                    errors, "descriptionRu", MAX_DESCRIPTION_LENGTH, locale);
                                            builder.uncommon(resourceSegment.getUncommon().orElse(false));
                                            if (errors.hasAnyErrors()) {
                                                return Result.failure(errors.build());
                                            }
                                            return Result.success(builder.build());
                                        })))
                ))
        );
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Result<Tuple2<ResourceSegmentModel, Boolean>>> validatePut(
            ResourceSegmentPutDto resourceSegment,
            String providerId,
            String segmentationId,
            Long version,
            ResourceSegmentModel existingResourceSegment,
            YaUserDetails currentUser,
            Locale locale,
            YdbTxSession session) {
        return validateProvider(session, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        p.getId(), currentUser, locale, p
                ))
                .flatMap(result -> result.andThenMono(pr -> validateSegmentation(session, segmentationId, providerId,
                        locale).map(rs ->
                        rs.andThen(provider -> {
                            ResourceSegmentModel.Builder builder = ResourceSegmentModel.builder();
                            ErrorCollection.Builder errors = ErrorCollection.builder();
                            builder.id(existingResourceSegment.getId());
                            builder.tenantId(existingResourceSegment.getTenantId());
                            builder.deleted(existingResourceSegment.isDeleted());
                            builder.segmentationId(existingResourceSegment.getSegmentationId());
                            builder.key(existingResourceSegment.getKey());
                            validateVersion(() -> Optional.ofNullable(version), existingResourceSegment::getVersion,
                                    builder::version, errors, "version", locale);
                            validateText(resourceSegment::getNameEn, builder::nameEn, errors, "nameEn",
                                    MAX_NAME_LENGTH, locale);
                            validateText(resourceSegment::getNameRu, builder::nameRu, errors, "nameRu",
                                    MAX_NAME_LENGTH, locale);
                            validateText(resourceSegment::getDescriptionEn, builder::descriptionEn, errors,
                                    "descriptionEn", MAX_DESCRIPTION_LENGTH, locale);
                            validateText(resourceSegment::getDescriptionRu, builder::descriptionRu, errors,
                                    "descriptionRu", MAX_DESCRIPTION_LENGTH, locale);
                            builder.uncommon(resourceSegment.getUncommon().orElse(false));
                            if (errors.hasAnyErrors()) {
                                return Result.failure(errors.build());
                            }
                            if (builder.hasChanges(existingResourceSegment)) {
                                return Result.success(Tuples.of(builder.build(), true));
                            }
                            return Result.success(Tuples.of(existingResourceSegment, false));
                        }))
                ))
        );
    }

    private void validateText(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private Mono<Result<String>> validateKey(YdbTxSession session, Supplier<Optional<String>> getter,
                                             String segmentationId, String fieldKey, Locale locale) {
        Optional<String> value = getter.get();
        if (value.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().isBlank()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().length() > MAX_KEY_LENGTH) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return resourceSegmentsDao.existsBySegmentationAndKey(session, segmentationId, Tenants.DEFAULT_TENANT_ID,
                value.get(), false).map(exists -> {
            if (exists) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey,
                        TypedError.conflict(messages
                                .getMessage("errors.resource.segment.key.already.exists", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(value.get());
        });
    }

    private void validateVersion(Supplier<Optional<Long>> getter, Supplier<Long> existingGetter,
                                 Consumer<Long> setter, ErrorCollection.Builder errors, String fieldKey,
                                 Locale locale) {
        Optional<Long> value = getter.get();
        Long existingValue = existingGetter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (!Objects.equals(value.get(), existingValue)) {
            errors.addError(fieldKey, TypedError.versionMismatch(messages
                    .getMessage("errors.version.mismatch", null, locale)));
        } else {
            setter.accept(existingValue + 1L);
        }
    }

}
