package ru.yandex.intranet.d.services.resources.types;

import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Supplier;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.resources.types.ResourceTypesDao;
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypeCreateDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypePatchDto;
import ru.yandex.intranet.d.web.model.resources.directory.types.ResourceTypePutDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Resource types service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourceTypesService {

    private static final int MAX_KEY_LENGTH = 256;
    private static final int MAX_NAME_LENGTH = 256;
    private static final int MAX_DESCRIPTION_LENGTH = 1024;

    private final ResourceTypesDao resourceTypesDao;
    private final ProvidersDao providersDao;
    private final YdbTableClient tableClient;
    private final ResourceTypesLoader resourceTypesLoader;
    private final ProvidersLoader providersLoader;
    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final ObjectReader continuationTokenReader;
    private final ObjectWriter continuationTokenWriter;
    private final UnitsEnsemblesDao unitsEnsemblesDao;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public ResourceTypesService(ResourceTypesDao resourceTypesDao,
                                ProvidersDao providersDao,
                                YdbTableClient tableClient,
                                ResourceTypesLoader resourceTypesLoader,
                                ProvidersLoader providersLoader,
                                @Qualifier("messageSource") MessageSource messages,
                                SecurityManagerService securityManagerService,
                                @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
                                UnitsEnsemblesDao unitsEnsemblesDao) {
        this.resourceTypesDao = resourceTypesDao;
        this.providersDao = providersDao;
        this.tableClient = tableClient;
        this.resourceTypesLoader = resourceTypesLoader;
        this.providersLoader = providersLoader;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.continuationTokenReader = objectMapper.getObjectMapper().readerFor(ResourceTypeContinuationToken.class);
        this.continuationTokenWriter = objectMapper.getObjectMapper().writerFor(ResourceTypeContinuationToken.class);
        this.unitsEnsemblesDao = unitsEnsemblesDao;
    }

    public Mono<Result<ResourceTypeModel>> getById(String id, String providerId, YaUserDetails currentUser,
                                                   Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThen(v ->
                validateId(id, locale))
                .andThenMono(u -> validateProvider(providerId, locale)
                        .flatMap(r -> r.andThenMono(pr ->
                                tableClient.usingSessionMonoRetryable(session -> resourceTypesDao
                                        .getById(immediateTx(session), id, Tenants.DEFAULT_TENANT_ID)
                                        .map(t -> validateExists(t.orElse(null), pr.getId(), locale))
                                )
                        ))
                ));
    }

    public Mono<Result<Page<ResourceTypeModel>>> getPage(String providerId, PageRequest pageRequest,
                                                         YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u -> {
            Result<PageRequest.Validated<ResourceTypeContinuationToken>> pageValidation
                    = pageRequest.validate(continuationTokenReader, messages, locale);
            return pageValidation.andThenDo(p -> validateContinuationToken(p, locale)).andThenMono(p -> {
                int limit = p.getLimit();
                String fromId = p.getContinuationToken().map(ResourceTypeContinuationToken::getId).orElse(null);
                return validateProvider(providerId, locale)
                        .flatMap(r -> r.andThenMono(pr -> tableClient.usingSessionMonoRetryable(session ->
                                resourceTypesDao.getByProvider(immediateTx(session), pr.getId(),
                                        Tenants.DEFAULT_TENANT_ID, fromId, limit + 1, false)
                                        .map(values -> values.size() > limit
                                                ? Page.page(values.subList(0, limit),
                                                prepareToken(values.get(limit - 1)))
                                                : Page.lastPage(values))
                                        .map(Result::success))));
            });
        }));
    }

    public Mono<Result<ResourceTypeModel>> create(ResourceTypeCreateDto resourceType, String providerId,
                                                  YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThenMono(u -> {
            String newId = UUID.randomUUID().toString();
            return tableClient.usingSessionMonoRetryable(session ->
                    session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                            validateCreation(resourceType, providerId, newId, currentUser, locale, ts)
                                    .flatMap(r -> r.andThenMono(validated ->
                                                    resourceTypesDao.upsertResourceTypeRetryable(ts, validated)
                                                            .doOnSuccess(v -> resourceTypesLoader.update(validated))
                                                            .thenReturn(Result.success(validated))
                                            )
                                    )
                    )
            );
        });
    }

    public Mono<Result<ResourceTypeModel>> put(String id, String providerId, Long version,
                                               ResourceTypePutDto resourceTypePutDto,
                                               YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> resourceTypesDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), providerId, locale)
                                        .andThenMono(p -> validatePut(resourceTypePutDto, providerId, version, p,
                                                currentUser, locale, ts)),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return resourceTypesDao.updateResourceTypeRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> resourceTypesLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    public Mono<Result<ResourceTypeModel>> patch(String id, String providerId, Long version,
                                                 ResourceTypePatchDto resourceTypePatch,
                                                 YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkWritePermissionsForProvider(currentUser, locale).andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> resourceTypesDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> validateExists(r.orElse(null), providerId, locale)
                                        .andThenMono(p -> validatePatch(resourceTypePatch, providerId, version, p,
                                                currentUser, locale, ts)),
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return resourceTypesDao.updateResourceTypeRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> resourceTypesLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(m.getT1()));
                                            } else {
                                                return ts.commitTransaction().thenReturn(Result.success(m.getT1()));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private String prepareToken(ResourceTypeModel lastItem) {
        return ContinuationTokens.encode(new ResourceTypeContinuationToken(lastItem.getId()),
                continuationTokenWriter);
    }

    private Result<Void> validateId(String resourceId, Locale locale) {
        if (!Uuids.isValidUuid(resourceId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.resource.type.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateContinuationToken(PageRequest.Validated<ResourceTypeContinuationToken> pageRequest,
                                                   Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        return validateId(pageRequest.getContinuationToken().get().getId(), locale);
    }

    private Result<ResourceTypeModel> validateExists(ResourceTypeModel resourceType, String providerId,
                                                     Locale locale) {
        if (resourceType == null || resourceType.isDeleted() || !resourceType.getProviderId().equals(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.resource.type.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(resourceType);
    }

    private Mono<Result<ProviderModel>> validateProvider(String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersLoader.getProviderByIdImmediate(providerId, Tenants.DEFAULT_TENANT_ID).map(provider -> {
            if (provider.isEmpty() || provider.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.provider.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider.get());
        });
    }

    private Mono<Result<ProviderModel>> validateProvider(YdbTxSession session, String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersDao.getById(session, providerId, Tenants.DEFAULT_TENANT_ID).map(provider -> {
            if (provider.isEmpty() || provider.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.provider.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider.get());
        });
    }

    private Mono<Result<UnitsEnsembleModel>> validateUnitsEnsemble(
            YdbTxSession session, Optional<String> unitsEnsembleIdOptional, Locale locale
    ) {
        if (unitsEnsembleIdOptional.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError("unitsEnsembleId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        String unitsEnsembleId = unitsEnsembleIdOptional.get();
        if (unitsEnsembleId.isBlank()) {
            ErrorCollection error = ErrorCollection.builder().addError("unitsEnsembleId", TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (!Uuids.isValidUuid(unitsEnsembleId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.units.ensemble.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return unitsEnsemblesDao.getById(session, unitsEnsembleId, Tenants.DEFAULT_TENANT_ID).map(units -> {
            if (units.isEmpty() || units.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.units.ensemble.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(units.get());
        });
    }

    private Mono<Result<ResourceTypeModel>> validateCreation(ResourceTypeCreateDto resourceType, String providerId,
                                                             String newId, YaUserDetails currentUser, Locale locale,
                                                             YdbTxSession session) {
        return validateProvider(session, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        providerId, currentUser, locale, p
                ))
                .flatMap(result -> result.andThenMono(provider ->
                        validateUnitsEnsemble(session, resourceType.getUnitsEnsembleId(), locale)
                                .flatMap(unitsEnsembleResult ->
                        validateKey(session, resourceType::getKey, provider.getId(), "key", locale).map(key -> {
                            ResourceTypeModel.Builder builder = ResourceTypeModel.builder();
                            ErrorCollection.Builder errors = ErrorCollection.builder();
                            builder.id(newId);
                            builder.tenantId(Tenants.getTenantId(currentUser));
                            builder.deleted(false);
                            builder.version(0L);
                            builder.providerId(provider.getId());
                            key.doOnSuccess(builder::key);
                            key.doOnFailure(errors::add);
                            validateText(resourceType::getNameEn, builder::nameEn, errors, "nameEn",
                                    MAX_NAME_LENGTH, locale);
                            validateText(resourceType::getNameRu, builder::nameRu, errors, "nameRu",
                                    MAX_NAME_LENGTH, locale);
                            validateText(resourceType::getDescriptionEn, builder::descriptionEn, errors,
                                    "descriptionEn", MAX_DESCRIPTION_LENGTH, locale);
                            validateText(resourceType::getDescriptionRu, builder::descriptionRu, errors,
                                    "descriptionRu", MAX_DESCRIPTION_LENGTH, locale);
                            unitsEnsembleResult.doOnSuccess(unitsEnsemble -> builder
                                    .unitsEnsembleId(unitsEnsemble.getId())
                                    .baseUnitId(UnitsComparator.getBaseUnit(unitsEnsemble).getId()));
                            unitsEnsembleResult.doOnFailure(errors::add);
                            builder.sortingOrder(resourceType.getSortingOrder().orElse(null));
                            AggregationSettingsHelper.validateAggregationSettings(resourceType::getAggregationSettings,
                                    builder::aggregationSettings, errors, "aggregationSettings", locale,
                                    messages);
                            if (errors.hasAnyErrors()) {
                                return Result.failure(errors.build());
                            }
                            return Result.success(builder.build());
                        })
                )))
        );
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Result<Tuple2<ResourceTypeModel, Boolean>>> validatePut(ResourceTypePutDto resourceType,
                                                                         String providerId,
                                                                         Long version,
                                                                         ResourceTypeModel existingResourceType,
                                                                         YaUserDetails currentUser,
                                                                         Locale locale,
                                                                         YdbTxSession session) {
        return validateProvider(session, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        providerId, currentUser, locale, p
                ))
                .map(result -> result.andThen(provider -> {
                    ResourceTypeModel.Builder builder = ResourceTypeModel.builder(existingResourceType);
                    ErrorCollection.Builder errors = ErrorCollection.builder();
                    validateVersion(() -> Optional.ofNullable(version), existingResourceType::getVersion,
                            builder::version, errors, "version", locale);
                    validateText(resourceType::getNameEn, builder::nameEn, errors, "nameEn",
                            MAX_NAME_LENGTH, locale);
                    validateText(resourceType::getNameRu, builder::nameRu, errors, "nameRu",
                            MAX_NAME_LENGTH, locale);
                    validateText(resourceType::getDescriptionEn, builder::descriptionEn, errors,
                            "descriptionEn", MAX_DESCRIPTION_LENGTH, locale);
                    validateText(resourceType::getDescriptionRu, builder::descriptionRu, errors,
                            "descriptionRu", MAX_DESCRIPTION_LENGTH, locale);
                    builder.unitsEnsembleId(existingResourceType.getUnitsEnsembleId());
                    builder.sortingOrder(resourceType.getSortingOrder());
                    AggregationSettingsHelper.validateAggregationSettings(resourceType::getAggregationSettings,
                            builder::aggregationSettings, errors, "aggregationSettings", locale,
                            messages);
                    if (errors.hasAnyErrors()) {
                        return Result.failure(errors.build());
                    }
                    if (builder.hasChanges(existingResourceType)) {
                        return Result.success(Tuples.of(builder.build(), true));
                    }
                    return Result.success(Tuples.of(existingResourceType, false));
                })));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Result<Tuple2<ResourceTypeModel, Boolean>>> validatePatch(ResourceTypePatchDto resourceType,
                                                                           String providerId,
                                                                           Long version,
                                                                           ResourceTypeModel existingResourceType,
                                                                           YaUserDetails currentUser,
                                                                           Locale locale,
                                                                           YdbTxSession session) {
        return validateProvider(session, providerId, locale).flatMap(providerR -> providerR
                .andThenMono(p -> securityManagerService.checkWritePermissionsForProvider(
                        providerId, currentUser, locale, p
                ))
                .map(result -> result.andThen(provider -> {
                    ResourceTypeModel.Builder builder = ResourceTypeModel.builder(existingResourceType);
                    ErrorCollection.Builder errors = ErrorCollection.builder();

                    validateVersion(() -> Optional.ofNullable(version), existingResourceType::getVersion,
                            builder::version, errors, "version", locale);

                    resourceType.getNameEn().acceptNewValueOptionalToSetter(s -> validateText(() -> s, builder::nameEn,
                            errors, "nameEn", MAX_NAME_LENGTH, locale), existingResourceType.getNameEn());
                    resourceType.getNameRu().acceptNewValueOptionalToSetter(s -> validateText(() -> s, builder::nameRu,
                            errors, "nameRu", MAX_NAME_LENGTH, locale), existingResourceType.getNameRu());
                    resourceType.getDescriptionEn().acceptNewValueOptionalToSetter(s -> validateText(() -> s,
                            builder::descriptionEn, errors, "descriptionEn", MAX_NAME_LENGTH, locale),
                            existingResourceType.getDescriptionEn());
                    resourceType.getDescriptionRu().acceptNewValueOptionalToSetter(s -> validateText(() -> s,
                            builder::descriptionRu, errors, "descriptionRu", MAX_NAME_LENGTH, locale),
                            existingResourceType.getDescriptionRu());
                    resourceType.getSortingOrder().acceptNewValueToSetter(builder::sortingOrder,
                            existingResourceType.getSortingOrder());
                    resourceType.getAggregationSettings().acceptNewValueOptionalToSetter(
                            as -> AggregationSettingsHelper.validateAggregationSettings(() -> as,
                                    builder::aggregationSettings, errors, "aggregationSettings", locale,
                                    messages),
                            existingResourceType.getAggregationSettings()
                                    .map(AggregationSettingsHelper::toAggregationSettingsDto)
                                    .map(AggregationSettingsHelper::toAggregationSettingsInputDto)
                                    .orElse(null)
                    );

                    if (errors.hasAnyErrors()) {
                        return Result.failure(errors.build());
                    }
                    if (builder.hasChanges(existingResourceType)) {
                        return Result.success(Tuples.of(builder.build(), true));
                    }
                    return Result.success(Tuples.of(existingResourceType, false));
                })));
    }

    private void validateText(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private Mono<Result<String>> validateKey(YdbTxSession session, Supplier<Optional<String>> getter,
                                             String providerId, String fieldKey, Locale locale) {
        Optional<String> value = getter.get();
        if (value.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().isBlank()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().length() > MAX_KEY_LENGTH) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return resourceTypesDao.existsByProviderAndKey(session, providerId, Tenants.DEFAULT_TENANT_ID,
                value.get(), false).map(exists -> {
            if (exists) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey,
                        TypedError.conflict(messages
                                .getMessage("errors.resource.type.key.already.exists", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(value.get());
        });
    }

    private void validateVersion(Supplier<Optional<Long>> getter, Supplier<Long> existingGetter,
                                 Consumer<Long> setter, ErrorCollection.Builder errors, String fieldKey,
                                 Locale locale) {
        Optional<Long> value = getter.get();
        Long existingValue = existingGetter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (!Objects.equals(value.get(), existingValue)) {
            errors.addError(fieldKey, TypedError.versionMismatch(messages
                    .getMessage("errors.version.mismatch", null, locale)));
        } else {
            setter.accept(existingValue + 1L);
        }
    }

}
