package ru.yandex.intranet.d.services.security;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.services.ServiceLoader;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.users.StaffAffiliation;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.model.users.UserServiceRoles;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static it.unimi.dsi.fastutil.longs.LongSets.EMPTY_SET;
import static ru.yandex.intranet.d.model.users.UserServiceRoles.QUOTA_MANAGER;
import static ru.yandex.intranet.d.model.users.UserServiceRoles.RESPONSIBLE_OF_PROVIDER;
import static ru.yandex.intranet.d.model.users.UserServiceRoles.SERVICE_PRODUCT_DEPUTY_HEAD;
import static ru.yandex.intranet.d.model.users.UserServiceRoles.SERVICE_PRODUCT_HEAD;
import static ru.yandex.intranet.d.model.users.UserServiceRoles.SERVICE_RESPONSIBLE;
import static ru.yandex.intranet.d.util.result.Result.success;
import static ru.yandex.intranet.d.util.result.TypedError.forbidden;

/**
 * Security manager.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 * @since 23.10.2020
 */
@SuppressWarnings("checkstyle:SimplifyBooleanExpression")
@Component
public final class SecurityManagerService {

    private final MessageSource messages;
    private final ProvidersLoader providersLoader;
    private final ServiceLoader serviceLoader;

    public SecurityManagerService(@Qualifier("messageSource") MessageSource messages, ProvidersLoader providersLoader,
                                  ServiceLoader serviceLoader) {
        this.messages = messages;
        this.providersLoader = providersLoader;
        this.serviceLoader = serviceLoader;
    }

    private static boolean checkHasAnyOfRolesForProvider(YaUserDetails yaUserDetails, String providerId,
                                                         DRole... roles) {
        return Arrays.stream(roles)
                .anyMatch(role -> role.hasRoleForProvider(yaUserDetails, providerId));
    }

    private static boolean checkHasAnyOfRolesForFolder(YaUserDetails yaUserDetails, String folderId,
                                                       DRole... roles) {
        return Arrays.stream(roles)
                .anyMatch(role -> role.hasRoleForFolder(yaUserDetails, folderId));
    }

    private static Mono<Boolean> checkHasAnyOfRolesForService(YaUserDetails yaUserDetails, long serviceId,
                                                              DRole... roles) {
        return Mono.just(Arrays.stream(roles)
                .anyMatch(role -> role.hasRoleForService(yaUserDetails, serviceId)));
    }

    public <T> Mono<Result<T>> checkReadPermissions(YaUserDetails user, Locale locale, T payload) {
        if (isDAdmin(user) ||
                isYandexUser(user) ||
                isYandexRobot(user) ||
                isProvider(user) ||
                isAnyProviderAdmin(user)
        ) {
            return Mono.just(success(payload));
        }
        return Mono.just(accessDenied(locale));
    }

    public Mono<Result<Void>> checkReadPermissions(YaUserDetails user, Locale locale) {
        return checkReadPermissions(user, locale, null);
    }

    public <T> Mono<Result<T>> checkReadPermissions(long serviceId, YaUserDetails user, Locale locale, T payload) {
        return checkReadPermissions(user, locale, payload)
                .filter(Result::isSuccess)
                .switchIfEmpty(isQuotaManagerOrServiceResponsible(serviceId, user)
                        .map(b -> b ? success(payload) : accessDenied(locale)));
    }

    public Mono<Result<Void>> checkReadPermissions(long serviceId, YaUserDetails user, Locale locale) {
        return checkReadPermissions(serviceId, user, locale, null);
    }

    public <T> Mono<Result<T>> checkReadPermissions(
            String folderId, YaUserDetails user, Locale locale, T payload
    ) {
        // todo: check role for folder
        return checkReadPermissions(user, locale, payload);
    }

    public <T> Mono<Result<T>> checkReadPermissions(
            FolderModel folder, YaUserDetails user, Locale locale, T payload
    ) {
        // todo: check role for folder
        return checkReadPermissions(user, locale, payload);
    }

    public <T> Mono<Result<T>> checkReadPermissions(
            List<String> folderIds, YaUserDetails user, Locale locale, T payload
    ) {
        // todo: check role for folder
        return checkReadPermissions(user, locale, payload);
    }

    public Result<Void> checkDirectoryEndpointPermissions(YaUserDetails user, Locale locale) {
        return checkDirectoryEndpointPermissions(user, locale, null);
    }

    public <T> Result<T> checkDirectoryEndpointPermissions(YaUserDetails user, Locale locale, T payload) {
        if (isDAdmin(user)
                || isProvider(user)
                || isAnyProviderAdmin(user)) {
            return success(payload);
        }
        return accessDenied(locale);
    }

    public Mono<Result<Void>> checkDirectoryEndpointPermissions(
            String providerId, YaUserDetails user, Locale locale
    ) {
        return checkDirectoryEndpointPermissions(providerId, user, locale, null);
    }

    public <T> Mono<Result<T>> checkDirectoryEndpointPermissions(
            String providerId, YaUserDetails user, Locale locale, T payload
    ) {
        if (isDAdmin(user)
                || isProviderByTVM(providerId, user)) {
            return Mono.just(success(payload));
        }
        return isProviderAdmin(providerId, user)
                .map(isProviderAdmin -> isProviderAdmin ? success(payload) : accessDenied(locale));
    }

    public Mono<Result<Void>> checkWritePermissionsForProvider(
            String providerId, YaUserDetails user, Locale locale
    ) {
        return checkWritePermissionsForProvider(providerId, user, locale, null);
    }

    public Result<Void> checkWritePermissionsForProvider(
            YaUserDetails user, Locale locale
    ) {
        return checkWritePermissionsForProvider(user, locale, null);
    }

    public <T> Mono<Result<T>> checkWritePermissionsForProvider(
            String providerId, YaUserDetails user, Locale locale, T payload
    ) {
        if (isDAdmin(user)
                || isProviderByTVM(providerId, user)) {
            return Mono.just(success(payload));
        }
        return isProviderAdmin(providerId, user)
                .map(isProviderAdmin -> isProviderAdmin ? success(payload) : accessDenied(locale));
    }

    private boolean isProviderByTVM(String providerId, YaUserDetails user) {
        return isProvider(user)
                && user.toProvider(providerId, Tenants.getTenantId(user)).isPresent();
    }

    public <T> Result<T> checkWritePermissionsForProvider(
            YaUserDetails user, Locale locale, T payload
    ) {
        if (isDAdmin(user)
                || isProvider(user)
                || isAnyProviderAdmin(user)) {
            return success(payload);
        }

        return accessDenied(locale);
    }

    public Result<Void> checkDictionaryWritePermissions(YaUserDetails user, Locale locale) {
        return checkDictionaryWritePermissions(user, locale, null);
    }

    public <T> Result<T> checkDictionaryWritePermissions(YaUserDetails user, Locale locale, T payload) {
        if (isDAdmin(user)) {
            return success(payload);
        }
        return accessDenied(locale, "errors.access.d.admin.only");
    }

    public <T> Mono<Result<T>> checkWritePermissions(long serviceId, YaUserDetails user, Locale locale, T payload) {
        if (isDAdmin(user)) {
            return Mono.just(success(payload));
        }

        return isQuotaManagerOrServiceResponsible(serviceId, user)
                .map(b -> b ? success(payload) : accessDenied(locale));
    }

    public Mono<Result<Void>> checkWritePermissions(long serviceId, YaUserDetails user, Locale locale) {
        return checkWritePermissions(serviceId, user, locale, null);
    }

    public Mono<Map<Long, Boolean>> hasUserWritePermissionsByServiceId(YdbTxSession ts, Set<Long> serviceIds,
                                                                       YaUserDetails user) {
        if (isDAdmin(user)) {
            return Mono.just(Maps.asMap(serviceIds, id -> true));
        }

        Set<Long> servicesWithQuotaWriteAccessRoles = getServicesWithQuotaWriteAccessRoles(user);

        if (servicesWithQuotaWriteAccessRoles.isEmpty()) {
            return Mono.just(Maps.asMap(serviceIds, id -> false));
        }

        Set<Long> toCheck = Sets.difference(serviceIds, servicesWithQuotaWriteAccessRoles);

        if (!toCheck.isEmpty()) {
            Set<Long> directQuotaWriteAccess = Sets.intersection(servicesWithQuotaWriteAccessRoles, serviceIds);
            return serviceLoader.getAllParentsByIds(ts, toCheck, Tenants.getTenantId(user))
                    .map(parentsByServiceId -> serviceIds.stream()
                            .collect(Collectors.toMap(Function.identity(),
                                    id -> directQuotaWriteAccess.contains(id) ||
                                            parentsByServiceId.getOrDefault(id, EMPTY_SET).longStream()
                                                    .anyMatch(servicesWithQuotaWriteAccessRoles::contains)
                            )));
        }

        return Mono.just(Maps.asMap(serviceIds, id -> true));
    }

    public Mono<Result<Void>> checkImportPermissions(
            YaUserDetails currentUser, Set<ProviderModel> providers, Locale locale
    ) {
        if (isDAdmin(currentUser) || isProvidersByTVM(providers, currentUser)) {
            return Mono.just(Result.success(null));
        }

        return Flux.fromStream(providers.stream())
                .flatMap(providerModel -> isProviderAdmin(providerModel.getId(), currentUser))
                .all(b -> b)
                .map(b -> b ? Result.success(null) : accessDenied(locale));
    }

    public Mono<Result<Void>> checkProvisionAndAccountPermissions(long serviceId,
                                                                  ProviderModel provider,
                                                                  YaUserDetails currentUser,
                                                                  Locale locale) {
        if (isDAdmin(currentUser)) {
            return Mono.just(success(null));
        }
        return isQuotaManagerOrServiceResponsible(serviceId, currentUser).flatMap(allowed -> {
            if (allowed) {
                return Mono.just(success(null));
            }
            return isProviderAdmin(provider, currentUser)
                    .map(providerAdmin -> providerAdmin ? success(null) : accessDenied(locale));
        });
    }

    public Mono<List<ProviderModel>> filterProvidersWithProviderAdminRole(List<ProviderModel> providers,
                                                                          YaUserDetails currentUser) {
        // Return Mono because later we are going to read roles directly from DB
        Set<Long> providerResponsibleServices = getServiceIdsByRole(currentUser, RESPONSIBLE_OF_PROVIDER);
        return Mono.just(providers.stream()
                .filter(provider -> providerResponsibleServices.contains(provider.getServiceId()))
                .collect(Collectors.toList()));
    }

    private boolean isProvidersByTVM(Set<ProviderModel> providers, YaUserDetails currentUser) {
        return providers.stream()
                .allMatch(providerModel -> isProviderByTVM(providerModel.getId(), currentUser));
    }

    public Result<Void> checkImportPermissions(YaUserDetails currentUser, Locale locale) {
        if (isProvider(currentUser)
                || isAnyProviderAdmin(currentUser)
                || isDAdmin(currentUser)) {
            return Result.success(null);
        }
        return accessDenied(locale);
    }

    private <T> Result<T> accessDenied(Locale locale) {
        return accessDenied(locale, "errors.access.denied");
    }

    private <T> Result<T> accessDenied(Locale locale, String messageCode) {
        return Result.failure(ErrorCollection.builder()
                .addError(forbidden(messages.getMessage(messageCode, null, locale)))
                .build());
    }

    private boolean isDAdmin(YaUserDetails user) {
        return user.getUser().map(UserModel::getDAdmin)
                .orElse(false);
    }

    private boolean isYandexUser(YaUserDetails user) {
        return user.getUser().flatMap(UserModel::getStaffAffiliation).map(StaffAffiliation.YANDEX::equals)
                .orElse(false);
    }

    private boolean isYandexRobot(YaUserDetails user) {
        return user.getUser().flatMap(UserModel::getStaffRobot)
                .orElse(false);
    }

    private boolean isProvider(YaUserDetails user) {
        return !user.getProviders().isEmpty();
    }

    private Mono<Boolean> isProviderAdmin(String providerId, YaUserDetails user) {
        Set<Long> userServiceIdsWithProviderAdminRole = getServiceIdsByRole(user, RESPONSIBLE_OF_PROVIDER);
        if (userServiceIdsWithProviderAdminRole.isEmpty()) {
            return Mono.just(Boolean.FALSE);
        }
        return providersLoader.getProviderByIdImmediate(providerId, Tenants.DEFAULT_TENANT_ID)
                .map(providerModel -> {
                    if (providerModel.isEmpty()) {
                        return false;
                    }
                    long serviceId = providerModel.get().getServiceId();
                    return userServiceIdsWithProviderAdminRole.contains(serviceId);
                });
    }

    private Mono<Boolean> isProviderAdmin(ProviderModel provider, YaUserDetails user) {
        // Return Mono because later we are going to read roles directly from DB
        return Mono.just(getServiceIdsByRole(user, RESPONSIBLE_OF_PROVIDER).contains(provider.getServiceId()));
    }

    private Mono<Boolean> isUserHaveRoleInServiceParent(Set<Long> userServiceIdsWithRole, long serviceId) {
        return serviceLoader.getAllParentsImmediate(serviceId, Tenants.DEFAULT_TENANT_ID)
                .map(parentsIds -> parentsIds.longStream()
                        .anyMatch(userServiceIdsWithRole::contains));
    }

    private boolean isAnyProviderAdmin(YaUserDetails user) {
        return user.getUser()
                .map(userModel -> !userModel.getRoles().getOrDefault(RESPONSIBLE_OF_PROVIDER, Set.of()).isEmpty())
                .orElse(false);
    }

    private Mono<Boolean> isQuotaManagerOrServiceResponsible(long serviceId, YaUserDetails user) {
        Set<Long> servicesWithQuotaWriteAccessRoles = getServicesWithQuotaWriteAccessRoles(user);

        if (servicesWithQuotaWriteAccessRoles.isEmpty()) {
            return Mono.just(Boolean.FALSE);
        }

        if (servicesWithQuotaWriteAccessRoles.contains(serviceId)) {
            return Mono.just(Boolean.TRUE);
        }

        return isUserHaveRoleInServiceParent(servicesWithQuotaWriteAccessRoles, serviceId);
    }

    private Set<Long> getServiceIdsByRole(YaUserDetails user, UserServiceRoles userServiceRoles) {
        return user.getUser()
                .map(userModel -> userModel.getRoles().getOrDefault(userServiceRoles, Set.of()))
                .orElse(Set.of());
    }

    private Set<Long> getServicesWithQuotaWriteAccessRoles(YaUserDetails user) {
        Set<Long> result = new HashSet<>();
        result.addAll(getServiceIdsByRole(user, QUOTA_MANAGER));
        result.addAll(getServiceIdsByRole(user, SERVICE_PRODUCT_HEAD));
        result.addAll(getServiceIdsByRole(user, SERVICE_PRODUCT_DEPUTY_HEAD));
        result.addAll(getServiceIdsByRole(user, SERVICE_RESPONSIBLE));
        return result;
    }

}
