package ru.yandex.intranet.d.services.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Lists;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.loans.ServiceLoansInDao;
import ru.yandex.intranet.d.dao.loans.ServiceLoansOutDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.loans.LoanStatus;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceReadOnlyState;
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.services.ClosingServiceDto;
import ru.yandex.intranet.d.web.model.services.ValueByServiceIdDto;

/**
 * Service for ABC Services
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */

@Component
public class AbcServicesService {

    private final ServicesDao servicesDao;
    private final FolderDao folderDao;
    private final QuotasDao quotasDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final OperationsInProgressDao operationsInProgressDao;
    private final YdbTableClient ydbTableClient;
    private final MessageSource messages;
    private final ResourcesLoader resourcesLoader;
    private final ServiceLoansInDao serviceLoansInDao;
    private final ServiceLoansOutDao serviceLoansOutDao;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public AbcServicesService(ServicesDao servicesDao, FolderDao folderDao, QuotasDao quotasDao,
                              AccountsQuotasDao accountsQuotasDao, OperationsInProgressDao operationsInProgressDao,
                              YdbTableClient ydbTableClient, @Qualifier("messageSource") MessageSource messages,
                              ResourcesLoader resourcesLoader, ServiceLoansInDao serviceLoansInDao,
                              ServiceLoansOutDao serviceLoansOutDao) {
        this.servicesDao = servicesDao;
        this.folderDao = folderDao;
        this.quotasDao = quotasDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.operationsInProgressDao = operationsInProgressDao;
        this.ydbTableClient = ydbTableClient;
        this.messages = messages;
        this.resourcesLoader = resourcesLoader;
        this.serviceLoansInDao = serviceLoansInDao;
        this.serviceLoansOutDao = serviceLoansOutDao;
    }

    public Mono<Result<ClosingServiceDto>> checkClosing(List<Long> serviceIds, TenantId tenantId, boolean ignoreState,
                                                         Locale locale) {
        if (serviceIds == null || serviceIds.isEmpty()) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError("serviceIds",
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)))
                    .build()));
        }
        return ydbTableClient.usingSessionMonoRetryable(session -> Flux.fromIterable(Lists.partition(serviceIds, 500))
                .concatMap(v -> servicesDao.getServiceStatesByIds(
                        session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY), v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()))
                .flatMap(services -> validateServices(serviceIds, services, locale)
                .andThenMono(u -> folderDao.getAllFoldersByServiceIds(
                        session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                        serviceIds.stream().map(id -> new WithTenant<>(tenantId, id)).collect(Collectors.toSet()))
                .flatMap(folders -> (folders.get().isEmpty() ? Mono.just(List.<QuotaModel>of()) :
                        quotasDao.getByFolders(
                            session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                            folders.get().stream().map(FolderModel::getId).collect(Collectors.toList()),
                            tenantId))
                .flatMap(quotas -> (folders.get().isEmpty() ?
                        Mono.just(new WithTxId<List<AccountsQuotasModel>>(List.of(), null)) :
                        accountsQuotasDao.getAllByFolderIds(
                            session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                            tenantId,
                            folders.get().stream().map(FolderModel::getId).collect(Collectors.toSet())))
                .flatMap(accountQuotas ->
                        (folders.get().isEmpty() ? Mono.just(List.<ResourceModel>of()) :
                            resourcesLoader.getResourcesByIds(
                                session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                                Stream.concat(
                                        quotas.stream().map(QuotaModel::getResourceId),
                                        accountQuotas.get().stream().map(AccountsQuotasModel::getResourceId))
                                    .map(resourceId -> Tuples.of(resourceId, tenantId))
                                    .collect(Collectors.toList())))
                .flatMap(resources ->
                        operationsInProgressDao.getAllByTenantAccounts(
                        session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                        tenantId,
                        accountQuotas.get().stream().map(AccountsQuotasModel::getAccountId).collect(Collectors.toSet()))
                .flatMap(operationOnProgress ->
                        serviceLoansInDao.filterServiceIdsByLoanStatusMono(
                                session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                                tenantId, serviceIds, LoanStatus.PENDING)
                .flatMap(servicesWithPendingLoansIn ->
                        serviceLoansOutDao.filterServiceIdsByLoanStatusMono(
                                session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                                tenantId, serviceIds, LoanStatus.PENDING)
                .map(servicesWithPendingLoansOut -> {
                    Set<Long> servicesWithPendingLoans = new HashSet<>(servicesWithPendingLoansIn);
                    servicesWithPendingLoans.addAll(servicesWithPendingLoansOut);
                    return createClosingServiceDto(services, folders.get(), quotas, accountQuotas.get(),
                            resources, operationOnProgress, servicesWithPendingLoans, ignoreState);
                }))))))))));
    }

    private Result<List<ServiceWithStatesModel>> validateServices(List<Long> serviceIds,
                                                                  List<ServiceWithStatesModel> services,
                                                                  Locale locale) {
        Set<Long> serviceFoundIds = services.stream().map(ServiceWithStatesModel::getId).collect(Collectors.toSet());
        for (int i = 0; i < serviceIds.size(); i++) {
            if (!serviceFoundIds.contains(serviceIds.get(i))) {
                String fieldKey = "serviceIds." + i;
                return Result.failure(ErrorCollection.builder().addError(fieldKey, TypedError.notFound(messages
                        .getMessage("errors.service.not.found", null, locale)))
                        .build());
            }
        }
        return Result.success(services);
    }

    @SuppressWarnings("ParameterNumber")
    private Result<ClosingServiceDto> createClosingServiceDto(
            List<ServiceWithStatesModel> services,
            List<FolderModel> folders,
            List<QuotaModel> quotas,
            List<AccountsQuotasModel> accountsQuotas,
            List<ResourceModel> resources,
            List<OperationInProgressModel> operationsInProgress,
            Set<Long> servicesWithPendingLoans,
            boolean ignoreState) {
        Map<Long, List<FolderModel>> foldersByServiceId = folders.stream()
                .collect(Collectors.groupingBy(FolderModel::getServiceId));
        Map<String, List<QuotaModel>> quotasByFolderId = quotas.stream()
                .collect(Collectors.groupingBy(QuotaModel::getFolderId));
        Map<String, List<AccountsQuotasModel>> accountsQuotasByFolderId = accountsQuotas.stream()
                .collect(Collectors.groupingBy(AccountsQuotasModel::getFolderId));
        Set<String> virtualResourcesIdsSet = resources.stream()
                .filter(ResourceModel::isVirtual)
                .map(ResourceModel::getId)
                .collect(Collectors.toSet());
        List<ValueByServiceIdDto> listServiceWithAllowedClosingOperation = new ArrayList<>();
        for (ServiceWithStatesModel service : services) {
            boolean allowedClosing = ignoreState || service.getReadOnlyState() == ServiceReadOnlyState.CLOSING ||
                    service.getReadOnlyState() == ServiceReadOnlyState.DELETING;
            allowedClosing &= foldersByServiceId.getOrDefault(service.getId(), List.of()).stream()
                    .allMatch(folder -> Stream.concat(
                                    quotasByFolderId.getOrDefault(folder.getId(), List.of()).stream()
                                            .filter(quota -> quota.getQuota() != 0)
                                            .map(QuotaModel::getResourceId),
                                    accountsQuotasByFolderId.getOrDefault(folder.getId(), List.of()).stream()
                                            .filter(accountsQuota -> accountsQuota.getProvidedQuota() != 0)
                                            .map(AccountsQuotasModel::getResourceId))
                            .distinct()
                            .allMatch(virtualResourcesIdsSet::contains));
            allowedClosing &= operationsInProgress.isEmpty();
            allowedClosing &= !servicesWithPendingLoans.contains(service.getId());
            listServiceWithAllowedClosingOperation.add(new ValueByServiceIdDto(service.getId(), allowedClosing));
        }
        return Result.success(new ClosingServiceDto(listServiceWithAllowedClosingOperation));
    }
}
