package ru.yandex.intranet.d.services.settings

import kotlinx.coroutines.reactor.awaitSingle
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.providers.ProvidersDao
import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao
import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao
import ru.yandex.intranet.d.dao.resources.types.ResourceTypesDao
import ru.yandex.intranet.d.dao.settings.RuntimeSettingsDao
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.model.settings.KnownProviders
import ru.yandex.intranet.d.model.settings.RateLimiterSettings
import ru.yandex.intranet.d.model.settings.SettingsKey
import ru.yandex.intranet.d.model.settings.YpUsageSyncSettings
import ru.yandex.intranet.d.model.settings.YtUsageSyncSettings
import ru.yandex.intranet.d.util.Uuids
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.settings.KnownProvidersDto
import ru.yandex.intranet.d.web.model.settings.RateLimiterSettingsDto
import ru.yandex.intranet.d.web.model.settings.YpUsageSyncSettingsDto
import ru.yandex.intranet.d.web.model.settings.YtUsageSyncSettingsDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.util.*

/**
 * Runtime settings admin service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class RuntimeSettingsAdminService(
    private val tableClient: YdbTableClient,
    private val runtimeSettingsDao: RuntimeSettingsDao,
    private val providersDao: ProvidersDao,
    private val resourceTypesDao: ResourceTypesDao,
    private val resourceSegmentationsDao: ResourceSegmentationsDao,
    private val resourceSegmentsDao: ResourceSegmentsDao,
    private val unitsEnsemblesDao: UnitsEnsemblesDao,
    @Qualifier("messageSource") private val messages: MessageSource
) {

    fun getKnownProvidersMono(user: YaUserDetails, locale: Locale): Mono<Result<KnownProvidersDto>> {
        return mono { getKnownProviders(user, locale) }
    }

    fun getYtUsageSyncSettingsMono(user: YaUserDetails, locale: Locale): Mono<Result<YtUsageSyncSettingsDto>> {
        return mono { getYtUsageSyncSettings(user, locale) }
    }

    fun getYpUsageSyncSettingsMono(user: YaUserDetails, locale: Locale): Mono<Result<YpUsageSyncSettingsDto>> {
        return mono { getYpUsageSyncSettings(user, locale) }
    }

    fun getRateLimiterSettingsMono(user: YaUserDetails, locale: Locale): Mono<Result<RateLimiterSettingsDto>> {
        return mono { getRateLimiterSettings(user, locale) }
    }

    fun putKnownProvidersMono(settings: KnownProvidersDto, user: YaUserDetails,
                              locale: Locale): Mono<Result<KnownProvidersDto>> {
        return mono { putKnownProviders(settings, user, locale) }
    }

    fun putYtUsageSyncSettingsMono(settings: YtUsageSyncSettingsDto, user: YaUserDetails,
                                   locale: Locale): Mono<Result<YtUsageSyncSettingsDto>> {
        return mono { putYtUsageSyncSettings(settings, user, locale) }
    }

    fun putYpUsageSyncSettingsMono(settings: YpUsageSyncSettingsDto, user: YaUserDetails,
                                   locale: Locale): Mono<Result<YpUsageSyncSettingsDto>> {
        return mono { putYpUsageSyncSettings(settings, user, locale) }
    }

    fun putRateLimiterSettingsMono(settings: RateLimiterSettingsDto, user: YaUserDetails,
                              locale: Locale): Mono<Result<RateLimiterSettingsDto>> {
        return mono { putRateLimiterSettings(settings, user, locale) }
    }

    private suspend fun getKnownProviders(user: YaUserDetails, locale: Locale): Result<KnownProvidersDto> = binding {
        validatePermissions(user, locale).bind()
        val settings = dbSessionRetryable(tableClient) {
            runtimeSettingsDao.get(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, SettingsKey.KNOWN_PROVIDERS)
        }
        return if (settings == null) {
            Result.success(KnownProvidersDto(
                yt = null,
                yp = null,
                ypPreStable = null
            ))
        } else {
            Result.success(KnownProvidersDto(
                yt = settings.yt,
                yp = settings.yp,
                ypPreStable = settings.ypPreStable
            ))
        }
    }

    private suspend fun getYtUsageSyncSettings(user: YaUserDetails, locale: Locale): Result<YtUsageSyncSettingsDto> = binding {
        validatePermissions(user, locale).bind()
        val settings = dbSessionRetryable(tableClient) {
            runtimeSettingsDao.get(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, SettingsKey.YT_USAGE_SYNC_SETTINGS)
        }
        return if (settings == null) {
            Result.success(YtUsageSyncSettingsDto(
                cpuStrongResourceType = null,
                poolTreeSegmentation = null,
                physicalPoolTreeSegment = null,
                clusterSegmentation = null,
                scopeSegmentation = null,
                computeScopeSegment = null,
                syncEnabled = null,
                cpuMetricGridSpacingSeconds = null,
                cpuMetricUnitsEnsemble = null,
                cpuMetricUnit = null
            ))
        } else {
            Result.success(YtUsageSyncSettingsDto(
                cpuStrongResourceType = settings.cpuStrongResourceType,
                poolTreeSegmentation = settings.poolTreeSegmentation,
                physicalPoolTreeSegment = settings.physicalPoolTreeSegment,
                clusterSegmentation = settings.clusterSegmentation,
                scopeSegmentation = settings.scopeSegmentation,
                computeScopeSegment = settings.computeScopeSegment,
                syncEnabled = settings.syncEnabled,
                cpuMetricGridSpacingSeconds = settings.cpuMetricGridSpacingSeconds,
                cpuMetricUnitsEnsemble = settings.cpuMetricUnitsEnsemble,
                cpuMetricUnit = settings.cpuMetricUnit
            ))
        }
    }

    private suspend fun getYpUsageSyncSettings(user: YaUserDetails, locale: Locale): Result<YpUsageSyncSettingsDto> = binding {
        validatePermissions(user, locale).bind()
        val settings = dbSessionRetryable(tableClient) {
            runtimeSettingsDao.get(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, SettingsKey.YP_USAGE_SYNC_SETTINGS)
        }
        return if (settings == null) {
            Result.success(YpUsageSyncSettingsDto(
                cpuResourceType = null,
                preStableCpuResourceType = null,
                clusterSegmentation = null,
                nodeSegmentSegmentation = null,
                preStableClusterSegmentation = null,
                preStableNodeSegmentSegmentation = null,
                defaultNodeSegment = null,
                devNodeSegment = null,
                preStableDefaultNodeSegment = null,
                preStableDevNodeSegment = null,
                coresUnitsEnsemble = null,
                coresUnit = null,
                syncEnabled = null
            ))
        } else {
            Result.success(YpUsageSyncSettingsDto(
                cpuResourceType = settings.cpuResourceType,
                preStableCpuResourceType = settings.preStableCpuResourceType,
                clusterSegmentation = settings.clusterSegmentation,
                nodeSegmentSegmentation = settings.nodeSegmentSegmentation,
                preStableClusterSegmentation = settings.preStableClusterSegmentation,
                preStableNodeSegmentSegmentation = settings.preStableNodeSegmentSegmentation,
                defaultNodeSegment = settings.defaultNodeSegment,
                devNodeSegment = settings.devNodeSegment,
                preStableDefaultNodeSegment = settings.preStableDefaultNodeSegment,
                preStableDevNodeSegment = settings.preStableDevNodeSegment,
                coresUnitsEnsemble = settings.coresUnitsEnsemble,
                coresUnit = settings.coresUnit,
                syncEnabled = settings.syncEnabled
            ))
        }
    }

    private suspend fun getRateLimiterSettings(user: YaUserDetails, locale: Locale): Result<RateLimiterSettingsDto> = binding {
        validatePermissions(user, locale).bind()
        val settings = dbSessionRetryable(tableClient) {
            runtimeSettingsDao.get(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, SettingsKey.RATE_LIMITER_SETTINGS)
        }
        return if (settings == null) {
            Result.success(RateLimiterSettingsDto(
                disabled = false
            ))
        } else {
            Result.success(RateLimiterSettingsDto(
                disabled = settings.disabled
            ))
        }
    }

    private suspend fun putKnownProviders(settings: KnownProvidersDto, user: YaUserDetails,
                                          locale: Locale): Result<KnownProvidersDto> = binding {
        validatePermissions(user, locale).bind()
        preValidateKnownProviders(settings, locale).bind()
        val newValue = validateKnownProviders(settings, locale).bind()!!
        dbSessionRetryable(tableClient) {
            runtimeSettingsDao.upsertOneRetryable(rwSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID,
                SettingsKey.KNOWN_PROVIDERS, newValue)
        }
        Result.success(KnownProvidersDto(
            yt = newValue.yt,
            yp = newValue.yp,
            ypPreStable = newValue.ypPreStable
        ))
    }

    private suspend fun putYtUsageSyncSettings(settings: YtUsageSyncSettingsDto, user: YaUserDetails,
                                               locale: Locale): Result<YtUsageSyncSettingsDto> = binding {
        validatePermissions(user, locale).bind()
        preValidateYtUsageSyncSettings(settings, locale).bind()
        val newValue = validateYtUsageSyncSettings(settings, locale).bind()!!
        dbSessionRetryable(tableClient) {
            runtimeSettingsDao.upsertOneRetryable(rwSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID,
                SettingsKey.YT_USAGE_SYNC_SETTINGS, newValue)
        }
        Result.success(YtUsageSyncSettingsDto(
            cpuStrongResourceType = newValue.cpuStrongResourceType,
            poolTreeSegmentation = newValue.poolTreeSegmentation,
            physicalPoolTreeSegment = newValue.physicalPoolTreeSegment,
            clusterSegmentation = newValue.clusterSegmentation,
            scopeSegmentation = newValue.scopeSegmentation,
            computeScopeSegment = newValue.computeScopeSegment,
            syncEnabled = newValue.syncEnabled,
            cpuMetricGridSpacingSeconds = newValue.cpuMetricGridSpacingSeconds,
            cpuMetricUnitsEnsemble = newValue.cpuMetricUnitsEnsemble,
            cpuMetricUnit = newValue.cpuMetricUnit
        ))
    }

    private suspend fun putYpUsageSyncSettings(settings: YpUsageSyncSettingsDto, user: YaUserDetails,
                                               locale: Locale): Result<YpUsageSyncSettingsDto> = binding {
        validatePermissions(user, locale).bind()
        preValidateYpUsageSyncSettings(settings, locale).bind()
        val newValue = validateYpUsageSyncSettings(settings, locale).bind()!!
        dbSessionRetryable(tableClient) {
            runtimeSettingsDao.upsertOneRetryable(rwSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID,
                SettingsKey.YP_USAGE_SYNC_SETTINGS, newValue)
        }
        Result.success(YpUsageSyncSettingsDto(
            cpuResourceType = settings.cpuResourceType,
            preStableCpuResourceType = settings.preStableCpuResourceType,
            clusterSegmentation = settings.clusterSegmentation,
            nodeSegmentSegmentation = settings.nodeSegmentSegmentation,
            preStableClusterSegmentation = settings.preStableClusterSegmentation,
            preStableNodeSegmentSegmentation = settings.preStableNodeSegmentSegmentation,
            defaultNodeSegment = settings.defaultNodeSegment,
            devNodeSegment = settings.devNodeSegment,
            preStableDefaultNodeSegment = settings.preStableDefaultNodeSegment,
            preStableDevNodeSegment = settings.preStableDevNodeSegment,
            coresUnitsEnsemble = settings.coresUnitsEnsemble,
            coresUnit = settings.coresUnit,
            syncEnabled = settings.syncEnabled
        ))
    }

    private suspend fun putRateLimiterSettings(settings: RateLimiterSettingsDto, user: YaUserDetails,
                                               locale: Locale): Result<RateLimiterSettingsDto> = binding {
        validatePermissions(user, locale).bind()
        preValidateRateLimiterSettings(settings, locale).bind()
        val newValue = RateLimiterSettings(
            disabled = settings.disabled!!
        )
        dbSessionRetryable(tableClient) {
            runtimeSettingsDao.upsertOneRetryable(rwSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID,
                SettingsKey.RATE_LIMITER_SETTINGS, newValue)
        }
        Result.success(RateLimiterSettingsDto(
            disabled = newValue.disabled
        ))
    }

    private fun validatePermissions(user: YaUserDetails, locale: Locale): Result<Unit> {
        if (user.user.isEmpty || !user.user.get().dAdmin) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                .getMessage("errors.access.denied", null, locale))).build())
        }
        return Result.success(Unit)
    }

    private fun preValidateKnownProviders(settings: KnownProvidersDto, locale: Locale): Result<Unit> {
        val errors = ErrorCollection.builder()
        validateId(settings.yt, "yt", errors, locale, "errors.provider.not.found", true)
        validateId(settings.yp, "yp", errors, locale, "errors.provider.not.found", true)
        validateId(settings.ypPreStable, "ypPreStable", errors, locale, "errors.provider.not.found", false)
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(Unit)
        }
    }

    private fun preValidateRateLimiterSettings(settings: RateLimiterSettingsDto, locale: Locale): Result<Unit> {
        val errors = ErrorCollection.builder()
        if (settings.disabled == null) {
            errors.addError("disabled", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(Unit)
        }
    }

    private fun preValidateYtUsageSyncSettings(settings: YtUsageSyncSettingsDto, locale: Locale): Result<Unit> {
        val errors = ErrorCollection.builder()
        validateId(settings.cpuStrongResourceType, "cpuStrongResourceType", errors, locale,
            "errors.resource.type.not.found", true)
        validateId(settings.poolTreeSegmentation, "poolTreeSegmentation", errors, locale,
            "errors.resource.segmentation.not.found", true)
        validateId(settings.physicalPoolTreeSegment, "physicalPoolTreeSegment", errors, locale,
            "errors.resource.segment.not.found", true)
        validateId(settings.clusterSegmentation, "clusterSegmentation", errors, locale,
            "errors.resource.segmentation.not.found", true)
        validateId(settings.scopeSegmentation, "scopeSegmentation", errors, locale,
            "errors.resource.segmentation.not.found", true)
        validateId(settings.computeScopeSegment, "computeScopeSegment", errors, locale,
            "errors.resource.segment.not.found", true)
        validateId(settings.cpuMetricUnitsEnsemble, "cpuMetricUnitsEnsemble", errors, locale,
            "errors.units.ensemble.not.found", true)
        validateId(settings.cpuMetricUnit, "cpuMetricUnit", errors, locale, "errors.unit.not.found", true)
        if (settings.syncEnabled == null) {
            errors.addError("syncEnabled", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (settings.cpuMetricGridSpacingSeconds == null) {
            errors.addError("cpuMetricGridSpacingSeconds", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(Unit)
        }
    }

    private fun preValidateYpUsageSyncSettings(settings: YpUsageSyncSettingsDto, locale: Locale): Result<Unit> {
        val errors = ErrorCollection.builder()
        validateId(settings.cpuResourceType, "cpuResourceType", errors, locale,
            "errors.resource.type.not.found", true)
        validateId(settings.preStableCpuResourceType, "cpuResourceType", errors, locale,
            "errors.resource.type.not.found", false)
        validateId(settings.clusterSegmentation, "clusterSegmentation", errors, locale,
            "errors.resource.segmentation.not.found", true)
        validateId(settings.nodeSegmentSegmentation, "nodeSegmentSegmentation", errors, locale,
            "errors.resource.segmentation.not.found", true)
        validateId(settings.preStableClusterSegmentation, "preStableClusterSegmentation", errors, locale,
            "errors.resource.segmentation.not.found", false)
        validateId(settings.preStableNodeSegmentSegmentation, "preStableNodeSegmentSegmentation", errors, locale,
            "errors.resource.segmentation.not.found", false)
        validateId(settings.defaultNodeSegment, "defaultNodeSegment", errors, locale,
            "errors.resource.segment.not.found", true)
        validateId(settings.devNodeSegment, "devNodeSegment", errors, locale,
            "errors.resource.segment.not.found", true)
        validateId(settings.preStableDefaultNodeSegment, "preStableDefaultNodeSegment", errors, locale,
            "errors.resource.segment.not.found", false)
        validateId(settings.preStableDevNodeSegment, "preStableDevNodeSegment", errors, locale,
            "errors.resource.segment.not.found", false)
        validateId(settings.coresUnitsEnsemble, "coresUnitsEnsemble", errors, locale,
            "errors.units.ensemble.not.found", true)
        validateId(settings.coresUnit, "coresUnit", errors, locale, "errors.unit.not.found", true)
        if (settings.syncEnabled == null) {
            errors.addError("syncEnabled", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(Unit)
        }
    }

    private fun validateId(id: String?, key: String, errors: ErrorCollection.Builder, locale: Locale, errorKey: String,
                           required: Boolean) {
        if (id == null && required) {
            errors.addError(key, TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else if (id != null) {
            if (!Uuids.isValidUuid(id)) {
                errors.addError(key, TypedError.invalid(messages
                    .getMessage(errorKey, null, locale)))
            }
        }
    }

    private suspend fun validateKnownProviders(settings: KnownProvidersDto, locale: Locale): Result<KnownProviders> {
        val errors = ErrorCollection.builder()
        validateProviderId(settings.yt!!, "yt", errors, locale)
        validateProviderId(settings.yp!!, "yp", errors, locale)
        validateProviderId(settings.ypPreStable, "ypPreStable", errors, locale)
        if ((settings.ypPreStable != null && setOf(settings.yt, settings.yp, settings.ypPreStable).size < 3)
            || (settings.ypPreStable == null && setOf(settings.yt, settings.yp).size < 2)) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.provider.id.duplicate", null, locale)))
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(KnownProviders(
                yt = settings.yt,
                yp = settings.yp,
                ypPreStable = settings.ypPreStable
            ))
        }
    }

    private suspend fun validateProviderId(id: String?, key: String, errors: ErrorCollection.Builder,
                                           locale: Locale) {
        if (id == null) {
            return
        }
        val provider = dbSessionRetryable(tableClient) {
            providersDao.getById(roStaleSingleRetryableCommit(), id, Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (provider.isEmpty || provider.get().isDeleted) {
            errors.addError(key, TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale)))
        }
    }

    private suspend fun validateYtUsageSyncSettings(settings: YtUsageSyncSettingsDto, locale: Locale): Result<YtUsageSyncSettings> {
        val errors = ErrorCollection.builder()
        val knownProviders = dbSessionRetryable(tableClient) {
            runtimeSettingsDao.get(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, SettingsKey.KNOWN_PROVIDERS)
        }
        if (knownProviders == null) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale)))
            return Result.failure(errors.build())
        }
        val cpuStrongResourceType = dbSessionRetryable(tableClient) {
            resourceTypesDao.getById(roStaleSingleRetryableCommit(), settings.cpuStrongResourceType!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (cpuStrongResourceType.isEmpty || cpuStrongResourceType.get().isDeleted
            || cpuStrongResourceType.get().providerId != knownProviders.yt) {
            errors.addError("cpuStrongResourceType", TypedError.invalid(messages
                .getMessage("errors.resource.type.not.found", null, locale)))
        }
        val poolTreeSegmentation = dbSessionRetryable(tableClient) {
            resourceSegmentationsDao.getById(roStaleSingleRetryableCommit(), settings.poolTreeSegmentation!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (poolTreeSegmentation.isEmpty || poolTreeSegmentation.get().isDeleted
            || poolTreeSegmentation.get().providerId != knownProviders.yt) {
            errors.addError("poolTreeSegmentation", TypedError.invalid(messages
                .getMessage("errors.resource.segmentation.not.found", null, locale)))
        }
        val physicalPoolTreeSegment = dbSessionRetryable(tableClient) {
            resourceSegmentsDao.getById(roStaleSingleRetryableCommit(), settings.physicalPoolTreeSegment!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (physicalPoolTreeSegment.isEmpty || physicalPoolTreeSegment.get().isDeleted
            || physicalPoolTreeSegment.get().segmentationId != settings.poolTreeSegmentation) {
            errors.addError("physicalPoolTreeSegment", TypedError.invalid(messages
                .getMessage("errors.resource.segment.not.found", null, locale)))
        }
        val clusterSegmentation = dbSessionRetryable(tableClient) {
            resourceSegmentationsDao.getById(roStaleSingleRetryableCommit(), settings.clusterSegmentation!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (clusterSegmentation.isEmpty || clusterSegmentation.get().isDeleted
            || clusterSegmentation.get().providerId != knownProviders.yt) {
            errors.addError("clusterSegmentation", TypedError.invalid(messages
                .getMessage("errors.resource.segmentation.not.found", null, locale)))
        }
        val scopeSegmentation = dbSessionRetryable(tableClient) {
            resourceSegmentationsDao.getById(roStaleSingleRetryableCommit(), settings.scopeSegmentation!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (scopeSegmentation.isEmpty || scopeSegmentation.get().isDeleted
            || scopeSegmentation.get().providerId != knownProviders.yt) {
            errors.addError("scopeSegmentation", TypedError.invalid(messages
                .getMessage("errors.resource.segmentation.not.found", null, locale)))
        }
        val computeScopeSegment = dbSessionRetryable(tableClient) {
            resourceSegmentsDao.getById(roStaleSingleRetryableCommit(), settings.computeScopeSegment!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (computeScopeSegment.isEmpty || computeScopeSegment.get().isDeleted
            || computeScopeSegment.get().segmentationId != settings.scopeSegmentation) {
            errors.addError("computeScopeSegment", TypedError.invalid(messages
                .getMessage("errors.resource.segment.not.found", null, locale)))
        }
        if (setOf(settings.poolTreeSegmentation, settings.clusterSegmentation, settings.scopeSegmentation).size != 3) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.resource.segmentation.id.duplicate", null, locale)))
        }
        if (setOf(settings.physicalPoolTreeSegment, settings.computeScopeSegment).size != 2) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.resource.segment.id.duplicate", null, locale)))
        }
        if (settings.cpuMetricGridSpacingSeconds!! <= 0) {
            errors.addError("cpuMetricGridSpacingSeconds", TypedError.invalid(messages
                .getMessage("errors.number.must.be.positive", null, locale)))
        }
        val cpuMetricUnitsEnsemble = dbSessionRetryable(tableClient) {
            unitsEnsemblesDao.getById(roStaleSingleRetryableCommit(), settings.cpuMetricUnitsEnsemble!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (cpuMetricUnitsEnsemble.isEmpty || cpuMetricUnitsEnsemble.get().isDeleted
            || (cpuStrongResourceType.isPresent
                && cpuStrongResourceType.get().unitsEnsembleId != cpuMetricUnitsEnsemble.get().id)) {
            errors.addError("cpuMetricUnitsEnsemble", TypedError.invalid(messages
                .getMessage("errors.units.ensemble.not.found", null, locale)))
        }
        if (cpuMetricUnitsEnsemble.isPresent && (cpuMetricUnitsEnsemble.get().unitById(settings.cpuMetricUnit).isEmpty
                || cpuMetricUnitsEnsemble.get().unitById(settings.cpuMetricUnit).get().isDeleted)) {
            errors.addError("cpuMetricUnit", TypedError.invalid(messages
                .getMessage("errors.unit.not.found", null, locale)))
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(YtUsageSyncSettings(
                cpuStrongResourceType = settings.cpuStrongResourceType,
                poolTreeSegmentation = settings.poolTreeSegmentation,
                physicalPoolTreeSegment = settings.physicalPoolTreeSegment,
                clusterSegmentation = settings.clusterSegmentation,
                scopeSegmentation = settings.scopeSegmentation,
                computeScopeSegment = settings.computeScopeSegment,
                syncEnabled = settings.syncEnabled,
                cpuMetricGridSpacingSeconds = settings.cpuMetricGridSpacingSeconds,
                cpuMetricUnitsEnsemble = settings.cpuMetricUnitsEnsemble,
                cpuMetricUnit = settings.cpuMetricUnit
            ))
        }
    }

    private suspend fun validateYpUsageSyncSettings(settings: YpUsageSyncSettingsDto, locale: Locale): Result<YpUsageSyncSettings> {
        val errors = ErrorCollection.builder()
        val knownProviders = dbSessionRetryable(tableClient) {
            runtimeSettingsDao.get(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, SettingsKey.KNOWN_PROVIDERS)
        }
        if (knownProviders == null) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale)))
            return Result.failure(errors.build())
        }
        if (knownProviders.ypPreStable == null
            && (settings.preStableCpuResourceType != null || settings.preStableClusterSegmentation != null
                || settings.preStableNodeSegmentSegmentation != null || settings.preStableDefaultNodeSegment != null
                || settings.preStableDevNodeSegment != null)) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale)))
        }
        val cpuResourceType = dbSessionRetryable(tableClient) {
            resourceTypesDao.getById(roStaleSingleRetryableCommit(), settings.cpuResourceType!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (cpuResourceType.isEmpty || cpuResourceType.get().isDeleted
            || cpuResourceType.get().providerId != knownProviders.yp) {
            errors.addError("cpuResourceType", TypedError.invalid(messages
                .getMessage("errors.resource.type.not.found", null, locale)))
        }
        val clusterSegmentation = dbSessionRetryable(tableClient) {
            resourceSegmentationsDao.getById(roStaleSingleRetryableCommit(), settings.clusterSegmentation!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (clusterSegmentation.isEmpty || clusterSegmentation.get().isDeleted
            || clusterSegmentation.get().providerId != knownProviders.yp) {
            errors.addError("clusterSegmentation", TypedError.invalid(messages
                .getMessage("errors.resource.segmentation.not.found", null, locale)))
        }
        val nodeSegmentSegmentation = dbSessionRetryable(tableClient) {
            resourceSegmentationsDao.getById(roStaleSingleRetryableCommit(), settings.nodeSegmentSegmentation!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (nodeSegmentSegmentation.isEmpty || nodeSegmentSegmentation.get().isDeleted
            || nodeSegmentSegmentation.get().providerId != knownProviders.yp) {
            errors.addError("nodeSegmentSegmentation", TypedError.invalid(messages
                .getMessage("errors.resource.segmentation.not.found", null, locale)))
        }
        val defaultNodeSegment = dbSessionRetryable(tableClient) {
            resourceSegmentsDao.getById(roStaleSingleRetryableCommit(), settings.defaultNodeSegment!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (defaultNodeSegment.isEmpty || defaultNodeSegment.get().isDeleted
            || defaultNodeSegment.get().segmentationId != settings.nodeSegmentSegmentation) {
            errors.addError("defaultNodeSegment", TypedError.invalid(messages
                .getMessage("errors.resource.segment.not.found", null, locale)))
        }
        val devNodeSegment = dbSessionRetryable(tableClient) {
            resourceSegmentsDao.getById(roStaleSingleRetryableCommit(), settings.devNodeSegment!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (devNodeSegment.isEmpty || devNodeSegment.get().isDeleted
            || devNodeSegment.get().segmentationId != settings.nodeSegmentSegmentation) {
            errors.addError("devNodeSegment", TypedError.invalid(messages
                .getMessage("errors.resource.segment.not.found", null, locale)))
        }
        val coresUnitsEnsemble = dbSessionRetryable(tableClient) {
            unitsEnsemblesDao.getById(roStaleSingleRetryableCommit(), settings.coresUnitsEnsemble!!,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (coresUnitsEnsemble.isEmpty || coresUnitsEnsemble.get().isDeleted
            || (cpuResourceType.isPresent
                && cpuResourceType.get().unitsEnsembleId != coresUnitsEnsemble.get().id)) {
            errors.addError("coresUnitsEnsemble", TypedError.invalid(messages
                .getMessage("errors.units.ensemble.not.found", null, locale)))
        }
        if (coresUnitsEnsemble.isPresent && (coresUnitsEnsemble.get().unitById(settings.coresUnit).isEmpty
                || coresUnitsEnsemble.get().unitById(settings.coresUnit).get().isDeleted)) {
            errors.addError("coresUnit", TypedError.invalid(messages
                .getMessage("errors.unit.not.found", null, locale)))
        }
        if (settings.clusterSegmentation == settings.nodeSegmentSegmentation) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.resource.segmentation.id.duplicate", null, locale)))
        }
        if (settings.defaultNodeSegment == settings.devNodeSegment) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.resource.segment.id.duplicate", null, locale)))
        }
        if (knownProviders.ypPreStable != null) {
            if (settings.preStableCpuResourceType == null) {
                errors.addError("preStableCpuResourceType", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
            } else {
                val preStableCpuResourceType = dbSessionRetryable(tableClient) {
                    resourceTypesDao.getById(roStaleSingleRetryableCommit(), settings.preStableCpuResourceType,
                        Tenants.DEFAULT_TENANT_ID).awaitSingle()
                }!!
                if (preStableCpuResourceType.isEmpty || preStableCpuResourceType.get().isDeleted
                    || preStableCpuResourceType.get().providerId != knownProviders.ypPreStable) {
                    errors.addError("preStableCpuResourceType", TypedError.invalid(messages
                        .getMessage("errors.resource.type.not.found", null, locale)))
                }
                if (preStableCpuResourceType.isPresent && coresUnitsEnsemble.isPresent
                        && preStableCpuResourceType.get().unitsEnsembleId != coresUnitsEnsemble.get().id) {
                    errors.addError("coresUnitsEnsemble", TypedError.invalid(messages
                        .getMessage("errors.units.ensemble.not.found", null, locale)))
                }
            }
            if (settings.preStableClusterSegmentation == null) {
                errors.addError("preStableClusterSegmentation", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
            } else {
                val preStableClusterSegmentation = dbSessionRetryable(tableClient) {
                    resourceSegmentationsDao.getById(roStaleSingleRetryableCommit(), settings.preStableClusterSegmentation,
                        Tenants.DEFAULT_TENANT_ID).awaitSingle()
                }!!
                if (preStableClusterSegmentation.isEmpty || preStableClusterSegmentation.get().isDeleted
                    || preStableClusterSegmentation.get().providerId != knownProviders.ypPreStable) {
                    errors.addError("preStableClusterSegmentation", TypedError.invalid(messages
                        .getMessage("errors.resource.segmentation.not.found", null, locale)))
                }
            }
            if (settings.preStableNodeSegmentSegmentation == null) {
                errors.addError("preStableNodeSegmentSegmentation", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
            } else {
                val preStableNodeSegmentSegmentation = dbSessionRetryable(tableClient) {
                    resourceSegmentationsDao.getById(roStaleSingleRetryableCommit(), settings.preStableNodeSegmentSegmentation,
                        Tenants.DEFAULT_TENANT_ID).awaitSingle()
                }!!
                if (preStableNodeSegmentSegmentation.isEmpty || preStableNodeSegmentSegmentation.get().isDeleted
                    || preStableNodeSegmentSegmentation.get().providerId != knownProviders.ypPreStable) {
                    errors.addError("preStableNodeSegmentSegmentation", TypedError.invalid(messages
                        .getMessage("errors.resource.segmentation.not.found", null, locale)))
                }
            }
            if (settings.preStableDefaultNodeSegment == null) {
                errors.addError("preStableDefaultNodeSegment", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
            } else {
                val preStableDefaultNodeSegment = dbSessionRetryable(tableClient) {
                    resourceSegmentsDao.getById(roStaleSingleRetryableCommit(), settings.preStableDefaultNodeSegment,
                        Tenants.DEFAULT_TENANT_ID).awaitSingle()
                }!!
                if (preStableDefaultNodeSegment.isEmpty || preStableDefaultNodeSegment.get().isDeleted
                    || (settings.preStableNodeSegmentSegmentation != null
                        && preStableDefaultNodeSegment.get().segmentationId != settings.preStableNodeSegmentSegmentation)) {
                    errors.addError("preStableDefaultNodeSegment", TypedError.invalid(messages
                        .getMessage("errors.resource.segment.not.found", null, locale)))
                }
            }
            if (settings.preStableDevNodeSegment == null) {
                errors.addError("preStableDevNodeSegment", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
            } else {
                val preStableDevNodeSegment = dbSessionRetryable(tableClient) {
                    resourceSegmentsDao.getById(roStaleSingleRetryableCommit(), settings.preStableDevNodeSegment,
                        Tenants.DEFAULT_TENANT_ID).awaitSingle()
                }!!
                if (preStableDevNodeSegment.isEmpty || preStableDevNodeSegment.get().isDeleted
                    || (settings.preStableNodeSegmentSegmentation != null
                        && preStableDevNodeSegment.get().segmentationId != settings.preStableNodeSegmentSegmentation)) {
                    errors.addError("preStableDevNodeSegment", TypedError.invalid(messages
                        .getMessage("errors.resource.segment.not.found", null, locale)))
                }
            }
            if (settings.preStableClusterSegmentation != null && settings.preStableNodeSegmentSegmentation != null
                && settings.preStableClusterSegmentation == settings.preStableNodeSegmentSegmentation) {
                errors.addError(TypedError.invalid(messages
                    .getMessage("errors.resource.segmentation.id.duplicate", null, locale)))
            }
            if (settings.preStableDefaultNodeSegment != null && settings.preStableDevNodeSegment != null
                && settings.preStableDefaultNodeSegment == settings.preStableDevNodeSegment) {
                errors.addError(TypedError.invalid(messages
                    .getMessage("errors.resource.segment.id.duplicate", null, locale)))
            }
        }
        return if (errors.hasAnyErrors()) {
            Result.failure(errors.build())
        } else {
            Result.success(YpUsageSyncSettings(
                cpuResourceType = settings.cpuResourceType,
                preStableCpuResourceType = settings.preStableCpuResourceType,
                clusterSegmentation = settings.clusterSegmentation,
                nodeSegmentSegmentation = settings.nodeSegmentSegmentation,
                preStableClusterSegmentation = settings.preStableClusterSegmentation,
                preStableNodeSegmentSegmentation = settings.preStableNodeSegmentSegmentation,
                defaultNodeSegment = settings.defaultNodeSegment,
                devNodeSegment = settings.devNodeSegment,
                preStableDefaultNodeSegment = settings.preStableDefaultNodeSegment,
                preStableDevNodeSegment = settings.preStableDevNodeSegment,
                coresUnitsEnsemble = settings.coresUnitsEnsemble,
                coresUnit = settings.coresUnit,
                syncEnabled = settings.syncEnabled
            ))
        }
    }

}
