package ru.yandex.intranet.d.services.sync;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.services.sync.model.ExternalAccount;
import ru.yandex.intranet.d.services.sync.model.ExternalProvision;
import ru.yandex.intranet.d.services.sync.model.GroupedAccounts;
import ru.yandex.intranet.d.services.sync.model.QuotasAndOperations;

/**
 * Service to sync providers accounts and quotas, postponed sync utils part.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class AccountsPostponedSyncUtils {

    private AccountsPostponedSyncUtils() {
    }

    public static Set<String> findPostponedFolders(ProviderModel provider,
                                                   Set<String> foldersPage,
                                                   GroupedAccounts groupedAccounts,
                                                   QuotasAndOperations quotasAndOperations) {
        Set<String> postponedFolders = new HashSet<>();
        foldersPage.forEach(folderId -> {
            Set<String> changedAccountsAffectingBalance = new HashSet<>();
            Map<String, ExternalAccount> accountsToCreate = groupedAccounts.getReceivedAccountsToCreate()
                    .getOrDefault(folderId, Collections.emptyMap());
            accountsToCreate.values().forEach(account -> {
                if (hasNonZeroProvisions(account)) {
                    changedAccountsAffectingBalance.add(account.getAccountId());
                }
            });
            Map<String, AccountModel> currentNonMovingAccountsToDelete = groupedAccounts
                    .getCurrentAccountsToDeleteStay().getOrDefault(folderId, Collections.emptyMap());
            Map<String, ExternalAccount> receivedNonMovingAccountsToDelete = groupedAccounts
                    .getReceivedAccountsToDeleteStay().getOrDefault(folderId, Collections.emptyMap());
            currentNonMovingAccountsToDelete.forEach((externalAccountId, currentAccount) -> {
                ExternalAccount receivedAccount = receivedNonMovingAccountsToDelete.get(externalAccountId);
                Map<String, AccountsQuotasModel> currentProvisions = quotasAndOperations
                        .getProvisionsByAccountResource().getOrDefault(currentAccount.getId(), Collections.emptyMap());
                if (deletionChangesProvisions(provider, receivedAccount, currentProvisions)) {
                    changedAccountsAffectingBalance.add(currentAccount.getOuterAccountIdInProvider());
                }
            });
            Map<String, AccountModel> currentNonMovingAccounts = groupedAccounts.getCurrentAccountsStay()
                    .getOrDefault(folderId, Collections.emptyMap());
            Map<String, ExternalAccount> receivedNonMovingAccounts = groupedAccounts.getReceivedAccountsStay()
                    .getOrDefault(folderId, Collections.emptyMap());
            currentNonMovingAccounts.forEach((externalAccountId, currentAccount) -> {
                ExternalAccount receivedAccount = receivedNonMovingAccounts.get(externalAccountId);
                Map<String, AccountsQuotasModel> currentProvisions = quotasAndOperations
                        .getProvisionsByAccountResource().getOrDefault(currentAccount.getId(), Collections.emptyMap());
                if (updateChangesProvisions(receivedAccount, currentProvisions)) {
                    changedAccountsAffectingBalance.add(currentAccount.getOuterAccountIdInProvider());
                }
            });
            Map<String, AccountModel> currentMovingOutSamePage = groupedAccounts.getCurrentAccountsMovingOutSamePage()
                    .getOrDefault(folderId, Collections.emptyMap());
            Map<String, AccountModel> currentMovingOutOtherPages = groupedAccounts
                    .getCurrentAccountsMovingOutOtherPages().getOrDefault(folderId, Collections.emptyMap());
            currentMovingOutSamePage.forEach((externalAccountId, currentAccount) -> {
                Map<String, AccountsQuotasModel> currentProvisions = quotasAndOperations
                        .getProvisionsByAccountResource().getOrDefault(currentAccount.getId(), Collections.emptyMap());
                if (hasNonZeroProvisions(currentProvisions)) {
                    changedAccountsAffectingBalance.add(currentAccount.getOuterAccountIdInProvider());
                }
            });
            currentMovingOutOtherPages.forEach((externalAccountId, currentAccount) -> {
                Map<String, AccountsQuotasModel> currentProvisions = quotasAndOperations
                        .getProvisionsByAccountResource().getOrDefault(currentAccount.getId(), Collections.emptyMap());
                if (hasNonZeroProvisions(currentProvisions)) {
                    changedAccountsAffectingBalance.add(currentAccount.getOuterAccountIdInProvider());
                }
            });
            Map<String, ExternalAccount> receivedMovingInSamePage = groupedAccounts
                    .getReceivedAccountsMovingInSamePage().getOrDefault(folderId, Collections.emptyMap());
            Map<String, ExternalAccount> receivedMovingInOtherPages = groupedAccounts
                    .getReceivedAccountsMovingInOtherPages().getOrDefault(folderId, Collections.emptyMap());
            receivedMovingInSamePage.forEach((externalAccountId, receivedAccount) -> {
                if (hasNonZeroProvisions(receivedAccount)) {
                    changedAccountsAffectingBalance.add(receivedAccount.getAccountId());
                }
            });
            receivedMovingInOtherPages.forEach((externalAccountId, receivedAccount) -> {
                if (hasNonZeroProvisions(receivedAccount)) {
                    changedAccountsAffectingBalance.add(receivedAccount.getAccountId());
                }
            });
            if (provider.isMultipleAccountsPerFolder() && !changedAccountsAffectingBalance.isEmpty()) {
                postponedFolders.add(folderId);
            }
        });
        Set<String> connectedPostponedFolders = new HashSet<>();
        foldersPage.forEach(folderId -> {
            if (!postponedFolders.contains(folderId)) {
                return;
            }
            connectedPostponedFolders.addAll(visitMoveConnections(folderId, groupedAccounts, new HashSet<>()));
        });
        return Sets.union(postponedFolders, connectedPostponedFolders);
    }

    private static Set<String> visitMoveConnections(String folderId, GroupedAccounts groupedAccounts,
                                                    Set<String> visitedFolders) {
        Set<String> postponedFolders = new HashSet<>();
        groupedAccounts.getReceivedAccountsMovingOutSamePage()
                .getOrDefault(folderId, Collections.emptyMap()).forEach((externalAccountId, account) -> {
            String targetFolderId = account.getFolderId();
            if (!visitedFolders.contains(targetFolderId)) {
                visitedFolders.add(targetFolderId);
                Set<String> collectedPostponedFolders = visitMoveConnections(targetFolderId, groupedAccounts,
                        visitedFolders);
                postponedFolders.addAll(collectedPostponedFolders);
                postponedFolders.add(targetFolderId);
            }
        });
        groupedAccounts.getReceivedAccountsMovingOutOtherPages()
                .getOrDefault(folderId, Collections.emptyMap()).forEach((externalAccountId, account) -> {
            String targetFolderId = account.getFolderId();
            if (!visitedFolders.contains(targetFolderId)) {
                visitedFolders.add(targetFolderId);
                Set<String> collectedPostponedFolders = visitMoveConnections(targetFolderId, groupedAccounts,
                        visitedFolders);
                postponedFolders.addAll(collectedPostponedFolders);
                postponedFolders.add(targetFolderId);
            }
        });
        groupedAccounts.getCurrentAccountsMovingInSamePage()
                .getOrDefault(folderId, Collections.emptyMap()).forEach((externalAccountId, account) -> {
            String targetFolderId = account.getFolderId();
            if (!visitedFolders.contains(targetFolderId)) {
                visitedFolders.add(targetFolderId);
                Set<String> collectedPostponedFolders = visitMoveConnections(targetFolderId, groupedAccounts,
                        visitedFolders);
                postponedFolders.addAll(collectedPostponedFolders);
                postponedFolders.add(targetFolderId);
            }
        });
        groupedAccounts.getCurrentAccountsMovingInOtherPages()
                .getOrDefault(folderId, Collections.emptyMap()).forEach((externalAccountId, account) -> {
            String targetFolderId = account.getFolderId();
            if (!visitedFolders.contains(targetFolderId)) {
                visitedFolders.add(targetFolderId);
                Set<String> collectedPostponedFolders = visitMoveConnections(targetFolderId, groupedAccounts,
                        visitedFolders);
                postponedFolders.addAll(collectedPostponedFolders);
                postponedFolders.add(targetFolderId);
            }
        });
        groupedAccounts.getCurrentAccountsOuterFoldersMovingIn()
                .getOrDefault(folderId, Collections.emptyMap()).forEach((externalAccountId, account) -> {
            String targetFolderId = account.getFolderId();
            if (!visitedFolders.contains(targetFolderId)) {
                visitedFolders.add(targetFolderId);
                Set<String> collectedPostponedFolders = visitMoveConnections(targetFolderId, groupedAccounts,
                        visitedFolders);
                postponedFolders.addAll(collectedPostponedFolders);
                postponedFolders.add(targetFolderId);
            }
        });
        groupedAccounts.getReceivedAccountsOuterFoldersMovingOut()
                .getOrDefault(folderId, Collections.emptyMap()).forEach((externalAccountId, account) -> {
            String targetFolderId = account.getFolderId();
            if (!visitedFolders.contains(targetFolderId)) {
                visitedFolders.add(targetFolderId);
                Set<String> collectedPostponedFolders = visitMoveConnections(targetFolderId, groupedAccounts,
                        visitedFolders);
                postponedFolders.addAll(collectedPostponedFolders);
                postponedFolders.add(targetFolderId);
            }
        });
        return postponedFolders;
    }

    private static boolean hasNonZeroProvisions(ExternalAccount account) {
        return account.getProvisions().stream().anyMatch(provision -> provision.getProvided() > 0L);
    }

    private static boolean hasNonZeroProvisions(Map<String, AccountsQuotasModel> provisions) {
        return provisions.values().stream()
                .anyMatch(provision -> provision.getProvidedQuota() != null && provision.getProvidedQuota() > 0L);
    }

    private static boolean updateChangesProvisions(ExternalAccount receivedAccount,
                                                   Map<String, AccountsQuotasModel> currentProvisions) {
        if (receivedAccount == null) {
            return false;
        }
        Map<String, ExternalProvision> receivedProvisions = receivedAccount.getProvisions().stream()
                .collect(Collectors.toMap(p -> p.getResource().getResource().getId(), Function.identity()));
        boolean currentMismatch = currentProvisions.entrySet().stream().anyMatch(e -> {
            long currentProvision = e.getValue().getProvidedQuota() != null ? e.getValue().getProvidedQuota() : 0L;
            long receivedProvision = receivedProvisions.containsKey(e.getKey())
                    ? receivedProvisions.get(e.getKey()).getProvided() : 0L;
            return currentProvision != receivedProvision;
        });
        boolean receivedMismatch = receivedProvisions.entrySet().stream().anyMatch(e -> {
            long receivedProvision = e.getValue().getProvided();
            long currentProvision;
            if (currentProvisions.containsKey(e.getKey())) {
                AccountsQuotasModel provision = currentProvisions.get(e.getKey());
                currentProvision = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
            } else {
                currentProvision = 0L;
            }
            return currentProvision != receivedProvision;
        });
        return currentMismatch || receivedMismatch;
    }

    private static boolean deletionChangesProvisions(ProviderModel provider, ExternalAccount receivedAccount,
                                                     Map<String, AccountsQuotasModel> currentProvisions) {
        if (provider.getAccountsSettings().isDeleteSupported()
                && !provider.getAccountsSettings().isSoftDeleteSupported() && receivedAccount == null) {
            return currentProvisions.values().stream().anyMatch(provision -> provision.getProvidedQuota() != null
                    && provision.getProvidedQuota() > 0L);
        }
        if (provider.getAccountsSettings().isDeleteSupported()
                && provider.getAccountsSettings().isSoftDeleteSupported() && receivedAccount != null) {
            Map<String, ExternalProvision> receivedProvisions = receivedAccount.getProvisions().stream()
                    .collect(Collectors.toMap(p -> p.getResource().getResource().getId(), Function.identity()));
            boolean currentMismatch = currentProvisions.entrySet().stream().anyMatch(e -> {
                long currentProvision = e.getValue().getProvidedQuota() != null ? e.getValue().getProvidedQuota() : 0L;
                long receivedProvision = receivedProvisions.containsKey(e.getKey())
                        ? receivedProvisions.get(e.getKey()).getProvided() : 0L;
                return currentProvision != receivedProvision;
            });
            boolean receivedMismatch = receivedProvisions.entrySet().stream().anyMatch(e -> {
                long receivedProvision = e.getValue().getProvided();
                long currentProvision;
                if (currentProvisions.containsKey(e.getKey())) {
                    AccountsQuotasModel provision = currentProvisions.get(e.getKey());
                    currentProvision = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
                } else {
                    currentProvision = 0L;
                }
                return currentProvision != receivedProvision;
            });
            return currentMismatch || receivedMismatch;
        }
        return false;
    }

}
