package ru.yandex.intranet.d.services.sync;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.folders.AccountHistoryModel;
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.OperationPhase;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionsByResource;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.services.sync.model.AccountMoveFolders;
import ru.yandex.intranet.d.services.sync.model.AccountMoveHistoryXRef;
import ru.yandex.intranet.d.services.sync.model.FolderHistoryGroupKey;
import ru.yandex.intranet.d.services.sync.model.FolderOperationAuthor;
import ru.yandex.intranet.d.util.units.Units;

/**
 * Service to sync providers accounts and quotas, oplog utils part.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class AccountsSyncOpLogUtils {

    private static final Logger LOG = LoggerFactory.getLogger(AccountsSyncOpLogUtils.class);

    private AccountsSyncOpLogUtils() {
    }

    @SuppressWarnings("ParameterNumber")
    public static List<FolderOperationLogModel.Builder> prepareFolderOperationLogs(
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> oldAccounts,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates,
            Map<String, List<String>> sortedFolderLogIdsByFolder,
            ProviderModel provider,
            String folderId,
            Instant now,
            Map<String, AccountMoveHistoryXRef> accountMoveXRefsByAccountId,
            Map<String, AccountMoveFolders> accountMoveFoldersByAccountId,
            Map<String, QuotaModel> currentQuotasByResourceId,
            Map<String, ResourceModel> resourceById
    ) {
        List<FolderOperationLogModel.Builder> result = new ArrayList<>();
        Set<FolderHistoryGroupKey> groupKeys = new HashSet<>();
        groupKeys.addAll(oldProvisions.keySet());
        groupKeys.addAll(newProvisions.keySet());
        groupKeys.addAll(oldAccounts.keySet());
        groupKeys.addAll(newAccounts.keySet());
        Map<FolderHistoryGroupKey, String> logIdByGroupKey = new HashMap<>();
        groupKeys.forEach(groupKey -> {
            String logId = UUID.randomUUID().toString();
            Optional<String> operationId = groupKey.getOperationId();
            Optional<OperationPhase> operationPhase = Optional.ofNullable(operationId.isPresent()
                    ? OperationPhase.RESTORE
                    : null);
            Set<FolderOperationAuthor> groupAuthors = operationAuthors.getOrDefault(groupKey, Collections.emptySet());
            FolderOperationAuthor author = groupAuthors.size() == 1
                    ? groupAuthors.iterator().next()
                    : new FolderOperationAuthor(null, null, provider.getId());
            FolderOperationLogModel.Builder builder = new FolderOperationLogModel.Builder();
            builder.setTenantId(Tenants.DEFAULT_TENANT_ID);
            builder.setFolderId(folderId);
            builder.setOperationDateTime(now);
            builder.setId(logId);
            builder.setProviderRequestId(null);
            builder.setOperationType(FolderOperationType.SYNC_WITH_PROVIDER);
            builder.setAuthorUserId(author.getAuthorUserId().orElse(null));
            builder.setAuthorUserUid(author.getAuthorUserUid().orElse(null));
            builder.setAuthorProviderId(author.getAuthorProviderId().orElse(null));
            builder.setSourceFolderOperationsLogId(null);
            builder.setDestinationFolderOperationsLogId(null);
            builder.setOldFolderFields(null);
            builder.setOldQuotas(new QuotasByResource(Collections.emptyMap()));
            builder.setOldBalance(new QuotasByResource(Collections.emptyMap()));
            builder.setOldProvisions(new QuotasByAccount(prepareProvisionsHistory(oldProvisions
                    .getOrDefault(groupKey, Collections.emptyMap()))));
            Map<String, AccountHistoryModel> oldAccountsDetails = oldAccounts
                    .getOrDefault(groupKey, Collections.emptyMap());
            if (!oldAccountsDetails.isEmpty()) {
                builder.setOldAccounts(new AccountsHistoryModel(oldAccountsDetails));
            }
            builder.setNewFolderFields(null);
            builder.setNewQuotas(new QuotasByResource(Collections.emptyMap()));
            builder.setNewBalance(new QuotasByResource(Collections.emptyMap()));
            Map<String, Map<String, ProvisionHistoryModel>> newProvisionsByAccountId = newProvisions.getOrDefault(
                    groupKey, Collections.emptyMap());
            builder.setNewProvisions(new QuotasByAccount(prepareProvisionsHistory(newProvisionsByAccountId)));
            Map<String, AccountHistoryModel> newAccountsDetails = newAccounts
                    .getOrDefault(groupKey, Collections.emptyMap());
            if (!newAccountsDetails.isEmpty()) {
                builder.setNewAccounts(new AccountsHistoryModel(newAccountsDetails));
                if (!newProvisionsByAccountId.isEmpty()) {
                    Map<String, Map<String, ProvisionHistoryModel>> defaultQuotasByAccountIdByResourceId =
                            builder.getDefaultQuotasByAccountIdByResourceId();
                    newAccountsDetails.keySet().forEach(accountId -> {
                        if (newProvisionsByAccountId.containsKey(accountId)) {
                            newProvisionsByAccountId.get(accountId).forEach((resourceId, provisionHistoryModel) -> {
                                ResourceModel resource = resourceById.get(resourceId);
                                if (resource != null && resource.getDefaultQuota().isPresent() &&
                                        resource.getDefaultQuota().get() != 0L
                                ) {
                                    defaultQuotasByAccountIdByResourceId
                                            .computeIfAbsent(accountId, s -> new HashMap<>())
                                            .put(resourceId, new ProvisionHistoryModel(
                                                    resource.getDefaultQuota().get(), 0L
                                            ));
                                }
                            });
                        }
                    });
                }
            }
            builder.setAccountsQuotasOperationsId(operationId.orElse(null));
            builder.setOperationPhase(operationPhase.orElse(null));
            builder.setQuotasDemandsId(null);
            result.add(builder);
            logIdByGroupKey.put(groupKey, logId);
            if (groupKey.getMovedAccountId().isPresent()) {
                String movedAccountId = groupKey.getMovedAccountId().get();
                if (accountMoveFoldersByAccountId.containsKey(movedAccountId)) {
                    AccountMoveFolders moveFolders = accountMoveFoldersByAccountId.get(movedAccountId);
                    if (moveFolders.getSourceFolderId().isPresent()
                            && moveFolders.getSourceFolderId().get().equals(folderId)) {
                        accountMoveXRefsByAccountId.computeIfAbsent(movedAccountId, k -> new AccountMoveHistoryXRef())
                                .setSourceOpLogId(logId);
                    }
                    if (moveFolders.getDestinationFolderId().isPresent()
                            && moveFolders.getDestinationFolderId().get().equals(folderId)) {
                        accountMoveXRefsByAccountId.computeIfAbsent(movedAccountId, k -> new AccountMoveHistoryXRef())
                                .setDestinationOpLogId(logId);
                    }
                }
            }
        });
        sortFolderLogIds(newestLastUpdates, sortedFolderLogIdsByFolder, folderId, groupKeys, logIdByGroupKey);
        prepareBalances(result, folderId, sortedFolderLogIdsByFolder, currentQuotasByResourceId);
        return result;
    }

    public static void prepareOpLogCrossReferences(
            List<FolderOperationLogModel.Builder> folderLogsToUpsert,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            Map<String, AccountMoveHistoryXRef> accountMoveXRefsByAccountId) {
        // TODO Support moves across folders pages
        Map<String, String> opLogXRefsSrc = new HashMap<>();
        Map<String, String> opLogXRefsDst = new HashMap<>();
        accountMoveXRefsByAccountId.values().forEach(xRef -> {
            if (xRef.getSourceOpLogId().isPresent() && xRef.getDestinationOpLogId().isPresent()) {
                opLogXRefsSrc.put(xRef.getSourceOpLogId().get(), xRef.getDestinationOpLogId().get());
                opLogXRefsDst.put(xRef.getDestinationOpLogId().get(), xRef.getSourceOpLogId().get());
            }
        });
        folderLogsToUpsert.forEach(opLog -> {
            if (opLog.getId().isPresent() && opLogXRefsSrc.containsKey(opLog.getId().get())) {
                String dstOpLogId = opLogXRefsSrc.get(opLog.getId().get());
                opLog.setDestinationFolderOperationsLogId(dstOpLogId);
            }
            if (opLog.getId().isPresent() && opLogXRefsDst.containsKey(opLog.getId().get())) {
                String srcOpLogId = opLogXRefsDst.get(opLog.getId().get());
                opLog.setSourceFolderOperationsLogId(srcOpLogId);
            }
        });
        Map<String, Set<FolderOperationLogModel.Builder>> opLogsByOperation = folderLogsToUpsert.stream()
                .filter(v -> v.getAccountsQuotasOperationsId().isPresent())
                .collect(Collectors.groupingBy(v -> v.getAccountsQuotasOperationsId().get(), Collectors.toSet()));
        opLogsByOperation.forEach((operationId, opLogs) -> {
            if (opLogs.size() < 2) {
                return;
            }
            AccountsQuotasOperationsModel operation = operationsById.get(operationId);
            if (operation == null) {
                return;
            }
            if (AccountsQuotasOperationsModel.OperationType.MOVE_PROVISION.equals(operation.getOperationType())) {
                String accountId = operation.getRequestedChanges().getAccountId().get();
                String destinationAccountId = operation.getRequestedChanges().getDestinationAccountId().get();
                List<FolderOperationLogModel.Builder> sourceOpLogs = new ArrayList<>();
                List<FolderOperationLogModel.Builder> destinationOpLogs = new ArrayList<>();
                opLogs.forEach(opLog -> {
                    Set<String> updatedAccounts = modifiedProvisionsAccountIdsFromOpLog(opLog);
                    boolean sourceMatch = updatedAccounts.contains(accountId);
                    boolean destinationMatch = updatedAccounts.contains(destinationAccountId);
                    if (sourceMatch && !destinationMatch) {
                        sourceOpLogs.add(opLog);
                    }
                    if (destinationMatch && !sourceMatch) {
                        destinationOpLogs.add(opLog);
                    }
                });
                if (sourceOpLogs.size() >= 1 && destinationOpLogs.size() >= 1) {
                    sourceOpLogs.get(0).setDestinationFolderOperationsLogId(destinationOpLogs.get(0).getId().get());
                    destinationOpLogs.get(0).setSourceFolderOperationsLogId(sourceOpLogs.get(0).getId().get());
                }
            }
        });
    }

    private static void prepareBalances(List<FolderOperationLogModel.Builder> opLogs,
                                        String folderId,
                                        Map<String, List<String>> sortedFolderLogIdsByFolder,
                                        Map<String, QuotaModel> currentQuotasByResourceId) {
        Map<String, FolderOperationLogModel.Builder> opLogsById = new HashMap<>();
        Map<String, Long> currentBalanceByResourceId = new HashMap<>();
        currentQuotasByResourceId.forEach((resourceId, quota) -> currentBalanceByResourceId
                .put(resourceId, quota.getBalance() != null ? quota.getBalance() : 0L));
        Map<String, Long> previousBalanceByResourceId = new HashMap<>(currentBalanceByResourceId);
        Map<String, Long> currentQuotaValuesByResourceId = new HashMap<>();
        currentQuotasByResourceId.forEach((resourceId, quota) -> currentQuotaValuesByResourceId
                .put(resourceId, quota.getQuota() != null ? quota.getQuota() : 0L));
        Map<String, Long> previousQuotaValuesByResourceId = new HashMap<>(currentQuotaValuesByResourceId);
        opLogs.forEach(opLog -> opLog.getId().ifPresent(id -> opLogsById.put(id, opLog)));
        for (String opLogId : sortedFolderLogIdsByFolder.getOrDefault(folderId, Collections.emptyList())) {
            FolderOperationLogModel.Builder opLog = opLogsById.get(opLogId);
            if (opLog.getOldProvisions().isEmpty() && opLog.getNewProvisions().isEmpty()) {
                continue;
            }
            if (opLog.getOldProvisions().isPresent() && opLog.getNewProvisions().isEmpty()) {
                QuotasByAccount oldProvisions = opLog.getOldProvisions().get();
                oldProvisions.forEach((account, oldProvisionsByResource) -> {
                    oldProvisionsByResource.forEach((resourceId, oldProvision) -> {
                        increaseBalance(folderId, currentBalanceByResourceId, account, resourceId, oldProvision);
                    });
                });
            }
            if (opLog.getOldProvisions().isEmpty() && opLog.getNewProvisions().isPresent()) {
                QuotasByAccount newProvisions = opLog.getNewProvisions().get();
                newProvisions.forEach((account, newProvisionsByResource) -> {
                    newProvisionsByResource.forEach((resourceId, newProvision) -> {
                        decreaseBalance(folderId, currentBalanceByResourceId, account, resourceId, newProvision);
                    });
                });
            }
            if (opLog.getOldProvisions().isPresent() && opLog.getNewProvisions().isPresent()) {
                QuotasByAccount oldProvisions = opLog.getOldProvisions().get();
                QuotasByAccount newProvisions = opLog.getNewProvisions().get();
                oldProvisions.forEach((account, oldProvisionsByResource) -> {
                    if (newProvisions.asMap().containsKey(account)) {
                        ProvisionsByResource newProvisionsByResource = newProvisions.asMap().get(account);
                        oldProvisionsByResource.forEach((resourceId, oldProvision) -> {
                            if (newProvisionsByResource.asMap().containsKey(resourceId)) {
                                ProvisionHistoryModel newProvision = newProvisionsByResource.asMap().get(resourceId);
                                adjustBalance(folderId, currentBalanceByResourceId, account, resourceId,
                                        oldProvision, newProvision);
                            } else {
                                increaseBalance(folderId, currentBalanceByResourceId, account, resourceId,
                                        oldProvision);
                            }
                        });
                        newProvisionsByResource.forEach((resourceId, newProvision) -> {
                            if (!oldProvisionsByResource.asMap().containsKey(resourceId)) {
                                decreaseBalance(folderId, currentBalanceByResourceId, account, resourceId,
                                        newProvision);
                            }
                        });
                    } else {
                        oldProvisionsByResource.forEach((resourceId, oldProvision) -> {
                            increaseBalance(folderId, currentBalanceByResourceId, account, resourceId, oldProvision);
                        });
                    }
                });
                newProvisions.forEach((account, newProvisionsByResource) -> {
                    if (!oldProvisions.asMap().containsKey(account)) {
                        newProvisionsByResource.forEach((resourceId, newProvision) -> {
                            decreaseBalance(folderId, currentBalanceByResourceId, account, resourceId, newProvision);
                        });
                    }
                });
            }
            if (opLog.getNewAccounts().isPresent() && !opLog.getDefaultQuotasByAccountIdByResourceId().isEmpty()) {
                opLog.getNewAccounts().get().getAccounts().keySet().forEach(accountId ->
                    opLog.getDefaultQuotasByAccountIdByResourceId().get(accountId).forEach(
                    (resourceId, defaultProvision) -> {
                        currentBalanceByResourceId.compute(resourceId, (s, balance) ->
                                (balance == null ? 0L : balance) + defaultProvision.getProvision());
                        currentQuotaValuesByResourceId.compute(resourceId, (s, quota) ->
                                (quota == null ? 0L : quota) + defaultProvision.getProvision());
                    })
                );
            }
            Map<String, Long> oldBalances = new HashMap<>();
            Map<String, Long> newBalances = new HashMap<>();
            for (Map.Entry<String, Long> entry : currentBalanceByResourceId.entrySet()) {
                String resourceId = entry.getKey();
                Long balance = entry.getValue();
                long previousBalance = previousBalanceByResourceId.getOrDefault(resourceId, 0L);
                if (balance != previousBalance) {
                    oldBalances.put(resourceId, previousBalance);
                    newBalances.put(resourceId, balance);
                }
            }
            opLog.setOldBalance(new QuotasByResource(oldBalances));
            opLog.setNewBalance(new QuotasByResource(newBalances));
            previousBalanceByResourceId = new HashMap<>(currentBalanceByResourceId);

            Map<String, Long> oldQuotas = new HashMap<>();
            Map<String, Long> newQuotas = new HashMap<>();
            for (Map.Entry<String, Long> entry : currentQuotaValuesByResourceId.entrySet()) {
                String resourceId = entry.getKey();
                Long quota = entry.getValue();
                long previousQuota = previousQuotaValuesByResourceId.getOrDefault(resourceId, 0L);
                if (quota != previousQuota) {
                    oldQuotas.put(resourceId, previousQuota);
                    newQuotas.put(resourceId, quota);
                }
            }
            opLog.setOldQuotas(new QuotasByResource(oldQuotas));
            opLog.setNewQuotas(new QuotasByResource(newQuotas));
            previousQuotaValuesByResourceId = new HashMap<>(currentQuotaValuesByResourceId);
        }
        // @todo хорошо бы результирующее состояние и записать в таблицу квот
        // (то, что в currentQuotaValuesByResourceId и currentBalanceByResourceId)
        // это избавит от дублирования кода и возможности расхождений.
        // @todo Вообще, но базе этого кода можно сделать универсальный метод подготовки истории и балансов,
        // и использовать его в спуске, перемещении, импорте...
    }

    private static void adjustBalance(String folderId, Map<String, Long> currentBalanceByResourceId, String account,
                                      String resourceId, ProvisionHistoryModel oldProvision,
                                      ProvisionHistoryModel newProvision) {
        long oldProvisionAmount = oldProvision.getProvision();
        long newProvisionAmount = newProvision.getProvision();
        Optional<Long> provisionIncrease = Units.subtract(newProvisionAmount,
                oldProvisionAmount);
        if (provisionIncrease.isEmpty()) {
            LOG.error("Failed to calculate updated balance for {} resource in account " +
                    "{} of folder {}", resourceId, account, folderId);
        } else {
            long currentBalance = currentBalanceByResourceId.getOrDefault(resourceId, 0L);
            Optional<Long> updatedBalance = Units.subtract(currentBalance,
                    provisionIncrease.get());
            if (updatedBalance.isEmpty()) {
                LOG.error("Failed to calculate updated balance for {} resource in account " +
                        "{} of folder {}", resourceId, account, folderId);
            } else {
                currentBalanceByResourceId.put(resourceId, updatedBalance.get());
            }
        }
    }

    private static void decreaseBalance(String folderId, Map<String, Long> currentBalanceByResourceId, String account,
                                        String resourceId, ProvisionHistoryModel newProvision) {
        long balanceIsDecreasedBy = newProvision.getProvision();
        long currentBalance = currentBalanceByResourceId.getOrDefault(resourceId, 0L);
        Optional<Long> updatedBalance = Units.subtract(currentBalance, balanceIsDecreasedBy);
        if (updatedBalance.isEmpty()) {
            LOG.error("Failed to calculate updated balance for {} resource in account {} of folder {}",
                    resourceId, account, folderId);
        } else {
            currentBalanceByResourceId.put(resourceId, updatedBalance.get());
        }
    }

    private static void increaseBalance(String folderId, Map<String, Long> currentBalanceByResourceId, String account,
                                        String resourceId, ProvisionHistoryModel oldProvision) {
        long balanceIsIncreasedBy = oldProvision.getProvision();
        long currentBalance = currentBalanceByResourceId.getOrDefault(resourceId, 0L);
        Optional<Long> updatedBalance = Units.add(currentBalance, balanceIsIncreasedBy);
        if (updatedBalance.isEmpty()) {
            LOG.error("Failed to calculate updated balance for {} resource in account {} of folder {}",
                    resourceId, account, folderId);
        } else {
            currentBalanceByResourceId.put(resourceId, updatedBalance.get());
        }
    }

    private static void sortFolderLogIds(Map<FolderHistoryGroupKey, Instant> newestLastUpdates,
                                         Map<String, List<String>> sortedFolderLogIdsByFolder,
                                         String folderId,
                                         Set<FolderHistoryGroupKey> groupKeys,
                                         Map<FolderHistoryGroupKey, String> logIdByGroupKey) {
        List<FolderHistoryGroupKey> sortedGroups = groupKeys.stream().sorted((left, right) -> {
            if (left.getMovedAccountId().isPresent() && right.getMovedAccountId().isEmpty()) {
                return 1;
            }
            if (left.getMovedAccountId().isEmpty() && right.getMovedAccountId().isPresent()) {
                return -1;
            }
            Instant leftTimestamp = newestLastUpdates.get(left);
            Instant rightTimestamp = newestLastUpdates.get(right);
            if (leftTimestamp != null && rightTimestamp != null) {
                return leftTimestamp.compareTo(rightTimestamp);
            }
            String leftLogId = logIdByGroupKey.get(left);
            String rightLogId = logIdByGroupKey.get(right);
            return leftLogId.compareTo(rightLogId);
        }).collect(Collectors.toList());
        List<String> sortedFolderLogIds = new ArrayList<>();
        sortedGroups.forEach(group -> sortedFolderLogIds.add(logIdByGroupKey.get(group)));
        sortedFolderLogIdsByFolder.put(folderId, sortedFolderLogIds);
    }

    private static Map<String, ProvisionsByResource>  prepareProvisionsHistory(
            Map<String, Map<String, ProvisionHistoryModel>> provisions) {
        Map<String, ProvisionsByResource> result = new HashMap<>();
        provisions.forEach((k, v) -> result.put(k, new ProvisionsByResource(v)));
        return result;
    }

    private static Set<String> modifiedProvisionsAccountIdsFromOpLog(FolderOperationLogModel.Builder opLog) {
        Set<String> accountIds = new HashSet<>();
        opLog.getOldProvisions().ifPresent(p -> p.forEach((accountId, v) -> accountIds.add(accountId)));
        opLog.getNewProvisions().ifPresent(p -> p.forEach((accountId, v) -> accountIds.add(accountId)));
        return accountIds;
    }

}
