package ru.yandex.intranet.d.services.sync;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;
import ru.yandex.intranet.d.services.integration.providers.ProvidersIntegrationService;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsResponseDto;
import ru.yandex.intranet.d.services.sync.model.AccumulatedAccounts;
import ru.yandex.intranet.d.services.sync.model.ExternalAccount;
import ru.yandex.intranet.d.services.sync.model.ExternalCompoundResourceKey;
import ru.yandex.intranet.d.services.sync.model.SyncFailureException;
import ru.yandex.intranet.d.services.sync.model.SyncResource;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.result.ErrorCollection;

/**
 * Service to sync providers accounts and quotas, requests part.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class AccountsSyncRequestService {

    private static final Logger LOG = LoggerFactory.getLogger(AccountsSyncRequestService.class);

    private final ProvidersIntegrationService providersIntegrationService;
    private final YdbTableClient tableClient;
    private final AccountsSyncStoreService accountsSyncStoreService;
    private final AccountsSyncValidationService accountsSyncValidationService;

    public AccountsSyncRequestService(ProvidersIntegrationService providersIntegrationService,
                                      YdbTableClient tableClient,
                                      AccountsSyncStoreService accountsSyncStoreService,
                                      AccountsSyncValidationService accountsSyncValidationService) {
        this.providersIntegrationService = providersIntegrationService;
        this.tableClient = tableClient;
        this.accountsSyncStoreService = accountsSyncStoreService;
        this.accountsSyncValidationService = accountsSyncValidationService;
    }

    public Mono<AccumulatedAccounts> requestAccountsFromProvider(
            ProviderModel provider, AccountsSpaceKeyRequestDto accountsSpaceKey,
            Map<ExternalCompoundResourceKey, SyncResource> externalIndex, Locale locale,
            ProvidersSyncStatusModel syncStatus
    ) {
        Boolean includeDeleted = provider.getAccountsSettings().isDeleteSupported()
                && provider.getAccountsSettings().isSoftDeleteSupported() ? true : null;
        ListAccountsRequestDto firstPageRequest = new ListAccountsRequestDto(
                provider.getAccountsSettings().getAccountsSyncPageSize(), null,
                true, includeDeleted, accountsSpaceKey);
        return AsyncMetrics.metric(providersIntegrationService.listAccounts(provider, firstPageRequest, Locales.ENGLISH)
                .flatMap(firstPageResult -> firstPageResult.match(firstPageResponse -> firstPageResponse
                    .match((accounts, requestId) -> {
                        if (accounts.getNextPageToken().isPresent() && !accounts.getNextPageToken().get().isEmpty()
                                && accounts.getAccounts().isPresent() && !accounts.getAccounts().get().isEmpty()) {
                            String nextPageToken = accounts.getNextPageToken().get();
                            return processAccountsPage(provider, accountsSpaceKey, externalIndex, accounts, requestId,
                                    locale, syncStatus).flatMap(acc -> requestNextPage(provider, accountsSpaceKey,
                                    externalIndex, nextPageToken, locale, syncStatus)
                                    .expand(tuple -> {
                                        if (tuple.getT2().isEmpty() || tuple.getT2().get().isEmpty() || tuple.getT3()) {
                                            return Mono.empty();
                                        } else {
                                            return requestNextPage(provider, accountsSpaceKey, externalIndex,
                                                    tuple.getT2().get(), locale, syncStatus);
                                        }
                                    }).map(Tuple2::getT1).reduce(acc, AccumulatedAccounts::concat));
                        }
                        return processAccountsPage(provider, accountsSpaceKey, externalIndex, accounts, requestId,
                                locale, syncStatus);
                    },
                    e -> Mono.error(new SyncFailureException("Sync failed with error", e)),
                    (e, requestId) -> Mono.error(new SyncFailureException("Sync failed with error "
                            + e + ", requestId=" + requestId))),
                e -> Mono.error(new SyncFailureException("Sync failed with error: " + e)))),
                (millis, success) -> LOG.info("Sync accounts download: duration = {} ms, success = {}",
                        millis, success));
    }

    private Mono<Tuple3<AccumulatedAccounts, Optional<String>, Boolean>> requestNextPage(
            ProviderModel provider, AccountsSpaceKeyRequestDto accountsSpaceKey,
            Map<ExternalCompoundResourceKey, SyncResource> externalIndex, String nextPageToken, Locale locale,
            ProvidersSyncStatusModel syncStatus) {
        Boolean includeDeleted = provider.getAccountsSettings().isDeleteSupported()
                && provider.getAccountsSettings().isSoftDeleteSupported() ? true : null;
        ListAccountsRequestDto nextPageRequest = new ListAccountsRequestDto(
                provider.getAccountsSettings().getAccountsSyncPageSize(), nextPageToken,
                true, includeDeleted, accountsSpaceKey);
        return providersIntegrationService.listAccounts(provider, nextPageRequest, locale)
                .flatMap(nextPageResult -> nextPageResult.match(nextPageResponse -> nextPageResponse
                    .match((accounts, requestId) -> processAccountsPage(provider, accountsSpaceKey, externalIndex,
                            accounts, requestId, locale, syncStatus).map(acc -> Tuples.of(acc,
                            accounts.getNextPageToken(),
                            accounts.getAccounts().isEmpty() || accounts.getAccounts().get().isEmpty())),
                            e -> Mono.error(new SyncFailureException("Sync failed with error", e)),
                            (e, requestId) -> Mono.error(new SyncFailureException("Sync failed with error "
                                    + e + ", requestId=" + requestId))),
                    e -> Mono.error(new SyncFailureException("Sync failed with error: " + e))));
    }

    private Mono<AccumulatedAccounts> processAccountsPage(ProviderModel provider,
                                                          AccountsSpaceKeyRequestDto accountsSpaceKey,
                                                          Map<ExternalCompoundResourceKey, SyncResource> externalIndex,
                                                          ListAccountsResponseDto accounts,
                                                          String requestId, Locale locale,
                                                          ProvidersSyncStatusModel syncStatus) {
        if (accounts.getAccounts().isEmpty() || accounts.getAccounts().get().isEmpty()) {
            return Mono.just(new AccumulatedAccounts(List.of(), Set.of(), false));
        }
        ErrorCollection.Builder errors = ErrorCollection.builder();
        Set<String> accumulatedAccountIds = new HashSet<>();
        List<ExternalAccount> validatedAccounts = accountsSyncValidationService.validateAccountsFields(provider,
                accountsSpaceKey, externalIndex, accounts, errors, accumulatedAccountIds, Locales.ENGLISH);
        if (errors.hasAnyErrors()) {
            LOG.error("Some of provider {} responses for account space {} can not be processed: {}, requestId={}",
                    provider.getKey(), accountsSpaceKey, errors.build(), requestId);
        }
        return tableClient.usingSessionMonoRetryable(session -> session
                .usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        accountsSyncStoreService.getTargetFolders(txSession, validatedAccounts)
                                .flatMap(folders -> accountsSyncValidationService.validateAccountsFolders(provider,
                                        validatedAccounts, accountsSpaceKey, folders, accumulatedAccountIds, requestId,
                                        locale, syncStatus))
                ))
                .flatMap(accumulatedAccounts -> {
                            if (errors.hasAnyErrors()) {
                                HashMap<String, String> details = new HashMap<>();
                                if (accountsSpaceKey != null) {
                                    details.put("accountsSpaceKey", accountsSpaceKey.toString());
                                }
                                if (requestId != null) {
                                    details.put("requestId", requestId);
                                }
                                return accountsSyncStoreService.insertSyncError(
                                        "Some of provider responses can not be processed", syncStatus,
                                        errors.build(),
                                        details
                                ).map(providersSyncErrorsModel -> accumulatedAccounts
                                        .mergeHasAnyErrors(errors.hasAnyErrors()));
                            }
                            return Mono.just(accumulatedAccounts);
                        });
    }
}
