package ru.yandex.intranet.d.services.sync;

import java.math.BigInteger;
import java.text.MessageFormat;
import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.folders.AccountHistoryModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncErrorsModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;
import ru.yandex.intranet.d.services.accounts.ReserveAccountsService;
import ru.yandex.intranet.d.services.sync.model.AccountMoveFolders;
import ru.yandex.intranet.d.services.sync.model.AccountMoveHistoryXRef;
import ru.yandex.intranet.d.services.sync.model.AccumulatedAccounts;
import ru.yandex.intranet.d.services.sync.model.ExternalAccount;
import ru.yandex.intranet.d.services.sync.model.ExternalCompoundResourceKey;
import ru.yandex.intranet.d.services.sync.model.FolderHistoryGroupKey;
import ru.yandex.intranet.d.services.sync.model.FolderOperationAuthor;
import ru.yandex.intranet.d.services.sync.model.GroupedAccounts;
import ru.yandex.intranet.d.services.sync.model.GroupedAuthors;
import ru.yandex.intranet.d.services.sync.model.ProcessFoldersPageResult;
import ru.yandex.intranet.d.services.sync.model.QuotasAndOperations;
import ru.yandex.intranet.d.services.sync.model.SyncResource;
import ru.yandex.intranet.d.util.units.Units;

import static ru.yandex.intranet.d.services.sync.AccountsSyncUtils.checkCoolDownPeriod;
import static ru.yandex.intranet.d.services.sync.AccountsSyncUtils.checkInProgressOperations;
import static ru.yandex.intranet.d.services.sync.AccountsSyncUtils.checkOperationsOrder;
import static ru.yandex.intranet.d.services.sync.AccountsSyncUtils.checkStaleVersions;
import static ru.yandex.intranet.d.services.sync.AccountsSyncUtils.updateAllocations;
import static ru.yandex.intranet.d.services.sync.AccountsSyncUtils.updateBalance;

/**
 * Service to sync providers accounts and quotas.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class AccountsSyncService {

    private static final int SYNC_FOLDERS_PAGE_SIZE = 100;
    private static final Logger LOG = LoggerFactory.getLogger(AccountsSyncService.class);

    private final YdbTableClient tableClient;
    private final AccountsSyncStoreService accountsSyncStoreService;
    private final AccountsSyncRequestService accountsSyncRequestService;
    private final ReserveAccountsService reserveAccountsService;

    public AccountsSyncService(YdbTableClient tableClient,
                               AccountsSyncStoreService accountsSyncStoreService,
                               AccountsSyncRequestService accountsSyncRequestService,
                               ReserveAccountsService reserveAccountsService) {
        this.tableClient = tableClient;
        this.accountsSyncStoreService = accountsSyncStoreService;
        this.accountsSyncRequestService = accountsSyncRequestService;
        this.reserveAccountsService = reserveAccountsService;
    }

    public Mono<ProvidersSyncStatusModel> syncOneProvider(ProviderModel provider, Locale locale, Clock clock) {
        // TODO Temporarily store received accounts in DB to avoid OOM when there are too many accounts
        return accountsSyncStoreService.upsertNewSyncStatus(
                Tenants.DEFAULT_TENANT_ID, provider.getId(), clock
        ).flatMap(syncStatus -> accountsSyncStoreService.getResources(provider)
                        .flatMap(resources -> {
            Map<ExternalCompoundResourceKey, SyncResource> externalIndex = AccountsSyncUtils
                    .prepareExternalIndex(resources);
            return accountsSyncStoreService.getAccountSpaces(provider).flatMap(accountsSpaces -> {
                if (accountsSpaces.isEmpty() || !provider.isAccountsSpacesSupported()) {
                    return syncOneAccountsSpace(
                            provider, null, externalIndex, false, true, locale, clock, syncStatus
                    ).onErrorResume(e -> {
                        ProvidersSyncErrorsModel error = accountsSyncStoreService.newError(
                                null, syncStatus, null, ExceptionUtils.getMessage(e));
                        LOG.error(MessageFormat.format("Failed to sync accounts for provider {0} (errorId={1})",
                                provider.getKey(), error.getErrorId()), e);
                        return accountsSyncStoreService.insertSyncError(error)
                                .map(u -> ProcessFoldersPageResult.error(error));
                    });
                } else {
                    return Flux.fromIterable(accountsSpaces).filter(AccountSpaceModel::isSyncEnabled)
                            .concatMap(accountsSpace -> syncOneAccountsSpace(provider, accountsSpace, externalIndex,
                                    false, true, locale, clock, syncStatus
                            ).onErrorResume(e -> {
                                ProvidersSyncErrorsModel error = accountsSyncStoreService.newError(
                                        accountsSpace, syncStatus, null, ExceptionUtils.getMessage(e));
                                LOG.error(MessageFormat.format("Failed to sync accounts for provider {0} in accounts " +
                                        "space {1} (errorId={2})", provider.getKey(), accountsSpace, error.getErrorId()
                                        ), e);
                                return accountsSyncStoreService.insertSyncError(error)
                                        .map(u -> ProcessFoldersPageResult.error(error))
                                        .onErrorResume(throwable -> {
                                            LOG.error("Can't save sync error information", throwable);
                                            return Mono.just(ProcessFoldersPageResult.error(error));
                                        });
                            }))
                            .collectList().map(ProcessFoldersPageResult::concat);
                }
            });
        }).flatMap(result -> accountsSyncStoreService.updateSyncStatus(syncStatus, result, clock)));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<ProcessFoldersPageResult> syncOneAccountsSpace(
            ProviderModel provider, AccountSpaceModel accountsSpace,
            Map<ExternalCompoundResourceKey, SyncResource> externalIndex, boolean doPostponedFolders,
            boolean paginateSync, Locale locale, Clock clock,
            ProvidersSyncStatusModel syncStatus
    ) {
        List<SyncResource> accountsSpaceResources = AccountsSyncUtils.filterResourcesByAccountsSpace(
                externalIndex.values(), accountsSpace);
        return accountsSyncStoreService.prepareAccountsSpaceKey(accountsSpace).flatMap(spaceKey ->
                accountsSyncRequestService.requestAccountsFromProvider(provider, spaceKey.orElse(null),
                        externalIndex, locale, syncStatus).flatMap(accounts ->
                        processReceivedAccounts(provider, accounts, accountsSpaceResources, accountsSpace,
                                doPostponedFolders, paginateSync, clock, syncStatus)
                        .flatMap(result -> processPostponedFolders(
                                provider, result, accountsSpaceResources, doPostponedFolders, paginateSync))
                        .flatMap(result -> processAccountsDeletion(
                                provider, accountsSpace, accountsSpaceResources, result, clock, syncStatus))
                .map(result -> result.setHasErrors(accounts.hasAnyErrors()))
        ));
    }

    private Mono<ProcessFoldersPageResult> processAccountsDeletion(
            ProviderModel provider, AccountSpaceModel accountsSpace,
            List<SyncResource> accountsSpaceResources,
            ProcessFoldersPageResult result, Clock clock, ProvidersSyncStatusModel syncStatus
    ) {

        if (!provider.getAccountsSettings().isDeleteSupported()
                && !provider.getAccountsSettings().isSoftDeleteSupported()
        ) {
            return Mono.just(result);
        }
        return tableClient.usingSessionMonoRetryable(session -> session
                .usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        accountsSyncStoreService.getAllNonDeletedAccountsByAccountsSpaceExcluding(txSession,
                                provider, accountsSpace, result.getAllReceivedAccountExternalIds())
                                .flatMap(deletedAccounts ->
                                        processAccountsDeletion(txSession, provider, accountsSpace,
                                                accountsSpaceResources, deletedAccounts, clock, syncStatus))))
                .thenReturn(result);
    }

    private Mono<Void> processAccountsDeletion(YdbTxSession session,
                                               ProviderModel provider,
                                               AccountSpaceModel accountsSpace,
                                               List<SyncResource> accountsSpaceResources,
                                               List<AccountModel> deletedAccounts,
                                               Clock clock, ProvidersSyncStatusModel syncStatus) {
        if (deletedAccounts.isEmpty()) {
            return Mono.empty();
        }
        Set<String> updatedFolderIds = deletedAccounts.stream().map(AccountModel::getFolderId)
                .collect(Collectors.toSet());
        return accountsSyncStoreService.getAccountsByFolders(session, updatedFolderIds, accountsSpace, provider)
                .flatMap(updatedFoldersAccounts -> {
                    Set<String> resourceIds = accountsSpaceResources.stream().map(r -> r.getResource().getId())
                            .collect(Collectors.toSet());
                    Map<String, ResourceModel> resourcesById = accountsSpaceResources.stream().collect(Collectors
                            .toMap(r -> r.getResource().getId(), SyncResource::getResource));
                    Set<String> accountIds = updatedFoldersAccounts.stream().map(AccountModel::getId)
                            .collect(Collectors.toSet());
                    return accountsSyncStoreService.getQuotas(session, updatedFolderIds, provider.getId(), resourceIds)
                            .flatMap(quotas -> accountsSyncStoreService.getProvisions(session, accountIds)
                                    .flatMap(provisions -> accountsSyncStoreService.loadOperations(session,
                                            updatedFolderIds, accountIds, Set.of(), provider.getId(), accountsSpace)
                                                .flatMap(opsTuple -> processAccountsDeletion(session, provider,
                                                        deletedAccounts, quotas, provisions, opsTuple.getT1(),
                                                        clock, syncStatus, resourcesById))));
                });
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Void> processAccountsDeletion(YdbTxSession session,
                                               ProviderModel provider,
                                               List<AccountModel> deletedAccounts,
                                               List<QuotaModel> quotas,
                                               List<AccountsQuotasModel> provisions,
                                               List<AccountsQuotasOperationsModel> operationsInProgress,
                                               Clock clock, ProvidersSyncStatusModel syncStatus,
                                               Map<String, ResourceModel> resourcesById
    ) {
        Map<String, Map<String, QuotaModel>> quotasByFolderResource = new HashMap<>();
        Map<String, List<AccountsQuotasOperationsModel>> operationsInProgressByAccount = new HashMap<>();
        Map<String, Map<String, AccountsQuotasModel>> provisionsByAccountResource = new HashMap<>();
        AccountsSyncUtils.groupQuotasAndOperation(quotas, provisions, operationsInProgress, quotasByFolderResource,
                operationsInProgressByAccount, provisionsByAccountResource);
        List<AccountModel> accountsToUpsert = new ArrayList<>();
        List<AccountModel> accountsToRemoveReserve = new ArrayList<>();
        List<QuotaModel> quotasToUpsert = new ArrayList<>();
        List<AccountsQuotasModel> provisionsToUpsert = new ArrayList<>();
        List<FolderOperationLogModel.Builder> folderLogsToUpsert = new ArrayList<>();
        Map<String, List<String>> sortedFolderLogIdsByFolder = new HashMap<>();
        Map<String, Map<String, Long>> updatedBalances = new HashMap<>();
        Map<String, Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>>> oldProvisions
                = new HashMap<>();
        Map<String, Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>>> newProvisions
                = new HashMap<>();
        Map<String, Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>>> oldAccounts = new HashMap<>();
        Map<String, Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>>> newAccounts = new HashMap<>();
        Map<String, Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>>> operationAuthors = new HashMap<>();
        Set<String> folderIdsToUpdate = new HashSet<>();
        Set<String> deletedAccountIds = new HashSet<>();
        Instant now = Instant.now(clock);
        deletedAccounts.forEach(deletedAccount -> {
            AccountsSyncUtils.processAccountDeletion(operationsInProgressByAccount, accountsToUpsert, oldAccounts,
                    newAccounts, folderIdsToUpdate, now, deletedAccount, deletedAccountIds, provider, syncStatus,
                    accountsToRemoveReserve);
        });
        folderIdsToUpdate.forEach(folderId -> {
            Map<String, Long> folderUpdatedBalances = updatedBalances.computeIfAbsent(folderId, k -> new HashMap<>());
            Map<String, QuotaModel> folderQuotas = quotasByFolderResource
                    .computeIfAbsent(folderId, k -> Collections.emptyMap());
            AccountsSyncUtils.prepareUpdatedBalances(folderUpdatedBalances, folderQuotas);
            operationAuthors.put(folderId, Map.of(new FolderHistoryGroupKey(null, null),
                    Set.of(new FolderOperationAuthor(null, null, provider.getId()))));
        });
        provisions.forEach(provision -> AccountsSyncUtils.processProvisionForAccountDeletion(provisionsToUpsert,
                updatedBalances, oldProvisions, newProvisions, deletedAccountIds, provision, now));
        updatedBalances.forEach((folderId, folderBalances) -> folderBalances.forEach((resourceId, balance) ->
                AccountsSyncUtils.prepareBalancesForAccountDeletion(provider, quotasByFolderResource, quotasToUpsert,
                folderId, resourceId, balance, resourcesById.get(resourceId))));
        folderIdsToUpdate.forEach(folderId -> folderLogsToUpsert.addAll(AccountsSyncOpLogUtils
                .prepareFolderOperationLogs(
                        oldProvisions.computeIfAbsent(folderId, k -> new HashMap<>()),
                        newProvisions.computeIfAbsent(folderId, k -> new HashMap<>()),
                        oldAccounts.computeIfAbsent(folderId, k -> new HashMap<>()),
                        newAccounts.computeIfAbsent(folderId, k -> new HashMap<>()),
                        operationAuthors.computeIfAbsent(folderId, k -> new HashMap<>()),
                        Map.of(), sortedFolderLogIdsByFolder, provider, folderId, now, Map.of(), Map.of(),
                        quotasByFolderResource.getOrDefault(folderId, Collections.emptyMap()), resourcesById)));
        return accountsSyncStoreService.prepareOrders(folderLogsToUpsert, Map.of(), sortedFolderLogIdsByFolder)
                .flatMap(tuple -> accountsSyncStoreService.doUpdates(session, accountsToUpsert, quotasToUpsert,
                        provisionsToUpsert, tuple.getT1(), tuple.getT2(), Set.of()).flatMap(l ->
                        reserveAccountsService.removeReserveAccountsMono(session, accountsToRemoveReserve)
                                .thenReturn(l))).then();
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<ProcessFoldersPageResult> processReceivedAccounts(
            ProviderModel provider,
            AccumulatedAccounts accounts,
            List<SyncResource> accountsSpaceResources,
            AccountSpaceModel accountsSpace,
            boolean doPostponedFolders,
            boolean paginateSync,
            Clock clock,
            ProvidersSyncStatusModel syncStatus
    ) {
        Map<String, ExternalAccount> accountsByExternalId = accounts.getAccountsToSync().stream()
                .collect(Collectors.toMap(ExternalAccount::getAccountId, Function.identity(),
                        AccountsSyncUtils::mergeAccounts));
        Map<String, List<ExternalAccount>> accountsByTargetFolder = accountsByExternalId.values().stream()
                .collect(Collectors.groupingBy(ExternalAccount::getFolderId, Collectors.toList()));
        Set<String> targetFolders = accountsByTargetFolder.keySet();
        return paginateFolder(targetFolders, paginateSync, provider, accountsSpace, accountsByExternalId,
                accountsByTargetFolder).flatMap(pages -> Flux.fromIterable(pages)
                .concatMap(page -> processFoldersPage(
                        provider,
                        accountsByTargetFolder,
                        accountsByExternalId,
                        new HashSet<>(page),
                        accountsSpace,
                        accounts.getAllAccountIds(),
                        accountsSpaceResources,
                        doPostponedFolders,
                        clock,
                        syncStatus
                ))
                .collectList().map(ProcessFoldersPageResult::concat)
                .flatMap(pr -> processLastPage(provider, accounts, accountsSpaceResources, accountsSpace,
                        doPostponedFolders, clock, accountsByExternalId, accountsByTargetFolder, pr, syncStatus))
        );
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<ProcessFoldersPageResult> processLastPage(
            ProviderModel provider, AccumulatedAccounts accounts,
            List<SyncResource> accountsSpaceResources, AccountSpaceModel accountsSpace,
            boolean doPostponedFolders, Clock clock,
            Map<String, ExternalAccount> accountsByExternalId,
            Map<String, List<ExternalAccount>> accountsByTargetFolder,
            ProcessFoldersPageResult foldersPageResult,
            ProvidersSyncStatusModel syncStatus
    ) {

        Set<String> foldersPage = foldersPageResult.getFolderIdsWithMovedAccounts();
        if (foldersPage.isEmpty()) {
            return Mono.just(foldersPageResult);
        }

        return processFoldersPage(provider, accountsByTargetFolder,
                accountsByExternalId,
                foldersPage, accountsSpace, accounts.getAllAccountIds(), accountsSpaceResources,
                doPostponedFolders, clock, syncStatus)
                .expand(pr -> {
                    Set<String> nextFolderPage = pr.getFolderIdsWithMovedAccounts();
                    if  (nextFolderPage.isEmpty()) {
                        return Mono.empty();
                    }
                    return processFoldersPage(provider,
                            accountsByTargetFolder, accountsByExternalId, nextFolderPage, accountsSpace,
                            accounts.getAllAccountIds(), accountsSpaceResources, doPostponedFolders, clock, syncStatus);
                })
                .collectList()
                .map(tail -> ProcessFoldersPageResult.concat(
                        Iterables.concat(Collections.singleton(foldersPageResult), tail)));
    }

    private Mono<List<List<String>>> paginateFolder(Set<String> folders, boolean paginateSync,
                                                    ProviderModel provider,
                                                    AccountSpaceModel accountsSpace,
                                                    Map<String, ExternalAccount> accountsByExternalId,
                                                    Map<String, List<ExternalAccount>> accountsByTargetFolder) {
        if (paginateSync) {
            return splitFoldersByPages(folders, provider, accountsSpace, accountsByExternalId, accountsByTargetFolder);
        }
        return Mono.just(List.of(Lists.newArrayList(folders)));
    }

    private Mono<ProcessFoldersPageResult> processPostponedFolders(ProviderModel provider,
                                                      ProcessFoldersPageResult result,
                                                      List<SyncResource> accountsSpaceResources,
                                                      boolean doPostponedFolders,
                                                      boolean paginateSync) {

        Set<String> postponedFolders = result.getPostponedFolderIds();
        Set<String> allReceivedAccountsIds = result.getAllReceivedAccountExternalIds();

        if (!doPostponedFolders) {
            return Mono.just(result);
        }
        // TODO Implement this
        return Mono.just(result);
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<ProcessFoldersPageResult> processFoldersPage(
            ProviderModel provider,
            Map<String, List<ExternalAccount>> accountsByTargetFolder,
            Map<String, ExternalAccount> accountsByExternalId,
            Set<String> targetFoldersPage,
            AccountSpaceModel accountsSpace,
            Set<String> allReceivedAccountExternalIds,
            List<SyncResource> accountsSpaceResources,
            boolean doPostponedFolders,
            Clock clock,
            ProvidersSyncStatusModel syncStatus
    ) {
        Map<String, String> preGeneratedAccountIds = AccountsSyncUtils.preGenerateAccountIds(accountsByTargetFolder,
                targetFoldersPage);
        Map<String, SyncResource> resourceById = accountsSpaceResources.stream()
                .collect(Collectors.toMap(r -> r.getResource().getId(), Function.identity()));
        return tableClient.usingSessionMonoRetryable(session -> session
                .usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        processFoldersPage(txSession, provider, accountsByTargetFolder, accountsByExternalId,
                                targetFoldersPage, accountsSpace, allReceivedAccountExternalIds,
                                resourceById, preGeneratedAccountIds, doPostponedFolders, clock, syncStatus)));
    }

    private Mono<List<List<String>>> splitFoldersByPages(
            Set<String> folders,
            ProviderModel provider,
            AccountSpaceModel accountsSpace,
            Map<String, ExternalAccount> accountsByExternalId,
            Map<String, List<ExternalAccount>> accountsByTargetFolder
    ) {
        return tableClient.usingSessionMonoRetryable(session -> session
                .usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        getAllAffectedAccounts(txSession, folders, provider, accountsSpace, accountsByTargetFolder)
                                .map(currentFoldersAccounts -> {
                                    Set<String> folderIdsWithMovedAccounts = new HashSet<>();
                                    currentFoldersAccounts.forEach(ca -> {
                                        var externalAccount
                                                = accountsByExternalId.get(ca.getOuterAccountIdInProvider());
                                        if (externalAccount != null
                                                && !externalAccount.getFolderId().equals(ca.getFolderId())) {
                                            folderIdsWithMovedAccounts.add(externalAccount.getFolderId());
                                            folderIdsWithMovedAccounts.add(ca.getFolderId());
                                        }
                                    });

                                    List<List<String>> pages;
                                    if (folderIdsWithMovedAccounts.isEmpty()) {
                                        pages = folders.isEmpty() ? List.of() :
                                                Lists.partition(Lists.newArrayList(folders), SYNC_FOLDERS_PAGE_SIZE);
                                    } else {
                                        Set<String> folderIdsWithoutMovedAccounts
                                                = Sets.difference(folders, folderIdsWithMovedAccounts);
                                        List<List<String>> otherPages = folderIdsWithoutMovedAccounts.isEmpty() ?
                                                List.of() : Lists.partition(Lists.newArrayList(
                                                        folderIdsWithoutMovedAccounts), SYNC_FOLDERS_PAGE_SIZE);
                                        pages = Lists.newArrayListWithCapacity(otherPages.size() + 1);
                                        pages.add(Lists.newArrayList(folderIdsWithMovedAccounts));
                                        pages.addAll(otherPages);
                                    }
                                    return pages;
                                })));
    }

    private Mono<Iterable<AccountModel>> getAllAffectedAccounts(
            YdbTxSession session,
            Set<String> folders,
            ProviderModel provider,
            AccountSpaceModel accountsSpace,
            Map<String, List<ExternalAccount>> accountsByTargetFolder) {

        return accountsSyncStoreService.getAccountsByFolders(session, folders, accountsSpace, provider)
                .flatMap(currentFoldersAccounts -> {
                    Set<String> externalAccountIdsCurrentlyInFolders = currentFoldersAccounts.stream()
                            .map(AccountModel::getOuterAccountIdInProvider).collect(Collectors.toSet());
                    Set<String> receivedExternalAccountIdsForFolders = folders.stream()
                            .flatMap(v -> accountsByTargetFolder.getOrDefault(v, Collections.emptyList()).stream())
                            .map(ExternalAccount::getAccountId).collect(Collectors.toSet());
                    Set<String> externalAccountIdsMovedTo = Sets.difference(
                            receivedExternalAccountIdsForFolders, externalAccountIdsCurrentlyInFolders);
                    return accountsSyncStoreService.getAccountsByExternalIds(session,
                            externalAccountIdsMovedTo, accountsSpace, provider)
                            .map(existingAccountsMovedToFolders -> Iterables.concat(
                                    currentFoldersAccounts, existingAccountsMovedToFolders)
                            );
                });
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<ProcessFoldersPageResult> processFoldersPage(
            YdbTxSession session,
            ProviderModel provider,
            Map<String, List<ExternalAccount>> accountsByTargetFolder,
            Map<String, ExternalAccount> accountsByExternalId,
            Set<String> foldersPage,
            AccountSpaceModel accountsSpace,
            Set<String> allReceivedAccountExternalIds,
            Map<String, SyncResource> resourceById,
            Map<String, String> preGeneratedAccountIds,
            boolean doPostponedFolders,
            Clock clock,
            ProvidersSyncStatusModel syncStatus
    ) {
        return accountsSyncStoreService.getAccountsByFolders(session, foldersPage, accountsSpace, provider)
                .flatMap(currentFoldersPageAccounts -> {
                    Set<String> externalAccountIdsCurrentlyInFoldersPage = currentFoldersPageAccounts.stream()
                            .map(AccountModel::getOuterAccountIdInProvider).collect(Collectors.toSet());
                    Set<String> receivedExternalAccountIdsForFoldersPage = foldersPage.stream()
                            .flatMap(v -> accountsByTargetFolder.getOrDefault(v, Collections.emptyList()).stream())
                            .map(ExternalAccount::getAccountId).collect(Collectors.toSet());
                    Set<String> externalAccountIdsMovedToFoldersPage = Sets.difference(
                            receivedExternalAccountIdsForFoldersPage, externalAccountIdsCurrentlyInFoldersPage);
                    Set<String> externalAccountIdsMovedFromFoldersPage = Sets.difference(
                            externalAccountIdsCurrentlyInFoldersPage, receivedExternalAccountIdsForFoldersPage);
                    Set<String> allExternalAccountIds = Sets.union(externalAccountIdsCurrentlyInFoldersPage,
                            receivedExternalAccountIdsForFoldersPage);
                    return accountsSyncStoreService.getAccountsByExternalIds(session,
                            externalAccountIdsMovedToFoldersPage, accountsSpace, provider)
                            .flatMap(existingAccountsMovedToFoldersPage -> {
                                Set<String> folderIdsFromWhichAccountsMoved = existingAccountsMovedToFoldersPage
                                        .stream().map(AccountModel::getFolderId).collect(Collectors.toSet());
                                Set<String> folderIdsToWhichAccountsMoved = externalAccountIdsMovedFromFoldersPage
                                        .stream().flatMap(id -> Optional.ofNullable(accountsByExternalId.get(id))
                                                .stream()).map(ExternalAccount::getFolderId)
                                        .collect(Collectors.toSet());
                                Sets.SetView<String> folderIdsWithMovedAccounts = Sets.union(
                                        folderIdsFromWhichAccountsMoved, folderIdsToWhichAccountsMoved);
                                Set<String> actualFoldersPage
                                        = Sets.difference(foldersPage, folderIdsWithMovedAccounts);
                                GroupedAccounts groupedAccounts = AccountsSyncGroupingUtils.groupAccounts(provider,
                                        foldersPage, accountsByTargetFolder, accountsByExternalId,
                                        currentFoldersPageAccounts, existingAccountsMovedToFoldersPage,
                                        allReceivedAccountExternalIds);
                                Set<String> resourceIds = resourceById.values().stream()
                                        .map(r -> r.getResource().getId()).collect(Collectors.toSet());
                                Set<String> receivedOperationIds = AccountsSyncUtils.collectOperationIds(
                                        provider, actualFoldersPage, accountsByTargetFolder);
                                return accountsSyncStoreService.loadQuotasAndOperations(session, provider.getId(),
                                        actualFoldersPage, resourceIds, groupedAccounts.getAllAccountIds(),
                                        receivedOperationIds, currentFoldersPageAccounts,
                                        existingAccountsMovedToFoldersPage, accountsSpace
                                ).flatMap(quotasAndOperations -> accountsSyncStoreService
                                        .getAuthors(session, allExternalAccountIds, accountsByExternalId)
                                        .flatMap(authors -> processFoldersPageAccounts(session, provider, accountsSpace,
                                                actualFoldersPage, groupedAccounts, quotasAndOperations,
                                                preGeneratedAccountIds, authors, doPostponedFolders, clock, syncStatus,
                                                resourceById)
                                        )
                                ).map(processed -> new ProcessFoldersPageResult(processed.getPostponedFolderIds(),
                                        folderIdsWithMovedAccounts, allReceivedAccountExternalIds,
                                        processed.getReceivedQuotasCount(), processed.getErrors(),
                                        processed.hasErrors()));
                            });
                });
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<ProcessFoldersPageAccountsResult> processFoldersPageAccounts(
            YdbTxSession session,
            ProviderModel provider,
            AccountSpaceModel accountsSpace,
            Set<String> foldersPage,
            GroupedAccounts groupedAccounts,
            QuotasAndOperations quotasAndOperations,
            Map<String, String> preGeneratedAccountIds,
            GroupedAuthors authors,
            boolean doPostponedFolders,
            Clock clock,
            ProvidersSyncStatusModel syncStatus,
            Map<String, SyncResource> resourceById
    ) {
        Set<String> postponedFolders = doPostponedFolders
                ? AccountsPostponedSyncUtils.findPostponedFolders(provider, foldersPage, groupedAccounts,
                        quotasAndOperations)
                : Collections.emptySet();
        Set<String> foldersToProcess = Sets.difference(foldersPage, postponedFolders);
        if (foldersToProcess.isEmpty()) {
            return Mono.just(new ProcessFoldersPageAccountsResult(postponedFolders, 0, null, false));
        }
        boolean accountAndProvisionsVersionedTogether = provider.getAccountsSettings().isPerAccountVersionSupported()
                && !provider.getAccountsSettings().isPerProvisionVersionSupported();
        boolean accountVersionedSeparately = provider.getAccountsSettings().isPerAccountVersionSupported()
                && provider.getAccountsSettings().isPerProvisionVersionSupported();
        boolean provisionsVersionedSeparately = provider.getAccountsSettings().isPerProvisionVersionSupported();
        boolean accountOperationsSupported = provider.getAccountsSettings().isPerAccountLastUpdateSupported();
        boolean provisionOperationsSupported = provider.getAccountsSettings().isPerProvisionLastUpdateSupported();
        boolean softDeleteSupported = provider.getAccountsSettings().isSoftDeleteSupported();
        boolean accountNameSupported = provider.getAccountsSettings().isDisplayNameSupported();
        boolean accountKeySupported = provider.getAccountsSettings().isKeySupported();
        List<AccountModel> accountsToUpsert = new ArrayList<>();
        List<QuotaModel> quotasToUpsert = new ArrayList<>();
        List<AccountsQuotasModel> provisionsToUpsert = new ArrayList<>();
        List<FolderOperationLogModel.Builder> folderLogsToUpsert = new ArrayList<>();
        Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert = new HashMap<>();
        Map<String, List<String>> sortedFolderLogIdsByFolder = new HashMap<>();
        Instant now = Instant.now(clock);
        Map<String, AccountMoveHistoryXRef> accountMoveXRefsByAccountId = new HashMap<>();
        Map<String, AccountMoveFolders> accountMoveFoldersByAccountId = new HashMap<>();
        Set<String> skippedMoveAccountsExternalIds = new HashSet<>();
        List<ProvidersSyncErrorsModel> errors = new ArrayList<>();
        List<AccountModel> accountsToRemoveReserve = new ArrayList<>();
        foldersToProcess.forEach(folderId -> {
            Map<String, Long> updatedBalances = new HashMap<>();
            Map<String, Long> defaultQuotas = new HashMap<>();
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions = new HashMap<>();
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions = new HashMap<>();
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> oldAccounts = new HashMap<>();
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts = new HashMap<>();
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors = new HashMap<>();
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates = new HashMap<>();
            Map<String, QuotaModel> currentQuotas = quotasAndOperations.getQuotasByFolderResource()
                    .getOrDefault(folderId, Collections.emptyMap());
            Set<String> operationsInProgress = quotasAndOperations.getOperationsInProgress();
            Map<String, AccountsQuotasOperationsModel> operationsById = quotasAndOperations.getOperationsById();
            AccountsSyncUtils.prepareUpdatedBalances(updatedBalances, currentQuotas);
            prepareNewAccounts(provider, accountsSpace, groupedAccounts, preGeneratedAccountIds,
                    accountAndProvisionsVersionedTogether, accountVersionedSeparately, provisionsVersionedSeparately,
                    accountOperationsSupported, provisionOperationsSupported, softDeleteSupported,
                    accountNameSupported, accountKeySupported, accountsToUpsert, provisionsToUpsert, now, folderId,
                    updatedBalances, defaultQuotas, oldProvisions, newProvisions, newAccounts, operationsInProgress,
                    operationsById, operationsToUpsert, authors, operationAuthors, newestLastUpdates, syncStatus,
                    errors);
            prepareStayAccounts(provider, groupedAccounts, accountAndProvisionsVersionedTogether,
                    accountVersionedSeparately, provisionsVersionedSeparately, accountOperationsSupported,
                    provisionOperationsSupported, softDeleteSupported, accountNameSupported, accountKeySupported,
                    accountsToUpsert, provisionsToUpsert, now, folderId, updatedBalances,
                    oldProvisions, newProvisions, oldAccounts, newAccounts, operationsInProgress, operationsById,
                    operationsToUpsert, authors, operationAuthors, newestLastUpdates, quotasAndOperations, syncStatus,
                    errors, accountsToRemoveReserve);
            prepareMoveAccountsDestination(provider, groupedAccounts, accountAndProvisionsVersionedTogether,
                    accountVersionedSeparately, provisionsVersionedSeparately, accountOperationsSupported,
                    provisionOperationsSupported, softDeleteSupported, accountNameSupported, accountKeySupported,
                    accountsToUpsert, provisionsToUpsert, now, folderId, updatedBalances, defaultQuotas,
                    oldProvisions, newProvisions, oldAccounts, newAccounts, operationsInProgress, operationsById,
                    operationsToUpsert, authors, operationAuthors, newestLastUpdates, quotasAndOperations,
                    accountMoveFoldersByAccountId, skippedMoveAccountsExternalIds, syncStatus, errors,
                    accountsToRemoveReserve);
            prepareMoveAccountsSource(provider, groupedAccounts, accountAndProvisionsVersionedTogether,
                    accountVersionedSeparately, provisionsVersionedSeparately, accountOperationsSupported,
                    provisionOperationsSupported, softDeleteSupported, accountNameSupported, accountKeySupported,
                    provisionsToUpsert, now, folderId, updatedBalances, defaultQuotas, oldProvisions,
                    newProvisions, oldAccounts, newAccounts, operationsInProgress, operationsById, authors,
                    operationAuthors, newestLastUpdates, quotasAndOperations, accountMoveFoldersByAccountId,
                    skippedMoveAccountsExternalIds, syncStatus, errors);
            // TODO Handle moves between pages
            // TODO Restore phase for reserve provision does not change quotas yet, should not be a real problem
            AccountsSyncUtils.prepareQuotasToUpsert(quotasToUpsert, updatedBalances, defaultQuotas, currentQuotas,
                    folderId, provider);
            folderLogsToUpsert.addAll(AccountsSyncOpLogUtils.prepareFolderOperationLogs(oldProvisions, newProvisions,
                    oldAccounts, newAccounts, operationAuthors, newestLastUpdates, sortedFolderLogIdsByFolder,
                    provider, folderId, now, accountMoveXRefsByAccountId, accountMoveFoldersByAccountId,
                    currentQuotas, resourceById.values().stream().collect(Collectors
                            .toMap(o -> o.getResource().getId(), SyncResource::getResource))));
        });
        AccountsSyncOpLogUtils.prepareOpLogCrossReferences(folderLogsToUpsert, quotasAndOperations.getOperationsById(),
                accountMoveXRefsByAccountId);
        return accountsSyncStoreService.prepareOrders(folderLogsToUpsert, operationsToUpsert,
                sortedFolderLogIdsByFolder)
                .flatMap(tuple -> accountsSyncStoreService.doUpdates(session, accountsToUpsert, quotasToUpsert,
                        provisionsToUpsert, tuple.getT1(), tuple.getT2(), postponedFolders)
                        .flatMap(l -> reserveAccountsService.removeReserveAccountsMono(session,
                                accountsToRemoveReserve).thenReturn(l)))
                .map(postponedFolderIds -> new ProcessFoldersPageAccountsResult(
                        postponedFolderIds,
                        quotasToUpsert.size(),
                        errors,
                        false));
    }

    @SuppressWarnings("ParameterNumber")
    private void prepareNewAccounts(
            ProviderModel provider,
            AccountSpaceModel accountsSpace,
            GroupedAccounts groupedAccounts,
            Map<String, String> preGeneratedAccountIds,
            boolean accountAndProvisionsVersionedTogether,
            boolean accountVersionedSeparately,
            boolean provisionsVersionedSeparately,
            boolean accountOperationsSupported,
            boolean provisionOperationsSupported,
            boolean softDeleteSupported,
            boolean accountNameSupported,
            boolean accountKeySupported,
            List<AccountModel> accountsToUpsert,
            List<AccountsQuotasModel> provisionsToUpsert,
            Instant now,
            String folderId,
            Map<String, Long> updatedBalances,
            Map<String, Long> defaultQuotas,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts,
            Set<String> operationsInProgress,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
            GroupedAuthors authors,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates,
            ProvidersSyncStatusModel syncStatus,
            List<ProvidersSyncErrorsModel> errors
    ) {
        groupedAccounts.getReceivedAccountsToCreate().getOrDefault(folderId, Collections.emptyMap())
                .forEach((externalAccountId, account) -> {
                    if (checkInProgressOperations(externalAccountId, account,
                            accountOperationsSupported, provisionOperationsSupported, operationsInProgress,
                            operationsById, folderId, true, null)) {
                        return;
                    }
                    Optional<String> accountCreationOperationId = AccountsSyncUtils.getAccountOperationId(account,
                            operationsById, accountOperationsSupported, folderId, null);
                    FolderHistoryGroupKey accountCreationHistoryGroupKey
                            = new FolderHistoryGroupKey(accountCreationOperationId.orElse(null), null);
                    Set<FolderOperationAuthor> accountCreationAuthors = AccountsSyncUtils.collectAuthors(provider,
                            account::getLastUpdate, authors);
                    operationAuthors.computeIfAbsent(accountCreationHistoryGroupKey, k -> new HashSet<>())
                            .addAll(accountCreationAuthors);
                    AccountsSyncUtils.refreshNewestLastUpdates(accountCreationHistoryGroupKey, account::getLastUpdate,
                            newestLastUpdates);
                    AccountModel newAccount = AccountsSyncUtils.buildNewAccount(provider, accountsSpace,
                            preGeneratedAccountIds, accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            accountOperationsSupported, softDeleteSupported, accountNameSupported,
                            accountKeySupported, now, folderId, externalAccountId, account, operationsById);
                    accountsToUpsert.add(newAccount);
                    AccountHistoryModel newAccountHistory = AccountHistoryModel.builder()
                            .version(newAccount.getVersion())
                            .providerId(newAccount.getProviderId())
                            .outerAccountIdInProvider(newAccount.getOuterAccountIdInProvider())
                            .outerAccountKeyInProvider(newAccount.getOuterAccountKeyInProvider().orElse(null))
                            .folderId(newAccount.getFolderId())
                            .displayName(newAccount.getDisplayName().orElse(null))
                            .deleted(newAccount.isDeleted())
                            .lastReceivedVersion(newAccount.getLastReceivedVersion().orElse(null))
                            .accountsSpacesId(newAccount.getAccountsSpacesId().orElse(null))
                            .build();
                    newAccounts.computeIfAbsent(accountCreationHistoryGroupKey, k -> new HashMap<>())
                            .put(newAccount.getId(), newAccountHistory);
                    AccountsSyncUtils.handleOperation(accountCreationOperationId.orElse(null), operationsToUpsert,
                            operationsById, now, null, folderId);
                    account.getProvisions().forEach(provision -> {
                        ResourceModel resource = provision.getResource().getResource();
                        Optional<Long> providedAmountO = Units.convertFromApi(provision.getProvided(),
                                resource, provision.getResource().getUnitsEnsemble(),
                                provision.getProvidedUnit());
                        if (providedAmountO.isEmpty()) {
                            errors.add(accountsSyncStoreService.newError(accountsSpace, syncStatus, Map.of(
                                    "accountId", account.getAccountId(),
                                    "resourceKey", resource.getKey(),
                                    "provided", Long.toString(provision.getProvided()),
                                    "providedUnitKey", provision.getProvidedUnit().getKey()
                            ), "Invalid provision (empty result)"));
                            LOG.error("Invalid provision {} {} received for resource {} in account {} of provider {}",
                                    provision.getProvided(), provision.getProvidedUnit().getKey(),
                                    resource.getKey(), account.getAccountId(),
                                    provider.getKey());
                        }
                        long providedAmount = providedAmountO.orElse(0L);
                        Optional<Long> allocatedAmountO = Units.convertFromApi(provision.getAllocated(),
                                resource, provision.getResource().getUnitsEnsemble(),
                                provision.getAllocatedUnit());
                        long allocatedAmount = allocatedAmountO.orElse(0L);
                        if (allocatedAmountO.isEmpty()) {
                            errors.add(accountsSyncStoreService.newError(accountsSpace, syncStatus, Map.of(
                                    "accountId", account.getAccountId(),
                                    "resourceKey", resource.getKey()
                            ), "Invalid allocation (empty result)"));
                            LOG.error("Invalid allocation received for resource {} in account {} of provider {}",
                                    resource.getKey(), account.getAccountId(),
                                    provider.getKey());
                        }
                        String resourceId = resource.getId();
                        updateBalanceByProvisionAndDefault(updatedBalances, defaultQuotas, resource, providedAmount,
                                    errors, syncStatus, account.getAccountId(), folderId);
                        if (providedAmount != 0L || allocatedAmount != 0L) {
                            Optional<String> provisionUpdateOperationId
                                    = AccountsSyncUtils.getProvisionUpdateOperationId(provision,
                                            operationsById, provisionOperationsSupported, folderId, null);
                            FolderHistoryGroupKey provisionUpdateHistoryGroupKey
                                    = new FolderHistoryGroupKey(provisionUpdateOperationId.orElse(null), null);
                            Set<FolderOperationAuthor> provisionUpdateAuthors = AccountsSyncUtils.collectAuthors(
                                    provider, provision::getLastUpdate, authors);
                            operationAuthors.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashSet<>())
                                    .addAll(provisionUpdateAuthors);
                            AccountsSyncUtils.refreshNewestLastUpdates(provisionUpdateHistoryGroupKey,
                                    provision::getLastUpdate, newestLastUpdates);
                            AccountsQuotasModel newProvision = AccountsSyncUtils.newProvision(provider,
                                    provisionsVersionedSeparately, provisionOperationsSupported, now, folderId,
                                    newAccount, provision, operationsById, providedAmount, allocatedAmount);
                            provisionsToUpsert.add(newProvision);
                            ProvisionHistoryModel oldProvisionHistory = new ProvisionHistoryModel(0L, null);
                            ProvisionHistoryModel newProvisionHistory = new ProvisionHistoryModel(providedAmount,
                                    provisionsVersionedSeparately
                                            ? provision.getQuotaVersion().orElse(null) : null);
                            oldProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                                    .computeIfAbsent(newAccount.getId(), k -> new HashMap<>())
                                    .put(resourceId, oldProvisionHistory);
                            newProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                                    .computeIfAbsent(newAccount.getId(), k -> new HashMap<>())
                                    .put(resourceId, newProvisionHistory);
                            AccountsSyncUtils.handleOperation(provisionUpdateOperationId.orElse(null),
                                    operationsToUpsert, operationsById, now, null, folderId);
                        }
                    });
                });
    }

    @SuppressWarnings("ParameterNumber")
    private void prepareStayAccounts(
            ProviderModel provider,
            GroupedAccounts groupedAccounts,
            boolean accountAndProvisionsVersionedTogether,
            boolean accountVersionedSeparately,
            boolean provisionsVersionedSeparately,
            boolean accountOperationsSupported,
            boolean provisionOperationsSupported,
            boolean softDeleteSupported,
            boolean accountNameSupported,
            boolean accountKeySupported,
            List<AccountModel> accountsToUpsert,
            List<AccountsQuotasModel> provisionsToUpsert,
            Instant now,
            String folderId,
            Map<String, Long> updatedBalances,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> oldAccounts,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts,
            Set<String> operationsInProgress,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
            GroupedAuthors authors,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates,
            QuotasAndOperations quotasAndOperations,
            ProvidersSyncStatusModel syncStatus,
            List<ProvidersSyncErrorsModel> errors,
            List<AccountModel> accountsToRemoveReserve
    ) {
        groupedAccounts.getCurrentAccountsStay().getOrDefault(folderId, Collections.emptyMap())
                .forEach((externalAccountId, account) -> {
                    ExternalAccount updatedAccount = groupedAccounts.getReceivedAccountsStay()
                            .getOrDefault(folderId, Collections.emptyMap()).get(externalAccountId);
                    boolean isAccountToDelete = groupedAccounts.getCurrentAccountsToDeleteStay()
                            .getOrDefault(folderId, Collections.emptyMap()).containsKey(externalAccountId);
                    boolean isUpdatedAccountReceived = updatedAccount != null;
                    if (isUpdatedAccountReceived && AccountsSyncUtils.checkInProgressOperations(externalAccountId,
                            updatedAccount, accountOperationsSupported, provisionOperationsSupported,
                            operationsInProgress, operationsById, folderId, false, account)) {
                        updateBalance(account, quotasAndOperations, updatedBalances, Map.of(), folderId);
                        updateAllocations(provisionsToUpsert, updatedAccount,
                                account, quotasAndOperations, provider);
                        return;
                    }
                    if (isUpdatedAccountReceived && AccountsSyncUtils.checkStaleVersions(account, updatedAccount,
                            quotasAndOperations, accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            provisionsVersionedSeparately, accountNameSupported, accountOperationsSupported,
                            accountKeySupported, provisionOperationsSupported)) {
                        updateBalance(account, quotasAndOperations, updatedBalances, Map.of(),
                                folderId);
                        updateAllocations(provisionsToUpsert, updatedAccount,
                                account, quotasAndOperations, provider);
                        return;
                    }
                    if (checkCoolDownPeriod(account, updatedAccount, quotasAndOperations,
                            accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            provisionsVersionedSeparately, accountOperationsSupported, provisionOperationsSupported,
                            isAccountToDelete, softDeleteSupported, accountNameSupported, accountKeySupported, now,
                            provider, syncStatus)) {
                        updateBalance(account, quotasAndOperations, updatedBalances, Map.of(), folderId);
                        if (isUpdatedAccountReceived) {
                            updateAllocations(provisionsToUpsert, updatedAccount,
                                    account, quotasAndOperations, provider);
                        }
                        return;
                    }
                    if (isUpdatedAccountReceived && checkOperationsOrder(account, updatedAccount,
                            quotasAndOperations, accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            provisionsVersionedSeparately, accountOperationsSupported,
                            provisionOperationsSupported, accountNameSupported, accountKeySupported)) {
                        updateBalance(account, quotasAndOperations, updatedBalances, Map.of(), folderId);
                        updateAllocations(provisionsToUpsert, updatedAccount,
                                account, quotasAndOperations, provider);
                        return;
                    }
                    if (!isUpdatedAccountReceived && !isAccountToDelete) {
                        updateBalance(account, quotasAndOperations, updatedBalances, Map.of(), folderId);
                        return;
                    }
                    if (!isUpdatedAccountReceived && isAccountToDelete) {
                        doFullAccountDeletion(provider, accountsToUpsert, provisionsToUpsert, now, oldProvisions,
                                newProvisions, oldAccounts, newAccounts, operationAuthors,
                                quotasAndOperations, account, accountsToRemoveReserve);
                        return;
                    }
                    doAccountStayUpdate(provider, accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            accountOperationsSupported, softDeleteSupported, accountNameSupported, accountKeySupported,
                            accountsToUpsert, now, oldAccounts, newAccounts, operationsById, operationsToUpsert,
                            authors, operationAuthors, newestLastUpdates, account, updatedAccount, isAccountToDelete,
                            folderId, quotasAndOperations, accountsToRemoveReserve);
                    doAccountStayDeletedProvisionsUpdate(provider, provisionsToUpsert, oldProvisions, newProvisions,
                            operationAuthors, quotasAndOperations, account, updatedAccount, now);
                    doAccountStayModifiedProvisionsUpdate(provider, provisionsVersionedSeparately,
                            provisionOperationsSupported, provisionsToUpsert, now, folderId,
                            updatedBalances, oldProvisions, newProvisions, operationsById, operationsToUpsert,
                            authors, operationAuthors, newestLastUpdates, quotasAndOperations, account,
                            updatedAccount, syncStatus, errors);
                });
    }

    @SuppressWarnings("ParameterNumber")
    private void prepareMoveAccountsDestination(
            ProviderModel provider,
            GroupedAccounts groupedAccounts,
            boolean accountAndProvisionsVersionedTogether,
            boolean accountVersionedSeparately,
            boolean provisionsVersionedSeparately,
            boolean accountOperationsSupported,
            boolean provisionOperationsSupported,
            boolean softDeleteSupported,
            boolean accountNameSupported,
            boolean accountKeySupported,
            List<AccountModel> accountsToUpsert,
            List<AccountsQuotasModel> provisionsToUpsert,
            Instant now,
            String folderId,
            Map<String, Long> updatedBalances,
            Map<String, Long> defaultQuotas,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> oldAccounts,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts,
            Set<String> operationsInProgress,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
            GroupedAuthors authors,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates,
            QuotasAndOperations quotasAndOperations,
            Map<String, AccountMoveFolders> accountMoveFoldersByAccountId,
            Set<String> skippedMoveAccountsExternalIds,
            ProvidersSyncStatusModel syncStatus,
            List<ProvidersSyncErrorsModel> errors,
            List<AccountModel> accountsToRemoveReserve
    ) {
        groupedAccounts.getCurrentAccountsMovingInSamePage().getOrDefault(folderId, Collections.emptyMap())
                .forEach((externalAccountId, account) -> {
                    ExternalAccount updatedAccount = groupedAccounts.getReceivedAccountsMovingInSamePage()
                            .getOrDefault(folderId, Collections.emptyMap()).get(externalAccountId);
                    boolean isAccountToDelete = groupedAccounts.getCurrentAccountsToDeleteMovingInSamePage()
                            .getOrDefault(folderId, Collections.emptyMap()).containsKey(externalAccountId);
                    if (checkInProgressOperations(externalAccountId, updatedAccount,
                            accountOperationsSupported, provisionOperationsSupported, operationsInProgress,
                            operationsById, folderId, false, account)) {
                        skippedMoveAccountsExternalIds.add(externalAccountId);
                        return;
                    }
                    if (checkStaleVersions(account, updatedAccount, quotasAndOperations,
                            accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            provisionsVersionedSeparately, accountNameSupported, accountOperationsSupported,
                            accountKeySupported, provisionOperationsSupported)) {
                        skippedMoveAccountsExternalIds.add(externalAccountId);
                        return;
                    }
                    if (checkCoolDownPeriod(account, updatedAccount, quotasAndOperations,
                            accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            provisionsVersionedSeparately, accountOperationsSupported, provisionOperationsSupported,
                            isAccountToDelete, softDeleteSupported, accountNameSupported, accountKeySupported, now,
                            provider, syncStatus)) {
                        skippedMoveAccountsExternalIds.add(externalAccountId);
                        return;
                    }
                    if (checkOperationsOrder(account, updatedAccount, quotasAndOperations,
                            accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            provisionsVersionedSeparately, accountOperationsSupported,
                            provisionOperationsSupported, accountNameSupported, accountKeySupported)) {
                        skippedMoveAccountsExternalIds.add(externalAccountId);
                        return;
                    }
                    if (skippedMoveAccountsExternalIds.contains(externalAccountId)) {
                        return;
                    }
                    FolderHistoryGroupKey moveGroupKey = doAccountMoveDestinationUpdate(provider,
                            accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            accountOperationsSupported, softDeleteSupported, accountNameSupported,
                            accountKeySupported, accountsToUpsert, now, oldAccounts, newAccounts,
                            operationsById, operationsToUpsert, authors, operationAuthors,
                            newestLastUpdates, account, updatedAccount, isAccountToDelete, folderId,
                            accountMoveFoldersByAccountId, quotasAndOperations, accountsToRemoveReserve);
                    doAccountMoveDestinationDeletedProvisionsUpdate(provider, provisionsToUpsert,
                            oldProvisions, newProvisions, operationAuthors, quotasAndOperations, account,
                            updatedAccount, now);
                    doAccountMoveDestinationModifiedProvisionsUpdate(provider, provisionsVersionedSeparately,
                            provisionOperationsSupported, provisionsToUpsert, now, folderId,
                            updatedBalances, defaultQuotas, newProvisions, operationsById, operationsToUpsert, authors,
                            operationAuthors, newestLastUpdates, quotasAndOperations, account, updatedAccount,
                            moveGroupKey, syncStatus, errors);
                });
    }

    @SuppressWarnings("ParameterNumber")
    private void prepareMoveAccountsSource(
            ProviderModel provider,
            GroupedAccounts groupedAccounts,
            boolean accountAndProvisionsVersionedTogether,
            boolean accountVersionedSeparately,
            boolean provisionsVersionedSeparately,
            boolean accountOperationsSupported,
            boolean provisionOperationsSupported,
            boolean softDeleteSupported,
            boolean accountNameSupported,
            boolean accountKeySupported,
            List<AccountsQuotasModel> provisionsToUpsert,
            Instant now,
            String folderId,
            Map<String, Long> updatedBalances,
            Map<String, Long> defaultQuotas,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> oldAccounts,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts,
            Set<String> operationsInProgress,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            GroupedAuthors authors,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates,
            QuotasAndOperations quotasAndOperations,
            Map<String, AccountMoveFolders> accountMoveFoldersByAccountId,
            Set<String> skippedMoveAccountsExternalIds,
            ProvidersSyncStatusModel syncStatus,
            List<ProvidersSyncErrorsModel> errors
    ) {
        groupedAccounts.getCurrentAccountsMovingOutSamePage().getOrDefault(folderId, Collections.emptyMap())
                .forEach((externalAccountId, account) -> {
                    ExternalAccount updatedAccount = groupedAccounts.getReceivedAccountsMovingOutSamePage()
                            .getOrDefault(folderId, Collections.emptyMap()).get(externalAccountId);
                    boolean isAccountToDelete = groupedAccounts.getCurrentAccountsToDeleteMovingOutSamePage()
                            .getOrDefault(folderId, Collections.emptyMap()).containsKey(externalAccountId);
                    updatedAccount.getProvisions().stream()
                            .map(updatedProvision -> updatedProvision.getResource().getResource())
                            .filter(resource -> resource.getDefaultQuota().isPresent())
                            .forEach(resource ->
                                    defaultQuotas.put(resource.getId(), -resource.getDefaultQuota().get()));
                    if (checkInProgressOperations(externalAccountId, updatedAccount,
                            accountOperationsSupported, provisionOperationsSupported, operationsInProgress,
                            operationsById, folderId, false, account)) {
                        skippedMoveAccountsExternalIds.add(externalAccountId);
                        updateBalance(account, quotasAndOperations, updatedBalances, defaultQuotas, folderId);
                        updateAllocations(provisionsToUpsert, updatedAccount,
                                account, quotasAndOperations, provider);
                        return;
                    }
                    if (checkStaleVersions(account, updatedAccount, quotasAndOperations,
                            accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            provisionsVersionedSeparately, accountNameSupported, accountOperationsSupported,
                            accountKeySupported, provisionOperationsSupported)) {
                        skippedMoveAccountsExternalIds.add(externalAccountId);
                        updateBalance(account, quotasAndOperations, updatedBalances, defaultQuotas, folderId);
                        updateAllocations(provisionsToUpsert, updatedAccount,
                                account, quotasAndOperations, provider);
                        return;
                    }
                    if (checkCoolDownPeriod(account, updatedAccount, quotasAndOperations,
                            accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            provisionsVersionedSeparately, accountOperationsSupported, provisionOperationsSupported,
                            isAccountToDelete, softDeleteSupported, accountNameSupported, accountKeySupported, now,
                            provider, syncStatus)) {
                        skippedMoveAccountsExternalIds.add(externalAccountId);
                        updateBalance(account, quotasAndOperations, updatedBalances, defaultQuotas, folderId);
                        updateAllocations(provisionsToUpsert, updatedAccount,
                                account, quotasAndOperations, provider);
                        return;
                    }
                    if (checkOperationsOrder(account, updatedAccount, quotasAndOperations,
                            accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            provisionsVersionedSeparately, accountOperationsSupported,
                            provisionOperationsSupported, accountNameSupported, accountKeySupported)) {
                        skippedMoveAccountsExternalIds.add(externalAccountId);
                        updateBalance(account, quotasAndOperations, updatedBalances, defaultQuotas, folderId);
                        updateAllocations(provisionsToUpsert, updatedAccount,
                                account, quotasAndOperations, provider);
                        return;
                    }
                    if (skippedMoveAccountsExternalIds.contains(externalAccountId)) {
                        return;
                    }
                    FolderHistoryGroupKey moveGroupKey = doAccountMoveSourceUpdate(provider,
                            accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                            accountOperationsSupported, softDeleteSupported, accountNameSupported,
                            accountKeySupported, oldAccounts, newAccounts, operationsById, authors,
                            operationAuthors, newestLastUpdates, account, updatedAccount, isAccountToDelete,
                            folderId, accountMoveFoldersByAccountId, quotasAndOperations);
                    doAccountMoveSourceDeletedProvisionsUpdate(provider, oldProvisions, newProvisions,
                            operationAuthors, quotasAndOperations, account, updatedAccount);
                    doAccountMoveSourceModifiedProvisionsUpdate(provider, provisionsVersionedSeparately,
                            provisionOperationsSupported, oldProvisions, newProvisions, operationsById,
                            authors, operationAuthors, newestLastUpdates, quotasAndOperations, account,
                            updatedAccount, moveGroupKey, folderId, syncStatus, errors);
                });
    }

    @SuppressWarnings({"ParameterNumber", "MethodLength"})
    private void doAccountStayModifiedProvisionsUpdate(ProviderModel provider,
            boolean provisionsVersionedSeparately, boolean provisionOperationsSupported,
            List<AccountsQuotasModel> provisionsToUpsert, Instant now, String folderId,
            Map<String, Long> updatedBalances,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
            GroupedAuthors authors, Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates, QuotasAndOperations quotasAndOperations,
            AccountModel account, ExternalAccount updatedAccount, ProvidersSyncStatusModel syncStatus,
            List<ProvidersSyncErrorsModel> errors
    ) {
        updatedAccount.getProvisions().forEach(updatedProvision -> {
            String resourceId = updatedProvision.getResource().getResource().getId();
            AccountsQuotasModel currentProvision = quotasAndOperations.getProvisionsByAccountResource()
                    .getOrDefault(account.getId(), Collections.emptyMap()).get(resourceId);
            Optional<Long> providedAmountO = Units.convertFromApi(updatedProvision.getProvided(),
                    updatedProvision.getResource().getResource(),
                    updatedProvision.getResource().getUnitsEnsemble(), updatedProvision.getProvidedUnit());
            if (providedAmountO.isEmpty()) {
                errors.add(accountsSyncStoreService.newError(null, syncStatus, Map.of(
                        "accountId", updatedAccount.getAccountId(),
                        "resourceKey", updatedProvision.getResource().getResource().getKey()
                ), "Invalid updated provision (empty result)"));
                LOG.error("Invalid provision {} {} received for resource {} in account {} of provider {}",
                        updatedProvision.getProvided(), updatedProvision.getProvidedUnit().getKey(),
                        updatedProvision.getResource().getResource().getKey(),
                        updatedAccount.getAccountId(), provider.getKey());
            }
            long providedAmount = providedAmountO.orElse(0L);
            Optional<Long> allocatedAmountO = Units.convertFromApi(updatedProvision.getAllocated(),
                    updatedProvision.getResource().getResource(),
                    updatedProvision.getResource().getUnitsEnsemble(),
                    updatedProvision.getAllocatedUnit());
            long allocatedAmount = allocatedAmountO.orElse(0L);
            if (allocatedAmountO.isEmpty()) {
                errors.add(accountsSyncStoreService.newError(null, syncStatus, Map.of(
                        "accountId", updatedAccount.getAccountId(),
                        "resourceKey", updatedProvision.getResource().getResource().getKey()
                ), "Invalid updated allocation (empty result)"));
                LOG.error("Invalid allocation received for resource {} in account {} of provider {}",
                        updatedProvision.getResource().getResource().getKey(),
                        updatedAccount.getAccountId(), provider.getKey());
            }
            if (providedAmount != 0L) {
                Optional<Long> updatedBalanceO = Units.subtract(updatedBalances
                        .getOrDefault(resourceId, 0L), providedAmount);
                if (updatedBalanceO.isPresent()) {
                    updatedBalances.put(resourceId, updatedBalanceO.get());
                } else {
                    errors.add(accountsSyncStoreService.newError(null, syncStatus, Map.of(
                            "accountId", updatedAccount.getAccountId(),
                            "resourceKey", updatedProvision.getResource().getResource().getKey()
                    ), "Invalid updated balance (empty result)"));
                    LOG.error("Invalid balance for resource {} in folder {}",
                            updatedProvision.getResource().getResource().getKey(), folderId);
                }
            }
            long currentProvidedAmount = currentProvision != null
                    ? (currentProvision.getProvidedQuota() != null
                        ? currentProvision.getProvidedQuota() : 0L)
                    : 0L;
            long currentAllocatedAmount = currentProvision != null
                    ? (currentProvision.getAllocatedQuota() != null
                        ? currentProvision.getAllocatedQuota() : 0L)
                    : 0L;
            Optional<Long> currentLastReceivedProvisionVersion = currentProvision != null
                    ? currentProvision.getLastReceivedProvisionVersion() : Optional.empty();
            Optional<String> currentLatestSuccessfulProvisionOperationId = currentProvision != null
                    ? currentProvision.getLatestSuccessfulProvisionOperationId() : Optional.empty();
            Optional<String> provisionUpdateOperationId = AccountsSyncUtils.getProvisionUpdateOperationId(
                    updatedProvision, operationsById, provisionOperationsSupported, folderId, account.getId());
            FolderHistoryGroupKey provisionUpdateHistoryGroupKey
                    = new FolderHistoryGroupKey(provisionUpdateOperationId.orElse(null), null);
            Set<FolderOperationAuthor> provisionUpdateAuthors = AccountsSyncUtils.collectAuthors(provider,
                    updatedProvision::getLastUpdate, authors);
            operationAuthors.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashSet<>())
                    .addAll(provisionUpdateAuthors);
            AccountsSyncUtils.refreshNewestLastUpdates(provisionUpdateHistoryGroupKey, updatedProvision::getLastUpdate,
                    newestLastUpdates);
            if (providedAmount != 0L || allocatedAmount != 0L) {
                if (currentProvidedAmount != providedAmount || currentAllocatedAmount != allocatedAmount
                        || (provisionsVersionedSeparately && !Objects.equal(currentLastReceivedProvisionVersion,
                        updatedProvision.getQuotaVersion()))
                        || (provisionOperationsSupported && !Objects.equal(currentLatestSuccessfulProvisionOperationId,
                        provisionUpdateOperationId))) {
                    AccountsQuotasModel modifiedProvision;
                    if (currentProvision != null) {
                        modifiedProvision = new AccountsQuotasModel.Builder(currentProvision)
                                .setProvidedQuota(providedAmount)
                                .setAllocatedQuota(allocatedAmount)
                                .setLastProvisionUpdate(now)
                                .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                                        ? updatedProvision.getQuotaVersion().orElse(null) : null)
                                .setLatestSuccessfulProvisionOperationId(AccountsSyncUtils.getValidOperationId(
                                        updatedProvision::getLastUpdate, provisionOperationsSupported,
                                        operationsById).orElse(currentProvision
                                        .getLatestSuccessfulProvisionOperationId().orElse(null)))
                                .build();
                    } else {
                        modifiedProvision = new AccountsQuotasModel.Builder()
                                .setAccountId(account.getId())
                                .setFolderId(folderId)
                                .setProviderId(provider.getId())
                                .setResourceId(resourceId)
                                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                                .setProvidedQuota(providedAmount)
                                .setAllocatedQuota(allocatedAmount)
                                .setLastProvisionUpdate(now)
                                .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                                        ? updatedProvision.getQuotaVersion().orElse(null) : null)
                                .setLatestSuccessfulProvisionOperationId(AccountsSyncUtils.getValidOperationId(
                                        updatedProvision::getLastUpdate, provisionOperationsSupported,
                                        operationsById).orElse(null))
                                .build();
                    }
                    provisionsToUpsert.add(modifiedProvision);
                }
                if (currentProvidedAmount != providedAmount
                        || (provisionsVersionedSeparately && !Objects.equal(currentLastReceivedProvisionVersion,
                        updatedProvision.getQuotaVersion()))) {
                    ProvisionHistoryModel oldProvisionHistory = new ProvisionHistoryModel(
                            currentProvidedAmount, currentLastReceivedProvisionVersion.orElse(null));
                    ProvisionHistoryModel newProvisionHistory = new ProvisionHistoryModel(providedAmount,
                            provisionsVersionedSeparately
                                    ? updatedProvision.getQuotaVersion().orElse(null) : null);
                    oldProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                            .computeIfAbsent(account.getId(), k -> new HashMap<>())
                            .put(resourceId, oldProvisionHistory);
                    newProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                            .computeIfAbsent(account.getId(), k -> new HashMap<>())
                            .put(resourceId, newProvisionHistory);
                    AccountsSyncUtils.handleOperation(provisionUpdateOperationId.orElse(null), operationsToUpsert,
                            operationsById, now, account.getId(), folderId);
                }
            } else if (currentProvision != null) {
                provisionsToUpsert.add(currentProvision.copyBuilder()
                        .setProvidedQuota(0L)
                        .setAllocatedQuota(0L)
                        .setLastProvisionUpdate(now)
                        .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                                ? updatedProvision.getQuotaVersion().orElse(null) : null)
                        .setLatestSuccessfulProvisionOperationId(AccountsSyncUtils.getValidOperationId(
                                updatedProvision::getLastUpdate, provisionOperationsSupported,
                                operationsById).orElse(currentProvision
                                .getLatestSuccessfulProvisionOperationId().orElse(null)))
                        .build());
                if (currentProvidedAmount != providedAmount
                        || (provisionsVersionedSeparately && !Objects.equal(currentLastReceivedProvisionVersion,
                        updatedProvision.getQuotaVersion()))) {
                    ProvisionHistoryModel oldProvisionHistory = new ProvisionHistoryModel(
                            currentProvidedAmount, currentLastReceivedProvisionVersion.orElse(null));
                    ProvisionHistoryModel newProvisionHistory = new ProvisionHistoryModel(providedAmount,
                            provisionsVersionedSeparately
                                    ? updatedProvision.getQuotaVersion().orElse(null) : null);
                    oldProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                            .computeIfAbsent(account.getId(), k -> new HashMap<>())
                            .put(resourceId, oldProvisionHistory);
                    newProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                            .computeIfAbsent(account.getId(), k -> new HashMap<>())
                            .put(resourceId, newProvisionHistory);
                    AccountsSyncUtils.handleOperation(provisionUpdateOperationId.orElse(null), operationsToUpsert,
                            operationsById, now, account.getId(), folderId);
                }
            }
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void doAccountMoveDestinationModifiedProvisionsUpdate(ProviderModel provider,
            boolean provisionsVersionedSeparately, boolean provisionOperationsSupported,
            List<AccountsQuotasModel> provisionsToUpsert, Instant now, String folderId,
            Map<String, Long> updatedBalances, Map<String, Long> defaultQuotas,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
            GroupedAuthors authors, Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates, QuotasAndOperations quotasAndOperations,
            AccountModel account, ExternalAccount updatedAccount, FolderHistoryGroupKey moveGroupKey,
            ProvidersSyncStatusModel syncStatus, List<ProvidersSyncErrorsModel> errors
    ) {
        updatedAccount.getProvisions().forEach(updatedProvision -> {
            ResourceModel resource = updatedProvision.getResource().getResource();
            String resourceId = resource.getId();
            AccountsQuotasModel currentProvision = quotasAndOperations.getProvisionsByAccountResource()
                    .getOrDefault(account.getId(), Collections.emptyMap()).get(resourceId);
            Optional<Long> providedAmountO = Units.convertFromApi(updatedProvision.getProvided(),
                    resource,
                    updatedProvision.getResource().getUnitsEnsemble(), updatedProvision.getProvidedUnit());
            if (providedAmountO.isEmpty()) {
                errors.add(accountsSyncStoreService.newError(null, syncStatus, Map.of(
                        "accountId", updatedAccount.getAccountId(),
                        "resourceKey", resource.getKey()
                ), "Invalid moved provision (empty result)"));
                LOG.error("Invalid provision {} {} received for resource {} in account {} of provider {}",
                        updatedProvision.getProvided(), updatedProvision.getProvidedUnit().getKey(),
                        resource.getKey(),
                        updatedAccount.getAccountId(), provider.getKey());
            }
            long providedAmount = providedAmountO.orElse(0L);
            Optional<Long> allocatedAmountO = Units.convertFromApi(updatedProvision.getAllocated(),
                    resource,
                    updatedProvision.getResource().getUnitsEnsemble(),
                    updatedProvision.getAllocatedUnit());
            long allocatedAmount = allocatedAmountO.orElse(0L);
            if (allocatedAmountO.isEmpty()) {
                errors.add(accountsSyncStoreService.newError(null, syncStatus, Map.of(
                        "accountId", updatedAccount.getAccountId(),
                        "resourceKey", resource.getKey()
                ), "Invalid moved allocation (empty result)"));
                LOG.error("Invalid allocation received for resource {} in account {} of provider {}",
                        resource.getKey(),
                        updatedAccount.getAccountId(), provider.getKey());
            }
            updateBalanceByProvisionAndDefault(updatedBalances, defaultQuotas, resource, providedAmount,
                        errors, syncStatus, updatedAccount.getAccountId(), folderId);
            long currentProvidedAmount = currentProvision != null
                    ? (currentProvision.getProvidedQuota() != null
                            ? currentProvision.getProvidedQuota() : 0L)
                    : 0L;
            long currentAllocatedAmount = currentProvision != null
                    ? (currentProvision.getAllocatedQuota() != null
                            ? currentProvision.getAllocatedQuota() : 0L)
                    : 0L;
            Optional<Long> currentLastReceivedProvisionVersion = currentProvision != null
                    ? currentProvision.getLastReceivedProvisionVersion() : Optional.empty();
            Optional<String> currentLatestSuccessfulProvisionOperationId = currentProvision != null
                    ? currentProvision.getLatestSuccessfulProvisionOperationId() : Optional.empty();
            Optional<String> provisionUpdateOperationId = AccountsSyncUtils.getProvisionUpdateOperationId(
                    updatedProvision, operationsById, provisionOperationsSupported, folderId, account.getId());
            FolderHistoryGroupKey provisionUpdateHistoryGroupKey
                    = new FolderHistoryGroupKey(provisionUpdateOperationId.orElse(null), null);
            Set<FolderOperationAuthor> provisionUpdateAuthors = AccountsSyncUtils.collectAuthors(provider,
                    updatedProvision::getLastUpdate, authors);
            operationAuthors.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashSet<>())
                    .addAll(provisionUpdateAuthors);
            AccountsSyncUtils.refreshNewestLastUpdates(provisionUpdateHistoryGroupKey, updatedProvision::getLastUpdate,
                    newestLastUpdates);
            if (providedAmount != 0L || allocatedAmount != 0L) {
                if (currentProvidedAmount != providedAmount || currentAllocatedAmount != allocatedAmount
                        || (provisionsVersionedSeparately && !Objects.equal(currentLastReceivedProvisionVersion,
                        updatedProvision.getQuotaVersion()))
                        || (provisionOperationsSupported && !Objects.equal(currentLatestSuccessfulProvisionOperationId,
                        provisionUpdateOperationId))
                        || !Objects.equal(account.getFolderId(), updatedAccount.getFolderId())) {
                    AccountsQuotasModel modifiedProvision;
                    if (currentProvision != null) {
                        modifiedProvision = new AccountsQuotasModel.Builder(currentProvision)
                                .setFolderId(updatedAccount.getFolderId())
                                .setProvidedQuota(providedAmount)
                                .setAllocatedQuota(allocatedAmount)
                                .setLastProvisionUpdate(now)
                                .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                                        ? updatedProvision.getQuotaVersion().orElse(null) : null)
                                .setLatestSuccessfulProvisionOperationId(AccountsSyncUtils.getValidOperationId(
                                        updatedProvision::getLastUpdate, provisionOperationsSupported,
                                        operationsById).orElse(currentProvision
                                        .getLatestSuccessfulProvisionOperationId().orElse(null)))
                                .build();
                    } else {
                        modifiedProvision = new AccountsQuotasModel.Builder()
                                .setAccountId(account.getId())
                                .setFolderId(updatedAccount.getFolderId())
                                .setProviderId(provider.getId())
                                .setResourceId(resourceId)
                                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                                .setProvidedQuota(providedAmount)
                                .setAllocatedQuota(allocatedAmount)
                                .setLastProvisionUpdate(now)
                                .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                                        ? updatedProvision.getQuotaVersion().orElse(null) : null)
                                .setLatestSuccessfulProvisionOperationId(AccountsSyncUtils.getValidOperationId(
                                        updatedProvision::getLastUpdate, provisionOperationsSupported,
                                        operationsById).orElse(null))
                                .build();
                    }
                    provisionsToUpsert.add(modifiedProvision);
                }
                if (currentProvidedAmount != providedAmount
                        || (provisionsVersionedSeparately && !Objects.equal(currentLastReceivedProvisionVersion,
                        updatedProvision.getQuotaVersion()))) {
                    AccountsSyncUtils.handleOperation(provisionUpdateOperationId.orElse(null), operationsToUpsert,
                            operationsById, now, account.getId(), folderId);
                }
            } else if (currentProvision != null) {
                provisionsToUpsert.add(currentProvision.copyBuilder()
                        .setProvidedQuota(0L)
                        .setAllocatedQuota(0L)
                        .setLastProvisionUpdate(now)
                        .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                                ? updatedProvision.getQuotaVersion().orElse(null) : null)
                        .setLatestSuccessfulProvisionOperationId(AccountsSyncUtils.getValidOperationId(
                                updatedProvision::getLastUpdate, provisionOperationsSupported,
                                operationsById).orElse(currentProvision
                                        .getLatestSuccessfulProvisionOperationId().orElse(null)))
                        .build());
                if (currentProvidedAmount != providedAmount
                        || (provisionsVersionedSeparately && !Objects.equal(currentLastReceivedProvisionVersion,
                        updatedProvision.getQuotaVersion()))) {
                    AccountsSyncUtils.handleOperation(provisionUpdateOperationId.orElse(null), operationsToUpsert,
                            operationsById, now, account.getId(), folderId);
                }
            }
            if (providedAmount != 0L) {
                ProvisionHistoryModel newMovedProvisionHistory = new ProvisionHistoryModel(providedAmount,
                        provisionsVersionedSeparately
                                ? updatedProvision.getQuotaVersion().orElse(null) : null);
                newProvisions.computeIfAbsent(moveGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, newMovedProvisionHistory);
            }
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void updateBalanceByProvisionAndDefault(Map<String, Long> updatedBalances,
            Map<String, Long> defaultQuotas, ResourceModel resource, long providedAmount,
            List<ProvidersSyncErrorsModel> errors, ProvidersSyncStatusModel syncStatus, String accountId,
            String folderId
    ) {
        if (providedAmount == 0L) {
            return;
        }
        String resourceId = resource.getId();
        BigInteger balance = BigInteger.valueOf(updatedBalances.getOrDefault(resourceId, 0L));
        if (resource.getDefaultQuota().isPresent()) {
            Long defaultQuota = resource.getDefaultQuota().get();
            defaultQuotas.put(resourceId, defaultQuota);
            balance = balance.add(BigInteger.valueOf(defaultQuota));
        }
        balance = balance.subtract(BigInteger.valueOf(providedAmount));
        OptionalLong updatedBalanceO = Units.longValue(balance);
        if (updatedBalanceO.isPresent()) {
            updatedBalances.put(resourceId, updatedBalanceO.getAsLong());
        } else {
            errors.add(accountsSyncStoreService.newError(null, syncStatus, Map.of(
                    "accountId", accountId,
                    "resourceKey", resource.getKey()
            ), "Invalid moved balance (empty result)"));
            LOG.error("Invalid balance for resource {} in folder {}",
                    resource.getKey(), folderId);
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void doAccountMoveSourceModifiedProvisionsUpdate(ProviderModel provider,
            boolean provisionsVersionedSeparately, boolean provisionOperationsSupported,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<String, AccountsQuotasOperationsModel> operationsById, GroupedAuthors authors,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates,
            QuotasAndOperations quotasAndOperations, AccountModel account,
            ExternalAccount updatedAccount, FolderHistoryGroupKey moveGroupKey, String folderId,
            ProvidersSyncStatusModel syncStatus, List<ProvidersSyncErrorsModel> errors) {
        updatedAccount.getProvisions().forEach(updatedProvision -> {
            String resourceId = updatedProvision.getResource().getResource().getId();
            AccountsQuotasModel currentProvision = quotasAndOperations.getProvisionsByAccountResource()
                    .getOrDefault(account.getId(), Collections.emptyMap()).get(resourceId);
            Optional<Long> providedAmountO = Units.convertFromApi(updatedProvision.getProvided(),
                    updatedProvision.getResource().getResource(),
                    updatedProvision.getResource().getUnitsEnsemble(), updatedProvision.getProvidedUnit());
            if (providedAmountO.isEmpty()) {
                errors.add(accountsSyncStoreService.newError(null, syncStatus, Map.of(
                        "accountId", updatedAccount.getAccountId(),
                        "resourceKey", updatedProvision.getResource().getResource().getKey()
                ), "Invalid moved provision (empty result)"));
                LOG.error("Invalid provision {} {} received for resource {} in account {} of provider {}",
                        updatedProvision.getProvided(), updatedProvision.getProvidedUnit().getKey(),
                        updatedProvision.getResource().getResource().getKey(),
                        updatedAccount.getAccountId(), provider.getKey());
            }
            long providedAmount = providedAmountO.orElse(0L);
            Optional<Long> allocatedAmountO = Units.convertFromApi(updatedProvision.getAllocated(),
                    updatedProvision.getResource().getResource(),
                    updatedProvision.getResource().getUnitsEnsemble(),
                    updatedProvision.getAllocatedUnit());
            long allocatedAmount = allocatedAmountO.orElse(0L);
            if (allocatedAmountO.isEmpty()) {
                errors.add(accountsSyncStoreService.newError(null, syncStatus, Map.of(
                        "accountId", updatedAccount.getAccountId(),
                        "resourceKey", updatedProvision.getResource().getResource().getKey()
                ), "Invalid moved allocation (empty result)"));
                LOG.error("Invalid allocation received for resource {} in account {} of provider {}",
                        updatedProvision.getResource().getResource().getKey(),
                        updatedAccount.getAccountId(), provider.getKey());
            }
            long currentProvidedAmount = currentProvision != null
                    ? (currentProvision.getProvidedQuota() != null
                            ? currentProvision.getProvidedQuota() : 0L)
                    : 0L;
            Optional<Long> currentLastReceivedProvisionVersion = currentProvision != null
                    ? currentProvision.getLastReceivedProvisionVersion() : Optional.empty();
            Optional<String> provisionUpdateOperationId = AccountsSyncUtils.getProvisionUpdateOperationId(
                    updatedProvision, operationsById, provisionOperationsSupported, folderId, account.getId());
            FolderHistoryGroupKey provisionUpdateHistoryGroupKey
                    = new FolderHistoryGroupKey(provisionUpdateOperationId.orElse(null), null);
            Set<FolderOperationAuthor> provisionUpdateAuthors = AccountsSyncUtils.collectAuthors(provider,
                    updatedProvision::getLastUpdate, authors);
            operationAuthors.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashSet<>())
                    .addAll(provisionUpdateAuthors);
            AccountsSyncUtils.refreshNewestLastUpdates(provisionUpdateHistoryGroupKey, updatedProvision::getLastUpdate,
                    newestLastUpdates);
            if (providedAmount != 0L || allocatedAmount != 0L) {
                if (currentProvidedAmount != providedAmount
                        || (provisionsVersionedSeparately && !Objects.equal(currentLastReceivedProvisionVersion,
                        updatedProvision.getQuotaVersion()))) {
                    ProvisionHistoryModel oldProvisionHistory = new ProvisionHistoryModel(
                            currentProvidedAmount, currentLastReceivedProvisionVersion.orElse(null));
                    ProvisionHistoryModel newProvisionHistory = new ProvisionHistoryModel(providedAmount,
                            provisionsVersionedSeparately
                                    ? updatedProvision.getQuotaVersion().orElse(null) : null);
                    oldProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                            .computeIfAbsent(account.getId(), k -> new HashMap<>())
                            .put(resourceId, oldProvisionHistory);
                    newProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                            .computeIfAbsent(account.getId(), k -> new HashMap<>())
                            .put(resourceId, newProvisionHistory);
                }
            } else if (currentProvision != null) {
                if (currentProvidedAmount != providedAmount
                        || (provisionsVersionedSeparately && !Objects.equal(currentLastReceivedProvisionVersion,
                        updatedProvision.getQuotaVersion()))) {
                    ProvisionHistoryModel oldProvisionHistory = new ProvisionHistoryModel(
                            currentProvidedAmount, currentLastReceivedProvisionVersion.orElse(null));
                    ProvisionHistoryModel newProvisionHistory = new ProvisionHistoryModel(providedAmount,
                            provisionsVersionedSeparately
                                    ? updatedProvision.getQuotaVersion().orElse(null) : null);
                    oldProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                            .computeIfAbsent(account.getId(), k -> new HashMap<>())
                            .put(resourceId, oldProvisionHistory);
                    newProvisions.computeIfAbsent(provisionUpdateHistoryGroupKey, k -> new HashMap<>())
                            .computeIfAbsent(account.getId(), k -> new HashMap<>())
                            .put(resourceId, newProvisionHistory);
                }
            }
            if (providedAmount != 0L) {
                ProvisionHistoryModel oldMovedProvisionHistory = new ProvisionHistoryModel(providedAmount,
                        provisionsVersionedSeparately
                                ? updatedProvision.getQuotaVersion().orElse(null) : null);
                oldProvisions.computeIfAbsent(moveGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, oldMovedProvisionHistory);
            }
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void doAccountStayDeletedProvisionsUpdate(ProviderModel provider,
            List<AccountsQuotasModel> provisionsToUpsert,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            QuotasAndOperations quotasAndOperations, AccountModel account, ExternalAccount updatedAccount,
            Instant now) {
        Set<String> currentResources = quotasAndOperations.getProvisionsByAccountResource()
                .getOrDefault(account.getId(), Collections.emptyMap()).keySet();
        Set<String> updatedResources = updatedAccount.getProvisions().stream()
                .map(p -> p.getResource().getResource().getId()).collect(Collectors.toSet());
        Set<String> deletedResources = Sets.difference(currentResources, updatedResources);
        deletedResources.forEach(resourceId -> {
            FolderHistoryGroupKey provisionDeletionHistoryGroupKey = new FolderHistoryGroupKey(null, null);
            Set<FolderOperationAuthor> provisionDeletionAuthors = Set.of(new FolderOperationAuthor(
                    null, null, provider.getId()));
            operationAuthors.computeIfAbsent(provisionDeletionHistoryGroupKey, k -> new HashSet<>())
                    .addAll(provisionDeletionAuthors);
            AccountsQuotasModel provision = quotasAndOperations.getProvisionsByAccountResource()
                    .getOrDefault(account.getId(), Collections.emptyMap()).get(resourceId);
            provisionsToUpsert.add(provision.copyBuilder()
                    .setProvidedQuota(0L)
                    .setAllocatedQuota(0L)
                    .setLastProvisionUpdate(now)
                    .setLastReceivedProvisionVersion(null)
                    .setLatestSuccessfulProvisionOperationId(null)
                    .build());
            if (!Objects.equal(provision.getProvidedQuota(), 0L)
                    || !Objects.equal(provision.getLastReceivedProvisionVersion().orElse(null), null)) {
                ProvisionHistoryModel oldProvisionHistory = new ProvisionHistoryModel(
                        provision.getProvidedQuota(), provision.getLastReceivedProvisionVersion()
                        .orElse(null));
                ProvisionHistoryModel newProvisionHistory = new ProvisionHistoryModel(0, null);
                oldProvisions.computeIfAbsent(provisionDeletionHistoryGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, oldProvisionHistory);
                newProvisions.computeIfAbsent(provisionDeletionHistoryGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, newProvisionHistory);
            }
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void doAccountMoveDestinationDeletedProvisionsUpdate(ProviderModel provider,
            List<AccountsQuotasModel> provisionsToUpsert,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            QuotasAndOperations quotasAndOperations, AccountModel account, ExternalAccount updatedAccount,
            Instant now) {
        Set<String> currentResources = quotasAndOperations.getProvisionsByAccountResource()
                .getOrDefault(account.getId(), Collections.emptyMap()).keySet();
        Set<String> updatedResources = updatedAccount.getProvisions().stream()
                .map(p -> p.getResource().getResource().getId()).collect(Collectors.toSet());
        Set<String> deletedResources = Sets.difference(currentResources, updatedResources);
        deletedResources.forEach(resourceId -> {
            FolderHistoryGroupKey provisionDeletionHistoryGroupKey = new FolderHistoryGroupKey(null, null);
            Set<FolderOperationAuthor> provisionDeletionAuthors = Set.of(new FolderOperationAuthor(
                    null, null, provider.getId()));
            operationAuthors.computeIfAbsent(provisionDeletionHistoryGroupKey, k -> new HashSet<>())
                    .addAll(provisionDeletionAuthors);
            AccountsQuotasModel provision = quotasAndOperations.getProvisionsByAccountResource()
                    .getOrDefault(account.getId(), Collections.emptyMap()).get(resourceId);
            provisionsToUpsert.add(provision.copyBuilder()
                    .setProvidedQuota(0L)
                    .setAllocatedQuota(0L)
                    .setLastProvisionUpdate(now)
                    .setLastReceivedProvisionVersion(null)
                    .setLatestSuccessfulProvisionOperationId(null)
                    .build());
            if (!Objects.equal(provision.getProvidedQuota(), 0L)
                    || !Objects.equal(provision.getLastReceivedProvisionVersion().orElse(null), null)) {
                ProvisionHistoryModel oldProvisionHistory = new ProvisionHistoryModel(
                        provision.getProvidedQuota(), provision.getLastReceivedProvisionVersion()
                        .orElse(null));
                ProvisionHistoryModel newProvisionHistory = new ProvisionHistoryModel(0, null);
                oldProvisions.computeIfAbsent(provisionDeletionHistoryGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, oldProvisionHistory);
                newProvisions.computeIfAbsent(provisionDeletionHistoryGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, newProvisionHistory);
            }
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void doAccountMoveSourceDeletedProvisionsUpdate(ProviderModel provider,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            QuotasAndOperations quotasAndOperations, AccountModel account, ExternalAccount updatedAccount) {
        Set<String> currentResources = quotasAndOperations.getProvisionsByAccountResource()
                .getOrDefault(account.getId(), Collections.emptyMap()).keySet();
        Set<String> updatedResources = updatedAccount.getProvisions().stream()
                .map(p -> p.getResource().getResource().getId()).collect(Collectors.toSet());
        Set<String> deletedResources = Sets.difference(currentResources, updatedResources);
        deletedResources.forEach(resourceId -> {
            FolderHistoryGroupKey provisionDeletionHistoryGroupKey = new FolderHistoryGroupKey(null, null);
            Set<FolderOperationAuthor> provisionDeletionAuthors = Set.of(new FolderOperationAuthor(
                    null, null, provider.getId()));
            operationAuthors.computeIfAbsent(provisionDeletionHistoryGroupKey, k -> new HashSet<>())
                    .addAll(provisionDeletionAuthors);
            AccountsQuotasModel provision = quotasAndOperations.getProvisionsByAccountResource()
                    .getOrDefault(account.getId(), Collections.emptyMap()).get(resourceId);
            if (!Objects.equal(provision.getProvidedQuota(), 0L)
                    || !Objects.equal(provision.getLastReceivedProvisionVersion().orElse(null), null)) {
                ProvisionHistoryModel oldProvisionHistory = new ProvisionHistoryModel(
                        provision.getProvidedQuota(), provision.getLastReceivedProvisionVersion()
                        .orElse(null));
                ProvisionHistoryModel newProvisionHistory = new ProvisionHistoryModel(0, null);
                oldProvisions.computeIfAbsent(provisionDeletionHistoryGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, oldProvisionHistory);
                newProvisions.computeIfAbsent(provisionDeletionHistoryGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, newProvisionHistory);
            }
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void doAccountStayUpdate(ProviderModel provider, boolean accountAndProvisionsVersionedTogether,
            boolean accountVersionedSeparately, boolean accountOperationsSupported, boolean softDeleteSupported,
            boolean accountNameSupported, boolean accountKeySupported, List<AccountModel> accountsToUpsert,
            Instant now, Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> oldAccounts,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
            GroupedAuthors authors, Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates, AccountModel account,
            ExternalAccount updatedAccount, boolean isAccountToDelete,
            String folderId, QuotasAndOperations quotasAndOperations, List<AccountModel> accountsToRemoveReserve) {
        Optional<String> accountUpdateOperationId = AccountsSyncUtils.getAccountOperationId(updatedAccount,
                operationsById, accountOperationsSupported, folderId, account.getId());
        FolderHistoryGroupKey accountUpdateHistoryGroupKey
                = new FolderHistoryGroupKey(accountUpdateOperationId.orElse(null), null);
        Set<FolderOperationAuthor> accountUpdateAuthors = AccountsSyncUtils.collectAuthors(provider,
                updatedAccount::getLastUpdate, authors);
        operationAuthors.computeIfAbsent(accountUpdateHistoryGroupKey, k -> new HashSet<>())
                .addAll(accountUpdateAuthors);
        AccountsSyncUtils.refreshNewestLastUpdates(accountUpdateHistoryGroupKey, updatedAccount::getLastUpdate,
                newestLastUpdates);
        if (AccountsSyncUtils.isAccountChanged(account, updatedAccount, quotasAndOperations, accountNameSupported,
                accountAndProvisionsVersionedTogether, accountVersionedSeparately, accountOperationsSupported,
                accountKeySupported)) {
            AccountModel.Builder modifiedAccountBuilder = new AccountModel.Builder(account)
                    .setDeleted(softDeleteSupported && isAccountToDelete)
                    .setVersion(account.getVersion() + 1L)
                    .setDisplayName(accountNameSupported
                            ? updatedAccount.getDisplayName().orElse(null) : null)
                    .setLastAccountUpdate(now)
                    .setLastReceivedVersion(accountAndProvisionsVersionedTogether
                            || accountVersionedSeparately
                            ? updatedAccount.getAccountVersion().orElse(null) : null)
                    .setLatestSuccessfulAccountOperationId(AccountsSyncUtils.getValidOperationId(
                            updatedAccount::getLastUpdate, accountOperationsSupported,
                            operationsById).orElse(account.getLatestSuccessfulAccountOperationId()
                            .orElse(null)))
                    .setOuterAccountKeyInProvider(accountKeySupported
                            ? updatedAccount.getKey().orElse(null) : null);
            if (softDeleteSupported && isAccountToDelete && account.getReserveType().isPresent()) {
                modifiedAccountBuilder.setReserveType(null);
            }
            AccountModel modifiedAccount = modifiedAccountBuilder.build();
            accountsToUpsert.add(modifiedAccount);
            if (modifiedAccount.isDeleted() && account.getReserveType().isPresent()) {
                accountsToRemoveReserve.add(modifiedAccount);
            }
            AccountHistoryModel.Builder oldAccountHistoryBuilder = AccountHistoryModel.builder()
                    .version(account.getVersion());
            AccountHistoryModel.Builder newAccountHistoryBuilder = AccountHistoryModel.builder()
                    .version(modifiedAccount.getVersion());
            boolean hasChanges = false;
            if (modifiedAccount.isDeleted() && account.getReserveType().isPresent()) {
                hasChanges = true;
                oldAccountHistoryBuilder.reserveType(account.getReserveType().get());
                newAccountHistoryBuilder.reserveType(null);
            }
            if (!Objects.equal(account.getOuterAccountKeyInProvider(),
                    modifiedAccount.getOuterAccountKeyInProvider())) {
                oldAccountHistoryBuilder.outerAccountKeyInProvider(account
                        .getOuterAccountKeyInProvider().orElse(null));
                newAccountHistoryBuilder.outerAccountKeyInProvider(modifiedAccount
                        .getOuterAccountKeyInProvider().orElse(null));
                hasChanges = true;
            }
            if (!Objects.equal(account.getDisplayName(), modifiedAccount.getDisplayName())) {
                oldAccountHistoryBuilder.displayName(account.getDisplayName().orElse(null));
                newAccountHistoryBuilder.displayName(modifiedAccount.getDisplayName().orElse(null));
                hasChanges = true;
            }
            if (account.isDeleted() != modifiedAccount.isDeleted()) {
                oldAccountHistoryBuilder.deleted(account.isDeleted());
                newAccountHistoryBuilder.deleted(modifiedAccount.isDeleted());
                hasChanges = true;
            }
            if (!Objects.equal(account.getLastReceivedVersion(),
                    modifiedAccount.getLastReceivedVersion())) {
                oldAccountHistoryBuilder.lastReceivedVersion(account
                        .getLastReceivedVersion().orElse(null));
                newAccountHistoryBuilder.lastReceivedVersion(modifiedAccount
                        .getLastReceivedVersion().orElse(null));
                hasChanges = true;
            }
            if (hasChanges) {
                AccountHistoryModel oldAccountHistory = oldAccountHistoryBuilder.build();
                AccountHistoryModel newAccountHistory = newAccountHistoryBuilder.build();
                oldAccounts.computeIfAbsent(accountUpdateHistoryGroupKey, k -> new HashMap<>())
                        .put(account.getId(), oldAccountHistory);
                newAccounts.computeIfAbsent(accountUpdateHistoryGroupKey, k -> new HashMap<>())
                        .put(account.getId(), newAccountHistory);
                AccountsSyncUtils.handleOperation(accountUpdateOperationId.orElse(null), operationsToUpsert,
                        operationsById, now, account.getId(), folderId);
            }
        }
    }

    @SuppressWarnings("ParameterNumber")
    private FolderHistoryGroupKey doAccountMoveDestinationUpdate(ProviderModel provider,
            boolean accountAndProvisionsVersionedTogether, boolean accountVersionedSeparately,
            boolean accountOperationsSupported, boolean softDeleteSupported, boolean accountNameSupported,
            boolean accountKeySupported, List<AccountModel> accountsToUpsert, Instant now,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> oldAccounts,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
            GroupedAuthors authors, Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates,
            AccountModel account, ExternalAccount updatedAccount, boolean isAccountToDelete, String folderId,
            Map<String, AccountMoveFolders> accountMoveFoldersByAccountId,
            QuotasAndOperations quotasAndOperations, List<AccountModel> accountsToRemoveReserve) {
        Optional<String> accountUpdateOperationId = AccountsSyncUtils.getAccountOperationId(updatedAccount,
                operationsById, accountOperationsSupported, folderId, account.getId());
        FolderHistoryGroupKey accountUpdateHistoryGroupKey
                = new FolderHistoryGroupKey(accountUpdateOperationId.orElse(null), account.getId());
        Set<FolderOperationAuthor> accountUpdateAuthors = AccountsSyncUtils.collectAuthors(provider,
                updatedAccount::getLastUpdate, authors);
        operationAuthors.computeIfAbsent(accountUpdateHistoryGroupKey, k -> new HashSet<>())
                .addAll(accountUpdateAuthors);
        AccountsSyncUtils.refreshNewestLastUpdates(accountUpdateHistoryGroupKey, updatedAccount::getLastUpdate,
                newestLastUpdates);
        if (AccountsSyncUtils.isAccountChanged(account, updatedAccount, quotasAndOperations, accountNameSupported,
                accountAndProvisionsVersionedTogether, accountVersionedSeparately, accountOperationsSupported,
                accountKeySupported)) {
            AccountModel.Builder modifiedAccountBuilder = new AccountModel.Builder(account)
                    .setFolderId(updatedAccount.getFolderId())
                    .setDeleted(softDeleteSupported && isAccountToDelete)
                    .setVersion(account.getVersion() + 1L)
                    .setDisplayName(accountNameSupported
                            ? updatedAccount.getDisplayName().orElse(null) : null)
                    .setLastAccountUpdate(now)
                    .setLastReceivedVersion(accountAndProvisionsVersionedTogether
                            || accountVersionedSeparately
                            ? updatedAccount.getAccountVersion().orElse(null) : null)
                    .setLatestSuccessfulAccountOperationId(AccountsSyncUtils.getValidOperationId(
                            updatedAccount::getLastUpdate, accountOperationsSupported,
                            operationsById).orElse(account.getLatestSuccessfulAccountOperationId()
                            .orElse(null)))
                    .setOuterAccountKeyInProvider(accountKeySupported
                            ? updatedAccount.getKey().orElse(null) : null);
            if (softDeleteSupported && isAccountToDelete && account.getReserveType().isPresent()) {
                modifiedAccountBuilder.setReserveType(null);
            }
            AccountModel modifiedAccount = modifiedAccountBuilder.build();
            if (modifiedAccount.isDeleted() && account.getReserveType().isPresent()) {
                accountsToRemoveReserve.add(modifiedAccount);
            }
            accountsToUpsert.add(modifiedAccount);
            AccountHistoryModel.Builder oldAccountHistoryBuilder = AccountHistoryModel.builder()
                    .version(account.getVersion())
                    .providerId(provider.getId())
                    .outerAccountIdInProvider(account.getOuterAccountIdInProvider())
                    .accountsSpacesId(account.getAccountsSpacesId().orElse(null));
            if (modifiedAccount.isDeleted() && account.getReserveType().isPresent()) {
                oldAccountHistoryBuilder.reserveType(account.getReserveType().get());
            }
            AccountHistoryModel.Builder newAccountHistoryBuilder = AccountHistoryModel.builder()
                    .version(modifiedAccount.getVersion())
                    .providerId(provider.getId())
                    .outerAccountIdInProvider(account.getOuterAccountIdInProvider())
                    .accountsSpacesId(account.getAccountsSpacesId().orElse(null));
            if (modifiedAccount.isDeleted() && account.getReserveType().isPresent()) {
                newAccountHistoryBuilder.reserveType(null);
            }
            oldAccountHistoryBuilder.folderId(account.getFolderId());
            newAccountHistoryBuilder.folderId(modifiedAccount.getFolderId());
            oldAccountHistoryBuilder.outerAccountKeyInProvider(account
                    .getOuterAccountKeyInProvider().orElse(null));
            newAccountHistoryBuilder.outerAccountKeyInProvider(modifiedAccount
                    .getOuterAccountKeyInProvider().orElse(null));
            oldAccountHistoryBuilder.displayName(account.getDisplayName().orElse(null));
            newAccountHistoryBuilder.displayName(modifiedAccount.getDisplayName().orElse(null));
            oldAccountHistoryBuilder.deleted(account.isDeleted());
            newAccountHistoryBuilder.deleted(modifiedAccount.isDeleted());
            oldAccountHistoryBuilder.lastReceivedVersion(account
                    .getLastReceivedVersion().orElse(null));
            newAccountHistoryBuilder.lastReceivedVersion(modifiedAccount
                    .getLastReceivedVersion().orElse(null));
            AccountHistoryModel oldAccountHistory = oldAccountHistoryBuilder.build();
            AccountHistoryModel newAccountHistory = newAccountHistoryBuilder.build();
            oldAccounts.computeIfAbsent(accountUpdateHistoryGroupKey, k -> new HashMap<>())
                    .put(account.getId(), oldAccountHistory);
            newAccounts.computeIfAbsent(accountUpdateHistoryGroupKey, k -> new HashMap<>())
                    .put(account.getId(), newAccountHistory);
            AccountsSyncUtils.handleOperation(accountUpdateOperationId.orElse(null), operationsToUpsert,
                    operationsById, now, account.getId(), folderId);
            accountMoveFoldersByAccountId.computeIfAbsent(account.getId(), k -> new AccountMoveFolders())
                    .setDestinationFolderId(folderId);
        }
        return accountUpdateHistoryGroupKey;
    }

    @SuppressWarnings("ParameterNumber")
    private FolderHistoryGroupKey doAccountMoveSourceUpdate(ProviderModel provider,
            boolean accountAndProvisionsVersionedTogether, boolean accountVersionedSeparately,
            boolean accountOperationsSupported, boolean softDeleteSupported, boolean accountNameSupported,
            boolean accountKeySupported, Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> oldAccounts,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            GroupedAuthors authors, Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            Map<FolderHistoryGroupKey, Instant> newestLastUpdates,
            AccountModel account, ExternalAccount updatedAccount,
            boolean isAccountToDelete, String folderId, Map<String, AccountMoveFolders> accountMoveFoldersByAccountId,
            QuotasAndOperations quotasAndOperations) {
        Optional<String> accountUpdateOperationId = AccountsSyncUtils.getAccountOperationId(updatedAccount,
                operationsById, accountOperationsSupported, folderId, account.getId());
        FolderHistoryGroupKey accountUpdateHistoryGroupKey
                = new FolderHistoryGroupKey(accountUpdateOperationId.orElse(null), account.getId());
        Set<FolderOperationAuthor> accountUpdateAuthors = AccountsSyncUtils.collectAuthors(provider,
                updatedAccount::getLastUpdate, authors);
        operationAuthors.computeIfAbsent(accountUpdateHistoryGroupKey, k -> new HashSet<>())
                .addAll(accountUpdateAuthors);
        AccountsSyncUtils.refreshNewestLastUpdates(accountUpdateHistoryGroupKey, updatedAccount::getLastUpdate,
                newestLastUpdates);
        if (AccountsSyncUtils.isAccountChanged(account, updatedAccount, quotasAndOperations, accountNameSupported,
                accountAndProvisionsVersionedTogether, accountVersionedSeparately, accountOperationsSupported,
                accountKeySupported)) {
            Optional<String> modifiedKey = Optional.ofNullable(accountKeySupported
                    ? updatedAccount.getKey().orElse(null) : null);
            Optional<String> modifiedDisplayName = Optional.ofNullable(accountNameSupported
                    ? updatedAccount.getDisplayName().orElse(null) : null);
            boolean modifiedDeleted = softDeleteSupported && isAccountToDelete;
            Optional<Long> modifiedLastReceivedVersion = Optional.ofNullable(accountAndProvisionsVersionedTogether
                    || accountVersionedSeparately ? updatedAccount.getAccountVersion().orElse(null) : null);
            AccountHistoryModel.Builder oldAccountHistoryBuilder = AccountHistoryModel.builder()
                    .version(account.getVersion())
                    .providerId(provider.getId())
                    .outerAccountIdInProvider(account.getOuterAccountIdInProvider())
                    .accountsSpacesId(account.getAccountsSpacesId().orElse(null));
            if (modifiedDeleted && account.getReserveType().isPresent()) {
                oldAccountHistoryBuilder.reserveType(account.getReserveType().get());
            }
            AccountHistoryModel.Builder newAccountHistoryBuilder = AccountHistoryModel.builder()
                    .version(account.getVersion() + 1L)
                    .providerId(provider.getId())
                    .outerAccountIdInProvider(account.getOuterAccountIdInProvider())
                    .accountsSpacesId(account.getAccountsSpacesId().orElse(null));
            if (modifiedDeleted && account.getReserveType().isPresent()) {
                newAccountHistoryBuilder.reserveType(null);
            }
            oldAccountHistoryBuilder.folderId(account.getFolderId());
            newAccountHistoryBuilder.folderId(updatedAccount.getFolderId());
            oldAccountHistoryBuilder.outerAccountKeyInProvider(account
                    .getOuterAccountKeyInProvider().orElse(null));
            newAccountHistoryBuilder.outerAccountKeyInProvider(modifiedKey.orElse(null));
            oldAccountHistoryBuilder.displayName(account.getDisplayName().orElse(null));
            newAccountHistoryBuilder.displayName(modifiedDisplayName.orElse(null));
            oldAccountHistoryBuilder.deleted(account.isDeleted());
            newAccountHistoryBuilder.deleted(modifiedDeleted);
            oldAccountHistoryBuilder.lastReceivedVersion(account.getLastReceivedVersion().orElse(null));
            newAccountHistoryBuilder.lastReceivedVersion(modifiedLastReceivedVersion.orElse(null));
            AccountHistoryModel oldAccountHistory = oldAccountHistoryBuilder.build();
            AccountHistoryModel newAccountHistory = newAccountHistoryBuilder.build();
            oldAccounts.computeIfAbsent(accountUpdateHistoryGroupKey, k -> new HashMap<>())
                    .put(account.getId(), oldAccountHistory);
            newAccounts.computeIfAbsent(accountUpdateHistoryGroupKey, k -> new HashMap<>())
                    .put(account.getId(), newAccountHistory);
            accountMoveFoldersByAccountId.computeIfAbsent(account.getId(), k -> new AccountMoveFolders())
                    .setSourceFolderId(folderId);
        }
        return accountUpdateHistoryGroupKey;
    }

    @SuppressWarnings("ParameterNumber")
    private void doFullAccountDeletion(ProviderModel provider, List<AccountModel> accountsToUpsert,
            List<AccountsQuotasModel> provisionsToUpsert, Instant now,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> oldProvisions,
            Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>> newProvisions,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> oldAccounts,
            Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>> newAccounts,
            Map<FolderHistoryGroupKey, Set<FolderOperationAuthor>> operationAuthors,
            QuotasAndOperations quotasAndOperations, AccountModel account,
            List<AccountModel> accountsToRemoveReserve) {
        FolderHistoryGroupKey accountDeletionHistoryGroupKey = new FolderHistoryGroupKey(null, null);
        Set<FolderOperationAuthor> accountDeletionAuthors = Set.of(new FolderOperationAuthor(null,
                null, provider.getId()));
        operationAuthors.computeIfAbsent(accountDeletionHistoryGroupKey, k -> new HashSet<>())
                .addAll(accountDeletionAuthors);
        AccountModel.Builder deletedAccountBuilder = new AccountModel.Builder(account)
                .setDeleted(true)
                .setVersion(account.getVersion() + 1L)
                .setLastAccountUpdate(now);
        if (account.getReserveType().isPresent()) {
            deletedAccountBuilder.setReserveType(null);
        }
        AccountModel deletedAccount = deletedAccountBuilder.build();
        if (account.getReserveType().isPresent()) {
            accountsToRemoveReserve.add(deletedAccount);
        }
        accountsToUpsert.add(deletedAccount);
        AccountHistoryModel.Builder oldAccountHistoryBuilder = AccountHistoryModel.builder()
                .version(account.getVersion())
                .deleted(account.isDeleted());
        if (account.getReserveType().isPresent()) {
            oldAccountHistoryBuilder.reserveType(account.getReserveType().get());
        }
        AccountHistoryModel oldAccountHistory = oldAccountHistoryBuilder.build();
        oldAccounts.computeIfAbsent(accountDeletionHistoryGroupKey, k -> new HashMap<>())
                .put(deletedAccount.getId(), oldAccountHistory);
        AccountHistoryModel.Builder newAccountHistoryBuilder = AccountHistoryModel.builder()
                .version(deletedAccount.getVersion())
                .deleted(deletedAccount.isDeleted());
        if (account.getReserveType().isPresent()) {
            newAccountHistoryBuilder.reserveType(null);
        }
        AccountHistoryModel newAccountHistory = newAccountHistoryBuilder.build();
        newAccounts.computeIfAbsent(accountDeletionHistoryGroupKey, k -> new HashMap<>())
                .put(deletedAccount.getId(), newAccountHistory);
        Map<String, AccountsQuotasModel> provisions = quotasAndOperations
                .getProvisionsByAccountResource().getOrDefault(account.getId(),
                        Collections.emptyMap());
        provisions.forEach((resourceId, provision) -> {
            FolderHistoryGroupKey provisionZeroHistoryGroupKey = new FolderHistoryGroupKey(null, null);
            Set<FolderOperationAuthor> provisionZeroAuthors = Set.of(new FolderOperationAuthor(null,
                    null, provider.getId()));
            operationAuthors.computeIfAbsent(provisionZeroHistoryGroupKey, k -> new HashSet<>())
                    .addAll(provisionZeroAuthors);
            provisionsToUpsert.add(provision.copyBuilder()
                    .setProvidedQuota(0L)
                    .setAllocatedQuota(0L)
                    .setLastProvisionUpdate(now)
                    .setLastReceivedProvisionVersion(null)
                    .setLatestSuccessfulProvisionOperationId(null)
                    .build());
            if (!Objects.equal(provision.getProvidedQuota(), 0L)
                    || !Objects.equal(provision.getLastReceivedProvisionVersion().orElse(null), null)) {
                ProvisionHistoryModel oldProvisionHistory = new ProvisionHistoryModel(
                        provision.getProvidedQuota(), provision.getLastReceivedProvisionVersion()
                        .orElse(null));
                ProvisionHistoryModel newProvisionHistory = new ProvisionHistoryModel(0L, null);
                oldProvisions.computeIfAbsent(provisionZeroHistoryGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, oldProvisionHistory);
                newProvisions.computeIfAbsent(provisionZeroHistoryGroupKey, k -> new HashMap<>())
                        .computeIfAbsent(account.getId(), k -> new HashMap<>())
                        .put(resourceId, newProvisionHistory);
            }
        });
    }
}
