package ru.yandex.intranet.d.services.sync;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao;
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao;
import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao;
import ru.yandex.intranet.d.dao.resources.types.ResourceTypesDao;
import ru.yandex.intranet.d.dao.sync.ProvidersSyncErrorsDao;
import ru.yandex.intranet.d.dao.sync.ProvidersSyncStatusDao;
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao;
import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel;
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.sync.Errors;
import ru.yandex.intranet.d.model.sync.ProvidersSyncErrorsModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;
import ru.yandex.intranet.d.model.sync.SyncStats;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyRequestDto;
import ru.yandex.intranet.d.services.operations.OperationsObservabilityService;
import ru.yandex.intranet.d.services.sync.model.ExternalAccount;
import ru.yandex.intranet.d.services.sync.model.ExternalLastUpdate;
import ru.yandex.intranet.d.services.sync.model.GroupedAuthors;
import ru.yandex.intranet.d.services.sync.model.ProcessFoldersPageResult;
import ru.yandex.intranet.d.services.sync.model.QuotasAndOperations;
import ru.yandex.intranet.d.services.sync.model.SyncResource;
import ru.yandex.intranet.d.services.sync.model.SyncSegment;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.result.ErrorCollection;

import static ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel.SyncStatuses.DONE_ERROR;
import static ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel.SyncStatuses.DONE_OK;

/**
 * Service to sync providers accounts and quotas, storage part.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class AccountsSyncStoreService {

    private static final Logger LOG = LoggerFactory.getLogger(AccountsSyncStoreService.class);

    private final AccountsQuotasOperationsDao accountsQuotasOperationsDao;
    private final OperationsInProgressDao operationsInProgressDao;
    private final AccountsSpacesDao accountsSpacesDao;
    private final UsersDao usersDao;
    private final FolderDao folderDao;
    private final AccountsQuotasDao accountsQuotasDao;
    private final QuotasDao quotasDao;
    private final AccountsDao accountsDao;
    private final ResourcesDao resourcesDao;
    private final ResourceTypesDao resourceTypesDao;
    private final ResourceSegmentationsDao resourceSegmentationsDao;
    private final ResourceSegmentsDao resourceSegmentsDao;
    private final UnitsEnsemblesDao unitsEnsemblesDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final YdbTableClient tableClient;
    private final ProvidersSyncStatusDao providersSyncStatusDao;
    private final ProvidersSyncErrorsDao providersSyncErrorsDao;
    private final OperationsObservabilityService operationsObservabilityService;

    @SuppressWarnings("ParameterNumber")
    public AccountsSyncStoreService(AccountsQuotasOperationsDao accountsQuotasOperationsDao,
                                    OperationsInProgressDao operationsInProgressDao,
                                    AccountsSpacesDao accountsSpacesDao,
                                    UsersDao usersDao,
                                    FolderDao folderDao,
                                    AccountsQuotasDao accountsQuotasDao,
                                    QuotasDao quotasDao,
                                    AccountsDao accountsDao,
                                    ResourcesDao resourcesDao,
                                    ResourceTypesDao resourceTypesDao,
                                    ResourceSegmentationsDao resourceSegmentationsDao,
                                    ResourceSegmentsDao resourceSegmentsDao,
                                    UnitsEnsemblesDao unitsEnsemblesDao,
                                    FolderOperationLogDao folderOperationLogDao,
                                    YdbTableClient tableClient,
                                    ProvidersSyncStatusDao providersSyncStatusDao,
                                    ProvidersSyncErrorsDao providersSyncErrorsDao,
                                    OperationsObservabilityService operationsObservabilityService) {
        this.accountsQuotasOperationsDao = accountsQuotasOperationsDao;
        this.operationsInProgressDao = operationsInProgressDao;
        this.accountsSpacesDao = accountsSpacesDao;
        this.usersDao = usersDao;
        this.folderDao = folderDao;
        this.accountsQuotasDao = accountsQuotasDao;
        this.quotasDao = quotasDao;
        this.accountsDao = accountsDao;
        this.resourcesDao = resourcesDao;
        this.resourceTypesDao = resourceTypesDao;
        this.resourceSegmentationsDao = resourceSegmentationsDao;
        this.resourceSegmentsDao = resourceSegmentsDao;
        this.unitsEnsemblesDao = unitsEnsemblesDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.tableClient = tableClient;
        this.providersSyncStatusDao = providersSyncStatusDao;
        this.providersSyncErrorsDao = providersSyncErrorsDao;
        this.operationsObservabilityService = operationsObservabilityService;
    }

    public Mono<Tuple2<List<FolderOperationLogModel>, List<AccountsQuotasOperationsModel>>> prepareOrders(
            List<FolderOperationLogModel.Builder> folderLogsToUpsert,
            Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
            Map<String, List<String>> sortedFolderLogIdsByFolder) {
        Map<String, Set<FolderOperationLogModel.Builder>> folderLogsByFolder = folderLogsToUpsert.stream()
                .collect(Collectors.groupingBy(builder -> builder.getFolderId().get(), Collectors.toSet()));
        List<String> folderIds = new ArrayList<>(folderLogsByFolder.keySet());
        return AsyncMetrics.metric(tableClient.usingSessionMonoRetryable(session ->
                Flux.fromIterable(Lists.partition(folderIds, 500)).concatMap(v ->
                        session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                                folderDao.getByIds(txSession, v, Tenants.DEFAULT_TENANT_ID).flatMap(folders -> {
                                    List<FolderModel> updatedFolders = new ArrayList<>();
                                    Map<String, Long> currentNextOpLogOrdersByFolder = new HashMap<>();
                                    folders.forEach(folder -> {
                                        currentNextOpLogOrdersByFolder.put(folder.getId(), folder.getNextOpLogOrder());
                                        long newOpLogCount = folderLogsByFolder.getOrDefault(folder.getId(),
                                                Collections.emptySet()).size();
                                        updatedFolders.add(folder.toBuilder()
                                                .setNextOpLogOrder(folder.getNextOpLogOrder() + newOpLogCount).build());
                                    });
                                    return folderDao.upsertAllRetryable(txSession, updatedFolders)
                                            .thenReturn(currentNextOpLogOrdersByFolder);
                                }))).collectList().map(l -> {
                                    Map<String, Long> result = new HashMap<>();
                                    l.forEach(result::putAll);
                                    return result;
                                })
        ).map(currentNextOpLogOrdersByFolder -> fillOrders(currentNextOpLogOrdersByFolder, folderLogsByFolder,
                operationsToUpsert, sortedFolderLogIdsByFolder)),
                (millis, success) -> LOG.info("Sync prepare orders: duration = {} ms, success = {}", millis, success));
    }

    public Mono<List<AccountModel>> getAllNonDeletedAccountsByAccountsSpaceExcluding(
            YdbTxSession session,
            ProviderModel provider,
            AccountSpaceModel accountsSpace,
            Set<String> externalAccountIdsToExclude) {
        return AsyncMetrics.metric(accountsDao.getAllNonDeletedByAccountsSpaceExcluding(session,
                Tenants.DEFAULT_TENANT_ID, provider.getId(), accountsSpace != null ? accountsSpace.getId() : null,
                externalAccountIdsToExclude),
                (millis, success) -> LOG.info("Sync load non-deleted accounts: duration = {} ms, success = {}",
                        millis, success));
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Set<String>> doUpdates(YdbTxSession session,
                                       List<AccountModel> accountsToUpsert,
                                       List<QuotaModel> quotasToUpsert,
                                       List<AccountsQuotasModel> provisionsToUpsert,
                                       List<FolderOperationLogModel> folderLogsToUpsert,
                                       List<AccountsQuotasOperationsModel> operationsToUpsert,
                                       Set<String> postponedFolders) {
        return AsyncMetrics.metric(accountsDao.upsertAllRetryable(session, accountsToUpsert)
                .then(quotasDao.upsertAllRetryable(session, quotasToUpsert)
                .then(accountsQuotasDao.upsertAllRetryable(session, provisionsToUpsert)
                .then(folderOperationLogDao.upsertAllRetryable(session, folderLogsToUpsert)
                .then(accountsQuotasOperationsDao.upsertAllRetryable(session, operationsToUpsert)))))
                .thenReturn(postponedFolders),
                (millis, success) -> LOG.info("Sync do updates: duration = {} ms, success = {}", millis, success));
    }

    public Mono<Optional<AccountsSpaceKeyRequestDto>> prepareAccountsSpaceKey(AccountSpaceModel accountsSpace) {
        if (accountsSpace == null) {
            return Mono.just(Optional.empty());
        }
        List<Tuple2<String, TenantId>> segmentationIds = accountsSpace.getSegments().stream()
                .map(ResourceSegmentSettingsModel::getSegmentationId).distinct()
                .map(v -> Tuples.of(v, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        List<Tuple2<String, TenantId>> segmentIds = accountsSpace.getSegments().stream()
                .map(ResourceSegmentSettingsModel::getSegmentId).distinct()
                .map(v -> Tuples.of(v, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return AsyncMetrics.metric(tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession -> Flux.fromIterable(Lists
                        .partition(segmentationIds, 500))
                        .concatMap(v -> resourceSegmentationsDao.getByIds(txSession, v))
                        .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()))
                        .flatMap(segmentations -> Flux.fromIterable(Lists.partition(segmentIds, 500))
                                .concatMap(v -> resourceSegmentsDao.getByIds(txSession, v)).collectList()
                                .map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()))
                                .map(segments -> Optional.of(createAccountsSpaceKey(segmentations,
                                        segments, accountsSpace)))))),
                (millis, success) -> LOG.info("Sync prepare accounts space key: duration = {} ms, success = {}",
                        millis, success));
    }

    public Mono<List<SyncResource>> getResources(ProviderModel provider) {
        return AsyncMetrics.metric(tableClient.usingSessionMonoRetryable(session -> session
                .usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession -> resourcesDao
                        .getAllByProvider(txSession, provider.getId(), Tenants.DEFAULT_TENANT_ID, false)
                        .flatMap(resources -> expandResources(txSession, resources)))),
                (millis, success) -> LOG.info("Sync resources load: duration = {} ms, success = {}", millis, success));
    }

    public Mono<List<AccountModel>> getAccountsByFolders(YdbTxSession session, Set<String> folderIds,
                                                          AccountSpaceModel accountsSpace, ProviderModel provider) {
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(new ArrayList<>(folderIds), 500))
                .concatMap(v -> accountsDao.getByFoldersForProvider(session, Tenants.DEFAULT_TENANT_ID,
                        provider.getId(), new HashSet<>(v), accountsSpace != null ? accountsSpace.getId() : null))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get accounts by folders: duration = {} ms, success = {}",
                        millis, success));
    }

    public Mono<List<AccountModel>> getAccountsByExternalIds(YdbTxSession session,
                                                              Set<String> externalIds,
                                                              AccountSpaceModel accountsSpace,
                                                              ProviderModel provider) {
        if (externalIds.isEmpty()) {
            return Mono.just(List.of());
        }
        String accountsSpaceId = accountsSpace != null ? accountsSpace.getId() : null;
        List<WithTenant<AccountModel.ExternalId>> ids = externalIds.stream()
                .map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID,
                        new AccountModel.ExternalId(provider.getId(), id, accountsSpaceId)))
                .collect(Collectors.toList());
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> accountsDao.getAllByExternalIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get accounts by external ids: duration = {} ms, success = {}",
                        millis, success));
    }

    public Mono<List<QuotaModel>> getQuotas(YdbTxSession session, Set<String> folderIds, String providerId,
                                             Set<String> resourceIds) {
        return AsyncMetrics.metric(quotasDao.getByProviderFoldersResources(session, Tenants.DEFAULT_TENANT_ID,
                folderIds, providerId, resourceIds),
                (millis, success) -> LOG.info("Sync get quotas: duration = {} ms, success = {}", millis, success));
    }

    public Mono<List<AccountsQuotasModel>> getProvisions(YdbTxSession session, Set<String> accountIds) {
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(new ArrayList<>(accountIds), 200))
                .concatMap(v -> accountsQuotasDao
                        .getAllByAccountIds(session, Tenants.DEFAULT_TENANT_ID, new HashSet<>(v)))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get provisions: duration = {} ms, success = {}", millis, success));
    }

    public Mono<List<FolderModel>> getTargetFolders(YdbTxSession session, List<ExternalAccount> externalAccounts) {
        List<String> folderIds = externalAccounts.stream().map(ExternalAccount::getFolderId).distinct()
                .collect(Collectors.toList());
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(folderIds, 500))
                .concatMap(v -> folderDao.getByIds(session, v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get target folders: duration = {} ms, success = {}",
                        millis, success));
    }

    public Mono<GroupedAuthors> getAuthors(YdbTxSession session, Set<String> externalAccountIds,
                                           Map<String, ExternalAccount> accountsByExternalId) {
        Set<String> logins = new HashSet<>();
        Set<String> uids = new HashSet<>();
        externalAccountIds.forEach(accountId -> {
            if (!accountsByExternalId.containsKey(accountId)) {
                return;
            }
            ExternalAccount account = accountsByExternalId.get(accountId);
            account.getLastUpdate().flatMap(ExternalLastUpdate::getAuthor).ifPresent(a -> {
                a.getPassportUid().ifPresent(uids::add);
                a.getStaffLogin().ifPresent(logins::add);
            });
            account.getProvisions().forEach(provision ->
                    provision.getLastUpdate().flatMap(ExternalLastUpdate::getAuthor).ifPresent(a -> {
                        a.getPassportUid().ifPresent(uids::add);
                        a.getStaffLogin().ifPresent(logins::add);
                    })
            );
        });
        if (logins.isEmpty() && uids.isEmpty()) {
            return Mono.just(new GroupedAuthors(Collections.emptyMap(), Collections.emptyMap()));
        }
        List<List<String>> partitionedLogins = Lists.partition(new ArrayList<>(logins), 250);
        List<List<String>> partitionedUids = Lists.partition(new ArrayList<>(uids), 250);
        List<Tuple2<List<Tuple2<String, TenantId>>, List<Tuple2<String, TenantId>>>> zipped = new ArrayList<>();
        for (int i = 0; i < Math.max(partitionedLogins.size(), partitionedUids.size()); i++) {
            List<Tuple2<String, TenantId>> loginPairs = i < partitionedLogins.size()
                    ? partitionedLogins.get(i).stream().map(v -> Tuples.of(v, Tenants.DEFAULT_TENANT_ID))
                    .collect(Collectors.toList())
                    : List.of();
            List<Tuple2<String, TenantId>> uidPairs = i < partitionedUids.size()
                    ? partitionedUids.get(i).stream().map(v -> Tuples.of(v, Tenants.DEFAULT_TENANT_ID))
                    .collect(Collectors.toList())
                    : List.of();
            zipped.add(Tuples.of(uidPairs, loginPairs));
        }
        return AsyncMetrics.metric(Flux.fromIterable(zipped)
                .concatMap(v -> usersDao.getByExternalIds(session, v.getT1(), v.getT2(), List.of()))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()))
                .map(users -> {
                    Map<String, UserModel> authorsByUid = new HashMap<>();
                    Map<String, UserModel> authorsByLogin = new HashMap<>();
                    users.forEach(user -> {
                        user.getPassportUid().ifPresent(v -> authorsByUid.put(v, user));
                        user.getPassportLogin().ifPresent(v -> authorsByLogin.put(v, user));
                    });
                    return new GroupedAuthors(authorsByUid, authorsByLogin);
                }), (millis, success) -> LOG.info("Sync get authors: duration = {} ms, success = {}", millis, success));
    }

    public Mono<List<AccountSpaceModel>> getAccountSpaces(ProviderModel provider) {
        return AsyncMetrics.metric(tableClient.usingSessionMonoRetryable(session -> session
                .usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession -> accountsSpacesDao
                        .getAllByProvider(txSession, Tenants.DEFAULT_TENANT_ID, provider.getId())))
                .map(WithTxId::get),
                (millis, success) -> LOG.info("Sync accounts spaces load: duration = {} ms, success = {}",
                        millis, success));
    }

    public Mono<Tuple2<List<AccountsQuotasOperationsModel>, Set<String>>> loadOperations(
            YdbTxSession session,
            Set<String> folderIds,
            Set<String> accountIds,
            Set<String> operationIds,
            String providerId,
            AccountSpaceModel accountsSpace) {
        return AsyncMetrics.metric(getInProgressOperationByFolders(session, folderIds)
                .flatMap(inProgressFolders -> getInProgressOperationByAccounts(session, accountIds)
                        .flatMap(inProgressAccounts -> {
                            Set<String> inProgressFoldersOperationIds = inProgressFolders.stream()
                                    .map(OperationInProgressModel::getOperationId).collect(Collectors.toSet());
                            Set<String> inProgressAccountsOperationIds = inProgressAccounts.stream()
                                    .map(OperationInProgressModel::getOperationId).collect(Collectors.toSet());
                            Set<String> inProgressOperationIds = Sets.union(inProgressFoldersOperationIds,
                                    inProgressAccountsOperationIds);
                            return getOperationsByIds(session, Sets.union(operationIds, inProgressOperationIds),
                                    providerId, accountsSpace)
                                    .map(operations -> Tuples.of(operations, inProgressOperationIds));
                        })),
                (millis, success) -> LOG.info("Sync load operations: duration = {} ms, success = {}", millis, success));
    }

    private Tuple2<List<FolderOperationLogModel>, List<AccountsQuotasOperationsModel>> fillOrders(
            Map<String, Long> currentNextOpLogOrdersByFolder,
            Map<String, Set<FolderOperationLogModel.Builder>> folderLogsByFolder,
            Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
            Map<String, List<String>> sortedFolderLogIdsByFolder) {
        List<FolderOperationLogModel> opLogsResult = new ArrayList<>();
        List<AccountsQuotasOperationsModel> operationsResult = new ArrayList<>();
        folderLogsByFolder.forEach((folderId, opLogs) -> {
            Map<String, Integer> opLogOrderById = new HashMap<>();
            List<String> sortedOpLogIds = sortedFolderLogIdsByFolder.getOrDefault(folderId, Collections.emptyList());
            for (int i = 0; i < sortedOpLogIds.size(); i++) {
                opLogOrderById.put(sortedOpLogIds.get(i), i);
            }
            List<FolderOperationLogModel.Builder> sortedOpLogs = opLogs.stream().sorted((left, right) -> {
                int leftOrder = opLogOrderById.getOrDefault(left.getId().get(), 0);
                int rightOrder = opLogOrderById.getOrDefault(right.getId().get(), 0);
                return Integer.compare(leftOrder, rightOrder);
            }).collect(Collectors.toList());
            long nextOrder = currentNextOpLogOrdersByFolder.get(folderId);
            for (FolderOperationLogModel.Builder builder : sortedOpLogs) {
                builder.setOrder(nextOrder);
                nextOrder++;
                FolderOperationLogModel opLog = builder.build();
                opLogsResult.add(opLog);
                Optional<String> operationId = opLog.getAccountsQuotasOperationsId();
                if (operationId.isPresent()) {
                    AccountsQuotasOperationsModel.Builder operationBuilder = operationsToUpsert.get(operationId.get());
                    if (operationBuilder != null) {
                        updateOperationOrders(operationBuilder, opLog);
                    }
                }
            }
        });
        operationsToUpsert.values().forEach(b -> operationsResult.add(b.build()));
        operationsResult.forEach(operationsObservabilityService::observeOperationFinished);
        return Tuples.of(opLogsResult, operationsResult);
    }

    private void updateOperationOrders(AccountsQuotasOperationsModel.Builder operationBuilder,
                                       FolderOperationLogModel opLog) {
        if (operationBuilder.getOrders().isEmpty() || operationBuilder.getOperationType().isEmpty()
                || operationBuilder.getRequestedChanges().isEmpty()) {
            return;
        }
        if (AccountsQuotasOperationsModel.OperationType.MOVE_ACCOUNT.equals(operationBuilder.getOperationType().get())
                && operationBuilder.getRequestedChanges().get().getDestinationFolderId().isPresent()) {
            String destinationFolderId = operationBuilder.getRequestedChanges().get().getDestinationFolderId().get();
            if (opLog.getFolderId().equals(destinationFolderId)) {
                if (operationBuilder.getOrders().get().getDestinationRestoreOrder().isEmpty()) {
                    operationBuilder.setOrders(OperationOrdersModel.builder(operationBuilder.getOrders().get())
                            .destinationRestoreOrder(opLog.getOrder()).build());
                }
            } else {
                if (operationBuilder.getOrders().get().getRestoreOrder().isEmpty()) {
                    operationBuilder.setOrders(OperationOrdersModel.builder(operationBuilder.getOrders().get())
                            .restoreOrder(opLog.getOrder()).build());
                }
            }
        } else if (AccountsQuotasOperationsModel.OperationType.MOVE_PROVISION
                .equals(operationBuilder.getOperationType().get()) && operationBuilder.getRequestedChanges().get()
                .getAccountId().isPresent() && operationBuilder.getRequestedChanges().get().getDestinationAccountId()
                .isPresent()) {
            String accountId = operationBuilder.getRequestedChanges().get().getAccountId().get();
            String destinationAccountId = operationBuilder.getRequestedChanges().get().getDestinationAccountId().get();
            Set<String> opLogAccountIds = modifiedProvisionsAccountIdsFromOpLog(opLog);
            if (opLogAccountIds.contains(accountId)) {
                if (operationBuilder.getOrders().get().getRestoreOrder().isEmpty()) {
                    operationBuilder.setOrders(OperationOrdersModel.builder(operationBuilder.getOrders().get())
                            .restoreOrder(opLog.getOrder()).build());
                }
            }
            if (opLogAccountIds.contains(destinationAccountId)) {
                if (operationBuilder.getOrders().get().getDestinationRestoreOrder().isEmpty()) {
                    operationBuilder.setOrders(OperationOrdersModel.builder(operationBuilder.getOrders().get())
                            .destinationRestoreOrder(opLog.getOrder()).build());
                }
            }
        } else {
            if (operationBuilder.getOrders().get().getRestoreOrder().isEmpty()) {
                operationBuilder.setOrders(OperationOrdersModel.builder(operationBuilder.getOrders().get())
                        .restoreOrder(opLog.getOrder()).build());
            }
        }
    }

    private Set<String> modifiedProvisionsAccountIdsFromOpLog(FolderOperationLogModel opLog) {
        Set<String> accountIds = new HashSet<>();
        opLog.getOldProvisions().forEach((accountId, v) -> accountIds.add(accountId));
        opLog.getNewProvisions().forEach((accountId, v) -> accountIds.add(accountId));
        return accountIds;
    }

    private AccountsSpaceKeyRequestDto createAccountsSpaceKey(List<ResourceSegmentationModel> segmentations,
                                                              List<ResourceSegmentModel> segments,
                                                              AccountSpaceModel accountsSpace) {
        Map<String, ResourceSegmentationModel> segmentationsById
                = segmentations.stream().collect(Collectors
                .toMap(ResourceSegmentationModel::getId, Function.identity()));
        Map<String, ResourceSegmentModel> segmentsById
                = segments.stream().collect(Collectors
                .toMap(ResourceSegmentModel::getId, Function.identity()));
        List<SegmentKeyRequestDto> keys = accountsSpace.getSegments().stream()
                .map(v -> new SegmentKeyRequestDto(segmentationsById.get(v.getSegmentationId()).getKey(),
                        segmentsById.get(v.getSegmentId()).getKey())).collect(Collectors.toList());
        return new AccountsSpaceKeyRequestDto(keys);
    }

    private Mono<List<SyncResource>> expandResources(YdbTxSession session, List<ResourceModel> resources) {
        return AsyncMetrics.metric(getResourceTypes(session, resources)
                .flatMap(resourceTypes -> getResourceSegmentations(session, resources)
                .flatMap(resourceSegmentations -> getResourceSegments(session, resources)
                .flatMap(resourceSegments -> getUnitsEnsembles(session, resources)
                .map(unitsEnsembles -> {
                    List<SyncResource> result = new ArrayList<>();
                    Map<String, ResourceTypeModel> resourceTypesById = resourceTypes.stream()
                            .collect(Collectors.toMap(ResourceTypeModel::getId, Function.identity()));
                    Map<String, ResourceSegmentationModel> segmentationsById = resourceSegmentations.stream()
                            .collect(Collectors.toMap(ResourceSegmentationModel::getId, Function.identity()));
                    Map<String, ResourceSegmentModel> segmentById = resourceSegments.stream()
                            .collect(Collectors.toMap(ResourceSegmentModel::getId, Function.identity()));
                    Map<String, UnitsEnsembleModel> ensemblesById = unitsEnsembles.stream()
                            .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
                    resources.forEach(resource -> {
                        ResourceTypeModel resourceType = resourceTypesById.get(resource.getResourceTypeId());
                        Set<SyncSegment> segments = new HashSet<>();
                        Optional.ofNullable(resource.getSegments()).orElse(Collections.emptySet()).forEach(segment ->
                                segments.add(new SyncSegment(segmentationsById.get(segment.getSegmentationId()),
                                        segmentById.get(segment.getSegmentId()))));
                        UnitsEnsembleModel unitsEnsemble = ensemblesById.get(resource.getUnitsEnsembleId());
                        result.add(new SyncResource(resource, resourceType, segments, unitsEnsemble));
                    });
                    return result;
                })))),
                (millis, success) -> LOG.info("Sync expand resources: duration = {} ms, success = {}",
                        millis, success));
    }

    private Mono<List<ResourceTypeModel>> getResourceTypes(YdbTxSession session, List<ResourceModel> resources) {
        List<String> resourceTypeIds = resources.stream().flatMap(r -> Optional.ofNullable(r.getResourceTypeId())
                .stream()).distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = resourceTypeIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceTypesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get resource types: duration = {} ms, success = {}",
                        millis, success));
    }

    private Mono<List<ResourceSegmentationModel>> getResourceSegmentations(YdbTxSession session,
                                                                           List<ResourceModel> resources) {
        List<String> resourceSegmentationIds = resources.stream().flatMap(r -> Optional.ofNullable(r.getSegments())
                .stream().flatMap(m -> m.stream().map(ResourceSegmentSettingsModel::getSegmentationId)))
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = resourceSegmentationIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentationsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get resource segmentations: duration = {} ms, success = {}",
                        millis, success));
    }

    private Mono<List<ResourceSegmentModel>> getResourceSegments(YdbTxSession session, List<ResourceModel> resources) {
        List<String> resourceSegmentationIds = resources.stream().flatMap(r -> Optional.ofNullable(r.getSegments())
                .stream().flatMap(m -> m.stream().map(ResourceSegmentSettingsModel::getSegmentId)))
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = resourceSegmentationIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get resource segments: duration = {} ms, success = {}",
                        millis, success));
    }

    private Mono<List<UnitsEnsembleModel>> getUnitsEnsembles(YdbTxSession session, List<ResourceModel> resources) {
        List<String> unitsEnsembleIds = resources.stream().map(ResourceModel::getUnitsEnsembleId)
                .distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> ids = unitsEnsembleIds.stream().map(id -> Tuples.of(id,
                Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> unitsEnsemblesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get units ensembles: duration = {} ms, success = {}",
                        millis, success));
    }

    private Mono<List<OperationInProgressModel>> getInProgressOperationByFolders(YdbTxSession session,
                                                                                 Set<String> folderIds) {
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(new ArrayList<>(folderIds), 500))
                .concatMap(v -> operationsInProgressDao
                        .getAllByTenantFolders(session, Tenants.DEFAULT_TENANT_ID, new HashSet<>(v)))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .filter(op -> op.getAccountId().isEmpty()).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get in progress ops by folders: duration = {} ms, success = {}",
                        millis, success));
    }

    private Mono<List<OperationInProgressModel>> getInProgressOperationByAccounts(YdbTxSession session,
                                                                                  Set<String> accountIds) {
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(new ArrayList<>(accountIds), 500))
                .concatMap(v -> operationsInProgressDao
                        .getAllByTenantAccounts(session, Tenants.DEFAULT_TENANT_ID, new HashSet<>(v)))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get in progress ops by accounts: duration = {} ms, success = {}",
                        millis, success));
    }

    private Mono<List<AccountsQuotasOperationsModel>> getOperationsByIds(YdbTxSession session,
                                                                         Set<String> operationIds,
                                                                         String providerId,
                                                                         AccountSpaceModel accountsSpace) {
        if (operationIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<WithTenant<String>> ids = operationIds.stream()
                .map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id)).collect(Collectors.toList());
        return AsyncMetrics.metric(Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> accountsQuotasOperationsDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream)
                        .filter(op -> {
                            if (accountsSpace == null) {
                                return providerId.equals(op.getProviderId()) && op.getAccountsSpaceId().isEmpty();
                            } else {
                                return providerId.equals(op.getProviderId()) && Objects.equal(op.getAccountsSpaceId(),
                                        Optional.of(accountsSpace).map(AccountSpaceModel::getId));
                            }
                        }).collect(Collectors.toList())),
                (millis, success) -> LOG.info("Sync get ops by ids: duration = {} ms, success = {}", millis, success));
    }

    public Mono<ProvidersSyncStatusModel> upsertNewSyncStatus(TenantId tenantId, String providerId, Clock clock) {
        Instant newSyncStart = Instant.now(clock);
        String newSyncId = UUID.randomUUID().toString();
        ProvidersSyncStatusModel.SyncStatuses newSyncStatus = ProvidersSyncStatusModel.SyncStatuses.RUNNING;
        return AsyncMetrics.metric(tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        providersSyncStatusDao.upsertNewSyncRetryable(
                                txSession, tenantId, providerId, newSyncStart, newSyncId, newSyncStatus
                        )
        )).map(unused -> new ProvidersSyncStatusModel.Builder()
                .setTenantId(tenantId)
                .setProviderId(providerId)
                .setNewSyncStart(newSyncStart)
                .setNewSyncId(newSyncId)
                .setNewSyncStatus(newSyncStatus)
                .build()
        ), (millis, success) -> LOG.info("Sync status upsert: duration = {} ms, success = {}", millis, success));
    }

    public Mono<ProvidersSyncStatusModel> updateSyncStatus(
            ProvidersSyncStatusModel syncStatus,
            ProcessFoldersPageResult syncResult,
            Clock clock
    ) {
        Instant now = Instant.now(clock);
        Instant syncStart = syncStatus.getNewSyncStart();
        String syncId = syncStatus.getNewSyncId();
        ProvidersSyncStatusModel.Builder status = new ProvidersSyncStatusModel.Builder(syncStatus)
                .setLastSyncStart(syncStart)
                .setLastSyncId(syncId)
                .setLastSyncFinish(now)
                .setLastSyncStatus(syncResult.hasErrors() ? DONE_ERROR : DONE_OK)
                .setLastSyncStats(new SyncStats(
                        (long) syncResult.getAllReceivedAccountExternalIdsSize(),
                        syncResult.getReceivedQuotasCount(),
                        Duration.between(syncStart, now)
                ));
        LOG.info("Sync status: {}", status);
        return AsyncMetrics.metric(
            tableClient.usingSessionMonoRetryable(session -> session.usingCompTxRetryable(
                tx -> providersSyncStatusDao.getByIdStartTx(tx, syncStatus.getProviderId(), syncStatus.getTenantId()),
                (tx, oldStatus) -> providersSyncStatusDao.upsertOneRetryable(tx,
                        status.setLastSuccessfulSyncFinish(syncResult.hasErrors() ?
                                oldStatus.map(ProvidersSyncStatusModel::getLastSuccessfulSyncFinish).orElse(null) : now
                        ).build()),
                (tx, result) -> providersSyncErrorsDao.clearOldErrorsRetryable(tx,
                        result.getTenantId(), result.getProviderId(), result.getLastSyncId()
                ).map(voidWithTxId -> result)
            )),
            (millis, success) -> LOG.info("Sync status update: duration = {} ms, success = {}", millis, success)
        );
    }

    public Mono<ProvidersSyncErrorsModel> insertSyncError(ProvidersSyncErrorsModel error) {
        return AsyncMetrics.metric(tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        providersSyncErrorsDao.upsertOneRetryable(txSession, error)
        )), (millis, success) -> LOG.info("Sync error upsert: duration = {} ms, success = {}", millis, success));
    }

    ProvidersSyncErrorsModel newError(
            AccountSpaceModel accountsSpace,
            ProvidersSyncStatusModel syncStatus,
            Map<String, String> details,
            String errorMessage
    ) {
        return new ProvidersSyncErrorsModel.Builder(
                syncStatus,
                accountsSpace != null ? accountsSpace.getId() : null,
                UUID.randomUUID().toString()
        )
                .setRequestTimestamp(Instant.now())
                .setErrors(new Errors(errorMessage,
                        details, null))
                .build();
    }

    @SuppressWarnings("ParameterNumber")
    Mono<QuotasAndOperations> loadQuotasAndOperations(
            YdbTxSession session,
            String providerId,
            Set<String> folderIds,
            Set<String> resourceIds,
            Set<String> accountIds,
            Set<String> receivedOperationIds,
            List<AccountModel> currentFoldersPageAccounts,
            List<AccountModel> existingAccountsMovedToFoldersPage,
            AccountSpaceModel accountsSpace
    ) {
        return getQuotas(session, folderIds, providerId, resourceIds).flatMap(quotas ->
                getProvisions(session, accountIds).flatMap(provisions -> {
                    Set<String> currentOperationIds = AccountsSyncUtils.collectCurrentOperationIds(
                            currentFoldersPageAccounts, existingAccountsMovedToFoldersPage,
                            provisions);
                    Set<String> operationIdsToLoad = Sets.union(currentOperationIds, receivedOperationIds);
                    return loadOperations(session, folderIds, accountIds, operationIdsToLoad,
                            providerId, accountsSpace).flatMap(tuple -> {
                        Map<String, Map<String, QuotaModel>> groupedQuotas = new HashMap<>();
                        quotas.forEach(q -> groupedQuotas.computeIfAbsent(q.getFolderId(), k -> new HashMap<>())
                                .put(q.getResourceId(), q));
                        Map<String, Map<String, AccountsQuotasModel>> groupedProvisions = new HashMap<>();
                        provisions.forEach(p -> groupedProvisions.computeIfAbsent(p.getAccountId(),
                                k -> new HashMap<>()).put(p.getResourceId(), p));
                        Map<String, List<AccountsQuotasOperationsModel>> groupedOperations = new HashMap<>();
                        Map<String, List<AccountsQuotasOperationsModel>> operationsInProgressByAccount
                                = new HashMap<>();
                        Map<String, AccountsQuotasOperationsModel> operationsById = new HashMap<>();
                        tuple.getT1().forEach(operation -> {
                            operationsById.put(operation.getOperationId(), operation);
                            if (operation.getRequestedChanges().getAccountId().isPresent()) {
                                groupedOperations.computeIfAbsent(operation.getRequestedChanges().getAccountId().get(),
                                        k -> new ArrayList<>()).add(operation);
                                if (tuple.getT2().contains(operation.getOperationId())) {
                                    operationsInProgressByAccount.computeIfAbsent(operation.getRequestedChanges()
                                            .getAccountId().get(), k -> new ArrayList<>()).add(operation);
                                }
                            }
                            if (operation.getRequestedChanges().getDestinationAccountId().isPresent()) {
                                groupedOperations.computeIfAbsent(operation.getRequestedChanges()
                                                .getDestinationAccountId().get(),
                                        k -> new ArrayList<>()).add(operation);
                                if (tuple.getT2().contains(operation.getOperationId())) {
                                    operationsInProgressByAccount.computeIfAbsent(operation.getRequestedChanges()
                                                    .getDestinationAccountId().get(),
                                            k -> new ArrayList<>()).add(operation);
                                }
                            }
                        });
                        return Mono.just(new QuotasAndOperations(groupedQuotas, groupedProvisions, groupedOperations,
                                tuple.getT2(), operationsInProgressByAccount, operationsById));
                    });
                }));
    }

    public Mono<ProvidersSyncErrorsModel> insertSyncError(
            String errorMessage,
            ProvidersSyncStatusModel syncStatus,
            ErrorCollection errorCollection,
            Map<String, String> details
    ) {
        HashMap<String, String> allDetails = new HashMap<>(details);
        allDetails.put("errorCollection", errorCollection.toString());
        ProvidersSyncErrorsModel error = new ProvidersSyncErrorsModel.Builder(
                syncStatus,
                null,
                UUID.randomUUID().toString()
        )
                .setRequestTimestamp(Instant.now())
                .setErrors(new Errors(errorMessage,
                        allDetails, null))
                .build();
        return AsyncMetrics.metric(tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        providersSyncErrorsDao.upsertOneRetryable(txSession, error)
        )), (millis, success) -> LOG.info("Sync error info upsert: duration = {} ms, success = {}", millis, success));
    }
}
