package ru.yandex.intranet.d.services.sync;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.base.Objects;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel;
import ru.yandex.intranet.d.model.folders.AccountHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.sync.model.ExternalAccount;
import ru.yandex.intranet.d.services.sync.model.ExternalCompoundResourceKey;
import ru.yandex.intranet.d.services.sync.model.ExternalLastUpdate;
import ru.yandex.intranet.d.services.sync.model.ExternalProvision;
import ru.yandex.intranet.d.services.sync.model.FolderHistoryGroupKey;
import ru.yandex.intranet.d.services.sync.model.FolderOperationAuthor;
import ru.yandex.intranet.d.services.sync.model.GroupedAuthors;
import ru.yandex.intranet.d.services.sync.model.QuotasAndOperations;
import ru.yandex.intranet.d.services.sync.model.SyncResource;
import ru.yandex.intranet.d.util.units.Units;

/**
 * Service to sync providers accounts and quotas, utils part.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class AccountsSyncUtils {

    public static final Duration SYNC_DELAY = Duration.ofMillis(300);

    private static final Logger LOG = LoggerFactory.getLogger(AccountsSyncUtils.class);

    private AccountsSyncUtils() {
    }

    public static boolean isCoolDown(Instant updateTime, Instant newSyncStartTime) {
        return Duration.between(updateTime, newSyncStartTime).compareTo(SYNC_DELAY) < 0;
    }

    public static boolean isCoolDown(Instant updateTime, ProvidersSyncStatusModel syncStatus) {
        return isCoolDown(updateTime, syncStatus.getNewSyncStart());
    }

    public static Map<ExternalCompoundResourceKey, SyncResource> prepareExternalIndex(List<SyncResource> resources) {
        Map<ExternalCompoundResourceKey, SyncResource> result = new HashMap<>();
        resources.forEach(resource -> {
            String resourceTypeKey = resource.getResourceType().getKey();
            Set<ExternalCompoundResourceKey.ExternalCompoundSegmentKey> segments = resource.getResourceSegments()
                    .stream().map(s -> new ExternalCompoundResourceKey.ExternalCompoundSegmentKey(s.getSegmentation()
                            .getKey(), s.getSegment().getKey())).collect(Collectors.toSet());
            ExternalCompoundResourceKey externalKey = new ExternalCompoundResourceKey(resourceTypeKey, segments);
            result.put(externalKey, resource);
        });
        return result;
    }

    @SuppressWarnings("ParameterNumber")
    public static void prepareBalancesForAccountDeletion(ProviderModel provider,
                                                         Map<String, Map<String, QuotaModel>> quotasByFolderResource,
                                                         List<QuotaModel> quotasToUpsert,
                                                         String folderId,
                                                         String resourceId,
                                                         Long balance,
                                                         ResourceModel resource
    ) {
        Long defaultQuota = (resource != null && resource.getDefaultQuota().isPresent()) ?
            resource.getDefaultQuota().get() : null;
        QuotaModel currentQuota = quotasByFolderResource
                .getOrDefault(folderId, Collections.emptyMap()).get(resourceId);
        long currentBalance = currentQuota != null
                ? (currentQuota.getBalance() != null ? currentQuota.getBalance() : 0L) : 0L;
        if (currentBalance == balance && defaultQuota == null) {
            return;
        }
        if (currentQuota != null) {
            QuotaModel.Builder updatedQuota = QuotaModel.builder(currentQuota);
            updatedQuota.balance(balance);
            if (defaultQuota != null && currentQuota.getQuota() != null &&
                    currentQuota.getQuota() >= defaultQuota &&
                    balance >= defaultQuota
            ) {
                updatedQuota
                        .quota(currentQuota.getQuota() - defaultQuota)
                        .balance(balance - defaultQuota)
                        .build();
            }
            quotasToUpsert.add(updatedQuota.build());
        } else {
            QuotaModel newQuota = QuotaModel.builder()
                    .tenantId(Tenants.DEFAULT_TENANT_ID)
                    .folderId(folderId)
                    .providerId(provider.getId())
                    .resourceId(resourceId)
                    .quota(0L)
                    .balance(balance)
                    .frozenQuota(0L)
                    .build();
            quotasToUpsert.add(newQuota);
        }
    }

    @SuppressWarnings("ParameterNumber")
    public static void processProvisionForAccountDeletion(
            List<AccountsQuotasModel> provisionsToUpsert,
            Map<String, Map<String, Long>> updatedBalances,
            Map<String, Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>>> oldProvisions,
            Map<String, Map<FolderHistoryGroupKey, Map<String, Map<String, ProvisionHistoryModel>>>> newProvisions,
            Set<String> deletedAccountIds,
            AccountsQuotasModel provision,
            Instant now) {
        if (deletedAccountIds.contains(provision.getAccountId())) {
            provisionsToUpsert.add(provision.copyBuilder()
                    .setProvidedQuota(0L)
                    .setAllocatedQuota(0L)
                    .setLastProvisionUpdate(now)
                    .setLastReceivedProvisionVersion(null)
                    .setLatestSuccessfulProvisionOperationId(null)
                    .build());
            if (!Objects.equal(provision.getProvidedQuota(), 0L)
                    || !Objects.equal(provision.getLastReceivedProvisionVersion().orElse(null), null)) {
                ProvisionHistoryModel oldProvision = new ProvisionHistoryModel(
                        provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L,
                        provision.getLastReceivedProvisionVersion().orElse(null));
                ProvisionHistoryModel newProvision = new ProvisionHistoryModel(0L, null);
                oldProvisions.computeIfAbsent(provision.getFolderId(), k -> new HashMap<>())
                        .computeIfAbsent(new FolderHistoryGroupKey(null, null), k -> new HashMap<>())
                        .computeIfAbsent(provision.getAccountId(), k -> new HashMap<>())
                        .put(provision.getResourceId(), oldProvision);
                newProvisions.computeIfAbsent(provision.getFolderId(), k -> new HashMap<>())
                        .computeIfAbsent(new FolderHistoryGroupKey(null, null), k -> new HashMap<>())
                        .computeIfAbsent(provision.getAccountId(), k -> new HashMap<>())
                        .put(provision.getResourceId(), newProvision);
            }
        } else {
            long providedAmount = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
            if (providedAmount != 0L) {
                Map<String, Long> folderUpdatedBalances = updatedBalances
                        .computeIfAbsent(provision.getFolderId(), k -> new HashMap<>());
                long currentBalance = folderUpdatedBalances
                        .getOrDefault(provision.getResourceId(), 0L);
                Optional<Long> updatedBalanceO = Units.subtract(currentBalance, providedAmount);
                if (updatedBalanceO.isPresent()) {
                    folderUpdatedBalances.put(provision.getResourceId(), updatedBalanceO.get());
                } else {
                    LOG.error("Invalid balance for resource {} in folder {}", provision.getResourceId(),
                            provision.getFolderId());
                }
            }
        }
    }

    public static void groupQuotasAndOperation(
            List<QuotaModel> quotas,
            List<AccountsQuotasModel> provisions,
            List<AccountsQuotasOperationsModel> operationsInProgress,
            Map<String, Map<String, QuotaModel>> quotasByFolderResource,
            Map<String, List<AccountsQuotasOperationsModel>> operationsInProgressByAccount,
            Map<String, Map<String, AccountsQuotasModel>> provisionsByAccountResource) {
        quotas.forEach(q -> quotasByFolderResource.computeIfAbsent(q.getFolderId(), k -> new HashMap<>())
                .put(q.getResourceId(), q));
        provisions.forEach(p -> provisionsByAccountResource.computeIfAbsent(p.getAccountId(),
                k -> new HashMap<>()).put(p.getResourceId(), p));
        operationsInProgress.forEach(operation -> {
            if (operation.getRequestedChanges().getAccountId().isPresent()) {
                operationsInProgressByAccount.computeIfAbsent(operation.getRequestedChanges()
                        .getAccountId().get(), k -> new ArrayList<>()).add(operation);
            }
            if (operation.getRequestedChanges().getDestinationAccountId().isPresent()) {
                operationsInProgressByAccount.computeIfAbsent(operation.getRequestedChanges()
                        .getDestinationAccountId().get(), k -> new ArrayList<>()).add(operation);
            }
        });
    }

    public static ExternalAccount mergeAccounts(ExternalAccount left, ExternalAccount right) {
        if (left.getAccountVersion().isPresent() && right.getAccountVersion().isPresent()) {
            return left.getAccountVersion().get() > right.getAccountVersion().get() ? left : right;
        }
        boolean versionedProvisions = left.getProvisions().stream().allMatch(p -> p.getQuotaVersion().isPresent())
                && right.getProvisions().stream().allMatch(p -> p.getQuotaVersion().isPresent());
        if (versionedProvisions) {
            Map<String, ExternalProvision> leftProvisionsByResource = left.getProvisions().stream()
                    .collect(Collectors.toMap(v -> v.getResource().getResource().getId(), Function.identity()));
            Map<String, ExternalProvision> rightProvisionsByResource = right.getProvisions().stream()
                    .collect(Collectors.toMap(v -> v.getResource().getResource().getId(), Function.identity()));
            boolean leftIsNewer = leftProvisionsByResource.entrySet().stream()
                    .allMatch(e -> !rightProvisionsByResource.containsKey(e.getKey())
                            || e.getValue().getQuotaVersion().get()
                            >= rightProvisionsByResource.get(e.getKey()).getQuotaVersion().get());
            return leftIsNewer ? left : right;
        }
        if (left.getLastUpdate().isPresent() && left.getLastUpdate().get().getTimestamp().isPresent()
                && right.getLastUpdate().isPresent() && right.getLastUpdate().get().getTimestamp().isPresent()) {
            return left.getLastUpdate().get().getTimestamp().get()
                    .compareTo(right.getLastUpdate().get().getTimestamp().get()) > 0 ? left : right;
        }
        boolean timestampedProvisions = left.getProvisions().stream()
                .allMatch(p -> p.getLastUpdate().isPresent() && p.getLastUpdate().get().getTimestamp().isPresent())
                && right.getProvisions().stream().allMatch(p -> p.getLastUpdate().isPresent()
                && p.getLastUpdate().get().getTimestamp().isPresent());
        if (timestampedProvisions) {
            Map<String, ExternalProvision> leftProvisionsByResource = left.getProvisions().stream()
                    .collect(Collectors.toMap(v -> v.getResource().getResource().getId(), Function.identity()));
            Map<String, ExternalProvision> rightProvisionsByResource = right.getProvisions().stream()
                    .collect(Collectors.toMap(v -> v.getResource().getResource().getId(), Function.identity()));
            boolean leftIsNewer = leftProvisionsByResource.entrySet().stream()
                    .allMatch(e -> !rightProvisionsByResource.containsKey(e.getKey())
                            || e.getValue().getLastUpdate().get().getTimestamp().get()
                            .compareTo(rightProvisionsByResource
                                    .get(e.getKey()).getLastUpdate().get().getTimestamp().get()) >= 0);
            return leftIsNewer ? left : right;
        }
        return left;
    }

    public static List<SyncResource> filterResourcesByAccountsSpace(Collection<SyncResource> allResources,
                                                                    AccountSpaceModel accountsSpace) {
        if (accountsSpace != null) {
            return allResources.stream().filter(r -> accountsSpace.getId()
                    .equals(r.getResource().getAccountsSpacesId())).collect(Collectors.toList());
        }
        return new ArrayList<>(allResources);
    }

    public static Map<String, String> preGenerateAccountIds(Map<String, List<ExternalAccount>> accountsByTargetFolder,
                                                            Set<String> targetFoldersPage) {
        Map<String, String> result = new HashMap<>();
        targetFoldersPage.forEach(folderId -> {
            accountsByTargetFolder.getOrDefault(folderId, Collections.emptyList()).forEach(account -> {
                result.put(account.getAccountId(), UUID.randomUUID().toString());
            });
        });
        return result;
    }

    public static Set<String> collectOperationIds(ProviderModel provider, Set<String> folderIds,
                                                  Map<String, List<ExternalAccount>> accountsByTargetFolder) {
        Set<String> operations = new HashSet<>();
        if (provider.getAccountsSettings().isPerAccountLastUpdateSupported()) {
            folderIds.forEach(folderId -> accountsByTargetFolder
                    .getOrDefault(folderId, Collections.emptyList()).forEach(account -> {
                        if (account.getLastUpdate().isPresent()
                                && account.getLastUpdate().get().getOperationId().isPresent()) {
                            operations.add(account.getLastUpdate().get().getOperationId().get());
                        }
                    }));
        }
        if (provider.getAccountsSettings().isPerProvisionLastUpdateSupported()) {
            folderIds.forEach(folderId -> accountsByTargetFolder
                    .getOrDefault(folderId, Collections.emptyList()).forEach(account ->
                            account.getProvisions().forEach(provision -> {
                                if (provision.getLastUpdate().isPresent()
                                        && provision.getLastUpdate().get().getOperationId().isPresent()) {
                                    operations.add(provision.getLastUpdate().get().getOperationId().get());
                                }
                            })));
        }
        return operations;
    }

    public static Set<String> collectCurrentOperationIds(List<AccountModel> currentFoldersPageAccounts,
                                                         List<AccountModel> existingAccountsMovedToFoldersPage,
                                                         List<AccountsQuotasModel> provisions) {
        Set<String> result = new HashSet<>();
        currentFoldersPageAccounts.forEach(account -> {
            if (account.getLatestSuccessfulAccountOperationId().isPresent()) {
                result.add(account.getLatestSuccessfulAccountOperationId().get());
            }
        });
        existingAccountsMovedToFoldersPage.forEach(account -> {
            if (account.getLatestSuccessfulAccountOperationId().isPresent()) {
                result.add(account.getLatestSuccessfulAccountOperationId().get());
            }
        });
        provisions.forEach(provision -> {
            if (provision.getLatestSuccessfulProvisionOperationId().isPresent()) {
                result.add(provision.getLatestSuccessfulProvisionOperationId().get());
            }
        });
        return result;
    }

    public static void prepareUpdatedBalances(Map<String, Long> updatedBalances,
                                              Map<String, QuotaModel> currentQuotas) {
        currentQuotas.forEach((resourceId, quota) -> updatedBalances.put(resourceId, quota.getQuota()));
        currentQuotas.forEach((resourceId, quota) -> {
            if (quota.getFrozenQuota() != 0L) {
                Optional<Long> updatedBalance = Units.subtract(updatedBalances.getOrDefault(resourceId, 0L),
                        quota.getFrozenQuota());
                if (updatedBalance.isPresent()) {
                    updatedBalances.put(resourceId, updatedBalance.get());
                } else {
                    LOG.error("Failed to update balance for resource {} in folder {}", resourceId,
                            quota.getFolderId());
                }
            }
        });
    }

    public static void prepareQuotasToUpsert(List<QuotaModel> quotasToUpsert, Map<String, Long> updatedBalances,
                                             Map<String, Long> defaultQuotas, Map<String, QuotaModel> currentQuotas,
                                             String folderId, ProviderModel provider) {
        updatedBalances.forEach((resourceId, updatedBalance) -> {
            long defaultQuota = defaultQuotas.getOrDefault(resourceId, 0L);
            if (currentQuotas.containsKey(resourceId)) {
                QuotaModel currentQuota = currentQuotas.get(resourceId);
                long currentBalance = currentQuota.getBalance() == null ? 0L : currentQuota.getBalance();
                long currentQuotaValue = currentQuota.getQuota() == null ? 0L : currentQuota.getQuota();
                Optional<Long> updatedQuotaValueO = Units.add(currentQuotaValue, defaultQuota);
                long updatedQuotaValue = updatedQuotaValueO.orElse(currentQuotaValue);
                if (updatedQuotaValueO.isEmpty()) {
                    LOG.error("Failed to update quota for resource {} in folder {}", resourceId, folderId);
                }
                if (currentBalance != updatedBalance || currentQuotaValue != updatedQuotaValue) {
                    QuotaModel modifiedQuota = QuotaModel.builder(currentQuota)
                            .balance(updatedBalance)
                            .quota(updatedQuotaValue)
                            .build();
                    quotasToUpsert.add(modifiedQuota);
                }
            } else {
                QuotaModel newQuota = QuotaModel.builder()
                        .tenantId(Tenants.DEFAULT_TENANT_ID)
                        .folderId(folderId)
                        .providerId(provider.getId())
                        .resourceId(resourceId)
                        .quota(defaultQuota)
                        .balance(updatedBalance)
                        .frozenQuota(0L)
                        .build();
                quotasToUpsert.add(newQuota);
            }
        });
    }

    public static void updateAllocations(List<AccountsQuotasModel> provisionsToUpsert,
                                         ExternalAccount updatedAccount,
                                         AccountModel account,
                                         QuotasAndOperations quotasAndOperations,
                                         ProviderModel provider) {
        updatedAccount.getProvisions().forEach(updatedProvision -> {
            String resourceId = updatedProvision.getResource().getResource().getId();
            AccountsQuotasModel currentProvision = quotasAndOperations.getProvisionsByAccountResource()
                    .getOrDefault(account.getId(), Collections.emptyMap()).get(resourceId);
            if (currentProvision == null) {
                return;
            }
            Optional<Long> updatedProvidedAmountO = Units.convertFromApi(updatedProvision.getProvided(),
                    updatedProvision.getResource().getResource(),
                    updatedProvision.getResource().getUnitsEnsemble(), updatedProvision.getProvidedUnit());
            if (updatedProvidedAmountO.isEmpty()) {
                LOG.error("Invalid provision {} {} received for resource {} in account {} of provider {}",
                        updatedProvision.getProvided(), updatedProvision.getProvidedUnit().getKey(),
                        updatedProvision.getResource().getResource().getKey(),
                        updatedAccount.getAccountId(), provider.getKey());
            }
            long updatedProvidedAmount = updatedProvidedAmountO.orElse(0L);
            long currentProvidedAmount = currentProvision.getProvidedQuota() != null
                    ? currentProvision.getProvidedQuota() : 0L;
            Optional<Long> updatedAllocatedAmountO = Units.convertFromApi(updatedProvision.getAllocated(),
                    updatedProvision.getResource().getResource(),
                    updatedProvision.getResource().getUnitsEnsemble(),
                    updatedProvision.getAllocatedUnit());
            if (updatedAllocatedAmountO.isEmpty()) {
                LOG.error("Invalid allocation received for resource {} in account {} of provider {}",
                        updatedProvision.getResource().getResource().getKey(),
                        updatedAccount.getAccountId(), provider.getKey());
            }
            long updatedAllocatedAmount = updatedAllocatedAmountO.orElse(0L);
            long currentAllocatedAmount = currentProvision.getAllocatedQuota() != null
                    ? currentProvision.getAllocatedQuota() : 0L;
            if (currentAllocatedAmount == updatedAllocatedAmount) {
                return;
            }
            if (currentProvidedAmount != updatedProvidedAmount) {
                return;
            }
            provisionsToUpsert.add(new AccountsQuotasModel.Builder(currentProvision)
                    .setAllocatedQuota(updatedAllocatedAmount).build());
        });
    }

    public static void updateBalance(
            AccountModel account, QuotasAndOperations quotasAndOperations, Map<String, Long> updatedBalances,
            Map<String, Long> defaultQuotas, String folderId
    ) {
        quotasAndOperations.getProvisionsByAccountResource().getOrDefault(account.getId(), Collections.emptyMap())
                .forEach((resourceId, provision) -> {
                    long providedAmount = provision.getProvidedQuota() != null ? provision.getProvidedQuota() : 0L;
                    if (providedAmount != 0L) {
                        Optional<Long> updatedBalanceO = Units.subtract(updatedBalances
                                .getOrDefault(resourceId, 0L), providedAmount);
                        if (updatedBalanceO.isPresent()) {
                            updatedBalances.put(resourceId, updatedBalanceO.get());
                        } else {
                            LOG.error("Invalid balance for resource {} in folder {}", resourceId, folderId);
                        }
                    }
                });

    }

    @SuppressWarnings("ParameterNumber")
    public static void processAccountDeletion(
            Map<String, List<AccountsQuotasOperationsModel>> operationsInProgressByAccount,
            List<AccountModel> accountsToUpsert,
            Map<String, Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>>> oldAccounts,
            Map<String, Map<FolderHistoryGroupKey, Map<String, AccountHistoryModel>>> newAccounts,
            Set<String> folderIdsToUpdate,
            Instant now,
            AccountModel deletedAccount,
            Set<String> deletedAccountIds,
            ProviderModel provider,
            ProvidersSyncStatusModel syncStatus,
            List<AccountModel> accountsToRemoveReserve) {
        if (operationsInProgressByAccount.containsKey(deletedAccount.getId())) {
            LOG.info("Skipping account {} deletion due to in progress operations still being present: {}",
                    deletedAccount.getOuterAccountIdInProvider(), operationsInProgressByAccount
                            .get(deletedAccount.getId()));
            return;
        }
        boolean coolDown = isCoolDown(deletedAccount.getLastAccountUpdate(), syncStatus);
        boolean coolDownDisabled = provider.getAccountsSettings().isSyncCoolDownDisabled();
        if (coolDown && !coolDownDisabled) {
            LOG.info("Account deletion {} is not synced due to active cool down period",
                    deletedAccount.getOuterAccountIdInProvider());
            return;
        }
        AccountModel.Builder updatedAccountBuilder = new AccountModel.Builder(deletedAccount)
                .setDeleted(true)
                .setVersion(deletedAccount.getVersion() + 1L);
        if (deletedAccount.getReserveType().isPresent()) {
            updatedAccountBuilder.setReserveType(null);
        }
        AccountModel updatedAccount = updatedAccountBuilder.build();
        if (deletedAccount.getReserveType().isPresent()) {
            accountsToRemoveReserve.add(updatedAccount);
        }
        accountsToUpsert.add(updatedAccount);
        folderIdsToUpdate.add(updatedAccount.getFolderId());
        deletedAccountIds.add(updatedAccount.getId());
        AccountHistoryModel.Builder oldAccountBuilder = AccountHistoryModel.builder()
                .version(deletedAccount.getVersion())
                .deleted(deletedAccount.isDeleted());
        if (deletedAccount.getReserveType().isPresent()) {
            oldAccountBuilder.reserveType(deletedAccount.getReserveType().get());
        }
        AccountHistoryModel oldAccount = oldAccountBuilder.build();
        AccountHistoryModel.Builder newAccountBuilder = AccountHistoryModel.builder()
                .version(deletedAccount.getVersion() + 1L)
                .deleted(true);
        if (deletedAccount.getReserveType().isPresent()) {
            newAccountBuilder.reserveType(null);
        }
        AccountHistoryModel newAccount = newAccountBuilder.build();
        oldAccounts.computeIfAbsent(deletedAccount.getFolderId(), k -> new HashMap<>())
                .computeIfAbsent(new FolderHistoryGroupKey(null, null),
                        k -> new HashMap<>()).put(deletedAccount.getId(), oldAccount);
        newAccounts.computeIfAbsent(deletedAccount.getFolderId(), k -> new HashMap<>())
                .computeIfAbsent(new FolderHistoryGroupKey(null, null),
                        k -> new HashMap<>()).put(deletedAccount.getId(), newAccount);
    }

    @SuppressWarnings("ParameterNumber")
    public static boolean checkOperationsOrder(AccountModel existingAccount,
                                               ExternalAccount updatedAccount,
                                               QuotasAndOperations quotasAndOperations,
                                               boolean accountAndProvisionsVersionedTogether,
                                               boolean accountVersionedSeparately,
                                               boolean provisionsVersionedSeparately,
                                               boolean accountOperationsSupported,
                                               boolean provisionOperationsSupported,
                                               boolean accountNameSupported,
                                               boolean accountKeySupported) {
        boolean updatedAccountVersioned = updatedAccount.getAccountVersion().isPresent();
        boolean existingAccountVersioned = existingAccount.getLastReceivedVersion().isPresent();
        if (accountAndProvisionsVersionedTogether && updatedAccountVersioned && existingAccountVersioned) {
            return false;
        }
        boolean checkAccountsOrder = !(accountVersionedSeparately && updatedAccountVersioned
                && existingAccountVersioned);
        if (checkAccountsOrder && accountOperationsSupported) {
            Optional<AccountsQuotasOperationsModel> existingAccountLatestOperation = existingAccount
                    .getLatestSuccessfulAccountOperationId().flatMap(id -> Optional.ofNullable(quotasAndOperations
                            .getOperationsById().get(id)));
            Optional<AccountsQuotasOperationsModel> updatedAccountLatestOperation = updatedAccount.getLastUpdate()
                    .flatMap(ExternalLastUpdate::getOperationId).flatMap(id -> Optional.ofNullable(quotasAndOperations
                            .getOperationsById().get(id)));
            if (existingAccountLatestOperation.isPresent() && updatedAccountLatestOperation.isPresent()) {
                long latestExistingOrder = getLatestOrder(existingAccountLatestOperation.get().getOrders());
                long latestUpdatedOrder = getLatestOrder(updatedAccountLatestOperation.get().getOrders());
                if (isAccountChanged(existingAccount, updatedAccount, quotasAndOperations, accountNameSupported,
                        accountAndProvisionsVersionedTogether, accountVersionedSeparately, accountOperationsSupported,
                        accountKeySupported) && latestExistingOrder > latestUpdatedOrder) {
                    LOG.info("Stale operation order {} for account {}, current order is {}, skipping sync",
                            latestUpdatedOrder, updatedAccount.getAccountId(), latestExistingOrder);
                    return true;
                }
            }
        }
        if (!provisionOperationsSupported) {
            return false;
        }
        Map<String, AccountsQuotasModel> provisions = quotasAndOperations.getProvisionsByAccountResource()
                .getOrDefault(existingAccount.getId(), Collections.emptyMap());
        for (ExternalProvision updatedProvision : updatedAccount.getProvisions()) {
            AccountsQuotasModel existingProvision = provisions.get(updatedProvision.getResource()
                    .getResource().getId());
            boolean updatedProvisionVersioned = updatedProvision.getQuotaVersion().isPresent();
            boolean existingProvisionVersioned = existingProvision != null
                    && existingProvision.getLastReceivedProvisionVersion().isPresent();
            boolean checkProvisionsOrder = !(provisionsVersionedSeparately && updatedProvisionVersioned
                    && existingProvisionVersioned);
            if (checkProvisionsOrder && existingProvision != null) {
                Optional<AccountsQuotasOperationsModel> existingProvisionLatestOperation = existingProvision
                        .getLatestSuccessfulProvisionOperationId().flatMap(id ->
                                Optional.ofNullable(quotasAndOperations.getOperationsById().get(id)));
                Optional<AccountsQuotasOperationsModel> updatedProvisionLatestOperation = updatedProvision
                        .getLastUpdate().flatMap(ExternalLastUpdate::getOperationId).flatMap(id ->
                                Optional.ofNullable(quotasAndOperations.getOperationsById().get(id)));
                if (existingProvisionLatestOperation.isPresent() && updatedProvisionLatestOperation.isPresent()) {
                    long latestExistingOrder = getLatestOrder(existingProvisionLatestOperation.get().getOrders());
                    long latestUpdatedOrder = getLatestOrder(updatedProvisionLatestOperation.get().getOrders());
                    if (isProvisionChanged(existingProvision, updatedProvision, quotasAndOperations,
                            provisionOperationsSupported, provisionsVersionedSeparately)
                            && latestExistingOrder > latestUpdatedOrder) {
                        LOG.info("Stale operation order {} for provision of resource {} in account {}, " +
                                        "current order is {}, skipping sync",
                                latestUpdatedOrder,
                                updatedProvision.getResource().getResource().getKey(),
                                updatedAccount.getAccountId(),
                                latestExistingOrder);
                        return true;
                    }
                }
            }
        }
        return false;
    }

    @SuppressWarnings("ParameterNumber")
    public static boolean checkCoolDownPeriod(AccountModel existingAccount,
                                              ExternalAccount updatedAccount,
                                              QuotasAndOperations quotasAndOperations,
                                              boolean accountAndProvisionsVersionedTogether,
                                              boolean accountVersionedSeparately,
                                              boolean provisionsVersionedSeparately,
                                              boolean accountOperationsSupported,
                                              boolean provisionOperationsSupported,
                                              boolean isAccountToDelete,
                                              boolean softDeleteSupported,
                                              boolean accountNameSupported,
                                              boolean accountKeySupported,
                                              Instant now,
                                              ProviderModel provider,
                                              ProvidersSyncStatusModel syncStatus) {
        boolean coolDownDisabled = provider.getAccountsSettings().isSyncCoolDownDisabled();
        if (updatedAccount == null) {
            if (isAccountToDelete && !softDeleteSupported) {
                boolean coolDown = isCoolDown(existingAccount.getLastAccountUpdate(), syncStatus);
                if (coolDown && !coolDownDisabled) {
                    LOG.info("Account {} is not synced due to active cool down period",
                            existingAccount.getOuterAccountIdInProvider());
                    return true;
                }
            }
            return false;
        }
        boolean updatedAccountVersioned = updatedAccount.getAccountVersion().isPresent();
        boolean existingAccountVersioned = existingAccount.getLastReceivedVersion().isPresent();
        boolean updatedAccountOperationKnown = updatedAccount.getLastUpdate().isPresent()
                && updatedAccount.getLastUpdate().get().getOperationId().isPresent()
                && quotasAndOperations.getOperationsById().containsKey(updatedAccount.getLastUpdate().get()
                .getOperationId().get());
        boolean existingAccountOperationKnown = existingAccount.getLatestSuccessfulAccountOperationId().isPresent();
        if (accountAndProvisionsVersionedTogether && updatedAccountVersioned && existingAccountVersioned) {
            return false;
        }
        boolean checkAccountCoolDown = !((accountVersionedSeparately && updatedAccountVersioned
                && existingAccountVersioned) || (accountOperationsSupported && updatedAccountOperationKnown
                && existingAccountOperationKnown));
        if (checkAccountCoolDown) {
            boolean coolDown = isAccountChanged(existingAccount, updatedAccount, quotasAndOperations,
                    accountNameSupported, accountAndProvisionsVersionedTogether, accountVersionedSeparately,
                    accountOperationsSupported, accountKeySupported)
                    && isCoolDown(existingAccount.getLastAccountUpdate(), syncStatus);
            if (coolDown && !coolDownDisabled) {
                LOG.info("Account {} is not synced due to active cool down period", updatedAccount.getAccountId());
                return true;
            }
        }
        Map<String, AccountsQuotasModel> provisions = quotasAndOperations.getProvisionsByAccountResource()
                .getOrDefault(existingAccount.getId(), Collections.emptyMap());
        for (ExternalProvision updatedProvision : updatedAccount.getProvisions()) {
            AccountsQuotasModel existingProvision = provisions.get(updatedProvision.getResource()
                    .getResource().getId());
            boolean updatedProvisionVersioned = updatedProvision.getQuotaVersion().isPresent();
            boolean existingProvisionVersioned = existingProvision != null
                    && existingProvision.getLastReceivedProvisionVersion().isPresent();
            boolean updatedProvisionOperationKnown = updatedProvision.getLastUpdate().isPresent()
                    && updatedProvision.getLastUpdate().get().getOperationId().isPresent()
                    && quotasAndOperations.getOperationsById().containsKey(updatedProvision.getLastUpdate().get()
                    .getOperationId().get());
            boolean existingProvisionOperationKnown = existingProvision != null && existingProvision
                    .getLatestSuccessfulProvisionOperationId().isPresent();
            boolean checkProvisionCoolDown = !((provisionsVersionedSeparately && updatedProvisionVersioned
                    && existingProvisionVersioned) || (provisionOperationsSupported && updatedProvisionOperationKnown
                    && existingProvisionOperationKnown));
            if (existingProvision != null && checkProvisionCoolDown) {
                boolean coolDown = isProvisionChanged(existingProvision, updatedProvision, quotasAndOperations,
                        provisionOperationsSupported, provisionsVersionedSeparately)
                        && isCoolDown(existingProvision.getLastProvisionUpdate(), syncStatus);
                if (coolDown && !coolDownDisabled) {
                    LOG.info("Account {} is not synced due to active cool down period for provisions",
                            updatedAccount.getAccountId());
                    return true;
                }
            }
        }
        Set<String> updatedResourceIds = updatedAccount.getProvisions().stream()
                .map(p -> p.getResource().getResource().getId()).collect(Collectors.toSet());
        for (Map.Entry<String, AccountsQuotasModel> entry : provisions.entrySet()) {
            if (updatedResourceIds.contains(entry.getKey())) {
                continue;
            }
            boolean coolDown = isCoolDown(entry.getValue().getLastProvisionUpdate(), syncStatus);
            if (coolDown && !coolDownDisabled) {
                LOG.info("Account {} is not synced due to active cool down period for removed provisions",
                        updatedAccount.getAccountId());
                return true;
            }
        }
        return false;
    }

    @SuppressWarnings("ParameterNumber")
    public static boolean checkStaleVersions(AccountModel existingAccount,
                                             ExternalAccount updatedAccount,
                                             QuotasAndOperations quotasAndOperations,
                                             boolean accountAndProvisionsVersionedTogether,
                                             boolean accountVersionedSeparately,
                                             boolean provisionsVersionedSeparately,
                                             boolean accountNameSupported,
                                             boolean accountOperationsSupported,
                                             boolean accountKeySupported,
                                             boolean provisionOperationsSupported) {
        if (accountAndProvisionsVersionedTogether && updatedAccount.getAccountVersion().isPresent()
                && existingAccount.getLastReceivedVersion().isPresent()) {
            if (updatedAccount.getAccountVersion().get() < existingAccount.getLastReceivedVersion().get()) {
                LOG.info("Stale version {} for account {} and it's provisions, current version is {}, skipping sync",
                        updatedAccount.getAccountVersion().get(), updatedAccount.getAccountId(),
                        existingAccount.getLastReceivedVersion().get());
                return true;
            }
            if (updatedAccount.getAccountVersion().get().equals(existingAccount.getLastReceivedVersion().get())
                    && (isAccountChanged(existingAccount, updatedAccount, quotasAndOperations, accountNameSupported,
                    accountAndProvisionsVersionedTogether, accountVersionedSeparately, accountOperationsSupported,
                    accountKeySupported)
                    || isProvisionsChanged(updatedAccount, existingAccount, quotasAndOperations,
                    provisionOperationsSupported, provisionsVersionedSeparately))) {
                LOG.info("Stale version {} for account {} and it's provisions, current version is {}, skipping sync",
                        updatedAccount.getAccountVersion().get(), updatedAccount.getAccountId(),
                        existingAccount.getLastReceivedVersion().get());
                return true;
            }
        }
        if (accountVersionedSeparately && updatedAccount.getAccountVersion().isPresent()
                && existingAccount.getLastReceivedVersion().isPresent()
                && isAccountChanged(existingAccount, updatedAccount, quotasAndOperations, accountNameSupported,
                accountAndProvisionsVersionedTogether, accountVersionedSeparately, accountOperationsSupported,
                accountKeySupported)) {
            if (updatedAccount.getAccountVersion().get() < existingAccount.getLastReceivedVersion().get()) {
                LOG.info("Stale version {} for account {}, current version is {}, skipping sync",
                        updatedAccount.getAccountVersion().get(), updatedAccount.getAccountId(),
                        existingAccount.getLastReceivedVersion().get());
                return true;
            }
            if (updatedAccount.getAccountVersion().get().equals(existingAccount.getLastReceivedVersion().get())
                    && isAccountChanged(existingAccount, updatedAccount, quotasAndOperations, accountNameSupported,
                    accountAndProvisionsVersionedTogether, accountVersionedSeparately, accountOperationsSupported,
                    accountKeySupported)) {
                LOG.info("Stale version {} for account {}, current version is {}, skipping sync",
                        updatedAccount.getAccountVersion().get(), updatedAccount.getAccountId(),
                        existingAccount.getLastReceivedVersion().get());
                return true;
            }
        }
        if (provisionsVersionedSeparately) {
            Map<String, AccountsQuotasModel> provisions = quotasAndOperations.getProvisionsByAccountResource()
                    .getOrDefault(existingAccount.getId(), Collections.emptyMap());
            for (ExternalProvision updatedProvision : updatedAccount.getProvisions()) {
                AccountsQuotasModel existingProvision = provisions.get(updatedProvision.getResource()
                        .getResource().getId());
                if (existingProvision != null && updatedProvision.getQuotaVersion().isPresent()
                        && existingProvision.getLastReceivedProvisionVersion().isPresent()
                        && isProvisionChanged(existingProvision, updatedProvision, quotasAndOperations,
                        provisionOperationsSupported, provisionsVersionedSeparately)) {
                    if (updatedProvision.getQuotaVersion().get()
                            < existingProvision.getLastReceivedProvisionVersion().get()) {
                        LOG.info("Stale version {} for provision of resource {} in account {}, " +
                                        "current version is {}, skipping sync",
                                updatedProvision.getQuotaVersion().get(),
                                updatedProvision.getResource().getResource().getKey(),
                                updatedAccount.getAccountId(),
                                existingProvision.getLastReceivedProvisionVersion().get());
                        return true;
                    }
                    if (updatedProvision.getQuotaVersion().get()
                            .equals(existingProvision.getLastReceivedProvisionVersion().get())
                            && isProvisionChanged(existingProvision, updatedProvision, quotasAndOperations,
                            provisionOperationsSupported, provisionsVersionedSeparately)) {
                        LOG.info("Stale version {} for provision of resource {} in account {}, " +
                                        "current version is {}, skipping sync",
                                updatedProvision.getQuotaVersion().get(),
                                updatedProvision.getResource().getResource().getKey(),
                                updatedAccount.getAccountId(),
                                existingProvision.getLastReceivedProvisionVersion().get());
                        return true;
                    }
                }
            }
        }
        return false;
    }

    public static boolean isProvisionsChanged(ExternalAccount updatedAccount, AccountModel existingAccount,
                                              QuotasAndOperations quotasAndOperations,
                                              boolean provisionOperationsSupported,
                                              boolean provisionsVersionedSeparately) {
        Map<String, AccountsQuotasModel> existingProvisions = quotasAndOperations.getProvisionsByAccountResource()
                .getOrDefault(existingAccount.getId(), Collections.emptyMap());
        boolean hasChanges = updatedAccount.getProvisions().stream().anyMatch(updatedProvision -> {
            AccountsQuotasModel existingProvision = existingProvisions.get(updatedProvision.getResource()
                    .getResource().getId());
            if (existingProvision == null && updatedProvision.getProvided() != 0L) {
                return true;
            } else if (existingProvision == null) {
                return false;
            }
            return isProvisionChanged(existingProvision, updatedProvision, quotasAndOperations,
                    provisionOperationsSupported, provisionsVersionedSeparately);
        });
        if (hasChanges) {
            return true;
        }
        Map<String, ExternalProvision> updatedProvisionsByResourceId = updatedAccount.getProvisions().stream()
                .collect(Collectors.toMap(v -> v.getResource().getResource().getId(), Function.identity()));
        return existingProvisions.entrySet().stream()
                .anyMatch(e -> e.getValue().getProvidedQuota() != null && e.getValue().getProvidedQuota() != 0L
                        && !updatedProvisionsByResourceId.containsKey(e.getKey()));
    }

    @SuppressWarnings("ParameterNumber")
    public static boolean isAccountChanged(AccountModel currentAccount, ExternalAccount externalAccount,
                                           QuotasAndOperations quotasAndOperations, boolean accountNameSupported,
                                           boolean accountAndProvisionsVersionedTogether,
                                           boolean accountVersionedSeparately, boolean accountOperationsSupported,
                                           boolean accountKeySupported) {
        return (accountKeySupported
                && !Objects.equal(currentAccount.getOuterAccountKeyInProvider(), externalAccount.getKey()))
                || !Objects.equal(currentAccount.getFolderId(), externalAccount.getFolderId())
                || (accountNameSupported
                        && !Objects.equal(currentAccount.getDisplayName(), externalAccount.getDisplayName()))
                || currentAccount.isDeleted() != externalAccount.isDeleted()
                || (accountOperationsSupported && !Objects.equal(currentAccount.getLatestSuccessfulAccountOperationId(),
                externalAccount.getLastUpdate().flatMap(ExternalLastUpdate::getOperationId)
                        .filter(v -> quotasAndOperations.getOperationsById().containsKey(v))))
                || ((accountAndProvisionsVersionedTogether || accountVersionedSeparately)
                    && !Objects.equal(currentAccount.getLastReceivedVersion(), externalAccount.getAccountVersion()));
    }

    public static void refreshNewestLastUpdates(FolderHistoryGroupKey groupKey,
                                                Supplier<Optional<ExternalLastUpdate>> lastUpdateSupplier,
                                                Map<FolderHistoryGroupKey, Instant> newestLastUpdates) {
        Optional<Instant> lastUpdate = lastUpdateSupplier.get().flatMap(ExternalLastUpdate::getTimestamp);
        if (lastUpdate.isEmpty()) {
            return;
        }
        if (!newestLastUpdates.containsKey(groupKey)) {
            newestLastUpdates.put(groupKey, lastUpdate.get());
        }
        Instant currentNewestLastUpdate = newestLastUpdates.get(groupKey);
        if (lastUpdate.get().compareTo(currentNewestLastUpdate) > 0) {
            newestLastUpdates.put(groupKey, lastUpdate.get());
        }
    }

    public static void handleOperation(String operationId,
                                       Map<String, AccountsQuotasOperationsModel.Builder> operationsToUpsert,
                                       Map<String, AccountsQuotasOperationsModel> operationsById,
                                       Instant now,
                                       String accountId,
                                       String folderId) {
        if (operationId == null) {
            return;
        }
        if (operationsToUpsert.containsKey(operationId)) {
            AccountsQuotasOperationsModel.Builder operation = operationsToUpsert.get(operationId);
            if (operation.getRequestStatus().isPresent()
                    && !AccountsQuotasOperationsModel.RequestStatus.WAITING.equals(operation.getRequestStatus().get())
                    && !AccountsQuotasOperationsModel.RequestStatus.OK.equals(operation.getRequestStatus().get())
                    && isMatchingOperation(() -> operation.getOperationType().get(),
                    () -> operation.getRequestedChanges().get(), accountId, folderId)) {
                operationsToUpsert.put(operationId, operation
                        .setUpdateDateTime(now)
                        .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                        .setErrorKind(null));
            }
        } else if (operationsById.containsKey(operationId)) {
            AccountsQuotasOperationsModel operation = operationsById.get(operationId);
            if (operation.getRequestStatus().isPresent()
                    && !AccountsQuotasOperationsModel.RequestStatus.WAITING.equals(operation.getRequestStatus().get())
                    && !AccountsQuotasOperationsModel.RequestStatus.OK.equals(operation.getRequestStatus().get())
                    && isMatchingOperation(operation::getOperationType,
                    operation::getRequestedChanges, accountId, folderId)) {
                operationsToUpsert.put(operationId, new AccountsQuotasOperationsModel.Builder(operation)
                        .setUpdateDateTime(now)
                        .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                        .setErrorKind(null));
            }
        }
    }

    public static Optional<String> getAccountOperationId(ExternalAccount account,
                                                         Map<String, AccountsQuotasOperationsModel> operationsById,
                                                         boolean accountOperationsSupported,
                                                         String folderId,
                                                         String accountId) {
        if (!accountOperationsSupported) {
            return Optional.empty();
        }
        return account.getLastUpdate().flatMap(ExternalLastUpdate::getOperationId)
                .filter(operationsById::containsKey).filter(operationId -> {
                    AccountsQuotasOperationsModel operation = operationsById.get(operationId);
                    return isMatchingOperation(operation::getOperationType, operation::getRequestedChanges,
                            accountId, folderId);
                });
    }

    public static Optional<String> getProvisionUpdateOperationId(
            ExternalProvision provision,
            Map<String, AccountsQuotasOperationsModel> operationsById,
            boolean provisionOperationsSupported,
            String folderId,
            String accountId) {
        if (!provisionOperationsSupported) {
            return Optional.empty();
        }
        return provision.getLastUpdate().flatMap(ExternalLastUpdate::getOperationId)
                .filter(operationsById::containsKey).filter(operationId -> {
                    AccountsQuotasOperationsModel operation = operationsById.get(operationId);
                    return isMatchingOperation(operation::getOperationType, operation::getRequestedChanges,
                            accountId, folderId);
                });
    }

    public static Set<FolderOperationAuthor> collectAuthors(ProviderModel provider,
                                                            Supplier<Optional<ExternalLastUpdate>> lastUpdateSupplier,
                                                            GroupedAuthors authors) {
        Set<FolderOperationAuthor> result = new HashSet<>();
        lastUpdateSupplier.get().flatMap(ExternalLastUpdate::getAuthor).ifPresent(userId -> {
            if (userId.getPassportUid().isPresent()
                    && authors.getAuthorsByUid().containsKey(userId.getPassportUid().get())) {
                UserModel user = authors.getAuthorsByUid().get(userId.getPassportUid().get());
                result.add(new FolderOperationAuthor(user.getId(), userId.getPassportUid().get(), null));
            } else if (userId.getStaffLogin().isPresent()
                    && authors.getAuthorsByLogin().containsKey(userId.getStaffLogin().get())) {
                UserModel user = authors.getAuthorsByLogin().get(userId.getStaffLogin().get());
                result.add(new FolderOperationAuthor(user.getId(), user.getPassportUid().orElse(null), null));
            } else if (userId.getPassportUid().isPresent()) {
                result.add(new FolderOperationAuthor(null, userId.getPassportUid().get(), null));
            } else {
                result.add(new FolderOperationAuthor(null, null, provider.getId()));
            }
        });
        return result;
    }

    @SuppressWarnings("ParameterNumber")
    public static boolean checkInProgressOperations(String externalAccountId,
                                                    ExternalAccount account,
                                                    boolean accountOperationsSupported,
                                                    boolean provisionOperationsSupported,
                                                    Set<String> operationsInProgressIds,
                                                    Map<String, AccountsQuotasOperationsModel> operationsById,
                                                    String folderId,
                                                    boolean newAccounts,
                                                    AccountModel existingAccount) {
        Set<String> collectedOperationIds = new HashSet<>();
        Map<String, AccountsQuotasOperationsModel> collectedOperations = new HashMap<>();
        collectOperations(account, accountOperationsSupported, provisionOperationsSupported,
                operationsById, collectedOperationIds, collectedOperations);
        if (!Sets.intersection(collectedOperationIds, operationsInProgressIds).isEmpty()) {
            LOG.info("Account creation operation (one of {}) is still in progress for account {}, " +
                    "skipping sync", collectedOperationIds, externalAccountId);
            return true;
        }
        boolean isInProgress = collectedOperations.values().stream().anyMatch(operation ->
                operation.getRequestStatus().isEmpty()
                        || AccountsQuotasOperationsModel.RequestStatus.WAITING
                        .equals(operation.getRequestStatus().get()));
        if (isInProgress) {
            LOG.info("At least one quota update operation (one of {}) is still in progress for " +
                    "account {}, skipping sync", collectedOperationIds, externalAccountId);
            return true;
        }
        List<AccountsQuotasOperationsModel> operationsInProgress = operationsInProgressIds.stream()
                .flatMap(id -> Optional.ofNullable(operationsById.get(id)).stream()).collect(Collectors.toList());
        if (newAccounts) {
            boolean folderAccountCreationInProgress = operationsInProgress.stream().anyMatch(op -> {
                boolean newAccountOp = AccountsQuotasOperationsModel.OperationType.CREATE_ACCOUNT
                        .equals(op.getOperationType())
                        || AccountsQuotasOperationsModel.OperationType.CREATE_ACCOUNT_AND_PROVIDE
                        .equals(op.getOperationType());
                boolean matchingFolder = op.getRequestedChanges().getAccountCreateParams().isPresent()
                        && folderId.equals(op.getRequestedChanges().getAccountCreateParams().get().getFolderId());
                return newAccountOp && matchingFolder;
            });
            if (folderAccountCreationInProgress) {
                LOG.info("Account creation operation (one of {}) is still in progress for folder {}, skipping sync",
                        operationsInProgressIds, folderId);
                return true;
            }
        } else {
            boolean accountOperationInProgress = operationsInProgress.stream()
                    .anyMatch(op -> op.getRequestedChanges().getAccountId().isPresent()
                            && existingAccount.getId().equals(op.getRequestedChanges().getAccountId().get()));
            if (accountOperationInProgress) {
                LOG.info("At least one quota update operation (one of {}) is still in progress for " +
                                "account {}, skipping sync", operationsInProgressIds,
                        existingAccount.getOuterAccountIdInProvider());
                return true;
            }
        }
        return false;
    }

    @SuppressWarnings("ParameterNumber")
    public static AccountsQuotasModel newProvision(ProviderModel provider,
                                                   boolean provisionsVersionedSeparately,
                                                   boolean provisionOperationsSupported,
                                                   Instant now,
                                                   String folderId,
                                                   AccountModel newAccount,
                                                   ExternalProvision provision,
                                                   Map<String, AccountsQuotasOperationsModel> operationsById,
                                                   long providedAmount,
                                                   long allocatedAmount) {
        return new AccountsQuotasModel.Builder()
                .setAccountId(newAccount.getId())
                .setFolderId(folderId)
                .setProviderId(provider.getId())
                .setResourceId(provision.getResource().getResource().getId())
                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                .setProvidedQuota(providedAmount)
                .setAllocatedQuota(allocatedAmount)
                .setLastProvisionUpdate(now)
                .setLastReceivedProvisionVersion(provisionsVersionedSeparately
                        ? provision.getQuotaVersion().orElse(null) : null)
                .setLatestSuccessfulProvisionOperationId(getValidOperationId(provision::getLastUpdate,
                        provisionOperationsSupported, operationsById).orElse(null))
                .build();
    }

    public static Optional<String> getValidOperationId(Supplier<Optional<ExternalLastUpdate>> lastUpdateSupplier,
                                                       boolean operationsSupported,
                                                       Map<String, AccountsQuotasOperationsModel> operationsById) {
        if (!operationsSupported) {
            return Optional.empty();
        }
        Optional<String> suppliedOperationId = lastUpdateSupplier.get().flatMap(ExternalLastUpdate::getOperationId);
        if (suppliedOperationId.isEmpty()) {
            return Optional.empty();
        }
        if (!operationsById.containsKey(suppliedOperationId.get())) {
            return Optional.empty();
        }
        return suppliedOperationId;
    }

    @SuppressWarnings("ParameterNumber")
    public static AccountModel buildNewAccount(ProviderModel provider,
                                               AccountSpaceModel accountsSpace,
                                               Map<String, String> preGeneratedAccountIds,
                                               boolean accountAndProvisionsVersionedTogether,
                                               boolean accountVersionedSeparately,
                                               boolean accountOperationsSupported,
                                               boolean softDeleteSupported,
                                               boolean accountNameSupported,
                                               boolean accountKeySupported,
                                               Instant now,
                                               String folderId,
                                               String externalAccountId,
                                               ExternalAccount account,
                                               Map<String, AccountsQuotasOperationsModel> operationsById) {
        return new AccountModel.Builder()
                .setId(preGeneratedAccountIds.get(externalAccountId))
                .setDeleted(softDeleteSupported && account.isDeleted())
                .setAccountsSpacesId(accountsSpace != null ? accountsSpace.getId() : null)
                .setProviderId(provider.getId())
                .setFolderId(folderId)
                .setVersion(0L)
                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                .setDisplayName(accountNameSupported ? account.getDisplayName().orElse(null) : null)
                .setLastAccountUpdate(now)
                .setLastReceivedVersion(accountAndProvisionsVersionedTogether
                        || accountVersionedSeparately
                        ? account.getAccountVersion().orElse(null) : null)
                .setLatestSuccessfulAccountOperationId(getValidOperationId(account::getLastUpdate,
                        accountOperationsSupported, operationsById).orElse(null))
                .setOuterAccountIdInProvider(externalAccountId)
                .setOuterAccountKeyInProvider(accountKeySupported
                        ? account.getKey().orElse(null) : null)
                .build();
    }

    private static long getLatestOrder(OperationOrdersModel orders) {
        if (orders.getCloseOrder().isEmpty() && orders.getRestoreOrder().isEmpty()) {
            return orders.getSubmitOrder();
        }
        if (orders.getCloseOrder().isPresent() && orders.getRestoreOrder().isEmpty()) {
            return Math.max(orders.getCloseOrder().get(), orders.getSubmitOrder());
        }
        if (orders.getCloseOrder().isEmpty() && orders.getRestoreOrder().isPresent()) {
            return Math.max(orders.getRestoreOrder().get(), orders.getSubmitOrder());
        }
        return Math.max(orders.getSubmitOrder(), Math.max(orders.getCloseOrder().get(),
                orders.getRestoreOrder().get()));
    }

    private static boolean isProvisionChanged(AccountsQuotasModel currentProvision,
                                              ExternalProvision externalProvision,
                                              QuotasAndOperations quotasAndOperations,
                                              boolean provisionOperationsSupported,
                                              boolean provisionsVersionedSeparately) {
        Optional<Long> providedAmountO = Units.convertFromApi(externalProvision.getProvided(),
                externalProvision.getResource().getResource(), externalProvision.getResource().getUnitsEnsemble(),
                externalProvision.getProvidedUnit());
        return !Objects.equal(currentProvision.getProvidedQuota(), providedAmountO.orElse(null))
                || (provisionOperationsSupported &&
                        !Objects.equal(currentProvision.getLatestSuccessfulProvisionOperationId(),
                                externalProvision.getLastUpdate().flatMap(ExternalLastUpdate::getOperationId)
                                        .filter(v -> quotasAndOperations.getOperationsById().containsKey(v))))
                || (provisionsVersionedSeparately && !Objects.equal(currentProvision.getLastReceivedProvisionVersion(),
                externalProvision.getQuotaVersion()));
    }

    private static boolean isMatchingOperation(
            Supplier<AccountsQuotasOperationsModel.OperationType> operationTypeSupplier,
            Supplier<OperationChangesModel> operationChangesSupplier,
            String accountId,
            String folderId) {
        AccountsQuotasOperationsModel.OperationType operationType = operationTypeSupplier.get();
        OperationChangesModel operationChanges = operationChangesSupplier.get();
        if (AccountsQuotasOperationsModel.OperationType.CREATE_ACCOUNT.equals(operationType)
                || AccountsQuotasOperationsModel.OperationType.CREATE_ACCOUNT_AND_PROVIDE.equals(operationType)) {
            return accountId == null && operationChanges.getAccountCreateParams().isPresent()
                    && operationChanges.getAccountCreateParams().get().getFolderId().equals(folderId);
        } else if (AccountsQuotasOperationsModel.OperationType.MOVE_PROVISION.equals(operationType)) {
            return (operationChanges.getAccountId().isPresent()
                    && operationChanges.getAccountId().get().equals(accountId))
                    || (operationChanges.getDestinationAccountId().isPresent()
                    && operationChanges.getDestinationAccountId().get().equals(accountId));
        } else {
            return operationChanges.getAccountId().isPresent()
                    && operationChanges.getAccountId().get().equals(accountId);
        }
    }

    private static void collectOperations(ExternalAccount account,
                                          boolean accountOperationsSupported,
                                          boolean provisionOperationsSupported,
                                          Map<String, AccountsQuotasOperationsModel> operationsById,
                                          Set<String> collectedOperationIds,
                                          Map<String, AccountsQuotasOperationsModel> collectedOperations) {
        if (accountOperationsSupported && account.getLastUpdate().isPresent()
                && account.getLastUpdate().get().getOperationId().isPresent()) {
            String operationId = account.getLastUpdate().get().getOperationId().get();
            collectedOperationIds.add(operationId);
            if (operationsById.containsKey(operationId)) {
                collectedOperations.put(operationId, operationsById.get(operationId));
            }
        }
        if (provisionOperationsSupported) {
            account.getProvisions().forEach(provision -> {
                if (provision.getLastUpdate().isPresent()
                        && provision.getLastUpdate().get().getOperationId().isPresent()) {
                    String operationId = provision.getLastUpdate().get().getOperationId().get();
                    collectedOperationIds.add(operationId);
                    if (operationsById.containsKey(operationId)) {
                        collectedOperations.put(operationId, operationsById.get(operationId));
                    }
                }
            });
        }
    }

}
