package ru.yandex.intranet.d.services.sync;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.LastUpdateDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyResponseDto;
import ru.yandex.intranet.d.services.sync.model.AccumulatedAccounts;
import ru.yandex.intranet.d.services.sync.model.ExternalAccount;
import ru.yandex.intranet.d.services.sync.model.ExternalAccountsSpaceKey;
import ru.yandex.intranet.d.services.sync.model.ExternalCompoundResourceKey;
import ru.yandex.intranet.d.services.sync.model.ExternalLastUpdate;
import ru.yandex.intranet.d.services.sync.model.ExternalProvision;
import ru.yandex.intranet.d.services.sync.model.ExternalUserId;
import ru.yandex.intranet.d.services.sync.model.SyncResource;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.TypedError;

/**
 * Service to sync providers accounts and quotas, validation part.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class AccountsSyncValidationService {

    private static final Logger LOG = LoggerFactory.getLogger(AccountsSyncValidationService.class);

    private final MessageSource messages;
    private final AccountsSyncStoreService accountsSyncStoreService;

    public AccountsSyncValidationService(
            @Qualifier("messageSource") MessageSource messages,
            AccountsSyncStoreService accountsSyncStoreService
    ) {
        this.messages = messages;
        this.accountsSyncStoreService = accountsSyncStoreService;
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<AccumulatedAccounts> validateAccountsFolders(ProviderModel provider,
                                                             List<ExternalAccount> externalAccounts,
                                                             AccountsSpaceKeyRequestDto accountsSpaceKey,
                                                             List<FolderModel> targetFolders,
                                                             Set<String> accumulatedAccountIds,
                                                             String requestId,
                                                             Locale locale, ProvidersSyncStatusModel syncStatus) {
        Map<String, FolderModel> targetFoldersById = targetFolders.stream()
                .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
        ErrorCollection.Builder errors = ErrorCollection.builder();
        List<ExternalAccount> validExternalAccounts = new ArrayList<>();
        externalAccounts.forEach(externalAccount -> {
            String targetFolderId = externalAccount.getFolderId();
            if (!targetFoldersById.containsKey(targetFolderId)) {
                errors.addError("accounts." + externalAccount.getIndex() + ".folderId", TypedError.invalid(messages
                        .getMessage("errors.folder.not.found", null, locale)))
                .addDetail("accounts." + externalAccount.getIndex() + ".folderId", targetFolderId)
                .addDetail("accounts." + externalAccount.getIndex() + ".accountId", externalAccount.getAccountId());
            } else {
                validExternalAccounts.add(externalAccount);
            }
        });
        AccumulatedAccounts accounts = new AccumulatedAccounts(
                validExternalAccounts, accumulatedAccountIds, errors.hasAnyErrors());
        if (errors.hasAnyErrors()) {
            ErrorCollection errorCollection = errors.build();
            LOG.error("Some of provider {} responses for account space {} can not be processed: {}, requestId={}",
                    provider.getKey(), accountsSpaceKey, errorCollection, requestId);
            return accountsSyncStoreService.insertSyncError(
                    "Some of provider responses can not be processed", syncStatus, errorCollection,
                    makeDetails(accountsSpaceKey, requestId)
            ).map(u -> accounts);
        }
        return Mono.just(accounts);
    }

    private Map<String, String> makeDetails(
            @Nullable
            AccountsSpaceKeyRequestDto accountsSpaceKey,
            String requestId
    ) {
        Map<String, String> stringStringMap = new java.util.HashMap<>();
        stringStringMap.put("requestId", requestId);
        if (accountsSpaceKey != null) {
            stringStringMap.put("accountsSpaceKey", accountsSpaceKey.toString());
        }
        return stringStringMap;
    }

    public List<ExternalAccount> validateAccountsFields(ProviderModel provider,
                                                        AccountsSpaceKeyRequestDto accountsSpaceKey,
                                                        Map<ExternalCompoundResourceKey, SyncResource> externalIndex,
                                                        ListAccountsResponseDto accounts,
                                                        ErrorCollection.Builder errors,
                                                        Set<String> accumulatedAccountIds,
                                                        Locale locale) {
        if (accounts.getAccounts().isEmpty()) {
            return List.of();
        }
        List<ExternalAccount> result = new ArrayList<>();
        for (int i = 0; i < accounts.getAccounts().get().size(); i++) {
            AccountDto account = accounts.getAccounts().get().get(i);
            if (account == null) {
                errors.addError("accounts." + i, TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                continue;
            }
            boolean matchingAccountsSpace = checkAccountsSpace(accountsSpaceKey, account, errors, i, locale);
            boolean validAccountFields = true;
            if (account.getAccountId().isEmpty() || account.getAccountId().get().isBlank()) {
                errors.addError("accounts." + i + ".accountId", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                validAccountFields = false;
            } else if (matchingAccountsSpace) {
                accumulatedAccountIds.add(account.getAccountId().get());
            }
            if (provider.getAccountsSettings().isKeySupported() && (account.getKey().isEmpty()
                    || account.getKey().get().isBlank())) {
                errors.addError("accounts." + i + ".key", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                validAccountFields = false;
            }
            if (provider.getAccountsSettings().isDisplayNameSupported() && (account.getDisplayName().isEmpty()
                    || account.getDisplayName().get().isBlank())) {
                errors.addError("accounts." + i + ".displayName", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                validAccountFields = false;
            }
            if (account.getFolderId().isEmpty()) {
                errors.addError("accounts." + i + ".folderId", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                validAccountFields = false;
            } else if (!Uuids.isValidUuid(account.getFolderId().get())) {
                errors.addError("accounts." + i + ".folderId", TypedError.invalid(messages
                                .getMessage("errors.folder.not.found", null, locale)))
                        .addDetail("accounts." + i + ".folderId", account.getFolderId().get())
                        .addDetail("accounts." + i + ".accountId", account.getAccountId().orElse("not sent"));
                validAccountFields = false;
            }
            if (!provider.getAccountsSettings().isSoftDeleteSupported()
                    && account.getDeleted().isPresent() && account.getDeleted().get()) {
                errors.addError("accounts." + i + ".deleted", TypedError.invalid(messages
                        .getMessage("errors.account.deletion.is.not.supported", null, locale)));
                validAccountFields = false;
            }
            if (provider.getAccountsSettings().isPerAccountVersionSupported()
                    && account.getAccountVersion().isEmpty()) {
                errors.addError("accounts." + i + ".accountVersion", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                validAccountFields = false;
            }
            if (provider.getAccountsSettings().isPerAccountLastUpdateSupported()
                    && account.getLastUpdate().isEmpty()) {
                errors.addError("accounts." + i + ".lastUpdate", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                validAccountFields = false;
            }
            List<ExternalProvision> provisions = new ArrayList<>();
            boolean allProvisionsValid = true;
            if (matchingAccountsSpace) {
                allProvisionsValid = checkProvisions(provider, account, provisions, externalIndex, errors, i, locale);
            }
            if (allProvisionsValid && matchingAccountsSpace && validAccountFields) {
                result.add(new ExternalAccount(account.getAccountId().get(), account.getKey().orElse(null),
                        account.getDisplayName().orElse(null), account.getFolderId().get(),
                        account.getDeleted().orElse(false), provisions, account.getAccountVersion().orElse(null),
                        toLastUpdate(account::getLastUpdate).orElse(null), i, account.isFreeTier().orElse(false)));
            }
        }
        return result;
    }

    private boolean checkProvisions(ProviderModel provider, AccountDto account, List<ExternalProvision> provisions,
                                    Map<ExternalCompoundResourceKey, SyncResource> externalIndex,
                                    ErrorCollection.Builder errors, int accountIndex, Locale locale) {
        if (account.getProvisions().isEmpty()) {
            return true;
        }
        boolean allProvisionsValid = true;
        for (int i = 0; i < account.getProvisions().get().size(); i++) {
            boolean validProvisionFields = true;
            ProvisionDto provision = account.getProvisions().get().get(i);
            if (provision == null) {
                errors.addError("accounts." + accountIndex + ".provisions." + i, TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                allProvisionsValid = false;
                continue;
            }
            SyncResource targetResource = null;
            if (provision.getResourceKey().isEmpty()) {
                errors.addError("accounts." + accountIndex + ".provisions." + i + ".resourceKey",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                allProvisionsValid = false;
                validProvisionFields = false;
            } else {
                String errorKey = "accounts." + accountIndex + ".provisions." + i;
                Optional<SyncResource> resourceO = getTargetResource(
                        account, provision, externalIndex, errors, errorKey, locale
                );
                if (resourceO.isEmpty()) {
                    errors.addError(errorKey + ".resourceKey",
                            TypedError.invalid(messages.getMessage("errors.resource.not.found", null, locale)));
                    allProvisionsValid = false;
                    validProvisionFields = false;
                } else {
                    targetResource = resourceO.get();
                }
            }
            Long providedAmount = null;
            if (provision.getProvidedAmount().isEmpty()) {
                errors.addError("accounts." + accountIndex + ".provisions." + i + ".providedAmount",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                allProvisionsValid = false;
                validProvisionFields = false;
            } else if (provision.getProvidedAmount().get() < 0L) {
                errors.addError("accounts." + accountIndex + ".provisions." + i + ".providedAmount",
                        TypedError.invalid(messages.getMessage("errors.number.must.be.non.negative", null, locale)));
                allProvisionsValid = false;
                validProvisionFields = false;
            } else {
                providedAmount = provision.getProvidedAmount().get();
            }
            UnitModel providedUnit = null;
            if (provision.getProvidedAmountUnitKey().isEmpty()) {
                errors.addError("accounts." + accountIndex + ".provisions." + i + ".providedAmountUnitKey",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                allProvisionsValid = false;
                validProvisionFields = false;
            } else {
                Optional<UnitModel> unitO = getTargetUnit(provision.getProvidedAmountUnitKey().get(), targetResource);
                if (unitO.isEmpty()) {
                    errors.addError("accounts." + accountIndex + ".provisions." + i + ".providedAmountUnitKey",
                            TypedError.invalid(messages.getMessage("errors.unit.not.found", null, locale)));
                    allProvisionsValid = false;
                    validProvisionFields = false;
                } else {
                    providedUnit = unitO.get();
                }
            }
            Long allocatedAmount = null;
            if (provision.getAllocatedAmount().isEmpty()) {
                errors.addError("accounts." + accountIndex + ".provisions." + i + ".allocatedAmount",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                allProvisionsValid = false;
                validProvisionFields = false;
            } else if (provision.getAllocatedAmount().get() < 0) {
                errors.addError("accounts." + accountIndex + ".provisions." + i + ".allocatedAmount",
                        TypedError.invalid(messages.getMessage("errors.number.must.be.non.negative", null, locale)));
                allProvisionsValid = false;
                validProvisionFields = false;
            } else {
                allocatedAmount = provision.getAllocatedAmount().get();
            }
            UnitModel allocatedUnit = null;
            if (provision.getAllocatedAmountUnitKey().isEmpty()) {
                errors.addError("accounts." + accountIndex + ".provisions." + i + ".allocatedAmountUnitKey",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                allProvisionsValid = false;
                validProvisionFields = false;
            } else {
                Optional<UnitModel> unitO = getTargetUnit(provision.getAllocatedAmountUnitKey().get(), targetResource);
                if (unitO.isEmpty()) {
                    String field = "accounts." + accountIndex + ".provisions." + i + ".allocatedAmountUnitKey";
                    errors
                            .addError(field, TypedError.invalid(
                                    messages.getMessage("errors.unit.not.found", null, locale)))
                            .addDetail(field, Map.of(
                                    "resource_id", Optional.ofNullable(targetResource)
                                            .map(r -> r.getResource().getId()),
                                    "unit_key", provision.getAllocatedAmountUnitKey().get()
                            ));
                    allProvisionsValid = false;
                    validProvisionFields = false;
                } else {
                    allocatedUnit = unitO.get();
                }
            }
            if (provider.getAccountsSettings().isPerProvisionLastUpdateSupported()
                    && provision.getLastUpdate().isEmpty()) {
                errors.addError("accounts." + accountIndex + ".provisions." + i + ".lastUpdate",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                allProvisionsValid = false;
                validProvisionFields = false;
            }
            if (provider.getAccountsSettings().isPerProvisionVersionSupported()
                    && provision.getQuotaVersion().isEmpty()) {
                errors.addError("accounts." + accountIndex + ".provisions." + i + ".quotaVersion",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                allProvisionsValid = false;
                validProvisionFields = false;
            }
            if (validProvisionFields) {
                provisions.add(new ExternalProvision(targetResource, providedAmount, providedUnit, allocatedAmount,
                        allocatedUnit, toLastUpdate(provision::getLastUpdate).orElse(null),
                        provision.getQuotaVersion().orElse(null), i));
            } else {
                errors.addDetail("accounts." + accountIndex + ".provisions." + i + ".data", provision);
            }
        }
        if (!allProvisionsValid) {
            errors.addDetail("accounts." + accountIndex + ".identity", Map.of(
                    "accountId", account.getAccountId(),
                    "key", account.getKey(),
                    "folderId", account.getFolderId(),
                    "accountVersion", account.getAccountVersion(),
                    "accountsSpaceKey", account.getAccountsSpaceKey(),
                    "lastUpdate", account.getLastUpdate()
            ));
        }
        return allProvisionsValid;
    }

    private Optional<SyncResource> getTargetResource(AccountDto account, ProvisionDto provision,
                                                     Map<ExternalCompoundResourceKey, SyncResource> externalIndex,
                                                     ErrorCollection.Builder errors, String errorKey, Locale locale) {
        if (provision.getResourceKey().isEmpty()) {
            errors.addError(errorKey + ".resourceKey",
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
            return Optional.empty();
        }
        if (provision.getResourceKey().get().getResourceTypeKey().isEmpty()) {
            errors.addError(errorKey + ".resourceKey.resourceTypeKey",
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
            return Optional.empty();
        }
        String resourceTypeKey = provision.getResourceKey().get().getResourceTypeKey().get();
        Set<SegmentKeyResponseDto> segments = new HashSet<>();
        if (account.getAccountsSpaceKey().isPresent()
                && account.getAccountsSpaceKey().get().getSegmentation().isPresent()) {
            segments.addAll(account.getAccountsSpaceKey().get().getSegmentation().get());
        }
        if (provision.getResourceKey().get().getSegmentation().isPresent()) {
            segments.addAll(provision.getResourceKey().get().getSegmentation().get());
        }
        boolean badSegments = segments.stream().anyMatch(segment -> segment == null
                || segment.getSegmentKey().isEmpty()
                || segment.getSegmentationKey().isEmpty()
                || segment.getSegmentKey().get().isBlank()
                || segment.getSegmentationKey().get().isBlank());
        if (badSegments) {
            errors.addError(errorKey + ".resourceKey.segmentation",
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
            return Optional.empty();
        }
        Map<String, List<String>> segmentsByKey = segments
                .stream().collect(Collectors.groupingBy(v -> v.getSegmentationKey().get(),
                        Collectors.mapping(v -> v.getSegmentKey().get(), Collectors.toList())));
        boolean duplicateSegments = segmentsByKey.values().stream().anyMatch(v -> v.size() > 1);
        if (duplicateSegments) {
            errors.addError(errorKey + ".resourceKey.segmentation.segmentationKey",
                    TypedError.invalid(messages.getMessage("errors.duplicate.keys.are.forbidden", null, locale)));
            return Optional.empty();
        }
        Set<ExternalCompoundResourceKey.ExternalCompoundSegmentKey> segmentKeys = segments.stream()
                .map(s -> new ExternalCompoundResourceKey.ExternalCompoundSegmentKey(s.getSegmentationKey().get(),
                        s.getSegmentKey().get())).collect(Collectors.toSet());
        ExternalCompoundResourceKey externalKey = new ExternalCompoundResourceKey(resourceTypeKey, segmentKeys);
        if (!externalIndex.containsKey(externalKey)) {
            return Optional.empty();
        }
        return Optional.of(externalIndex.get(externalKey));
    }

    private Optional<UnitModel> getTargetUnit(String unitKey, SyncResource resource) {
        if (resource == null) {
            return Optional.empty();
        }
        return resource.getUnitsEnsemble().unitByKey(unitKey);
    }

    private Optional<ExternalLastUpdate> toLastUpdate(Supplier<Optional<LastUpdateDto>> supplier) {
        Optional<LastUpdateDto> lastUpdate = supplier.get();
        if (lastUpdate.isEmpty()) {
            return Optional.empty();
        }
        ExternalUserId author = lastUpdate.get().getAuthor()
                .map(user -> new ExternalUserId(user.getPassportUid().orElse(null),
                        user.getStaffLogin().orElse(null))).orElse(null);
        String operationId = lastUpdate.get().getOperationId().isPresent()
                && !lastUpdate.get().getOperationId().get().isBlank() ? lastUpdate.get().getOperationId().get() : null;
        return Optional.of(new ExternalLastUpdate(lastUpdate.get().getTimestamp()
                .map(Instant::ofEpochMilli).orElse(null), author, operationId));
    }

    private boolean checkAccountsSpace(AccountsSpaceKeyRequestDto requestAccountsSpace, AccountDto account,
                                       ErrorCollection.Builder errors, int accountIndex, Locale locale) {
        if ((requestAccountsSpace == null) != account.getAccountsSpaceKey().isEmpty()) {
            errors.addError("accounts." + accountIndex + ".accountsSpaceKey", TypedError.invalid(messages
                    .getMessage("errors.accounts.space.mismatch", null, locale)));
            return false;
        }
        if (account.getAccountsSpaceKey().isPresent()) {
            if (account.getAccountsSpaceKey().get().getSegmentation().isEmpty()
                    || account.getAccountsSpaceKey().get().getSegmentation().get().isEmpty()) {
                errors.addError("accounts." + accountIndex + ".accountsSpaceKey", TypedError.invalid(messages
                        .getMessage("errors.accounts.space.mismatch", null, locale)));
                return false;
            }
            boolean badSegments = account.getAccountsSpaceKey().get().getSegmentation().get().stream()
                    .anyMatch(segment -> segment == null || segment.getSegmentKey().isEmpty()
                            || segment.getSegmentationKey().isEmpty()
                            || segment.getSegmentKey().get().isBlank()
                            || segment.getSegmentationKey().get().isBlank());
            if (badSegments) {
                errors.addError("accounts." + accountIndex + ".accountsSpaceKey", TypedError.invalid(messages
                        .getMessage("errors.accounts.space.mismatch", null, locale)));
                return false;
            }
            Map<String, List<String>> segmentsByKey = account.getAccountsSpaceKey().get().getSegmentation().get()
                    .stream().collect(Collectors.groupingBy(v -> v.getSegmentationKey().get(),
                            Collectors.mapping(v -> v.getSegmentKey().get(), Collectors.toList())));
            boolean duplicateSegments = segmentsByKey.values().stream().anyMatch(v -> v.size() > 1);
            if (duplicateSegments) {
                errors.addError("accounts." + accountIndex + ".accountsSpaceKey", TypedError.invalid(messages
                        .getMessage("errors.accounts.space.mismatch", null, locale)));
                return false;
            }
        }
        if (requestAccountsSpace != null && account.getAccountsSpaceKey().isPresent()) {
            ExternalAccountsSpaceKey requestKey = new ExternalAccountsSpaceKey(requestAccountsSpace.getSegmentation()
                    .orElse(Collections.emptyList()).stream()
                    .map(v -> new ExternalAccountsSpaceKey.ExternalSegmentKey(v.getSegmentationKey(),
                            v.getSegmentKey())).collect(Collectors.toSet()));
            ExternalAccountsSpaceKey responseKey = new ExternalAccountsSpaceKey(account.getAccountsSpaceKey().get()
                    .getSegmentation().orElse(Collections.emptyList()).stream()
                    .map(v -> new ExternalAccountsSpaceKey.ExternalSegmentKey(v.getSegmentationKey().get(),
                            v.getSegmentKey().get())).collect(Collectors.toSet()));
            if (!requestKey.equals(responseKey)) {
                errors.addError("accounts." + accountIndex + ".accountsSpaceKey", TypedError.invalid(messages
                        .getMessage("errors.accounts.space.mismatch", null, locale)));
                return false;
            }
        }
        return true;
    }

}
