package ru.yandex.intranet.d.services.sync.model;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import com.google.common.collect.Streams;

/**
 * Accumulated accounts.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class AccumulatedAccounts {

    private final List<ExternalAccount> accountsToSync;
    private final Set<String> allAccountIds;
    private final boolean hasAnyErrors;

    public AccumulatedAccounts(List<ExternalAccount> accountsToSync, Set<String> allAccountIds, boolean hasAnyErrors) {
        this.accountsToSync = accountsToSync;
        this.allAccountIds = allAccountIds;
        this.hasAnyErrors = hasAnyErrors;
    }

    public List<ExternalAccount> getAccountsToSync() {
        return accountsToSync;
    }

    public Set<String> getAllAccountIds() {
        return allAccountIds;
    }

    public boolean hasAnyErrors() {
        return hasAnyErrors;
    }

    public AccumulatedAccounts concat(AccumulatedAccounts r) {
        return new AccumulatedAccounts(
                Streams.concat(accountsToSync.stream(), r.getAccountsToSync().stream()).collect(Collectors.toList()),
                Sets.union(allAccountIds, r.getAllAccountIds()),
                hasAnyErrors || r.hasAnyErrors
        );
    }

    public AccumulatedAccounts mergeHasAnyErrors(boolean hasAnyErrors) {
        return new AccumulatedAccounts(accountsToSync, allAccountIds, this.hasAnyErrors || hasAnyErrors);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccumulatedAccounts that = (AccumulatedAccounts) o;
        return hasAnyErrors == that.hasAnyErrors &&
                Objects.equals(accountsToSync, that.accountsToSync) &&
                Objects.equals(allAccountIds, that.allAccountIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(accountsToSync, allAccountIds, hasAnyErrors);
    }

    @Override
    public String toString() {
        return "AccumulatedAccounts{" +
                "accountsToSync=" + accountsToSync +
                ", allAccountIds=" + allAccountIds +
                ", hasAnyErrors=" + hasAnyErrors +
                '}';
    }

}
