package ru.yandex.intranet.d.services.tracker;

import java.time.Duration;
import java.util.Collections;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import io.netty.channel.ChannelOption;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.reactive.function.client.ClientRequest;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.ExchangeFilterFunctions;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import org.springframework.web.util.UriComponentsBuilder;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;
import reactor.netty.http.client.HttpClient;
import reactor.util.retry.Retry;
import reactor.util.retry.RetrySpec;

import ru.yandex.intranet.d.util.ResolverHolder;
import ru.yandex.intranet.d.util.http.YaReactorClientHttpConnector;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.tracker.TrackerCreateTicketDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerCreateTicketResponseDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerErrorDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerTransitionExecuteDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerUpdateTicketDto;

/**
 * Tracker client
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class TrackerClientImpl implements TrackerClient {
    private static final Logger LOG = LoggerFactory.getLogger(TrackerClientImpl.class);

    private static final long RESPONSE_SIZE_LIMIT = 10485760L;

    private final long maxAttempts;
    private final long minBackoffMs;
    private final long timeoutMs;
    private final String baseUrl;
    private final String token;
    private final String userAgent;
    private final WebClient client;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public TrackerClientImpl(@Value("${tracker.client.connectTimeoutMs}") int connectTimeoutMillis,
                         @Value("${tracker.client.readTimeoutMs}") long readTimeoutMillis,
                         @Value("${tracker.client.writeTimeoutMs}") long writeTimeoutMillis,
                         @Value("${tracker.client.maxAttempts}") long maxAttempts,
                         @Value("${tracker.client.minBackoffMs}") long minBackoffMs,
                         @Value("${tracker.client.baseUrl}") String baseUrl,
                         @Value("${tracker.client.token}") String token,
                         @Value("${http.client.userAgent}") String userAgent) {
        this.maxAttempts = maxAttempts;
        this.minBackoffMs = minBackoffMs;
        this.timeoutMs = readTimeoutMillis;
        this.baseUrl = baseUrl;
        this.token = token;
        this.userAgent = userAgent;
        this.client = createWebClient(connectTimeoutMillis, readTimeoutMillis, writeTimeoutMillis);
    }

    @Override
    public Mono<Result<TrackerCreateTicketResponseDto>> createTicket(TrackerCreateTicketDto request) {
        String uri = UriComponentsBuilder.fromHttpUrl(baseUrl).pathSegment("v2", "issues").toUriString();
        return client.post()
                .uri(uri)
                .header("Authorization", "OAuth " + token)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .header("X-Request-ID", UUID.randomUUID().toString())
                .accept(MediaType.APPLICATION_JSON)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> {
                    if (response.rawStatusCode() == HttpStatus.CONFLICT.value()) {
                        String key = response.headers().asHttpHeaders().getFirst("X-Ticket-Key");
                        if (key != null) {
                            return Mono.just(Result.success(new TrackerCreateTicketResponseDto(key)));
                        }
                    }
                    return toResult(response, TrackerCreateTicketResponseDto.class, uri);
                })
                .timeout(Duration.ofMillis(timeoutMs))
                .retryWhen(retryRequest());
    }

    @Override
    public Mono<Result<Void>> updateTicket(String ticketKey, TrackerUpdateTicketDto request) {
        String uri = UriComponentsBuilder.fromHttpUrl(baseUrl).pathSegment("v2", "issues", ticketKey).toUriString();
        return client.put()
                .uri(uri)
                .header("Authorization", "OAuth " + token)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .header("X-Request-ID", UUID.randomUUID().toString())
                .accept(MediaType.APPLICATION_JSON)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResult(response, uri))
                .timeout(Duration.ofMillis(timeoutMs))
                .retryWhen(retryRequest());
    }

    @Override
    public Mono<Result<Void>> closeTicket(String ticketKey, String transition,
                                          TrackerTransitionExecuteDto request) {
        String uri = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("v2", "issues", ticketKey, "transitions", transition, "_execute").toUriString();
        return client.post()
                .uri(uri)
                .header("Authorization", "OAuth " + token)
                .header(HttpHeaders.USER_AGENT, userAgent)
                .header("X-Request-ID", UUID.randomUUID().toString())
                .accept(MediaType.APPLICATION_JSON)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(request)
                .exchangeToMono(response -> toResult(response, uri))
                .timeout(Duration.ofMillis(timeoutMs))
                .retryWhen(retryRequest());
    }

    private <T> Mono<Result<T>> toResult(ClientResponse clientResponse, Class<T> responseClass, String uri) {
        return isSuccess(clientResponse) ?
                clientResponse.bodyToMono(responseClass).map(Result::success) :
                toError(clientResponse, uri);
    }

    private Mono<Result<Void>> toResult(ClientResponse clientResponse, String uri) {
        return isSuccess(clientResponse) ?
                clientResponse.toEntity(Void.class).map(u -> Result.success(null)) :
                toError(clientResponse, uri);
    }

    private <T> Mono<Result<T>> toError(ClientResponse clientResponse, String uri) {
        return clientResponse.bodyToMono(TrackerErrorDto.class).map(
                error -> {
                    String message = String.format(
                            "Received %s error code for %s tracker API endpoint. ErrorMessages: %s. Errors: %s.",
                            clientResponse.rawStatusCode(), uri, error.getErrorMessages(), error.getErrors());
                    LOG.error(message);
                    return Result.failure(ErrorCollection.builder().addError(TypedError.badRequest(message)).build());
                });
    }

    private boolean isSuccess(ClientResponse clientResponse) {
        int statusCode = clientResponse.rawStatusCode();
        return statusCode >= 200 && statusCode < 300;
    }

    private Retry retryRequest() {
        return RetrySpec.backoff(maxAttempts, Duration.ofMillis(minBackoffMs)).filter(e -> {
            if (e instanceof WebClientResponseException) {
                WebClientResponseException webException = (WebClientResponseException) e;
                return isRetryableCode(webException.getRawStatusCode());
            }
            return !Exceptions.isRetryExhausted(e);
        });
    }

    private static WebClient createWebClient(int connectTimeoutMillis,
                                             long readTimeoutMillis,
                                             long writeTimeoutMillis) {
        HttpClient httpClient = HttpClient.create()
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectTimeoutMillis)
                .resolver(ResolverHolder.RESOLVER_INSTANCE)
                .doOnConnected(connection -> {
                    connection.addHandlerLast(new ReadTimeoutHandler(readTimeoutMillis, TimeUnit.MILLISECONDS));
                    connection.addHandlerLast(new WriteTimeoutHandler(writeTimeoutMillis, TimeUnit.MILLISECONDS));
                });
        return WebClient.builder()
                .clientConnector(new YaReactorClientHttpConnector(httpClient))
                .filter(ExchangeFilterFunctions.limitResponseSize(RESPONSE_SIZE_LIMIT))
                .filter((request, next) -> next.exchange(request)
                        .flatMap(clientResponse -> {
                            if (isRetryableCode(clientResponse.statusCode().value())) {
                                return Mono.just(ClientRequest
                                        .from(request)
                                        .headers(headers -> headers.replace("X-Request-ID",
                                                Collections.singletonList(UUID.randomUUID().toString())))
                                        .build())
                                        .flatMap(next::exchange);
                            } else {
                                return Mono.just(clientResponse);
                            }
                        }))
                .codecs(configurer -> configurer
                        .defaultCodecs()
                        .maxInMemorySize(Math.toIntExact(RESPONSE_SIZE_LIMIT)))
                .build();
    }

    private static boolean isRetryableCode(int statusCode) {
        return statusCode == HttpStatus.TOO_MANY_REQUESTS.value()
                || statusCode == HttpStatus.INTERNAL_SERVER_ERROR.value()
                || statusCode == HttpStatus.BAD_GATEWAY.value()
                || statusCode == HttpStatus.SERVICE_UNAVAILABLE.value()
                || statusCode == HttpStatus.GATEWAY_TIMEOUT.value();
    }
}
