package ru.yandex.intranet.d.services.tracker;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.web.model.tracker.TrackerCreateTicketDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerCreateTicketResponseDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerTransitionExecuteDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerUpdateTicketDto;

/**
 * Tracker client for tests
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
@Profile({"local", "dev", "integration-tests"})
@Primary
public class TrackerClientStub implements TrackerClient {
    private static final Logger LOG = LoggerFactory.getLogger(TrackerClientStub.class);

    private final Map<String, AtomicLong> countTicketInQueue = new ConcurrentHashMap<>();
    private final Map<String, Ticket> ticketByKey = new ConcurrentHashMap<>();

    @Override
    public Mono<Result<TrackerCreateTicketResponseDto>> createTicket(TrackerCreateTicketDto request) {
        countTicketInQueue.putIfAbsent(request.getQueue(), new AtomicLong(0));
        long tickerNumber = countTicketInQueue.get(request.getQueue()).incrementAndGet();
        String ticketKey = request.getQueue() + "-" + tickerNumber;
        Ticket ticket = new Ticket(ticketKey, request.getQueue(), request.getSummary(),
                request.getDescription(), request.getCreatedBy(), request.getComponents(), request.getTags(),
                request.getUnique(), TicketStatus.OPENED, null, null);
        ticketByKey.put(ticketKey, ticket);
        LOG.info("Create ticket {}: {}", ticketKey, request);
        return Mono.just(Result.success(new TrackerCreateTicketResponseDto(ticketKey)));
    }

    @Override
    public Mono<Result<Void>> updateTicket(String ticketKey, TrackerUpdateTicketDto request) {
        ticketByKey.computeIfPresent(ticketKey, (key, ticket) -> new Ticket(key, ticket.getQueue(),
                Objects.requireNonNullElse(request.getSummary(), ticket.getSummary()),
                Objects.requireNonNullElse(request.getDescription(), ticket.getDescription()),
                ticket.getCreatedBy(),
                Objects.requireNonNullElse(request.getComponents(), ticket.getComponents()),
                Objects.requireNonNullElse(request.getTags(), ticket.getTags()),
                ticket.getUnique(), ticket.getStatus(), ticket.getLastTransition(), ticket.getResolution()));
        LOG.info("Update ticket {}: {}", ticketKey, request);
        return Mono.just(Result.success(null));
    }

    @Override
    public Mono<Result<Void>> closeTicket(String ticketKey, String transition, TrackerTransitionExecuteDto request) {
        ticketByKey.computeIfPresent(ticketKey, (key, ticket) -> new Ticket(key, ticket.getQueue(), ticket.getSummary(),
                ticket.getDescription(), ticket.getCreatedBy(), ticket.getComponents(), ticket.getTags(),
                ticket.getUnique(), TicketStatus.CLOSED, transition, request.getResolution()));
        LOG.info("Close ticket {}. Transition - {}, resolution - {}", ticketKey, transition, request.getResolution());
        return Mono.just(Result.success(null));
    }

    @Nullable
    public Ticket getTicket(String key) {
        return ticketByKey.get(key);
    }

    public static class Ticket {
        private final String key;
        private final String queue;
        private final String summary;
        private final String description;
        private final String createdBy;
        private final List<Long> components;
        private final List<String> tags;
        private final String unique;
        private final TicketStatus status;
        @Nullable
        private final String lastTransition;
        @Nullable
        private final String resolution;

        @SuppressWarnings("ParameterNumber")
        public Ticket(String key, String queue, String summary, String description, String createdBy,
                      List<Long> components, List<String> tags, String unique, TicketStatus status,
                      @Nullable String lastTransition, @Nullable String resolution) {
            this.key = key;
            this.queue = queue;
            this.summary = summary;
            this.description = description;
            this.createdBy = createdBy;
            this.components = components;
            this.tags = tags;
            this.unique = unique;
            this.status = status;
            this.lastTransition = lastTransition;
            this.resolution = resolution;
        }

        public String getKey() {
            return key;
        }

        public String getQueue() {
            return queue;
        }

        public String getSummary() {
            return summary;
        }

        public String getDescription() {
            return description;
        }

        public String getCreatedBy() {
            return createdBy;
        }

        public List<Long> getComponents() {
            return components;
        }

        public List<String> getTags() {
            return tags;
        }

        public String getUnique() {
            return unique;
        }

        public TicketStatus getStatus() {
            return status;
        }

        @Nullable
        public String getLastTransition() {
            return lastTransition;
        }

        @Nullable
        public String getResolution() {
            return resolution;
        }

        @Override
        public String toString() {
            return "Ticket{" +
                    "key='" + key + '\'' +
                    ", queue='" + queue + '\'' +
                    ", summary='" + summary + '\'' +
                    ", description='" + description + '\'' +
                    ", createdBy='" + createdBy + '\'' +
                    ", components=" + components +
                    ", tags=" + tags +
                    ", unique='" + unique + '\'' +
                    ", status=" + status +
                    ", lastTransition='" + lastTransition + '\'' +
                    ", resolution='" + resolution + '\'' +
                    '}';
        }
    }

    public enum TicketStatus {
        OPENED,
        CLOSED,
    }
}
