package ru.yandex.intranet.d.services.transfer;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogCommentDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao;
import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogCommentModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceBaseIdentity;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.validators.AbcServiceValidator;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.ResultTx;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.quotas.QuotaTransferInputDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * TransferQuotaService
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@Component
public class BulkQuotaMoveService {
    private static final int MAX_COMMENT_LENGTH = 100_000;

    private final MessageSource messages;
    private final QuotasDao quotasDao;
    private final ResourcesLoader resourcesLoader;
    private final ResourceTypesLoader resourceTypesLoader;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final FolderDao folderDao;
    private final SecurityManagerService securityManagerService;
    private final YdbTableClient tableClient;
    private final ResourcesDao resourcesDao;
    private final ResourceSegmentationsDao resourceSegmentationsDao;
    private final ResourceSegmentsDao resourceSegmentsDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final FolderOperationLogCommentDao folderOperationLogCommentDao;
    private final AbcServiceValidator abcServiceValidator;

    @SuppressWarnings("ParameterNumber")
    public BulkQuotaMoveService(@Qualifier("messageSource") MessageSource messages,
                                QuotasDao quotasDao,
                                ResourcesLoader resourcesLoader,
                                ResourceTypesLoader resourceTypesLoader,
                                UnitsEnsemblesLoader unitsEnsemblesLoader,
                                FolderDao folderDao,
                                SecurityManagerService securityManagerService,
                                YdbTableClient tableClient,
                                ResourcesDao resourcesDao,
                                ResourceSegmentationsDao resourceSegmentationsDao,
                                ResourceSegmentsDao resourceSegmentsDao,
                                FolderOperationLogDao folderOperationLogDao,
                                FolderOperationLogCommentDao folderOperationLogCommentDao,
                                AbcServiceValidator abcServiceValidator) {
        this.messages = messages;
        this.quotasDao = quotasDao;
        this.resourcesLoader = resourcesLoader;
        this.resourceTypesLoader = resourceTypesLoader;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.folderDao = folderDao;
        this.securityManagerService = securityManagerService;
        this.tableClient = tableClient;
        this.resourcesDao = resourcesDao;
        this.resourceSegmentationsDao = resourceSegmentationsDao;
        this.resourceSegmentsDao = resourceSegmentsDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.folderOperationLogCommentDao = folderOperationLogCommentDao;
        this.abcServiceValidator = abcServiceValidator;
    }


    public Mono<Result<FolderOperationLogCommentModel>> moveQuotas(QuotaTransferInputDto body,
                                                                   YaUserDetails currentUser,
                                                                   Locale locale) {
        return validateBody(body, locale)
                .andThenMono(b -> moveQuotasWithValidatedBody(b, currentUser, locale));
    }

    private Mono<Result<FolderOperationLogCommentModel>> moveQuotasWithValidatedBody(QuotaTransferInputDto body,
                                                                                     YaUserDetails currentUser,
                                                                                     Locale locale) {

        TenantId tenantId = Tenants.getTenantId(currentUser);

        Set<String> providerIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        Set<Tuple2<String, String>> resourceIds = new HashSet<>();

        Map<QuotaTransferInputDto.ExternalResourceId, Tuple2<Tuple2<String, String>, TenantId>>
                typeByExternalResource = new HashMap<>();

        Map<QuotaTransferInputDto.ExternalResourceId, Set<Tuple2<ResourceSegmentationModel.ProviderKey, String>>>
                segmentsByExternalResource = new HashMap<>();

        Multimap<QuotaTransferInputDto.ExternalResourceId, String> unitByExternalResource = HashMultimap.create();
        Multimap<String, String> unitByInternalResource = HashMultimap.create();

        for (QuotaTransferInputDto.Transfer transfer : body.getTransfers()) {
            folderIds.add(transfer.getSourceFolderId());
            folderIds.add(transfer.getDestinationFolderId());

            QuotaTransferInputDto.Resource resource = transfer.getResource();

            providerIds.add(resource.getProviderId());

            if (resource.getResourceId() != null) {
                resourceIds.add(Tuples.of(resource.getProviderId(), resource.getResourceId()));
                unitByInternalResource.put(resource.getResourceId(), transfer.getAmountUnitKey());
            } else {
                QuotaTransferInputDto.ExternalResourceId externalResourceId = resource.getExternalResourceId();
                if (externalResourceId != null) {
                    String typeKey = externalResourceId.getTypeKey();

                    Set<Tuple2<ResourceSegmentationModel.ProviderKey, String>> segments = new HashSet<>();
                    for (QuotaTransferInputDto.Segment segment : externalResourceId.getSegmentation()) {
                        ResourceSegmentationModel.ProviderKey key = new ResourceSegmentationModel.ProviderKey(
                                resource.getProviderId(), segment.getSegmentationKey());

                        segments.add(Tuples.of(key, segment.getSegmentKey()));
                    }

                    typeByExternalResource.put(externalResourceId,
                            Tuples.of(Tuples.of(resource.getProviderId(), typeKey), tenantId));
                    segmentsByExternalResource.put(externalResourceId, segments);
                    unitByExternalResource.put(externalResourceId, transfer.getAmountUnitKey());
                }
            }
        }

        return validatePermissions(currentUser, locale, providerIds)
                .flatMap(r -> r.andThenMono(b ->
                        validateFolderIds(tenantId, folderIds, locale)))
                .flatMap(r2 -> r2.andThenMono(fs ->
                        validateService(body.getTransfers(), fs, locale)))
                .flatMap(r3 -> r3.andThenMono(u ->
                        validateResourceIds(resourceIds, tenantId, locale)))
                .flatMap(r4 -> r4.andThenMono(intResources ->
                        validateExternalResourceTypeIds(typeByExternalResource, locale)
                                .flatMap(r5 -> r5.andThenMono(externalResourceTypes ->
                                        validateSegmentations(tenantId, segmentsByExternalResource, locale)
                                                .flatMap(r6 -> r6.andThenMono(externalSegmentKey ->
                                                                validateSegments(tenantId, externalSegmentKey, locale))
                                                        .flatMap(r7 -> r7.andThenMono(externalSegment ->
                                                                        validateExternalResources(externalResourceTypes,
                                                                                externalSegment, locale))
                                                                .map(r8 -> r8.apply(extResources ->
                                                                        Tuples.of(intResources, extResources)))
                                                        ))))))
                .flatMap(r -> r.andThenMono(resources ->
                        validateUnits(resources.getT1(), resources.getT2(),
                                unitByInternalResource, unitByExternalResource, locale)
                                .map(r2 -> r2.andThen(units ->
                                        toQuotaDiffs(body.getTransfers(), resources.getT1(), resources.getT2(), units,
                                                locale)
                                ))))
                .flatMap(r -> r.andThenMono(qd -> makeTransfers(qd, tenantId, currentUser, body.getComment(), locale)));
    }

    private Mono<Result<Void>> validatePermissions(YaUserDetails currentUser, Locale locale, Set<String> providerIds) {
        return Flux.fromIterable(providerIds)
                .flatMap(pid -> securityManagerService.checkWritePermissionsForProvider(pid, currentUser, locale))
                .reduce(ErrorCollection.builder(), (errors, r) -> {
                    r.doOnFailure(errors::add);
                    return errors;
                }).map(errors -> errors.hasAnyErrors() ? Result.failure(errors.build()) : Result.success(null));
    }

    private Result<QuotaTransferInputDto> validateBody(QuotaTransferInputDto body, Locale locale) {
        if (body.getTransfers() == null || body.getTransfers().isEmpty()) {
            return Result.failure(ErrorCollection.builder()
                    .addError("transfers", TypedError.invalid(messages.getMessage(
                            "errors.field.is.required", null, locale)))
                    .build());
        }
        if (body.getTransfers().size() > 1000) {
            return Result.failure(ErrorCollection.builder()
                    .addError("transfers", TypedError.invalid(messages.getMessage(
                            "errors.transfers.limit.exceeded", null, locale)))
                    .build());
        }

        boolean hasInvalidResources = body.getTransfers().stream()
                .map(QuotaTransferInputDto.Transfer::getResource)
                .anyMatch(r -> (r == null) || (r.getResourceId() == null && r.getExternalResourceId() == null) ||
                        (r.getResourceId() != null && r.getExternalResourceId() != null));

        if (hasInvalidResources) {
            return Result.failure(ErrorCollection.builder()
                    .addError(TypedError.invalid(messages.getMessage(
                            "errors.transfers.invalid.resource", null, locale)))
                    .build());
        }

        boolean hasTransfersWithoutRequiredFields = body.getTransfers().stream()
                .anyMatch(t -> t.getDestinationFolderId() == null
                        || t.getSourceFolderId() == null
                        || t.getResource().getProviderId() == null);

        if (hasTransfersWithoutRequiredFields) {
            return Result.failure(ErrorCollection.builder()
                    .addError(TypedError.invalid(messages.getMessage(
                            "errors.transfers.missing.transfer.fields", null, locale)))
                    .build());
        }

        if (StringUtils.isEmpty(body.getComment())) {
            return Result.failure(ErrorCollection.builder()
                    .addError("comment", TypedError.invalid(messages.getMessage(
                            "errors.field.is.required", null, locale)))
                    .build());
        }
        if (body.getComment().length() > MAX_COMMENT_LENGTH) {
            return Result.failure(ErrorCollection.builder()
                    .addError("comment", TypedError.invalid(messages.getMessage(
                            "errors.text.is.too.long", null, locale)))
                    .build());
        }

        return Result.success(body);
    }

    private Mono<Result<FolderOperationLogCommentModel>> makeTransfers(List<QuotaDiff> diffs,
                                                                       TenantId tenantId,
                                                                       YaUserDetails currentUser,
                                                                       String comment,
                                                                       Locale locale) {

        List<WithTenant<QuotaModel.Key>> keys = diffs.stream()
                .flatMap(diff -> Stream.of(diff.getSourceKey(), diff.getDestKey()))
                .map(k -> new WithTenant<>(tenantId, k))
                .collect(Collectors.toList());

        List<String> folderIds = keys.stream()
                .map(k -> k.getIdentity().getFolderId())
                .distinct()
                .collect(Collectors.toList());

        FolderOperationLogCommentModel commentModel =
                prepareComment(tenantId, currentUser.getUser().orElseThrow(), comment);
        String commentId = commentModel.getId();

        return tableClient.usingSessionMonoRetryable(session ->
                session.usingCompResultTxRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                        ts -> folderOperationLogCommentDao.getByIdStartTx(ts, commentId, tenantId)
                                .map(tx -> ResultTx.success(tx.get(), tx.getTransactionId())),
                        (ts, createdComment) -> createdComment.map(c -> Mono.just(Result.success(c)))
                                .orElseGet(() ->
                                        folderDao.getByIds(ts, folderIds, tenantId)
                                                .flatMap(folders -> quotasDao.getByKeys(ts, keys)
                                                        .flatMap(quotas -> upsertQuotasAndOperation(diffs, tenantId,
                                                                currentUser, locale, commentModel, commentId, ts,
                                                                folders, quotas)))
                                )
                                .map(r -> new WithTxId<>(r, ts.getId())),
                        (ts, b) -> ts.commitTransaction().thenReturn(new WithTxId<>(b, null))
                )).map(r -> r.andThen(Function.identity()));
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<Result<FolderOperationLogCommentModel>>
    upsertQuotasAndOperation(List<QuotaDiff> diffs, TenantId tenantId,
                             YaUserDetails currentUser, Locale locale,
                             FolderOperationLogCommentModel commentModel,
                             String commentId, YdbTxSession ts,
                             List<FolderModel> folders,
                             List<QuotaModel> quotas) {
        return getUpdatedQuotas(diffs, tenantId, quotas, locale)
                .applyMono(newQuotas -> quotasDao.upsertAllRetryable(ts, newQuotas)
                        .thenReturn(getOpLogsAndFolders(folders, quotas, diffs, currentUser, commentId))
                        .flatMap(opLogsAndFolders ->
                                folderOperationLogDao.upsertAllRetryable(ts, opLogsAndFolders.getT1())
                                        .then(folderDao.upsertAllRetryable(ts, opLogsAndFolders.getT2())))
                        .then(folderOperationLogCommentDao.upsertOneRetryable(ts, commentModel))
                );
    }

    @NotNull
    private FolderOperationLogCommentModel prepareComment(TenantId tenantId, UserModel user, String comment) {
        return FolderOperationLogCommentModel.builder()
                .setId(UUID.randomUUID().toString())
                .setTenantId(tenantId)
                .setCreatedAt(Instant.now())
                .setCreatedBy(user.getId())
                .setText(comment)
                .build();
    }

    private Tuple2<List<FolderOperationLogModel>, List<FolderModel>> getOpLogsAndFolders(List<FolderModel> folders,
                                                                                         List<QuotaModel> oldQuotas,
                                                                                         List<QuotaDiff> diffs,
                                                                                         YaUserDetails currentUser,
                                                                                         String commentId) {
        Map<String, Long> nextOpByFolderId = folders.stream()
                .collect(Collectors.toMap(FolderModel::getId, FolderModel::getNextOpLogOrder));

        Map<QuotaModel.Key, Long> quotaByKey = oldQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::toKey, QuotaModel::getQuota));
        Map<QuotaModel.Key, Long> balanceByKey = oldQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::toKey, QuotaModel::getBalance));

        TenantId tenantId = Tenants.getTenantId(currentUser);
        UserModel author = currentUser.getUser().orElseThrow();

        Map<Tuple2<String, String>, List<QuotaDiff>> diffsBySourceDestination = new HashMap<>();
        diffs.forEach(diff -> {
            String sourceFolderId = diff.getSourceKey().getFolderId();
            String destinationFolderId = diff.getDestKey().getFolderId();
            diffsBySourceDestination.computeIfAbsent(Tuples.of(sourceFolderId,
                    destinationFolderId), k -> new ArrayList<>()).add(diff);
            diffsBySourceDestination.computeIfAbsent(Tuples.of(destinationFolderId,
                    sourceFolderId), k -> new ArrayList<>()).add(diff);
        });

        Map<Set<String>, QuotaDiff> diffBySourceDestinationFoldersId = new HashMap<>();

        List<FolderOperationLogModel> folderOperationLogs = new ArrayList<>();

        diffsBySourceDestination.forEach((k, v) -> {
            Map<String, Long> oldQuotasLog = new HashMap<>();
            Map<String, Long> oldBalanceLog = new HashMap<>();

            String first = k.getT1();
            String second = k.getT2();

            Map<String, Long> newQuotasLog = new HashMap<>();
            Map<String, Long> newBalanceLog = new HashMap<>();

            v.forEach(diff -> {
                QuotaModel.Key key = diff.getDestKey().getFolderId().equals(first) ? diff.getDestKey() :
                        diff.getSourceKey();

                long amount = diff.getDestKey().equals(key) ? diff.getAmount() : -diff.getAmount();

                Long quotaByKeyOrDefault = quotaByKey.getOrDefault(key, 0L);
                Long balanceByKeyOrDefault = balanceByKey.getOrDefault(key, 0L);
                oldQuotasLog.put(key.getResourceId(), quotaByKeyOrDefault);
                oldBalanceLog.put(key.getResourceId(), balanceByKeyOrDefault);

                long newQuota = quotaByKeyOrDefault + amount;
                long newBalance = balanceByKeyOrDefault + amount;
                newQuotasLog.put(key.getResourceId(), newQuota);
                newBalanceLog.put(key.getResourceId(), newBalance);

                quotaByKey.put(key, newQuota);
                balanceByKey.put(key, newBalance);
            });

            Long opOrder = nextOpByFolderId.get(first);
            QuotaDiff diff = diffBySourceDestinationFoldersId.computeIfAbsent(
                    Set.of(first, second), key -> v.get(0));

            FolderOperationLogModel folderLog = getFolderLog(tenantId, first,
                    opOrder,
                    diff,
                    oldQuotasLog,
                    oldBalanceLog,
                    newQuotasLog,
                    newBalanceLog,
                    author,
                    commentId
            );

            folderOperationLogs.add(folderLog);
            nextOpByFolderId.put(first, opOrder + 1);
        });

        List<FolderModel> updatedFolders = folders.stream()
                .map(folderModel -> folderModel.toBuilder()
                        .setNextOpLogOrder(nextOpByFolderId.get(folderModel.getId()))
                        .build())
                .collect(Collectors.toList());

        return Tuples.of(folderOperationLogs, updatedFolders);
    }

    @SuppressWarnings("ParameterNumber")
    private FolderOperationLogModel getFolderLog(TenantId tenantId,
                                                 String folderId,
                                                 Long nextOpLogOrder,
                                                 QuotaDiff diff,
                                                 Map<String, Long> oldQuotasLog,
                                                 Map<String, Long> oldBalanceLog,
                                                 Map<String, Long> newQuotasLog,
                                                 Map<String, Long> newBalanceLog,
                                                 UserModel author, String commentId) {
        return FolderOperationLogModel.builder()
                .setTenantId(tenantId)
                .setFolderId(folderId)
                .setOperationDateTime(Instant.now())
                .setId(UUID.randomUUID().toString())
                .setProviderRequestId(null)
                .setOperationType(FolderOperationType.BULK_QUOTA_MOVE)
                .setAuthorUserId(author.getId())
                .setAuthorUserUid(author.getPassportUid().orElse(null))
                .setSourceFolderOperationsLogId(diff.getSourceLogId())
                .setDestinationFolderOperationsLogId(diff.getDestLogId())
                .setOldFolderFields(null)
                .setNewFolderFields(null)
                .setOldProvisions(new QuotasByAccount(Map.of()))
                .setNewProvisions(new QuotasByAccount(Map.of()))
                .setOldAccounts(null)
                .setNewAccounts(null)
                .setActuallyAppliedProvisions(null)
                .setAccountsQuotasOperationsId(null)
                .setQuotasDemandsId(null)
                .setOperationPhase(null)
                .setOldQuotas(new QuotasByResource(oldQuotasLog))
                .setNewQuotas(new QuotasByResource(newQuotasLog))
                .setOldBalance(new QuotasByResource(oldBalanceLog))
                .setNewBalance(new QuotasByResource(newBalanceLog))
                .setOrder(nextOpLogOrder)
                .setCommentId(commentId)
                .build();
    }

    private Result<List<QuotaModel>> getUpdatedQuotas(List<QuotaDiff> diffs,
                                                      TenantId tenantId,
                                                      List<QuotaModel> quotas,
                                                      Locale locale) {
        Map<QuotaModel.Key, QuotaModel> quotasByKey = quotas.stream()
                .collect(Collectors.toMap(QuotaModel::toKey, Function.identity()));

        Map<QuotaModel.Key, Long> diffByQuotaKey = new HashMap<>();

        ErrorCollection.Builder errorBuilder = ErrorCollection.builder();

        for (QuotaDiff diff : diffs) {
            List<Tuple2<QuotaModel.Key, Long>> diffByKey = List.of(
                    Tuples.of(diff.getSourceKey(), -diff.getAmount()),
                    Tuples.of(diff.getDestKey(), diff.getAmount())
            );
            for (Tuple2<QuotaModel.Key, Long> keyAndDiff : diffByKey) {
                QuotaModel.Key key = keyAndDiff.getT1();
                Long initDiff = diffByQuotaKey.getOrDefault(key, 0L);
                Optional<Long> resultDiff = Units.add(initDiff, keyAndDiff.getT2());
                if (resultDiff.isEmpty()) {
                    errorBuilder.addError(TypedError.invalid(messages.getMessage(
                            "errors.transfers.amount.overflow", new Object[]{key.getFolderId(), key.getProviderId(),
                                    key.getResourceId(), diff.getAmount()}, locale)));
                    break;
                }
                diffByQuotaKey.put(key, resultDiff.get());
            }
        }

        diffByQuotaKey.forEach((key, value) -> validateChange(errorBuilder, key, quotasByKey.get(key), value, locale));

        if (errorBuilder.hasAnyErrors()) {
            return Result.failure(errorBuilder.build());
        }

        List<QuotaModel> updatedModels = diffByQuotaKey.entrySet().stream()
                .map(e -> {
                    QuotaModel quotaModel = quotasByKey.get(e.getKey());
                    if (quotaModel != null) {
                        return QuotaModel.builder(quotaModel)
                                .quota(quotaModel.getQuota() + e.getValue())
                                .balance(quotaModel.getBalance() + e.getValue())
                                .build();
                    }
                    return QuotaModel.builder()
                            .folderId(e.getKey().getFolderId())
                            .providerId(e.getKey().getProviderId())
                            .resourceId(e.getKey().getResourceId())
                            .tenantId(tenantId)
                            .quota(e.getValue())
                            .balance(e.getValue())
                            .frozenQuota(0L)
                            .build();
                })
                .collect(Collectors.toList());

        return Result.success(updatedModels);
    }

    private void validateChange(ErrorCollection.Builder errorBuilder,
                                QuotaModel.Key quotaKey,
                                @Nullable QuotaModel quotaModel,
                                Long diff,
                                Locale locale) {
        if (quotaModel == null) {
            if (diff < 0) {
                errorBuilder.addError(TypedError.invalid(messages.getMessage(
                        "errors.transfers.negative.balance", new Object[]{quotaKey.getFolderId()}, locale)));
            }
            return;
        }

        Optional<Long> sum = Units.add(quotaModel.getBalance(), diff);
        if (sum.isEmpty()) {
            errorBuilder.addError(TypedError.invalid(messages.getMessage(
                    "errors.transfers.amount.overflow", new Object[]{quotaModel.getFolderId(),
                            quotaModel.getProviderId(), quotaModel.getResourceId(), diff}, locale)));
        } else if (sum.get() < 0) {
            errorBuilder.addError(TypedError.invalid(messages.getMessage(
                    "errors.transfers.negative.balance", new Object[]{quotaKey.getFolderId()}, locale)));
        }

    }

    private Result<List<QuotaDiff>> toQuotaDiffs(
            List<QuotaTransferInputDto.Transfer> transfers,
            Map<String, ResourceModel> internalResources,
            Map<QuotaTransferInputDto.ExternalResourceId, ResourceModel> externalResources,
            Map<ResourceModel, Map<String, Tuple2<UnitModel, UnitsEnsembleModel>>> units,
            Locale locale
    ) {


        List<QuotaDiff> quotaDiffs = new ArrayList<>(transfers.size());

        Set<QuotaModel.Key> destinationKeys = new HashSet<>();
        Set<QuotaModel.Key> sourceKeys = new HashSet<>();
        Set<Tuple3<String, String, String>> foldersResourceKey = new HashSet<>();

        for (QuotaTransferInputDto.Transfer transfer : transfers) {
            QuotaTransferInputDto.Resource resource = transfer.getResource();
            ResourceModel resourceModel;
            if (resource.getResourceId() != null) {
                resourceModel = internalResources.get(resource.getResourceId());
            } else {
                resourceModel = externalResources.get(resource.getExternalResourceId());
            }

            Tuple3<String, String, String> key = Tuples.of(transfer.getSourceFolderId(),
                    transfer.getDestinationFolderId(), resourceModel.getId());

            if (foldersResourceKey.contains(key)) {
                return Result.failure(ErrorCollection.builder()
                        .addError(TypedError.invalid(messages.getMessage(
                                "errors.transfers.duplicate.folders.pair.and.resource", key.toArray(), locale)))
                        .build());
            }
            foldersResourceKey.add(key);

            Tuple2<UnitModel, UnitsEnsembleModel> unit = units.get(resourceModel).get(transfer.getAmountUnitKey());
            Optional<Long> amount;
            try {
                amount = Units.convertFromApi(transfer.getAmount(), resourceModel, unit.getT2(),
                        unit.getT1());
            } catch (IllegalStateException e) {
                return Result.failure(ErrorCollection.builder()
                        .addError(TypedError.invalid("cant convert amount " + e.getMessage()))
                        .build());
            }
            if (amount.isEmpty()) {
                return Result.failure(ErrorCollection.builder()
                        .addError(TypedError.invalid(messages.getMessage(
                                "errors.transfers.amount.overflow", new Object[]{transfer.getDestinationFolderId(),
                                        resourceModel.getProviderId(), resourceModel.getId(), transfer.getAmount()},
                                locale)))
                        .build());
            }

            if (amount.get() < 0) {
                return Result.failure(ErrorCollection.builder()
                        .addError(TypedError.invalid(messages.getMessage(
                                "errors.transfers.negative.amount", null, locale)))
                        .build());
            }

            QuotaModel.Key destinationKey = new QuotaModel.Key(transfer.getDestinationFolderId(),
                    resourceModel.getProviderId(),
                    resourceModel.getId()
            );

            if (sourceKeys.contains(destinationKey)) {
                return Result.failure(ErrorCollection.builder()
                        .addError(TypedError.invalid(messages.getMessage(
                                "errors.transfers.invalid.source", null, locale)))
                        .build());
            }

            destinationKeys.add(destinationKey);

            QuotaModel.Key sourceKey = new QuotaModel.Key(transfer.getSourceFolderId(),
                    resourceModel.getProviderId(),
                    resourceModel.getId()
            );

            if (destinationKeys.contains(sourceKey)) {
                return Result.failure(ErrorCollection.builder()
                        .addError(TypedError.invalid(messages.getMessage(
                                "errors.transfers.invalid.source", null, locale)))
                        .build());
            }

            sourceKeys.add(sourceKey);

            quotaDiffs.add(new QuotaDiff(sourceKey, destinationKey, amount.get()));
        }

        return Result.success(quotaDiffs);
    }

    private Mono<Result<Map<ResourceModel, Map<String, Tuple2<UnitModel, UnitsEnsembleModel>>>>> validateUnits(
            Map<String, ResourceModel> internalResources,
            Map<QuotaTransferInputDto.ExternalResourceId, ResourceModel> externalResources,
            Multimap<String, String> unitByInternalResource,
            Multimap<QuotaTransferInputDto.ExternalResourceId, String> unitByExternalResource,
            Locale locale) {

        Multimap<ResourceModel, String> units = HashMultimap.create();

        for (String resourceKey : unitByInternalResource.keySet()) {
            units.putAll(internalResources.get(resourceKey), unitByInternalResource.get(resourceKey));
        }
        for (QuotaTransferInputDto.ExternalResourceId externalResourceId : unitByExternalResource.keySet()) {
            units.putAll(externalResources.get(externalResourceId), unitByExternalResource.get(externalResourceId));
        }

        Set<Tuple2<String, TenantId>> ensembleIds = units.keySet().stream()
                .map(rm -> Tuples.of(rm.getUnitsEnsembleId(), rm.getTenantId()))
                .collect(Collectors.toSet());

        return tableClient.usingSessionMonoRetryable(session -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(
                session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY), new ArrayList<>(ensembleIds)).map(es -> {
            Map<String, UnitsEnsembleModel> ensembleById = es.stream()
                    .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));

            Map<ResourceModel, Map<String, Tuple2<UnitModel, UnitsEnsembleModel>>> resourceUnits =
                    units.keys().elementSet().stream().collect(Collectors.toMap(Function.identity(),
                            rm -> {
                                UnitsEnsembleModel unitsEnsembleModel = ensembleById.get(rm.getUnitsEnsembleId());
                                return unitsEnsembleModel.getUnits().stream()
                                        .filter(u -> rm.getResourceUnits().getAllowedUnitIds().contains(u.getId()))
                                        .filter(u -> units.get(rm).contains(u.getKey()))
                                        .collect(Collectors.toMap(UnitModel::getKey, u ->
                                                Tuples.of(u, unitsEnsembleModel)));
                            }
                    ));

            List<Map.Entry<ResourceModel, String>> invalidUnits = units.entries().stream()
                    .filter(e -> !resourceUnits.get(e.getKey()).containsKey(e.getValue()))
                    .collect(Collectors.toList());

            if (!invalidUnits.isEmpty()) {
                return missingIdsResult("errors.unit.with.key.not.found", invalidUnits,
                        e -> new Object[]{e.getKey().getKey(), e.getValue()}, locale);
            }

            return Result.success(resourceUnits);

        }));
    }

    private Mono<Result<Map<QuotaTransferInputDto.ExternalResourceId, ResourceModel>>> validateExternalResources(
            Map<QuotaTransferInputDto.ExternalResourceId, ResourceBaseIdentity> typeByExternalResource,
            Map<QuotaTransferInputDto.ExternalResourceId, Set<ResourceSegmentSettingsModel>> segmentsByExternalResource,
            Locale locale) {

        Map<QuotaTransferInputDto.ExternalResourceId, Tuple2<ResourceBaseIdentity, Set<ResourceSegmentSettingsModel>>>
                resourceInfoByExternalResource = typeByExternalResource.keySet().stream()
                .collect(Collectors.toMap(Function.identity(), er ->
                        Tuples.of(typeByExternalResource.get(er), segmentsByExternalResource.get(er))));

        Set<ResourceBaseIdentity> ids = new HashSet<>(typeByExternalResource.values());

        return tableClient.usingSessionMonoRetryable(session ->
                resourcesDao.getAllByBaseIdentities(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                        new ArrayList<>(ids), false).map(rs -> {
                    Map<Tuple2<ResourceBaseIdentity, Set<ResourceSegmentSettingsModel>>, ResourceModel>
                            resourceByTypeAndSegments = rs.stream()
                            .collect(Collectors.toMap(r -> Tuples.of(
                                    new ResourceBaseIdentity(r.getTenantId(), r.getProviderId(), r.getResourceTypeId()),
                                    r.getSegments()), Function.identity()
                            ));

                    Set<Tuple2<ResourceBaseIdentity, Set<ResourceSegmentSettingsModel>>> invalidResourceInfo =
                            Sets.difference(new HashSet<>(resourceInfoByExternalResource.values()),
                                    resourceByTypeAndSegments.keySet());

                    if (!invalidResourceInfo.isEmpty()) {
                        return missingIdsResult("errors.resource.with.key.not.found", invalidResourceInfo,
                                t -> new Object[]{t.getT1().getResourceTypeId(), t.getT2().stream()
                                        .map(ResourceSegmentSettingsModel::getSegmentId)
                                        .collect(Collectors.joining(", "))}, locale);
                    }

                    return Result.success(resourceInfoByExternalResource.entrySet().stream()
                            .collect(Collectors.toMap(Map.Entry::getKey,
                                    e -> resourceByTypeAndSegments.get(e.getValue())))
                    );
                }));
    }

    private Mono<Result<Map<QuotaTransferInputDto.ExternalResourceId, Set<ResourceSegmentSettingsModel>>>>
    validateSegments(TenantId tenantId, Map<QuotaTransferInputDto.ExternalResourceId,
            Set<ResourceSegmentModel.SegmentationAndKey>> segmentsByExternalResource,
                     Locale locale) {

        Set<ResourceSegmentModel.SegmentationAndKey> keys = segmentsByExternalResource.values().stream()
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());

        List<Tuple2<ResourceSegmentModel.SegmentationAndKey, TenantId>> keysWithTenant =
                keys.stream()
                        .map(k -> Tuples.of(k, tenantId))
                        .collect(Collectors.toList());

        return tableClient.usingSessionMonoRetryable(session ->
                resourceSegmentsDao.getAllBySegmentationAndKey(
                        session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY), keysWithTenant).map(ss -> {
                    Map<ResourceSegmentModel.SegmentationAndKey, ResourceSegmentModel> segmentByKey = ss.stream()
                            .collect(Collectors.toMap(s -> new ResourceSegmentModel.SegmentationAndKey(
                                    s.getSegmentationId(), s.getKey()), Function.identity()));

                    Set<ResourceSegmentModel.SegmentationAndKey> invalidKeys =
                            Sets.difference(keys, segmentByKey.keySet());
                    if (!invalidKeys.isEmpty()) {
                        return missingIdsResult("errors.segment.with.key.not.found", invalidKeys,
                                sk -> new Object[]{sk.getSegmentationId(), sk.getKey()}, locale);
                    }
                    return Result.success(segmentsByExternalResource.entrySet().stream()
                            .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().stream()
                                    .map(k -> new ResourceSegmentSettingsModel(
                                            segmentByKey.get(k).getSegmentationId(), segmentByKey.get(k).getId()))
                                    .collect(Collectors.toSet())
                            ))
                    );

                }));
    }

    private Mono<Result<Map<QuotaTransferInputDto.ExternalResourceId, Set<ResourceSegmentModel.SegmentationAndKey>>>>
    validateSegmentations(TenantId tenantId, Map<QuotaTransferInputDto.ExternalResourceId,
            Set<Tuple2<ResourceSegmentationModel.ProviderKey, String>>> segmentsByExternalId,
                          Locale locale) {

        Set<ResourceSegmentationModel.ProviderKey> keys = segmentsByExternalId.values().stream()
                .flatMap(Collection::stream)
                .map(Tuple2::getT1)
                .collect(Collectors.toSet());

        List<Tuple2<ResourceSegmentationModel.ProviderKey, TenantId>> segmentationKeys =
                keys.stream()
                        .map(k -> Tuples.of(k, tenantId))
                        .collect(Collectors.toList());

        return tableClient.usingSessionMonoRetryable(session -> resourceSegmentationsDao.getAllByProvidersAndKeys(
                session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY), segmentationKeys).map(ss -> {
            Map<ResourceSegmentationModel.ProviderKey, ResourceSegmentationModel> segmentationById = ss.stream()
                    .collect(Collectors.toMap(s -> new ResourceSegmentationModel.ProviderKey(
                            s.getProviderId(), s.getKey()), Function.identity()));

            Set<ResourceSegmentationModel.ProviderKey> invalidSegmentationKeys =
                    Sets.difference(keys, segmentationById.keySet());
            if (!invalidSegmentationKeys.isEmpty()) {
                return missingIdsResult("errors.segmentation.with.key.not.found", invalidSegmentationKeys,
                        pk -> new Object[]{pk.getProviderId(), pk.getKey()}, locale);
            }
            return Result.success(segmentsByExternalId.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().stream()
                            .map(segmentationKeyAndSegmentKey -> new ResourceSegmentModel.SegmentationAndKey(
                                    segmentationById.get(segmentationKeyAndSegmentKey.getT1()).getId(),
                                    segmentationKeyAndSegmentKey.getT2()
                            ))
                            .collect(Collectors.toSet())
                    )));
        }));
    }

    private Mono<Result<Map<QuotaTransferInputDto.ExternalResourceId, ResourceBaseIdentity>>>
    validateExternalResourceTypeIds(Map<QuotaTransferInputDto.ExternalResourceId,
            Tuple2<Tuple2<String, String>, TenantId>> externalResourceTypeIds,
                                    Locale locale) {
        return tableClient.usingSessionMonoRetryable(session ->
                resourceTypesLoader.getResourceTypesByProviderAndKeys(
                        session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                        new ArrayList<>(externalResourceTypeIds.values())).map(ts -> {
                    Map<Tuple2<String, String>, ResourceBaseIdentity> resourceTypeById = ts.stream()
                            .collect(Collectors.toMap(t -> Tuples.of(t.getProviderId(), t.getKey()), t ->
                                    new ResourceBaseIdentity(t.getTenantId(), t.getProviderId(), t.getId())));
                    Set<Tuple2<String, String>> ids = externalResourceTypeIds.values().stream()
                            .map(Tuple2::getT1)
                            .collect(Collectors.toSet());

                    Set<Tuple2<String, String>> invalidResourceTypes = Sets.difference(ids, resourceTypeById.keySet());
                    if (!invalidResourceTypes.isEmpty()) {
                        return missingIdsResult("errors.resource.type.with.key.not.found",
                                invalidResourceTypes, Tuple2::toArray, locale);
                    }
                    return Result.success(externalResourceTypeIds.entrySet().stream()
                            .collect(Collectors.toMap(Map.Entry::getKey,
                                    e -> resourceTypeById.get(e.getValue().getT1()))));
                }));
    }

    private Mono<Result<Map<String, ResourceModel>>> validateResourceIds(Set<Tuple2<String, String>> resourceIds,
                                                                         TenantId tenantId, Locale locale) {

        List<Tuple2<String, TenantId>> resourceIdWithTenants = resourceIds.stream()
                .map(r -> Tuples.of(r.getT2(), tenantId))
                .distinct()
                .collect(Collectors.toList());

        return tableClient.usingSessionMonoRetryable(session ->
                resourcesLoader.getResourcesByIds(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                        resourceIdWithTenants).map(rs -> {
                    Set<Tuple2<String, String>> resultIds = rs.stream()
                            .map(r -> Tuples.of(r.getProviderId(), r.getId()))
                            .collect(Collectors.toSet());

                    Set<Tuple2<String, String>> invalidResourceIds = Sets.difference(resourceIds, resultIds);
                    if (!invalidResourceIds.isEmpty()) {
                        return missingIdsResult("errors.resource.with.id.not.found", invalidResourceIds,
                                Tuple2::toArray, locale);
                    }
                    return Result.success(rs.stream()
                            .collect(Collectors.toMap(ResourceModel::getId, Function.identity())));
                }));
    }

    private Mono<Result<Map<String, FolderModel>>> validateFolderIds(TenantId tenantId, Set<String> folderIds,
                                                                     Locale locale) {
        return tableClient.usingSessionMonoRetryable(session ->
                folderDao.getByIds(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                        new ArrayList<>(folderIds), tenantId).map(fs -> {
                            Map<String, FolderModel> folderById = fs.stream()
                                    .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
                            Set<String> invalidFolderIds = Sets.difference(folderIds, folderById.keySet());
                            if (!invalidFolderIds.isEmpty()) {
                                return missingIdsResult("errors.folder.with.id.not.found", invalidFolderIds,
                                        s -> new Object[]{s}, locale);
                            }
                            return Result.success(folderById);
                        }
                ));
    }

    private Mono<Result<Void>> validateService(List<QuotaTransferInputDto.Transfer> transfers,
                                               Map<String, FolderModel> folderById, Locale locale) {
        Set<Long> serviceIdsForSourceFolders = new HashSet<>();
        Set<Long> serviceIdsForDestinationFolders = new HashSet<>();

        for (QuotaTransferInputDto.Transfer transfer : transfers) {
            FolderModel sourceFolder = folderById.get(transfer.getSourceFolderId());
            FolderModel destinationFolder = folderById.get(transfer.getDestinationFolderId());

            serviceIdsForSourceFolders.add(sourceFolder.getServiceId());
            serviceIdsForDestinationFolders.add(destinationFolder.getServiceId());
        }

        return tableClient.usingSessionMonoRetryable(session ->
                        abcServiceValidator.validateAbcServices(new ArrayList<>(serviceIdsForSourceFolders),
                                locale,
                                session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                                AbcServiceValidator.ALLOWED_SERVICE_STATES_FOR_QUOTA_EXPORT,
                                AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES_FOR_QUOTA_EXPORT,
                                true).flatMap(u ->
                                abcServiceValidator.validateAbcServices(
                                        new ArrayList<>(serviceIdsForDestinationFolders),
                                        locale,
                                        session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                                        AbcServiceValidator.ALLOWED_SERVICE_STATES,
                                        AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES,
                                        false)))
                .map(result -> result.apply(u -> null));
    }

    private <R, T> Result<R> missingIdsResult(String code,
                                              Collection<T> invalidIds,
                                              Function<T, Object[]> toParams,
                                              Locale locale) {
        ErrorCollection.Builder builder = ErrorCollection.builder();
        invalidIds.stream()
                .map(ids -> TypedError.notFound(messages.getMessage(code, toParams.apply(ids), locale)))
                .forEach(builder::addError);
        return Result.failure(builder.build());
    }

    private static class QuotaDiff {
        private final QuotaModel.Key sourceKey;
        private final QuotaModel.Key destKey;
        private final Long amount;
        private final String sourceLogId;
        private final String destLogId;

        private QuotaDiff(QuotaModel.Key sourceKey, QuotaModel.Key destKey, Long amount) {
            this.sourceKey = sourceKey;
            this.destKey = destKey;
            this.amount = amount;
            sourceLogId = UUID.randomUUID().toString();
            destLogId = UUID.randomUUID().toString();
        }

        public QuotaModel.Key getSourceKey() {
            return sourceKey;
        }

        public QuotaModel.Key getDestKey() {
            return destKey;
        }

        public Long getAmount() {
            return amount;
        }

        public String getSourceLogId() {
            return sourceLogId;
        }

        public String getDestLogId() {
            return destLogId;
        }
    }
}
