package ru.yandex.intranet.d.services.transfer;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.providers.RelatedCoefficient;
import ru.yandex.intranet.d.model.providers.RelatedResourceMapping;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer;
import ru.yandex.intranet.d.model.transfers.QuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer;
import ru.yandex.intranet.d.model.transfers.TransferParameters;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.quotas.QuotasHelper;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedProvisionTransfer;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedQuotaTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedReserveTransfer;
import ru.yandex.intranet.d.services.transfer.model.ProvisionTransferPreApplicationData;
import ru.yandex.intranet.d.services.transfer.model.QuotaTransferPreApplicationData;
import ru.yandex.intranet.d.services.transfer.model.ResponsibleAndNotified;
import ru.yandex.intranet.d.services.transfer.model.ValidatedProvisionTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedProvisionTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaResourceTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedReserveTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedReserveTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestAccountWarnings;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestFolderWarnings;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestModel;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestPermissions;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestResult;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestWarnings;
import ru.yandex.intranet.d.util.ProviderUtil;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TransferRequestErrorMessages;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.AmountDto;
import ru.yandex.intranet.d.web.model.QuotaDto;
import ru.yandex.intranet.d.web.model.recipe.AccountQuotaDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunCreateTransferRequestDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.services.recipe.RecipeService.toAccountQuotaDto;
import static ru.yandex.intranet.d.services.transfer.TransferRequestProvisionService.toProvisionTransfer;
import static ru.yandex.intranet.d.services.transfer.TransferRequestReserveService.getQuotaTransfer;
import static ru.yandex.intranet.d.services.transfer.TransferRequestValidationService.getAllocated;
import static ru.yandex.intranet.d.services.transfer.TransferRequestValidationService.getAmount;
import static ru.yandex.intranet.d.services.transfer.TransferRequestValidationService.getBalance;
import static ru.yandex.intranet.d.services.transfer.TransferRequestValidationService.getFrozenQuota;
import static ru.yandex.intranet.d.services.transfer.TransferRequestValidationService.getProvided;
import static ru.yandex.intranet.d.util.Util.indexBy;

/**
 * TransferDryRunRequestService
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@Component
public class DryRunTransferRequestService {

    private final TransferRequestValidationService validationService;
    private final TransferRequestStoreService storeService;
    private final TransferRequestQuotaService transferRequestQuotaService;
    private final TransferRequestReserveService transferRequestReserveService;
    private final TransferRequestResponsibleAndNotifyService transferRequestResponsibleAndNotifyService;
    private final SecurityManagerService securityManagerService;
    private final TransferRequestAnswerService answerService;
    private final MessageSource messages;
    private final YdbTableClient tableClient;
    private final TransferRequestErrorMessages transferRequestErrorMessages;
    private final TransferRequestProvisionService transferRequestProvisionService;

    @SuppressWarnings("ParameterNumber")
    public DryRunTransferRequestService(TransferRequestValidationService validationService,
                                        TransferRequestStoreService storeService,
                                        TransferRequestQuotaService transferRequestQuotaService,
                                        TransferRequestReserveService transferRequestReserveService,
                                        TransferRequestResponsibleAndNotifyService
                                                    transferRequestResponsibleAndNotifyService,
                                        SecurityManagerService securityManagerService,
                                        TransferRequestAnswerService answerService,
                                        @Qualifier("messageSource") MessageSource messages,
                                        YdbTableClient tableClient,
                                        @Qualifier("transferRequestErrorMessages")
                                                    TransferRequestErrorMessages transferRequestErrorMessages,
                                        TransferRequestProvisionService transferRequestProvisionService) {
        this.validationService = validationService;
        this.storeService = storeService;
        this.transferRequestQuotaService = transferRequestQuotaService;
        this.transferRequestReserveService = transferRequestReserveService;
        this.transferRequestResponsibleAndNotifyService = transferRequestResponsibleAndNotifyService;
        this.securityManagerService = securityManagerService;
        this.answerService = answerService;
        this.messages = messages;
        this.tableClient = tableClient;
        this.transferRequestErrorMessages = transferRequestErrorMessages;
        this.transferRequestProvisionService = transferRequestProvisionService;
    }

    private Mono<Result<DryRunTransferRequestResult>> dryRun(YdbTxSession txSession,
                                                             FrontDryRunCreateTransferRequestDto transferRequest,
                                                             YaUserDetails currentUser,
                                                             Locale locale) {
        if (currentUser.getUser().isEmpty()) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                    .getMessage("errors.access.denied", null, locale)))
                    .build()));
        }
        Result<TransferRequestType> typeR = validationService
                .validateTransferRequestType(transferRequest::getRequestType, locale);
        return typeR.andThenMono(type -> switch (type) {
            case QUOTA_TRANSFER -> dryRunQuotaTransfer(txSession, transferRequest, currentUser, locale);
            case RESERVE_TRANSFER -> dryRunReserveTransfer(txSession, transferRequest, currentUser, locale);
            case PROVISION_TRANSFER ->
                    dryRunProvisionTransfer(txSession, transferRequest, currentUser, locale);
            case FOLDER_TRANSFER, ACCOUNT_TRANSFER ->
                    throw new IllegalArgumentException("Unsupported transfer request type: " + type);
        });
    }

    public Mono<Result<DryRunTransferRequestResult>> dryRunProvisionTransfer(
            YdbTxSession txSession,
            FrontDryRunCreateTransferRequestDto transferRequest,
            YaUserDetails currentUser,
            Locale locale
    ) {
        ErrorCollection.Builder preErrorsBuilder = ErrorCollection.builder();
        if (transferRequest.getParameters().isEmpty()) {
            preErrorsBuilder.addError("parameters", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        if (transferRequest.getParameters().isPresent()) {
            List<PreValidatedProvisionTransfer.Builder> preValidatedProvisionTransfersBuilders =
                    validationService.preValidateProvisionTransferParameters(transferRequest.getParameters().get(),
                            null, preErrorsBuilder, locale, false);
            if (preErrorsBuilder.hasAnyErrors()) {
                return Mono.just(Result.failure(preErrorsBuilder.build()));
            }
            List<PreValidatedProvisionTransfer> preValidatedProvisionTransfers =
                    preValidatedProvisionTransfersBuilders.stream()
                            .map(PreValidatedProvisionTransfer.Builder::build)
                            .collect(Collectors.toList());
            return transferRequestProvisionService.validateProvisionTransfersParametersMono(txSession,
                            preValidatedProvisionTransfers, currentUser, locale, false)
                    .flatMap(r -> r.andThenMono(parameters -> getRelatedResources(txSession, parameters)
                    .flatMap(relatedResources -> getUnitsEnsembles(txSession, relatedResources)
                    .flatMap(relatedUnitsEnsembles ->
                            validateProvisionTransferApplication(txSession, parameters, locale)
                    .flatMap(preApplicationR -> preApplicationR.applyMono(preApplication -> getResult(
                            txSession, transferRequest, currentUser, parameters, preApplication, relatedResources,
                            relatedUnitsEnsembles)
                    ))))));
        } else {
            return Mono.just(Result.failure(preErrorsBuilder.build()));
        }
    }

    public Mono<List<ResourceModel>> getRelatedResources(
            YdbTxSession txSession,
            ValidatedProvisionTransferParameters parameters
    ) {
        Map<String, RelatedResourceMapping> relatedResourceMappingByResourceId = new HashMap<>();
        for (ProviderModel provider : parameters.getProviders()) {
            Map<String, RelatedResourceMapping> providerRelatedResource =
                    provider.getRelatedResourcesByResourceId().orElse(Map.of());
            relatedResourceMappingByResourceId.putAll(providerRelatedResource);
        }
        Set<String> relatedResourceIds = parameters.getResources().stream()
                .map(r -> relatedResourceMappingByResourceId.get(r.getId()))
                .filter(Objects::nonNull)
                .flatMap(mapping -> mapping.getRelatedCoefficientMap().keySet().stream())
                .collect(Collectors.toSet());
        return storeService.loadResources(txSession, relatedResourceIds)
                .map(resources -> resources.stream()
                        .filter(r -> !r.isReadOnly() && !r.isDeleted() && r.isManaged() && !r.isVirtual())
                        .collect(Collectors.toList()));
    }

    public Mono<List<UnitsEnsembleModel>> getUnitsEnsembles(
            YdbTxSession txSession,
            List<ResourceModel> relatedResources
    ) {
        Set<String> unitsEnsembleIds = relatedResources.stream()
                .map(ResourceModel::getUnitsEnsembleId)
                .collect(Collectors.toSet());
        return storeService.loadUnitsEnsembles(txSession, unitsEnsembleIds);
    }

    public Mono<Result<ValidatedProvisionTransferApplication>>
    validateProvisionTransferApplication(
            YdbTxSession txSession,
            ValidatedProvisionTransferParameters parameters,
            Locale locale
    ) {
        Set<String> folderIds = new HashSet<>();
        Set<String> accountIds = new HashSet<>();

        List<ValidatedProvisionTransfer> provisionTransfers = parameters.getProvisionTransfers();
        Map<Tuple2<String, String>, Tuple2<ValidatedQuotaResourceTransfer, String>> aggregatedTransfers =
                TransferRequestProvisionService.aggregateProvisionTransfers(parameters);

        for (ValidatedProvisionTransfer provisionTransfer : provisionTransfers) {
            Stream<ValidatedQuotaResourceTransfer> accountTransfersStream = Stream.of(
                    provisionTransfer.getSourceAccountTransfers(), provisionTransfer.getDestinationAccountTransfers())
                    .flatMap(Collection::stream);
            Result<ValidatedProvisionTransferApplication> accountSumValidationResult =
                    validationService.validateQuotaTransfersSum(accountTransfersStream, locale);
            if (accountSumValidationResult.isFailure()) {
                return Mono.just(accountSumValidationResult);
            }
            folderIds.add(provisionTransfer.getSourceFolder().getId());
            folderIds.add(provisionTransfer.getDestinationFolder().getId());
            accountIds.add(provisionTransfer.getSourceAccount().getId());
            accountIds.add(provisionTransfer.getDestinationAccount().getId());
        }

        Map<String, ProviderModel> providerById = indexBy(parameters.getProviders(),
                ProviderModel::getId);
        return storeService.loadQuotas(txSession, folderIds).flatMap(folderQuotas ->
            storeService.loadAccountsQuotas(txSession, accountIds).map(accountQuotas -> {
                Map<String, Map<String, AccountsQuotasModel>> quotasByAccountResource = indexBy(accountQuotas,
                        AccountsQuotasModel::getAccountId, AccountsQuotasModel::getResourceId);
                ErrorCollection.Builder errors = ErrorCollection.builder();
                Map<String, List<String>> warningsPerResource = new HashMap<>();
                Map<String, Map<String, Object>> detailsPerResource = new HashMap<>();
                Map<String, DryRunTransferRequestFolderWarnings> warningsPerFolder = new HashMap<>();
                Map<String, DryRunTransferRequestAccountWarnings> warningsPerAccount = new HashMap<>();
                aggregatedTransfers.forEach((accountResource, transferField) ->
                        processAccountQuotaTransfer(transferField.getT1(), accountResource.getT1(), providerById,
                                quotasByAccountResource, errors, warningsPerResource, detailsPerResource,
                                warningsPerAccount, transferField.getT2(), locale));
                DryRunTransferRequestWarnings druRunWarnings = new DryRunTransferRequestWarnings(
                        new ArrayList<>(), warningsPerResource, warningsPerFolder, detailsPerResource,
                        warningsPerAccount);
                ProvisionTransferPreApplicationData preApplicationData = new ProvisionTransferPreApplicationData(
                        folderQuotas, accountQuotas, Map.of());
                ValidatedProvisionTransferApplication validatedApplication = new ValidatedProvisionTransferApplication(
                        preApplicationData,
                        druRunWarnings,
                        errors.build()
                );
                return Result.success(validatedApplication);
            }));
    }

    public List<QuotaModel> getUpdatedQuotaModels(List<QuotaModel> quotas,
                                                  List<ValidatedQuotaTransfer> quotaTransfers) {
        Map<String, Map<String, QuotaModel>> quotasByFolderResource = new HashMap<>();
        quotas.forEach(q -> quotasByFolderResource.computeIfAbsent(q.getFolderId(), k -> new HashMap<>())
                .put(q.getResourceId(), q));
        List<QuotaModel> updatedQuotas = new ArrayList<>();
        for (ValidatedQuotaTransfer quotaTransfer : quotaTransfers) {
            for (ValidatedQuotaResourceTransfer resourceTransfer : quotaTransfer.getTransfers()) {
                QuotaModel currentQuota = quotasByFolderResource
                        .getOrDefault(quotaTransfer.getDestinationFolder().getId(), Collections.emptyMap())
                        .get(resourceTransfer.getResource().getId());
                if (currentQuota != null) {
                    QuotaModel updatedQuota = QuotaModel.builder(currentQuota)
                            .quota((currentQuota.getQuota() != null ? currentQuota.getQuota() : 0L)
                                    + resourceTransfer.getDelta())
                            .balance((currentQuota.getBalance() != null ? currentQuota.getBalance() : 0L)
                                    + resourceTransfer.getDelta())
                            .build();
                    updatedQuotas.add(updatedQuota);
                } else {
                    QuotaModel newQuota = QuotaModel.builder()
                            .tenantId(Tenants.DEFAULT_TENANT_ID)
                            .folderId(quotaTransfer.getDestinationFolder().getId())
                            .resourceId(resourceTransfer.getResource().getId())
                            .providerId(resourceTransfer.getResource().getProviderId())
                            .quota(resourceTransfer.getDelta())
                            .balance(resourceTransfer.getDelta())
                            .frozenQuota(0L)
                            .build();
                    updatedQuotas.add(newQuota);
                }
            }
        }
        return updatedQuotas;
    }

    private DryRunTransferRequestResult prepareDryRunCreateQuotaTransferRequest(
            ValidatedQuotaTransferParameters validatedParameters,
            Tuple2<QuotaTransferPreApplicationData, DryRunTransferRequestWarnings> preApplicationData,
            @Nullable TransferResponsible responsible,
            @Nullable DryRunTransferRequestPermissions permissions) {

        List<QuotaModel> quotas = preApplicationData.getT1().getQuotas();

        TransferParameters.Builder parametersBuilder = TransferParameters.builder();
        validatedParameters.getTransfers().forEach(quotaTransfer -> {
            QuotaTransfer.Builder quotaTransferBuilder = QuotaTransfer.builder();
            quotaTransferBuilder.folderId(quotaTransfer.getDestinationFolder().getId());
            quotaTransferBuilder.serviceId(quotaTransfer.getDestinationService().getId());
            quotaTransfer.getTransfers().forEach(resourceTransfer -> {
                ResourceQuotaTransfer.Builder resourceQuotaTransferBuilder = ResourceQuotaTransfer.builder();
                resourceQuotaTransferBuilder.resourceId(resourceTransfer.getResource().getId());
                resourceQuotaTransferBuilder.delta(resourceTransfer.getDelta());
                quotaTransferBuilder.addTransfer(resourceQuotaTransferBuilder.build());
            });
            parametersBuilder.addQuotaTransfer(quotaTransferBuilder.build());
        });

        List<QuotaModel> updatedQuotaModels = getUpdatedQuotaModels(quotas, validatedParameters.getTransfers());
        Set<QuotaModel.Key> updatedQuotaKeys = getUpdatedQuotaKeys(updatedQuotaModels);

        return DryRunTransferRequestResult.builder()
                .folders(validatedParameters.getFolders())
                .services(validatedParameters.getServices())
                .resources(validatedParameters.getResources())
                .unitsEnsembles(validatedParameters.getUnitsEnsembles())
                .providers(validatedParameters.getProviders())
                .transferRequest(DryRunTransferRequestModel.builder()
                        .type(TransferRequestType.QUOTA_TRANSFER)
                        .quotasOld(getOldQuota(quotas, updatedQuotaKeys))
                        .quotasNew(getNewQuota(updatedQuotaModels))
                        .parameters(parametersBuilder.build())
                        .build())
                .responsible(responsible)
                .permissions(permissions)
                .warnings(preApplicationData.getT2())
                .build();
    }

    public Mono<Result<Tuple2<QuotaTransferPreApplicationData, DryRunTransferRequestWarnings>>>
    validateQuotaTransferApplication(YdbTxSession txSession, ValidatedQuotaTransferParameters parameters,
                                     Locale locale) {
        Map<String, BigInteger> sumPerResource = new HashMap<>();
        parameters.getTransfers().forEach(quotaTransfer -> {
            quotaTransfer.getTransfers().forEach(resourceTransfer -> {
                BigInteger currentSum = sumPerResource.getOrDefault(resourceTransfer.getResource().getId(),
                        BigInteger.ZERO);
                sumPerResource.put(resourceTransfer.getResource().getId(),
                        currentSum.add(BigInteger.valueOf(resourceTransfer.getDelta())));
            });
        });
        if (sumPerResource.values().stream().anyMatch(v -> v.compareTo(BigInteger.ZERO) != 0)) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                    .getMessage("errors.transfer.request.has.non.zero.quota.sum", null, locale)))
                    .build()));
        }
        Set<String> folderIds = new HashSet<>();
        parameters.getTransfers().forEach(t -> folderIds.add(t.getDestinationFolder().getId()));
        return storeService.loadQuotas(txSession, folderIds).map(quotas -> {
            Map<String, Map<String, QuotaModel>> quotasByFolderResource = new HashMap<>();
            quotas.forEach(quota -> quotasByFolderResource.computeIfAbsent(quota.getFolderId(),
                    k -> new HashMap<>()).put(quota.getResourceId(), quota));
            ErrorCollection.Builder errors = ErrorCollection.builder();
            Map<String, List<String>> warningsPerResource = new HashMap<>();
            Map<String, Map<String, Object>> detailsPerResource = new HashMap<>();
            Map<String, DryRunTransferRequestFolderWarnings> warningsPerFolder = new HashMap<>();
            parameters.getTransfers().forEach(quotaTransfer -> {
                quotaTransfer.getTransfers().forEach(resourceTransfer -> {
                    processFolderQuotaTransfer(resourceTransfer, quotaTransfer.getDestinationFolder().getId(),
                            quotasByFolderResource, errors, warningsPerResource,
                            detailsPerResource, warningsPerFolder,
                            "parameters.quotaTransfers." + quotaTransfer.getIndex() + ".resourceTransfers", locale);
                });
            });
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
            QuotaTransferPreApplicationData quotaTransferData =
                    QuotaTransferPreApplicationData.builder().addQuotas(quotas).build();
            return Result.success(Tuples.of(quotaTransferData, new DryRunTransferRequestWarnings(new ArrayList<>(),
                    warningsPerResource, warningsPerFolder, detailsPerResource, Map.of())));
        });
    }

    @SuppressWarnings("ParameterNumber")
    private void processFolderQuotaTransfer(
            ValidatedQuotaResourceTransfer resourceTransfer,
            String folderId,
            Map<String, Map<String, QuotaModel>> quotasByFolderResource,
            ErrorCollection.Builder errors,
            Map<String, List<String>> warningsPerResource,
            Map<String, Map<String, Object>> detailsPerResource,
            Map<String, DryRunTransferRequestFolderWarnings> warningsPerFolder,
            String field,
            Locale locale) {
        QuotaModel currentQuota = quotasByFolderResource
                .getOrDefault(folderId, Collections.emptyMap())
                .getOrDefault(resourceTransfer.getResource().getId(), null);
        long currentBalance = getBalance(currentQuota);
        long currentAmount = getAmount(currentQuota);
        long currentProvided = currentAmount != 0L ? (currentAmount - currentBalance) : 0L;
        long delta = resourceTransfer.getDelta();
        if (delta < 0 && Math.abs(delta) > currentBalance) {
            String message = transferRequestErrorMessages.getNotEnoughBalanceErrorMessage(
                    resourceTransfer, currentBalance, currentProvided, locale);
            warningsPerResource.computeIfAbsent(resourceTransfer.getResource().getId(),
                    rid -> new ArrayList<>())
                    .add(message);
            DryRunTransferRequestFolderWarnings folderWarnings =
                    warningsPerFolder.computeIfAbsent(folderId,
                            fid -> new DryRunTransferRequestFolderWarnings(new ArrayList<>(),
                                    new HashMap<>(), new HashMap<>()));
            folderWarnings.getMessages().add(message);
            folderWarnings.getPerResource().computeIfAbsent(resourceTransfer.getResource().getId(),
                    rid -> new ArrayList<>())
                    .add(message);
            if (currentBalance > 0) {
                AmountDto suggestedAmount = QuotasHelper.getAmountDto(QuotasHelper.toBigDecimal(currentBalance),
                        resourceTransfer.getResource(), resourceTransfer.getResourceUnitsEnsemble(), locale);
                Map<String, Object> details = Map.of("suggestedAmount", suggestedAmount,
                        "suggestedAmountPrompt", messages.getMessage(
                                "errors.not.enough.balance.for.quota.transfer.non.zero.balance.button",
                                null, locale)
                );
                folderWarnings.getDetailsPerResource().computeIfAbsent(resourceTransfer.getResource().getId(),
                                rid -> new HashMap<>())
                        .putAll(details);
                detailsPerResource.computeIfAbsent(resourceTransfer.getResource().getId(),
                                rid -> new HashMap<>())
                        .putAll(details);

            }
        }
        if (delta == 0L || Units.add(currentAmount, delta).isEmpty()
                || Units.add(currentBalance, delta).isEmpty()) {
            errors.addError(field + "." + resourceTransfer.getIndex() + ".delta",
                    TypedError.invalid(messages.getMessage("errors.number.out.of.range", null, locale)));
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void processAccountQuotaTransfer(
            ValidatedQuotaResourceTransfer resourceTransfer,
            String accountId,
            Map<String, ProviderModel> providerById,
            Map<String, Map<String, AccountsQuotasModel>> quotasByAccountResource,
            ErrorCollection.Builder errors,
            Map<String, List<String>> warningsPerResource,
            Map<String, Map<String, Object>> detailsPerResource,
            Map<String, DryRunTransferRequestAccountWarnings> warningsPerAccount,
            String field,
            Locale locale) {
        ProviderModel provider = providerById.get(resourceTransfer.getResource().getProviderId());
        boolean isAllocatedSupported = ProviderUtil.isAllocatedSupported(provider, resourceTransfer.getResource());
        AccountsQuotasModel currentQuota = quotasByAccountResource
                .getOrDefault(accountId, Collections.emptyMap())
                .getOrDefault(resourceTransfer.getResource().getId(), null);

        long currentProvided = getProvided(currentQuota);
        long currentFrozen = getFrozenQuota(currentQuota);
        long currentAllocated = isAllocatedSupported ? getAllocated(currentQuota) : 0L;
        long delta = resourceTransfer.getDelta();
        long currentAvailable = Math.max(0L, currentProvided - currentFrozen - currentAllocated);
        if (delta < 0) {
            if (Math.abs(delta) > currentProvided) {
                String message = transferRequestErrorMessages.getProvidedQuotaLessThanRequestedErrorMessage(
                        resourceTransfer, currentProvided, currentAllocated, currentFrozen, locale);
                errors.addError(field + "." + resourceTransfer.getIndex() + ".delta",
                        TypedError.invalid(message));
                if (currentAvailable > 0) {
                    addSuggestedAmountAccount(errors, currentAvailable, isAllocatedSupported, resourceTransfer, locale);
                }
            } else if (Math.abs(delta) > currentAvailable) {
                String message = transferRequestErrorMessages.getNotEnoughAccountAvailableQuotaErrorMessage(
                        resourceTransfer, currentProvided, currentAllocated, currentFrozen, locale);

                warningsPerResource.computeIfAbsent(resourceTransfer.getResource().getId(),
                                rid -> new ArrayList<>())
                        .add(message);

                DryRunTransferRequestAccountWarnings accountWarnings =
                        warningsPerAccount.computeIfAbsent(accountId,
                                fid -> new DryRunTransferRequestAccountWarnings(new ArrayList<>(),
                                        new HashMap<>(), new HashMap<>()));

                accountWarnings.getMessages().add(message);
                accountWarnings.getPerResource().computeIfAbsent(resourceTransfer.getResource().getId(),
                                rid -> new ArrayList<>())
                        .add(message);
                if (currentAvailable > 0) {
                    AmountDto suggestedAmount = QuotasHelper.getAmountDto(QuotasHelper.toBigDecimal(currentAvailable),
                            resourceTransfer.getResource(), resourceTransfer.getResourceUnitsEnsemble(), locale);
                    Map<String, Object> details = Map.of(
                            "suggestedProvisionAmount", suggestedAmount,
                            "suggestedProvisionAmountPrompt", messages.getMessage(
                                    getSuggestedAmountPromptCode(isAllocatedSupported),
                                    null, locale)
                    );
                    accountWarnings.getDetailsPerResource().computeIfAbsent(resourceTransfer.getResource().getId(),
                                    rid -> new HashMap<>())
                            .putAll(details);
                    detailsPerResource.computeIfAbsent(resourceTransfer.getResource().getId(),
                                    rid -> new HashMap<>())
                            .putAll(details);
                }
            }
        }
        if (delta == 0L || Units.add(currentProvided, delta).isEmpty()) {
            errors.addError(field + "." + resourceTransfer.getIndex() + ".delta",
                    TypedError.invalid(messages.getMessage("errors.number.out.of.range", null, locale)));
        }
    }

    private static String getSuggestedAmountPromptCode(boolean isAllocatedSupported) {
        if (isAllocatedSupported) {
            return "errors.not.enough.available.for.provision.transfer.not.allocated.button";
        } else {
            return "errors.not.enough.available.for.provision.transfer.provided.button";
        }
    }

    private void addSuggestedAmountAccount(ErrorCollection.Builder errors, long currentAvailable,
                                           boolean isAllocatedSupported,
                                           ValidatedQuotaResourceTransfer resourceTransfer, Locale locale) {
        errors.addDetail(resourceTransfer.getResource().getId() + ".suggestedProvisionAmount",
                QuotasHelper.getAmountDto(QuotasHelper.toBigDecimal(currentAvailable), resourceTransfer.getResource(),
                        resourceTransfer.getResourceUnitsEnsemble(), locale));
        errors.addDetail("suggestedProvisionAmountPrompt", messages.getMessage(
                getSuggestedAmountPromptCode(isAllocatedSupported),
                null, locale));
    }

    public Mono<Result<DryRunTransferRequestResult>> dryRunQuotaTransfer(
            YdbTxSession txSession,
            FrontDryRunCreateTransferRequestDto transferRequest,
            YaUserDetails currentUser,
            Locale locale) {
        ErrorCollection.Builder preErrorsBuilder = ErrorCollection.builder();
        if (transferRequest.getParameters().isEmpty()) {
            preErrorsBuilder.addError("parameters", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        if (transferRequest.getParameters().isPresent()) {
            PreValidatedQuotaTransferParameters.Builder preValidatedParametersBuilder
                    = PreValidatedQuotaTransferParameters.builder();
            validationService.preValidateQuotaTransferParameters(transferRequest.getParameters().get(),
                    preValidatedParametersBuilder, preErrorsBuilder, locale, false);
            if (preErrorsBuilder.hasAnyErrors()) {
                return Mono.just(Result.failure(preErrorsBuilder.build()));
            }
            PreValidatedQuotaTransferParameters preValidatedParameters = preValidatedParametersBuilder.build();

            return transferRequestQuotaService.validateQuotaTransferParameters(txSession, preValidatedParameters,
                    currentUser, locale, false).flatMap(r -> r.andThenMono(parameters ->
                    validateQuotaTransferApplication(txSession, parameters, locale).flatMap(preApplicationR ->
                            preApplicationR.applyMono(preApplication -> getResult(
                                    txSession, transferRequest, currentUser, parameters, preApplication)
                            ))));
        } else {
            return Mono.just(Result.failure(preErrorsBuilder.build()));
        }
    }


    private Mono<DryRunTransferRequestResult> getResult(
            YdbTxSession txSession,
            FrontDryRunCreateTransferRequestDto transferRequest,
            YaUserDetails currentUser,
            ValidatedProvisionTransferParameters parameters,
            ValidatedProvisionTransferApplication preApplication,
            List<ResourceModel> relatedResources,
            List<UnitsEnsembleModel> relatedUnitsEnsembles
    ) {
        if (transferRequest.isPrepareResponsible() || transferRequest.isPreparePermissions()) {
            return transferRequestResponsibleAndNotifyService.calculateForProvisionTransferCreate(
                    txSession, parameters, currentUser).map(responsible -> prepareDryRunCreateProvisionTransferRequest(
                    parameters, preApplication, relatedResources, relatedUnitsEnsembles, responsible.getResponsible(),
                    getPermissions(responsible, currentUser)));
        }
        return Mono.just(prepareDryRunCreateProvisionTransferRequest(parameters, preApplication, relatedResources,
                relatedUnitsEnsembles, null, null));
    }

    private Mono<DryRunTransferRequestResult> getResult(
            YdbTxSession txSession,
            FrontDryRunCreateTransferRequestDto transferRequest, YaUserDetails currentUser,
            ValidatedQuotaTransferParameters parameters,
            Tuple2<QuotaTransferPreApplicationData, DryRunTransferRequestWarnings> preApplication) {

        if (transferRequest.isPrepareResponsible() || transferRequest.isPreparePermissions()) {
            return transferRequestResponsibleAndNotifyService.calculateForQuotaTransferCreate(
                    txSession, parameters, currentUser).map(responsibles -> prepareDryRunCreateQuotaTransferRequest(
                    parameters, preApplication, responsibles.getResponsible(),
                    getPermissions(responsibles, currentUser)));
        }
        return Mono.just(prepareDryRunCreateQuotaTransferRequest(parameters, preApplication, null, null));
    }


    public Mono<Result<ExpandedTransferRequests<DryRunTransferRequestResult>>> dryRun(
            FrontDryRunCreateTransferRequestDto transferRequest,
            YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(ra -> ra.andThenMono(vb ->
                tableClient.usingSessionMonoRetryable(session -> session.usingTxMonoRetryable(
                        TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                                dryRun(txSession, transferRequest, currentUser, locale))
                        .flatMap(rd -> rd.applyMono(re -> answerService.expand(session, re)))
                )));
    }

    public DryRunTransferRequestPermissions getPermissions(
            ResponsibleAndNotified responsibleAndNotified,
            YaUserDetails currentUser) {
        TransferResponsible responsible = responsibleAndNotified.getResponsible();
        boolean canAutoConfirmAsProviderResponsible = responsibleAndNotified.isProviderResponsibleAutoConfirm();

        String userId = currentUser.getUser().orElseThrow().getId();
        boolean canVote = responsible.getResponsible().stream()
                .anyMatch(foldersResponsible -> foldersResponsible.getResponsible()
                        .stream().anyMatch(serviceResponsible -> serviceResponsible.getResponsibleIds()
                                .contains(userId)))
                || responsible.getReserveResponsibleModel().map(reserveResponsible -> reserveResponsible
                .getResponsibleIds().contains(userId)).orElse(false);

        boolean canConfirmSingleHandedly;
        if (!canVote) {
            canConfirmSingleHandedly = false;
        } else {
            canConfirmSingleHandedly = responsible.getResponsible().stream()
                    .allMatch(foldersResponsible -> foldersResponsible.getResponsible()
                            .stream().anyMatch(serviceResponsible -> serviceResponsible.getResponsibleIds()
                                    .contains(userId)))
                    && responsible.getReserveResponsibleModel()
                    .map(reserveResponsible -> reserveResponsible.getResponsibleIds()
                            .contains(userId)).orElse(true);
        }

        return new DryRunTransferRequestPermissions(canVote,
                canAutoConfirmAsProviderResponsible, canConfirmSingleHandedly);
    }

    public Mono<Result<DryRunTransferRequestResult>> dryRunReserveTransfer(
            YdbTxSession txSession,
            FrontDryRunCreateTransferRequestDto transferRequest,
            YaUserDetails currentUser,
            Locale locale) {
        ErrorCollection.Builder preErrorsBuilder = ErrorCollection.builder();
        if (transferRequest.getParameters().isEmpty()) {
            preErrorsBuilder.addError("parameters", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return Mono.just(Result.failure(preErrorsBuilder.build()));
        } else {
            PreValidatedReserveTransfer.Builder preValidatedParametersBuilder
                    = PreValidatedReserveTransfer.builder();
            validationService.preValidateReserveTransferParameters(transferRequest.getParameters().get(),
                    preValidatedParametersBuilder, preErrorsBuilder, locale, false);
            if (preErrorsBuilder.hasAnyErrors()) {
                return Mono.just(Result.failure(preErrorsBuilder.build()));
            }
            PreValidatedReserveTransfer preValidatedParameters = preValidatedParametersBuilder.build();

            return transferRequestReserveService.validateReserveTransfer(txSession, preValidatedParameters,
                    currentUser, locale, false).flatMap(r -> r.andThenMono(parameters ->
                    validateReserveTransferApplication(txSession, parameters, locale).flatMap(preApplicationR ->
                            preApplicationR.applyMono(preApplication -> getResult(
                                    txSession, transferRequest, currentUser, parameters, preApplication)
                            ))));
        }
    }

    public Mono<Result<Tuple2<QuotaTransferPreApplicationData, DryRunTransferRequestWarnings>>>
    validateReserveTransferApplication(YdbTxSession txSession, ValidatedReserveTransferParameters parameters,
                                       Locale locale) {
        String reserveFolderId = parameters.getTransfer().getReserveFolder().getId();
        String destinationFolderId = parameters.getTransfer().getDestinationFolder().getId();
        Set<String> folderIds = Set.of(reserveFolderId, destinationFolderId);
        return storeService.loadQuotas(txSession, folderIds).map(quotas -> {
            Map<String, Map<String, QuotaModel>> quotasByFolderResource = new HashMap<>();
            quotas.forEach(quota -> quotasByFolderResource.computeIfAbsent(quota.getFolderId(),
                    k -> new HashMap<>()).put(quota.getResourceId(), quota));
            Map<String, List<String>> warningsPerResource = new HashMap<>();
            Map<String, Map<String, Object>> detailsPerResource = new HashMap<>();
            Map<String, DryRunTransferRequestFolderWarnings> warningsPerFolder = new HashMap<>();
            ErrorCollection.Builder errors = ErrorCollection.builder();
            parameters.getTransfer().getTransfers().forEach(resourceTransfer -> {
                QuotaModel currentReserveQuota = quotasByFolderResource
                        .getOrDefault(reserveFolderId, Collections.emptyMap())
                        .getOrDefault(resourceTransfer.getResource().getId(), null);
                long currentReserveBalance = getBalance(currentReserveQuota);
                long currentReserveAmount = getAmount(currentReserveQuota);
                long currentReserveProvided = currentReserveAmount != 0L ?
                        (currentReserveAmount - currentReserveBalance) : 0L;
                long delta = resourceTransfer.getDelta();
                if (delta > currentReserveBalance) {
                    String message = transferRequestErrorMessages.getNotEnoughBalanceErrorMessage(
                            resourceTransfer, currentReserveBalance, currentReserveProvided, locale);
                    warningsPerResource.computeIfAbsent(resourceTransfer.getResource().getId(),
                            rid -> new ArrayList<>())
                            .add(message);
                    DryRunTransferRequestFolderWarnings folderWarnings =
                            warningsPerFolder.computeIfAbsent(reserveFolderId,
                                    fid -> new DryRunTransferRequestFolderWarnings(new ArrayList<>(),
                                            new HashMap<>(), new HashMap<>()));
                    folderWarnings.getMessages().add(message);
                    folderWarnings.getPerResource().computeIfAbsent(resourceTransfer.getResource().getId(),
                            rid -> new ArrayList<>())
                            .add(message);

                    if (currentReserveBalance > 0) {
                        AmountDto suggestedAmount = QuotasHelper.getAmountDto(
                                QuotasHelper.toBigDecimal(currentReserveBalance),
                                resourceTransfer.getResource(), resourceTransfer.getResourceUnitsEnsemble(), locale);
                        Map<String, Object> details = Map.of("suggestedAmount", suggestedAmount,
                                "suggestedAmountPrompt", messages.getMessage(
                                        "errors.not.enough.balance.for.quota.transfer.non.zero.balance.button",
                                        null, locale));
                        detailsPerResource.computeIfAbsent(resourceTransfer.getResource().getId(),
                                rid -> new HashMap<>())
                                .putAll(details);
                        folderWarnings.getDetailsPerResource().computeIfAbsent(resourceTransfer.getResource().getId(),
                                rid -> new HashMap<>())
                                .putAll(details);
                    }
                }

                QuotaModel currentQuota = quotasByFolderResource
                        .getOrDefault(destinationFolderId, Collections.emptyMap())
                        .getOrDefault(resourceTransfer.getResource().getId(), null);

                long currentBalance = getBalance(currentQuota);
                long currentAmount = getAmount(currentQuota);
                if (Units.add(currentAmount, delta).isEmpty()
                        || Units.add(currentBalance, delta).isEmpty()) {
                    errors.addError("parameters.reserveTransfer.resourceTransfers." + resourceTransfer.getIndex()
                            + ".delta", TypedError.invalid(messages.getMessage("errors.number.out.of.range", null,
                            locale)));
                }
            });
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
            QuotaTransferPreApplicationData quotaTransferData =
                    QuotaTransferPreApplicationData.builder().addQuotas(quotas).build();
            return Result.success(Tuples.of(quotaTransferData, new DryRunTransferRequestWarnings(new ArrayList<>(),
                    warningsPerResource, warningsPerFolder, detailsPerResource, Map.of())));
        });
    }

    private Mono<DryRunTransferRequestResult> getResult(
            YdbTxSession txSession,
            FrontDryRunCreateTransferRequestDto transferRequest, YaUserDetails currentUser,
            ValidatedReserveTransferParameters parameters,
            Tuple2<QuotaTransferPreApplicationData, DryRunTransferRequestWarnings> preApplication) {

        if (transferRequest.isPrepareResponsible() || transferRequest.isPreparePermissions()) {
            return transferRequestResponsibleAndNotifyService.calculateForReserveTransferCreate(
                    txSession, parameters, currentUser).map(responsibles -> prepareDryRunCreateQuotaTransferRequest(
                    parameters, preApplication, responsibles.getResponsible(),
                    getPermissions(responsibles, currentUser)));
        }
        return Mono.just(prepareDryRunCreateQuotaTransferRequest(parameters, preApplication, null, null));
    }

    private DryRunTransferRequestResult prepareDryRunCreateQuotaTransferRequest(
            ValidatedReserveTransferParameters validatedParameters,
            Tuple2<QuotaTransferPreApplicationData, DryRunTransferRequestWarnings> preApplicationData,
            @Nullable TransferResponsible responsible,
            @Nullable DryRunTransferRequestPermissions permissions) {
        ValidatedReserveTransfer transfer = validatedParameters.getTransfer();
        long destinationServiceId = transfer.getDestinationService().getId();
        String destinationFolderId = transfer.getDestinationFolder().getId();
        String reserveFolderId = transfer.getReserveFolder().getId();
        long reserveServiceId = transfer.getReserveFolder().getServiceId();

        TransferParameters.Builder parametersBuilder = TransferParameters.builder();
        List<ValidatedQuotaResourceTransfer> transfers = transfer.getTransfers();
        parametersBuilder.addQuotaTransfer(getQuotaTransfer(destinationFolderId, destinationServiceId, transfers,
                true));
        parametersBuilder.addQuotaTransfer(getQuotaTransfer(reserveFolderId, reserveServiceId, transfers, false));

        List<QuotaModel> quotas = preApplicationData.getT1().getQuotas();
        TransferParameters transferParameters = parametersBuilder.build();
        List<QuotaModel> updatedQuotaModels = getUpdatedQuotaModels(quotas, transferParameters);
        Set<QuotaModel.Key> updatedQuotaKeys = getUpdatedQuotaKeys(updatedQuotaModels);

        return DryRunTransferRequestResult.builder()
                .folders(validatedParameters.getFolders())
                .services(validatedParameters.getServices())
                .resources(validatedParameters.getResources())
                .unitsEnsembles(validatedParameters.getUnitsEnsembles())
                .providers(List.of(transfer.getProvider()))
                .transferRequest(DryRunTransferRequestModel.builder()
                        .type(TransferRequestType.RESERVE_TRANSFER)
                        .quotasOld(getOldQuota(quotas, updatedQuotaKeys))
                        .quotasNew(getNewQuota(updatedQuotaModels))
                        .parameters(transferParameters)
                        .build())
                .responsible(responsible)
                .permissions(permissions)
                .warnings(preApplicationData.getT2())
                .build();
    }

    private DryRunTransferRequestResult prepareDryRunCreateProvisionTransferRequest(
            ValidatedProvisionTransferParameters validatedParameters,
            ValidatedProvisionTransferApplication preApplicationData,
            List<ResourceModel> relatedResources,
            List<UnitsEnsembleModel> relatedUnitsEnsembles,
            @Nullable TransferResponsible responsible,
            @Nullable DryRunTransferRequestPermissions permissions) {
        TransferParameters.Builder parametersBuilder = TransferParameters.builder();
        for (ValidatedProvisionTransfer validatedTransfer : validatedParameters.getProvisionTransfers()) {
            ProvisionTransfer provisionTransfer = toProvisionTransfer(validatedTransfer);
            parametersBuilder.addProvisionTransfer(provisionTransfer);
        }

        TransferParameters transferParameters = parametersBuilder.build();

        List<QuotaModel> quotas = preApplicationData.preApplicationData().getFolderQuotas();
        List<AccountsQuotasModel> accountQuotas = preApplicationData.preApplicationData().getAccountQuotas();
        Tuple2<List<AccountsQuotasModel>, List<QuotaModel>> updatedProvisionQuotas = getUpdatedProvisionQuotas(quotas,
                accountQuotas, transferParameters.getProvisionTransfers());
        Map<String, ProviderModel> providerById = indexBy(validatedParameters.getProviders(), ProviderModel::getId);
        Set<ProvisionTransfer> relatedResourcesTransfers = getRelatedResourcesTransfers(
                transferParameters.getProvisionTransfers(), accountQuotas, providerById);
        DryRunTransferParameters dryRunTransferParameters = new DryRunTransferParameters(relatedResourcesTransfers);

        Set<QuotaModel.Key> updatedQuotaKeys = getUpdatedQuotaKeys(updatedProvisionQuotas.getT2());
        Set<AccountsQuotasModel.Identity> updatedAccountQuotaKeys =
                getUpdatedAccountQuotaKeys(updatedProvisionQuotas.getT1());

        Map<String, UnitsEnsembleModel> unitsEnsembleById = indexBy(validatedParameters.getUnitsEnsembles(),
                UnitsEnsembleModel::getId);
        Map<String, ResourceModel> resourceById = indexBy(validatedParameters.getResources(),
                ResourceModel::getId);
        Map<String, AccountModel> accountsById = indexBy(validatedParameters.getAccounts(),
                AccountModel::getId);

        Function<AccountsQuotasModel, AccountQuotaDto> dtoMapper =
                a -> toAccountQuotaDto(accountsById, resourceById, unitsEnsembleById, a);

        Set<ResourceModel> resources = new HashSet<>();
        resources.addAll(validatedParameters.getResources());
        resources.addAll(relatedResources);
        Set<UnitsEnsembleModel> unitsEnsembles = new HashSet<>();
        unitsEnsembles.addAll(validatedParameters.getUnitsEnsembles());
        unitsEnsembles.addAll(relatedUnitsEnsembles);
        return DryRunTransferRequestResult.builder()
                .folders(validatedParameters.getFolders())
                .services(validatedParameters.getServices())
                .resources(List.copyOf(resources))
                .unitsEnsembles(List.copyOf(unitsEnsembles))
                .providers(validatedParameters.getProviders())
                .transferRequest(DryRunTransferRequestModel.builder()
                        .type(TransferRequestType.PROVISION_TRANSFER)
                        .quotasOld(getOldQuota(quotas, updatedQuotaKeys))
                        .quotasNew(getNewQuota(updatedProvisionQuotas.getT2()))
                        .accountQuotasOld(getOldAccountQuota(accountQuotas, updatedAccountQuotaKeys, dtoMapper))
                        .accountQuotasNew(getNewAccountQuota(updatedProvisionQuotas.getT1(), dtoMapper))
                        .parameters(transferParameters)
                        .dryRunParameters(dryRunTransferParameters)
                        .build())
                .responsible(responsible)
                .permissions(permissions)
                .warnings(preApplicationData.warnings())
                .errors(preApplicationData.errors())
                .build();
    }

    private static Set<QuotaModel.Key> getUpdatedQuotaKeys(List<QuotaModel> updatedQuotaModels) {
        return updatedQuotaModels.stream()
                .map(QuotaModel::toKey)
                .collect(Collectors.toSet());
    }

    private static Set<AccountsQuotasModel.Identity> getUpdatedAccountQuotaKeys(
            List<AccountsQuotasModel> updateAccountQuotas
    ) {
        return updateAccountQuotas.stream()
                .map(AccountsQuotasModel::getIdentity)
                .collect(Collectors.toSet());
    }

    private List<QuotaDto> getOldQuota(List<QuotaModel> quotas, Set<QuotaModel.Key> updatedQuotaKeys) {
        return quotas.stream()
                .filter(q -> updatedQuotaKeys.contains(q.toKey()))
                .map(QuotaDto::new)
                .collect(Collectors.toList());
    }

    private List<QuotaDto> getNewQuota(List<QuotaModel> updatedQuotaModels) {
        return updatedQuotaModels.stream()
                .map(QuotaDto::new)
                .collect(Collectors.toList());
    }

    private static List<AccountQuotaDto> getOldAccountQuota(List<AccountsQuotasModel> quotas,
                                                            Set<AccountsQuotasModel.Identity> updatedQuotaKeys,
                                                            Function<AccountsQuotasModel, AccountQuotaDto> dtoMapper) {
        return quotas.stream()
                .filter(q -> updatedQuotaKeys.contains(q.getIdentity()))
                .map(dtoMapper)
                .collect(Collectors.toList());
    }

    private static List<AccountQuotaDto> getNewAccountQuota(List<AccountsQuotasModel> quotas,
                                                            Function<AccountsQuotasModel, AccountQuotaDto> dtoMapper) {
        return quotas.stream()
                .map(dtoMapper)
                .collect(Collectors.toList());
    }

    public List<QuotaModel> getUpdatedQuotaModels(List<QuotaModel> quotas,
                                                  TransferParameters transferParameters) {
        Map<String, Map<String, QuotaModel>> quotasByFolderResource = new HashMap<>();
        quotas.forEach(q -> quotasByFolderResource.computeIfAbsent(q.getFolderId(), k -> new HashMap<>())
                .put(q.getResourceId(), q));
        Set<QuotaTransfer> quotaTransfers = transferParameters.getQuotaTransfers();
        List<QuotaModel> updatedQuotas = new ArrayList<>();
        for (QuotaTransfer quotaTransfer : quotaTransfers) {
            for (ResourceQuotaTransfer resourceTransfer : quotaTransfer.getTransfers()) {
                updatedQuotas.add(updateQuota(resourceTransfer, quotaTransfer.getDestinationFolderId(),
                        quotasByFolderResource));
            }
        }
        return updatedQuotas;
    }

    public Set<ProvisionTransfer> getRelatedResourcesTransfers(
            Set<ProvisionTransfer> transfers,
            List<AccountsQuotasModel> accountsQuotas,
            Map<String, ProviderModel> providerById
    ) {
        Set<ProvisionTransfer> result = new HashSet<>();
        Map<String, Map<String, AccountsQuotasModel>> accountQuotaByAccountIdResourceId = indexBy(accountsQuotas,
                AccountsQuotasModel::getAccountId, AccountsQuotasModel::getResourceId);
        for (ProvisionTransfer transfer : transfers) {
            ProviderModel provider = providerById.get(transfer.getProviderId());
            Map<String, RelatedResourceMapping> relatedResourceMappingByResourceId =
                    provider.getRelatedResourcesByResourceId().orElse(Map.of());
            Map<String, BigDecimal> relatedResourceDeltaById = new HashMap<>();
            Set<ResourceQuotaTransfer> destinationAccountTransfers = transfer.getDestinationAccountTransfers();
            for (ResourceQuotaTransfer destinationAccountTransfer : destinationAccountTransfers) {
                String resourceId = destinationAccountTransfer.getResourceId();
                RelatedResourceMapping relatedResourceMapping = relatedResourceMappingByResourceId.get(resourceId);
                if (relatedResourceMapping == null) {
                    continue;
                }
                Map<String, RelatedCoefficient> relatedCoefficientMap =
                        relatedResourceMapping.getRelatedCoefficientMap();
                relatedCoefficientMap.forEach((relatedResourceId, coef) -> {
                    BigDecimal delta = BigDecimal.valueOf(destinationAccountTransfer.getDelta());
                    BigDecimal numerator = BigDecimal.valueOf(coef.getNumerator());
                    BigDecimal denominator = BigDecimal.valueOf(coef.getDenominator());
                    BigDecimal relatedResourceDelta = delta.multiply(numerator).divide(denominator, RoundingMode.UP);
                    relatedResourceDeltaById.merge(relatedResourceId, relatedResourceDelta, BigDecimal::add);
                });
            }
            Set<ResourceQuotaTransfer> relatedSourceTransfers = new HashSet<>();
            Set<ResourceQuotaTransfer> relatedDestinationTransfers = new HashSet<>();
            relatedResourceDeltaById.forEach((relatedResourceId, delta) -> {
                boolean deltaIsPositive = delta.signum() > 0;
                String accountId = deltaIsPositive ? transfer.getSourceAccountId()
                        : transfer.getDestinationAccountId();
                AccountsQuotasModel sourceQuota = accountQuotaByAccountIdResourceId.getOrDefault(accountId,
                        Map.of()).get(relatedResourceId);
                if (!deltaIsPositive) {
                    delta = delta.negate();
                }
                long newDelta = getRelatedResourceDeltaWithQuota(provider, delta, sourceQuota);
                if (newDelta != 0L) {
                    relatedSourceTransfers.add(new ResourceQuotaTransfer(relatedResourceId, -newDelta));
                    relatedDestinationTransfers.add(new ResourceQuotaTransfer(relatedResourceId, newDelta));
                }
            });
            if (!relatedSourceTransfers.isEmpty()) {
                result.add(new ProvisionTransfer(transfer.getSourceAccountId(), transfer.getDestinationAccountId(),
                        transfer.getSourceFolderId(), transfer.getDestinationFolderId(), transfer.getSourceServiceId(),
                        transfer.getDestinationServiceId(), relatedSourceTransfers, relatedDestinationTransfers,
                        transfer.getProviderId(), transfer.getAccountsSpacesId(), transfer.getOperationId()));
            }
        }
        return result;
    }

    public long getRelatedResourceDeltaWithQuota(
            ProviderModel provider,
            BigDecimal relatedDelta,
            @Nullable AccountsQuotasModel quota
    ) {
        boolean isAllocatedSupported = provider.isAllocatedSupported().orElse(false);
        long provided = getProvided(quota);
        long allocated = isAllocatedSupported ? getAllocated(quota) : 0L;
        long frozenQuota = getFrozenQuota(quota);
        long available = Math.max(0, provided - allocated - frozenQuota);
        long delta = relatedDelta.longValueExact();
        return Math.min(delta, available);
    }

    public Tuple2<List<AccountsQuotasModel>, List<QuotaModel>> getUpdatedProvisionQuotas(
            List<QuotaModel> folderQuotas,
            List<AccountsQuotasModel> accountQuotas,
            Set<ProvisionTransfer> provisionTransfers) {
        Map<String, Map<String, AccountsQuotasModel>> quotasByAccountResource = indexBy(accountQuotas,
                AccountsQuotasModel::getAccountId, AccountsQuotasModel::getResourceId);
        Map<String, Map<String, QuotaModel>> quotasByFolderResource = indexBy(folderQuotas, QuotaModel::getFolderId,
                QuotaModel::getResourceId);

        Set<QuotaModel.Key> updatedQuotaKeys = new HashSet<>();
        Set<AccountsQuotasModel.Identity> updatedAccountQuotaKeys = new HashSet<>();
        String fakeOperationId = UUID.randomUUID().toString();
        for (ProvisionTransfer provisionTransfer : provisionTransfers) {
            Map<String, Map<String, QuotaModel>> updatedQuotasByFolderResource = new HashMap<>();
            Map<String, Set<ResourceQuotaTransfer>> accountTransfers = Map.of(
                    provisionTransfer.getSourceAccountId(), provisionTransfer.getSourceAccountTransfers(),
                    provisionTransfer.getDestinationAccountId(), provisionTransfer.getDestinationAccountTransfers()
            );
            Map<String, String> folderByAccountId = Map.of(
                    provisionTransfer.getSourceAccountId(), provisionTransfer.getSourceFolderId(),
                    provisionTransfer.getDestinationAccountId(), provisionTransfer.getDestinationFolderId()
            );
            accountTransfers.forEach((accountId, transfers) -> transfers.forEach(transfer -> {
                String folderId = folderByAccountId.get(accountId);
                String anotherFolderId = folderId.equals(provisionTransfer.getSourceFolderId()) ?
                        provisionTransfer.getDestinationFolderId() :
                        provisionTransfer.getSourceFolderId();
                AccountsQuotasModel updatedAccountQuota = updateAccountQuota(transfer, accountId, folderId,
                        fakeOperationId, quotasByAccountResource);
                updatedAccountQuotaKeys.add(updatedAccountQuota.getIdentity());
                quotasByAccountResource.computeIfAbsent(accountId, x -> new HashMap<>())
                        .put(transfer.getResourceId(), updatedAccountQuota);


                QuotaModel updatedQuota = updateQuotaForProvisionTransfer(transfer, folderId,
                        anotherFolderId, provisionTransfer.getProviderId(), quotasByFolderResource);
                updatedQuotaKeys.add(updatedQuota.toKey());
                updatedQuotasByFolderResource.computeIfAbsent(folderId, x -> new HashMap<>())
                        .put(transfer.getResourceId(), updatedQuota);
            }));
            updatedQuotasByFolderResource.forEach((folderId, updatedQuotasByResourceId) ->
                    updatedQuotasByResourceId.forEach((resourceId, updatedQuota) ->
                            quotasByFolderResource.computeIfAbsent(folderId, x -> new HashMap<>())
                                    .put(resourceId, updatedQuota)
            ));
        }
        List<QuotaModel> updatedQuotas = updatedQuotaKeys.stream()
                .map(k -> quotasByFolderResource.get(k.getFolderId()).get(k.getResourceId()))
                .collect(Collectors.toList());
        List<AccountsQuotasModel> updatedAccountQuotas = updatedAccountQuotaKeys.stream()
                .map(k -> quotasByAccountResource.get(k.getAccountId()).get(k.getResourceId()))
                .collect(Collectors.toList());
        return Tuples.of(updatedAccountQuotas, updatedQuotas);
    }

    private static QuotaModel updateQuota(
            ResourceQuotaTransfer resourceTransfer,
            String folderId,
            Map<String, Map<String, QuotaModel>> quotasByFolderResource
    ) {
        QuotaModel currentQuota = quotasByFolderResource
                .getOrDefault(folderId, Collections.emptyMap())
                .get(resourceTransfer.getResourceId());
        if (currentQuota != null) {
            return QuotaModel.builder(currentQuota)
                    .quota((currentQuota.getQuota() != null ? currentQuota.getQuota() : 0L)
                            + resourceTransfer.getDelta())
                    .balance((currentQuota.getBalance() != null ? currentQuota.getBalance() : 0L)
                            + resourceTransfer.getDelta())
                    .build();
        } else {
            return QuotaModel.builder()
                    .tenantId(Tenants.DEFAULT_TENANT_ID)
                    .folderId(folderId)
                    .resourceId(resourceTransfer.getResourceId())
                    .providerId(resourceTransfer.getResourceId())
                    .quota(resourceTransfer.getDelta())
                    .balance(resourceTransfer.getDelta())
                    .frozenQuota(0L)
                    .build();
        }
    }

    private static QuotaModel updateQuotaForProvisionTransfer(
            ResourceQuotaTransfer resourceTransfer,
            String folderId,
            String anotherFolderId,
            String providerId,
            Map<String, Map<String, QuotaModel>> quotasByFolderResource
    ) {
        QuotaModel currentQuota = quotasByFolderResource
                .getOrDefault(folderId, Collections.emptyMap())
                .get(resourceTransfer.getResourceId());
        QuotaModel anotherFolderQuota = quotasByFolderResource
                .getOrDefault(anotherFolderId, Collections.emptyMap())
                .get(resourceTransfer.getResourceId());
        if (currentQuota == null) {
            currentQuota = emptyQuotaModel(
                    folderId,
                    resourceTransfer.getResourceId(),
                    providerId
            );
        }
        if (anotherFolderQuota == null) {
            anotherFolderQuota = emptyQuotaModel(
                    folderId,
                    resourceTransfer.getResourceId(),
                    providerId
            );
        }
        return calculateNewQuotaModel(resourceTransfer.getDelta(), currentQuota, anotherFolderQuota);
    }

    private static QuotaModel emptyQuotaModel(
            String folderId,
            String resourceId,
            String providerId
    ) {
        return QuotaModel.builder()
                .tenantId(Tenants.DEFAULT_TENANT_ID)
                .folderId(folderId)
                .resourceId(resourceId)
                .providerId(providerId)
                .quota(0L)
                .balance(0L)
                .frozenQuota(0L)
                .build();
    }

    private static QuotaModel calculateNewQuotaModel(
            long delta,
            QuotaModel quotaModel,
            QuotaModel anotherFolderQuota
    ) {
        long updatedQuota = delta < 0 ?
                Math.max(quotaModel.getQuota() + delta, 0) :
                quotaModel.getQuota() + Math.min(delta, anotherFolderQuota.getQuota());
        long quotaDelta = updatedQuota - quotaModel.getQuota();
        long balanceDelta = quotaDelta - delta;
        long balance = quotaModel.getBalance() + balanceDelta;
        return QuotaModel.builder(quotaModel)
                .balance(balance)
                .quota(updatedQuota)
                .build();
    }

    private static AccountsQuotasModel updateAccountQuota(
            ResourceQuotaTransfer resourceTransfer,
            String accountId,
            String folderId,
            String fakeOperationId,
            Map<String, Map<String, AccountsQuotasModel>> quotasByAccountResource
    ) {
        AccountsQuotasModel currentQuota = quotasByAccountResource
                .getOrDefault(accountId, Collections.emptyMap())
                .get(resourceTransfer.getResourceId());
        if (currentQuota != null) {
            return currentQuota.copyBuilder()
                    .setProvidedQuota((currentQuota.getProvidedQuota() != null ? currentQuota.getProvidedQuota() : 0L)
                            + resourceTransfer.getDelta())
                    .build();
        } else {
            return new AccountsQuotasModel.Builder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setFolderId(folderId)
                    .setAccountId(accountId)
                    .setResourceId(resourceTransfer.getResourceId())
                    .setProviderId(resourceTransfer.getResourceId())
                    .setProvidedQuota(resourceTransfer.getDelta())
                    .setAllocatedQuota(0L)
                    .setFrozenProvidedQuota(0L)
                    .setLastProvisionUpdate(Instant.now())
                    .setLastReceivedProvisionVersion(0L)
                    .setLatestSuccessfulProvisionOperationId(fakeOperationId)
                    .build();
        }
    }

    private record ValidatedProvisionTransferApplication(
            ProvisionTransferPreApplicationData preApplicationData,
            DryRunTransferRequestWarnings warnings,
            ErrorCollection errors
    ) { }
}
