package ru.yandex.intranet.d.services.transfer;

import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.transfers.PendingTransferRequestsDao;
import ru.yandex.intranet.d.dao.transfers.TransferRequestsDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.metrics.TransferRequestRefreshMetrics;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.transfers.PendingTransferRequestsModel;
import ru.yandex.intranet.d.model.transfers.TransferNotified;
import ru.yandex.intranet.d.model.transfers.TransferRequestEventType;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryFields;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;
import ru.yandex.intranet.d.services.transfer.model.RefreshTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ResponsibleAndNotified;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestIndices;
import ru.yandex.intranet.d.services.transfer.ticket.TransferRequestTicketService;

/**
 * Transfer request refresh service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class RefreshTransferRequestService {

    private static final Logger LOG = LoggerFactory.getLogger(RefreshTransferRequestService.class);

    private final YdbTableClient tableClient;
    private final TransferRequestsDao transferRequestsDao;
    private final TransferRequestResponsibleAndNotifyService transferRequestResponsibleAndNotifyService;
    private final TransferRequestStoreService storeService;
    private final TransferRequestIndicesService indicesService;
    private final TransferRequestAnswerService transferRequestAnswerService;
    private final TransferRequestVoteService transferRequestVoteService;
    private final TransferRequestNotificationsService transferRequestNotificationsService;
    private final PendingTransferRequestsDao pendingTransferRequestsDao;
    private final TransferRequestRefreshMetrics transferRequestRefreshMetrics;
    private final TransferRequestTicketService transferRequestTicketService;

    @SuppressWarnings("ParameterNumber")
    public RefreshTransferRequestService(
            YdbTableClient tableClient,
            TransferRequestsDao transferRequestsDao,
            TransferRequestResponsibleAndNotifyService transferRequestResponsibleAndNotifyService,
            TransferRequestStoreService storeService,
            TransferRequestIndicesService indicesService,
            TransferRequestAnswerService transferRequestAnswerService,
            TransferRequestVoteService transferRequestVoteService,
            TransferRequestNotificationsService transferRequestNotificationsService,
            PendingTransferRequestsDao pendingTransferRequestsDao,
            TransferRequestRefreshMetrics transferRequestRefreshMetrics,
            TransferRequestTicketService transferRequestTicketService) {
        this.tableClient = tableClient;
        this.transferRequestsDao = transferRequestsDao;
        this.transferRequestResponsibleAndNotifyService = transferRequestResponsibleAndNotifyService;
        this.storeService = storeService;
        this.indicesService = indicesService;
        this.transferRequestTicketService = transferRequestTicketService;
        this.transferRequestAnswerService = transferRequestAnswerService;
        this.transferRequestVoteService = transferRequestVoteService;
        this.transferRequestNotificationsService = transferRequestNotificationsService;
        this.pendingTransferRequestsDao = pendingTransferRequestsDao;
        this.transferRequestRefreshMetrics = transferRequestRefreshMetrics;
    }

    public Mono<Void> refresh() {
        LOG.info("Running refresh transfer requests...");
        return loadPending()
                .flatMap(pending ->
                        Flux.fromIterable(pending)
                                .concatMap(this::refreshSinglePendingTransfer)
                                .collectList()
                ).doOnNext(this::refreshStats)
                .doOnError(this::refreshStatsOnError)
                .then();
    }

    private void refreshStats(List<Boolean> results) {
        long successes = 0L;
        long failures = 0L;
        for (boolean result : results) {
            if (result) {
                successes++;
            } else {
                failures++;
            }
        }
        transferRequestRefreshMetrics.updateRefreshTransferRequestResultMetrics(successes, failures);
        LOG.info("Refresh transfer requests finished. {} successful, {} failed.", successes, failures);
    }

    private void refreshStatsOnError(Throwable e) {
        transferRequestRefreshMetrics.updateRefreshTransferRequestResultMetrics(0, 1);
        LOG.info("Refresh transfer requests failed.", e);
    }

    private Mono<List<PendingTransferRequestsModel>> loadPending() {
        return tableClient.usingSessionMonoRetryable(session -> pendingTransferRequestsDao
                .getAllByTenant(session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY),
                        Tenants.DEFAULT_TENANT_ID));
    }

    private Mono<Boolean> refreshSinglePendingTransfer(PendingTransferRequestsModel pendingTransfer) {
        return refreshTransferRequest(pendingTransfer).flatMap(tr -> {
                    if (tr.isEmpty()) {
                        return Mono.empty();
                    }
                    RefreshTransferRequest transfer = tr.get();
                    return expandTransferRequest(transfer)
                            .flatMap(expanded -> updateTrackerIssue(expanded, transfer))
                            .doOnError(e -> LOG.error("Failed to refresh tracker issue for transfer request "
                                    + pendingTransfer.getRequestId(), e))
                            .onErrorResume(e -> Mono.empty());
                }).thenReturn(true)
                .doOnError(e -> LOG.error("Failed to refresh transfer request " + pendingTransfer.getRequestId(), e))
                .onErrorReturn(false);
    }

    private Mono<Optional<RefreshTransferRequest>> refreshTransferRequest(
            PendingTransferRequestsModel pendingTransfer) {
        return tableClient.usingSessionMonoRetryable(session ->
                session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession -> {
                    String transferId = pendingTransfer.getRequestId();
                    TenantId tenantId = pendingTransfer.getTenantId();
                    return transferRequestsDao.getById(txSession, transferId, tenantId).flatMap(transferO -> {
                        if (transferO.isEmpty() || !transferO.get().getStatus().equals(TransferRequestStatus.PENDING)) {
                            return Mono.just(Optional.empty());
                        }
                        return refreshTransferRequest(txSession, transferO.get());
                    });
                })
        );
    }

    private Mono<Optional<RefreshTransferRequest>> refreshTransferRequest(YdbTxSession ts,
                                                                          TransferRequestModel transfer) {
        return transferRequestResponsibleAndNotifyService.
                calculateForTransferRequestModel(ts, transfer, true).flatMap(newResp ->
                        refreshTransferRequest(ts, transfer, newResp))
                .doOnSuccess(p -> p.ifPresent(prepared -> sendNotifications(transfer, prepared.getNotifiedUsers())));
    }

    @NotNull
    public Mono<Optional<RefreshTransferRequest>> refreshTransferRequest(YdbTxSession ts,
                                                                         TransferRequestModel transfer,
                                                                         ResponsibleAndNotified newResponsible) {
        if (newResponsible.getResponsible().equals(transfer.getResponsible()) ||
                transfer.getStatus() != TransferRequestStatus.PENDING) {
            return Mono.just(Optional.empty());
        }
        return indicesService.loadTransferRequestIndices(ts, transfer).flatMap(currentIndices -> {
            RefreshTransferRequest prepared = prepareRefreshTransfer(transfer, currentIndices, newResponsible);
            return storeService.saveRefresh(ts, prepared)
                    .thenReturn(Optional.of(prepared));
        });
    }

    private Mono<ExpandedTransferRequests<TransferRequestModel>> expandTransferRequest(
            RefreshTransferRequest transfer) {
        return tableClient.usingSessionMonoRetryable(session ->
                transferRequestAnswerService.expand(session, transfer.getRefreshedTransferRequest()));
    }

    private Mono<Void> updateTrackerIssue(ExpandedTransferRequests<TransferRequestModel> expandedTransfer,
                                          RefreshTransferRequest transfer) {
        return transferRequestTicketService.updateTicket(expandedTransfer, transfer).flatMap(r -> r.match(t -> {
                    TransferRequestModel refreshedTransfer = transfer.getRefreshedTransferRequest();
                    if (refreshedTransfer.getStatus().equals(TransferRequestStatus.REJECTED)) {
                        return transferRequestTicketService.closeTicket(refreshedTransfer, t).flatMap(s -> s.match(
                                u -> Mono.empty(),
                                e -> {
                                    LOG.error("Failed to close tracker issue for transfer request {}: {}",
                                            transfer.getRefreshedTransferRequest().getId(), e);
                                    return Mono.empty();
                                }
                        ));
                    }
                    return Mono.empty();
                },
                e -> {
                    LOG.error("Failed to refresh tracker issue for transfer request {}: {}",
                            transfer.getRefreshedTransferRequest().getId(), e);
                    return Mono.empty();
                }));
    }

    private RefreshTransferRequest prepareRefreshTransfer(TransferRequestModel transferRequest,
                                                          TransferRequestIndices currentTransferRequestIndices,
                                                          ResponsibleAndNotified newResponsibleAndNotified) {
        Instant now = Instant.now();

        RefreshTransferRequest.Builder builder = RefreshTransferRequest.builder();
        boolean unableToConfirm = transferRequestVoteService.pendingTransferRequestUnableToConfirm(
                newResponsibleAndNotified.getResponsible(),
                transferRequest.getVotes(),
                Collections.emptySet()
        );
        TransferRequestStatus targetStatus = unableToConfirm ?
                TransferRequestStatus.REJECTED : TransferRequestStatus.PENDING;
        TransferRequestModel refreshedTransferRequest = TransferRequestModel.builder(transferRequest)
                .version(transferRequest.getVersion() + 1)
                .responsible(newResponsibleAndNotified.getResponsible())
                .status(targetStatus)
                .nextHistoryOrder(transferRequest.getNextHistoryOrder() + 1)
                .updatedAt(now)
                .transferNotified(TransferNotified.from(newResponsibleAndNotified))
                .build();
        builder.refreshedTransferRequest(refreshedTransferRequest);
        TransferRequestIndices newTransferRequestIndices =
                TransferRequestIndicesService.calculateTransferRequestIndices(refreshedTransferRequest);
        TransferRequestIndices.Difference difference =
                TransferRequestIndicesService.difference(currentTransferRequestIndices, newTransferRequestIndices);
        builder.indicesDifference(difference);
        builder.history(generateHistory(transferRequest, refreshedTransferRequest));
        builder.notified(newResponsibleAndNotified.getNotifiedUsers());

        return builder.build();
    }

    private TransferRequestHistoryModel generateHistory(TransferRequestModel oldTransferRequest,
                                                        TransferRequestModel refreshedTransferRequest) {
        TransferRequestHistoryModel.Builder refreshHistoryBuilder = TransferRequestHistoryModel.builder();
        refreshHistoryBuilder.id(UUID.randomUUID().toString());
        refreshHistoryBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        refreshHistoryBuilder.transferRequestId(oldTransferRequest.getId());
        refreshHistoryBuilder.timestamp(refreshedTransferRequest.getUpdatedAt().orElseThrow());
        TransferRequestHistoryFields.Builder oldFields = TransferRequestHistoryFields.builder()
                .version(oldTransferRequest.getVersion());
        TransferRequestHistoryFields.Builder newFields = TransferRequestHistoryFields.builder()
                .version(refreshedTransferRequest.getVersion());
        if (refreshedTransferRequest.getStatus().equals(TransferRequestStatus.PENDING)) {
            refreshHistoryBuilder.type(TransferRequestEventType.UPDATED);
        } else {
            refreshHistoryBuilder.type(TransferRequestEventType.REJECTED);
            oldFields.status(oldTransferRequest.getStatus());
            newFields.status(refreshedTransferRequest.getStatus());
        }
        refreshHistoryBuilder.oldFields(oldFields.build());
        refreshHistoryBuilder.newFields(newFields.build());
        refreshHistoryBuilder.oldResponsible(oldTransferRequest.getResponsible());
        refreshHistoryBuilder.newResponsible(refreshedTransferRequest.getResponsible());
        refreshHistoryBuilder.order(oldTransferRequest.getNextHistoryOrder());
        return refreshHistoryBuilder.build();
    }

    public void sendNotifications(TransferRequestModel currentTransferRequest, Set<UserModel> newNotifiedUsers) {
        try {
            Set<UserModel> userModels = TransferRequestResponsibleAndNotifyService
                    .excludeAlreadyNotifiedUsers(currentTransferRequest, newNotifiedUsers);
            transferRequestNotificationsService.sendNotifications(userModels, currentTransferRequest.getId());
        } catch (Exception e) {
            LOG.error("Failed to send notifications after refresh", e);
        }
    }

}
