package ru.yandex.intranet.d.services.transfer

import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.loans.LoansDao
import ru.yandex.intranet.d.dao.loans.LoansHistoryDao
import ru.yandex.intranet.d.dao.loans.PendingLoansDao
import ru.yandex.intranet.d.dao.loans.ServiceLoansBalanceDao
import ru.yandex.intranet.d.dao.loans.ServiceLoansInDao
import ru.yandex.intranet.d.dao.loans.ServiceLoansOutDao
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.kotlin.orFalse
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountReserveType
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.loans.LoanActionSubject
import ru.yandex.intranet.d.model.loans.LoanActionSubjects
import ru.yandex.intranet.d.model.loans.LoanAmount
import ru.yandex.intranet.d.model.loans.LoanAmounts
import ru.yandex.intranet.d.model.loans.LoanDueDate
import ru.yandex.intranet.d.model.loans.LoanEventType
import ru.yandex.intranet.d.model.loans.LoanModel
import ru.yandex.intranet.d.model.loans.LoanStatus
import ru.yandex.intranet.d.model.loans.LoanSubject
import ru.yandex.intranet.d.model.loans.LoanSubjectType
import ru.yandex.intranet.d.model.loans.LoanType
import ru.yandex.intranet.d.model.loans.LoansHistoryFields
import ru.yandex.intranet.d.model.loans.LoansHistoryKey
import ru.yandex.intranet.d.model.loans.LoansHistoryModel
import ru.yandex.intranet.d.model.loans.PendingLoanKey
import ru.yandex.intranet.d.model.loans.PendingLoanModel
import ru.yandex.intranet.d.model.loans.ServiceLoanBalanceKey
import ru.yandex.intranet.d.model.loans.ServiceLoanBalanceModel
import ru.yandex.intranet.d.model.loans.ServiceLoanInModel
import ru.yandex.intranet.d.model.loans.ServiceLoanOutModel
import ru.yandex.intranet.d.model.services.ServiceMinimalModel
import ru.yandex.intranet.d.model.transfers.LoanMeta
import ru.yandex.intranet.d.model.transfers.LoanOperationType
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.model.transfers.VoteType
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision
import ru.yandex.intranet.d.services.transfer.model.PreValidatedLoanBorrowParameters
import ru.yandex.intranet.d.services.transfer.model.PreValidatedLoanParameters
import ru.yandex.intranet.d.services.transfer.model.PreValidatedLoanPayOffParameters
import ru.yandex.intranet.d.services.transfer.model.ValidatedLoanBorrowParameters
import ru.yandex.intranet.d.services.transfer.model.ValidatedLoanParameters
import ru.yandex.intranet.d.services.transfer.model.ValidatedLoanPayOffParameters
import ru.yandex.intranet.d.services.transfer.model.ValidatedProvisionTransferParameters
import ru.yandex.intranet.d.util.Uuids
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanParametersDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.math.BigInteger
import java.time.Instant
import java.time.LocalDate
import java.time.ZoneId
import java.util.*

private val logger = KotlinLogging.logger {}

/**
 * Transfer request loans service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class TransferLoansService(
    private val loansDao: LoansDao,
    private val loansHistoryDao: LoansHistoryDao,
    private val pendingLoansDao: PendingLoansDao,
    private val serviceLoansInDao: ServiceLoansInDao,
    private val serviceLoansOutDao: ServiceLoansOutDao,
    private val serviceLoansBalanceDao: ServiceLoansBalanceDao,
    @Qualifier("messageSource") private val messages: MessageSource
) {

    fun preValidateLoanFields(loanParameters: FrontTransferLoanParametersDto?, errors: ErrorCollection.Builder,
                              currentUser: YaUserDetails,  locale: Locale): PreValidatedLoanParameters? {
        if (loanParameters == null) {
            return null
        }
        if (loanParameters.borrowParameters == null && loanParameters.payOffParameters == null) {
            if (loanParameters.provideOverCommitReserve.orFalse()) {
                errors.addError("parameters.loanParameters", TypedError.invalid(
                        messages.getMessage("errors.invalid.missing.borrow.parameters.on.reserve.provision",
                            null, locale)))
            }
            return null
        }
        if (loanParameters.borrowParameters != null && loanParameters.payOffParameters != null) {
            errors.addError("loanParameters", TypedError.invalid(messages
                .getMessage("errors.simultaneous.borrow.and.pay.off.not.supported", null, locale)))
            return null
        }
        if (loanParameters.borrowParameters != null) {
            val dueDate = loanParameters.borrowParameters.dueDate
            return if (dueDate == null) {
                errors.addError("loanParameters.borrowParameters.dueDate", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)))
                null
            } else {
                val currentLocalDate = currentLocalDate(currentUser)
                if (dueDate.isBefore(currentLocalDate)) {
                    errors.addError("loanParameters.borrowParameters.dueDate", TypedError.invalid(messages
                            .getMessage("errors.loan.due.day.can.not.be.in.the.past", null, locale)))
                    null
                } else {
                    PreValidatedLoanParameters(PreValidatedLoanBorrowParameters(dueDate), null,
                        loanParameters.provideOverCommitReserve)
                }
            }
        }
        if (loanParameters.payOffParameters != null) {
            val loanId = loanParameters.payOffParameters.loanId
            return if (loanId == null) {
                errors.addError("loanParameters.borrowParameters.loanId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                null
            } else {
                if (!Uuids.isValidUuid(loanId)) {
                    errors.addError("loanParameters.borrowParameters.loanId", TypedError.invalid(messages
                        .getMessage("errors.loan.not.found", null, locale)))
                    null
                } else {
                    PreValidatedLoanParameters(null, PreValidatedLoanPayOffParameters(loanId),
                        loanParameters.provideOverCommitReserve)
                }
            }
        }
        return null
    }

    suspend fun validateLoanFields(txSession: YdbTxSession, loanParameters: PreValidatedLoanParameters?,
                                   transferParameters: ValidatedProvisionTransferParameters,
                                   currentUser: YaUserDetails, locale: Locale): Result<ValidatedLoanParameters?> {
        if (loanParameters == null) {
            return Result.success(null)
        }
        return if (loanParameters.borrowParameters != null) {
            validateLoanBorrowFields(loanParameters, loanParameters.borrowParameters, transferParameters,
                currentUser, locale)
        } else if (loanParameters.payOffParameters != null) {
            validateLoanPayOffFields(txSession, loanParameters, loanParameters.payOffParameters, transferParameters,
                locale)
        } else {
            throw IllegalArgumentException("Invalid loan parameters $loanParameters")
        }
    }

    suspend fun processLoan(session: YdbTxSession,
                            transferRequest: TransferRequestModel,
                            now: Instant,
                            sourceAccount: AccountModel,
                            destinationAccount: AccountModel,
                            sourceFolder: FolderModel,
                            destinationFolder: FolderModel,
                            knownDestinationProvisionsByResourceId: Map<ResourceId, AccountsQuotasModel>,
                            receivedDestinationProvisionByResourceId: Map<ResourceId, ValidatedReceivedProvision>,
                            knownSourceProvisionsByResourceId: Map<ResourceId, AccountsQuotasModel>,
                            receivedSourceProvisionByResourceId: Map<ResourceId, ValidatedReceivedProvision>): LoanId? {
        if (transferRequest.loanMeta.isEmpty) {
            return null
        }
        val loanMeta = transferRequest.loanMeta.get()
        return when (loanMeta.operationType) {
            LoanOperationType.BORROW -> {
                processLoanBorrow(session, transferRequest, loanMeta, now, sourceAccount,
                    destinationAccount, sourceFolder, destinationFolder, knownDestinationProvisionsByResourceId,
                    receivedDestinationProvisionByResourceId)
            }
            LoanOperationType.PAY_OFF -> {
                processLoanPayOff(session, transferRequest, loanMeta, sourceAccount, destinationAccount,
                    knownSourceProvisionsByResourceId, receivedSourceProvisionByResourceId, now)
                null
            }
        }
    }

    suspend fun processLoanBorrow(
        session: YdbTxSession,
        transferRequest: TransferRequestModel,
        loanMeta: LoanMeta,
        now: Instant,
        sourceAccount: AccountModel,
        destinationAccount: AccountModel,
        sourceFolder: FolderModel,
        destinationFolder: FolderModel,
        knownDestinationProvisionsByResourceId: Map<ResourceId, AccountsQuotasModel>,
        receivedDestinationProvisionByResourceId: Map<ResourceId, ValidatedReceivedProvision>
    ): LoanId {
        val provideOverCommitReserve = loanMeta.provideOverCommitReserve ?: false
        val loanSource = getBorrowLoanSource(sourceFolder, sourceAccount, provideOverCommitReserve)
        val loanDestination = getBorrowLoanDestination(destinationFolder, destinationAccount)
        val loanAmounts = getLoanAmounts(transferRequest, destinationAccount, knownDestinationProvisionsByResourceId,
            receivedDestinationProvisionByResourceId)
        val approximateDueAt = approximateDueAtTimestamp(loanMeta.borrowDueDate!!)
        val newLoan = LoanModel(
            tenantId = Tenants.DEFAULT_TENANT_ID,
            id = UUID.randomUUID().toString(),
            status = LoanStatus.PENDING,
            type = getBorrowLoanType(sourceAccount, provideOverCommitReserve),
            createdAt = now,
            dueAt = loanMeta.borrowDueDate,
            settledAt = null,
            updatedAt = null,
            version = 0L,
            requestedBy = getLoanAuthor(transferRequest),
            requestApprovedBy = getLoanApprovers(transferRequest),
            borrowTransferRequestId = transferRequest.id,
            borrowedFrom = loanSource,
            borrowedTo = loanDestination,
            payOffFrom = loanDestination,
            payOffTo = loanSource,
            borrowedAmounts = loanAmounts,
            payOffAmounts = loanAmounts,
            dueAmounts = loanAmounts,
            dueAtTimestamp = approximateDueAt
        )
        val newPendingLoan = PendingLoanModel(
            key = PendingLoanKey(
                tenantId = Tenants.DEFAULT_TENANT_ID,
                dueAtTimestamp = approximateDueAt,
                loanId = newLoan.id
            ),
            dueAt = newLoan.dueAt,
            // TODO Support notifications
            notifications = null
        )
        val serviceLoanIn = ServiceLoanInModel(
            tenantId = Tenants.DEFAULT_TENANT_ID,
            serviceId = sourceFolder.serviceId,
            status = newLoan.status,
            dueAt = approximateDueAt,
            loanId = newLoan.id
        )
        val serviceLoanOut = ServiceLoanOutModel(
            tenantId = Tenants.DEFAULT_TENANT_ID,
            serviceId = destinationFolder.serviceId,
            status = newLoan.status,
            dueAt = approximateDueAt,
            loanId = newLoan.id
        )
        val newLoanHistory = LoansHistoryModel(
            key = LoansHistoryKey(
                tenantId = Tenants.DEFAULT_TENANT_ID,
                id = UUID.randomUUID().toString(),
                loanId = newLoan.id,
                eventTimestamp = now
            ),
            eventAuthor = newLoan.requestedBy,
            eventApprovedBy = newLoan.requestApprovedBy,
            eventType = LoanEventType.LOAN_CREATED,
            transferRequestId = transferRequest.id,
            oldFields = null,
            newFields = LoansHistoryFields(
                version = newLoan.version,
                status = newLoan.status,
                dueAt = newLoan.dueAt,
                payOffFrom = newLoan.payOffFrom,
                payOffTo = newLoan.payOffTo,
                borrowedAmounts = newLoan.borrowedAmounts,
                payOffAmounts = newLoan.payOffAmounts,
                dueAmounts = newLoan.dueAmounts
            )
        )
        val currentSourceLoanBalances = getCurrentLoanBalances(session, newLoan, sourceFolder.serviceId,
            sourceAccount.providerId)
        val currentDestinationLoanBalances = getCurrentLoanBalances(session, newLoan, destinationFolder.serviceId,
            destinationAccount.providerId)
        val updatedLoanBalances = updateLoanBalances(currentSourceLoanBalances, currentDestinationLoanBalances,
            sourceAccount, sourceFolder, destinationAccount, destinationFolder, newLoan)
        loansDao.upsertOneRetryable(session, newLoan)
        pendingLoansDao.upsertOneRetryable(session, newPendingLoan)
        serviceLoansInDao.upsertOneRetryable(session, serviceLoanIn)
        serviceLoansOutDao.upsertOneRetryable(session, serviceLoanOut)
        serviceLoansBalanceDao.upsertManyRetryable(session, updatedLoanBalances)
        loansHistoryDao.upsertOneRetryable(session, newLoanHistory)
        return newLoan.id
    }

    private fun updateLoanBalances(currentSourceLoanBalances: List<ServiceLoanBalanceModel>,
                                   currentDestinationLoanBalances: List<ServiceLoanBalanceModel>,
                                   sourceAccount: AccountModel,
                                   sourceFolder: FolderModel,
                                   destinationAccount: AccountModel,
                                   destinationFolder: FolderModel,
                                   loan: LoanModel): List<ServiceLoanBalanceModel> {
        val result = mutableListOf<ServiceLoanBalanceModel>()
        val currentSourceByResource = currentSourceLoanBalances.associateBy { it.key.resourceId }
        val currentDestinationByResource = currentDestinationLoanBalances.associateBy { it.key.resourceId }
        loan.payOffAmounts.amounts.forEach { amount ->
            val currentSource = currentSourceByResource[amount.resource]
            if (currentSource != null) {
                result.add(ServiceLoanBalanceModel(
                    key = currentSource.key,
                    amountIn = currentSource.amountIn.add(amount.amount),
                    amountOut = currentSource.amountOut
                ))
            } else {
                result.add(ServiceLoanBalanceModel(
                    key = ServiceLoanBalanceKey(
                        tenantId = Tenants.DEFAULT_TENANT_ID,
                        serviceId = sourceFolder.serviceId,
                        providerId = sourceAccount.providerId,
                        resourceId = amount.resource
                    ),
                    amountIn = amount.amount,
                    amountOut = BigInteger.ZERO
                ))
            }
            val currentDestination = currentDestinationByResource[amount.resource]
            if (currentDestination != null) {
                result.add(ServiceLoanBalanceModel(
                    key = currentDestination.key,
                    amountIn = currentDestination.amountIn,
                    amountOut = currentDestination.amountOut.add(amount.amount)
                ))
            } else {
                result.add(ServiceLoanBalanceModel(
                    key = ServiceLoanBalanceKey(
                        tenantId = Tenants.DEFAULT_TENANT_ID,
                        serviceId = destinationFolder.serviceId,
                        providerId = destinationAccount.providerId,
                        resourceId = amount.resource
                    ),
                    amountIn = BigInteger.ZERO,
                    amountOut = amount.amount
                ))
            }
        }
        return result
    }

    private suspend fun getCurrentLoanBalances(session: YdbTxSession,
                                               loan: LoanModel,
                                               serviceId: ServiceId,
                                               providerId: ProviderId): List<ServiceLoanBalanceModel> {
        val resourceIds = loan.payOffAmounts.amounts.map { it.resource }.distinct()
        val keys = resourceIds.map { ServiceLoanBalanceKey(Tenants.DEFAULT_TENANT_ID, serviceId, providerId, it) }
        return keys.chunked(1000).flatMap { p -> serviceLoansBalanceDao.getByIds(session, p) }
    }

    private fun approximateDueAtTimestamp(dueAt: LoanDueDate): Instant {
        // Loan is due at the end of its due date
        val dayAfterDue = dueAt.localDate.plusDays(1)
        val startOfDayAfterDue = dayAfterDue.atStartOfDay(dueAt.timeZone)
        return startOfDayAfterDue.toInstant()
    }

    suspend fun processLoanPayOff(
        session: YdbTxSession,
        transferRequest: TransferRequestModel,
        loanMeta: LoanMeta,
        sourceAccount: AccountModel,
        destinationAccount: AccountModel,
        knownSourceProvisionsByResourceId: Map<ResourceId, AccountsQuotasModel>,
        receivedSourceProvisionByResourceId: Map<ResourceId, ValidatedReceivedProvision>,
        now: Instant
    ) {
        if (loanMeta.payOffLoanId == null) {
            logger.warn { "No loan to pay off for transfer request ${transferRequest.id}" }
            return
        }
        val loanToPayOff = loansDao.getById(session, loanMeta.payOffLoanId, Tenants.DEFAULT_TENANT_ID)!!
        if (loanToPayOff.status != LoanStatus.PENDING) {
            logger.warn { "Loan ${loanToPayOff.id} is already payed off (processing transfer request ${transferRequest.id})" }
            return
        }
        val payOffAmounts = getPayOffAmounts(transferRequest, sourceAccount, loanToPayOff,
            knownSourceProvisionsByResourceId, receivedSourceProvisionByResourceId)
        if (payOffAmounts.isEmpty()) {
            logger.warn { "No resources to pay off, loan ${loanToPayOff.id}, transfer request ${transferRequest.id}" }
            return
        }
        val updatedDueAmounts = getAmountsAfterPayOff(loanToPayOff.dueAmounts, payOffAmounts, loanToPayOff, transferRequest)
        val settled = updatedDueAmounts.amounts.isEmpty()

        val updatedLoan = loanToPayOff.copy(
            status = if (settled) {
                LoanStatus.SETTLED
            } else {
                loanToPayOff.status
            },
            settledAt = if (settled) {
                now
            } else {
                loanToPayOff.settledAt
            },
            updatedAt = now,
            version = loanToPayOff.version + 1L,
            dueAmounts = updatedDueAmounts,
        )
        val newLoanHistory = LoansHistoryModel(
            key = LoansHistoryKey(
                tenantId = loanToPayOff.tenantId,
                id = UUID.randomUUID().toString(),
                loanId = loanToPayOff.id,
                eventTimestamp = now
            ),
            eventAuthor = getLoanAuthor(transferRequest),
            eventApprovedBy = getLoanApprovers(transferRequest),
            eventType = if (settled) {
                LoanEventType.LOAN_SETTLED
            } else {
                LoanEventType.LOAN_PAY_OFF
            },
            transferRequestId = transferRequest.id,
            oldFields = LoansHistoryFields(
                version = loanToPayOff.version,
                status = if (settled) {
                    loanToPayOff.status
                } else {
                    null
                },
                dueAt = null,
                payOffFrom = null,
                payOffTo = null,
                borrowedAmounts = null,
                payOffAmounts = null,
                dueAmounts = loanToPayOff.dueAmounts
            ),
            newFields = LoansHistoryFields(
                version = updatedLoan.version,
                status = if (settled) {
                    updatedLoan.status
                } else {
                    null
                },
                dueAt = null,
                payOffFrom = null,
                payOffTo = null,
                borrowedAmounts = null,
                payOffAmounts = null,
                dueAmounts = updatedLoan.dueAmounts
            )
        )
        val currentSourceLoanBalances = getCurrentLoanBalances(session, loanToPayOff, loanToPayOff.payOffFrom.service,
            sourceAccount.providerId)
        val currentDestinationLoanBalances = getCurrentLoanBalances(session, loanToPayOff, loanToPayOff.payOffTo.service,
            destinationAccount.providerId)
        val updatedBalances = updateLoanBalancesOnPayOff(currentSourceLoanBalances, currentDestinationLoanBalances,
            loanToPayOff.dueAmounts, payOffAmounts, loanToPayOff, transferRequest)
        loansDao.upsertOneRetryable(session, updatedLoan)
        loansHistoryDao.upsertOneRetryable(session, newLoanHistory)
        serviceLoansBalanceDao.upsertManyRetryable(session, updatedBalances)
        if (settled) {
            val pendingLoanKey = PendingLoanKey(
                tenantId = Tenants.DEFAULT_TENANT_ID,
                dueAtTimestamp = loanToPayOff.dueAtTimestamp,
                loanId = loanToPayOff.id
            )
            pendingLoansDao.deleteOneRetryable(session, pendingLoanKey)
            val oldServiceLoanIn = ServiceLoanInModel(
                tenantId = loanToPayOff.tenantId,
                serviceId = loanToPayOff.payOffTo.service,
                status = loanToPayOff.status,
                dueAt = loanToPayOff.dueAtTimestamp,
                loanId = loanToPayOff.id
            )
            val oldServiceLoanOut = ServiceLoanOutModel(
                tenantId = loanToPayOff.tenantId,
                serviceId = loanToPayOff.payOffFrom.service,
                status = loanToPayOff.status,
                dueAt = loanToPayOff.dueAtTimestamp,
                loanId = loanToPayOff.id
            )
            val newServiceLoanIn = ServiceLoanInModel(
                tenantId = updatedLoan.tenantId,
                serviceId = updatedLoan.payOffTo.service,
                status = updatedLoan.status,
                dueAt = updatedLoan.dueAtTimestamp,
                loanId = updatedLoan.id
            )
            val newServiceLoanOut = ServiceLoanOutModel(
                tenantId = updatedLoan.tenantId,
                serviceId = updatedLoan.payOffFrom.service,
                status = updatedLoan.status,
                dueAt = updatedLoan.dueAtTimestamp,
                loanId = updatedLoan.id
            )
            serviceLoansInDao.deleteById(session, oldServiceLoanIn)
            serviceLoansOutDao.deleteById(session, oldServiceLoanOut)
            serviceLoansInDao.upsertOneRetryable(session, newServiceLoanIn)
            serviceLoansOutDao.upsertOneRetryable(session, newServiceLoanOut)
        }
    }

    private fun getPayOffAmounts(transferRequest: TransferRequestModel,
                                 sourceAccount: AccountModel,
                                 loan: LoanModel,
                                 knownSourceProvisionsByResourceId: Map<ResourceId, AccountsQuotasModel>,
                                 receivedSourceProvisionByResourceId: Map<ResourceId, ValidatedReceivedProvision>
    ): Map<ResourceId, BigInteger> {
        val transferResourceIds = mutableSetOf<ResourceId>()
        transferRequest.parameters.provisionTransfers.forEach { provisionTransfer ->
            if (provisionTransfer.sourceAccountId == sourceAccount.id) {
                provisionTransfer.sourceAccountTransfers.forEach { accountTransfer ->
                    transferResourceIds.add(accountTransfer.resourceId)
                }
            }
        }
        val pendingLoanResourceIds = loan.dueAmounts.amounts.filter { it.amount > BigInteger.ZERO }
            .map { it.resource }.toSet()
        val matchingResourceIds = transferResourceIds.intersect(pendingLoanResourceIds)
        val result = mutableMapOf<ResourceId, BigInteger>()
        matchingResourceIds.forEach { resourceId ->
            val oldProvision = knownSourceProvisionsByResourceId[resourceId]?.providedQuota ?: 0L
            val newProvision = receivedSourceProvisionByResourceId[resourceId]?.providedAmount ?: 0L
            val delta = BigInteger.valueOf(oldProvision).subtract(BigInteger.valueOf(newProvision))
            if (delta > BigInteger.ZERO) {
                result[resourceId] = delta
            } else {
                logger.warn { "Non-positive pay off delta $delta for resource $resourceId in source account while paying " +
                    "off ${loan.id}, transfer request ${transferRequest.id}" }
            }
        }
        return result
    }

    private fun getAmountsAfterPayOff(oldAmounts: LoanAmounts,
                                      payOffAmounts: Map<ResourceId, BigInteger>,
                                      loan: LoanModel,
                                      transferRequest: TransferRequestModel): LoanAmounts {
        val result = mutableListOf<LoanAmount>()
        oldAmounts.amounts.forEach { oldAmount ->
            val payOffAmount = payOffAmounts[oldAmount.resource] ?: BigInteger.ZERO
            val newAmount = if (oldAmount.amount < payOffAmount) {
                logger.warn { "Pay off $payOffAmount is greater than due amount ${oldAmount.amount} for " +
                    "resource ${oldAmount.resource}, loan ${loan.id}, transfer request ${transferRequest.id}" }
                BigInteger.ZERO
            } else {
                oldAmount.amount.subtract(payOffAmount)
            }
            if (newAmount > BigInteger.ZERO) {
                result.add(LoanAmount(oldAmount.resource, newAmount))
            }
        }
        return LoanAmounts(result)
    }

    private fun updateLoanBalancesOnPayOff(currentSourceLoanBalances: List<ServiceLoanBalanceModel>,
                                           currentDestinationLoanBalances: List<ServiceLoanBalanceModel>,
                                           oldAmounts: LoanAmounts,
                                           payOffAmounts: Map<ResourceId, BigInteger>,
                                           loan: LoanModel,
                                           transferRequest: TransferRequestModel): List<ServiceLoanBalanceModel> {
        val result = mutableListOf<ServiceLoanBalanceModel>()
        val currentSourceByResource = currentSourceLoanBalances.associateBy { it.key.resourceId }
        val currentDestinationByResource = currentDestinationLoanBalances.associateBy { it.key.resourceId }
        val oldAmountByResource = oldAmounts.amounts.associateBy { it.resource }.mapValues { it.value.amount }
        payOffAmounts.forEach { (resourceId, payOffAmount) ->
            val oldAmount = oldAmountByResource[resourceId] ?: BigInteger.ZERO
            val delta = if (oldAmount < payOffAmount) {
                oldAmount
            } else {
                payOffAmount
            }
            val currentSource = currentSourceByResource[resourceId]
            val currentDestination = currentDestinationByResource[resourceId]
            if (currentSource != null) {
                val currentOut = currentSource.amountOut
                val updatedAmountOut = if (currentOut < delta) {
                    logger.warn { "Loan out balance of $resourceId for ${loan.payOffFrom.service} is less then pay off amount, " +
                        "$currentOut vs $delta, loan is ${loan.id}, transfer is ${transferRequest.id}" }
                    BigInteger.ZERO
                } else {
                    currentOut.subtract(delta)
                }
                result.add(ServiceLoanBalanceModel(
                    key = currentSource.key,
                    amountIn = currentSource.amountIn,
                    amountOut = updatedAmountOut
                ))
            } else {
                logger.warn { "Loan out balance of $resourceId for ${loan.payOffFrom.service} is less then pay off amount, " +
                    "0 vs $delta, loan is ${loan.id}, transfer is ${transferRequest.id}" }
            }

            if (currentDestination != null) {
                val currentIn = currentDestination.amountIn
                val updatedAmountIn = if (currentIn < delta) {
                    logger.warn { "Loan in balance of $resourceId for ${loan.payOffTo.service} is less then pay off amount, " +
                        "$currentIn vs $delta, loan is ${loan.id}, transfer is ${transferRequest.id}" }
                    BigInteger.ZERO
                } else {
                    currentIn.subtract(delta)
                }
                result.add(ServiceLoanBalanceModel(
                    key = currentDestination.key,
                    amountIn = updatedAmountIn,
                    amountOut = currentDestination.amountOut
                ))
            } else {
                logger.warn { "Loan in balance of $resourceId for ${loan.payOffTo.service} is less then pay off amount, " +
                    "0 vs $delta, loan is ${loan.id}, transfer is ${transferRequest.id}" }
            }
        }
        return result
    }

    private fun getBorrowLoanType(sourceAccount: AccountModel, provideOverCommitReserve: Boolean): LoanType {
        if (sourceAccount.reserveType.isPresent && sourceAccount.reserveType.get() == AccountReserveType.PROVIDER) {
            return if (provideOverCommitReserve) {
                LoanType.PROVIDER_RESERVE_OVER_COMMIT
            } else {
                LoanType.PROVIDER_RESERVE
            }
        }
        // TODO Support other loan types
        return LoanType.PEER_TO_PEER
    }

    private fun getLoanAuthor(transferRequest: TransferRequestModel): LoanActionSubject {
        return LoanActionSubject(transferRequest.createdBy, null)
    }

    private fun getLoanApprovers(transferRequest: TransferRequestModel): LoanActionSubjects {
        return LoanActionSubjects(transferRequest.votes.votes.filter { it.type == VoteType.CONFIRM }
            .map { LoanActionSubject(it.userId, null) })
    }

    private fun getBorrowLoanSource(
        sourceFolder: FolderModel,
        sourceAccount: AccountModel,
        provideOverCommitReserve: Boolean
    ): LoanSubject {
        // TODO Support other subject types, like service reserves
        val subjectType = if (sourceAccount.reserveType.isPresent
            && sourceAccount.reserveType.get() == AccountReserveType.PROVIDER)
        {
            if (provideOverCommitReserve) {
                LoanSubjectType.PROVIDER_RESERVE_OVER_COMMIT
            } else {
                LoanSubjectType.PROVIDER_RESERVE_ACCOUNT
            }
        } else {
            LoanSubjectType.SERVICE
        }
        return LoanSubject(
            type = subjectType,
            service = sourceFolder.serviceId,
            reserveService = null,
            account = sourceAccount.id,
            folder = sourceFolder.id,
            provider = if (subjectType == LoanSubjectType.PROVIDER_RESERVE_ACCOUNT
                || subjectType == LoanSubjectType.PROVIDER_RESERVE_OVER_COMMIT) {
                sourceAccount.providerId
            } else {
                null
            },
            accountsSpace = if (subjectType == LoanSubjectType.PROVIDER_RESERVE_ACCOUNT
                || subjectType == LoanSubjectType.PROVIDER_RESERVE_OVER_COMMIT) {
                sourceAccount.accountsSpacesId.orElse(null)
            } else {
                null
            }
        )
    }

    private fun getBorrowLoanDestination(destinationFolder: FolderModel, destinationAccount: AccountModel): LoanSubject {
        return LoanSubject(
            type = LoanSubjectType.SERVICE,
            service = destinationFolder.serviceId,
            reserveService = null,
            account = destinationAccount.id,
            folder = destinationFolder.id,
            provider = null,
            accountsSpace = null
        )
    }

    private fun getLoanAmounts(transferRequest: TransferRequestModel,
                               destinationAccount: AccountModel,
                               knownDestinationProvisionsByResourceId: Map<ResourceId, AccountsQuotasModel>,
                               receivedDestinationProvisionByResourceId: Map<ResourceId, ValidatedReceivedProvision>
    ): LoanAmounts {
        val matchingResourceIds = mutableSetOf<ResourceId>()
        transferRequest.parameters.provisionTransfers.forEach { provisionTransfer ->
            if (provisionTransfer.destinationAccountId == destinationAccount.id) {
                provisionTransfer.destinationAccountTransfers.forEach { accountTransfer ->
                    matchingResourceIds.add(accountTransfer.resourceId)
                }
            }
        }
        val loanAmounts = mutableListOf<LoanAmount>()
        matchingResourceIds.forEach { resourceId ->
            val oldProvision = knownDestinationProvisionsByResourceId[resourceId]?.providedQuota ?: 0L
            val newProvision = receivedDestinationProvisionByResourceId[resourceId]?.providedAmount ?: 0L
            val delta = BigInteger.valueOf(newProvision).subtract(BigInteger.valueOf(oldProvision))
            if (delta > BigInteger.ZERO) {
                loanAmounts.add(LoanAmount(resourceId, delta))
            } else {
                logger.warn { "Negative delta $delta for resource $resourceId in destination account while borrowing, " +
                    "transfer request ${transferRequest.id}" }
            }
        }
        return LoanAmounts(loanAmounts)
    }

    private fun validateLoanBorrowFields(loanParameters: PreValidatedLoanParameters,
                                         borrowParameters: PreValidatedLoanBorrowParameters,
                                         transferParameters: ValidatedProvisionTransferParameters,
                                         currentUser: YaUserDetails,
                                         locale: Locale): Result<ValidatedLoanParameters?> {
        val errors = ErrorCollection.builder()
        var validDeltaSigns = true
        var nonZeroTransfer = false
        var providerReserveSourceCount = 0L
        var otherReserveSourceCount = 0L
        // All borrowing transfers must be from source to destination
        // At least one transfer must be non-zero
        transferParameters.provisionTransfers.forEachIndexed { index, provisionTransfer ->
            if (provisionTransfer.sourceAccount.reserveType.orElse(null) == AccountReserveType.PROVIDER) {
                providerReserveSourceCount += 1L
            } else {
                otherReserveSourceCount += 1L
                if (loanParameters.provideOverCommitReserve.orFalse()) {
                    errors.addError("parameters.provisionTransfers.$index.sourceAccount",
                        TypedError.invalid(messages.getMessage("errors.invalid.source.for.over.commit.reserve",
                            null, locale)))
                }
            }
            provisionTransfer.sourceAccountTransfers.forEach { accountTransfer ->
                validDeltaSigns = validDeltaSigns && accountTransfer.delta <= 0L
                nonZeroTransfer = nonZeroTransfer || accountTransfer.delta != 0L
            }
            provisionTransfer.destinationAccountTransfers.forEach { accountTransfer ->
                validDeltaSigns = validDeltaSigns && accountTransfer.delta >= 0L
                nonZeroTransfer = nonZeroTransfer || accountTransfer.delta != 0L
            }
        }
        if (!validDeltaSigns || !nonZeroTransfer) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.invalid.borrow.transfer", null, locale)))
        }
        if (providerReserveSourceCount != 0L && otherReserveSourceCount != 0L) {
            errors.addError(TypedError.invalid(messages
                .getMessage("errors.invalid.loan.transfer.source", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        val dueDate = LoanDueDate(borrowParameters.dueDate, userTimeZone(currentUser))
        return Result.success(ValidatedLoanParameters(ValidatedLoanBorrowParameters(dueDate), null,
            loanParameters.provideOverCommitReserve))
    }

    private suspend fun validateLoanPayOffFields(txSession: YdbTxSession,
                                                 loanParameters: PreValidatedLoanParameters,
                                                 payOffParameters: PreValidatedLoanPayOffParameters,
                                                 transferParameters: ValidatedProvisionTransferParameters,
                                                 locale: Locale): Result<ValidatedLoanParameters?> {
        val errors = ErrorCollection.builder()
        var validDeltaSigns = true
        var nonZeroTransfer = false
        // All paying off transfers must be from source to destination
        // At least one transfer must be non-zero
        if (loanParameters.provideOverCommitReserve.orFalse()) {
            errors.addError("loanParameters.provideOverCommitReserve",
                TypedError.invalid(messages.getMessage("errors.invalid.overcommit.reserve.with.pay.off",
                    null, locale)))
        }
        transferParameters.provisionTransfers.forEachIndexed { index, provisionTransfer ->
            provisionTransfer.sourceAccountTransfers.forEach { accountTransfer ->
                validDeltaSigns = validDeltaSigns && accountTransfer.delta <= 0L
                nonZeroTransfer = nonZeroTransfer || accountTransfer.delta != 0L
            }
            provisionTransfer.destinationAccountTransfers.forEach { accountTransfer ->
                validDeltaSigns = validDeltaSigns && accountTransfer.delta >= 0L
                nonZeroTransfer = nonZeroTransfer || accountTransfer.delta != 0L
            }
        }
        val loan = loansDao.getById(txSession, payOffParameters.loanId, Tenants.DEFAULT_TENANT_ID)
        if (loan == null) {
            // Loan must exist
            errors.addError("loanParameters.borrowParameters.loanId", TypedError.invalid(messages
                .getMessage("errors.loan.not.found", null, locale)))
        } else {
            val validSourceDestination = validatePayOffSourceAndDestination(transferParameters, loan)
            if (!validSourceDestination) {
                // Source and destination must match the loan
                errors.addError(TypedError.invalid(messages
                    .getMessage("errors.invalid.loan.source.destination", null, locale)))
            }
            if (loan.status != LoanStatus.PENDING) {
                // Loan must still be pending
                errors.addError(TypedError.invalid(messages
                    .getMessage("errors.loan.is.not.pending", null, locale)))
            } else {
                val invalidPayOffAmounts = validatePayOffAmounts(transferParameters, loan)
                if (invalidPayOffAmounts) {
                    // Pay off amounts must not be greater than due amounts
                    errors.addError(TypedError.invalid(messages
                        .getMessage("errors.loan.pay.off.is.not.due", null, locale)))
                }
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(ValidatedLoanParameters(null, ValidatedLoanPayOffParameters(loan!!),
            loanParameters.provideOverCommitReserve))
    }

    private fun validatePayOffAmounts(transferParameters: ValidatedProvisionTransferParameters,
                                      loan: LoanModel): Boolean {
        val paidOffAmounts = mutableMapOf<ResourceId, BigInteger>()
        transferParameters.provisionTransfers.forEach { provisionTransfer ->
            provisionTransfer.destinationAccountTransfers.forEach { accountTransfer ->
                paidOffAmounts.merge(accountTransfer.resource.id,
                    accountTransfer.delta.toBigInteger()) { old, new -> old.add(new) }

            }
        }
        val dueAmounts = loan.dueAmounts.amounts.associateBy { it.resource }.mapValues { it.value.amount }
        val payOffIsNotDue = paidOffAmounts.any { payOffEntry ->
            val dueAmount = dueAmounts[payOffEntry.key]
            dueAmount == null || payOffEntry.value > dueAmount
        }
        return payOffIsNotDue
    }

    private fun validatePayOffSourceAndDestination(transferParameters: ValidatedProvisionTransferParameters,
                                                   loan: LoanModel): Boolean {
        val payOffFrom = loan.payOffFrom
        val payOffTo = loan.payOffTo
        val sources = transferParameters.provisionTransfers
            .map { TransferSubject(it.sourceAccount, it.sourceFolder, it.sourceService) }
        val destinations = transferParameters.provisionTransfers
            .map { TransferSubject(it.destinationAccount, it.destinationFolder, it.destinationService) }
        if (payOffFrom.type == LoanSubjectType.ACCOUNT) {
            if (sources.isEmpty() || sources.any { it.account.id != payOffFrom.account }) {
                // Pay off must be from the predefined account
                return false
            }
        } else {
            if (sources.isEmpty() || sources.any { it.service.id != payOffFrom.service }) {
                // Pay off must be from the predefined service
                return false
            }
        }
        when (loan.type) {
            LoanType.PROVIDER_RESERVE -> {
                val providerId = payOffTo.provider
                val accountsSpaceId = payOffTo.accountsSpace
                if (destinations.isEmpty() || destinations.any { it.account.providerId != providerId
                        || it.account.accountsSpacesId.orElse(null) != accountsSpaceId
                        || it.account.reserveType.orElse(null) != AccountReserveType.PROVIDER })
                {
                    // Pay off must be to provider reserve account
                    return false
                }
            }
            LoanType.PROVIDER_RESERVE_OVER_COMMIT -> {
                val providerId = payOffTo.provider
                val accountsSpaceId = payOffTo.accountsSpace
                if (destinations.isEmpty() || destinations.any { it.account.providerId != providerId
                        || it.account.accountsSpacesId.orElse(null) != accountsSpaceId
                        || it.account.reserveType.orElse(null) != AccountReserveType.PROVIDER })
                {
                    // Pay off must be to provider reserve account
                    return false
                }
            }
            LoanType.SERVICE_RESERVE -> {
                // TODO Check that destination is a service reserve account
                if (destinations.isEmpty() || destinations.any { it.service.id != payOffTo.service }) {
                    // TODO This check is incomplete, destination may actually be any service reserve account
                    return false
                }
            }
            LoanType.PEER_TO_PEER -> {
                if (payOffTo.type == LoanSubjectType.ACCOUNT) {
                    if (destinations.isEmpty() || destinations.any { it.account.id != payOffTo.account }) {
                        // Pay off must be to the predefined account
                        return false
                    }
                } else {
                    if (destinations.isEmpty() || destinations.any { it.service.id != payOffTo.service }) {
                        // Pay off must be to the predefined service
                        return false
                    }
                }
            }
        }
        return true
    }

    private fun currentLocalDate(currentUser: YaUserDetails): LocalDate {
        return LocalDate.now(userTimeZone(currentUser))
    }

    private fun userTimeZone(currentUser: YaUserDetails): ZoneId {
        return if (currentUser.user.isPresent && currentUser.user.get().timeZone.isPresent) {
            val timeZone = currentUser.user.get().timeZone.get()
            val zoneId = try {
                ZoneId.of(timeZone)
            } catch (e: Exception) {
                logger.warn(e) { "Unexpected timezone $timeZone for ${currentUser.user}" }
                ZoneId.of("Europe/Moscow")
            }
            zoneId
        } else {
            ZoneId.of("Europe/Moscow")
        }
    }

    private data class TransferSubject (
        val account: AccountModel,
        val folder: FolderModel,
        val service: ServiceMinimalModel
    )

}
