package ru.yandex.intranet.d.services.transfer;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.Streams;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.FoldersResponsible;
import ru.yandex.intranet.d.model.transfers.LoanMeta;
import ru.yandex.intranet.d.model.transfers.LocalizedTransferApplicationErrors;
import ru.yandex.intranet.d.model.transfers.OperationStatus;
import ru.yandex.intranet.d.model.transfers.ProviderResponsible;
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer;
import ru.yandex.intranet.d.model.transfers.QuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ReserveResponsibleModel;
import ru.yandex.intranet.d.model.transfers.TransferApplicationDetails;
import ru.yandex.intranet.d.model.transfers.TransferApplicationErrors;
import ru.yandex.intranet.d.model.transfers.TransferParameters;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryFields;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.transfers.TransferVote;
import ru.yandex.intranet.d.model.transfers.TransferVotes;
import ru.yandex.intranet.d.model.units.DecimalWithUnit;
import ru.yandex.intranet.d.model.units.GrammaticalCase;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.quotas.QuotasHelper;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestCreationResult;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestHistoryContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.ValidatedTransferRequestContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.ValidatedTransferRequestHistoryContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestModel;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestResult;
import ru.yandex.intranet.d.services.transfer.model.dryrun.DryRunTransferRequestWarnings;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.util.FrontStringUtil;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.web.errors.Errors;
import ru.yandex.intranet.d.web.model.QuotaDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontAccountDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontAccountsSpaceDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontFolderDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontProviderDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceSegmentDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceSegmentationDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontResourceTypeDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontServiceDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontUnitDictionaryElementDto;
import ru.yandex.intranet.d.web.model.dictionaries.FrontUserDictionaryElementDto;
import ru.yandex.intranet.d.web.model.recipe.AccountQuotaDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestHistoryEventTypeDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestStatusDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestSubtypeDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestVoteTypeDto;
import ru.yandex.intranet.d.web.model.transfers.front.DictionaryBuilder;
import ru.yandex.intranet.d.web.model.transfers.front.FrontProvisionTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontQuotaResourceTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontQuotaTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontReserveResponsibleDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontSingleTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanBorrowMetaDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanMetaDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferLoanPayOffMetaDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferOperationStatusDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestApplicationDetailsDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestApplicationErrorsDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestFieldHistoryDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestHistoryEventDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestHistoryEventsPageDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestProviderResponsibleDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestResponsibleDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestResponsibleForServiceDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestResponsibleGroupDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestVoteDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestVotesDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestsPageDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunSingleTransferResultDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferAccountQuotaDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferAccountQuotasDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferAccountWarningsDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferFolderWarningsDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferParameters;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferPermissionsDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferQuotaDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferQuotasDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.dryrun.FrontDryRunTransferWarningsDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.services.transfer.TransferRequestPermissionService.canCancel;
import static ru.yandex.intranet.d.services.transfer.TransferRequestPermissionService.canProvideOverCommitReserve;
import static ru.yandex.intranet.d.services.transfer.TransferRequestPermissionService.canUpdate;
import static ru.yandex.intranet.d.services.transfer.TransferRequestPermissionService.canVote;

/**
 * Transfer request security service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class TransferRequestAnswerService {

    private final TransferRequestStoreService storeService;
    private final ObjectWriter transferRequestContinuationTokenWriter;
    private final ObjectWriter transferRequestHistoryContinuationTokenWriter;

    public TransferRequestAnswerService(
            TransferRequestStoreService storeService,
            @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper) {
        this.storeService = storeService;
        this.transferRequestContinuationTokenWriter = objectMapper.getObjectMapper()
                .writerFor(TransferRequestContinuationToken.class);
        this.transferRequestHistoryContinuationTokenWriter = objectMapper.getObjectMapper()
                .writerFor(TransferRequestHistoryContinuationToken.class);
    }

    public Mono<ExpandedTransferRequests<TransferRequestModel>> expand(YdbSession session,
                                                                       TransferRequestModel model) {
        Set<String> userIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        Set<Long> serviceIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        Set<String> accountIds = new HashSet<>();
        extractDependenciesOfRequest(model, userIds, folderIds, serviceIds, resourceIds, accountIds);
        return loadExpandedDependencies(session, model, userIds, folderIds, serviceIds, resourceIds, accountIds);
    }

    public Mono<ExpandedTransferRequests<TransferRequestModel>> expand(YdbSession session,
                                                                       TransferRequestCreationResult creationResult) {
        Set<String> userIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        Set<Long> serviceIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        Set<String> accountIds = new HashSet<>();
        extractDependenciesOfRequest(creationResult.getRequest(), userIds, folderIds, serviceIds,
                resourceIds, accountIds);
        return loadExpandedDependencies(session, creationResult.getRequest(), userIds, folderIds, serviceIds,
                resourceIds, accountIds, creationResult.getFolders(), creationResult.getServices(),
                creationResult.getResources(), creationResult.getProviders(), creationResult.getUnitsEnsembles());
    }


    public Mono<ExpandedTransferRequests<DryRunTransferRequestResult>> expand(YdbSession session,
                                                                              DryRunTransferRequestResult result) {
        Set<String> userIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        Set<Long> serviceIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        Set<String> accountIds = new HashSet<>();

        result.getTransferResponsible().ifPresent(
                responsible -> extractDependenciesOfResponsible(userIds, folderIds, serviceIds, responsible)
        );

        TransferParameters parameters = result.getTransferRequest().getParameters();
        extractDependenciesOfFolderTransfers(folderIds, serviceIds, parameters);
        extractDependenciesOfQuotaTransfers(folderIds, serviceIds, resourceIds, parameters);
        extractDependenciesOfAccountTransfers(folderIds, serviceIds, resourceIds, accountIds, parameters);
        extractDependenciesOfProvisionTransfers(folderIds, serviceIds, resourceIds, accountIds, parameters);
        result.getTransferRequest().getDryRunParameters().ifPresent(params -> params.getRelatedProvisionTransfers()
                .forEach(p -> extractDependenciesOfProvisionTransfer(folderIds, serviceIds, resourceIds, accountIds,
                        p)));

        return loadExpandedDependencies(session, result, userIds, folderIds, serviceIds,
                resourceIds, accountIds, result.getFolders(), result.getServices(),
                result.getResources(), result.getProviders(), result.getUnitsEnsembles());
    }


    public Mono<ExpandedTransferRequests<Page<TransferRequestModel>>> expandTransferRequests(
            YdbSession session,
            PageRequest.Validated<ValidatedTransferRequestContinuationToken> page,
            List<TransferRequestModel> requests) {
        List<TransferRequestModel> resultList = requests.size() > page.getLimit()
                ? requests.subList(0, page.getLimit()) : requests;
        Set<String> userIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        Set<Long> serviceIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        Set<String> accountIds = new HashSet<>();
        resultList.forEach(model -> extractDependenciesOfRequest(model, userIds, folderIds,
                serviceIds, resourceIds, accountIds));
        Page<TransferRequestModel> resultPage = requests.size() > page.getLimit()
                ? Page.page(resultList, prepareToken(resultList.get(resultList.size() - 1)))
                : Page.lastPage(resultList);
        return loadExpandedDependencies(session, resultPage, userIds, folderIds, serviceIds, resourceIds, accountIds);
    }

    public Mono<ExpandedTransferRequests<Page<TransferRequestHistoryModel>>> expandHistory(
            YdbSession session,
            PageRequest.Validated<ValidatedTransferRequestHistoryContinuationToken> page,
            List<TransferRequestHistoryModel> events) {
        List<TransferRequestHistoryModel> resultList = events.size() > page.getLimit()
                ? events.subList(0, page.getLimit()) : events;
        Set<String> userIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        Set<Long> serviceIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        Set<String> accountIds = new HashSet<>();
        resultList.forEach(model -> extractDependenciesOfEvent(model, userIds, folderIds,
                serviceIds, resourceIds, accountIds));
        Page<TransferRequestHistoryModel> resultPage = events.size() > page.getLimit()
                ? Page.page(resultList, prepareTokenHistory(resultList.get(resultList.size() - 1)))
                : Page.lastPage(resultList);
        return loadExpandedDependencies(session, resultPage, userIds, folderIds, serviceIds, resourceIds, accountIds);
    }

    private void fillDictionary(
            ExpandedTransferRequests<?> expanded, Locale locale, Set<String> actuallyUsedUnits,
            DictionaryBuilder builder
    ) {
        expanded.getFolders().forEach((id, folder) -> builder
                .addFolder(id, new FrontFolderDictionaryElementDto(folder)));
        expanded.getServices().forEach((id, service) -> builder
                .addService(id, new FrontServiceDictionaryElementDto(service, locale)));
        expanded.getResources().forEach((id, resource) -> builder
                .addResource(id, new FrontResourceDictionaryElementDto(resource, locale)));
        expanded.getProviders().forEach((id, provider) -> builder
                .addProvider(id, new FrontProviderDictionaryElementDto(provider, locale)));
        expanded.getResourceTypes().forEach((id, resourceType) -> builder
                .addResourceType(id, new FrontResourceTypeDictionaryElementDto(resourceType, locale)));
        expanded.getAccountsSpaces().forEach((id, accountsSpace) -> builder
                .addAccountsSpace(id, new FrontAccountsSpaceDictionaryElementDto(accountsSpace, locale)));
        expanded.getSegmentations().forEach((id, segmentation) -> builder
                .addSegmentation(id, new FrontResourceSegmentationDictionaryElementDto(segmentation, locale)));
        expanded.getSegments().forEach((id, segment) -> builder
                .addSegment(id, new FrontResourceSegmentDictionaryElementDto(segment, locale)));
        expanded.getUnits().forEach((id, unit) -> {
            if (actuallyUsedUnits.contains(unit.getId())) {
                builder.addUnit(id, toUnitDictionary(unit, locale));
            }
        });
        expanded.getUsers().forEach((id, user) -> builder
                .addUser(id, new FrontUserDictionaryElementDto(user, locale)));
        expanded.getAccounts().forEach((id, account) ->
                builder.addAccount(id, new FrontAccountDictionaryElementDto(account))
        );
    }

    public FrontSingleTransferRequestDto toSingleRequest(ExpandedTransferRequests<TransferRequestModel> expanded,
                                                         YaUserDetails currentUser,
                                                         Locale locale) {
        FrontSingleTransferRequestDto.Builder builder = FrontSingleTransferRequestDto.builder();
        builder.transfer(toRequest(expanded.getTransferRequests(), expanded, currentUser, locale));
        fillDictionary(expanded, locale, getActuallyUsedUnits(builder), builder);
        return builder.build();
    }

    public FrontTransferRequestsPageDto toRequestsPage(
            ExpandedTransferRequests<Page<TransferRequestModel>> expanded,
            YaUserDetails currentUser,
            Locale locale) {
        FrontTransferRequestsPageDto.Builder builder = FrontTransferRequestsPageDto.builder();
        expanded.getTransferRequests().getItems().forEach(transferRequest -> builder
                .addTransfer(toRequest(transferRequest, expanded, currentUser, locale)));
        fillDictionary(expanded, locale, getActuallyUsedUnits(builder), builder);
        builder.nextPageToken(expanded.getTransferRequests().getContinuationToken().orElse(null));
        return builder.build();
    }

    public FrontTransferRequestHistoryEventsPageDto toHistoryPage(
            ExpandedTransferRequests<Page<TransferRequestHistoryModel>> expanded,
            Locale locale) {
        FrontTransferRequestHistoryEventsPageDto.Builder builder = FrontTransferRequestHistoryEventsPageDto.builder();
        expanded.getTransferRequests().getItems().forEach(transferRequest -> builder
                .addEvent(toEvent(transferRequest, expanded.getResources(), expanded.getUnitsEnsembles(), locale)));
        fillDictionary(expanded, locale, getActuallyUsedUnits(builder), builder);
        builder.nextPageToken(expanded.getTransferRequests().getContinuationToken().orElse(null));
        return builder.build();
    }

    private Set<String> getActuallyUsedUnits(FrontSingleTransferRequestDto.Builder builder) {
        Set<String> actuallyUsedUnits = new HashSet<>();
        builder.getTransfer().getParameters().getQuotaTransfers().forEach(transfer -> transfer.getResourceTransfers()
                .forEach(resourceTransfer -> actuallyUsedUnits.add(resourceTransfer.getDeltaUnitId())));
        return actuallyUsedUnits;
    }

    private Set<String> getActuallyUsedUnits(FrontTransferRequestsPageDto.Builder builder) {
        Set<String> actuallyUsedUnits = new HashSet<>();
        builder.getTransfers().forEach(transfer -> transfer.getParameters().getQuotaTransfers()
                .forEach(quotaTransfer -> quotaTransfer.getResourceTransfers()
                        .forEach(resourceTransfer -> actuallyUsedUnits.add(resourceTransfer.getDeltaUnitId()))));
        return actuallyUsedUnits;
    }

    private Set<String> getActuallyUsedUnits(FrontTransferRequestHistoryEventsPageDto.Builder builder) {
        Set<String> actuallyUsedUnits = new HashSet<>();
        builder.getEvents().forEach(event -> {
            event.getOldParameters().ifPresent(parameters -> parameters.getQuotaTransfers()
                    .forEach(quotaTransfer -> quotaTransfer.getResourceTransfers()
                            .forEach(resourceTransfer -> actuallyUsedUnits.add(resourceTransfer.getDeltaUnitId()))));
            event.getNewParameters().ifPresent(parameters -> parameters.getQuotaTransfers()
                    .forEach(quotaTransfer -> quotaTransfer.getResourceTransfers()
                            .forEach(resourceTransfer -> actuallyUsedUnits.add(resourceTransfer.getDeltaUnitId()))));
        });
        return actuallyUsedUnits;
    }

    private FrontTransferRequestHistoryEventDto toEvent(TransferRequestHistoryModel model,
                                                        Map<String, ResourceModel> resources,
                                                        Map<String, UnitsEnsembleModel> unitsEnsembles,
                                                        Locale locale) {
        return FrontTransferRequestHistoryEventDto.builder()
                .id(model.getId())
                .eventType(TransferRequestHistoryEventTypeDto.fromModel(model.getType()))
                .eventTimestamp(model.getTimestamp())
                .authorId(model.getAuthorId().orElse(null))
                .oldFields(model.getOldFields().map(this::toEventFields).orElse(null))
                .newFields(model.getNewFields().map(this::toEventFields).orElse(null))
                .oldParameters(model.getOldParameters()
                        .map(p -> toParameters(p, resources, unitsEnsembles, locale)).orElse(null))
                .newParameters(model.getNewParameters()
                        .map(p -> toParameters(p, resources, unitsEnsembles, locale)).orElse(null))
                .oldTransferResponsible(model.getOldResponsible().map(this::toResponsible).orElse(null))
                .newTransferResponsible(model.getNewResponsible().map(this::toResponsible).orElse(null))
                .oldTransferVotes(model.getOldVotes().map(this::toVotes).orElse(null))
                .newTransferVotes(model.getNewVotes().map(this::toVotes).orElse(null))
                .oldApplicationDetails(model.getOldApplicationDetails()
                        .map(d -> toApplicationDetails(d, locale)).orElse(null))
                .newApplicationDetails(model.getNewApplicationDetails()
                        .map(d -> toApplicationDetails(d, locale)).orElse(null))
                .oldLoanMeta(model.getOldLoanMeta().map(this::toLoanMeta).orElse(null))
                .newLoanMeta(model.getNewLoanMeta().map(this::toLoanMeta).orElse(null))
                .build();
    }

    private FrontTransferRequestFieldHistoryDto toEventFields(TransferRequestHistoryFields model) {
        return FrontTransferRequestFieldHistoryDto.builder()
                .version(model.getVersion().map(Objects::toString).orElse(null))
                .summary(model.getSummary().orElse(null))
                .description(model.getDescription().orElse(null))
                .trackerIssueKey(model.getTrackerIssueKey().orElse(null))
                .type(model.getType().map(TransferRequestTypeDto::fromModel).orElse(null))
                .status(model.getStatus().map(TransferRequestStatusDto::fromModel).orElse(null))
                .build();
    }

    private FrontTransferRequestDto toRequest(TransferRequestModel model,
                                              ExpandedTransferRequests<?> expanded,
                                              YaUserDetails currentUser,
                                              Locale locale) {
        return FrontTransferRequestDto.builder()
                .id(model.getId())
                .version(String.valueOf(model.getVersion()))
                .summary(model.getSummary().orElse(null))
                .description(model.getDescription().orElse(null))
                .trackerIssueKey(model.getTrackerIssueKey().orElse(null))
                .requestType(TransferRequestTypeDto.fromModel(model.getType()))
                .requestSubtype(model.getSubtype().map(TransferRequestSubtypeDto::fromModel).orElse(null))
                .status(TransferRequestStatusDto.fromModel(model.getStatus()))
                .createdBy(model.getCreatedBy())
                .updatedBy(model.getUpdatedBy().orElse(null))
                .createdAt(model.getCreatedAt())
                .updatedAt(model.getUpdatedAt().orElse(null))
                .appliedAt(model.getAppliedAt().orElse(null))
                .parameters(toParameters(model.getParameters(), expanded.getResources(), expanded.getUnitsEnsembles(),
                        locale))
                .transferResponsible(toResponsible(model.getResponsible()))
                .transferVotes(toVotes(model.getVotes()))
                .applicationDetails(model.getApplicationDetails()
                        .map(d -> toApplicationDetails(d, locale)).orElse(null))
                .canUpdate(canUpdate(currentUser, model, expanded.getProviders(), expanded.getAccounts()))
                .canCancel(canCancel(currentUser, model, expanded.getProviders(), expanded.getAccounts()))
                .canVote(canVote(currentUser, model, expanded.getAccounts()))
                .canProvideOverCommitReserve(canProvideOverCommitReserve(currentUser, model, expanded.getProviders(),
                        expanded.getAccounts()))
                .loanMeta(model.getLoanMeta().map(this::toLoanMeta).orElse(null))
                .build();
    }

    private FrontTransferLoanMetaDto toLoanMeta(LoanMeta loanMeta) {
        FrontTransferLoanBorrowMetaDto borrowMeta = null;
        FrontTransferLoanPayOffMetaDto payOffMeta = null;
        switch (loanMeta.getOperationType()) {
            case BORROW -> borrowMeta = new FrontTransferLoanBorrowMetaDto(
                    Objects.requireNonNull(loanMeta.getBorrowDueDate()).getLocalDate(),
                    new ArrayList<>(Objects.requireNonNullElse(loanMeta.getBorrowLoanIds(), Set.of())));
            case PAY_OFF -> payOffMeta = new FrontTransferLoanPayOffMetaDto(Objects.requireNonNull(loanMeta
                    .getPayOffLoanId()));
            default -> {
            }
        }
        return new FrontTransferLoanMetaDto(borrowMeta, payOffMeta, loanMeta.getProvideOverCommitReserve());
    }

    private FrontTransferRequestResponsibleDto toResponsible(TransferResponsible model) {
        FrontTransferRequestResponsibleDto.Builder builder = FrontTransferRequestResponsibleDto.builder();
        model.getResponsible().forEach(responsible -> builder.addResponsibleGroup(toResponsibleGroup(responsible)));
        model.getProviderResponsible()
                .forEach(responsible -> builder.addProviderResponsible(toProviderResponsible(responsible)));
        builder.reserveResponsible(model.getReserveResponsibleModel()
                .map(reserveResponsible -> FrontReserveResponsibleDto.builder()
                        .providerId(reserveResponsible.getProviderId())
                        .addResponsibleIds(reserveResponsible.getResponsibleIds())
                        .serviceId(reserveResponsible.getServiceId())
                        .folderId(reserveResponsible.getFolderId())
                        .build())
                .orElse(null));
        return builder.build();
    }

    private FrontTransferRequestResponsibleGroupDto toResponsibleGroup(FoldersResponsible foldersResponsible) {
        FrontTransferRequestResponsibleGroupDto.Builder builder = FrontTransferRequestResponsibleGroupDto.builder();
        Map<String, Set<String>> userServices = new HashMap<>();
        foldersResponsible.getResponsible().forEach(serviceResponsible -> serviceResponsible.getResponsibleIds()
                .forEach(userId -> userServices.computeIfAbsent(userId, k -> new HashSet<>())
                        .add(String.valueOf(serviceResponsible.getServiceId()))));
        foldersResponsible.getFolderIds().forEach(builder::addFolder);
        userServices.forEach((userId, serviceIds) -> {
            FrontTransferRequestResponsibleForServiceDto.Builder responsibleBuilder
                    = FrontTransferRequestResponsibleForServiceDto.builder();
            responsibleBuilder.responsibleId(userId);
            serviceIds.forEach(responsibleBuilder::addServiceId);
            builder.addResponsible(responsibleBuilder.build());
        });
        return builder.build();
    }

    private FrontTransferRequestProviderResponsibleDto toProviderResponsible(ProviderResponsible providerResponsible) {
        FrontTransferRequestProviderResponsibleDto.Builder builder = FrontTransferRequestProviderResponsibleDto
                .builder();
        builder.responsibleId(providerResponsible.getResponsibleId());
        builder.addProviderIds(providerResponsible.getProviderIds());
        return builder.build();
    }

    private FrontTransferRequestVotesDto toVotes(TransferVotes model) {
        FrontTransferRequestVotesDto.Builder builder = FrontTransferRequestVotesDto.builder();
        model.getVotes().forEach(vote -> builder.addVote(toVote(vote)));
        return builder.build();
    }

    private FrontTransferRequestVoteDto toVote(TransferVote model) {
        FrontTransferRequestVoteDto.Builder builder = FrontTransferRequestVoteDto.builder();
        builder.userId(model.getUserId());
        model.getFolderIds().forEach(builder::addFolderId);
        model.getServiceIds().forEach(serviceId -> builder.addServiceId(String.valueOf(serviceId)));
        builder.timestamp(model.getTimestamp());
        builder.voteType(TransferRequestVoteTypeDto.fromModel(model.getType()));
        model.getProviderIds().forEach(builder::addProviderId);
        return builder.build();
    }

    private FrontTransferRequestApplicationDetailsDto toApplicationDetails(TransferApplicationDetails model,
                                                                           Locale locale) {
        return FrontTransferRequestApplicationDetailsDto.builder()
                .transferErrors(toTransferErrors(model, locale).orElse(null))
                .transferErrorsByOperationId(Maps.transformValues(model.getErrorsByOperationId(),
                        e -> toTransferErrors(e, locale)))
                .operationStatusById(Maps.transformValues(model.getOperationStatusById(),
                        TransferRequestAnswerService::toOperationStatusDto))
                .build();
    }

    private Optional<FrontTransferRequestApplicationErrorsDto> toTransferErrors(TransferApplicationDetails model,
                                                                                Locale locale) {
        if (model.getErrorsEn().isEmpty() && model.getErrorsRu().isEmpty()) {
            return Optional.empty();
        }
        TransferApplicationErrors errors = Locales.selectObject(model.getErrorsEn().orElse(null),
                model.getErrorsRu().orElse(null), locale);
        FrontTransferRequestApplicationErrorsDto result = FrontTransferRequestApplicationErrorsDto.builder()
                .addErrors(errors.getErrors())
                .addFieldErrors(errors.getFieldErrors())
                .addDetails(errors.getDetails())
                .build();
        return Optional.of(result);
    }

    private FrontTransferRequestApplicationErrorsDto toTransferErrors(
            LocalizedTransferApplicationErrors operationErrors,
            Locale locale
    ) {
        TransferApplicationErrors errors = Locales.selectObject(operationErrors.getErrorsEn(),
                operationErrors.getErrorsRu(), locale);
        return FrontTransferRequestApplicationErrorsDto.builder()
                .addErrors(errors.getErrors())
                .addFieldErrors(errors.getFieldErrors())
                .addDetails(errors.getDetails())
                .build();
    }

    private static FrontTransferOperationStatusDto toOperationStatusDto(OperationStatus status) {
        return switch (status) {
            case COMPLETED -> FrontTransferOperationStatusDto.COMPLETED;
            case FAILED -> FrontTransferOperationStatusDto.FAILED;
            case EXECUTING -> FrontTransferOperationStatusDto.EXECUTING;
            case UNKNOWN -> FrontTransferOperationStatusDto.UNKNOWN;
        };
    }

    private FrontTransferRequestParametersDto toParameters(TransferParameters model,
                                                           Map<String, ResourceModel> resources,
                                                           Map<String, UnitsEnsembleModel> unitsEnsembles,
                                                           Locale locale) {
        FrontTransferRequestParametersDto.Builder builder = FrontTransferRequestParametersDto.builder();
        model.getQuotaTransfers().forEach(quotaTransfer -> builder
                .addQuotaTransfer(toQuotaTransfer(quotaTransfer, resources, unitsEnsembles, locale)));
        model.getProvisionTransfers().forEach(provisionTransfer -> builder
                .addProvisionTransfer(FrontProvisionTransferDto.fromModel(
                        provisionTransfer, resources, unitsEnsembles, locale)));
        return builder.build();
    }

    private FrontQuotaTransferDto toQuotaTransfer(QuotaTransfer model,
                                                  Map<String, ResourceModel> resources,
                                                  Map<String, UnitsEnsembleModel> unitsEnsembles,
                                                  Locale locale) {
        FrontQuotaTransferDto.Builder builder = FrontQuotaTransferDto.builder();
        builder.destinationFolderId(model.getDestinationFolderId());
        builder.destinationServiceId(String.valueOf(model.getDestinationServiceId()));
        model.getTransfers().forEach(resourceQuotaTransfer -> builder
                .addResourceTransfer(FrontQuotaResourceTransferDto.fromModel(resourceQuotaTransfer, resources,
                        unitsEnsembles, locale)));
        return builder.build();
    }

    private FrontUnitDictionaryElementDto toUnitDictionary(UnitModel model, Locale locale) {
        return FrontUnitDictionaryElementDto.builder()
                .shortName(Locales.select(model.getShortNameSingularEn(), model.getShortNameSingularRu()
                        .get(GrammaticalCase.NOMINATIVE), locale))
                .build();
    }

    @SuppressWarnings("ParameterNumber")
    private <T> Mono<ExpandedTransferRequests<T>> loadExpandedDependencies(
            YdbSession session, T model, Set<String> userIds, Set<String> folderIds,
            Set<Long> serviceIds, Set<String> resourceIds, Set<String> accountIds,
            List<FolderModel> knownFolders, List<ServiceMinimalModel> knownServices,
            List<ResourceModel> knownResources, List<ProviderModel> knownProviders,
            List<UnitsEnsembleModel> knownUnitsEnsembles) {
        return storeService.loadUsers(session, userIds).flatMap(users ->
            storeService.loadAccounts(session, accountIds).flatMap(accounts -> {
                Set<String> accountsFolderIds = accounts.stream().map(AccountModel::getFolderId)
                        .collect(Collectors.toSet());
                Set<String> folderIdsToLoad = Sets.union(folderIds, accountsFolderIds);
        return storeService.loadFolders(session, folderIdsToLoad, knownFolders).flatMap(folders -> {
            Set<Long> folderServiceIds = folders.stream().map(FolderModel::getServiceId)
                    .collect(Collectors.toSet());
            Set<Long> serviceIdsToLoad = Sets.union(serviceIds, folderServiceIds);
        return storeService.loadServices(session, serviceIdsToLoad, knownServices).flatMap(services ->
            storeService.loadResources(session, resourceIds, knownResources).flatMap(resources -> {
                Set<String> providerIds = resources.stream()
                        .map(ResourceModel::getProviderId).collect(Collectors.toSet());
        return storeService.loadProviders(session, providerIds, knownProviders)
            .flatMap(providers -> {
                Set<String> unitsEnsembleIds = resources.stream()
                        .map(ResourceModel::getUnitsEnsembleId)
                        .collect(Collectors.toSet());
        return storeService.loadUnitsEnsembles(session, unitsEnsembleIds, knownUnitsEnsembles)
            .flatMap(unitsEnsembles -> {
                List<String> accountsSpaceIds = Streams.concat(
                                resources.stream().map(ResourceModel::getAccountsSpacesId),
                                accounts.stream().map(a -> a.getAccountsSpacesId().orElse(null)))
                        .filter(Objects::nonNull)
                        .distinct()
                        .toList();
        return storeService.loadAccountSpaces(session, accountsSpaceIds)
            .flatMap(accountsSpaces -> {
                List<String> segmentationIds = Streams.concat(
                                resources.stream().flatMap(r -> r.getSegments().stream()
                                        .map(ResourceSegmentSettingsModel::getSegmentationId)),
                                accountsSpaces.stream().flatMap(as -> as.getSegments().stream()
                                        .map(ResourceSegmentSettingsModel::getSegmentationId)))
                        .distinct()
                        .toList();
        return storeService.loadResourceSegmentations(session, segmentationIds)
            .flatMap(segmentations -> {
                List<String> segmentIds = Streams.concat(
                                resources.stream().flatMap(r -> r.getSegments().stream()
                                        .map(ResourceSegmentSettingsModel::getSegmentId)),
                                accountsSpaces.stream().flatMap(as -> as.getSegments().stream()
                                        .map(ResourceSegmentSettingsModel::getSegmentId)))
                        .distinct()
                        .toList();
        return storeService.loadResourceSegments(session, segmentIds)
            .flatMap(segments -> {
                List<String> resourceTypeIds = resources.stream()
                        .map(ResourceModel::getResourceTypeId)
                        .distinct()
                        .toList();
        return storeService.loadResourceTypes(session, resourceTypeIds)
            .map(resourceTypes -> {
                    List<UnitModel> units = new ArrayList<>();
                    unitsEnsembles.forEach(ensemble -> units.addAll(ensemble.getUnits()));
                    return ExpandedTransferRequests.<T>builder()
                            .transferRequests(model)
                            .addFolders(folders)
                            .addServices(services)
                            .addResources(resources)
                            .addProviders(providers)
                            .addUnitsEnsembles(unitsEnsembles)
                            .addUnits(units)
                            .addUsers(users)
                            .addAccounts(accounts)
                            .addResourceTypes(resourceTypes)
                            .addSegmentations(segmentations)
                            .addSegments(segments)
                            .addAccountsSpaces(accountsSpaces)
                            .build();
                    }); }); }); }); }); }); }));
                });
            }));
    }

    @SuppressWarnings("ParameterNumber")
    private <T> Mono<ExpandedTransferRequests<T>> loadExpandedDependencies(
            YdbSession session, T model, Set<String> userIds, Set<String> folderIds,
            Set<Long> serviceIds, Set<String> resourceIds, Set<String> accountIds) {
        return storeService.loadUsers(session, userIds).flatMap(users -> storeService.loadAccounts(session, accountIds)
                .flatMap(accounts -> {
                    Set<String> accountsFolderIds = accounts.stream().map(AccountModel::getFolderId)
                            .collect(Collectors.toSet());
                    Set<String> folderIdsToLoad = Sets.union(folderIds, accountsFolderIds);
        return storeService.loadFolders(session, folderIdsToLoad)
                .flatMap(folders -> {
                    Set<Long> folderServiceIds = folders.stream().map(FolderModel::getServiceId)
                            .collect(Collectors.toSet());
                    Set<Long> serviceIdsToLoad = Sets.union(serviceIds, folderServiceIds);
        return storeService.loadServices(session, serviceIdsToLoad)
                .flatMap(services -> storeService.loadResources(session, resourceIds).flatMap(resources -> {
                    Set<String> providerIds = resources.stream()
                            .map(ResourceModel::getProviderId).collect(Collectors.toSet());
        return storeService.loadProviders(session, providerIds)
                .flatMap(providers -> {
                    Set<String> unitsEnsembleIds = resources.stream()
                            .map(ResourceModel::getUnitsEnsembleId)
                            .collect(Collectors.toSet());
        return storeService.loadUnitsEnsembles(session, unitsEnsembleIds)
                .flatMap(unitsEnsembles -> {
                    List<String> accountsSpaceIds = Streams.concat(
                                    resources.stream().map(ResourceModel::getAccountsSpacesId),
                                    accounts.stream().map(a -> a.getAccountsSpacesId().orElse(null)))
                            .filter(Objects::nonNull)
                            .distinct()
                            .toList();
        return storeService.loadAccountSpaces(session, accountsSpaceIds)
                .flatMap(accountsSpaces -> {
                    List<String> segmentationIds = Streams.concat(
                                    resources.stream().flatMap(r -> r.getSegments().stream()
                                            .map(ResourceSegmentSettingsModel::getSegmentationId)),
                                    accountsSpaces.stream().flatMap(as -> as.getSegments().stream()
                                            .map(ResourceSegmentSettingsModel::getSegmentationId)))
                            .distinct()
                            .toList();
        return storeService.loadResourceSegmentations(session, segmentationIds)
                .flatMap(segmentations -> {
                    List<String> segmentIds = Streams.concat(
                                    resources.stream().flatMap(r -> r.getSegments().stream()
                                            .map(ResourceSegmentSettingsModel::getSegmentId)),
                                    accountsSpaces.stream().flatMap(as -> as.getSegments().stream()
                                            .map(ResourceSegmentSettingsModel::getSegmentId)))
                            .distinct()
                            .toList();
        return storeService.loadResourceSegments(session, segmentIds)
                .flatMap(segments -> {
                    List<String> resourceTypeIds = resources.stream()
                            .map(ResourceModel::getResourceTypeId)
                            .distinct()
                            .toList();
        return storeService.loadResourceTypes(session, resourceTypeIds)
                .map(resourceTypes -> {
                    List<UnitModel> units = new ArrayList<>();
                    unitsEnsembles.forEach(ensemble -> units.addAll(ensemble.getUnits()));
        return ExpandedTransferRequests.<T>builder()
                .transferRequests(model)
                .addFolders(folders)
                .addServices(services)
                .addResources(resources)
                .addProviders(providers)
                .addUnitsEnsembles(unitsEnsembles)
                .addUnits(units)
                .addUsers(users)
                .addAccounts(accounts)
                .addResourceTypes(resourceTypes)
                .addSegmentations(segmentations)
                .addSegments(segments)
                .addAccountsSpaces(accountsSpaces)
                .build();
        }); }); }); }); }); }); })); });
        }));
    }

    private void extractDependenciesOfRequest(TransferRequestModel model, Set<String> userIds, Set<String> folderIds,
                                              Set<Long> serviceIds, Set<String> resourceIds, Set<String> accountIds) {
        userIds.add(model.getCreatedBy());
        model.getUpdatedBy().ifPresent(userIds::add);
        extractDependenciesOfResponsible(userIds, folderIds, serviceIds, model.getResponsible());
        extractDependenciesOfVotes(userIds, folderIds, serviceIds, model.getVotes());
        extractDependenciesOfFolderTransfers(folderIds, serviceIds, model.getParameters());
        extractDependenciesOfQuotaTransfers(folderIds, serviceIds, resourceIds, model.getParameters());
        extractDependenciesOfAccountTransfers(folderIds, serviceIds, resourceIds, accountIds, model.getParameters());
        extractDependenciesOfProvisionTransfers(folderIds, serviceIds, resourceIds, accountIds, model.getParameters());
    }


    private void extractDependenciesOfEvent(TransferRequestHistoryModel model, Set<String> userIds,
                                            Set<String> folderIds, Set<Long> serviceIds, Set<String> resourceIds,
                                            Set<String> accountIds) {
        model.getAuthorId().ifPresent(userIds::add);
        model.getOldResponsible().ifPresent(x -> extractDependenciesOfResponsible(userIds, folderIds, serviceIds, x));
        model.getNewResponsible().ifPresent(x -> extractDependenciesOfResponsible(userIds, folderIds, serviceIds, x));
        model.getOldVotes().ifPresent(x -> extractDependenciesOfVotes(userIds, folderIds, serviceIds, x));
        model.getNewVotes().ifPresent(x -> extractDependenciesOfVotes(userIds, folderIds, serviceIds, x));
        model.getOldParameters().ifPresent(x -> extractDependenciesOfFolderTransfers(folderIds, serviceIds, x));
        model.getNewParameters().ifPresent(x -> extractDependenciesOfFolderTransfers(folderIds, serviceIds, x));
        model.getOldParameters().ifPresent(x -> extractDependenciesOfQuotaTransfers(folderIds, serviceIds,
                resourceIds, x));
        model.getNewParameters().ifPresent(x -> extractDependenciesOfQuotaTransfers(folderIds, serviceIds,
                resourceIds, x));
        model.getOldParameters().ifPresent(x -> extractDependenciesOfAccountTransfers(folderIds, serviceIds,
                resourceIds, accountIds, x));
        model.getNewParameters().ifPresent(x -> extractDependenciesOfAccountTransfers(folderIds, serviceIds,
                resourceIds, accountIds, x));
        model.getOldParameters().ifPresent(x -> extractDependenciesOfProvisionTransfers(folderIds, serviceIds,
                resourceIds, accountIds, x));
        model.getNewParameters().ifPresent(x -> extractDependenciesOfProvisionTransfers(folderIds, serviceIds,
                resourceIds, accountIds, x));
    }

    private void extractDependenciesOfProvisionTransfers(Set<String> folderIds, Set<Long> serviceIds,
                                                         Set<String> resourceIds, Set<String> accountIds,
                                                         TransferParameters x) {
        x.getProvisionTransfers().forEach(t ->
                extractDependenciesOfProvisionTransfer(folderIds, serviceIds, resourceIds, accountIds, t));
    }

    private void extractDependenciesOfProvisionTransfer(Set<String> folderIds, Set<Long> serviceIds,
                                                        Set<String> resourceIds, Set<String> accountIds,
                                                        ProvisionTransfer transfer) {
        folderIds.add(transfer.getSourceFolderId());
        folderIds.add(transfer.getDestinationFolderId());
        serviceIds.add(transfer.getSourceServiceId());
        serviceIds.add(transfer.getDestinationServiceId());
        accountIds.add(transfer.getSourceAccountId());
        accountIds.add(transfer.getDestinationAccountId());
        transfer.getSourceAccountTransfers().forEach(v -> resourceIds.add(v.getResourceId()));
        transfer.getDestinationAccountTransfers().forEach(v -> resourceIds.add(v.getResourceId()));
    }

    private void extractDependenciesOfAccountTransfers(Set<String> folderIds, Set<Long> serviceIds,
                                                       Set<String> resourceIds, Set<String> accountIds,
                                                       TransferParameters x) {
        x.getAccountTransfers().forEach(t -> {
            folderIds.add(t.getSourceFolderId());
            folderIds.add(t.getDestinationFolderId());
            serviceIds.add(t.getSourceServiceId());
            serviceIds.add(t.getDestinationServiceId());
            accountIds.add(t.getAccountId());
            t.getQuotaTransfers().forEach(v -> resourceIds.add(v.getResourceId()));
        });
    }

    private void extractDependenciesOfQuotaTransfers(Set<String> folderIds, Set<Long> serviceIds,
                                                     Set<String> resourceIds, TransferParameters x) {
        x.getQuotaTransfers().forEach(t -> {
            folderIds.add(t.getDestinationFolderId());
            serviceIds.add(t.getDestinationServiceId());
            t.getTransfers().forEach(v -> resourceIds.add(v.getResourceId()));
        });
    }

    private void extractDependenciesOfFolderTransfers(Set<String> folderIds, Set<Long> serviceIds,
                                                      TransferParameters x) {
        x.getFolderTransfers().forEach(t -> {
            serviceIds.add(t.getSourceServiceId());
            serviceIds.add(t.getDestinationServiceId());
            folderIds.add(t.getFolderId());
        });
    }

    private void extractDependenciesOfVotes(Set<String> userIds, Set<String> folderIds, Set<Long> serviceIds,
                                            TransferVotes x) {
        x.getVotes().forEach(vote -> {
            userIds.add(vote.getUserId());
            serviceIds.addAll(vote.getServiceIds());
            folderIds.addAll(vote.getFolderIds());
        });
    }

    private void extractDependenciesOfResponsible(Set<String> userIds, Set<String> folderIds, Set<Long> serviceIds,
                                                  TransferResponsible x) {
        x.getResponsible().forEach(r -> {
            r.getResponsible().forEach(v -> {
                userIds.addAll(v.getResponsibleIds());
                serviceIds.add(v.getServiceId());
            });
            folderIds.addAll(r.getFolderIds());
        });
        x.getProviderResponsible().forEach(r -> userIds.add(r.getResponsibleId()));
        x.getReserveResponsibleModel().map(ReserveResponsibleModel::getResponsibleIds).ifPresent(userIds::addAll);
    }

    private String prepareToken(TransferRequestModel lastItem) {
        return ContinuationTokens.encode(new TransferRequestContinuationToken(lastItem.getId(),
                lastItem.getCreatedAt().toEpochMilli()), transferRequestContinuationTokenWriter);
    }

    private String prepareTokenHistory(TransferRequestHistoryModel lastItem) {
        return ContinuationTokens.encode(new TransferRequestHistoryContinuationToken(lastItem.getId(),
                lastItem.getTimestamp().toEpochMilli()), transferRequestHistoryContinuationTokenWriter);
    }


    public static FrontDryRunTransferQuotaDto fromModel(QuotaDto quotaDto, List<UnitModel> sortedUnits,
                                                        UnitModel baseUnit, Locale locale) {

        DecimalWithUnit quota = QuotasHelper.convertToReadable(new BigDecimal(quotaDto.getQuota()),
                sortedUnits, baseUnit);

        DecimalWithUnit balance = QuotasHelper.convertToReadable(new BigDecimal(quotaDto.getBalance()),
                sortedUnits, baseUnit);

        DecimalWithUnit frozen = QuotasHelper.convertToReadable(new BigDecimal(quotaDto.getFrozenQuota()),
                sortedUnits, baseUnit);

        return new FrontDryRunTransferQuotaDto(
                quotaDto.getResourceId(),
                FrontStringUtil.toString(QuotasHelper.roundForDisplay(quota.getAmount())),
                quota.getUnit().getId(),
                QuotasHelper.getUnitName(quota.getUnit(), locale),
                FrontStringUtil.toString(QuotasHelper.roundForDisplay(balance.getAmount())),
                balance.getUnit().getId(),
                QuotasHelper.getUnitName(balance.getUnit(), locale),
                FrontStringUtil.toString(QuotasHelper.roundForDisplay(frozen.getAmount())),
                frozen.getUnit().getId(),
                QuotasHelper.getUnitName(frozen.getUnit(), locale)
        );
    }

    public static FrontDryRunTransferAccountQuotaDto fromModel(AccountQuotaDto quotaDto, List<UnitModel> sortedUnits,
                                                               UnitModel baseUnit, Locale locale) {

        DecimalWithUnit provided = QuotasHelper.convertToReadable(new BigDecimal(quotaDto.getProvidedQuota()),
                sortedUnits, baseUnit);

        DecimalWithUnit allocated = QuotasHelper.convertToReadable(new BigDecimal(quotaDto.getAllocatedQuota()),
                sortedUnits, baseUnit);

        DecimalWithUnit frozen = QuotasHelper.convertToReadable(new BigDecimal(quotaDto.getFrozenProvidedQuota()),
                sortedUnits, baseUnit);

        return new FrontDryRunTransferAccountQuotaDto(
                quotaDto.getResourceId(),
                FrontStringUtil.toString(QuotasHelper.roundForDisplay(provided.getAmount())),
                provided.getUnit().getId(),
                QuotasHelper.getUnitName(provided.getUnit(), locale),
                FrontStringUtil.toString(QuotasHelper.roundForDisplay(allocated.getAmount())),
                allocated.getUnit().getId(),
                QuotasHelper.getUnitName(allocated.getUnit(), locale),
                FrontStringUtil.toString(QuotasHelper.roundForDisplay(frozen.getAmount())),
                frozen.getUnit().getId(),
                QuotasHelper.getUnitName(frozen.getUnit(), locale)
        );
    }

    public FrontDryRunSingleTransferResultDto toSingleDryRunResult(
            ExpandedTransferRequests<DryRunTransferRequestResult> expanded,
            YaUserDetails currentUser, Locale locale) {

        FrontDryRunSingleTransferResultDto.Builder builder = FrontDryRunSingleTransferResultDto.builder();
        DryRunTransferRequestModel model = expanded.getTransferRequests().getTransferRequest();

        Map<String, List<UnitModel>> sortedUnitModelsByEnsembleId = new HashMap<>();

        List<FrontDryRunTransferQuotasDto> oldQuotasDto = model.getQuotasOld().stream()
                .collect(Collectors.groupingBy(QuotaDto::getFolderId, Collectors.mapping(quotaModel ->
                        getFrontDryRunTransferQuotaDto(expanded, locale, sortedUnitModelsByEnsembleId,
                                quotaModel), Collectors.toList())))
                .entrySet().stream()
                .map(e -> new FrontDryRunTransferQuotasDto(e.getKey(), e.getValue()))
                .collect(Collectors.toList());

        List<FrontDryRunTransferQuotasDto> newQuotasDto = model.getQuotasNew().stream()
                .collect(Collectors.groupingBy(QuotaDto::getFolderId, Collectors.mapping(quotaDto ->
                        getFrontDryRunTransferQuotaDto(expanded, locale, sortedUnitModelsByEnsembleId,
                                quotaDto), Collectors.toList())))
                .entrySet().stream()
                .map(e -> new FrontDryRunTransferQuotasDto(e.getKey(), e.getValue()))
                .collect(Collectors.toList());

        List<FrontDryRunTransferAccountQuotasDto> oldAccountQuotas = getFrontDryRunTransferAccountQuotasDto(
                model.getAccountQuotasOld(), expanded, locale, sortedUnitModelsByEnsembleId);
        List<FrontDryRunTransferAccountQuotasDto> newAccountQuotas = getFrontDryRunTransferAccountQuotasDto(
                model.getAccountQuotasNew(), expanded, locale, sortedUnitModelsByEnsembleId);
        FrontDryRunTransferParameters dryRunTransferParameters = new FrontDryRunTransferParameters(
                model.getDryRunParameters()
                        .stream()
                        .flatMap(x -> x.getRelatedProvisionTransfers().stream())
                        .map(x -> FrontProvisionTransferDto.fromModel(x, expanded.getResources(),
                                expanded.getUnitsEnsembles(), locale))
                        .collect(Collectors.toList()));
        builder.transfer(new FrontDryRunTransferRequestDto(
                TransferRequestTypeDto.fromModel(model.getType()), oldQuotasDto, newQuotasDto,
                oldAccountQuotas, newAccountQuotas, dryRunTransferParameters));
        Set<String> actuallyUsedUnits = new HashSet<>();
        Stream.concat(newQuotasDto.stream(), oldQuotasDto.stream())
                .flatMap(qs -> qs.getQuotas().stream())
                .flatMap(q -> Stream.of(q.getQuotaUnitId(), q.getBalanceUnitId(), q.getFrozenQuotaUnitId()))
                .forEach(actuallyUsedUnits::add);
        Stream.concat(newAccountQuotas.stream(), oldAccountQuotas.stream())
                .flatMap(a -> a.getQuotas().stream())
                .flatMap(aq -> Stream.of(aq.getAllocatedQuotaUnitId(), aq.getProvidedQuotaUnitId(),
                        aq.getFrozenProvidedQuotaUnitId()))
                .forEach(actuallyUsedUnits::add);
        dryRunTransferParameters.getRelatedProvisionTransfers().stream()
                .flatMap(rpt -> Stream.concat(rpt.getSourceAccountTransfers().stream(),
                        rpt.getDestinationAccountTransfers().stream()))
                .map(FrontQuotaResourceTransferDto::getDeltaUnitId)
                .forEach(actuallyUsedUnits::add);
        fillDictionary(expanded, locale, actuallyUsedUnits, builder);
        expanded.getTransferRequests().getPermissions().ifPresent(permissions ->
                builder.permissions(new FrontDryRunTransferPermissionsDto(
                        permissions.isCanVote(),
                        permissions.isCanAutoConfirmAsProviderResponsible(),
                        permissions.isCanConfirmSingleHandedly()
                ))
        );

        expanded.getTransferRequests().getTransferResponsible().ifPresent(responsible ->
                builder.responsible(toResponsible(responsible))
        );

        DryRunTransferRequestWarnings warnings = expanded.getTransferRequests().getWarnings();
        builder.warnings(new FrontDryRunTransferWarningsDto(
                warnings.getMessages(),
                warnings.getPerResource(),
                warnings.getPerFolder().entrySet().stream()
                        .collect(Collectors.toMap(Map.Entry::getKey, e -> new FrontDryRunTransferFolderWarningsDto(
                                e.getValue().getMessages(),
                                e.getValue().getPerResource(),
                                e.getValue().getDetailsPerResource()
                        ))),
                warnings.getDetailsPerResource(),
                warnings.getPerAccount().entrySet().stream()
                        .collect(Collectors.toMap(Map.Entry::getKey, e -> new FrontDryRunTransferAccountWarningsDto(
                                e.getValue().getMessages(),
                                e.getValue().getPerResource(),
                                e.getValue().getDetailsPerResource()
                        )))
        ));

        builder.errors(Errors.toDto(expanded.getTransferRequests().getErrors()));

        return builder.build();
    }

    private static List<FrontDryRunTransferAccountQuotasDto> getFrontDryRunTransferAccountQuotasDto(
            List<AccountQuotaDto> accountQuotas,
            ExpandedTransferRequests<?> expanded,
            Locale locale,
            Map<String, List<UnitModel>> sortedUnitModelsByEnsembleId
    ) {
        return accountQuotas.stream()
                .collect(Collectors.groupingBy(AccountQuotaDto::getAccountId, Collectors.mapping(accountQuotaDto ->
                        getFrontDryRunTransferAccountQuotaDto(expanded, locale, sortedUnitModelsByEnsembleId,
                                accountQuotaDto), Collectors.toList())))
                .entrySet().stream()
                .map(e -> new FrontDryRunTransferAccountQuotasDto(e.getKey(), e.getValue()))
                .collect(Collectors.toList());
    }

    private static FrontDryRunTransferAccountQuotaDto getFrontDryRunTransferAccountQuotaDto(
            ExpandedTransferRequests<?> expanded,
            Locale locale,
            Map<String, List<UnitModel>> sortedUnitModelsByEnsembleId,
            AccountQuotaDto accountQuotaDto
    ) {
        ResourceModel resourceModel = expanded.getResources().get(accountQuotaDto.getResourceId());
        UnitsEnsembleModel unitsEnsembleModel = expanded.getUnitsEnsembles().get(resourceModel.getUnitsEnsembleId());
        Optional<UnitModel> baseUnit = unitsEnsembleModel.unitById(resourceModel.getBaseUnitId());
        return fromModel(accountQuotaDto, sortedUnitModelsByEnsembleId.computeIfAbsent(
                        unitsEnsembleModel.getId(),
                        eid -> unitsEnsembleModel.getUnits().stream()
                                .sorted(UnitsComparator.INSTANCE).collect(Collectors.toList())
                ),
                baseUnit.get(), locale);
    }

    private FrontDryRunTransferQuotaDto getFrontDryRunTransferQuotaDto(
            ExpandedTransferRequests<?> expanded, Locale locale,
            Map<String, List<UnitModel>> sortedUnitModelsByEnsembleId, QuotaDto quotaDto) {
        ResourceModel resourceModel = expanded.getResources().get(quotaDto.getResourceId());
        UnitsEnsembleModel unitsEnsembleModel =
                expanded.getUnitsEnsembles().get(resourceModel.getUnitsEnsembleId());
        Optional<UnitModel> baseUnit = unitsEnsembleModel.unitById(resourceModel.getBaseUnitId());
        return fromModel(quotaDto, sortedUnitModelsByEnsembleId.computeIfAbsent(
                unitsEnsembleModel.getId(),
                eid -> unitsEnsembleModel.getUnits().stream()
                        .sorted(UnitsComparator.INSTANCE).collect(Collectors.toList())
                ),
                baseUnit.get(), locale);
    }
}
