package ru.yandex.intranet.d.services.transfer;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.transfers.ReserveResponsibleModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByFolderModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByResponsibleModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByServiceModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestIndices;

/**
 * TransferRequestIndicesService.
 * Сервис для работы с индексами TransferRequest-а.
 * Позволяет загружать сущесвующие, вычислять новые, находить разницу и т.д.
 *
 * @author Petr Surkov <petrsurkov@yandex-team.ru>
 */
@Component
public class TransferRequestIndicesService {
    private final TransferRequestStoreService storeService;

    public TransferRequestIndicesService(TransferRequestStoreService storeService) {
        this.storeService = storeService;
    }

    /**
     * Считает разницу между двумя индексами
     */
    public static TransferRequestIndices.Difference difference(
            TransferRequestIndices oldIndices,
            TransferRequestIndices newIndices
    ) {
        return new TransferRequestIndices.Difference(
                indexDifference(newIndices, oldIndices, TransferRequestIndices::getResponsibleIndices),
                indexDifference(oldIndices, newIndices, TransferRequestIndices::getResponsibleIndices),
                indexDifference(newIndices, oldIndices, TransferRequestIndices::getFolderIndices),
                indexDifference(oldIndices, newIndices, TransferRequestIndices::getFolderIndices),
                indexDifference(newIndices, oldIndices, TransferRequestIndices::getServiceIndices),
                indexDifference(oldIndices, newIndices, TransferRequestIndices::getServiceIndices)
        );
    }

    /**
     * Загружает существующие индексы для трансфер реквеста
     */
    public Mono<TransferRequestIndices> loadTransferRequestIndices(YdbTxSession txSession,
                                                                   TransferRequestModel request) {
        return storeService.loadByResponsibleIndex(txSession, request).flatMap(byResponsibleIndices ->
                storeService.loadByFolderIndex(txSession, request).flatMap(byFolderIndices ->
                        storeService.loadByServiceIndex(txSession, request).map(byServiceIndices ->
                                new TransferRequestIndices(byResponsibleIndices, byFolderIndices, byServiceIndices))));
    }

    /**
     * <p>Считает новые индексы для трансфер реквеста</p>
     *
     * <p>Можно использовать вкупе с {@link #difference(TransferRequestIndices, TransferRequestIndices)}}
     * чтобы найти разницу со старыми индексами</p>
     */
    public static TransferRequestIndices calculateTransferRequestIndices(TransferRequestModel transferRequestModel) {
        TransferRequestType type = transferRequestModel.getType();
        return switch (type) {
            case QUOTA_TRANSFER, RESERVE_TRANSFER -> calculateTransferQuotaRequestIndices(transferRequestModel);
            case PROVISION_TRANSFER -> calculateTransferProvisionRequestIndices(transferRequestModel);
            default -> throw new IllegalArgumentException("Unsupported transfer request type '" + type
                    + "' for index creation");
        };
    }

    private static TransferRequestIndices calculateTransferQuotaRequestIndices(TransferRequestModel transferRequest) {

        /* Responsible indices */
        Set<TransferRequestByResponsibleModel> responsibleIndices = calculateResponsibleIndices(transferRequest,
                transferRequest.getResponsible());

        /* Folder indices */
        Set<TransferRequestByFolderModel> folderIndices = transferRequest.getParameters()
                .getQuotaTransfers()
                .stream()
                .map(quotaTransfer -> byFolderIndex(quotaTransfer.getDestinationFolderId(), transferRequest))
                .collect(Collectors.toSet());

        /* Service indices */
        Set<TransferRequestByServiceModel> serviceIndices = transferRequest.getParameters()
                .getQuotaTransfers()
                .stream()
                .map(quotaTransfer -> byServiceIndex(quotaTransfer.getDestinationServiceId(), transferRequest))
                .collect(Collectors.toSet());
        return new TransferRequestIndices(responsibleIndices, folderIndices, serviceIndices);
    }

    private static TransferRequestIndices calculateTransferProvisionRequestIndices(
            TransferRequestModel transferRequest
    ) {
        /* Responsible indices */
        Set<TransferRequestByResponsibleModel> responsibleIndices = calculateResponsibleIndices(transferRequest,
                transferRequest.getResponsible());

        /* Folder indices */
        Set<TransferRequestByFolderModel> folderIndices = transferRequest.getParameters()
                .getProvisionTransfers()
                .stream()
                .flatMap(provisionTransfer -> Stream.of(
                        byFolderIndex(provisionTransfer.getDestinationFolderId(), transferRequest),
                        byFolderIndex(provisionTransfer.getSourceFolderId(), transferRequest)))
                .collect(Collectors.toSet());

        /* Service indices */
        Set<TransferRequestByServiceModel> serviceIndices = transferRequest.getParameters()
                .getProvisionTransfers()
                .stream()
                .flatMap(provisionTransfer -> Stream.of(
                        byServiceIndex(provisionTransfer.getSourceServiceId(), transferRequest),
                        byServiceIndex(provisionTransfer.getDestinationServiceId(), transferRequest)))
                .collect(Collectors.toSet());
        return new TransferRequestIndices(responsibleIndices, folderIndices, serviceIndices);
    }

    @NotNull
    private static TransferRequestByServiceModel byServiceIndex(long serviceId,
                                                                TransferRequestModel transferRequest) {
        return TransferRequestByServiceModel.builder()
                .tenantId(Tenants.DEFAULT_TENANT_ID)
                .serviceId(serviceId)
                .status(transferRequest.getStatus())
                .createdAt(transferRequest.getCreatedAt())
                .transferRequestId(transferRequest.getId())
                .build();
    }

    @NotNull
    private static TransferRequestByFolderModel byFolderIndex(String provisionTransfer,
                                                              TransferRequestModel transferRequest) {
        return TransferRequestByFolderModel.builder()
                .tenantId(Tenants.DEFAULT_TENANT_ID)
                .folderId(provisionTransfer)
                .status(transferRequest.getStatus())
                .createdAt(transferRequest.getCreatedAt())
                .transferRequestId(transferRequest.getId())
                .build();
    }

    public static Set<TransferRequestByResponsibleModel> calculateResponsibleIndices(
            TransferRequestModel transferRequest,
            TransferResponsible responsible) {
        Set<TransferRequestByResponsibleModel> responsibleIndices = new HashSet<>();
        responsible.getResponsible().forEach(foldersResponsible ->
                foldersResponsible.getResponsible().forEach(serviceResponsible ->
                        serviceResponsible.getResponsibleIds().forEach(responsibleId ->
                                responsibleIndices.add(TransferRequestByResponsibleModel.builder()
                                        .tenantId(Tenants.DEFAULT_TENANT_ID)
                                        .responsibleId(responsibleId)
                                        .status(transferRequest.getStatus())
                                        .createdAt(transferRequest.getCreatedAt())
                                        .transferRequestId(transferRequest.getId())
                                        .build()))));
        // если разносить switchом, то надо быть аккуратным тут
        // дело в том, что reserveTransfer может ссылаться с некоторого момента на quotaTransfer
        // так что по getReserveResponsibleModel возможно надо пройтись даже для quotaTransfer
        responsible.getReserveResponsibleModel()
                .map(ReserveResponsibleModel::getResponsibleIds)
                .ifPresent(responsibleIds -> responsibleIds.forEach(responsibleId ->
                        responsibleIndices.add(TransferRequestByResponsibleModel.builder()
                                .tenantId(Tenants.DEFAULT_TENANT_ID)
                                .responsibleId(responsibleId)
                                .status(transferRequest.getStatus())
                                .createdAt(transferRequest.getCreatedAt())
                                .transferRequestId(transferRequest.getId())
                                .build())));
        return responsibleIndices;
    }

    private static <T> Set<T> indexDifference(TransferRequestIndices left,
                                              TransferRequestIndices right,
                                              Function<TransferRequestIndices, List<T>> getter) {
        return Sets.difference(
                new HashSet<>(getter.apply(left)),
                new HashSet<>(getter.apply(right))
        );
    }
}
