package ru.yandex.intranet.d.services.transfer

import com.yandex.ydb.table.transaction.TransactionMode
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.meter
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.kotlin.withMDC
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.model.transfers.TransferRequestType
import ru.yandex.intranet.d.services.quotas.MoveProvisionOperationResult
import ru.yandex.intranet.d.services.quotas.TransferRequestCloser
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests
import ru.yandex.intranet.d.services.transfer.model.TransferRequestProviderData
import ru.yandex.intranet.d.services.transfer.model.ResponsibleAndNotified
import ru.yandex.intranet.d.services.transfer.model.TransferRequestCreationResult
import ru.yandex.intranet.d.services.transfer.ticket.TransferRequestTicketService
import ru.yandex.intranet.d.services.uniques.RequestUniqueService
import ru.yandex.intranet.d.util.MdcKey
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestDto
import ru.yandex.intranet.d.web.model.transfers.front.FrontPutTransferRequestDto
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestVotingDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.util.*

private val logger = KotlinLogging.logger {}

/**
 * Transfer request logic service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class TransferRequestLogicService(private val securityManagerService: SecurityManagerService,
                                  private val requestUniqueService: RequestUniqueService,
                                  private val securityService: TransferRequestSecurityService,
                                  private val answerService: TransferRequestAnswerService,
                                  private val requestService: TransferRequestService,
                                  private val transferRequestTicketService: TransferRequestTicketService,
                                  private val transferRequestNotificationsService: TransferRequestNotificationsService,
                                  private val validationService: TransferRequestValidationService,
                                  private val storeService: TransferRequestStoreService,
                                  private val responsibleService: TransferRequestResponsibleAndNotifyService,
                                  private val tableClient: YdbTableClient,
                                  private val provisionService: TransferRequestProvisionService,
                                  private val permissionService: TransferRequestPermissionService,
                                  @Qualifier("messageSource") private val messages: MessageSource
): TransferRequestCloser {

    fun createMono(transferRequest: FrontCreateTransferRequestDto,
                   currentUser: YaUserDetails,
                   locale: Locale,
                   publicApi: Boolean,
                   idempotencyKey: String?,
                   delayValidation: Boolean
    ): Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> {
        return mono { create(transferRequest, currentUser, locale, publicApi, idempotencyKey, delayValidation) }
    }

    suspend fun create(
        transferRequest: FrontCreateTransferRequestDto,
        currentUser: YaUserDetails,
        locale: Locale,
        publicApi: Boolean,
        idempotencyKey: String?,
        delayValidation: Boolean,
    ): Result<ExpandedTransferRequests<TransferRequestModel>> = binding {
        securityManagerService.checkReadPermissions(currentUser, locale).awaitSingle().bind()
        val checkedIdempotencyKey = requestUniqueService.validateIdempotencyKey(idempotencyKey, locale).bind()
        val existing = dbSessionRetryable(tableClient) {
            val existingTransferRequest = requestUniqueService.checkCreateTransferRequest(
                session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY), checkedIdempotencyKey, currentUser)
            if (existingTransferRequest != null) {
                answerService.expand(session, existingTransferRequest).awaitSingle()
            } else {
                null
            }
        }
        if (existing == null) {
            val providerData = getPreCreateData(transferRequest, currentUser, locale, publicApi).bind()
            val (expanded, created) = dbSessionRetryable(tableClient) {
                val createdRequest = rwTxRetryable {
                    val checkedCreate = requestService.validateCreate(txSession, transferRequest, providerData,
                        currentUser, locale, publicApi, delayValidation).awaitSingle().bind()
                    val applicableCreate = if (delayValidation && checkedCreate!!.isApply) {
                        requestService.validateCreateApplicable(checkedCreate)
                    } else {
                        checkedCreate
                    }
                    requestService.saveCreate(txSession, applicableCreate!!, currentUser, checkedIdempotencyKey)
                        .awaitSingleOrNull()
                }!!
                Pair(answerService.expand(session, createdRequest).awaitSingle(), createdRequest)
            }!!
            val createdIssueResult = transferRequestTicketService.createTicket(Tuples.of(expanded, created))
                .awaitSingle()
            createdIssueResult.doOnFailure { e -> logger.warn { "Failed to create tracker issue: $e" } }
            transferRequestNotificationsService.sendNotifications(created.notifiedUsers, created.request.id)
            requestService.deferredCreate(created, locale).awaitSingle()
            return Result.success(expanded)
        } else {
            securityService.checkReadPermissions(existing.transferRequests, currentUser, locale).awaitSingle().bind()
            return Result.success(existing)
        }
    }

    private suspend fun getPreCreateData(
        transferRequest: FrontCreateTransferRequestDto,
        currentUser: YaUserDetails,
        locale: Locale,
        publicApi: Boolean,
    ): Result<TransferRequestProviderData?> = binding {
        if (currentUser.user.isEmpty) {
            return Result.failure(ErrorCollection.builder().addError(
                TypedError.forbidden(messages.getMessage("errors.access.denied", null, locale))
            ).build())
        }
        val type = validationService.validateTransferRequestType({ transferRequest.requestType }, locale).bind()
        val preCreateProviderData = when (type) {
            TransferRequestType.PROVISION_TRANSFER -> {
                val result: Result<TransferRequestProviderData> = binding {
                    val preCreateTransferRequestData = dbSessionRetryable(tableClient) {
                        meter(logger, "Provision transfer request create, load data for request") {
                            provisionService.extractPreCreateData(session, transferRequest, locale, publicApi).bind()
                        }
                    }!!
                    meter(logger, "Provision transfer request create, get accounts from provider") {
                        provisionService.preCreateProviderData(preCreateTransferRequestData, locale)
                    }
                }
                result.match({ it }, {
                    logger.warn { "Error while pre-fetch provider accounts: $it" }
                    null
                })
            }
            else -> null
        }
        return Result.success(preCreateProviderData)
    }

    fun putMono(id: String,
                    version: Long?,
                    transferRequest: FrontPutTransferRequestDto,
                    currentUser: YaUserDetails,
                    locale: Locale,
                    publicApi: Boolean,
                    idempotencyKey: String?,
                    delayValidation: Boolean): Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> {
        return mono {
            withMDC(MdcKey.COMMON_TRANSFER_REQUEST_ID to id) {
                put(id, version, transferRequest, currentUser, locale, publicApi, idempotencyKey, delayValidation)
            }
        }
    }

    suspend fun put(id: String,
                    version: Long?,
                    transferRequest: FrontPutTransferRequestDto,
                    currentUser: YaUserDetails,
                    locale: Locale,
                    publicApi: Boolean,
                    idempotencyKey: String?,
                    delayValidation: Boolean): Result<ExpandedTransferRequests<TransferRequestModel>> = binding {
        securityManagerService.checkReadPermissions(currentUser, locale).awaitSingle().bind()
        validationService.validateId(id, locale).bind()
        val checkedIdempotencyKey = requestUniqueService.validateIdempotencyKey(idempotencyKey, locale).bind()
        val (updated, expanded) = dbSessionRetryable(tableClient) {
            val (existingRequest, updatedRequestR) = rwTxRetryable {
                val requestO = storeService.getById(txSession, id).awaitSingle()
                val request = validationService.validateExists(requestO.orElse(null), locale).bind()!!
                securityService.checkReadPermissions(request, currentUser, locale).awaitSingle().bind()
                val existingTransferRequest = requestUniqueService.checkPutTransferRequest(txSession,
                    checkedIdempotencyKey, currentUser)
                if (existingTransferRequest != null) {
                    validateIdempotencyKeyMatch(existingTransferRequest, request, locale).bind()
                    Pair (existingTransferRequest, null)
                } else {
                    val responsible = responsibleService.calculateForTransferRequestModel(txSession, request, false)
                        .awaitSingle()
                    val putR = doPut(txSession, transferRequest, request, responsible, version, currentUser,
                        locale, publicApi, checkedIdempotencyKey, delayValidation)
                    val updatedTransferRequestR = putR.match(
                        { tr -> requestService.toRefreshResult(putR, tr, request) },
                        { requestService.refreshRequest(txSession, putR, request, responsible, currentUser) }
                    ).awaitSingle()
                    Pair (null, updatedTransferRequestR)
                }
            }!!
            if (updatedRequestR != null) {
                val updatedRefreshed = requestService.completeRefresh(session, updatedRequestR, false)
                    .awaitSingle().bind()!!
                val updatedTransferRequest = requestService.deferredPut(updatedRefreshed, locale).awaitSingle()
                val expandedTransferRequest = answerService.expand(session, updatedTransferRequest).awaitSingle()
                Pair (updatedTransferRequest, expandedTransferRequest)
            } else {
                val expandedTransferRequest = answerService.expand(session, existingRequest).awaitSingle()
                Pair (null, expandedTransferRequest)
            }
        }!!
        if (updated != null) {
            val updatedIssueResult = transferRequestTicketService.updateTicket(expanded, updated).awaitSingle()
            updatedIssueResult.doOnFailure { e -> logger.warn { "Failed to update tracker issue: $e" } }
            val closeIssueResult = requestService.closeIfNeeded(expanded.transferRequests, expanded).awaitSingle()
            closeIssueResult.doOnFailure { e -> logger.warn { "Failed to close tracker issue: $e" } }
        }
        return Result.success(expanded)
    }

    fun cancelMono(id: String,
                   version: Long?,
                   currentUser: YaUserDetails,
                   locale: Locale,
                   idempotencyKey: String?): Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> {
        return mono {
            withMDC(MdcKey.COMMON_TRANSFER_REQUEST_ID to id) {
                cancel(id, version, currentUser, locale, idempotencyKey)
            }
        }
    }

    suspend fun cancel(id: String,
                       version: Long?,
                       currentUser: YaUserDetails,
                       locale: Locale,
                       idempotencyKey: String?): Result<ExpandedTransferRequests<TransferRequestModel>> = binding {
        securityManagerService.checkReadPermissions(currentUser, locale).awaitSingle().bind()
        validationService.validateId(id, locale).bind()
        val checkedIdempotencyKey = requestUniqueService.validateIdempotencyKey(idempotencyKey, locale).bind()
        val (cancelled, expanded) = dbSessionRetryable(tableClient) {
            val (existingRequest, cancelledRequestR) = rwTxRetryable {
                val requestO = storeService.getById(txSession, id).awaitSingle()
                val request = validationService.validateExists(requestO.orElse(null), locale).bind()!!
                val existingTransferRequest = requestUniqueService.checkCancelTransferRequest(txSession,
                    checkedIdempotencyKey, currentUser)
                if (existingTransferRequest != null) {
                    validateIdempotencyKeyMatch(existingTransferRequest, request, locale).bind()
                    securityService.checkReadPermissions(request, currentUser, locale).awaitSingle().bind()
                    Pair (existingTransferRequest, null)
                } else {
                    val responsible = responsibleService.calculateForTransferRequestModel(txSession, request, false)
                        .awaitSingle()
                    val cancelR = doCancel(txSession, request, responsible, version, currentUser, locale,
                        checkedIdempotencyKey)
                    val cancelledTransferRequestR = cancelR.match(
                        { tr -> requestService.toCancelRefreshResult(cancelR, tr) },
                        { requestService.refreshRequest(txSession, cancelR, request, responsible, currentUser) }
                    ).awaitSingle()
                    Pair (null, cancelledTransferRequestR)
                }
            }!!
            if (cancelledRequestR != null) {
                val cancelledRefreshed = requestService.completeRefresh(session, cancelledRequestR, true)
                    .awaitSingle().bind()!!
                val expandedTransferRequest = answerService.expand(session, cancelledRefreshed).awaitSingle()
                Pair (cancelledRefreshed, expandedTransferRequest)
            } else {
                val expandedTransferRequest = answerService.expand(session, existingRequest).awaitSingle()
                Pair (null, expandedTransferRequest)
            }
        }!!
        if (cancelled != null) {
            val closeIssueResult = transferRequestTicketService.closeTicket(cancelled, cancelled).awaitSingle()
            closeIssueResult.doOnFailure { e -> logger.warn { "Failed to close tracker issue: $e" } }
        }
        return Result.success(expanded)
    }

    fun voteMono(id: String,
             version: Long?,
             voteParameters: FrontTransferRequestVotingDto,
             currentUser: YaUserDetails,
             locale: Locale,
             idempotencyKey: String?): Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> {
        return mono {
            withMDC(MdcKey.COMMON_TRANSFER_REQUEST_ID to id) {
                vote(id, version, voteParameters, currentUser, locale, idempotencyKey)
            }
        }
    }

    suspend fun vote(id: String,
                     version: Long?,
                     voteParameters: FrontTransferRequestVotingDto,
                     currentUser: YaUserDetails,
                     locale: Locale,
                     idempotencyKey: String?): Result<ExpandedTransferRequests<TransferRequestModel>> = binding {
        securityManagerService.checkReadPermissions(currentUser, locale).awaitSingle().bind()
        validationService.validateId(id, locale).bind()
        val checkedIdempotencyKey = requestUniqueService.validateIdempotencyKey(idempotencyKey, locale).bind()
        val (votedFor, expanded) = dbSessionRetryable(tableClient) {
            val (existingRequest, votedForRequestR) = rwTxRetryable {
                val requestO = storeService.getById(txSession, id).awaitSingle()
                val request = validationService.validateExists(requestO.orElse(null), locale).bind()!!
                securityService.checkReadPermissions(request, currentUser, locale).awaitSingle().bind()
                val existingTransferRequest = requestUniqueService.checkVoteForTransferRequest(txSession,
                    checkedIdempotencyKey, currentUser)
                if (existingTransferRequest != null) {
                    validateIdempotencyKeyMatch(existingTransferRequest, request, locale).bind()
                    Pair (existingTransferRequest, null)
                } else {
                    val responsible = responsibleService.calculateForTransferRequestModel(txSession, request, false)
                        .awaitSingle()
                    val voteR = doVote(txSession, voteParameters, request, responsible, currentUser, version, locale,
                        checkedIdempotencyKey)
                    val votedForTransferRequestR = voteR.match(
                        { tr -> requestService.toRefreshResult(voteR, tr, request) },
                        { requestService.refreshRequest(txSession, voteR, request, responsible, currentUser)}
                    ).awaitSingle()
                    Pair (null, votedForTransferRequestR)
                }
            }!!
            if (votedForRequestR != null) {
                val votedForRefreshed = requestService.completeRefresh(session, votedForRequestR, true)
                    .awaitSingle().bind()!!
                val votedForTransferRequest = requestService.deferredVote(votedForRefreshed, locale).awaitSingle()
                val expandedTransferRequest = answerService.expand(session, votedForTransferRequest).awaitSingle()
                Pair (votedForTransferRequest, expandedTransferRequest)
            } else {
                val expandedTransferRequest = answerService.expand(session, existingRequest).awaitSingle()
                Pair (null, expandedTransferRequest)
            }
        }!!
        if (votedFor != null) {
            val closeIssueResult = transferRequestTicketService.closeTicket(votedFor.request, votedFor)
                .awaitSingle()
            closeIssueResult.doOnFailure { e -> logger.warn { "Failed to close tracker issue: $e" } }
        }
        return Result.success(expanded)
    }

    private suspend fun doPut(txSession: YdbTxSession,
                              transferRequest: FrontPutTransferRequestDto,
                              request: TransferRequestModel,
                              responsible: ResponsibleAndNotified,
                              version: Long?,
                              currentUser: YaUserDetails,
                              locale: Locale,
                              publicApi: Boolean,
                              unique: String?,
                              delayValidation: Boolean): Result<TransferRequestCreationResult> = binding {
        val validated = requestService.validatePut(txSession, transferRequest,
            request, responsible, version, currentUser, locale, publicApi, delayValidation).awaitSingle().bind()
        val applicable = if (delayValidation && validated!!.isApply && validated.transferRequest.isPresent) {
            requestService.validatePutApplicable(validated)
        } else {
            validated
        }
        return Result.success(requestService.savePut(txSession, applicable, currentUser, unique).awaitSingle())
    }

    private suspend fun doCancel(txSession: YdbTxSession,
                                 request: TransferRequestModel,
                                 responsible: ResponsibleAndNotified,
                                 version: Long?,
                                 currentUser: YaUserDetails,
                                 locale: Locale,
                                 unique: String?): Result<TransferRequestModel> = binding {
        permissionService.checkUserCanCancelRequest(request, responsible.responsible, currentUser, locale).bind()
        validationService.validateVersion(request, version, locale).bind()
        validationService.validateCancellation(request, locale).bind()
        return Result.success(requestService.cancelRequest(txSession, request, responsible, currentUser, unique)
            .awaitSingle())
    }

    private suspend fun doVote(txSession: YdbTxSession,
                               voteParameters: FrontTransferRequestVotingDto,
                               request: TransferRequestModel,
                               responsible: ResponsibleAndNotified,
                               currentUser: YaUserDetails,
                               version: Long?,
                               locale: Locale,
                               unique: String?): Result<TransferRequestCreationResult> = binding {
        val validated = requestService.validateVote(txSession, voteParameters, request, responsible, version,
            currentUser, locale).awaitSingle().bind()
        return Result.success(requestService.saveVote(txSession, validated, currentUser, unique).awaitSingle())
    }

    private fun validateIdempotencyKeyMatch(existingTransferRequest: TransferRequestModel,
                                            requestedTransferRequest: TransferRequestModel,
                                            locale: Locale): Result<Void?> {
        if (existingTransferRequest.id != requestedTransferRequest.id) {
            return Result.failure(ErrorCollection.builder().addError("idempotencyKey", TypedError.invalid(messages
                .getMessage("errors.idempotency.key.mismatch", null, locale))).build())
        }
        return Result.success(null)
    }

    override suspend fun moveProvisionOperationFinished(
        tx: YdbTxSession,
        transferRequest: TransferRequestModel,
        operationId: String,
        operationResult: MoveProvisionOperationResult,
        borrowedLoanId: LoanId?
    ): suspend () -> Unit {
        return when (transferRequest.type) {
            TransferRequestType.PROVISION_TRANSFER -> {
                val result = provisionService.finishRequestOperation(tx, transferRequest, operationId,
                    operationResult.opLogIdsByFolderId, operationResult.requestStatus,
                    operationResult.errorsEn, operationResult.errorsRu, borrowedLoanId)
                if (result.closed) {
                    suspend {
                        postClose(result.updatedTransferRequest)
                    }
                } else {
                    suspend {  }
                }
            }
            else -> suspend {  }
        }
    }

    suspend fun postClose(transferRequestModel: TransferRequestModel) {
        val closeResult = requestService.closeIfNeeded(transferRequestModel, transferRequestModel).awaitSingle()
        closeResult.doOnFailure { e -> logger.warn { "Failed to close tracker issue: $e" } }
    }
}
