package ru.yandex.intranet.d.services.transfer

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.kotlin.AccountId
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.UserId
import ru.yandex.intranet.d.kotlin.getOrNull
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountReserveType
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus
import ru.yandex.intranet.d.model.transfers.TransferRequestType
import ru.yandex.intranet.d.model.transfers.TransferResponsible
import ru.yandex.intranet.d.model.users.UserServiceRoles
import ru.yandex.intranet.d.services.transfer.TransferRequestResponsibleAndNotifyService.isResponsible
import ru.yandex.intranet.d.services.transfer.model.ResponsibleAndNotified
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.Result.Companion.failure
import ru.yandex.intranet.d.util.result.Result.Companion.success
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.util.*


@Component
class TransferRequestPermissionService(
    @Qualifier("messageSource")
    val messages: MessageSource,
) {
    companion object {
        @JvmStatic
        fun canUpdate(
            currentUser: YaUserDetails,
            model: TransferRequestModel,
            providerById: Map<ProviderId, ProviderModel>,
            accountById: Map<AccountId, AccountModel>
        ): Boolean {
            return TransferRequestStatus.PENDING == model.status &&
                (isAuthor(currentUser, model.createdBy) || isResponsible(currentUser, model.responsible)
                    || canUpdateAsProviderResponsible(currentUser, model, providerById, accountById))
        }

        @JvmStatic
        fun canCancel(
            currentUser: YaUserDetails,
            model: TransferRequestModel,
            providerById: Map<ProviderId, ProviderModel>,
            accountById: Map<AccountId, AccountModel>
        ): Boolean = canUpdate(currentUser, model, providerById, accountById)

        @JvmStatic
        fun isResponsible(currentUser: YaUserDetails, responsible: TransferResponsible): Boolean {
            return currentUser.user.isPresent &&
                isResponsible(currentUser.user.get(), responsible)
        }

        @JvmStatic
        fun isAuthor(currentUser: YaUserDetails, authorId: String): Boolean {
            return currentUser.user.isPresent && currentUser.user.get().id == authorId
        }

        @JvmStatic
        fun canVote(
            currentUser: YaUserDetails,
            model: TransferRequestModel,
            accountById: Map<AccountId, AccountModel>,
        ): Boolean {
            if (currentUser.user.isEmpty) {
                return false
            }
            val currentUserId = currentUser.user.get().id
            val notYetVoted = model.votes.votes.none { vote -> currentUserId == vote.userId }
            var hasMoreFoldersToVote = false
            if (!notYetVoted) {
                val votedFolders = HashSet<String>()
                model.votes.votes.forEach { vote ->
                    if (vote.userId == currentUserId) {
                        votedFolders.addAll(vote.folderIds)
                    }
                }
                val foldersToVote = HashSet<String>()
                model.responsible.responsible.forEach { foldersResponsible ->
                    val canVoteForFolders = foldersResponsible.responsible.any { serviceResponsible ->
                            serviceResponsible.responsibleIds
                                .contains(currentUserId)
                        }
                    if (canVoteForFolders) {
                        foldersToVote.addAll(foldersResponsible.folderIds)
                    }
                }
                model.responsible.reserveResponsibleModel.ifPresent { reserveResponsible ->
                    if (reserveResponsible.responsibleIds.contains(currentUserId)) {
                        foldersToVote.add(reserveResponsible.folderId)
                    }
                }
                if ((foldersToVote - votedFolders).isNotEmpty()) {
                    hasMoreFoldersToVote = true
                }
            }
            val canVoteAsProviderResponsible = canVoteAsProviderResponsible(model, currentUserId, accountById)
            val canVote = model.responsible.responsible.any { responsible ->
                    responsible.responsible.any {
                        it.responsibleIds.contains(currentUserId)
                    }
                }
                || model.responsible.reserveResponsibleModel.isPresent
                && model.responsible.reserveResponsibleModel.get().responsibleIds.contains(currentUserId)
                || canVoteAsProviderResponsible
            return TransferRequestStatus.PENDING == model.status && (notYetVoted || hasMoreFoldersToVote) && canVote
        }

        @JvmStatic
        fun canProvideOverCommitReserve(
            currentUser: YaUserDetails,
            model: TransferRequestModel,
            providerById: Map<ProviderId, ProviderModel>,
            accountById: Map<AccountId, AccountModel>,
        ): Boolean = canUpdateAsProviderResponsible(currentUser, model, providerById, accountById)

        private fun canVoteAsProviderResponsible(
            model: TransferRequestModel,
            currentUserId: UserId,
            accountById: Map<AccountId, AccountModel>,
        ): Boolean {
            //todo проверять в зависимости от подтипа
            val hasSourceReserveAccount = model.parameters.provisionTransfers.any {
                AccountReserveType.PROVIDER == accountById[it.sourceAccountId]?.reserveType.getOrNull()
            }
            return model.type == TransferRequestType.PROVISION_TRANSFER
                && hasSourceReserveAccount
                && isProviderResponsible(model.responsible, currentUserId)
        }

        private fun isProviderResponsible(
            responsible: TransferResponsible,
            currentUserId: UserId
        ) = responsible.providerResponsible.any { it.responsibleId == currentUserId }

        private fun canUpdateAsProviderResponsible(
            currentUser: YaUserDetails,
            model: TransferRequestModel,
            providerById: Map<ProviderId, ProviderModel>,
            accountById: Map<AccountId, AccountModel>,
        ): Boolean {
            if (currentUser.user.isEmpty) {
                return false
            }
            //todo проверять в зависимости от подтипа
            val providerServiceIds = currentUser.user.get().roles[UserServiceRoles.RESPONSIBLE_OF_PROVIDER] ?: setOf()
            val allProviderResponsible = model.parameters.provisionTransfers.all {
                val providerServiceId = providerById[it.providerId]?.serviceId
                providerServiceId in providerServiceIds
            }
            val allAccountsAreReserve = model.parameters.provisionTransfers.all { pt ->
                accountById[pt.sourceAccountId]!!.reserveType.map { it == AccountReserveType.PROVIDER }.orElse(false)
            }
            return model.status == TransferRequestStatus.PENDING
                && model.type == TransferRequestType.PROVISION_TRANSFER
                && allProviderResponsible
                && allAccountsAreReserve
        }

        @JvmStatic
        fun canUserPutRequest(
            transferRequest: TransferRequestModel,
            responsibleAndNotified: ResponsibleAndNotified,
            currentUser: YaUserDetails
        ): Boolean {
            return currentUser.user.map { it.id == transferRequest.createdBy }.orElse(false)
                || isResponsible(currentUser.user.get(), responsibleAndNotified.responsible)
        }
    }

    fun checkUserCanVoteForRequest(
        model: TransferRequestModel,
        responsibleAndNotified: ResponsibleAndNotified,
        currentUser: YaUserDetails,
        locale: Locale
    ): Result<Void?> {
        val currentUserId = currentUser.user.get().id
        val responsible = responsibleAndNotified.responsible
        val canVote = responsible.responsible.any { foldersResponsible ->
            foldersResponsible.responsible.any { serviceResponsible ->
                serviceResponsible.responsibleIds.contains(currentUserId)
            }
        }
            || responsible.reserveResponsibleModel.isPresent
            && responsible.reserveResponsibleModel.get().responsibleIds.contains(currentUserId)
        val notYetVoted = model.votes.votes.none { vote -> currentUserId == vote.userId }
        var hasMoreFoldersToVote = false
        if (!notYetVoted) {
            val votedFolders = HashSet<String>()
            model.votes.votes.forEach { vote ->
                if (vote.userId == currentUserId) {
                    votedFolders.addAll(vote.folderIds)
                }
            }
            val foldersToVote = HashSet<String>()
            responsible.responsible.forEach { foldersResponsible ->
                val canVoteForFolders = foldersResponsible.responsible.any { serviceResponsible ->
                    serviceResponsible.responsibleIds
                        .contains(currentUserId)
                }
                if (canVoteForFolders) {
                    foldersToVote.addAll(foldersResponsible.folderIds)
                }
            }
            responsible.reserveResponsibleModel
                .map { it.folderId }
                .ifPresent { foldersToVote.add(it) }
            if ((foldersToVote - votedFolders).isNotEmpty()) {
                hasMoreFoldersToVote = true
            }
        }
        if (!canVote) {
            return failure(
                ErrorCollection.builder()
                    .addError(TypedError.forbidden(messages.getMessage("errors.access.denied", null, locale)))
                    .build()
            )
        }
        return if (!(notYetVoted || hasMoreFoldersToVote)) {
            failure(
                ErrorCollection.builder().addError(
                    TypedError
                        .invalid(messages.getMessage("errors.transfer.request.already.voted", null, locale))
                ).build()
            )
        } else success(null)
    }

    fun checkUserCanVoteForReserveProvisionRequest(
        model: TransferRequestModel,
        responsibleAndNotified: ResponsibleAndNotified,
        currentUser: YaUserDetails,
        locale: Locale,
    ): Result<Void?> {
        val userId = currentUser.user.get().id
        val canVote = isProviderResponsible(responsibleAndNotified.responsible, userId)
        val alreadyVoted = model.votes.votes.any { userId == it.userId }
        if (!canVote) {
            return failure(
                ErrorCollection.builder()
                    .addError(TypedError.forbidden(messages.getMessage("errors.access.denied", null, locale)))
                    .build()
            )
        }
        return if (alreadyVoted) {
            failure(ErrorCollection.builder().addError(
                    TypedError
                        .invalid(messages.getMessage("errors.transfer.request.already.voted", null, locale))
                ).build())
        } else success(null)
    }

    fun checkUserCanCancelRequest(
        request: TransferRequestModel,
        responsible: TransferResponsible,
        currentUser: YaUserDetails,
        locale: Locale
    ): Result<Void?> {
        //todo нужен доступ в зависимости от типа и подтипа заявки
        return if (currentUser.user.isPresent && (request.createdBy == currentUser.user.get().id ||
                isResponsible(currentUser.user.get(), responsible))
        ) {
            success(null)
        } else {
            failure(
                ErrorCollection.builder().addError(
                    TypedError.forbidden(
                        messages
                            .getMessage("errors.only.responsible.can.cancel.transfer.request", null, locale)
                    )
                ).build()
            )
        }
    }

}
