package ru.yandex.intranet.d.services.transfer

import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuple2
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.users.UsersDao
import ru.yandex.intranet.d.datasource.model.YdbSession
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.*
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountReserveType
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.loans.LoanType
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.services.ServiceMinimalModel
import ru.yandex.intranet.d.model.transfers.*
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.model.users.UserModel
import ru.yandex.intranet.d.services.integration.providers.ProvidersIntegrationService
import ru.yandex.intranet.d.services.integration.providers.getUsersFromReceivedAccount
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.GetAccountRequestDto
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyRequestDto
import ru.yandex.intranet.d.services.integration.providers.validateProviderAccountDto
import ru.yandex.intranet.d.services.integration.providers.validateReceivedProvisions
import ru.yandex.intranet.d.services.operations.model.RetryResult
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision
import ru.yandex.intranet.d.services.quotas.MoveProvisionApplicationContext
import ru.yandex.intranet.d.services.quotas.MoveProvisionLogicService
import ru.yandex.intranet.d.services.transfer.model.*
import ru.yandex.intranet.d.util.DisplayUtil
import ru.yandex.intranet.d.util.MdcKey
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.errors.Errors
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestDto
import ru.yandex.intranet.d.web.model.transfers.front.FrontPutTransferRequestDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.time.Instant
import java.util.*
import kotlin.streams.asStream

@Component
class TransferRequestProvisionService(
    private val validationService: TransferRequestValidationService,
    private val responsibleService: TransferRequestResponsibleAndNotifyService,
    private val storeService: TransferRequestStoreService,
    private val securityService: TransferRequestSecurityService,
    private val indicesService: TransferRequestIndicesService,
    private val voteService: TransferRequestVoteService,
    private val moveProvisionLogicService: MoveProvisionLogicService,
    private val transferLoansService: TransferLoansService,
    @Qualifier("messageSource")
    private val messages: MessageSource,
    private val permissionService: TransferRequestPermissionService,
    private val providersIntegrationService: ProvidersIntegrationService,
    private val usersDao: UsersDao,
) {

    suspend fun extractPreCreateData(
        session: YdbSession,
        transferRequest: FrontCreateTransferRequestDto,
        locale: Locale,
        publicApi: Boolean,
    ): Result<PreCreateProvisionTransferData> = binding {
        val preErrorBuilder = ErrorCollection.builder()
        validationService.validateParameters(transferRequest::getParameters, preErrorBuilder, locale)
        if (transferRequest.parameters.isPresent) {
            val preValidatedProvisionTransfersBuilders = validationService.preValidateProvisionTransferParameters(
                transferRequest.parameters.get(), null, preErrorBuilder, locale, publicApi)
            if (preErrorBuilder.hasAnyErrors()) {
                return Result.failure(preErrorBuilder.build())
            }
            val preValidatedProvisionTransfers = preValidatedProvisionTransfersBuilders.map { it.build() }
            val accountsToLoad = preValidatedProvisionTransfers.flatMap {
                listOf(it.sourceAccountId, it.destinationAccountId)
            }.toSet()
            val loadedAccounts = storeService.loadAccounts(session, accountsToLoad).awaitSingle()

            val accountFolderIds = loadedAccounts.map { it.folderId }.toSet()
            val requestFolderIds = preValidatedProvisionTransfers.flatMap {
                listOf(it.sourceFolderId, it.destinationFolderId)
            }.toSet()
            val loadedFolders = storeService.loadFolders(session, accountFolderIds + requestFolderIds).awaitSingle()

            val providerIds = loadedAccounts.map { it.providerId }.toSet()
            val providers = storeService.loadProviders(session, providerIds).awaitSingle()

            val accountSpaceIds = loadedAccounts.mapNotNull { it.accountsSpacesId.getOrNull() }.toList()
            val accountSpaces = if (accountSpaceIds.isNotEmpty()) {
                storeService.loadAccountSpaces(session, accountSpaceIds).awaitSingle()
            } else {
                listOf()
            }

            val segmentationIds = accountSpaces.flatMap { it.segments.map { it.segmentationId } }.toList()
            val segmentations = storeService.loadResourceSegmentations(session, segmentationIds).awaitSingle()

            val segmentIds = accountSpaces.flatMap { it.segments.map { it.segmentId } }.toList()
            val segments = storeService.loadResourceSegments(session, segmentIds).awaitSingle()

            return Result.success(PreCreateProvisionTransferData(loadedAccounts, loadedFolders, providers,
                accountSpaces, segmentations, segments))
        }
        return Result.failure(preErrorBuilder.build())
    }

    suspend fun preCreateProviderData(
        preCreateData: PreCreateProvisionTransferData,
        locale: Locale,
    ): Result<TransferRequestProviderData> = binding {
        val folderById = preCreateData.folders.associateBy { it.id }
        val providerById = preCreateData.providers.associateBy { it.id }
        val segmentById = preCreateData.segments.associateBy { it.id }
        val segmentationById = preCreateData.segmentations.associateBy { it.id }
        val accountSpaceById = preCreateData.accountSpaces.associateBy { it.id }
        val receivedAccounts = preCreateData.accounts.map { account ->
            val provider = providerById[account.providerId]!!
            val folder = folderById[account.folderId]!!
            val accountSpace = account.accountsSpacesId.map { accountSpaceById[it] }.getOrNull()
            val getAccountRequestDto = GetAccountRequestDto(true, true, folder.id, folder.serviceId,
                prepareAccountSpaceKey(accountSpace, segmentById, segmentationById)
            )
            val result = providersIntegrationService.getAccount(
                account.id, provider, getAccountRequestDto,
                locale
            ).awaitSingle().bind()!!
            val receivedAccount = result.matchSuspend({ accountDto, requestId ->
                validateProviderAccountDto(accountDto, messages, locale)
            }, {
                Result.failure(
                    ErrorCollection.builder()
                        .addError(TypedError.invalid("Couldn't get account ${account.id} from provider: $it"))
                        .build()
                )
            }, { err, requestId ->
                val providerError = Errors.flattenProviderErrorResponse(err, null)
                Result.failure(
                    ErrorCollection.builder()
                        .addError(TypedError.invalid("Couldn't get account ${account.id} from provider: $providerError"))
                        .build()
                )
            }).bind()
            receivedAccount!!
        }.toList()
        return Result.success(ProvisionTransferProviderData(receivedAccounts))
    }

    private fun prepareAccountSpaceKey(
        accountSpaceModel: AccountSpaceModel?,
        segmentById: Map<SegmentId, ResourceSegmentModel>,
        segmentationById: Map<SegmentationId, ResourceSegmentationModel>,
    ): AccountsSpaceKeyRequestDto? {
        if (accountSpaceModel == null) {
            return null
        }
        val segments = accountSpaceModel.segments.map {
            val segmentModel = segmentById[it.segmentId]!!
            val segmentationModel = segmentationById[it.segmentationId]!!
            SegmentKeyRequestDto(segmentationModel.key, segmentModel.key)
        }.toList()
        return AccountsSpaceKeyRequestDto(segments)
    }

    fun validateProvisionTransferMono(
        txSession: YdbTxSession,
        transferRequest: FrontCreateTransferRequestDto,
        providerData: TransferRequestProviderData?,
        currentUser: YaUserDetails,
        locale: Locale,
        publicApi: Boolean,
        delayValidation: Boolean
    ): Mono<Result<ValidatedCreateTransferRequest>> = mono {
        validateProvisionTransfers(txSession, transferRequest, providerData, currentUser, locale, publicApi,
            delayValidation)
    }

    suspend fun validateProvisionTransfers(
        txSession: YdbTxSession,
        transferRequest: FrontCreateTransferRequestDto,
        transferRequestProviderData: TransferRequestProviderData?,
        currentUser: YaUserDetails,
        locale: Locale,
        publicApi: Boolean,
        delayValidation: Boolean
    ): Result<ValidatedCreateTransferRequest> = binding {
        val providerData = transferRequestProviderData as? ProvisionTransferProviderData?
        val preValidatedParametersBuilder = PreValidatedTransferRequestParameters.builder()
        val preErrorBuilder = ErrorCollection.builder()
        validationService.validateCreateFields(transferRequest::getDescription, transferRequest::getParameters,
            preValidatedParametersBuilder, preErrorBuilder, locale)
        val preValidatedLoanParameters = transferLoansService.preValidateLoanFields(transferRequest.loanParameters
            .orElse(null), preErrorBuilder, currentUser, locale)
        if (transferRequest.parameters.isPresent) {
            val preValidatedProvisionTransfersBuilders = validationService.preValidateProvisionTransferParameters(
                transferRequest.parameters.get(), preValidatedLoanParameters,
                preErrorBuilder, locale, publicApi)
            if (preErrorBuilder.hasAnyErrors()) {
                return Result.failure(preErrorBuilder.build())
            }
            val preValidatedParameters = preValidatedParametersBuilder.build()
            val preValidatedProvisionTransfers = preValidatedProvisionTransfersBuilders.map { it.build() }
            val validatedParameters = validateProvisionTransfersParameters(txSession,
                preValidatedProvisionTransfers, providerData, currentUser, locale, publicApi).bind()!!
            val validatedLoanParameters = transferLoansService.validateLoanFields(txSession,
                preValidatedLoanParameters, validatedParameters, currentUser, locale).bind()
            val responsible = calculateResponsible(txSession, validatedParameters, currentUser)
            validateUserAction(validatedParameters, validatedLoanParameters, responsible.responsible, currentUser,
                locale).bind()
            val confirmationData = validationService.validateResponsible(responsible,
                transferRequest::getAddConfirmation, currentUser, locale).bind()!!
            val applicationData = validateProvisionTransferApplication(txSession, validatedParameters,
                validatedLoanParameters, providerData, locale, delayValidation).bind()!!
            val validatedCreateTransferRequest = prepareCreateProvisionTransferRequest(validatedParameters,
                preValidatedParameters, applicationData, responsible, confirmationData, validatedLoanParameters)

            return Result.success(validatedCreateTransferRequest)
        } else {
            return Result.failure(preErrorBuilder.build())
        }
    }

    private suspend fun calculateResponsible(
        txSession: YdbTxSession,
        validatedParameters: ValidatedProvisionTransferParameters,
        currentUser: YaUserDetails
    ): ResponsibleAndNotified {
        //todo перейти на поддержку подтипов
        return if (requireProviderResponsible(validatedParameters)) {
            responsibleService.calculateForReserveProvisionTransfer(txSession, validatedParameters, currentUser)
                .awaitSingle()
        } else {
            responsibleService.calculateForProvisionTransferCreate(txSession, validatedParameters,
                currentUser).awaitSingle()
        }
    }

    private fun requireProviderResponsible(validatedParameters: ValidatedProvisionTransferParameters): Boolean {
        return validatedParameters.provisionTransfers.any {
            AccountReserveType.PROVIDER == it.sourceAccount.reserveType.getOrNull()
        }
    }

    private fun requireProviderResponsible(
        transferRequestModel: TransferRequestModel,
        accountsById: Map<AccountId, AccountModel>,
    ): Boolean {
        return transferRequestModel.parameters.provisionTransfers.any {
            AccountReserveType.PROVIDER == accountsById[it.sourceAccountId]?.reserveType.getOrNull()
        }
    }

    private fun validateUserAction(
        validatedProvisionTransferParameters: ValidatedProvisionTransferParameters,
        loanParameters: ValidatedLoanParameters?,
        responsible: TransferResponsible,
        currentUser: YaUserDetails,
        locale: Locale,
    ): Result<Unit> {
        val errors = ErrorCollection.builder()
        val providerResponsible = responsible.providerResponsible.associateBy { it.responsibleId }
        val providerIds = validatedProvisionTransferParameters.provisionTransfers
            .map { it.sourceAccount.providerId }
            .toSet()
        if (loanParameters?.provideOverCommitReserve.orFalse()) {
            val userId = currentUser.user.orElse(null)?.id
            val userProviders = currentUser.providers.map { it.id }
            if ((userId !in providerResponsible || !providerResponsible[userId]!!.providerIds.containsAll(providerIds))
                && !userProviders.containsAll(providerIds)) {
                errors.addError("parameters.provideOverCommitReserve",
                    TypedError.invalid(messages.getMessage(
                        "errors.transfers.only.provider.responsible.can.provide.over.commit.reserve",
                        null, locale)))
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(Unit)
    }

    fun validatePutProvisionTransferMono(
        txSession: YdbTxSession,
        frontPutTransferRequestDto: FrontPutTransferRequestDto,
        transferRequestModel: TransferRequestModel,
        yaUserDetails: YaUserDetails,
        locale: Locale,
        publicApi: Boolean,
        delayValidation: Boolean
    ) = mono {
        validatePutProvisionTransfer(txSession, frontPutTransferRequestDto, transferRequestModel,
            yaUserDetails, locale, publicApi, delayValidation)
    }

    suspend fun validatePutProvisionTransfer(
        txSession: YdbTxSession,
        putRequest: FrontPutTransferRequestDto,
        transferRequestModel: TransferRequestModel,
        currentUser: YaUserDetails,
        locale: Locale,
        publicApi: Boolean,
        delayValidation: Boolean
    ): Result<ValidatedPutTransferRequest> = binding {
        val errors = ErrorCollection.builder()
        val preValidatedParametersBuilder = PreValidatedTransferRequestParameters.builder()
        validationService.validateCreateFields(putRequest::getDescription, putRequest::getParameters,
            preValidatedParametersBuilder, errors, locale)
        val preValidatedLoanParameters = transferLoansService.preValidateLoanFields(putRequest.loanParameters
            .orElse(null), errors, currentUser, locale)
        if (putRequest.parameters.isPresent) {
            val preValidatedProvisionTransfersBuilders = validationService.preValidateProvisionTransferParameters(
                putRequest.parameters.get(), preValidatedLoanParameters, errors, locale, publicApi)
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build())
            }
            val preValidateParameters = preValidatedParametersBuilder.build()
            val preValidatedProvisionTransfers = preValidatedProvisionTransfersBuilders.map { it.build() }
            val validatedParameters = validateProvisionTransfersParameters(txSession, preValidatedProvisionTransfers,
                null, currentUser, locale, publicApi).bind()!!
            val updatedResponsible = calculateResponsible(txSession, validatedParameters, currentUser)
            val validatedLoanParameters = transferLoansService.validateLoanFields(txSession,
                preValidatedLoanParameters, validatedParameters, currentUser, locale).bind()
            validateUserAction(validatedParameters, validatedLoanParameters, updatedResponsible.responsible,
                currentUser, locale).bind()
            val preApplication = validateProvisionTransferApplication(txSession, validatedParameters,
                validatedLoanParameters, null, locale, delayValidation).bind()!!
            val confirmation = validationService.validateResponsiblePut(updatedResponsible,
                putRequest::getAddConfirmation, currentUser, locale).bind()!!
            val indices = indicesService.loadTransferRequestIndices(txSession, transferRequestModel).awaitSingle()
            val preparePutProvisionTransferRequest = preparePutProvisionTransferRequest(validatedParameters,
                preValidateParameters, preApplication, updatedResponsible, confirmation, indices,
                transferRequestModel, validatedLoanParameters)
            return Result.success(preparePutProvisionTransferRequest)
        } else {
            return Result.failure(errors.build())
        }
    }

    fun validateVoteProvisionTransferMono(
        txSession: YdbTxSession,
        voteType: VoteType,
        currentTransferRequest: TransferRequestModel,
        responsibleAndNotified: ResponsibleAndNotified,
        currentUser: YaUserDetails,
        locale: Locale
    ) = mono {
        validateVoteProvisionTransfer(txSession, voteType, currentTransferRequest,
            responsibleAndNotified, currentUser, locale)
    }

    suspend fun validateVoteProvisionTransfer(
        txSession: YdbTxSession,
        voteType: VoteType,
        currentTransferRequest: TransferRequestModel,
        responsibleAndNotified: ResponsibleAndNotified,
        currentUser: YaUserDetails,
        locale: Locale
    ): Result<ValidatedVoteTransferRequest> = binding {
        val context = loadProvisionTransferParametersContext(txSession,
            currentTransferRequest.parameters.provisionTransfers)
        val accountById = context.accounts.associateBy { it.id }
        if (requireProviderResponsible(currentTransferRequest, accountById)) {
            permissionService.checkUserCanVoteForReserveProvisionRequest(currentTransferRequest,
                responsibleAndNotified, currentUser, locale).bind()
        } else {
            permissionService.checkUserCanVoteForRequest(currentTransferRequest, responsibleAndNotified, currentUser,
                locale).bind()
        }
        val indices = indicesService.loadTransferRequestIndices(txSession, currentTransferRequest).awaitSingle()
        val voteTransferRequest = prepareVoteProvisionTransferRequest(voteType, currentTransferRequest, context,
            indices, responsibleAndNotified, currentUser)
        return Result.success(voteTransferRequest)
    }

    private fun prepareVoteProvisionTransferRequest(
        voteType: VoteType,
        transferRequest: TransferRequestModel,
        context: ProvisionTransferParametersContext,
        indices: TransferRequestIndices,
        responsibleAndNotified: ResponsibleAndNotified,
        currentUser: YaUserDetails
    ): ValidatedVoteTransferRequest {
        val targetStatus: TransferRequestStatus
        val now = Instant.now()
        val builder = ValidatedVoteTransferRequest.builder()
        with(builder) {
            folders(context.folders)
            quotas(context.quotas)
            addAccounts(context.accounts)
            addAccountsQuotas(context.accountQuotas)
            services(context.services)
            resources(context.resources)
            unitsEnsembles(context.unitsEnsembles)
            providers(context.providers)
            now(now)
        }
        val historyBuilder = TransferRequestHistoryModel.builder()
        val transferRequestBuilder = TransferRequestModel.builder(transferRequest)
        val updatedVotes = updateVotes(transferRequest.votes, voteType, currentUser, now,
            responsibleAndNotified.responsible, context.folders)
        if (VoteType.CONFIRM == voteType) {
            val apply = isApplyProvisionTransfer(transferRequest, responsibleAndNotified.responsible, currentUser)
            val errorsEn = ErrorCollection.builder()
            val errorsRu = ErrorCollection.builder()
            val canBeApplied = canProvisionTransferBeApplied(transferRequest, context, mapOf(), errorsRu, errorsEn)
            builder.apply(apply && canBeApplied)
            targetStatus = if (apply) {
                if (canBeApplied) {
                    TransferRequestStatus.EXECUTING
                } else {
                    TransferRequestStatus.FAILED
                }
            } else {
                transferRequest.status
            }
            val applicationDetails = updateTransferApplicationDetails(canBeApplied, errorsEn.build(), errorsRu.build())
            with(transferRequestBuilder) {
                appliedAt(now)
                applicationDetails(applicationDetails)
            }
            with(historyBuilder) {
                oldApplicationDetails(null)
                newApplicationDetails(applicationDetails)
            }
        } else {
            builder.apply(false)
            targetStatus = when (voteType) {
                VoteType.REJECT -> TransferRequestStatus.REJECTED
                VoteType.ABSTAIN -> if (voteService.pendingTransferRequestUnableToConfirm(
                        responsibleAndNotified.responsible, transferRequest.votes, setOf(currentUser.user.get().id))) {
                    TransferRequestStatus.REJECTED
                } else {
                    transferRequest.status
                }
                else -> transferRequest.status
            }
        }
        val newTransferRequest = with(transferRequestBuilder) {
            version(transferRequest.version + 1)
            nextHistoryOrder(transferRequest.nextHistoryOrder + 1)
            votes(updatedVotes)
            responsible(responsibleAndNotified.responsible)
            transferNotified(TransferNotified.from(responsibleAndNotified))
            status(targetStatus)
        }.build()
        builder.transferRequest(newTransferRequest)
        with(historyBuilder) {
            id(UUID.randomUUID().toString())
            tenantId(Tenants.DEFAULT_TENANT_ID)
            transferRequestId(transferRequest.id)
            val type = when (targetStatus) {
                TransferRequestStatus.EXECUTING -> TransferRequestEventType.START_EXECUTING
                TransferRequestStatus.FAILED -> TransferRequestEventType.FAILED
                TransferRequestStatus.REJECTED -> TransferRequestEventType.REJECTED
                else -> TransferRequestEventType.VOTED
            }
            type(type)
            timestamp(now)
            authorId(currentUser.user.get().id)
            val oldFields = TransferRequestHistoryFields.builder()
                .version(transferRequest.version)
            val newFields = TransferRequestHistoryFields.builder()
                .version(transferRequest.version + 1)
            if (!transferRequest.status.equals(targetStatus)) {
                oldFields.status(transferRequest.status)
                newFields.status(targetStatus)
            }
            oldFields(oldFields.build())
            newFields(newFields.build())
            oldVotes(transferRequest.votes)
            newVotes(updatedVotes)
            order(transferRequest.nextHistoryOrder)
            if (!responsibleAndNotified.responsible.equals(transferRequest.responsible)) {
                oldResponsible(transferRequest.responsible)
                newResponsible(responsibleAndNotified.responsible)
            }
        }
        builder.history(historyBuilder.build())
        val newTransferRequestIndices = TransferRequestIndicesService
            .calculateTransferRequestIndices(newTransferRequest)
        val difference = TransferRequestIndicesService.difference(indices, newTransferRequestIndices)
        builder.indicesDifference(difference)
        if (transferRequest.status.equals(targetStatus)) {
            val notifiedUsers = TransferRequestResponsibleAndNotifyService.excludeAlreadyNotifiedUsers(transferRequest,
                responsibleAndNotified.notifiedUsers, currentUser)
            builder.notifiedUsers(notifiedUsers)
        }
        return builder.build()
    }

    private fun updateTransferApplicationDetails(
        canBeApplied: Boolean,
        errorsEn: ErrorCollection,
        errorsRu: ErrorCollection
    ): TransferApplicationDetails {
        val applicationDetailsBuilder = TransferApplicationDetails.builder()
        if (!canBeApplied) {
            fillApplicationErrors(applicationDetailsBuilder, errorsEn, errorsRu)
        }
        return applicationDetailsBuilder.build()
    }

    private fun fillApplicationErrors(
        applicationDetailsBuilder: TransferApplicationDetails.Builder,
        errorsEn: ErrorCollection,
        errorsRu: ErrorCollection
    ) {
        val (applicationErrorsEn, applicationErrorsRu) = toLocalizedErrors(errorsEn, errorsRu)
        applicationDetailsBuilder.errorsEn(applicationErrorsEn)
        applicationDetailsBuilder.errorsRu(applicationErrorsRu)
    }

    private fun toLocalizedErrors(
        errorsEn: ErrorCollection,
        errorsRu: ErrorCollection
    ): LocalizedTransferApplicationErrors {
        val errorsEnBuilder = TransferApplicationErrors.builder()
        val errorsRuBuilder = TransferApplicationErrors.builder()
        errorsEn.errors.forEach { errorsEnBuilder.addError(it.error) }
        errorsRu.errors.forEach { errorsRuBuilder.addError(it.error) }
        errorsEn.fieldErrors.forEach { (field, errors) ->
            errors.forEach { errorsEnBuilder.addFieldError(field, it.error) }
        }
        errorsRu.fieldErrors.forEach { (field, errors) ->
            errors.forEach { errorsRuBuilder.addFieldError(field, it.error) }
        }
        errorsEnBuilder.addDetails(errorsEn.details)
        errorsRuBuilder.addDetails(errorsRu.details)
        val applicationErrorsEn = errorsEnBuilder.build()
        val applicationErrorsRu = errorsRuBuilder.build()
        return LocalizedTransferApplicationErrors(applicationErrorsEn, applicationErrorsRu)
    }

    private fun isApplyProvisionTransfer(transferRequestModel: TransferRequestModel,
                                         responsible: TransferResponsible,
                                         currentUser: YaUserDetails): Boolean {
        val currentUserId = currentUser.user.get().id
        if (responsible.providerResponsible.any { it.responsibleId == currentUserId }) {
            return true
        }
        val confirmedFolderIds = hashSetOf<String>()
        transferRequestModel.votes.votes.forEach {
            if (it.userId.equals(currentUserId)) {
                return@forEach
            }
            if (VoteType.CONFIRM == it.type) {
                confirmedFolderIds += it.folderIds
            }
        }
        responsible.responsible.forEach {
            val isResponsible = it.responsible.any { sr -> currentUserId in sr.responsibleIds }
            if (isResponsible) {
                confirmedFolderIds += it.folderIds
            }
        }
        responsible.reserveResponsibleModel.ifPresent {
            if (currentUserId in it.responsibleIds) {
                confirmedFolderIds += it.folderId
            }
        }
        return transferRequestModel.parameters.provisionTransfers.all {
            it.sourceFolderId in confirmedFolderIds && it.destinationFolderId in confirmedFolderIds
        }
    }

    private fun canProvisionTransferBeApplied(
        transferRequestModel: TransferRequestModel,
        context: ProvisionTransferParametersContext,
        receivedProvisionByAccountResource: Map<AccountId, Map<ResourceId, ValidatedReceivedProvision>>,
        errorsRu: ErrorCollection.Builder,
        errorsEn: ErrorCollection.Builder
    ): Boolean {
        val quotasByAccountResource = context.accountQuotas.groupBy { it.accountId }
            .mapValues { (_, quotas) -> quotas.associateBy { it.resourceId } }
        val resourceById = context.resources.associateBy { it.id }
        val accountById = context.accounts.associateBy { it.id }
        val unitEnsembleById = context.unitsEnsembles.associateBy { it.id }
        val providerById = context.providers.associateBy { it.id }
        transferRequestModel.parameters.provisionTransfers.forEach { provisionTransfer ->
            val checkAvailableQuota = !(transferRequestModel.loanMeta.getOrNull()?.provideOverCommitReserve ?: false)
            sequenceOf(
                provisionTransfer.sourceAccountId to provisionTransfer.sourceAccountTransfers,
                provisionTransfer.destinationAccountId to provisionTransfer.destinationAccountTransfers
            ).forEach { (accountId, transfers) ->
                val account = accountById[accountId]!!
                val quotaByResource = quotasByAccountResource[accountId]
                val provisionByResource = receivedProvisionByAccountResource[accountId] ?: mapOf()
                val provider = providerById[account.providerId]!!
                transfers.forEach { transfer ->
                    val quota = quotaByResource?.get(transfer.resourceId)
                    val resource = resourceById[transfer.resourceId]!!
                    val unitEnsemble = unitEnsembleById[resource.unitsEnsembleId]
                        ?: throw IllegalStateException("Unit ensemble not found ${resource.unitsEnsembleId}")
                    val receivedProvision = provisionByResource[transfer.resourceId]
                    validationService.validateAccountQuotaTransfer(transfer, quota, receivedProvision, resource,
                        account, provider, unitEnsemble, errorsEn, errorsRu, checkAvailableQuota
                    )
                }
            }
        }
        return !errorsEn.hasAnyErrors() && !errorsRu.hasAnyErrors()
    }

    private suspend fun loadProvisionTransferParametersContext(
        txSession: YdbTxSession,
        provisionTransfers: Set<ProvisionTransfer>
    ): ProvisionTransferParametersContext {
        val accountIds = mutableSetOf<String>()
        val folderIds = mutableSetOf<String>()
        val serviceIds = mutableSetOf<Long>()
        val resourceIds = mutableSetOf<String>()
        provisionTransfers.forEach { pt ->
            accountIds += pt.sourceAccountId
            accountIds += pt.destinationAccountId
            folderIds += pt.sourceFolderId
            folderIds += pt.destinationFolderId
            serviceIds += pt.sourceServiceId
            serviceIds += pt.destinationServiceId
            sequenceOf(pt.sourceAccountTransfers, pt.destinationAccountTransfers)
                .flatMap { it.asSequence() }
                .forEach {
                    resourceIds += it.resourceId
                }
        }
        val folders = storeService.loadFolders(txSession, folderIds).awaitSingle()
        val services = storeService.loadServices(txSession, serviceIds).awaitSingle()
        val accounts = storeService.loadAccounts(txSession, accountIds).awaitSingle()
        val resources = storeService.loadResources(txSession, resourceIds).awaitSingle()
        val providerIds = mutableSetOf<String>()
        val unitEnsembleIds = mutableSetOf<String>()
        resources.forEach {
            providerIds += it.providerId
            unitEnsembleIds += it.unitsEnsembleId
        }
        val providers = storeService.loadProviders(txSession, providerIds).awaitSingle()
        val unitEnsembles = storeService.loadUnitsEnsembles(txSession, unitEnsembleIds).awaitSingle()
        val quotas = storeService.loadQuotas(txSession, folderIds).awaitSingle()
        val accountQuotas = storeService.loadAccountsQuotas(txSession, accountIds).awaitSingle()
        return ProvisionTransferParametersContext(accounts, folders, services, resources, providers, unitEnsembles,
            quotas, accountQuotas)
    }

    fun validateProvisionTransfersParametersMono(
        txSession: YdbTxSession,
        preValidatedProvisionTransfers: List<PreValidatedProvisionTransfer>,
        currentUser: YaUserDetails,
        locale: Locale,
        publicApi: Boolean
    ): Mono<Result<ValidatedProvisionTransferParameters>> = mono {
        validateProvisionTransfersParameters(txSession, preValidatedProvisionTransfers,
            null, currentUser, locale, publicApi)
    }

    suspend fun validateProvisionTransfersParameters(
        txSession: YdbTxSession,
        preValidatedProvisionTransfers: List<PreValidatedProvisionTransfer>,
        preCreateProviderData: ProvisionTransferProviderData?,
        currentUser: YaUserDetails,
        locale: Locale,
        publicApi: Boolean
    ): Result<ValidatedProvisionTransferParameters> {
        val accountIds = hashSetOf<String>()
        val folderIds = hashSetOf<String>()
        val serviceIds = hashSetOf<Long>()
        val resourceIds = hashSetOf<String>()

        preValidatedProvisionTransfers.forEach { transfer ->
            accountIds += transfer.sourceAccountId
            accountIds += transfer.destinationAccountId
            folderIds += transfer.sourceFolderId
            folderIds += transfer.destinationFolderId
            if (transfer.sourceServiceId != null) {
                serviceIds += transfer.sourceServiceId
            }
            if (transfer.destinationServiceId != null) {
                serviceIds += transfer.destinationServiceId
            }
            sequenceOf(
                transfer.sourceAccountTransfers,
                transfer.destinationAccountTransfers,
            ).forEach { transfers ->
                transfers.forEach {
                    resourceIds += it.resourceId
                }
            }
        }
        val accounts = storeService.loadAccounts(txSession, accountIds).awaitSingle()
        val accountFolderIds = accounts.map { it.folderId }.toSet()
        val folders = storeService.loadFolders(txSession, folderIds union accountFolderIds).awaitSingle()
        val folderServiceIds = folders.map { it.serviceId }.toSet()
        val services = storeService.loadServices(txSession, serviceIds union folderServiceIds).awaitSingle()
        val resources = storeService.loadResources(txSession, resourceIds).awaitSingle()
        val segmentationIds = resources.flatMap { it.segments.map { it.segmentationId } }.toList()
        val segmentations = storeService.loadResourceSegmentations(txSession, segmentationIds).awaitSingle()
        val segmentIds = resources.flatMap { it.segments.map { it.segmentId } }.toList()
        val segments = storeService.loadResourceSegments(txSession, segmentIds).awaitSingle()
        val resourceTypeIds = resources.map { it.resourceTypeId }.toList()
        val resourceTypes = storeService.loadResourceTypes(txSession, resourceTypeIds).awaitSingle()
        val unitEnsembleIds = resources.map { it.unitsEnsembleId }.toSet()
        val providerIds = resources.map { it.providerId }.toSet() union accounts.map { it.providerId }.toSet()
        val unitsEnsembles = storeService.loadUnitsEnsembles(txSession, unitEnsembleIds).awaitSingle()
        val providers = storeService.loadProviders(txSession, providerIds).awaitSingle()
        val users = loadUsers(txSession, preCreateProviderData)

        return validateProvisionTransfers(preValidatedProvisionTransfers, currentUser, locale,
            accounts, folders, services, resources, resourceTypes, segments, segmentations,
            unitsEnsembles, providers, users, publicApi)
    }

    private suspend fun loadUsers(
        txSession: YdbTxSession,
        preCreateProviderData: ProvisionTransferProviderData?,
    ): List<UserModel> {
        if (preCreateProviderData == null) {
            return emptyList()
        }
        val userIds = preCreateProviderData.receivedAccounts.flatMap { getUsersFromReceivedAccount(it) }.toSet()
        if (userIds.isEmpty()) {
            return emptyList()
        }
        val uidSet = userIds.mapNotNull { it.passportUid.orElse(null) }.toSet()
        val loginSet = userIds.mapNotNull { it.staffLogin.orElse(null) }.toSet()
        if (uidSet.isEmpty() && loginSet.isEmpty()) {
            return emptyList()
        }
        return usersDao.getByExternalIds(txSession,
                uidSet.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) },
                loginSet.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) },
                emptyList()).awaitSingle()
    }

    private fun prepareCreateProvisionTransferRequest(
        validatedParameters: ValidatedProvisionTransferParameters,
        preValidatedTransferRequestParameters: PreValidatedTransferRequestParameters,
        preApplicationData: ProvisionTransferPreApplicationData,
        responsibleAndNotified: ResponsibleAndNotified,
        confirmation: CreateTransferRequestConfirmationData,
        loanParameters: ValidatedLoanParameters?
    ): ValidatedCreateTransferRequest {
        val responsible = responsibleAndNotified.responsible
        val now = Instant.now()
        val validatedBuilder = ValidatedCreateTransferRequest.builder()
        with(validatedBuilder) {
            quotas(preApplicationData.folderQuotas)
            folders(validatedParameters.folders)
            addAccounts(validatedParameters.accounts)
            addAccountsQuotas(preApplicationData.accountQuotas)
            services(validatedParameters.services)
            resources(validatedParameters.resources)
            unitsEnsembles(validatedParameters.unitsEnsembles)
            providers(validatedParameters.providers)
            addReceivedProvisionByAccountResource(preApplicationData.receivedProvisionsByAccountResource)
            now(now)
            apply(confirmation.isApply)
        }
        val transferRequestBuilder = TransferRequestModel.builder()
        val parametersBuilder = TransferParameters.builder()

        validatedParameters.provisionTransfers.forEach {
            parametersBuilder.addProvisionTransfer(toProvisionTransfer(it))
        }

        val votesBuilder = TransferVotes.builder()
        if (confirmation.isConfirm) {
            if (confirmation.isProviderResponsibleAutoConfirm) {
                val vote = prepareProviderConfirmVote(responsible, now, confirmation.user,
                    validatedParameters.folders)
                votesBuilder.addVote(vote)
            } else {
                val vote = TransferRequestQuotaService.prepareConfirmationVote(responsible, now, confirmation.user)
                votesBuilder.addVote(vote)
            }
        }
        val transferRequestId = UUID.randomUUID().toString()
        val transferParameters = parametersBuilder.build()
        val transferVotes = votesBuilder.build()
        val summary = generateSummary(validatedParameters)
        val transferRequestStatus = if (confirmation.isApply) {
            TransferRequestStatus.EXECUTING
        } else {
            TransferRequestStatus.PENDING
        }
        val loanMeta = prepareLoanMeta(loanParameters)
        val historyBuilder = TransferRequestHistoryModel.builder()
        val historyFieldsBuilder = TransferRequestHistoryFields.builder()
        val historyFields = with(historyFieldsBuilder) {
            version(0L)
            summary(summary)
            description(preValidatedTransferRequestParameters.description.orElse(null))
            type(TransferRequestType.PROVISION_TRANSFER)
            status(transferRequestStatus)
        }.build()
        val historyModel = with(historyBuilder) {
            id(UUID.randomUUID().toString())
            tenantId(Tenants.DEFAULT_TENANT_ID)
            transferRequestId(transferRequestId)
            type(TransferRequestEventType.CREATED)
            timestamp(now)
            authorId(confirmation.user.id)
            oldFields(null)
            oldParameters(null)
            oldResponsible(null)
            oldVotes(null)
            oldApplicationDetails(null)
            newFields(historyFields)
            newParameters(transferParameters)
            newVotes(transferVotes)
            newResponsible(responsible)
            newApplicationDetails(null)
            oldLoanMeta(null)
            newLoanMeta(loanMeta)
            order(0L)
        }.build()
        val transferRequestModel = with(transferRequestBuilder) {
            id(transferRequestId)
            tenantId(Tenants.DEFAULT_TENANT_ID)
            version(0L)
            summary(summary)
            description(preValidatedTransferRequestParameters.description.orElse(null))
            type(TransferRequestType.PROVISION_TRANSFER)
            subtype(getSubtype(validatedParameters.provisionTransfers, loanParameters))
            status(transferRequestStatus)
            createdBy(confirmation.user.id)
            updatedBy(null)
            createdAt(now)
            updatedAt(null)
            appliedAt(if (confirmation.isApply) now else null)
            parameters(transferParameters)
            responsible(responsible)
            votes(transferVotes)
            applicationDetails(null)
            nextHistoryOrder(1L)
            transferNotified(TransferNotified.from(responsibleAndNotified))
            loanMeta(loanMeta)
        }.build()
        with(validatedBuilder) {
            transferRequest(transferRequestModel)
            history(historyModel)
        }
        validatedParameters.folders.forEach {
            validatedBuilder.addFolderIndex(TransferRequestByFolderModel(Tenants.DEFAULT_TENANT_ID, it.id,
                transferRequestStatus, now, transferRequestId))
        }
        validatedParameters.services.forEach {
            validatedBuilder.addServiceIndex(TransferRequestByServiceModel(Tenants.DEFAULT_TENANT_ID, it.id,
                transferRequestStatus, now, transferRequestId))
        }
        val responsibleIndices = TransferRequestIndicesService
            .calculateResponsibleIndices(transferRequestModel, responsible)
        validatedBuilder.addResponsibleIndices(responsibleIndices)
        TransferRequestQuotaService.prepareNotifiedUsersCreate(confirmation, validatedBuilder, responsibleAndNotified)
        return validatedBuilder.build()
    }

    private fun prepareLoanMeta(loanParameters: ValidatedLoanParameters?): LoanMeta? {
        if (loanParameters == null) {
            return null
        }
        return if (loanParameters.borrowParameters != null) {
            LoanMeta(loanParameters.borrowParameters.dueDate, null, null, LoanOperationType.BORROW,
                loanParameters.provideOverCommitReserve)
        } else if (loanParameters.payOffParameters != null) {
            LoanMeta(null, null, loanParameters.payOffParameters.loan.id, LoanOperationType.PAY_OFF,
                loanParameters.provideOverCommitReserve)
        } else {
            throw IllegalArgumentException("Invalid loan parameters $loanParameters")
        }
    }

    private fun isSufficientUpdate(
        oldLoanMeta: LoanMeta?,
        newLoanMeta: LoanMeta?
    ): Boolean {
        if (oldLoanMeta == newLoanMeta) {
            return false
        }
        return oldLoanMeta?.borrowDueDate != newLoanMeta?.borrowDueDate
            || oldLoanMeta?.borrowLoanIds != newLoanMeta?.borrowLoanIds
            || oldLoanMeta?.payOffLoanId != newLoanMeta?.payOffLoanId
            || oldLoanMeta?.operationType != newLoanMeta?.operationType
    }

    private fun generateSummary(validatedParameters: ValidatedProvisionTransferParameters): String {
        if (validatedParameters.provisionTransfers.size == 1) {
            val provisionTransfer = validatedParameters.provisionTransfers.first()
            return "Перенос спущенной квоты между ${provisionTransfer.sourceService.slug}" +
                ":${provisionTransfer.sourceFolder.displayName}" +
                ":${DisplayUtil.getAccountDisplayString(provisionTransfer.sourceAccount)} " +
                "и ${provisionTransfer.destinationService.slug}:${provisionTransfer.destinationFolder.displayName}" +
                ":${DisplayUtil.getAccountDisplayString(provisionTransfer.destinationAccount)}"
        } else {
            return "Перенос спущенной квоты"
        }
    }

    private fun preparePutProvisionTransferRequest(
        validatedParameters: ValidatedProvisionTransferParameters,
        preValidatedTransferRequestParameters: PreValidatedTransferRequestParameters,
        preApplicationData: ProvisionTransferPreApplicationData,
        responsibleAndNotified: ResponsibleAndNotified,
        confirmation: PutTransferRequestConfirmationData,
        indices: TransferRequestIndices,
        currentTransferRequest: TransferRequestModel,
        loanParameters: ValidatedLoanParameters?
    ): ValidatedPutTransferRequest {
        val responsible = responsibleAndNotified.responsible
        val now = Instant.now()
        val validatedBuilder = ValidatedPutTransferRequest.builder()
        with(validatedBuilder) {
            quotas(preApplicationData.folderQuotas)
            addAccountsQuotas(preApplicationData.accountQuotas)
            folders(validatedParameters.folders)
            addAccounts(validatedParameters.accounts)
            services(validatedParameters.services)
            resources(validatedParameters.resources)
            unitsEnsembles(validatedParameters.unitsEnsembles)
            providers(validatedParameters.providers)
            now(now)
            oldTransferRequest(currentTransferRequest)
        }
        val transferRequestBuilder = TransferRequestModel.builder(currentTransferRequest)
        val parametersBuilder = TransferParameters.builder()

        validatedParameters.provisionTransfers.forEach {
            parametersBuilder.addProvisionTransfer(toProvisionTransfer(it))
        }

        val updatedTransferParameters = parametersBuilder.build()
        val oldLoanMeta = currentTransferRequest.loanMeta.getOrNull()
        val newLoanMeta = prepareLoanMeta(loanParameters)
        val parametersUpdated = !updatedTransferParameters.equals(currentTransferRequest.parameters)
        val responsibleUpdated = !responsible.equals(currentTransferRequest.responsible)
        val votesBuilder = TransferVotes.builder()
        if (confirmation.isConfirm) {
            val vote = prepareVote(responsible, now, confirmation.user, validatedParameters.folders)
            votesBuilder.addVote(vote)
        }
        if (!parametersUpdated && !isSufficientUpdate(oldLoanMeta, newLoanMeta)) {
            TransferRequestQuotaService.copyVotes(responsible, confirmation, currentTransferRequest, votesBuilder)
        }
        val updatedTransferVotes = votesBuilder.build()
        val votesUpdated = !updatedTransferVotes.equals(currentTransferRequest.votes)
        val apply = updatedProvisionTransferRequestCanBeApplied(updatedTransferVotes, updatedTransferParameters)
        val targetStatus = if (apply) {
            TransferRequestStatus.EXECUTING
        } else {
            TransferRequestStatus.PENDING
        }

        val historyBuilder = TransferRequestHistoryModel.builder()
        with(historyBuilder) {
            id(UUID.randomUUID().toString())
            tenantId(Tenants.DEFAULT_TENANT_ID)
            transferRequestId(currentTransferRequest.id)
            type(TransferRequestEventType.UPDATED)
            timestamp(now)
            authorId(confirmation.user.id)
        }
        val oldFieldsBuilder = TransferRequestHistoryFields.builder()
            .version(currentTransferRequest.version)
        val newFieldsBuilder = TransferRequestHistoryFields.builder()
            .version(currentTransferRequest.version + 1)
        if (!currentTransferRequest.description.equals(preValidatedTransferRequestParameters.description)) {
            oldFieldsBuilder.description(currentTransferRequest.description.orElse(null))
            newFieldsBuilder.description(preValidatedTransferRequestParameters.description.orElse(null))
        }
        if (!currentTransferRequest.status.equals(targetStatus)) {
            oldFieldsBuilder.status(currentTransferRequest.status)
            newFieldsBuilder.status(targetStatus)
        }
        oldFieldsBuilder.type(null)
        newFieldsBuilder.type(null)
        val historyModel = with(historyBuilder) {
            oldFields(oldFieldsBuilder.build())
            newFields(newFieldsBuilder.build())
            if (parametersUpdated) {
                oldParameters(currentTransferRequest.parameters)
                newParameters(updatedTransferParameters)
            } else {
                oldParameters(null)
                newParameters(null)
            }
            if (responsibleUpdated) {
                oldResponsible(currentTransferRequest.responsible)
                newResponsible(responsible)
            } else {
                oldResponsible(null)
                newResponsible(null)
            }
            if (votesUpdated) {
                oldVotes(currentTransferRequest.votes)
                newVotes(updatedTransferVotes)
            } else {
                oldVotes(null)
                newVotes(null)
            }
            oldApplicationDetails(null)
            newApplicationDetails(null)
            order(currentTransferRequest.nextHistoryOrder)
            oldLoanMeta(oldLoanMeta)
            newLoanMeta(newLoanMeta)
        }.build()
        with(transferRequestBuilder) {
            version(currentTransferRequest.version + 1)
            description(preValidatedTransferRequestParameters.description.orElse(null))
            status(targetStatus)
            subtype(getSubtype(validatedParameters.provisionTransfers, loanParameters))
            updatedBy(confirmation.user.id)
            updatedAt(now)
            appliedAt(if (apply) now else null)
            parameters(updatedTransferParameters)
            responsible(responsible)
            votes(updatedTransferVotes)
            applicationDetails(null)
            nextHistoryOrder(currentTransferRequest.nextHistoryOrder + 1)
            loanMeta(newLoanMeta)
        }
        if (!transferRequestBuilder.hasChanges(currentTransferRequest)) {
            return with(validatedBuilder) {
                apply(false)
                transferRequest(null)
                history(null)
            }.build()
        }
        validatedBuilder.apply(apply)
        transferRequestBuilder.transferNotified(TransferNotified.from(responsibleAndNotified))
        val newTransferRequest = transferRequestBuilder.build()
        validatedBuilder.transferRequest(newTransferRequest)
        validatedBuilder.history(historyModel)
        val newIndices = TransferRequestIndicesService.calculateTransferRequestIndices(newTransferRequest)
        val difference = TransferRequestIndicesService.difference(indices, newIndices)
        validatedBuilder.indicesDifference(difference)
        TransferRequestQuotaService.prepareNotifiedUsersPut(confirmation, validatedBuilder, responsibleAndNotified,
            parametersUpdated, apply, currentTransferRequest)
        return validatedBuilder.build()
    }

    private fun updateVotes(
        currentVotes: TransferVotes,
        voteType: VoteType,
        currentUser: YaUserDetails,
        now: Instant,
        responsible: TransferResponsible,
        folders: List<FolderModel>
    ): TransferVotes {
        val updatedVotes = TransferVotes.builder()
        val user = currentUser.user.get()
        currentVotes.votes.forEach { vote: TransferVote ->
            if (vote.userId == user.id) {
                return@forEach
            }
            updatedVotes.addVote(vote)
        }
        updatedVotes.addVote(prepareVote(responsible, now, user, folders, voteType))
        return updatedVotes.build()
    }

    private fun prepareVote(
        responsible: TransferResponsible,
        now: Instant,
        user: UserModel,
        folders: List<FolderModel>,
        voteType: VoteType = VoteType.CONFIRM
    ): TransferVote {
        val providerVote = responsible.providerResponsible.any { it.responsibleId == user.id }
        if (providerVote) {
            return prepareProviderConfirmVote(responsible, now, user, folders)
        }
        val voteBuilder = TransferVote.builder()
        voteBuilder.userId(user.id)
        voteBuilder.type(voteType)
        voteBuilder.timestamp(now)
        val voteFolderIds = HashSet<String>()
        val voteServiceIds = HashSet<Long>()
        responsible.responsible.forEach { foldersResponsible ->
            var responsibleForFolders = false
            for (serviceResponsible in foldersResponsible.responsible) {
                if (serviceResponsible.responsibleIds.contains(user.id)) {
                    voteServiceIds.add(serviceResponsible.serviceId)
                    responsibleForFolders = true
                }
            }
            if (responsibleForFolders) {
                voteFolderIds.addAll(foldersResponsible.folderIds)
            }
        }
        responsible.reserveResponsibleModel.ifPresent { reserveResponsible ->
            if (reserveResponsible.responsibleIds.contains(user.id)) {
                voteFolderIds.add(reserveResponsible.folderId)
                voteServiceIds.add(reserveResponsible.serviceId)
            }
        }
        voteBuilder.addFolderIds(voteFolderIds)
        voteBuilder.addServiceIds(voteServiceIds)
        return voteBuilder.build()
    }

    private fun updatedProvisionTransferRequestCanBeApplied(
        updatedTransferVotes: TransferVotes,
        updatedTransferParameters: TransferParameters
    ): Boolean {
        val confirmationsByFolderId: MutableMap<String, MutableSet<String>> = HashMap()
        updatedTransferVotes.votes.forEach { vote ->
            if (VoteType.CONFIRM == vote.type) {
                vote.folderIds.forEach { folderId ->
                    confirmationsByFolderId.computeIfAbsent(folderId) { HashSet() }.add(vote.userId)
                }
            }
        }
        return updatedTransferParameters.provisionTransfers.none { transfer ->
            confirmationsByFolderId.getOrDefault(transfer.sourceFolderId, emptySet()).isEmpty()
                || confirmationsByFolderId.getOrDefault(transfer.destinationFolderId, emptySet()).isEmpty()
        }
    }

    private fun prepareProviderConfirmVote(
        responsible: TransferResponsible,
        now: Instant,
        user: UserModel,
        folders: List<FolderModel>
    ): TransferVote {
        return with(TransferVote.builder()) {
            userId(user.id)
            type(VoteType.CONFIRM)
            timestamp(now)
            addFolderIds(folders.map { it.id }.toSet())
            addProviderIds(responsible.providerResponsible
                .find { it.responsibleId == user.id }
                ?.providerIds ?: setOf()
            )
        }.build()
    }

    private suspend fun validateProvisionTransfers(
        provisionTransfers: List<PreValidatedProvisionTransfer>,
        currentUser: YaUserDetails,
        locale: Locale,
        accounts: List<AccountModel>,
        folders: List<FolderModel>,
        services: List<ServiceMinimalModel>,
        resources: List<ResourceModel>,
        resourceTypes: List<ResourceTypeModel>,
        segments: List<ResourceSegmentModel>,
        segmentations: List<ResourceSegmentationModel>,
        unitEnsembles: List<UnitsEnsembleModel>,
        providers: List<ProviderModel>,
        users: List<UserModel>,
        publicApi: Boolean
    ): Result<ValidatedProvisionTransferParameters> {
        val accountsById = accounts.associateBy { it.id }
        val foldersById = folders.associateBy { it.id }
        val servicesById = services.associateBy { it.id }
        val resourcesById = resources.associateBy { it.id }
        val unitEnsembleById = unitEnsembles.associateBy { it.id }
        val providersById = providers.associateBy { it.id }
        val provisionTransferErrors = ErrorCollection.builder()
        val field = "parameters.provisionTransfers"
        val builders = mutableListOf<ValidatedProvisionTransfer.Builder>()
        val accountResources = mutableSetOf<Pair<AccountId, ResourceId>>()
        provisionTransfers.forEachIndexed { index, provisionTransfer ->
            val provisionTransferBuilder = ValidatedProvisionTransfer.builder()
            val sourceAccount = accountsById[provisionTransfer.sourceAccountId]
            val destinationAccount = accountsById[provisionTransfer.destinationAccountId]
            val fieldIndex = "$field.$index"
            validationService.validateAccountWithFolder(sourceAccount, provisionTransfer.sourceFolderId,
                provisionTransferBuilder::sourceAccount, provisionTransferErrors, "$fieldIndex.sourceAccountId",
                locale, publicApi)
            validationService.validateAccountWithFolder(destinationAccount, provisionTransfer.destinationFolderId,
                provisionTransferBuilder::destinationAccount, provisionTransferErrors,
                "$fieldIndex.destinationAccountId", locale, publicApi)
            val sourceFolder = foldersById[provisionTransfer.sourceFolderId]
            val destinationFolder = foldersById[provisionTransfer.destinationFolderId]
            validationService.validateFolderWithService(sourceFolder, provisionTransfer.sourceServiceId,
                provisionTransferBuilder::sourceFolder, provisionTransferErrors, "$fieldIndex.sourceFolderId",
                locale, publicApi)
            validationService.validateFolderWithService(destinationFolder, provisionTransfer.destinationServiceId,
                provisionTransferBuilder::destinationFolder, provisionTransferErrors,
                "$fieldIndex.destinationFolderId", locale, publicApi)
            val (sourceService, destinationService) = if (!publicApi) {
                Pair(servicesById[provisionTransfer.sourceServiceId],
                    servicesById[provisionTransfer.destinationServiceId])
            } else {
                Pair(servicesById[sourceFolder?.serviceId], servicesById[destinationFolder?.serviceId])
            }
            validationService.validateServiceTransfers(sourceService, provisionTransfer.sourceAccountTransfers,
                provisionTransferErrors, provisionTransferBuilder::sourceService,
                fieldIndex + if (publicApi) "" else ".sourceServiceId", locale)
            validationService.validateServiceTransfers(destinationService,
                provisionTransfer.destinationAccountTransfers, provisionTransferErrors,
                provisionTransferBuilder::destinationService,
                fieldIndex + if (publicApi) "" else ".destinationServiceId", locale)

            if (sourceAccount != null && destinationAccount != null) {
                validationService.validateProvisionTransferAccounts(sourceAccount, destinationAccount,
                    provisionTransferErrors, fieldIndex, locale)
                sequenceOf(sourceAccount, destinationAccount)
                    .map { it.providerId }
                    .distinct()
                    .forEach {
                        val provider = providersById[it]!!
                        validationService.validateMoveProvisionSupportForProvider(provider, provisionTransferErrors,
                            fieldIndex, locale)
                    }
                sequenceOf(
                    PreValidatedAccountTransfers(sourceAccount, provisionTransfer.sourceAccountTransfers,
                        "$fieldIndex.sourceAccountTransfers", provisionTransferBuilder::addSourceAccountTransfer,
                        false),
                    PreValidatedAccountTransfers(destinationAccount, provisionTransfer.destinationAccountTransfers,
                        "$fieldIndex.destinationAccountTransfers",
                        provisionTransferBuilder::addDestinationAccountTransfer,
                        true),
                ).forEach { (account, transfers, transferField, setter, isDestinationTransfers) ->
                    transfers.forEach {
                        val indexedTransferField = "$transferField.${it.index}"
                        val resource = resourcesById[it.resourceId]
                        val accountResource = account.id to it.resourceId
                        if (accountResource in accountResources) {
                            validationService.duplicateAccountResourceError(sourceAccount,
                                provisionTransferErrors, indexedTransferField, locale)
                        }
                        accountResources.add(accountResource)
                        validationService.validateAccountResource(sourceAccount, resource, provisionTransferErrors,
                            indexedTransferField, locale)
                        validationService.validateQuotaResourceTransfer(it, resourcesById, providersById,
                            unitEnsembleById, setter, provisionTransferErrors, transferField, locale,
                            publicApi, isDestinationTransfers)
                    }
                }
            }
            builders.add(provisionTransferBuilder)
        }
        validationService.validateExchangeTransfer(provisionTransfers, accountsById,
            provisionTransferErrors, field, locale)
        validationService.validateReserveTransfer(provisionTransfers, accountsById,
            provisionTransferErrors, field, locale)

        if (provisionTransferErrors.hasAnyErrors()) {
            return Result.failure(provisionTransferErrors.build())
        }
        val validatedProvisionTransfers = builders.map { it.build() }
        val validatedProvisionTransferParameters = ValidatedProvisionTransferParameters(
            validatedProvisionTransfers, accounts, folders, services, resources, resourceTypes,
            segments, segmentations, providers, unitEnsembles, users)
        return securityService.checkProvisionTransferReadPermissions(validatedProvisionTransferParameters,
            currentUser, locale).awaitSingle()
    }

    private data class PreValidatedAccountTransfers(
        val account: AccountModel,
        val transfers: List<PreValidatedQuotaResourceTransfer>,
        val field: String,
        val setter: (ValidatedQuotaResourceTransfer) -> Unit,
        val isDestinationTransfers: Boolean
    )

    private suspend fun validateProvisionTransferApplication(
        txSession: YdbTxSession,
        transferParameters: ValidatedProvisionTransferParameters,
        loanParameters: ValidatedLoanParameters?,
        preCreateProviderData: ProvisionTransferProviderData?,
        locale: Locale,
        delayValidation: Boolean
    ): Result<ProvisionTransferPreApplicationData> = binding {
        val accountIds = mutableSetOf<String>()
        val folderIds = mutableSetOf<String>()
        val overCommitPayOff = LoanType.PROVIDER_RESERVE_OVER_COMMIT == loanParameters?.payOffParameters?.loan?.type
        val provideOverCommitReserve = loanParameters?.provideOverCommitReserve ?: false
        transferParameters.provisionTransfers.forEach { provisionTransfer ->
            val sourceAccountId = provisionTransfer.sourceAccount.id
            val destinationAccountId = provisionTransfer.destinationAccount.id
            val sourceFolderId = provisionTransfer.sourceFolder.id
            val destinationFolderId = provisionTransfer.destinationFolder.id
            accountIds += sourceAccountId
            accountIds += destinationAccountId
            folderIds += sourceFolderId
            folderIds += destinationFolderId
            val accountTransfersSequence = sequenceOf(
                provisionTransfer.sourceAccountTransfers,
                provisionTransfer.destinationAccountTransfers
            ).flatMap { it.asSequence() }
            if (!provideOverCommitReserve && !overCommitPayOff) {
                validationService.validateQuotaTransfersSum<Any>(accountTransfersSequence.asStream(), locale).bind()
            }
        }
        val folderQuotas = storeService.loadQuotas(txSession, folderIds).awaitSingle()
        val accountsQuotas = storeService.loadAccountsQuotas(txSession, accountIds).awaitSingle()
        val quotasByAccountResource = accountsQuotas.groupBy { it.accountId }
            .mapValues { (_, quotas) -> quotas.associateBy { it.resourceId } }
        val receivedAccountById = preCreateProviderData?.receivedAccounts?.associateBy { it.accountId } ?: mapOf()
        val providerById = transferParameters.providers.associateBy { it.id }
        val segmentationById = transferParameters.segmentations.associateBy { it.id }
        val segmentById = transferParameters.segments.associateBy { it.id }
        val resourceTypeById = transferParameters.resourceTypes.associateBy { it.id }
        val unitEnsembleById = transferParameters.unitsEnsembles.associateBy { it.id }
        val usersByLogin = transferParameters.users.associateBy { it.passportLogin.orElseThrow() }
        val usersByUid = transferParameters.users.associateBy { it.passportUid.orElseThrow() }
        val receivedProvisionsByAccount = mutableMapOf<AccountId, Map<ResourceId, ValidatedReceivedProvision>>()
        val errors = ErrorCollection.builder()
        transferParameters.provisionTransfers.forEachIndexed { index, provisionTransfer ->
            sequenceOf(
                ValidatedAccountTransfers(provisionTransfer.sourceAccount, provisionTransfer.sourceAccountTransfers,
                    "parameters.provisionTransfers.$index.sourceAccountTransfers"),
                ValidatedAccountTransfers(provisionTransfer.destinationAccount,
                    provisionTransfer.destinationAccountTransfers,
                    "parameters.provisionTransfers.$index.destinationAccountTransfers"),
            ).forEach { (account, transfers, field) ->
                val quotaByAccount = quotasByAccountResource[account.id]
                val receivedAccount = receivedAccountById[account.id]
                val provider = providerById[account.providerId]!!
                val receivedProvisionByResourceId = if (receivedAccount != null) {
                    val validationErrors = ErrorCollection.builder()
                    val receivedProvisions = validateReceivedProvisions(
                        receivedAccount.provisions, receivedAccount.accountsSpaceKey.getOrNull(), usersByUid,
                        usersByLogin, mapOf(), transferParameters.resources, resourceTypeById, segmentationById,
                        segmentById, unitEnsembleById, validationErrors, messages, locale
                    )
                    if (validationErrors.hasAnyErrors()) {
                        mapOf()
                    } else {
                        receivedProvisions.associateBy { it.resource.id }
                    }
                } else {
                    mapOf()
                }
                receivedProvisionsByAccount[account.id] = receivedProvisionByResourceId
                transfers.forEach {
                    val quota = quotaByAccount?.get(it.resource.id)
                    val receivedProvision = receivedProvisionByResourceId[it.resource.id]
                    val skipValidation = !overCommitPayOff && provideOverCommitReserve
                    validationService.validateAccountQuotaTransfer(it, quota, receivedProvision, provider, errors,
                        field, locale, delayValidation || skipValidation)
                }
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(ProvisionTransferPreApplicationData(folderQuotas, accountsQuotas,
            receivedProvisionsByAccount))
    }

    fun validateCreateApplicable(
        transferRequest: ValidatedCreateTransferRequest
    ): ValidatedCreateTransferRequest {
        val errorsBuilderRu = ErrorCollection.builder()
        val errorsBuilderEn = ErrorCollection.builder()
        val context = ProvisionTransferParametersContext(
            transferRequest.accounts,
            transferRequest.folders,
            transferRequest.services,
            transferRequest.resources,
            transferRequest.providers,
            transferRequest.unitsEnsembles,
            transferRequest.quotas,
            transferRequest.accountsQuotas
        )
        val canBeApplied: Boolean = canProvisionTransferBeApplied(
            transferRequest.transferRequest,
            context,
            transferRequest.receivedProvisionByAccountResource,
            errorsBuilderRu,
            errorsBuilderEn
        )
        return if (!canBeApplied || errorsBuilderEn.hasAnyErrors() || errorsBuilderRu.hasAnyErrors()) {
            var applicationDetailsBuilder = TransferApplicationDetails.builder()
            if (transferRequest.transferRequest.applicationDetails.isPresent) {
                applicationDetailsBuilder = TransferApplicationDetails.builder(
                    transferRequest.transferRequest.applicationDetails.get()
                )
            }
            val applicationDetails = applicationDetailsBuilder
                .errorsEn(TransferApplicationErrors(errorsBuilderEn.build()))
                .errorsRu(TransferApplicationErrors(errorsBuilderRu.build()))
                .build()
            ValidatedCreateTransferRequest.builder(transferRequest)
                .transferRequest(
                    TransferRequestModel.builder(transferRequest.transferRequest)
                        .status(TransferRequestStatus.FAILED)
                        .applicationDetails(applicationDetails)
                        .build()
                )
                .history(
                    TransferRequestHistoryModel.builder(transferRequest.history)
                        .newApplicationDetails(applicationDetails)
                        .build()
                )
                .apply(false)
                .build()
        } else {
            transferRequest
        }
    }

    fun validatePutApplicable(transferRequest: ValidatedPutTransferRequest): ValidatedPutTransferRequest? {
        val errorsBuilderRu = ErrorCollection.builder()
        val errorsBuilderEn = ErrorCollection.builder()
        val context = ProvisionTransferParametersContext(
            transferRequest.accounts,
            transferRequest.folders,
            transferRequest.services,
            transferRequest.resources,
            transferRequest.providers,
            transferRequest.unitsEnsembles,
            transferRequest.quotas,
            transferRequest.accountsQuotas
        )
        val request = transferRequest.transferRequest.orElseThrow()
        val canBeApplied = canProvisionTransferBeApplied(
            request,
            context,
            mapOf(),
            errorsBuilderRu,
            errorsBuilderEn
        )
        return if (!canBeApplied || errorsBuilderEn.hasAnyErrors() || errorsBuilderRu.hasAnyErrors()) {
            var applicationDetailsBuilder = TransferApplicationDetails.builder()
            if (request.applicationDetails.isPresent) {
                applicationDetailsBuilder = TransferApplicationDetails.builder(
                    request.applicationDetails.get()
                )
            }
            val applicationDetails = applicationDetailsBuilder
                .errorsEn(TransferApplicationErrors(errorsBuilderEn.build()))
                .errorsRu(TransferApplicationErrors(errorsBuilderRu.build()))
                .build()
            val result = ValidatedPutTransferRequest.builder(transferRequest)
                .transferRequest(
                    TransferRequestModel.builder(request)
                        .status(TransferRequestStatus.FAILED)
                        .applicationDetails(applicationDetails)
                        .build()
                )
                .apply(false)
            if (transferRequest.history.isPresent) {
                result.history(
                    TransferRequestHistoryModel.builder(transferRequest.history.get())
                        .newApplicationDetails(applicationDetails)
                        .build()
                )
            }
            result.build()
        } else {
            transferRequest
        }
    }

    fun applyProvisionRequestMono(
        txSession: YdbTxSession,
        transferRequestModel: TransferRequestModel,
        historyModel: TransferRequestHistoryModel,
        applicationContext: MoveProvisionApplicationContext,
        currentUser: YaUserDetails
    ): Mono<ApplyResult> = mono {
        applyProvisionRequest(txSession, transferRequestModel, historyModel, applicationContext, currentUser)
    }

    suspend fun applyProvisionRequest(
        txSession: YdbTxSession,
        transferRequestModel: TransferRequestModel,
        historyModel: TransferRequestHistoryModel,
        applicationContext: MoveProvisionApplicationContext,
        currentUser: YaUserDetails
    ): ApplyResult = withMDC(MdcKey.COMMON_TRANSFER_REQUEST_ID to transferRequestModel.id) {
        val record = moveProvisionLogicService.createMoveProvisionOperationRecord(txSession, transferRequestModel,
            applicationContext, currentUser)
        val updatedTransfers = transferRequestModel.parameters.provisionTransfers.map { transfer ->
            val operationId = record.operationIdByProvisionTransfer[transfer]
            if (operationId != null) {
                transfer.copy(operationId = operationId)
            } else {
                transfer
            }
        }
        val transferStatuses = record.operationIds.associateWith { OperationStatus.EXECUTING }
        val updatedApplicationDetails = TransferApplicationDetails.builder()
            .addOperationIds(record.operationIds)
            .addFolderOpLogIds(record.opLogIdsByFolderId)
            .addOperationStatuses(transferStatuses)
            .build()
        val updatedParameters = TransferParameters.builder()
            .addProvisionTransfers(updatedTransfers)
            .build()
        val updatedRequest = transferRequestModel.copyBuilder()
            .applicationDetails(updatedApplicationDetails)
            .parameters(updatedParameters)
            .build()
        val updatedHistory = historyModel.copyBuilder()
            .oldParameters(if (historyModel.type != TransferRequestEventType.CREATED)
                transferRequestModel.parameters else null)
            .newParameters(updatedParameters)
            .newApplicationDetails(updatedApplicationDetails)
            .build()
        return@withMDC ApplyResult(updatedRequest, updatedHistory)
    }

    fun startOperationIfExistsMono(
        transferRequestModel: TransferRequestModel
    ): Mono<Void> = mono {
        startOperationIfExists(transferRequestModel)
    }.then()

    suspend fun startOperationIfExists(
        transferRequestModel: TransferRequestModel
    ) {
        if (transferRequestModel.status == TransferRequestStatus.EXECUTING) {
            if (transferRequestModel.applicationDetails.isPresent) {
                val applicationDetails = transferRequestModel.applicationDetails.get()
                applicationDetails.operationIds.forEach { operationId ->
                    moveProvisionLogicService.runOperation(operationId)
                }
            }
        }
    }

    suspend fun finishRequestOperation(
        txSession: YdbTxSession,
        transferRequest: TransferRequestModel,
        operationId: String,
        opLogIdsByFolderId: Map<String, Set<String>>,
        operationStatus: RetryResult,
        errorsEn: ErrorCollection,
        errorsRu: ErrorCollection,
        borrowedLoanId: LoanId?
    ): OperationFinishResult {
        val applicationDetailsBuilder = if (transferRequest.applicationDetails.isPresent) {
            TransferApplicationDetails.builder(transferRequest.applicationDetails.get())
        } else {
            TransferApplicationDetails.builder()
        }
        applicationDetailsBuilder.addFolderOpLogIds(opLogIdsByFolderId)
        val opStatus = when (operationStatus) {
            RetryResult.SUCCESS -> OperationStatus.COMPLETED
            RetryResult.FATAL_FAILURE, RetryResult.CONFLICT -> OperationStatus.FAILED
            else -> OperationStatus.UNKNOWN
        }
        applicationDetailsBuilder.addOperationId(operationId)
        applicationDetailsBuilder.addOperationStatus(operationId, opStatus)
        if (errorsEn.hasAnyErrors() || errorsRu.hasAnyErrors()) {
            applicationDetailsBuilder.addOperationErrors(operationId, toLocalizedErrors(errorsEn, errorsRu))
        }
        val applicationDetails = applicationDetailsBuilder.build()
        val hasUnfinishedOperations = applicationDetails.operationStatusById.containsValue(OperationStatus.EXECUTING)
        val requestStatus = if (!hasUnfinishedOperations) {
            val operationStatuses = applicationDetails.operationStatusById.values
            if (operationStatuses.all { it != OperationStatus.COMPLETED }) {
                TransferRequestStatus.FAILED
            } else if (operationStatuses.any { it != OperationStatus.COMPLETED }) {
                TransferRequestStatus.PARTLY_APPLIED
            } else {
                TransferRequestStatus.APPLIED
            }
        } else {
            null
        }
        val eventType = when (requestStatus) {
            TransferRequestStatus.APPLIED -> TransferRequestEventType.APPLIED
            TransferRequestStatus.PARTLY_APPLIED -> TransferRequestEventType.PARTLY_APPLIED
            TransferRequestStatus.FAILED -> TransferRequestEventType.FAILED
            else -> TransferRequestEventType.UPDATED
        }
        val oldLoanMeta = transferRequest.loanMeta.orElse(null)
        val newLoanMeta = if (borrowedLoanId != null && oldLoanMeta != null) {
            val newBorrowLoanIds = mutableSetOf<LoanId>()
            newBorrowLoanIds.addAll(oldLoanMeta.borrowLoanIds ?: emptySet())
            newBorrowLoanIds.add(borrowedLoanId)
            oldLoanMeta.copy(borrowLoanIds = newBorrowLoanIds)
        } else {
            oldLoanMeta
        }
        val transferRequestBuilder = TransferRequestModel.builder(transferRequest)
        val updatedRequest = with(transferRequestBuilder) {
            if (requestStatus != null) {
                status(requestStatus)
            }
            applicationDetails(applicationDetails)
            version(transferRequest.version + 1)
            nextHistoryOrder(transferRequest.nextHistoryOrder + 1)
            loanMeta(newLoanMeta)
        }.build()
        val historyBuilder = TransferRequestHistoryModel.builder()
        val historyModel = with(historyBuilder) {
            id(UUID.randomUUID().toString())
            tenantId(Tenants.DEFAULT_TENANT_ID)
            transferRequestId(transferRequest.id)
            type(eventType)
            timestamp(Instant.now())
            order(transferRequest.nextHistoryOrder)
            if (requestStatus != null) {
                val oldFields = TransferRequestHistoryFields.builder()
                    .status(transferRequest.status)
                    .build()
                val newFields = TransferRequestHistoryFields.builder()
                    .status(requestStatus)
                    .build()
                oldFields(oldFields)
                newFields(newFields)
            }
            if (oldLoanMeta != newLoanMeta) {
                oldLoanMeta(oldLoanMeta)
                newLoanMeta(newLoanMeta)
            }
            oldApplicationDetails(transferRequest.applicationDetails.orElse(null))
            newApplicationDetails(applicationDetails)
        }.build()
        val indicesDifference = if (requestStatus != null) {
            val currentIndices = indicesService.loadTransferRequestIndices(txSession, transferRequest).awaitSingle()
            val calculatedIndices = TransferRequestIndicesService
                .calculateTransferRequestIndices(updatedRequest)
            TransferRequestIndicesService.difference(currentIndices, calculatedIndices)
        } else {
            TransferRequestIndices.Difference.empty()
        }
        storeService.saveRefresh(txSession, updatedRequest, historyModel, indicesDifference).awaitSingleOrNull()
        return OperationFinishResult(updatedRequest, requestStatus != null)
    }

    data class OperationFinishResult(
        val updatedTransferRequest: TransferRequestModel,
        val closed: Boolean
    )

    data class ApplyResult(
        val updatedTransferRequest: TransferRequestModel,
        val updatedTransferHistory: TransferRequestHistoryModel
    )

    private data class ValidatedAccountTransfers(
        val account: AccountModel,
        val transfers: List<ValidatedQuotaResourceTransfer>,
        val field: String,
    )

    companion object {
        @JvmStatic
        fun toProvisionTransfer(provisionTransfer: ValidatedProvisionTransfer): ProvisionTransfer {
            val sourceAccountTransfers = provisionTransfer.sourceAccountTransfers
                .map { ResourceQuotaTransfer(it.resource.id, it.delta) }.toSet()
            val destinationAccountTransfers = provisionTransfer.destinationAccountTransfers
                .map { ResourceQuotaTransfer(it.resource.id, it.delta) }.toSet()
            return ProvisionTransfer(
                sourceAccountId = provisionTransfer.sourceAccount.id,
                sourceFolderId = provisionTransfer.sourceFolder.id,
                sourceServiceId = provisionTransfer.sourceService.id,
                destinationAccountId = provisionTransfer.destinationAccount.id,
                destinationFolderId = provisionTransfer.destinationFolder.id,
                destinationServiceId = provisionTransfer.destinationService.id,
                sourceAccountTransfers = sourceAccountTransfers,
                destinationAccountTransfers = destinationAccountTransfers,
                providerId = provisionTransfer.sourceAccount.providerId,
                accountsSpacesId = provisionTransfer.sourceAccount.accountsSpacesId.orElse(null),
                operationId = null,
            )
        }

        @JvmStatic
        private fun aggregateTransfers(
            transferByAccountResource: HashMap<Tuple2<AccountId, ResourceId>,
                Tuple2<ValidatedQuotaResourceTransfer, String>>,
            transfers: List<ValidatedQuotaResourceTransfer>,
            accountId: AccountId,
            field: String
        ) {
            transfers.forEach {
                val accountResource = Tuples.of(accountId, it.resource.id)
                if (accountResource in transferByAccountResource) {
                    val oldTransfer = transferByAccountResource[accountResource]!!
                    val newTransfer = ValidatedQuotaResourceTransfer(it.resource, it.delta + oldTransfer.t1.delta,
                        it.resourceUnitsEnsemble, it.deltaUnit, it.index)
                    transferByAccountResource[accountResource] = Tuples.of(newTransfer, field)
                } else {
                    transferByAccountResource[accountResource] = Tuples.of(it, field)
                }
            }
        }

        @JvmStatic
        fun aggregateProvisionTransfers(
            provisionTransferParameters: ValidatedProvisionTransferParameters
        ): Map<Tuple2<AccountId, ResourceId>, Tuple2<ValidatedQuotaResourceTransfer, String>> {
            val result = hashMapOf<Tuple2<AccountId, ResourceId>, Tuple2<ValidatedQuotaResourceTransfer, String>>()
            provisionTransferParameters.provisionTransfers.forEachIndexed { index, transfer ->
                aggregateTransfers(result, transfer.sourceAccountTransfers, transfer.sourceAccount.id,
                    "parameters.provisionTransfers.$index.sourceAccountTransfers")
                aggregateTransfers(result, transfer.destinationAccountTransfers, transfer.destinationAccount.id,
                    "parameters.provisionTransfers.$index.destinationAccountTransfers")
            }
            return result
        }
    }

    private fun getSubtype(
        provisionTransfers: List<ValidatedProvisionTransfer>,
        loanParameters: ValidatedLoanParameters?
    ): TransferRequestSubtype {
        return if (loanParameters == null) {
            val countTransfersWithReserveAccounts = provisionTransfers.stream()
                .filter { it.sourceAccount.reserveType.isPresent || it.destinationAccount.reserveType.isPresent }
                .count()

            if (countTransfersWithReserveAccounts == 0L) {
                TransferRequestSubtype.DEFAULT_PROVISION_TRANSFER
            } else {
                val sourceFolder = mutableSetOf<FolderModel>()
                val destinationFolder = mutableSetOf<FolderModel>()
                for (provisionTransfer in provisionTransfers) {
                    sourceFolder.add(provisionTransfer.sourceFolder)
                    destinationFolder.add(provisionTransfer.destinationFolder)
                }

                val isExchangeTransfer = provisionTransfers.size == 2 &&
                    sourceFolder.size == 2 &&
                    sourceFolder == destinationFolder &&
                    provisionTransfers.stream()
                        .allMatch{it.sourceAccount.reserveType.isPresent xor
                            it.destinationAccount.reserveType.isPresent }
                if (isExchangeTransfer) {
                    TransferRequestSubtype.EXCHANGE_PROVISION_TRANSFER
                } else {
                    TransferRequestSubtype.RESERVE_PROVISION_TRANSFER
                }
            }
        } else {
            TransferRequestSubtype.LOAN_PROVISION_TRANSFER
        }
    }
}
