package ru.yandex.intranet.d.services.transfer;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Objects;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.folders.TransferMetaHistoryModel;
import ru.yandex.intranet.d.model.folders.TransferMetaHistoryModel.RoleInTransfer;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.ProviderResponsible;
import ru.yandex.intranet.d.model.transfers.QuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ServiceResponsible;
import ru.yandex.intranet.d.model.transfers.TransferApplicationDetails;
import ru.yandex.intranet.d.model.transfers.TransferApplicationErrors;
import ru.yandex.intranet.d.model.transfers.TransferNotified;
import ru.yandex.intranet.d.model.transfers.TransferParameters;
import ru.yandex.intranet.d.model.transfers.TransferRequestByFolderModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByResponsibleModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByServiceModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestEventType;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryFields;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;
import ru.yandex.intranet.d.model.transfers.TransferRequestSubtype;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.transfers.TransferVote;
import ru.yandex.intranet.d.model.transfers.TransferVotes;
import ru.yandex.intranet.d.model.transfers.VoteType;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.transfer.model.CreateTransferRequestConfirmationData;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedQuotaTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedTransferRequestParameters;
import ru.yandex.intranet.d.services.transfer.model.PutTransferRequestConfirmationData;
import ru.yandex.intranet.d.services.transfer.model.QuotaTransferParametersContext;
import ru.yandex.intranet.d.services.transfer.model.QuotaTransferPreApplicationData;
import ru.yandex.intranet.d.services.transfer.model.ResponsibleAndNotified;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestIndices;
import ru.yandex.intranet.d.services.transfer.model.ValidatedCreateTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ValidatedPutTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedVoteTransferRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontPutTransferRequestDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Transfer request quota service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class TransferRequestQuotaService {

    private final TransferRequestValidationService validationService;
    private final TransferRequestStoreService storeService;
    private final TransferRequestIndicesService indicesService;
    private final TransferRequestResponsibleAndNotifyService transferRequestResponsibleAndNotifyService;
    private final TransferRequestVoteService transferRequestVoteService;
    private final TransferRequestPermissionService permissionService;
    private final MessageSource messages;

    public TransferRequestQuotaService(
            TransferRequestValidationService validationService,
            TransferRequestStoreService storeService,
            TransferRequestIndicesService indicesService,
            TransferRequestResponsibleAndNotifyService transferRequestResponsibleAndNotifyService,
            TransferRequestVoteService transferRequestVoteService,
            TransferRequestPermissionService permissionService,
            @Qualifier("messageSource") MessageSource messages) {
        this.validationService = validationService;
        this.storeService = storeService;
        this.indicesService = indicesService;
        this.transferRequestResponsibleAndNotifyService = transferRequestResponsibleAndNotifyService;
        this.transferRequestVoteService = transferRequestVoteService;
        this.permissionService = permissionService;
        this.messages = messages;
    }

    public Mono<Result<ValidatedCreateTransferRequest>> validateQuotaTransfer(
            YdbTxSession txSession,
            FrontCreateTransferRequestDto transferRequest,
            YaUserDetails currentUser,
            Locale locale,
            boolean publicApi,
            boolean delayValidation
    ) {
        PreValidatedTransferRequestParameters.Builder preValidatedRequestBuilder
                = PreValidatedTransferRequestParameters.builder();
        ErrorCollection.Builder preErrorsBuilder = ErrorCollection.builder();
        validationService.validateCreateFields(transferRequest::getDescription,
                transferRequest::getParameters, preValidatedRequestBuilder, preErrorsBuilder, locale);
        if (transferRequest.getLoanParameters().isPresent()) {
            preErrorsBuilder.addError("loanParameters", TypedError.invalid(messages
                    .getMessage("errors.loan.is.not.supported", null, locale)));
        }
        if (transferRequest.getParameters().isPresent()) {
            PreValidatedQuotaTransferParameters.Builder preValidatedParametersBuilder
                    = PreValidatedQuotaTransferParameters.builder();
            validationService.preValidateQuotaTransferParameters(transferRequest.getParameters().get(),
                    preValidatedParametersBuilder, preErrorsBuilder, locale, publicApi);
            if (preErrorsBuilder.hasAnyErrors()) {
                return Mono.just(Result.failure(preErrorsBuilder.build()));
            }
            PreValidatedQuotaTransferParameters preValidatedParameters = preValidatedParametersBuilder.build();
            PreValidatedTransferRequestParameters preValidatedRequest = preValidatedRequestBuilder.build();
            return validateQuotaTransferParameters(txSession, preValidatedParameters, currentUser, locale, publicApi)
                    .flatMap(r -> r.andThenMono(parameters ->
                            validationService.validateQuotaTransferApplication(txSession, parameters, locale,
                                            delayValidation)
                                    .flatMap(preApplicationR -> preApplicationR.andThenMono(preApplication ->
                                            transferRequestResponsibleAndNotifyService.calculateForQuotaTransferCreate(
                                                    txSession, parameters, currentUser)
                                                    .map(responsible -> validationService.validateResponsible(
                                                            responsible, transferRequest::getAddConfirmation,
                                                            currentUser, locale).apply(confirmation ->
                                                            prepareCreateQuotaTransferRequest(preValidatedRequest,
                                                                    parameters, preApplication, responsible,
                                                                    confirmation)))))));
        } else {
            return Mono.just(Result.failure(preErrorsBuilder.build()));
        }
    }

    public ValidatedCreateTransferRequest validateCreateApplicable(ValidatedCreateTransferRequest transferRequest) {
        ErrorCollection.Builder errorsBuilderRu = ErrorCollection.builder();
        ErrorCollection.Builder errorsBuilderEn = ErrorCollection.builder();
        QuotaTransferParametersContext context = QuotaTransferParametersContext.builder()
                .addFolders(transferRequest.getFolders())
                .addProviders(transferRequest.getProviders())
                .addQuotas(transferRequest.getQuotas())
                .addServices(transferRequest.getServices())
                .addResources(transferRequest.getResources())
                .addUnitsEnsembles(transferRequest.getUnitsEnsembles())
                .build();
        boolean canBeApplied = canQuotaTransferBeApplied(
                transferRequest.getTransferRequest(),
                errorsBuilderEn,
                errorsBuilderRu,
                context
        );
        if (!canBeApplied || errorsBuilderEn.hasAnyErrors() || errorsBuilderRu.hasAnyErrors()) {
            TransferApplicationDetails.Builder applicationDetailsBuilder = TransferApplicationDetails.builder();
            if (transferRequest.getTransferRequest().getApplicationDetails().isPresent()) {
                applicationDetailsBuilder = TransferApplicationDetails.builder(
                        transferRequest.getTransferRequest().getApplicationDetails().get()
                );
            }
            TransferApplicationDetails applicationDetails = applicationDetailsBuilder
                    .errorsEn(new TransferApplicationErrors(errorsBuilderEn.build()))
                    .errorsRu(new TransferApplicationErrors(errorsBuilderRu.build()))
                    .build();
            return ValidatedCreateTransferRequest.builder(transferRequest)
                    .transferRequest(
                            TransferRequestModel.builder(transferRequest.getTransferRequest())
                                    .status(TransferRequestStatus.FAILED)
                                    .applicationDetails(applicationDetails)
                                    .build()
                    )
                    .history(TransferRequestHistoryModel.builder(transferRequest.getHistory())
                            .newApplicationDetails(applicationDetails)
                            .build()
                    )
                    .apply(false)
                    .build();
        } else {
            return transferRequest;
        }
    }

    public ValidatedPutTransferRequest validatePutApplicable(ValidatedPutTransferRequest transferRequest) {
        ErrorCollection.Builder errorsBuilderRu = ErrorCollection.builder();
        ErrorCollection.Builder errorsBuilderEn = ErrorCollection.builder();
        QuotaTransferParametersContext context = QuotaTransferParametersContext.builder()
                .addFolders(transferRequest.getFolders())
                .addProviders(transferRequest.getProviders())
                .addQuotas(transferRequest.getQuotas())
                .addServices(transferRequest.getServices())
                .addResources(transferRequest.getResources())
                .addUnitsEnsembles(transferRequest.getUnitsEnsembles())
                .build();
        var request = transferRequest.getTransferRequest().orElseThrow();
        boolean canBeApplied = canQuotaTransferBeApplied(
                request,
                errorsBuilderEn,
                errorsBuilderRu,
                context
        );
        if (!canBeApplied || errorsBuilderEn.hasAnyErrors() || errorsBuilderRu.hasAnyErrors()) {
            TransferApplicationDetails.Builder applicationDetailsBuilder = TransferApplicationDetails.builder();
            if (request.getApplicationDetails().isPresent()) {
                applicationDetailsBuilder = TransferApplicationDetails.builder(
                        request.getApplicationDetails().get()
                );
            }
            TransferApplicationDetails applicationDetails = applicationDetailsBuilder
                    .errorsEn(new TransferApplicationErrors(errorsBuilderEn.build()))
                    .errorsRu(new TransferApplicationErrors(errorsBuilderRu.build()))
                    .build();
            ValidatedPutTransferRequest.Builder result = ValidatedPutTransferRequest.builder(transferRequest)
                    .transferRequest(
                            TransferRequestModel.builder(request)
                                    .status(TransferRequestStatus.FAILED)
                                    .applicationDetails(applicationDetails)
                                    .build()
                    )
                    .apply(false);

            if (transferRequest.getHistory().isPresent()) {
                result.history(TransferRequestHistoryModel.builder(transferRequest.getHistory().get())
                        .newApplicationDetails(applicationDetails)
                        .build()
                );
            }
            return result.build();
        } else {
            return transferRequest;
        }
    }

    @SuppressWarnings("checkstyle:ParameterNumber")
    public Mono<Result<ValidatedPutTransferRequest>> validatePutQuotaTransfer(
            YdbTxSession txSession, FrontPutTransferRequestDto putTransferRequest,
            TransferRequestModel transferRequest, ResponsibleAndNotified responsible,
            YaUserDetails currentUser, Locale locale, boolean publicApi, boolean delayValidation) {
        PreValidatedTransferRequestParameters.Builder preValidatedRequestBuilder
                = PreValidatedTransferRequestParameters.builder();
        ErrorCollection.Builder preErrorsBuilder = ErrorCollection.builder();
        validationService.validateCreateFields(putTransferRequest::getDescription,
                putTransferRequest::getParameters, preValidatedRequestBuilder, preErrorsBuilder, locale);
        if (putTransferRequest.getLoanParameters().isPresent()) {
            preErrorsBuilder.addError("loanParameters", TypedError.invalid(messages
                    .getMessage("errors.loan.is.not.supported", null, locale)));
        }
        if (putTransferRequest.getParameters().isPresent()) {
            PreValidatedQuotaTransferParameters.Builder preValidatedParametersBuilder
                    = PreValidatedQuotaTransferParameters.builder();
            validationService.preValidateQuotaTransferParameters(putTransferRequest.getParameters().get(),
                    preValidatedParametersBuilder, preErrorsBuilder, locale, publicApi);
            if (preErrorsBuilder.hasAnyErrors()) {
                return Mono.just(Result.failure(preErrorsBuilder.build()));
            }
            PreValidatedQuotaTransferParameters preValidatedParameters = preValidatedParametersBuilder.build();
            PreValidatedTransferRequestParameters preValidatedRequest = preValidatedRequestBuilder.build();
            return validateQuotaTransferParameters(txSession, preValidatedParameters, currentUser, locale, publicApi)
                    .flatMap(r -> r.andThenMono(parameters -> validationService
                            .validateQuotaTransferApplication(txSession, parameters, locale, delayValidation)
                            .flatMap(preApplicationR -> preApplicationR.andThenMono(preApplication ->
                                    validationService.validateResponsiblePut(responsible,
                                            putTransferRequest::getAddConfirmation, currentUser, locale)
                                            .applyMono(confirmation ->
                                                    indicesService.loadTransferRequestIndices(txSession,
                                                            transferRequest)
                                                            .map(indices -> preparePutQuotaTransferRequest(
                                                                    preValidatedRequest, parameters,
                                                                    preApplication, responsible, confirmation,
                                                                    indices, transferRequest)))))));

        } else {
            return Mono.just(Result.failure(preErrorsBuilder.build()));
        }
    }

    public Mono<Result<ValidatedVoteTransferRequest>> validateVoteQuotaTransfer(
            YdbTxSession txSession, VoteType voteType, TransferRequestModel transferRequest,
            ResponsibleAndNotified reloadedResponsible, YaUserDetails currentUser, Locale locale) {
        return loadQuotaTransferParametersContext(txSession,
                transferRequest.getParameters().getQuotaTransfers()).flatMap(context ->
                indicesService.loadTransferRequestIndices(txSession, transferRequest).map(indices ->
                        permissionService.checkUserCanVoteForRequest(transferRequest, reloadedResponsible, currentUser,
                                        locale)
                                .apply(v -> prepareVoteQuotaTransferRequest(voteType, transferRequest, context, indices,
                                        reloadedResponsible, currentUser))));
    }

    private ValidatedVoteTransferRequest prepareVoteQuotaTransferRequest(VoteType voteType,
                                                                         TransferRequestModel transferRequest,
                                                                         QuotaTransferParametersContext context,
                                                                         TransferRequestIndices indices,
                                                                         ResponsibleAndNotified responsibleAndNotified,
                                                                         YaUserDetails currentUser) {
        switch (voteType) {
            case REJECT:
                return prepareRejectQuotaTransferRequest(voteType, transferRequest, context, indices,
                        responsibleAndNotified, currentUser);
            case ABSTAIN:
                if (transferRequestVoteService.pendingTransferRequestUnableToConfirm(
                        responsibleAndNotified.getResponsible(),
                        transferRequest.getVotes(),
                        Set.of(currentUser.getUser().get().getId())
                )) {
                    return prepareRejectQuotaTransferRequest(voteType, transferRequest, context, indices,
                            responsibleAndNotified, currentUser);
                } else {
                    return prepareAbstainQuotaTransferRequest(voteType, transferRequest, context, indices,
                            responsibleAndNotified, currentUser);
                }
            case CONFIRM:
                return prepareConfirmQuotaTransferRequest(voteType, transferRequest, context, indices,
                        responsibleAndNotified, currentUser);
            default:
                throw new IllegalArgumentException("Unexpected vote type: " + voteType);
        }
    }

    private ValidatedVoteTransferRequest prepareRejectQuotaTransferRequest(VoteType voteType,
                                                                           TransferRequestModel transferRequest,
                                                                           QuotaTransferParametersContext context,
                                                                           TransferRequestIndices indices,
                                                                           ResponsibleAndNotified responsible,
                                                                           YaUserDetails currentUser) {
        Instant now = Instant.now();
        ValidatedVoteTransferRequest.Builder builder = ValidatedVoteTransferRequest.builder();
        prepareVoteBuilder(builder, context, now);
        builder.apply(false);
        TransferRequestModel.Builder transferRequestBuilder = TransferRequestModel.builder(transferRequest);
        transferRequestBuilder.status(TransferRequestStatus.REJECTED);
        transferRequestBuilder.version(transferRequest.getVersion() + 1L);
        transferRequestBuilder.nextHistoryOrder(transferRequest.getNextHistoryOrder() + 1L);
        TransferVotes updatedVotes = updateVotes(transferRequest.getVotes(), voteType, currentUser, now,
                transferRequest.getResponsible());
        transferRequestBuilder.votes(updatedVotes);
        transferRequestBuilder.responsible(responsible.getResponsible());
        TransferRequestModel newTransferRequest = transferRequestBuilder.build();
        builder.transferRequest(newTransferRequest);
        TransferRequestHistoryModel.Builder historyBuilder = TransferRequestHistoryModel.builder();
        historyBuilder.id(UUID.randomUUID().toString());
        historyBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        historyBuilder.transferRequestId(transferRequest.getId());
        historyBuilder.type(TransferRequestEventType.REJECTED);
        historyBuilder.timestamp(now);
        historyBuilder.authorId(currentUser.getUser().get().getId());
        TransferRequestHistoryFields.Builder oldFields = TransferRequestHistoryFields.builder()
                .version(transferRequest.getVersion());
        TransferRequestHistoryFields.Builder newFields = TransferRequestHistoryFields.builder()
                .version(transferRequest.getVersion() + 1L);
        if (!transferRequest.getStatus().equals(TransferRequestStatus.REJECTED)) {
            oldFields.status(transferRequest.getStatus());
            newFields.status(TransferRequestStatus.REJECTED);
        }
        historyBuilder.oldFields(oldFields.build());
        historyBuilder.newFields(newFields.build());
        historyBuilder.oldVotes(transferRequest.getVotes());
        historyBuilder.newVotes(updatedVotes);
        historyBuilder.order(transferRequest.getNextHistoryOrder());
        if (!responsible.getResponsible().equals(transferRequest.getResponsible())) {
            historyBuilder.oldResponsible(transferRequest.getResponsible());
            historyBuilder.newResponsible(responsible.getResponsible());
        }
        builder.history(historyBuilder.build());
        if (!transferRequest.getStatus().equals(TransferRequestStatus.REJECTED)) {
            TransferRequestIndices newTransferRequestIndices =
                    TransferRequestIndicesService.calculateTransferRequestIndices(newTransferRequest);
            TransferRequestIndices.Difference difference =
                    TransferRequestIndicesService.difference(indices, newTransferRequestIndices);
            builder.indicesDifference(difference);
        }
        return builder.build();
    }

    private ValidatedVoteTransferRequest prepareAbstainQuotaTransferRequest(VoteType voteType,
                                                                            TransferRequestModel transferRequest,
                                                                            QuotaTransferParametersContext context,
                                                                            TransferRequestIndices indices,
                                                                            ResponsibleAndNotified responsibleN,
                                                                            YaUserDetails currentUser) {
        Instant now = Instant.now();
        ValidatedVoteTransferRequest.Builder builder = ValidatedVoteTransferRequest.builder();
        prepareVoteBuilder(builder, context, now);
        builder.apply(false);
        TransferRequestModel.Builder transferRequestBuilder = TransferRequestModel.builder(transferRequest);
        transferRequestBuilder.version(transferRequest.getVersion() + 1L);
        transferRequestBuilder.nextHistoryOrder(transferRequest.getNextHistoryOrder() + 1L);
        TransferVotes updatedVotes = updateVotes(transferRequest.getVotes(), voteType, currentUser, now,
                transferRequest.getResponsible());
        transferRequestBuilder.votes(updatedVotes);
        TransferResponsible responsible = responsibleN.getResponsible();
        transferRequestBuilder.responsible(responsible);
        transferRequestBuilder.transferNotified(TransferNotified.from(responsibleN));
        final TransferRequestModel newTransferRequest = transferRequestBuilder.build();
        builder.transferRequest(newTransferRequest);
        TransferRequestHistoryModel.Builder historyBuilder = TransferRequestHistoryModel.builder();
        historyBuilder.id(UUID.randomUUID().toString());
        historyBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        historyBuilder.transferRequestId(transferRequest.getId());
        historyBuilder.type(TransferRequestEventType.VOTED);
        historyBuilder.timestamp(now);
        historyBuilder.authorId(currentUser.getUser().get().getId());
        TransferRequestHistoryFields.Builder oldFields = TransferRequestHistoryFields.builder()
                .version(transferRequest.getVersion());
        TransferRequestHistoryFields.Builder newFields = TransferRequestHistoryFields.builder()
                .version(transferRequest.getVersion() + 1L);
        historyBuilder.oldFields(oldFields.build());
        historyBuilder.newFields(newFields.build());
        historyBuilder.oldVotes(transferRequest.getVotes());
        historyBuilder.newVotes(updatedVotes);
        historyBuilder.order(transferRequest.getNextHistoryOrder());
        if (!responsible.equals(transferRequest.getResponsible())) {
            historyBuilder.oldResponsible(transferRequest.getResponsible());
            historyBuilder.newResponsible(responsible);
        }
        final TransferRequestIndices newIndices =
                TransferRequestIndicesService.calculateTransferRequestIndices(newTransferRequest);
        final TransferRequestIndices.Difference indicesDifference =
                TransferRequestIndicesService.difference(indices, newIndices);
        builder.indicesDifference(indicesDifference);
        Set<UserModel> notifiedUsers = TransferRequestResponsibleAndNotifyService
                .excludeAlreadyNotifiedUsers(transferRequest, responsibleN.getNotifiedUsers(), currentUser);
        builder.notifiedUsers(notifiedUsers);
        builder.history(historyBuilder.build());
        return builder.build();
    }

    private ValidatedVoteTransferRequest prepareConfirmQuotaTransferRequest(VoteType voteType,
                                                                            TransferRequestModel transferRequest,
                                                                            QuotaTransferParametersContext context,
                                                                            TransferRequestIndices indices,
                                                                            ResponsibleAndNotified responsible,
                                                                            YaUserDetails currentUser) {
        Instant now = Instant.now();
        ValidatedVoteTransferRequest.Builder builder = ValidatedVoteTransferRequest.builder();
        prepareVoteBuilder(builder, context, now);
        boolean apply = isApplyQuotaTransfer(transferRequest, responsible.getResponsible(), currentUser);
        ErrorCollection.Builder errorsEn = ErrorCollection.builder();
        ErrorCollection.Builder errorsRu = ErrorCollection.builder();
        boolean canBeApplied = apply && canQuotaTransferBeApplied(transferRequest, errorsEn, errorsRu, context);
        builder.apply(apply && canBeApplied);
        TransferRequestModel.Builder transferRequestBuilder = TransferRequestModel.builder(transferRequest);
        TransferRequestStatus targetStatus = apply
                ? (canBeApplied ? TransferRequestStatus.APPLIED : TransferRequestStatus.FAILED)
                : transferRequest.getStatus();
        transferRequestBuilder.status(targetStatus);
        transferRequestBuilder.version(transferRequest.getVersion() + 1L);
        transferRequestBuilder.nextHistoryOrder(transferRequest.getNextHistoryOrder() + 1L);
        if (apply && canBeApplied) {
            transferRequestBuilder.appliedAt(now);
        }
        TransferApplicationDetails updatedApplicationDetails = null;
        if (apply) {
            updatedApplicationDetails = updateTransferApplicationDetails(canBeApplied, errorsEn.build(),
                    errorsRu.build(), transferRequest.getParameters().getQuotaTransfers(), builder);
        } else {
            transferRequestBuilder.transferNotified(TransferNotified.from(responsible));
            builder.notifiedUsers(TransferRequestResponsibleAndNotifyService
                    .excludeAlreadyNotifiedUsers(transferRequest, responsible.getNotifiedUsers(), currentUser));
        }
        transferRequestBuilder.applicationDetails(updatedApplicationDetails);
        TransferVotes updatedVotes = updateVotes(transferRequest.getVotes(), voteType, currentUser, now,
                transferRequest.getResponsible());
        transferRequestBuilder.votes(updatedVotes);
        transferRequestBuilder.responsible(responsible.getResponsible());
        TransferRequestModel newTransferRequest = transferRequestBuilder.build();
        builder.transferRequest(newTransferRequest);
        TransferRequestHistoryModel.Builder historyBuilder = TransferRequestHistoryModel.builder();
        TransferRequestEventType eventType = targetStatus == TransferRequestStatus.APPLIED
                ? TransferRequestEventType.APPLIED : (targetStatus == TransferRequestStatus.FAILED
                ? TransferRequestEventType.FAILED : TransferRequestEventType.VOTED);
        historyBuilder.id(UUID.randomUUID().toString());
        historyBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        historyBuilder.transferRequestId(transferRequest.getId());
        historyBuilder.type(eventType);
        historyBuilder.timestamp(now);
        historyBuilder.authorId(currentUser.getUser().get().getId());
        TransferRequestHistoryFields.Builder oldFields = TransferRequestHistoryFields.builder()
                .version(transferRequest.getVersion());
        TransferRequestHistoryFields.Builder newFields = TransferRequestHistoryFields.builder()
                .version(transferRequest.getVersion() + 1L);
        if (!transferRequest.getStatus().equals(targetStatus)) {
            oldFields.status(transferRequest.getStatus());
            newFields.status(targetStatus);
        }
        historyBuilder.oldFields(oldFields.build());
        historyBuilder.newFields(newFields.build());
        historyBuilder.oldVotes(transferRequest.getVotes());
        historyBuilder.newVotes(updatedVotes);
        historyBuilder.oldApplicationDetails(null);
        historyBuilder.newApplicationDetails(updatedApplicationDetails);
        historyBuilder.order(transferRequest.getNextHistoryOrder());
        if (!responsible.getResponsible().equals(transferRequest.getResponsible())) {
            historyBuilder.oldResponsible(transferRequest.getResponsible());
            historyBuilder.newResponsible(responsible.getResponsible());
        }
        builder.history(historyBuilder.build());
        if (!transferRequest.getStatus().equals(targetStatus)) {
            TransferRequestIndices newTransferRequestIndices =
                    TransferRequestIndicesService.calculateTransferRequestIndices(newTransferRequest);
            TransferRequestIndices.Difference difference =
                    TransferRequestIndicesService.difference(indices, newTransferRequestIndices);
            builder.indicesDifference(difference);
        }
        return builder.build();
    }

    private void prepareVoteBuilder(ValidatedVoteTransferRequest.Builder builder,
                                    QuotaTransferParametersContext context, Instant now) {
        builder.quotas(context.getQuotas());
        builder.folders(context.getFolders());
        builder.services(context.getServices());
        builder.resources(context.getResources());
        builder.unitsEnsembles(context.getUnitsEnsembles());
        builder.providers(context.getProviders());
        builder.now(now);
    }

    private TransferApplicationDetails updateTransferApplicationDetails(
            boolean canBeApplied, ErrorCollection errorsEn, ErrorCollection errorsRu, Set<QuotaTransfer> quotaTransfers,
            ValidatedVoteTransferRequest.Builder resultBuilder) {
        TransferApplicationDetails.Builder detailsBuilder = TransferApplicationDetails.builder();
        if (canBeApplied) {
            quotaTransfers.forEach(quotaTransfer -> {
                String folderId = quotaTransfer.getDestinationFolderId();
                String opLogId = UUID.randomUUID().toString();
                resultBuilder.putPreGeneratedFolderOpLogId(folderId, opLogId);
                detailsBuilder.addFolderOpLogId(folderId, opLogId);
            });
        } else {
            TransferApplicationErrors.Builder errorsBuilderEn = TransferApplicationErrors.builder();
            TransferApplicationErrors.Builder errorsBuilderRu = TransferApplicationErrors.builder();
            errorsEn.getErrors().forEach(e -> errorsBuilderEn.addError(e.getError()));
            errorsRu.getErrors().forEach(e -> errorsBuilderRu.addError(e.getError()));
            errorsEn.getFieldErrors().forEach((field, errors) ->
                    errors.forEach(e -> errorsBuilderEn.addFieldError(field, e.getError())));
            errorsRu.getFieldErrors().forEach((field, errors) ->
                    errors.forEach(e -> errorsBuilderRu.addFieldError(field, e.getError())));
            detailsBuilder.errorsEn(errorsBuilderEn.build());
            detailsBuilder.errorsRu(errorsBuilderRu.build());
        }
        return detailsBuilder.build();
    }

    static TransferVotes updateVotes(TransferVotes currentVotes, VoteType voteType, YaUserDetails currentUser,
                                     Instant now, TransferResponsible responsible) {
        TransferVotes.Builder updatedVotes = TransferVotes.builder();
        String currentUserId = currentUser.getUser().get().getId();
        currentVotes.getVotes().forEach(vote -> {
            if (vote.getUserId().equals(currentUserId)) {
                return;
            }
            updatedVotes.addVote(vote);
        });
        Set<String> voteFolderIds = new HashSet<>();
        Set<Long> voteServiceIds = new HashSet<>();
        responsible.getResponsible().forEach(foldersResponsible -> {
            boolean responsibleForFolders = false;
            for (ServiceResponsible serviceResponsible : foldersResponsible.getResponsible()) {
                if (serviceResponsible.getResponsibleIds().contains(currentUserId)) {
                    voteServiceIds.add(serviceResponsible.getServiceId());
                    responsibleForFolders = true;
                }
            }
            if (responsibleForFolders) {
                voteFolderIds.addAll(foldersResponsible.getFolderIds());
            }
        });
        responsible.getReserveResponsibleModel().ifPresent(reserveResponsible -> {
            if (reserveResponsible.getResponsibleIds().contains(currentUserId)) {
                voteServiceIds.add(reserveResponsible.getServiceId());
                voteFolderIds.add(reserveResponsible.getFolderId());
            }
        });
        updatedVotes.addVote(TransferVote.builder()
                .userId(currentUserId)
                .timestamp(now)
                .type(voteType)
                .addFolderIds(voteFolderIds)
                .addServiceIds(voteServiceIds)
                .build());
        return updatedVotes.build();
    }

    private boolean isApplyQuotaTransfer(TransferRequestModel transferRequest, TransferResponsible responsible,
                                         YaUserDetails currentUser) {
        String currentUserId = currentUser.getUser().get().getId();
        Set<String> confirmedFolderIds = new HashSet<>();
        transferRequest.getVotes().getVotes().forEach(vote -> {
            if (vote.getUserId().equals(currentUserId)) {
                return;
            }
            if (VoteType.CONFIRM.equals(vote.getType())) {
                confirmedFolderIds.addAll(vote.getFolderIds());
            }
        });
        responsible.getResponsible().forEach(foldersResponsible -> {
            boolean isResponsible = foldersResponsible.getResponsible().stream().anyMatch(serviceResponsible ->
                    serviceResponsible.getResponsibleIds().contains(currentUserId));
            if (isResponsible) {
                confirmedFolderIds.addAll(foldersResponsible.getFolderIds());
            }
        });
        responsible.getReserveResponsibleModel().ifPresent(reserveResponsible -> {
            if (reserveResponsible.getResponsibleIds().contains(currentUserId)) {
                confirmedFolderIds.add(reserveResponsible.getFolderId());
            }
        });
        return transferRequest.getParameters().getQuotaTransfers().stream().allMatch(quotaTransfer ->
                confirmedFolderIds.contains(quotaTransfer.getDestinationFolderId()));
    }

    boolean canQuotaTransferBeApplied(TransferRequestModel transferRequest,
                                      ErrorCollection.Builder errorsEn,
                                      ErrorCollection.Builder errorsRu,
                                      QuotaTransferParametersContext context) {
        Map<String, Map<String, QuotaModel>> quotasByFolderResource = new HashMap<>();
        context.getQuotas().forEach(quota -> quotasByFolderResource.computeIfAbsent(quota.getFolderId(),
                k -> new HashMap<>()).put(quota.getResourceId(), quota));
        Map<String, ResourceModel> resourcesById = context.getResources().stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, FolderModel> foldersById = context.getFolders().stream()
                .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
        transferRequest.getParameters().getQuotaTransfers().forEach(quotaTransfer -> {
            quotaTransfer.getTransfers().forEach(resourceTransfer -> {
                QuotaModel currentQuota = quotasByFolderResource
                        .getOrDefault(quotaTransfer.getDestinationFolderId(), Collections.emptyMap())
                        .getOrDefault(resourceTransfer.getResourceId(), null);
                ResourceModel resource = resourcesById.get(resourceTransfer.getResourceId());
                FolderModel folder = foldersById.get(quotaTransfer.getDestinationFolderId());
                UnitsEnsembleModel unitsEnsemble = context.getUnitsEnsembles().stream()
                        .filter(e -> e.getId().equals(resource.getUnitsEnsembleId())).findFirst()
                        .orElseThrow(() -> new IllegalStateException(
                                "Unit ensemble not found " + resource.getUnitsEnsembleId()));
                validationService.validateQuotaTransfer(resourceTransfer, currentQuota, resource, folder,
                        unitsEnsemble, errorsEn, errorsRu);
            });
        });
        return !errorsEn.hasAnyErrors() && !errorsRu.hasAnyErrors();
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Void> applyQuotaTransferRequest(YdbTxSession txSession, List<QuotaModel> quotas,
                                                TransferRequestModel transferRequest, UserModel author,
                                                Map<String, String> preGeneratedFolderOpLogIdsByFolderId,
                                                List<ResourceModel> resources, List<FolderModel> folders,
                                                Instant now) {
        Map<String, Map<String, QuotaModel>> quotasByFolderResource = new HashMap<>();
        quotas.forEach(q -> quotasByFolderResource.computeIfAbsent(q.getFolderId(), k -> new HashMap<>())
                .put(q.getResourceId(), q));
        Map<String, ResourceModel> resourcesById = resources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, FolderModel> foldersById = folders.stream()
                .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
        List<QuotaModel> updatedQuotas = new ArrayList<>();
        List<FolderOperationLogModel.Builder> folderOpLogBuilders = new ArrayList<>();
        List<FolderOperationLogModel> newFolderOpLogs = new ArrayList<>();
        List<FolderModel> updatedFolders = new ArrayList<>();
        Map<String, TransferMetaHistoryModel> transfersMeta = prepareTransfersMeta(
                transferRequest.getId(), transferRequest.getParameters().getQuotaTransfers()
        );
        for (QuotaTransfer quotaTransfer : transferRequest.getParameters().getQuotaTransfers()) {
            FolderModel currentFolder = foldersById.get(quotaTransfer.getDestinationFolderId());
            FolderOperationLogModel.Builder opLogBuilder = FolderOperationLogModel.builder();
            opLogBuilder.setTenantId(Tenants.DEFAULT_TENANT_ID);
            opLogBuilder.setFolderId(quotaTransfer.getDestinationFolderId());
            opLogBuilder.setOperationDateTime(now);
            opLogBuilder.setId(preGeneratedFolderOpLogIdsByFolderId.get(quotaTransfer.getDestinationFolderId()));
            opLogBuilder.setProviderRequestId(null);
            switch (transferRequest.getType()) {
                case QUOTA_TRANSFER -> opLogBuilder.setOperationType(FolderOperationType.QUOTA_TRANSFER);
                case RESERVE_TRANSFER -> opLogBuilder.setOperationType(FolderOperationType.RESERVE_TRANSFER);
                default -> throw new IllegalArgumentException(
                        "Unexpected transfer request type: " + transferRequest.getType()
                );
            }
            opLogBuilder.setAuthorUserId(author.getId());
            opLogBuilder.setAuthorUserUid(author.getPassportUid().orElse(null));
            opLogBuilder.setSourceFolderOperationsLogId(null);
            opLogBuilder.setDestinationFolderOperationsLogId(null);
            opLogBuilder.setOldFolderFields(null);
            opLogBuilder.setNewFolderFields(null);
            opLogBuilder.setOldProvisions(new QuotasByAccount(Map.of()));
            opLogBuilder.setNewProvisions(new QuotasByAccount(Map.of()));
            opLogBuilder.setOldAccounts(null);
            opLogBuilder.setNewAccounts(null);
            opLogBuilder.setActuallyAppliedProvisions(null);
            opLogBuilder.setAccountsQuotasOperationsId(null);
            opLogBuilder.setQuotasDemandsId(transferRequest.getId());
            opLogBuilder.setTransferMeta(transfersMeta.get(quotaTransfer.getDestinationFolderId()));
            opLogBuilder.setOperationPhase(null);
            opLogBuilder.setOrder(currentFolder.getNextOpLogOrder());
            Map<String, Long> oldQuotasByResourceId = new HashMap<>();
            Map<String, Long> newQuotasByResourceId = new HashMap<>();
            Map<String, Long> oldBalanceByResourceId = new HashMap<>();
            Map<String, Long> newBalanceByResourceId = new HashMap<>();
            for (ResourceQuotaTransfer resourceTransfer : quotaTransfer.getTransfers()) {
                QuotaModel currentQuota = quotasByFolderResource
                        .getOrDefault(quotaTransfer.getDestinationFolderId(), Collections.emptyMap())
                        .get(resourceTransfer.getResourceId());
                long delta = resourceTransfer.getDelta();
                if (currentQuota != null) {
                    QuotaModel updatedQuota = QuotaModel.builder(currentQuota)
                            .quota((currentQuota.getQuota() != null ? currentQuota.getQuota() : 0L)
                                    + delta)
                            .balance((currentQuota.getBalance() != null ? currentQuota.getBalance() : 0L)
                                    + delta)
                            .build();
                    updatedQuotas.add(updatedQuota);
                    oldQuotasByResourceId.put(resourceTransfer.getResourceId(),
                            currentQuota.getQuota() != null ? currentQuota.getQuota() : 0L);
                    newQuotasByResourceId.put(resourceTransfer.getResourceId(),
                            updatedQuota.getQuota() != null ? updatedQuota.getQuota() : 0L);
                    oldBalanceByResourceId.put(resourceTransfer.getResourceId(),
                            currentQuota.getBalance() != null ? currentQuota.getBalance() : 0L);
                    newBalanceByResourceId.put(resourceTransfer.getResourceId(),
                            updatedQuota.getBalance() != null ? updatedQuota.getBalance() : 0L);
                } else {
                    QuotaModel newQuota = QuotaModel.builder()
                            .tenantId(Tenants.DEFAULT_TENANT_ID)
                            .folderId(quotaTransfer.getDestinationFolderId())
                            .resourceId(resourceTransfer.getResourceId())
                            .providerId(resourcesById.get(resourceTransfer.getResourceId()).getProviderId())
                            .quota(delta)
                            .balance(delta)
                            .frozenQuota(0L)
                            .build();
                    updatedQuotas.add(newQuota);
                    oldQuotasByResourceId.put(resourceTransfer.getResourceId(), 0L);
                    newQuotasByResourceId.put(resourceTransfer.getResourceId(),
                            newQuota.getQuota() != null ? newQuota.getQuota() : 0L);
                    oldBalanceByResourceId.put(resourceTransfer.getResourceId(), 0L);
                    newBalanceByResourceId.put(resourceTransfer.getResourceId(),
                            newQuota.getBalance() != null ? newQuota.getBalance() : 0L);
                }
            }
            opLogBuilder.setOldQuotas(new QuotasByResource(oldQuotasByResourceId));
            opLogBuilder.setNewQuotas(new QuotasByResource(newQuotasByResourceId));
            opLogBuilder.setOldBalance(new QuotasByResource(oldBalanceByResourceId));
            opLogBuilder.setNewBalance(new QuotasByResource(newBalanceByResourceId));
            folderOpLogBuilders.add(opLogBuilder);
            updatedFolders.add(currentFolder.toBuilder()
                    .setNextOpLogOrder(currentFolder.getNextOpLogOrder() + 1L)
                    .build());
        }
        if (folderOpLogBuilders.size() == 2 && transferRequest.getParameters().getQuotaTransfers().size() == 2) {
            Optional<QuotaTransfer> destinationTransfer = transferRequest.getParameters().getQuotaTransfers().stream()
                    .filter(quotaTransfer -> quotaTransfer.getTransfers().stream()
                            .allMatch(resourceTransfer -> resourceTransfer.getDelta() > 0)).findFirst();
            Optional<QuotaTransfer> sourceTransfer = transferRequest.getParameters().getQuotaTransfers().stream()
                    .filter(quotaTransfer -> quotaTransfer.getTransfers().stream()
                            .allMatch(resourceTransfer -> resourceTransfer.getDelta() < 0)).findFirst();
            if (destinationTransfer.isPresent() && sourceTransfer.isPresent()) {
                String destinationFolderId = destinationTransfer.get().getDestinationFolderId();
                String sourceFolderId = sourceTransfer.get().getDestinationFolderId();
                Optional<FolderOperationLogModel.Builder> sourceOpLog = folderOpLogBuilders.stream()
                        .filter(opLog -> Objects.equal(sourceFolderId, opLog.getFolderId().get())).findFirst();
                Optional<FolderOperationLogModel.Builder> destinationOpLog = folderOpLogBuilders.stream()
                        .filter(opLog -> Objects.equal(destinationFolderId, opLog.getFolderId().get())).findFirst();
                if (sourceOpLog.isPresent() && destinationOpLog.isPresent()) {
                    sourceOpLog.get().setDestinationFolderOperationsLogId(destinationOpLog.get().getId().get());
                    destinationOpLog.get().setSourceFolderOperationsLogId(sourceOpLog.get().getId().get());
                }
            }
        }
        folderOpLogBuilders.forEach(builder -> newFolderOpLogs.add(builder.build()));
        return storeService.upsertQuotas(txSession, updatedQuotas)
                .then(Mono.defer(() -> storeService.upsertFolderOperationLog(txSession, newFolderOpLogs)))
                .then(Mono.defer(() -> storeService.upsertFolders(txSession, updatedFolders)));
    }

    private static Map<String, TransferMetaHistoryModel> prepareTransfersMeta(
            String transferRequestId, Set<QuotaTransfer> quotaTransfers
    ) {
        Map<String, RoleInTransfer> roleByFolderId = new HashMap<>();
        for (QuotaTransfer transfer : quotaTransfers) {
            long positiveCount = transfer.getTransfers().stream()
                    .filter(quotaTransfer -> quotaTransfer.getDelta() > 0)
                    .count();
            long negativeCount = transfer.getTransfers().size() - positiveCount;
            roleByFolderId.put(
                    transfer.getDestinationFolderId(),
                    positiveCount > negativeCount ? RoleInTransfer.DESTINATION : RoleInTransfer.SOURCE
            );
        }
        Map<String, TransferMetaHistoryModel> result = new HashMap<>();
        for (String folderId : roleByFolderId.keySet()) {
            RoleInTransfer role = roleByFolderId.get(folderId);
            Set<TransferMetaHistoryModel.Another> anotherParticipants = quotaTransfers.stream()
                    .filter(transfer -> !folderId.equals(transfer.getDestinationFolderId()))
                    .filter(transfer -> role != roleByFolderId.get(transfer.getDestinationFolderId()))
                    .map(transfer -> new TransferMetaHistoryModel.Another(
                            transfer.getDestinationServiceId(),
                            transfer.getDestinationFolderId(),
                            null
                    ))
                    .collect(Collectors.toSet());
            result.put(folderId, new TransferMetaHistoryModel(
                    transferRequestId,
                    role,
                    anotherParticipants
            ));
        }
        return result;
    }

    private ValidatedPutTransferRequest preparePutQuotaTransferRequest(
            PreValidatedTransferRequestParameters preValidatedRequest,
            ValidatedQuotaTransferParameters validatedParameters,
            QuotaTransferPreApplicationData preApplicationData,
            ResponsibleAndNotified responsibleAndNotified,
            PutTransferRequestConfirmationData confirmation,
            TransferRequestIndices transferRequestIndices,
            TransferRequestModel currentTransferRequest) {
        TransferResponsible responsible = responsibleAndNotified.getResponsible();
        Instant now = Instant.now();
        ValidatedPutTransferRequest.Builder validatedBuilder = ValidatedPutTransferRequest.builder();
        validatedBuilder.quotas(preApplicationData.getQuotas());
        validatedBuilder.folders(validatedParameters.getFolders());
        validatedBuilder.services(validatedParameters.getServices());
        validatedBuilder.resources(validatedParameters.getResources());
        validatedBuilder.unitsEnsembles(validatedParameters.getUnitsEnsembles());
        validatedBuilder.providers(validatedParameters.getProviders());
        validatedBuilder.now(now);
        validatedBuilder.oldTransferRequest(currentTransferRequest);
        TransferRequestModel.Builder transferRequestBuilder = TransferRequestModel.builder(currentTransferRequest);
        TransferParameters.Builder parametersBuilder = TransferParameters.builder();
        validatedParameters.getTransfers().forEach(quotaTransfer -> {
            QuotaTransfer.Builder quotaTransferBuilder = QuotaTransfer.builder();
            quotaTransferBuilder.folderId(quotaTransfer.getDestinationFolder().getId());
            quotaTransferBuilder.serviceId(quotaTransfer.getDestinationService().getId());
            quotaTransfer.getTransfers().forEach(resourceTransfer -> {
                ResourceQuotaTransfer.Builder resourceQuotaTransferBuilder = ResourceQuotaTransfer.builder();
                resourceQuotaTransferBuilder.resourceId(resourceTransfer.getResource().getId());
                resourceQuotaTransferBuilder.delta(resourceTransfer.getDelta());
                quotaTransferBuilder.addTransfer(resourceQuotaTransferBuilder.build());
            });
            parametersBuilder.addQuotaTransfer(quotaTransferBuilder.build());
        });
        TransferParameters updatedTransferParameters = parametersBuilder.build();
        boolean parametersUpdated = !updatedTransferParameters.equals(currentTransferRequest.getParameters());
        boolean responsibleUpdated = !responsible.equals(currentTransferRequest.getResponsible());
        TransferVotes.Builder votesBuilder = TransferVotes.builder();
        if (confirmation.isConfirm()) {
            TransferVote vote = prepareConfirmationVote(responsible, now, confirmation.getUser());
            votesBuilder.addVote(vote);
        }
        if (!parametersUpdated) {
            copyVotes(responsible, confirmation, currentTransferRequest, votesBuilder);
        }
        TransferVotes updatedTransferVotes = votesBuilder.build();
        boolean votesUpdated = !updatedTransferVotes.equals(currentTransferRequest.getVotes());
        boolean apply = updatedQuotaTransferRequestMustBeApplied(updatedTransferVotes, updatedTransferParameters);
        TransferRequestStatus targetStatus = apply
                ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING;
        TransferApplicationDetails.Builder applicationDetailsBuilder = TransferApplicationDetails.builder();
        if (apply) {
            validatedParameters.getTransfers().forEach(quotaTransfer -> {
                String folderId = quotaTransfer.getDestinationFolder().getId();
                String opLogId = UUID.randomUUID().toString();
                validatedBuilder.putPreGeneratedFolderOpLogId(folderId, opLogId);
                applicationDetailsBuilder.addFolderOpLogId(folderId, opLogId);
            });
        }
        TransferRequestHistoryModel.Builder historyBuilder = TransferRequestHistoryModel.builder();
        String transferRequestId = currentTransferRequest.getId();
        TransferApplicationDetails transferApplicationDetails = applicationDetailsBuilder.build();
        historyBuilder.id(UUID.randomUUID().toString());
        historyBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        historyBuilder.transferRequestId(transferRequestId);
        historyBuilder.type(TransferRequestEventType.UPDATED);
        historyBuilder.timestamp(now);
        historyBuilder.authorId(confirmation.getUser().getId());
        TransferRequestHistoryFields.Builder oldFields = TransferRequestHistoryFields.builder()
                .version(currentTransferRequest.getVersion());
        TransferRequestHistoryFields.Builder newFields = TransferRequestHistoryFields.builder()
                .version(currentTransferRequest.getVersion() + 1L);
        if (!currentTransferRequest.getDescription().equals(preValidatedRequest.getDescription())) {
            oldFields.description(currentTransferRequest.getDescription().orElse(null));
            newFields.description(preValidatedRequest.getDescription().orElse(null));
        }
        if (!currentTransferRequest.getStatus().equals(targetStatus)) {
            oldFields.status(currentTransferRequest.getStatus());
            newFields.status(targetStatus);
        }
        oldFields.type(null);
        newFields.type(null);
        historyBuilder.oldFields(oldFields.build());
        historyBuilder.newFields(newFields.build());
        if (parametersUpdated) {
            historyBuilder.oldParameters(currentTransferRequest.getParameters());
            historyBuilder.newParameters(updatedTransferParameters);
        } else {
            historyBuilder.oldParameters(null);
            historyBuilder.newParameters(null);
        }
        if (responsibleUpdated) {
            historyBuilder.oldResponsible(currentTransferRequest.getResponsible());
            historyBuilder.newResponsible(responsible);
        } else {
            historyBuilder.oldResponsible(null);
            historyBuilder.newResponsible(null);
        }
        if (votesUpdated) {
            historyBuilder.oldVotes(currentTransferRequest.getVotes());
            historyBuilder.newVotes(updatedTransferVotes);
        } else {
            historyBuilder.oldVotes(null);
            historyBuilder.newVotes(null);
        }
        historyBuilder.oldApplicationDetails(null);
        if (apply) {
            historyBuilder.newApplicationDetails(transferApplicationDetails);
        } else {
            historyBuilder.newApplicationDetails(null);
        }
        historyBuilder.order(currentTransferRequest.getNextHistoryOrder());
        transferRequestBuilder.version(currentTransferRequest.getVersion() + 1L);
        transferRequestBuilder.description(preValidatedRequest.getDescription().orElse(null));
        transferRequestBuilder.status(targetStatus);
        transferRequestBuilder.updatedBy(confirmation.getUser().getId());
        transferRequestBuilder.updatedAt(now);
        transferRequestBuilder.appliedAt(apply ? now : null);
        transferRequestBuilder.parameters(updatedTransferParameters);
        transferRequestBuilder.responsible(responsible);
        transferRequestBuilder.votes(updatedTransferVotes);
        transferRequestBuilder.applicationDetails(apply ? transferApplicationDetails : null);
        transferRequestBuilder.nextHistoryOrder(currentTransferRequest.getNextHistoryOrder() + 1L);
        if (!transferRequestBuilder.hasChanges(currentTransferRequest)) {
            validatedBuilder.apply(false);
            validatedBuilder.transferRequest(null);
            validatedBuilder.history(null);
            return validatedBuilder.build();
        }
        validatedBuilder.apply(apply);
        transferRequestBuilder.transferNotified(TransferNotified.from(responsibleAndNotified));
        TransferRequestModel newTransferRequest = transferRequestBuilder.build();
        validatedBuilder.transferRequest(newTransferRequest);
        validatedBuilder.history(historyBuilder.build());

        TransferRequestIndices newTransferRequestIndices =
                TransferRequestIndicesService.calculateTransferRequestIndices(newTransferRequest);
        TransferRequestIndices.Difference difference =
                TransferRequestIndicesService.difference(transferRequestIndices, newTransferRequestIndices);
        validatedBuilder.indicesDifference(difference);

        prepareNotifiedUsersPut(confirmation, validatedBuilder, responsibleAndNotified, parametersUpdated, apply,
                currentTransferRequest);

        return validatedBuilder.build();
    }

    static void copyVotes(TransferResponsible responsible, PutTransferRequestConfirmationData confirmation,
                          TransferRequestModel currentTransferRequest, TransferVotes.Builder votesBuilder) {
        // If parameters was not updated then existing votes are saved
        // except votes for folders where user is not responsible anymore.
        // This means that the request can't become rejected this way.
        // That's because pending request can not have any rejections and so request can at most become accepted.
        currentTransferRequest.getVotes().getVotes().forEach(vote -> {
            if (confirmation.isConfirm() && confirmation.getUser().getId().equals(vote.getUserId())) {
                return;
            }
            TransferVote.Builder voteBuilder = TransferVote.builder();
            voteBuilder.userId(vote.getUserId());
            voteBuilder.type(vote.getType());
            voteBuilder.timestamp(vote.getTimestamp());
            Set<String> voteFolderIds = new HashSet<>();
            Set<Long> voteServiceIds = new HashSet<>();
            responsible.getResponsible().forEach(foldersResponsible -> {
                for (String folderId : foldersResponsible.getFolderIds()) {
                    if (vote.getFolderIds().contains(folderId)) {
                        boolean responsibleForFolders = false;
                        for (ServiceResponsible serviceResponsible : foldersResponsible.getResponsible()) {
                            if (serviceResponsible.getResponsibleIds().contains(vote.getUserId())) {
                                voteServiceIds.add(serviceResponsible.getServiceId());
                                responsibleForFolders = true;
                            }
                        }
                        if (responsibleForFolders) {
                            voteFolderIds.add(folderId);
                        }
                    }
                }
            });
            responsible.getReserveResponsibleModel().ifPresent(reserveResponsible -> {
                if (reserveResponsible.getResponsibleIds().contains(vote.getUserId())) {
                    voteServiceIds.add(reserveResponsible.getServiceId());
                    voteFolderIds.add(reserveResponsible.getFolderId());
                }
            });
            if (!voteFolderIds.isEmpty()) {
                voteBuilder.addFolderIds(voteFolderIds);
                voteBuilder.addServiceIds(voteServiceIds);
                votesBuilder.addVote(voteBuilder.build());
            }
        });
    }

    static boolean updatedQuotaTransferRequestMustBeApplied(TransferVotes updatedTransferVotes,
                                                            TransferParameters updatedTransferParameters) {
        Map<String, Set<String>> confirmationsByFolderId = new HashMap<>();
        updatedTransferVotes.getVotes().forEach(vote -> {
            if (VoteType.CONFIRM.equals(vote.getType())) {
                vote.getFolderIds().forEach(folderId -> confirmationsByFolderId
                        .computeIfAbsent(folderId, k -> new HashSet<>()).add(vote.getUserId()));
            }
        });
        return updatedTransferParameters.getQuotaTransfers().stream()
                .noneMatch(transfer -> confirmationsByFolderId.getOrDefault(transfer.getDestinationFolderId(),
                        Collections.emptySet()).isEmpty());
    }

    private ValidatedCreateTransferRequest prepareCreateQuotaTransferRequest(
            PreValidatedTransferRequestParameters preValidatedRequest,
            ValidatedQuotaTransferParameters validatedParameters,
            QuotaTransferPreApplicationData preApplicationData,
            ResponsibleAndNotified responsibleAndNotified,
            CreateTransferRequestConfirmationData confirmation) {
        TransferResponsible responsible = responsibleAndNotified.getResponsible();
        Instant now = Instant.now();
        ValidatedCreateTransferRequest.Builder validatedBuilder = ValidatedCreateTransferRequest.builder();
        validatedBuilder.quotas(preApplicationData.getQuotas());
        validatedBuilder.folders(validatedParameters.getFolders());
        validatedBuilder.services(validatedParameters.getServices());
        validatedBuilder.resources(validatedParameters.getResources());
        validatedBuilder.unitsEnsembles(validatedParameters.getUnitsEnsembles());
        validatedBuilder.providers(validatedParameters.getProviders());
        validatedBuilder.now(now);
        validatedBuilder.apply(confirmation.isApply());
        TransferRequestModel.Builder transferRequestBuilder = TransferRequestModel.builder();
        TransferParameters.Builder parametersBuilder = TransferParameters.builder();
        validatedParameters.getTransfers().forEach(quotaTransfer -> {
            QuotaTransfer.Builder quotaTransferBuilder = QuotaTransfer.builder();
            quotaTransferBuilder.folderId(quotaTransfer.getDestinationFolder().getId());
            quotaTransferBuilder.serviceId(quotaTransfer.getDestinationService().getId());
            quotaTransfer.getTransfers().forEach(resourceTransfer -> {
                ResourceQuotaTransfer.Builder resourceQuotaTransferBuilder = ResourceQuotaTransfer.builder();
                resourceQuotaTransferBuilder.resourceId(resourceTransfer.getResource().getId());
                resourceQuotaTransferBuilder.delta(resourceTransfer.getDelta());
                quotaTransferBuilder.addTransfer(resourceQuotaTransferBuilder.build());
            });
            parametersBuilder.addQuotaTransfer(quotaTransferBuilder.build());
        });
        TransferVotes.Builder votesBuilder = TransferVotes.builder();
        if (confirmation.isConfirm()) {
            if (confirmation.isProviderResponsibleAutoConfirm()) {
                // See DISPENSER-3540
                TransferVote vote = prepareProviderAutoConfirmationVote(responsible, now,
                        confirmation.getUser(), validatedParameters);
                votesBuilder.addVote(vote);
            } else {
                TransferVote vote = prepareConfirmationVote(responsible, now, confirmation.getUser());
                votesBuilder.addVote(vote);
            }
        }
        TransferApplicationDetails.Builder applicationDetailsBuilder = TransferApplicationDetails.builder();
        if (confirmation.isApply()) {
            validatedParameters.getTransfers().forEach(quotaTransfer -> {
                String folderId = quotaTransfer.getDestinationFolder().getId();
                String opLogId = UUID.randomUUID().toString();
                validatedBuilder.putPreGeneratedFolderOpLogId(folderId, opLogId);
                applicationDetailsBuilder.addFolderOpLogId(folderId, opLogId);
            });
        }
        TransferRequestHistoryModel.Builder historyBuilder = TransferRequestHistoryModel.builder();
        String transferRequestId = UUID.randomUUID().toString();
        TransferParameters transferParameters = parametersBuilder.build();
        TransferVotes transferVotes = votesBuilder.build();
        TransferApplicationDetails transferApplicationDetails = applicationDetailsBuilder.build();
        historyBuilder.id(UUID.randomUUID().toString());
        historyBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        historyBuilder.transferRequestId(transferRequestId);
        historyBuilder.type(TransferRequestEventType.CREATED);
        historyBuilder.timestamp(now);
        historyBuilder.authorId(confirmation.getUser().getId());
        historyBuilder.oldFields(null);
        historyBuilder.oldParameters(null);
        historyBuilder.oldResponsible(null);
        historyBuilder.oldVotes(null);
        historyBuilder.oldApplicationDetails(null);
        String summary = generateSummary(transferParameters,
                validatedParameters.getServices(), validatedParameters.getFolders());
        historyBuilder.newFields(TransferRequestHistoryFields.builder()
                .version(0L)
                .summary(summary)
                .description(preValidatedRequest.getDescription().orElse(null))
                .type(TransferRequestType.QUOTA_TRANSFER)
                .status(confirmation.isApply()
                        ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING)
                .build());
        historyBuilder.newParameters(transferParameters);
        historyBuilder.newVotes(transferVotes);
        historyBuilder.newResponsible(responsible);
        historyBuilder.newApplicationDetails(confirmation.isApply() ? transferApplicationDetails : null);
        historyBuilder.order(0L);
        transferRequestBuilder.id(transferRequestId);
        transferRequestBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        transferRequestBuilder.version(0L);
        transferRequestBuilder.summary(summary);
        transferRequestBuilder.description(preValidatedRequest.getDescription().orElse(null));
        transferRequestBuilder.type(TransferRequestType.QUOTA_TRANSFER);
        transferRequestBuilder.subtype(TransferRequestSubtype.DEFAULT_QUOTA_TRANSFER);
        transferRequestBuilder.status(confirmation.isApply()
                ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING);
        transferRequestBuilder.createdBy(confirmation.getUser().getId());
        transferRequestBuilder.updatedBy(null);
        transferRequestBuilder.createdAt(now);
        transferRequestBuilder.updatedAt(null);
        transferRequestBuilder.appliedAt(confirmation.isApply() ? now : null);
        transferRequestBuilder.parameters(transferParameters);
        transferRequestBuilder.responsible(responsible);
        transferRequestBuilder.votes(transferVotes);
        transferRequestBuilder.applicationDetails(confirmation.isApply() ? transferApplicationDetails : null);
        transferRequestBuilder.nextHistoryOrder(1L);
        transferRequestBuilder.transferNotified(TransferNotified.from(responsibleAndNotified));
        validatedBuilder.transferRequest(transferRequestBuilder.build());
        validatedBuilder.history(historyBuilder.build());
        validatedParameters.getTransfers().forEach(quotaTransfer -> {
            validatedBuilder.addFolderIndex(TransferRequestByFolderModel.builder()
                    .tenantId(Tenants.DEFAULT_TENANT_ID)
                    .folderId(quotaTransfer.getDestinationFolder().getId())
                    .status(confirmation.isApply()
                            ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING)
                    .createdAt(now)
                    .transferRequestId(transferRequestId)
                    .build());
            validatedBuilder.addServiceIndex(TransferRequestByServiceModel.builder()
                    .tenantId(Tenants.DEFAULT_TENANT_ID)
                    .serviceId(quotaTransfer.getDestinationService().getId())
                    .status(confirmation.isApply()
                            ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING)
                    .createdAt(now)
                    .transferRequestId(transferRequestId)
                    .build());
        });
        addResponsibleIndex(confirmation, responsible, now, validatedBuilder, transferRequestId);
        prepareNotifiedUsersCreate(confirmation, validatedBuilder, responsibleAndNotified);
        return validatedBuilder.build();
    }

    static void addResponsibleIndex(
            CreateTransferRequestConfirmationData confirmation, TransferResponsible responsible, Instant now,
            ValidatedCreateTransferRequest.Builder validatedBuilder, String transferRequestId) {
        responsible.getResponsible().forEach(foldersResponsible ->
                foldersResponsible.getResponsible().forEach(serviceResponsible ->
                        serviceResponsible.getResponsibleIds().forEach(responsibleId ->
                                validatedBuilder.addResponsibleIndex(
                                        TransferRequestByResponsibleModel.builder()
                                                .tenantId(Tenants.DEFAULT_TENANT_ID)
                                                .responsibleId(responsibleId)
                                                .status(confirmation.isApply()
                                                        ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING)
                                                .createdAt(now)
                                                .transferRequestId(transferRequestId)
                                                .build()))));
        responsible.getReserveResponsibleModel().ifPresent(reserveResponsible ->
                reserveResponsible.getResponsibleIds().forEach(responsibleId ->
                        validatedBuilder.addResponsibleIndex(
                                TransferRequestByResponsibleModel.builder()
                                        .tenantId(Tenants.DEFAULT_TENANT_ID)
                                        .responsibleId(responsibleId)
                                        .status(confirmation.isApply()
                                                ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING)
                                        .createdAt(now)
                                        .transferRequestId(transferRequestId)
                                        .build())));
    }

    static TransferVote prepareConfirmationVote(TransferResponsible responsible, Instant now, UserModel user) {
        TransferVote.Builder voteBuilder = TransferVote.builder();
        voteBuilder.userId(user.getId());
        voteBuilder.type(VoteType.CONFIRM);
        voteBuilder.timestamp(now);
        Set<String> voteFolderIds = new HashSet<>();
        Set<Long> voteServiceIds = new HashSet<>();
        responsible.getResponsible().forEach(foldersResponsible -> {
            boolean responsibleForFolders = false;
            for (ServiceResponsible serviceResponsible : foldersResponsible.getResponsible()) {
                if (serviceResponsible.getResponsibleIds().contains(user.getId())) {
                    voteServiceIds.add(serviceResponsible.getServiceId());
                    responsibleForFolders = true;
                }
            }
            if (responsibleForFolders) {
                voteFolderIds.addAll(foldersResponsible.getFolderIds());
            }
        });
        responsible.getReserveResponsibleModel().ifPresent(reserveResponsible -> {
            if (reserveResponsible.getResponsibleIds().contains(user.getId())) {
                voteFolderIds.add(reserveResponsible.getFolderId());
                voteServiceIds.add(reserveResponsible.getServiceId());
            }
        });
        voteBuilder.addFolderIds(voteFolderIds);
        voteBuilder.addServiceIds(voteServiceIds);
        return voteBuilder.build();
    }

    private TransferVote prepareProviderAutoConfirmationVote(TransferResponsible responsible, Instant now,
                                                             UserModel user,
                                                             ValidatedQuotaTransferParameters validatedParameters) {
        TransferVote.Builder voteBuilder = TransferVote.builder();
        voteBuilder.userId(user.getId());
        voteBuilder.type(VoteType.CONFIRM);
        voteBuilder.timestamp(now);
        validatedParameters.getTransfers().forEach(t -> voteBuilder.addFolderId(t.getDestinationFolder().getId()));
        voteBuilder.addProviderIds(responsible.getProviderResponsible().stream()
                .filter(r -> r.getResponsibleId().equals(user.getId()))
                .map(ProviderResponsible::getProviderIds).findFirst().orElse(Set.of()));
        return voteBuilder.build();
    }

    Mono<Result<ValidatedQuotaTransferParameters>> validateQuotaTransferParameters(
            YdbTxSession txSession, PreValidatedQuotaTransferParameters parameters, YaUserDetails currentUser,
            Locale locale, boolean publicApi) {
        Set<String> folderIds = new HashSet<>();
        Set<Long> serviceIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        parameters.getTransfers().forEach(quotaTransfer -> {
            folderIds.add(quotaTransfer.getDestinationFolderId());
            quotaTransfer.getDestinationServiceId().ifPresent(serviceIds::add);
            quotaTransfer.getTransfers()
                    .forEach(resourceTransfer -> resourceIds.add(resourceTransfer.getResourceId()));
        });
        return storeService.loadFolders(txSession, folderIds).flatMap(folders -> {
            Set<Long> folderServiceIds = folders.stream().map(FolderModel::getServiceId).collect(Collectors.toSet());
            return storeService.loadServices(txSession, Sets.union(serviceIds, folderServiceIds)).flatMap(services ->
                    storeService.loadResources(txSession, resourceIds).flatMap(resources -> {
                        Set<String> unitsEnsemblesIds = resources.stream().map(ResourceModel::getUnitsEnsembleId)
                                .collect(Collectors.toSet());
                        Set<String> providerIds = resources.stream().map(ResourceModel::getProviderId)
                                .collect(Collectors.toSet());
                        return storeService.loadUnitsEnsembles(txSession, unitsEnsemblesIds).flatMap(unitsEnsembles ->
                                storeService.loadProviders(txSession, providerIds).flatMap(providers ->
                                        validationService.validateQuotaTransferParameters(parameters, currentUser,
                                                locale, folders, services, resources, unitsEnsembles, providers,
                                                publicApi)));
                    }));
        });
    }

    private Mono<QuotaTransferParametersContext> loadQuotaTransferParametersContext(YdbTxSession txSession,
                                                                                    Set<QuotaTransfer> transfers) {
        Set<String> folderIds = new HashSet<>();
        Set<Long> serviceIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        transfers.forEach(quotaTransfer -> {
            folderIds.add(quotaTransfer.getDestinationFolderId());
            serviceIds.add(quotaTransfer.getDestinationServiceId());
            quotaTransfer.getTransfers()
                    .forEach(resourceTransfer -> resourceIds.add(resourceTransfer.getResourceId()));
        });
        return storeService.loadFolders(txSession, folderIds).flatMap(folders -> {
            Set<Long> folderServiceIds = folders.stream().map(FolderModel::getServiceId).collect(Collectors.toSet());
            return storeService.loadServices(txSession, Sets.union(serviceIds, folderServiceIds)).flatMap(services ->
                    storeService.loadResources(txSession, resourceIds).flatMap(resources -> {
                        Set<String> unitsEnsemblesIds = resources.stream().map(ResourceModel::getUnitsEnsembleId)
                                .collect(Collectors.toSet());
                        Set<String> providerIds = resources.stream().map(ResourceModel::getProviderId)
                                .collect(Collectors.toSet());
                        return storeService.loadUnitsEnsembles(txSession, unitsEnsemblesIds).flatMap(unitsEnsembles ->
                                storeService.loadProviders(txSession, providerIds).flatMap(providers ->
                                        storeService.loadQuotas(txSession, folderIds).map(quotas ->
                                                QuotaTransferParametersContext.builder()
                                                        .addFolders(folders)
                                                        .addServices(services)
                                                        .addResources(resources)
                                                        .addUnitsEnsembles(unitsEnsembles)
                                                        .addProviders(providers)
                                                        .addQuotas(quotas)
                                                        .build())));
                    }));
        });
    }

    static void prepareNotifiedUsersCreate(CreateTransferRequestConfirmationData confirmation,
                                           ValidatedCreateTransferRequest.Builder validatedBuilder,
                                           ResponsibleAndNotified responsibleAndNotified) {
        if (confirmation.isApply()) {
            return;
        }
        if (confirmation.isConfirm()) {
            responsibleAndNotified.getNotifiedUsers().stream()
                    .filter(u -> !u.getId().equals(confirmation.getUser().getId()))
                    .forEach(validatedBuilder::addNotifiedUser);
        } else {
            validatedBuilder.addNotifiedUsers(responsibleAndNotified.getNotifiedUsers());
        }
    }

    static void prepareNotifiedUsersPut(PutTransferRequestConfirmationData confirmation,
                                        ValidatedPutTransferRequest.Builder validatedBuilder,
                                        ResponsibleAndNotified responsibleAndNotified,
                                        boolean parametersUpdated,
                                        boolean apply,
                                        TransferRequestModel currentTransferRequest) {
        if (apply) {
            return;
        }
        if (parametersUpdated) {
            Set<UserModel> notified;
            if (confirmation.isConfirm()) {
                notified = responsibleAndNotified.getNotifiedUsers().stream()
                        .filter(u -> !u.getId().equals(confirmation.getUser().getId()))
                        .collect(Collectors.toSet());
            } else {
                notified = responsibleAndNotified.getNotifiedUsers();
            }
            Set<String> responsibleIds =
                    TransferRequestResponsibleAndNotifyService.getResponsibleIds(currentTransferRequest);
            notified.stream()
                    .filter(um -> responsibleIds.contains(um.getId()))
                    .forEach(validatedBuilder::addUpdateNotifiedUser);
            notified.stream()
                    .filter(um -> !responsibleIds.contains(um.getId()))
                    .forEach(validatedBuilder::addNewNotifiedUser);
        } else {
            Set<UserModel> notifiedUsers = TransferRequestResponsibleAndNotifyService
                    .excludeAlreadyNotifiedUsers(currentTransferRequest, responsibleAndNotified.getNotifiedUsers());
            if (confirmation.isConfirm()) {
                notifiedUsers.removeIf(userModel -> userModel.getId().equals(confirmation.getUser().getId()));
            }
            validatedBuilder.addNewNotifiedUsers(notifiedUsers);
        }
    }

    private String generateSummary(TransferParameters parameters,
                                   List<ServiceMinimalModel> services, List<FolderModel> folders) {
        if (parameters.getQuotaTransfers().size() == 2) {
            Set<QuotaTransfer> quotaTransferSet = new HashSet<>(parameters.getQuotaTransfers());
            Optional<QuotaTransfer> quotaTransferSourceOptional = parameters.getQuotaTransfers().stream()
                    .filter(this::transferHasNegativeQuota).findFirst();
            QuotaTransfer quotaTransferSource = quotaTransferSourceOptional.orElseThrow();
            quotaTransferSet.remove(quotaTransferSource);
            Optional<QuotaTransfer> quotaTransferDestinationOptional = quotaTransferSet.stream().findFirst();
            QuotaTransfer quotaTransferDestination = quotaTransferDestinationOptional.orElseThrow();
            String serviceSourceSlug = services.stream()
                    .filter(s -> s.getId() == quotaTransferSource.getDestinationServiceId())
                    .findFirst().orElseThrow().getSlug();
            String serviceDestinationSlug = services.stream()
                    .filter(s -> s.getId() == quotaTransferDestination.getDestinationServiceId())
                    .findFirst().orElseThrow().getSlug();
            String folderSource = folders.stream()
                    .filter(f -> f.getId().equals(quotaTransferSource.getDestinationFolderId()))
                    .findFirst().orElseThrow().getDisplayName();
            String folderDestination = folders.stream()
                    .filter(f -> f.getId().equals(quotaTransferDestination.getDestinationFolderId()))
                    .findFirst().orElseThrow().getDisplayName();

            return String.format("Перенос квоты из %s:%s в %s:%s",
                    serviceSourceSlug, folderSource,
                    serviceDestinationSlug, folderDestination);
        } else {
            return "Перенос квоты";
        }
    }

    private boolean transferHasNegativeQuota(QuotaTransfer quotaTransfer) {
        return quotaTransfer.getTransfers().stream()
                .anyMatch(resourceQuotaTransfer -> resourceQuotaTransfer.getDelta() < 0);
    }

}
